/* valareturnstatement.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valareturnstatement.h>
#include <stdlib.h>
#include <string.h>
#include <gee/list.h>
#include <vala/valaexpression.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valareport.h>
#include <vala/valavoidtype.h>
#include <vala/valadatatype.h>
#include <vala/valalocalvariable.h>
#include <vala/valanullliteral.h>




struct _ValaReturnStatementPrivate {
	ValaExpression* _return_expression;
};

#define VALA_RETURN_STATEMENT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_RETURN_STATEMENT, ValaReturnStatementPrivate))
enum  {
	VALA_RETURN_STATEMENT_DUMMY_PROPERTY
};
static void vala_return_statement_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_return_statement_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_return_statement_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static gboolean vala_return_statement_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static void vala_return_statement_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection);
static void vala_return_statement_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection);
static gpointer vala_return_statement_parent_class = NULL;
static ValaStatementIface* vala_return_statement_vala_statement_parent_iface = NULL;
static void vala_return_statement_finalize (ValaCodeNode* obj);



/**
 * Creates a new return statement.
 *
 * @param return_expression the return expression
 * @param source_reference  reference to source code
 * @return                  newly created return statement
 */
ValaReturnStatement* vala_return_statement_construct (GType object_type, ValaExpression* return_expression, ValaSourceReference* source_reference) {
	ValaReturnStatement* self;
	self = (ValaReturnStatement*) g_type_create_instance (object_type);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source_reference);
	vala_return_statement_set_return_expression (self, return_expression);
	return self;
}


ValaReturnStatement* vala_return_statement_new (ValaExpression* return_expression, ValaSourceReference* source_reference) {
	return vala_return_statement_construct (VALA_TYPE_RETURN_STATEMENT, return_expression, source_reference);
}


static void vala_return_statement_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaReturnStatement * self;
	self = (ValaReturnStatement*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_return_statement (visitor, self);
}


static void vala_return_statement_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaReturnStatement * self;
	self = (ValaReturnStatement*) base;
	g_return_if_fail (visitor != NULL);
	if (vala_return_statement_get_return_expression (self) != NULL) {
		vala_code_node_accept ((ValaCodeNode*) vala_return_statement_get_return_expression (self), visitor);
		vala_code_visitor_visit_end_full_expression (visitor, vala_return_statement_get_return_expression (self));
	}
}


static void vala_return_statement_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaReturnStatement * self;
	self = (ValaReturnStatement*) base;
	g_return_if_fail (old_node != NULL);
	g_return_if_fail (new_node != NULL);
	if (vala_return_statement_get_return_expression (self) == old_node) {
		vala_return_statement_set_return_expression (self, new_node);
	}
}


static gboolean vala_return_statement_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaReturnStatement * self;
	gboolean _tmp1;
	gboolean _tmp11;
	ValaLocalVariable* _tmp14;
	ValaSymbol* _tmp13;
	ValaLocalVariable* local;
	gboolean _tmp15;
	gboolean _tmp16;
	gboolean _tmp18;
	GeeList* _tmp21;
	gboolean _tmp22;
	self = (ValaReturnStatement*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	if (vala_return_statement_get_return_expression (self) != NULL) {
		vala_expression_set_target_type (vala_return_statement_get_return_expression (self), analyzer->current_return_type);
	}
	_tmp1 = FALSE;
	if (vala_return_statement_get_return_expression (self) != NULL) {
		_tmp1 = !vala_code_node_check ((ValaCodeNode*) vala_return_statement_get_return_expression (self), analyzer);
	} else {
		_tmp1 = FALSE;
	}
	if (_tmp1) {
		/* ignore inner error*/
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return FALSE;
	}
	if (analyzer->current_return_type == NULL) {
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Return not allowed in this context");
		return FALSE;
	}
	if (vala_return_statement_get_return_expression (self) == NULL) {
		if (!VALA_IS_VOID_TYPE (analyzer->current_return_type)) {
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Return without value in non-void function");
		}
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	if (VALA_IS_VOID_TYPE (analyzer->current_return_type)) {
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Return with value in void function");
		return FALSE;
	}
	if (vala_expression_get_value_type (vala_return_statement_get_return_expression (self)) == NULL) {
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Invalid expression in return value");
		return FALSE;
	}
	if (!vala_data_type_compatible (vala_expression_get_value_type (vala_return_statement_get_return_expression (self)), analyzer->current_return_type)) {
		char* _tmp9;
		char* _tmp8;
		char* _tmp7;
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		_tmp9 = NULL;
		_tmp8 = NULL;
		_tmp7 = NULL;
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp9 = g_strdup_printf ("Return: Cannot convert from `%s' to `%s'", _tmp7 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_return_statement_get_return_expression (self))), _tmp8 = vala_code_node_to_string ((ValaCodeNode*) analyzer->current_return_type)));
		_tmp9 = (g_free (_tmp9), NULL);
		_tmp8 = (g_free (_tmp8), NULL);
		_tmp7 = (g_free (_tmp7), NULL);
		return FALSE;
	}
	_tmp11 = FALSE;
	if (vala_data_type_is_disposable (vala_expression_get_value_type (vala_return_statement_get_return_expression (self)))) {
		_tmp11 = !vala_data_type_get_value_owned (analyzer->current_return_type);
	} else {
		_tmp11 = FALSE;
	}
	if (_tmp11) {
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Return value transfers ownership but method return type hasn't been declared to transfer ownership");
		return FALSE;
	}
	_tmp14 = NULL;
	_tmp13 = NULL;
	local = (_tmp14 = (_tmp13 = vala_expression_get_symbol_reference (vala_return_statement_get_return_expression (self)), VALA_IS_LOCAL_VARIABLE (_tmp13) ? ((ValaLocalVariable*) _tmp13) : NULL), (_tmp14 == NULL) ? NULL : vala_code_node_ref (_tmp14));
	_tmp15 = FALSE;
	_tmp16 = FALSE;
	if (local != NULL) {
		_tmp16 = vala_data_type_is_disposable (vala_local_variable_get_variable_type (local));
	} else {
		_tmp16 = FALSE;
	}
	if (_tmp16) {
		_tmp15 = !vala_data_type_get_value_owned (analyzer->current_return_type);
	} else {
		_tmp15 = FALSE;
	}
	if (_tmp15) {
		gboolean _tmp17;
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Local variable with strong reference used as return value and method return type has not been declared to transfer ownership");
		return (_tmp17 = FALSE, (local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL)), _tmp17);
	}
	_tmp18 = FALSE;
	if (VALA_IS_NULL_LITERAL (vala_return_statement_get_return_expression (self))) {
		_tmp18 = !vala_data_type_get_nullable (analyzer->current_return_type);
	} else {
		_tmp18 = FALSE;
	}
	if (_tmp18) {
		char* _tmp20;
		char* _tmp19;
		_tmp20 = NULL;
		_tmp19 = NULL;
		vala_report_warning (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp20 = g_strdup_printf ("`null' incompatible with return type `%s`", _tmp19 = vala_code_node_to_string ((ValaCodeNode*) analyzer->current_return_type)));
		_tmp20 = (g_free (_tmp20), NULL);
		_tmp19 = (g_free (_tmp19), NULL);
	}
	_tmp21 = NULL;
	vala_code_node_add_error_types ((ValaCodeNode*) self, _tmp21 = vala_code_node_get_error_types ((ValaCodeNode*) vala_return_statement_get_return_expression (self)));
	(_tmp21 == NULL) ? NULL : (_tmp21 = (gee_collection_object_unref (_tmp21), NULL));
	return (_tmp22 = !vala_code_node_get_error ((ValaCodeNode*) self), (local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL)), _tmp22);
}


static void vala_return_statement_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaReturnStatement * self;
	self = (ValaReturnStatement*) base;
	g_return_if_fail (collection != NULL);
	if (vala_return_statement_get_return_expression (self) != NULL) {
		vala_code_node_get_defined_variables ((ValaCodeNode*) vala_return_statement_get_return_expression (self), collection);
	}
}


static void vala_return_statement_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaReturnStatement * self;
	self = (ValaReturnStatement*) base;
	g_return_if_fail (collection != NULL);
	if (vala_return_statement_get_return_expression (self) != NULL) {
		vala_code_node_get_used_variables ((ValaCodeNode*) vala_return_statement_get_return_expression (self), collection);
	}
}


ValaExpression* vala_return_statement_get_return_expression (ValaReturnStatement* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_return_expression;
}


void vala_return_statement_set_return_expression (ValaReturnStatement* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_return_expression = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_return_expression == NULL) ? NULL : (self->priv->_return_expression = (vala_code_node_unref (self->priv->_return_expression), NULL)), _tmp2);
	if (self->priv->_return_expression != NULL) {
		vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_return_expression, (ValaCodeNode*) self);
	}
}


static void vala_return_statement_class_init (ValaReturnStatementClass * klass) {
	vala_return_statement_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_return_statement_finalize;
	g_type_class_add_private (klass, sizeof (ValaReturnStatementPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_return_statement_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_return_statement_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_return_statement_real_replace_expression;
	VALA_CODE_NODE_CLASS (klass)->check = vala_return_statement_real_check;
	VALA_CODE_NODE_CLASS (klass)->get_defined_variables = vala_return_statement_real_get_defined_variables;
	VALA_CODE_NODE_CLASS (klass)->get_used_variables = vala_return_statement_real_get_used_variables;
}


static void vala_return_statement_vala_statement_interface_init (ValaStatementIface * iface) {
	vala_return_statement_vala_statement_parent_iface = g_type_interface_peek_parent (iface);
}


static void vala_return_statement_instance_init (ValaReturnStatement * self) {
	self->priv = VALA_RETURN_STATEMENT_GET_PRIVATE (self);
}


static void vala_return_statement_finalize (ValaCodeNode* obj) {
	ValaReturnStatement * self;
	self = VALA_RETURN_STATEMENT (obj);
	(self->priv->_return_expression == NULL) ? NULL : (self->priv->_return_expression = (vala_code_node_unref (self->priv->_return_expression), NULL));
	VALA_CODE_NODE_CLASS (vala_return_statement_parent_class)->finalize (obj);
}


GType vala_return_statement_get_type (void) {
	static GType vala_return_statement_type_id = 0;
	if (vala_return_statement_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaReturnStatementClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_return_statement_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaReturnStatement), 0, (GInstanceInitFunc) vala_return_statement_instance_init, NULL };
		static const GInterfaceInfo vala_statement_info = { (GInterfaceInitFunc) vala_return_statement_vala_statement_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_return_statement_type_id = g_type_register_static (VALA_TYPE_CODE_NODE, "ValaReturnStatement", &g_define_type_info, 0);
		g_type_add_interface_static (vala_return_statement_type_id, VALA_TYPE_STATEMENT, &vala_statement_info);
	}
	return vala_return_statement_type_id;
}




