/* hashset.vala
 *
 * Copyright (C) 1995-1997  Peter Mattis, Spencer Kimball and Josh MacDonald
 * Copyright (C) 1997-2000  GLib Team and others
 * Copyright (C) 2007-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>


#define GEE_TYPE_COLLECTION_OBJECT (gee_collection_object_get_type ())
#define GEE_COLLECTION_OBJECT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_COLLECTION_OBJECT, GeeCollectionObject))
#define GEE_COLLECTION_OBJECT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_TYPE_COLLECTION_OBJECT, GeeCollectionObjectClass))
#define GEE_IS_COLLECTION_OBJECT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_COLLECTION_OBJECT))
#define GEE_IS_COLLECTION_OBJECT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_TYPE_COLLECTION_OBJECT))
#define GEE_COLLECTION_OBJECT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_TYPE_COLLECTION_OBJECT, GeeCollectionObjectClass))

typedef struct _GeeCollectionObject GeeCollectionObject;
typedef struct _GeeCollectionObjectClass GeeCollectionObjectClass;
typedef struct _GeeCollectionObjectPrivate GeeCollectionObjectPrivate;

#define GEE_TYPE_ITERABLE (gee_iterable_get_type ())
#define GEE_ITERABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_ITERABLE, GeeIterable))
#define GEE_IS_ITERABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_ITERABLE))
#define GEE_ITERABLE_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_ITERABLE, GeeIterableIface))

typedef struct _GeeIterable GeeIterable;
typedef struct _GeeIterableIface GeeIterableIface;

#define GEE_TYPE_ITERATOR (gee_iterator_get_type ())
#define GEE_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_ITERATOR, GeeIterator))
#define GEE_IS_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_ITERATOR))
#define GEE_ITERATOR_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_ITERATOR, GeeIteratorIface))

typedef struct _GeeIterator GeeIterator;
typedef struct _GeeIteratorIface GeeIteratorIface;

#define GEE_TYPE_COLLECTION (gee_collection_get_type ())
#define GEE_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_COLLECTION, GeeCollection))
#define GEE_IS_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_COLLECTION))
#define GEE_COLLECTION_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_COLLECTION, GeeCollectionIface))

typedef struct _GeeCollection GeeCollection;
typedef struct _GeeCollectionIface GeeCollectionIface;

#define GEE_TYPE_SET (gee_set_get_type ())
#define GEE_SET(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_SET, GeeSet))
#define GEE_IS_SET(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_SET))
#define GEE_SET_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_SET, GeeSetIface))

typedef struct _GeeSet GeeSet;
typedef struct _GeeSetIface GeeSetIface;

#define GEE_TYPE_HASH_SET (gee_hash_set_get_type ())
#define GEE_HASH_SET(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_HASH_SET, GeeHashSet))
#define GEE_HASH_SET_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_TYPE_HASH_SET, GeeHashSetClass))
#define GEE_IS_HASH_SET(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_HASH_SET))
#define GEE_IS_HASH_SET_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_TYPE_HASH_SET))
#define GEE_HASH_SET_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_TYPE_HASH_SET, GeeHashSetClass))

typedef struct _GeeHashSet GeeHashSet;
typedef struct _GeeHashSetClass GeeHashSetClass;
typedef struct _GeeHashSetPrivate GeeHashSetPrivate;
typedef struct _GeeHashSetNode GeeHashSetNode;

#define GEE_HASH_SET_TYPE_ITERATOR (gee_hash_set_iterator_get_type ())
#define GEE_HASH_SET_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_HASH_SET_TYPE_ITERATOR, GeeHashSetIterator))
#define GEE_HASH_SET_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_HASH_SET_TYPE_ITERATOR, GeeHashSetIteratorClass))
#define GEE_HASH_SET_IS_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_HASH_SET_TYPE_ITERATOR))
#define GEE_HASH_SET_IS_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_HASH_SET_TYPE_ITERATOR))
#define GEE_HASH_SET_ITERATOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_HASH_SET_TYPE_ITERATOR, GeeHashSetIteratorClass))

typedef struct _GeeHashSetIterator GeeHashSetIterator;
typedef struct _GeeHashSetIteratorClass GeeHashSetIteratorClass;
typedef struct _GeeHashSetIteratorPrivate GeeHashSetIteratorPrivate;

/**
 * Base class for all collections.
 */
struct _GeeCollectionObject {
	GTypeInstance parent_instance;
	volatile int ref_count;
	GeeCollectionObjectPrivate * priv;
};

struct _GeeCollectionObjectClass {
	GTypeClass parent_class;
	void (*finalize) (GeeCollectionObject *self);
};

/**
 * Implemented by classes that support a simple iteration over instances of the
 * collection.
 */
struct _GeeIterableIface {
	GTypeInterface parent_iface;
	GType (*get_element_type) (GeeIterable* self);
	GeeIterator* (*iterator) (GeeIterable* self);
};

/**
 * Implemented by classes that support a simple iteration over instances of the
 * collection.
 */
struct _GeeIteratorIface {
	GTypeInterface parent_iface;
	gboolean (*next) (GeeIterator* self);
	gpointer (*get) (GeeIterator* self);
};

/**
 * Serves as the base interface for implementing collection classes. Defines
 * size, iteration, and modification methods.
 */
struct _GeeCollectionIface {
	GTypeInterface parent_iface;
	gboolean (*contains) (GeeCollection* self, gconstpointer item);
	gboolean (*add) (GeeCollection* self, gconstpointer item);
	gboolean (*remove) (GeeCollection* self, gconstpointer item);
	void (*clear) (GeeCollection* self);
	gint (*get_size) (GeeCollection* self);
};

/**
 * A set is a collection without duplicates.
 */
struct _GeeSetIface {
	GTypeInterface parent_iface;
};

/**
 * Hashtable implementation of the Set interface.
 */
struct _GeeHashSet {
	GeeCollectionObject parent_instance;
	GeeHashSetPrivate * priv;
};

struct _GeeHashSetClass {
	GeeCollectionObjectClass parent_class;
};

struct _GeeHashSetPrivate {
	GType g_type;
	GBoxedCopyFunc g_dup_func;
	GDestroyNotify g_destroy_func;
	gint _array_size;
	gint _nnodes;
	GeeHashSetNode** _nodes;
	gint _nodes_length1;
	gint _nodes_size;
	gint _stamp;
	GHashFunc _hash_func;
	GEqualFunc _equal_func;
};

struct _GeeHashSetNode {
	gpointer key;
	GeeHashSetNode* next;
	guint key_hash;
};

struct _GeeHashSetIterator {
	GeeCollectionObject parent_instance;
	GeeHashSetIteratorPrivate * priv;
};

struct _GeeHashSetIteratorClass {
	GeeCollectionObjectClass parent_class;
};

struct _GeeHashSetIteratorPrivate {
	GType g_type;
	GBoxedCopyFunc g_dup_func;
	GDestroyNotify g_destroy_func;
	GeeHashSet* _set;
	gint _index;
	GeeHashSetNode* _node;
	gint _stamp;
};



gpointer gee_collection_object_ref (gpointer instance);
void gee_collection_object_unref (gpointer instance);
GParamSpec* gee_param_spec_collection_object (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void gee_value_set_collection_object (GValue* value, gpointer v_object);
gpointer gee_value_get_collection_object (const GValue* value);
GType gee_collection_object_get_type (void);
GType gee_iterator_get_type (void);
GType gee_iterable_get_type (void);
GType gee_collection_get_type (void);
GType gee_set_get_type (void);
GType gee_hash_set_get_type (void);
#define GEE_HASH_SET_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_TYPE_HASH_SET, GeeHashSetPrivate))
enum  {
	GEE_HASH_SET_DUMMY_PROPERTY
};
#define GEE_HASH_SET_MIN_SIZE 11
#define GEE_HASH_SET_MAX_SIZE 13845163
void gee_hash_set_set_hash_func (GeeHashSet* self, GHashFunc value);
void gee_hash_set_set_equal_func (GeeHashSet* self, GEqualFunc value);
GeeHashSet* gee_hash_set_new (GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GHashFunc hash_func, GEqualFunc equal_func);
GeeHashSet* gee_hash_set_construct (GType object_type, GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GHashFunc hash_func, GEqualFunc equal_func);
GeeHashSet* gee_hash_set_new (GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GHashFunc hash_func, GEqualFunc equal_func);
static GeeHashSetNode** gee_hash_set_lookup_node (GeeHashSet* self, gconstpointer key);
static gboolean gee_hash_set_real_contains (GeeCollection* base, gconstpointer key);
static GType gee_hash_set_real_get_element_type (GeeIterable* base);
static GeeHashSetIterator* gee_hash_set_iterator_new (GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GeeHashSet* set);
static GeeHashSetIterator* gee_hash_set_iterator_construct (GType object_type, GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GeeHashSet* set);
static GType gee_hash_set_iterator_get_type (void);
static GeeIterator* gee_hash_set_real_iterator (GeeIterable* base);
static GeeHashSetNode* gee_hash_set_node_new (gpointer k, guint hash);
static GeeHashSetNode* gee_hash_set_node_new (gpointer k, guint hash);
static void gee_hash_set_resize (GeeHashSet* self);
static gboolean gee_hash_set_real_add (GeeCollection* base, gconstpointer key);
static gboolean gee_hash_set_real_remove (GeeCollection* base, gconstpointer key);
static void gee_hash_set_real_clear (GeeCollection* base);
void gee_collection_clear (GeeCollection* self);
static void gee_hash_set_node_free (GeeHashSetNode* self);
static void gee_hash_set_node_instance_init (GeeHashSetNode * self);
#define GEE_HASH_SET_ITERATOR_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_HASH_SET_TYPE_ITERATOR, GeeHashSetIteratorPrivate))
enum  {
	GEE_HASH_SET_ITERATOR_DUMMY_PROPERTY
};
static void gee_hash_set_iterator_set_set (GeeHashSetIterator* self, GeeHashSet* value);
static GeeHashSetIterator* gee_hash_set_iterator_new (GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GeeHashSet* set);
static gboolean gee_hash_set_iterator_real_next (GeeIterator* base);
static gpointer gee_hash_set_iterator_real_get (GeeIterator* base);
static gpointer gee_hash_set_iterator_parent_class = NULL;
static GeeIteratorIface* gee_hash_set_iterator_gee_iterator_parent_iface = NULL;
static void gee_hash_set_iterator_finalize (GeeCollectionObject* obj);
static gpointer gee_hash_set_parent_class = NULL;
static GeeIterableIface* gee_hash_set_gee_iterable_parent_iface = NULL;
static GeeCollectionIface* gee_hash_set_gee_collection_parent_iface = NULL;
static GeeSetIface* gee_hash_set_gee_set_parent_iface = NULL;
static void gee_hash_set_finalize (GeeCollectionObject* obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);



GeeHashSet* gee_hash_set_construct (GType object_type, GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GHashFunc hash_func, GEqualFunc equal_func) {
	GeeHashSet* self;
	GeeHashSetNode** _tmp0;
	self = (GeeHashSet*) g_type_create_instance (object_type);
	self->priv->g_type = g_type;
	self->priv->g_dup_func = g_dup_func;
	self->priv->g_destroy_func = g_destroy_func;
	gee_hash_set_set_hash_func (self, hash_func);
	gee_hash_set_set_equal_func (self, equal_func);
	self->priv->_array_size = GEE_HASH_SET_MIN_SIZE;
	_tmp0 = NULL;
	self->priv->_nodes = (_tmp0 = g_new0 (GeeHashSetNode*, self->priv->_array_size + 1), self->priv->_nodes = (_vala_array_free (self->priv->_nodes, self->priv->_nodes_length1, (GDestroyNotify) gee_hash_set_node_free), NULL), self->priv->_nodes_length1 = self->priv->_array_size, self->priv->_nodes_size = self->priv->_nodes_length1, _tmp0);
	return self;
}


GeeHashSet* gee_hash_set_new (GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GHashFunc hash_func, GEqualFunc equal_func) {
	return gee_hash_set_construct (GEE_TYPE_HASH_SET, g_type, g_dup_func, g_destroy_func, hash_func, equal_func);
}


static GeeHashSetNode** gee_hash_set_lookup_node (GeeHashSet* self, gconstpointer key) {
	guint hash_value;
	GeeHashSetNode** node;
	g_return_val_if_fail (self != NULL, NULL);
	hash_value = self->priv->_hash_func (key);
	node = &self->priv->_nodes[hash_value % self->priv->_array_size];
	while (TRUE) {
		gboolean _tmp0;
		_tmp0 = FALSE;
		if ((*node) != NULL) {
			gboolean _tmp1;
			_tmp1 = FALSE;
			if (hash_value != (*node)->key_hash) {
				_tmp1 = TRUE;
			} else {
				_tmp1 = !self->priv->_equal_func ((*node)->key, key);
			}
			_tmp0 = _tmp1;
		} else {
			_tmp0 = FALSE;
		}
		if (!_tmp0) {
			break;
		}
		node = &(*node)->next;
	}
	return node;
}


static gboolean gee_hash_set_real_contains (GeeCollection* base, gconstpointer key) {
	GeeHashSet * self;
	GeeHashSetNode** node;
	self = (GeeHashSet*) base;
	node = gee_hash_set_lookup_node (self, key);
	return (*node) != NULL;
}


static GType gee_hash_set_real_get_element_type (GeeIterable* base) {
	GeeHashSet * self;
	self = (GeeHashSet*) base;
	return self->priv->g_type;
}


static GeeIterator* gee_hash_set_real_iterator (GeeIterable* base) {
	GeeHashSet * self;
	self = (GeeHashSet*) base;
	return (GeeIterator*) gee_hash_set_iterator_new (self->priv->g_type, (GBoxedCopyFunc) self->priv->g_dup_func, self->priv->g_destroy_func, self);
}


static gboolean gee_hash_set_real_add (GeeCollection* base, gconstpointer key) {
	GeeHashSet * self;
	GeeHashSetNode** node;
	self = (GeeHashSet*) base;
	node = gee_hash_set_lookup_node (self, key);
	if ((*node) != NULL) {
		return FALSE;
	} else {
		guint hash_value;
		gconstpointer _tmp1;
		hash_value = self->priv->_hash_func (key);
		_tmp1 = NULL;
		*node = gee_hash_set_node_new ((_tmp1 = key, ((_tmp1 == NULL) || (self->priv->g_dup_func == NULL)) ? ((gpointer) _tmp1) : self->priv->g_dup_func ((gpointer) _tmp1)), hash_value);
		self->priv->_nnodes++;
		gee_hash_set_resize (self);
		self->priv->_stamp++;
		return TRUE;
	}
}


static gboolean gee_hash_set_real_remove (GeeCollection* base, gconstpointer key) {
	GeeHashSet * self;
	GeeHashSetNode** node;
	self = (GeeHashSet*) base;
	node = gee_hash_set_lookup_node (self, key);
	if ((*node) != NULL) {
		gpointer _tmp0;
		_tmp0 = NULL;
		(*node)->key = (_tmp0 = NULL, (((*node)->key == NULL) || (self->priv->g_destroy_func == NULL)) ? NULL : ((*node)->key = (self->priv->g_destroy_func ((*node)->key), NULL)), _tmp0);
		*node = (*node)->next;
		self->priv->_nnodes--;
		gee_hash_set_resize (self);
		self->priv->_stamp++;
		return TRUE;
	}
	return FALSE;
}


static void gee_hash_set_real_clear (GeeCollection* base) {
	GeeHashSet * self;
	self = (GeeHashSet*) base;
	{
		gint i;
		i = 0;
		for (; i < self->priv->_array_size; i++) {
			GeeHashSetNode* _tmp0;
			GeeHashSetNode* node;
			_tmp0 = NULL;
			node = (_tmp0 = self->priv->_nodes[i], self->priv->_nodes[i] = NULL, _tmp0);
			while (node != NULL) {
				GeeHashSetNode* _tmp1;
				GeeHashSetNode* next;
				gpointer _tmp2;
				GeeHashSetNode* _tmp4;
				GeeHashSetNode* _tmp3;
				_tmp1 = NULL;
				next = (_tmp1 = node->next, node->next = NULL, _tmp1);
				_tmp2 = NULL;
				node->key = (_tmp2 = NULL, ((node->key == NULL) || (self->priv->g_destroy_func == NULL)) ? NULL : (node->key = (self->priv->g_destroy_func (node->key), NULL)), _tmp2);
				_tmp4 = NULL;
				_tmp3 = NULL;
				node = (_tmp4 = (_tmp3 = next, next = NULL, _tmp3), (node == NULL) ? NULL : (node = (gee_hash_set_node_free (node), NULL)), _tmp4);
				(next == NULL) ? NULL : (next = (gee_hash_set_node_free (next), NULL));
			}
			(node == NULL) ? NULL : (node = (gee_hash_set_node_free (node), NULL));
		}
	}
	self->priv->_nnodes = 0;
	gee_hash_set_resize (self);
}


static void gee_hash_set_resize (GeeHashSet* self) {
	gboolean _tmp0;
	gboolean _tmp1;
	g_return_if_fail (self != NULL);
	_tmp0 = FALSE;
	_tmp1 = FALSE;
	if (self->priv->_array_size >= (3 * self->priv->_nnodes)) {
		_tmp1 = self->priv->_array_size >= GEE_HASH_SET_MIN_SIZE;
	} else {
		_tmp1 = FALSE;
	}
	if (_tmp1) {
		_tmp0 = TRUE;
	} else {
		gboolean _tmp2;
		_tmp2 = FALSE;
		if ((3 * self->priv->_array_size) <= self->priv->_nnodes) {
			_tmp2 = self->priv->_array_size < GEE_HASH_SET_MAX_SIZE;
		} else {
			_tmp2 = FALSE;
		}
		_tmp0 = _tmp2;
	}
	if (_tmp0) {
		gint new_array_size;
		GeeHashSetNode** _tmp3;
		gint new_nodes_size;
		gint new_nodes_length1;
		GeeHashSetNode** new_nodes;
		GeeHashSetNode** _tmp15;
		GeeHashSetNode** _tmp14;
		new_array_size = (gint) g_spaced_primes_closest ((guint) self->priv->_nnodes);
		new_array_size = CLAMP (new_array_size, GEE_HASH_SET_MIN_SIZE, GEE_HASH_SET_MAX_SIZE);
		_tmp3 = NULL;
		new_nodes = (_tmp3 = g_new0 (GeeHashSetNode*, new_array_size + 1), new_nodes_length1 = new_array_size, new_nodes_size = new_nodes_length1, _tmp3);
		{
			gint i;
			i = 0;
			for (; i < self->priv->_array_size; i++) {
				GeeHashSetNode* node;
				GeeHashSetNode* next;
				GeeHashSetNode* _tmp7;
				GeeHashSetNode* _tmp6;
				GeeHashSetNode* _tmp5;
				GeeHashSetNode* _tmp4;
				node = NULL;
				next = NULL;
				_tmp7 = NULL;
				_tmp6 = NULL;
				_tmp5 = NULL;
				_tmp4 = NULL;
				for (node = (_tmp5 = (_tmp4 = self->priv->_nodes[i], self->priv->_nodes[i] = NULL, _tmp4), (node == NULL) ? NULL : (node = (gee_hash_set_node_free (node), NULL)), _tmp5); node != NULL; node = (_tmp7 = (_tmp6 = next, next = NULL, _tmp6), (node == NULL) ? NULL : (node = (gee_hash_set_node_free (node), NULL)), _tmp7)) {
					GeeHashSetNode* _tmp9;
					GeeHashSetNode* _tmp8;
					guint hash_val;
					GeeHashSetNode* _tmp11;
					GeeHashSetNode* _tmp10;
					GeeHashSetNode* _tmp13;
					GeeHashSetNode* _tmp12;
					_tmp9 = NULL;
					_tmp8 = NULL;
					next = (_tmp9 = (_tmp8 = node->next, node->next = NULL, _tmp8), (next == NULL) ? NULL : (next = (gee_hash_set_node_free (next), NULL)), _tmp9);
					hash_val = node->key_hash % new_array_size;
					_tmp11 = NULL;
					_tmp10 = NULL;
					node->next = (_tmp11 = (_tmp10 = new_nodes[hash_val], new_nodes[hash_val] = NULL, _tmp10), (node->next == NULL) ? NULL : (node->next = (gee_hash_set_node_free (node->next), NULL)), _tmp11);
					_tmp13 = NULL;
					_tmp12 = NULL;
					new_nodes[hash_val] = (_tmp13 = (_tmp12 = node, node = NULL, _tmp12), (new_nodes[hash_val] == NULL) ? NULL : (new_nodes[hash_val] = (gee_hash_set_node_free (new_nodes[hash_val]), NULL)), _tmp13);
				}
				(node == NULL) ? NULL : (node = (gee_hash_set_node_free (node), NULL));
				(next == NULL) ? NULL : (next = (gee_hash_set_node_free (next), NULL));
			}
		}
		_tmp15 = NULL;
		_tmp14 = NULL;
		self->priv->_nodes = (_tmp15 = (_tmp14 = new_nodes, new_nodes = NULL, _tmp14), self->priv->_nodes = (_vala_array_free (self->priv->_nodes, self->priv->_nodes_length1, (GDestroyNotify) gee_hash_set_node_free), NULL), self->priv->_nodes_length1 = new_nodes_length1, self->priv->_nodes_size = self->priv->_nodes_length1, _tmp15);
		self->priv->_array_size = new_array_size;
		new_nodes = (_vala_array_free (new_nodes, new_nodes_length1, (GDestroyNotify) gee_hash_set_node_free), NULL);
	}
}


static gint gee_hash_set_real_get_size (GeeCollection* base) {
	GeeHashSet* self;
	self = (GeeHashSet*) base;
	return self->priv->_nnodes;
}


void gee_hash_set_set_hash_func (GeeHashSet* self, GHashFunc value) {
	g_return_if_fail (self != NULL);
	self->priv->_hash_func = value;
}


void gee_hash_set_set_equal_func (GeeHashSet* self, GEqualFunc value) {
	g_return_if_fail (self != NULL);
	self->priv->_equal_func = value;
}


static GeeHashSetNode* gee_hash_set_node_new (gpointer k, guint hash) {
	GeeHashSetNode* self;
	gpointer _tmp0;
	self = g_slice_new0 (GeeHashSetNode);
	gee_hash_set_node_instance_init (self);
	_tmp0 = NULL;
	self->key = (_tmp0 = k, k = NULL, _tmp0);
	self->key_hash = hash;
	return self;
}


static void gee_hash_set_node_instance_init (GeeHashSetNode * self) {
}


static void gee_hash_set_node_free (GeeHashSetNode* self) {
	(self->next == NULL) ? NULL : (self->next = (gee_hash_set_node_free (self->next), NULL));
	g_slice_free (GeeHashSetNode, self);
}


static GeeHashSetIterator* gee_hash_set_iterator_construct (GType object_type, GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GeeHashSet* set) {
	GeeHashSetIterator* self;
	g_return_val_if_fail (set != NULL, NULL);
	self = (GeeHashSetIterator*) g_type_create_instance (object_type);
	self->priv->g_type = g_type;
	self->priv->g_dup_func = g_dup_func;
	self->priv->g_destroy_func = g_destroy_func;
	gee_hash_set_iterator_set_set (self, set);
	return self;
}


static GeeHashSetIterator* gee_hash_set_iterator_new (GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GeeHashSet* set) {
	return gee_hash_set_iterator_construct (GEE_HASH_SET_TYPE_ITERATOR, g_type, g_dup_func, g_destroy_func, set);
}


static gboolean gee_hash_set_iterator_real_next (GeeIterator* base) {
	GeeHashSetIterator * self;
	self = (GeeHashSetIterator*) base;
	if (self->priv->_node != NULL) {
		self->priv->_node = self->priv->_node->next;
	}
	while (TRUE) {
		gboolean _tmp0;
		_tmp0 = FALSE;
		if (self->priv->_node == NULL) {
			_tmp0 = (self->priv->_index + 1) < self->priv->_set->priv->_array_size;
		} else {
			_tmp0 = FALSE;
		}
		if (!_tmp0) {
			break;
		}
		self->priv->_index++;
		self->priv->_node = self->priv->_set->priv->_nodes[self->priv->_index];
	}
	return self->priv->_node != NULL;
}


static gpointer gee_hash_set_iterator_real_get (GeeIterator* base) {
	GeeHashSetIterator * self;
	gconstpointer _tmp0;
	self = (GeeHashSetIterator*) base;
	g_assert (self->priv->_stamp == self->priv->_set->priv->_stamp);
	g_assert (self->priv->_node != NULL);
	_tmp0 = NULL;
	return (_tmp0 = self->priv->_node->key, ((_tmp0 == NULL) || (self->priv->g_dup_func == NULL)) ? ((gpointer) _tmp0) : self->priv->g_dup_func ((gpointer) _tmp0));
}


static void gee_hash_set_iterator_set_set (GeeHashSetIterator* self, GeeHashSet* value) {
	GeeHashSet* _tmp1;
	GeeHashSet* _tmp0;
	g_return_if_fail (self != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->_set = (_tmp1 = (_tmp0 = value, (_tmp0 == NULL) ? NULL : gee_collection_object_ref (_tmp0)), (self->priv->_set == NULL) ? NULL : (self->priv->_set = (gee_collection_object_unref (self->priv->_set), NULL)), _tmp1);
	self->priv->_stamp = self->priv->_set->priv->_stamp;
}


static void gee_hash_set_iterator_class_init (GeeHashSetIteratorClass * klass) {
	gee_hash_set_iterator_parent_class = g_type_class_peek_parent (klass);
	GEE_COLLECTION_OBJECT_CLASS (klass)->finalize = gee_hash_set_iterator_finalize;
	g_type_class_add_private (klass, sizeof (GeeHashSetIteratorPrivate));
}


static void gee_hash_set_iterator_gee_iterator_interface_init (GeeIteratorIface * iface) {
	gee_hash_set_iterator_gee_iterator_parent_iface = g_type_interface_peek_parent (iface);
	iface->next = gee_hash_set_iterator_real_next;
	iface->get = gee_hash_set_iterator_real_get;
}


static void gee_hash_set_iterator_instance_init (GeeHashSetIterator * self) {
	self->priv = GEE_HASH_SET_ITERATOR_GET_PRIVATE (self);
	self->priv->_index = -1;
	self->priv->_stamp = 0;
}


static void gee_hash_set_iterator_finalize (GeeCollectionObject* obj) {
	GeeHashSetIterator * self;
	self = GEE_HASH_SET_ITERATOR (obj);
	(self->priv->_set == NULL) ? NULL : (self->priv->_set = (gee_collection_object_unref (self->priv->_set), NULL));
	GEE_COLLECTION_OBJECT_CLASS (gee_hash_set_iterator_parent_class)->finalize (obj);
}


static GType gee_hash_set_iterator_get_type (void) {
	static GType gee_hash_set_iterator_type_id = 0;
	if (gee_hash_set_iterator_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeHashSetIteratorClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_hash_set_iterator_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeHashSetIterator), 0, (GInstanceInitFunc) gee_hash_set_iterator_instance_init, NULL };
		static const GInterfaceInfo gee_iterator_info = { (GInterfaceInitFunc) gee_hash_set_iterator_gee_iterator_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_hash_set_iterator_type_id = g_type_register_static (GEE_TYPE_COLLECTION_OBJECT, "GeeHashSetIterator", &g_define_type_info, 0);
		g_type_add_interface_static (gee_hash_set_iterator_type_id, GEE_TYPE_ITERATOR, &gee_iterator_info);
	}
	return gee_hash_set_iterator_type_id;
}


static void gee_hash_set_class_init (GeeHashSetClass * klass) {
	gee_hash_set_parent_class = g_type_class_peek_parent (klass);
	GEE_COLLECTION_OBJECT_CLASS (klass)->finalize = gee_hash_set_finalize;
	g_type_class_add_private (klass, sizeof (GeeHashSetPrivate));
}


static void gee_hash_set_gee_iterable_interface_init (GeeIterableIface * iface) {
	gee_hash_set_gee_iterable_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_element_type = gee_hash_set_real_get_element_type;
	iface->iterator = gee_hash_set_real_iterator;
}


static void gee_hash_set_gee_collection_interface_init (GeeCollectionIface * iface) {
	gee_hash_set_gee_collection_parent_iface = g_type_interface_peek_parent (iface);
	iface->contains = gee_hash_set_real_contains;
	iface->add = gee_hash_set_real_add;
	iface->remove = gee_hash_set_real_remove;
	iface->clear = gee_hash_set_real_clear;
	iface->get_size = gee_hash_set_real_get_size;
}


static void gee_hash_set_gee_set_interface_init (GeeSetIface * iface) {
	gee_hash_set_gee_set_parent_iface = g_type_interface_peek_parent (iface);
}


static void gee_hash_set_instance_init (GeeHashSet * self) {
	self->priv = GEE_HASH_SET_GET_PRIVATE (self);
	self->priv->_stamp = 0;
}


static void gee_hash_set_finalize (GeeCollectionObject* obj) {
	GeeHashSet * self;
	self = GEE_HASH_SET (obj);
	{
		gee_collection_clear ((GeeCollection*) self);
	}
	self->priv->_nodes = (_vala_array_free (self->priv->_nodes, self->priv->_nodes_length1, (GDestroyNotify) gee_hash_set_node_free), NULL);
	GEE_COLLECTION_OBJECT_CLASS (gee_hash_set_parent_class)->finalize (obj);
}


GType gee_hash_set_get_type (void) {
	static GType gee_hash_set_type_id = 0;
	if (gee_hash_set_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeHashSetClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_hash_set_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeHashSet), 0, (GInstanceInitFunc) gee_hash_set_instance_init, NULL };
		static const GInterfaceInfo gee_iterable_info = { (GInterfaceInitFunc) gee_hash_set_gee_iterable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		static const GInterfaceInfo gee_collection_info = { (GInterfaceInitFunc) gee_hash_set_gee_collection_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		static const GInterfaceInfo gee_set_info = { (GInterfaceInitFunc) gee_hash_set_gee_set_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_hash_set_type_id = g_type_register_static (GEE_TYPE_COLLECTION_OBJECT, "GeeHashSet", &g_define_type_info, 0);
		g_type_add_interface_static (gee_hash_set_type_id, GEE_TYPE_ITERABLE, &gee_iterable_info);
		g_type_add_interface_static (gee_hash_set_type_id, GEE_TYPE_COLLECTION, &gee_collection_info);
		g_type_add_interface_static (gee_hash_set_type_id, GEE_TYPE_SET, &gee_set_info);
	}
	return gee_hash_set_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
	g_free (array);
}




