/* readonlylist.vala
 *
 * Copyright (C) 2007-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>


#define GEE_TYPE_COLLECTION_OBJECT (gee_collection_object_get_type ())
#define GEE_COLLECTION_OBJECT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_COLLECTION_OBJECT, GeeCollectionObject))
#define GEE_COLLECTION_OBJECT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_TYPE_COLLECTION_OBJECT, GeeCollectionObjectClass))
#define GEE_IS_COLLECTION_OBJECT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_COLLECTION_OBJECT))
#define GEE_IS_COLLECTION_OBJECT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_TYPE_COLLECTION_OBJECT))
#define GEE_COLLECTION_OBJECT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_TYPE_COLLECTION_OBJECT, GeeCollectionObjectClass))

typedef struct _GeeCollectionObject GeeCollectionObject;
typedef struct _GeeCollectionObjectClass GeeCollectionObjectClass;
typedef struct _GeeCollectionObjectPrivate GeeCollectionObjectPrivate;

#define GEE_TYPE_ITERABLE (gee_iterable_get_type ())
#define GEE_ITERABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_ITERABLE, GeeIterable))
#define GEE_IS_ITERABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_ITERABLE))
#define GEE_ITERABLE_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_ITERABLE, GeeIterableIface))

typedef struct _GeeIterable GeeIterable;
typedef struct _GeeIterableIface GeeIterableIface;

#define GEE_TYPE_ITERATOR (gee_iterator_get_type ())
#define GEE_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_ITERATOR, GeeIterator))
#define GEE_IS_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_ITERATOR))
#define GEE_ITERATOR_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_ITERATOR, GeeIteratorIface))

typedef struct _GeeIterator GeeIterator;
typedef struct _GeeIteratorIface GeeIteratorIface;

#define GEE_TYPE_COLLECTION (gee_collection_get_type ())
#define GEE_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_COLLECTION, GeeCollection))
#define GEE_IS_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_COLLECTION))
#define GEE_COLLECTION_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_COLLECTION, GeeCollectionIface))

typedef struct _GeeCollection GeeCollection;
typedef struct _GeeCollectionIface GeeCollectionIface;

#define GEE_TYPE_LIST (gee_list_get_type ())
#define GEE_LIST(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_LIST, GeeList))
#define GEE_IS_LIST(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_LIST))
#define GEE_LIST_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_LIST, GeeListIface))

typedef struct _GeeList GeeList;
typedef struct _GeeListIface GeeListIface;

#define GEE_TYPE_READ_ONLY_LIST (gee_read_only_list_get_type ())
#define GEE_READ_ONLY_LIST(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_READ_ONLY_LIST, GeeReadOnlyList))
#define GEE_READ_ONLY_LIST_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_TYPE_READ_ONLY_LIST, GeeReadOnlyListClass))
#define GEE_IS_READ_ONLY_LIST(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_READ_ONLY_LIST))
#define GEE_IS_READ_ONLY_LIST_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_TYPE_READ_ONLY_LIST))
#define GEE_READ_ONLY_LIST_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_TYPE_READ_ONLY_LIST, GeeReadOnlyListClass))

typedef struct _GeeReadOnlyList GeeReadOnlyList;
typedef struct _GeeReadOnlyListClass GeeReadOnlyListClass;
typedef struct _GeeReadOnlyListPrivate GeeReadOnlyListPrivate;

#define GEE_READ_ONLY_LIST_TYPE_ITERATOR (gee_read_only_list_iterator_get_type ())
#define GEE_READ_ONLY_LIST_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_READ_ONLY_LIST_TYPE_ITERATOR, GeeReadOnlyListIterator))
#define GEE_READ_ONLY_LIST_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_READ_ONLY_LIST_TYPE_ITERATOR, GeeReadOnlyListIteratorClass))
#define GEE_READ_ONLY_LIST_IS_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_READ_ONLY_LIST_TYPE_ITERATOR))
#define GEE_READ_ONLY_LIST_IS_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_READ_ONLY_LIST_TYPE_ITERATOR))
#define GEE_READ_ONLY_LIST_ITERATOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_READ_ONLY_LIST_TYPE_ITERATOR, GeeReadOnlyListIteratorClass))

typedef struct _GeeReadOnlyListIterator GeeReadOnlyListIterator;
typedef struct _GeeReadOnlyListIteratorClass GeeReadOnlyListIteratorClass;
typedef struct _GeeReadOnlyListIteratorPrivate GeeReadOnlyListIteratorPrivate;

/**
 * Base class for all collections.
 */
struct _GeeCollectionObject {
	GTypeInstance parent_instance;
	volatile int ref_count;
	GeeCollectionObjectPrivate * priv;
};

struct _GeeCollectionObjectClass {
	GTypeClass parent_class;
	void (*finalize) (GeeCollectionObject *self);
};

/**
 * Implemented by classes that support a simple iteration over instances of the
 * collection.
 */
struct _GeeIteratorIface {
	GTypeInterface parent_iface;
	gboolean (*next) (GeeIterator* self);
	gpointer (*get) (GeeIterator* self);
};

/**
 * Implemented by classes that support a simple iteration over instances of the
 * collection.
 */
struct _GeeIterableIface {
	GTypeInterface parent_iface;
	GType (*get_element_type) (GeeIterable* self);
	GeeIterator* (*iterator) (GeeIterable* self);
};

/**
 * Serves as the base interface for implementing collection classes. Defines
 * size, iteration, and modification methods.
 */
struct _GeeCollectionIface {
	GTypeInterface parent_iface;
	gboolean (*contains) (GeeCollection* self, gconstpointer item);
	gboolean (*add) (GeeCollection* self, gconstpointer item);
	gboolean (*remove) (GeeCollection* self, gconstpointer item);
	void (*clear) (GeeCollection* self);
	gint (*get_size) (GeeCollection* self);
};

/**
 * Represents a collection of items in a well-defined order.
 */
struct _GeeListIface {
	GTypeInterface parent_iface;
	gpointer (*get) (GeeList* self, gint index);
	void (*set) (GeeList* self, gint index, gconstpointer item);
	gint (*index_of) (GeeList* self, gconstpointer item);
	void (*insert) (GeeList* self, gint index, gconstpointer item);
	void (*remove_at) (GeeList* self, gint index);
};

/**
 * Represents a read-only collection of items in a well-defined order.
 */
struct _GeeReadOnlyList {
	GeeCollectionObject parent_instance;
	GeeReadOnlyListPrivate * priv;
};

struct _GeeReadOnlyListClass {
	GeeCollectionObjectClass parent_class;
};

struct _GeeReadOnlyListPrivate {
	GType g_type;
	GBoxedCopyFunc g_dup_func;
	GDestroyNotify g_destroy_func;
	GeeList* _list;
};

struct _GeeReadOnlyListIterator {
	GeeCollectionObject parent_instance;
	GeeReadOnlyListIteratorPrivate * priv;
};

struct _GeeReadOnlyListIteratorClass {
	GeeCollectionObjectClass parent_class;
};

struct _GeeReadOnlyListIteratorPrivate {
	GType g_type;
	GBoxedCopyFunc g_dup_func;
	GDestroyNotify g_destroy_func;
};


static gpointer gee_read_only_list_iterator_parent_class = NULL;
static GeeIteratorIface* gee_read_only_list_iterator_gee_iterator_parent_iface = NULL;
static gpointer gee_read_only_list_parent_class = NULL;
static GeeIterableIface* gee_read_only_list_gee_iterable_parent_iface = NULL;
static GeeCollectionIface* gee_read_only_list_gee_collection_parent_iface = NULL;
static GeeListIface* gee_read_only_list_gee_list_parent_iface = NULL;

gpointer gee_collection_object_ref (gpointer instance);
void gee_collection_object_unref (gpointer instance);
GParamSpec* gee_param_spec_collection_object (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void gee_value_set_collection_object (GValue* value, gpointer v_object);
gpointer gee_value_get_collection_object (const GValue* value);
GType gee_collection_object_get_type (void);
GType gee_iterator_get_type (void);
GType gee_iterable_get_type (void);
GType gee_collection_get_type (void);
GType gee_list_get_type (void);
GType gee_read_only_list_get_type (void);
#define GEE_READ_ONLY_LIST_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_TYPE_READ_ONLY_LIST, GeeReadOnlyListPrivate))
enum  {
	GEE_READ_ONLY_LIST_DUMMY_PROPERTY
};
void gee_read_only_list_set_list (GeeReadOnlyList* self, GeeList* value);
GeeReadOnlyList* gee_read_only_list_new (GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GeeList* list);
GeeReadOnlyList* gee_read_only_list_construct (GType object_type, GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GeeList* list);
GeeReadOnlyList* gee_read_only_list_new (GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GeeList* list);
static GType gee_read_only_list_real_get_element_type (GeeIterable* base);
static GeeReadOnlyListIterator* gee_read_only_list_iterator_new (GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func);
static GeeReadOnlyListIterator* gee_read_only_list_iterator_construct (GType object_type, GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func);
static GType gee_read_only_list_iterator_get_type (void);
GeeIterator* gee_iterable_iterator (GeeIterable* self);
static GeeIterator* gee_read_only_list_real_iterator (GeeIterable* base);
gboolean gee_collection_contains (GeeCollection* self, gconstpointer item);
static gboolean gee_read_only_list_real_contains (GeeCollection* base, gconstpointer item);
gint gee_list_index_of (GeeList* self, gconstpointer item);
static gint gee_read_only_list_real_index_of (GeeList* base, gconstpointer item);
static gboolean gee_read_only_list_real_add (GeeCollection* base, gconstpointer item);
static gboolean gee_read_only_list_real_remove (GeeCollection* base, gconstpointer item);
static void gee_read_only_list_real_insert (GeeList* base, gint index, gconstpointer item);
static void gee_read_only_list_real_remove_at (GeeList* base, gint index);
gpointer gee_list_get (GeeList* self, gint index);
static gpointer gee_read_only_list_real_get (GeeList* base, gint index);
static void gee_read_only_list_real_set (GeeList* base, gint index, gconstpointer o);
static void gee_read_only_list_real_clear (GeeCollection* base);
gint gee_collection_get_size (GeeCollection* self);
#define GEE_READ_ONLY_LIST_ITERATOR_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_READ_ONLY_LIST_TYPE_ITERATOR, GeeReadOnlyListIteratorPrivate))
enum  {
	GEE_READ_ONLY_LIST_ITERATOR_DUMMY_PROPERTY
};
static gboolean gee_read_only_list_iterator_real_next (GeeIterator* base);
static gpointer gee_read_only_list_iterator_real_get (GeeIterator* base);
static GeeReadOnlyListIterator* gee_read_only_list_iterator_new (GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func);
static void gee_read_only_list_finalize (GeeCollectionObject* obj);



GeeReadOnlyList* gee_read_only_list_construct (GType object_type, GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GeeList* list) {
	GeeReadOnlyList* self;
	self = (GeeReadOnlyList*) g_type_create_instance (object_type);
	self->priv->g_type = g_type;
	self->priv->g_dup_func = g_dup_func;
	self->priv->g_destroy_func = g_destroy_func;
	gee_read_only_list_set_list (self, list);
	return self;
}


GeeReadOnlyList* gee_read_only_list_new (GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GeeList* list) {
	return gee_read_only_list_construct (GEE_TYPE_READ_ONLY_LIST, g_type, g_dup_func, g_destroy_func, list);
}


static GType gee_read_only_list_real_get_element_type (GeeIterable* base) {
	GeeReadOnlyList * self;
	GType result;
	self = (GeeReadOnlyList*) base;
	result = self->priv->g_type;
	return result;
}


static GeeIterator* gee_read_only_list_real_iterator (GeeIterable* base) {
	GeeReadOnlyList * self;
	GeeIterator* result;
	self = (GeeReadOnlyList*) base;
	if (self->priv->_list == NULL) {
		result = (GeeIterator*) gee_read_only_list_iterator_new (self->priv->g_type, (GBoxedCopyFunc) self->priv->g_dup_func, self->priv->g_destroy_func);
		return result;
	}
	result = gee_iterable_iterator ((GeeIterable*) self->priv->_list);
	return result;
}


static gboolean gee_read_only_list_real_contains (GeeCollection* base, gconstpointer item) {
	GeeReadOnlyList * self;
	gboolean result;
	self = (GeeReadOnlyList*) base;
	if (self->priv->_list == NULL) {
		result = FALSE;
		return result;
	}
	result = gee_collection_contains ((GeeCollection*) self->priv->_list, item);
	return result;
}


static gint gee_read_only_list_real_index_of (GeeList* base, gconstpointer item) {
	GeeReadOnlyList * self;
	gint result;
	self = (GeeReadOnlyList*) base;
	if (self->priv->_list == NULL) {
		result = -1;
		return result;
	}
	result = gee_list_index_of (self->priv->_list, item);
	return result;
}


static gboolean gee_read_only_list_real_add (GeeCollection* base, gconstpointer item) {
	GeeReadOnlyList * self;
	gboolean result;
	self = (GeeReadOnlyList*) base;
	g_assert_not_reached ();
}


static gboolean gee_read_only_list_real_remove (GeeCollection* base, gconstpointer item) {
	GeeReadOnlyList * self;
	gboolean result;
	self = (GeeReadOnlyList*) base;
	g_assert_not_reached ();
}


static void gee_read_only_list_real_insert (GeeList* base, gint index, gconstpointer item) {
	GeeReadOnlyList * self;
	self = (GeeReadOnlyList*) base;
	g_assert_not_reached ();
}


static void gee_read_only_list_real_remove_at (GeeList* base, gint index) {
	GeeReadOnlyList * self;
	self = (GeeReadOnlyList*) base;
	g_assert_not_reached ();
}


static gpointer gee_read_only_list_real_get (GeeList* base, gint index) {
	GeeReadOnlyList * self;
	gpointer result;
	self = (GeeReadOnlyList*) base;
	if (self->priv->_list == NULL) {
		result = NULL;
		return result;
	}
	result = gee_list_get (self->priv->_list, index);
	return result;
}


static void gee_read_only_list_real_set (GeeList* base, gint index, gconstpointer o) {
	GeeReadOnlyList * self;
	self = (GeeReadOnlyList*) base;
	g_assert_not_reached ();
}


static void gee_read_only_list_real_clear (GeeCollection* base) {
	GeeReadOnlyList * self;
	self = (GeeReadOnlyList*) base;
	g_assert_not_reached ();
}


static gint gee_read_only_list_real_get_size (GeeCollection* base) {
	gint result;
	GeeReadOnlyList* self;
	self = (GeeReadOnlyList*) base;
	result = gee_collection_get_size ((GeeCollection*) self->priv->_list);
	return result;
}


void gee_read_only_list_set_list (GeeReadOnlyList* self, GeeList* value) {
	GeeList* _tmp1_;
	GeeList* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_list = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : gee_collection_object_ref (_tmp0_)), (self->priv->_list == NULL) ? NULL : (self->priv->_list = (gee_collection_object_unref (self->priv->_list), NULL)), _tmp1_);
}


static gboolean gee_read_only_list_iterator_real_next (GeeIterator* base) {
	GeeReadOnlyListIterator * self;
	gboolean result;
	self = (GeeReadOnlyListIterator*) base;
	result = FALSE;
	return result;
}


static gpointer gee_read_only_list_iterator_real_get (GeeIterator* base) {
	GeeReadOnlyListIterator * self;
	gpointer result;
	self = (GeeReadOnlyListIterator*) base;
	result = NULL;
	return result;
}


static GeeReadOnlyListIterator* gee_read_only_list_iterator_construct (GType object_type, GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func) {
	GeeReadOnlyListIterator* self;
	self = (GeeReadOnlyListIterator*) g_type_create_instance (object_type);
	self->priv->g_type = g_type;
	self->priv->g_dup_func = g_dup_func;
	self->priv->g_destroy_func = g_destroy_func;
	return self;
}


static GeeReadOnlyListIterator* gee_read_only_list_iterator_new (GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func) {
	return gee_read_only_list_iterator_construct (GEE_READ_ONLY_LIST_TYPE_ITERATOR, g_type, g_dup_func, g_destroy_func);
}


static void gee_read_only_list_iterator_class_init (GeeReadOnlyListIteratorClass * klass) {
	gee_read_only_list_iterator_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (GeeReadOnlyListIteratorPrivate));
}


static void gee_read_only_list_iterator_gee_iterator_interface_init (GeeIteratorIface * iface) {
	gee_read_only_list_iterator_gee_iterator_parent_iface = g_type_interface_peek_parent (iface);
	iface->next = gee_read_only_list_iterator_real_next;
	iface->get = gee_read_only_list_iterator_real_get;
}


static void gee_read_only_list_iterator_instance_init (GeeReadOnlyListIterator * self) {
	self->priv = GEE_READ_ONLY_LIST_ITERATOR_GET_PRIVATE (self);
}


static GType gee_read_only_list_iterator_get_type (void) {
	static GType gee_read_only_list_iterator_type_id = 0;
	if (gee_read_only_list_iterator_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeReadOnlyListIteratorClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_read_only_list_iterator_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeReadOnlyListIterator), 0, (GInstanceInitFunc) gee_read_only_list_iterator_instance_init, NULL };
		static const GInterfaceInfo gee_iterator_info = { (GInterfaceInitFunc) gee_read_only_list_iterator_gee_iterator_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_read_only_list_iterator_type_id = g_type_register_static (GEE_TYPE_COLLECTION_OBJECT, "GeeReadOnlyListIterator", &g_define_type_info, 0);
		g_type_add_interface_static (gee_read_only_list_iterator_type_id, GEE_TYPE_ITERATOR, &gee_iterator_info);
	}
	return gee_read_only_list_iterator_type_id;
}


static void gee_read_only_list_class_init (GeeReadOnlyListClass * klass) {
	gee_read_only_list_parent_class = g_type_class_peek_parent (klass);
	GEE_COLLECTION_OBJECT_CLASS (klass)->finalize = gee_read_only_list_finalize;
	g_type_class_add_private (klass, sizeof (GeeReadOnlyListPrivate));
}


static void gee_read_only_list_gee_iterable_interface_init (GeeIterableIface * iface) {
	gee_read_only_list_gee_iterable_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_element_type = gee_read_only_list_real_get_element_type;
	iface->iterator = gee_read_only_list_real_iterator;
}


static void gee_read_only_list_gee_collection_interface_init (GeeCollectionIface * iface) {
	gee_read_only_list_gee_collection_parent_iface = g_type_interface_peek_parent (iface);
	iface->contains = gee_read_only_list_real_contains;
	iface->add = gee_read_only_list_real_add;
	iface->remove = gee_read_only_list_real_remove;
	iface->clear = gee_read_only_list_real_clear;
	iface->get_size = gee_read_only_list_real_get_size;
}


static void gee_read_only_list_gee_list_interface_init (GeeListIface * iface) {
	gee_read_only_list_gee_list_parent_iface = g_type_interface_peek_parent (iface);
	iface->index_of = gee_read_only_list_real_index_of;
	iface->insert = gee_read_only_list_real_insert;
	iface->remove_at = gee_read_only_list_real_remove_at;
	iface->get = gee_read_only_list_real_get;
	iface->set = gee_read_only_list_real_set;
}


static void gee_read_only_list_instance_init (GeeReadOnlyList * self) {
	self->priv = GEE_READ_ONLY_LIST_GET_PRIVATE (self);
}


static void gee_read_only_list_finalize (GeeCollectionObject* obj) {
	GeeReadOnlyList * self;
	self = GEE_READ_ONLY_LIST (obj);
	(self->priv->_list == NULL) ? NULL : (self->priv->_list = (gee_collection_object_unref (self->priv->_list), NULL));
	GEE_COLLECTION_OBJECT_CLASS (gee_read_only_list_parent_class)->finalize (obj);
}


GType gee_read_only_list_get_type (void) {
	static GType gee_read_only_list_type_id = 0;
	if (gee_read_only_list_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeReadOnlyListClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_read_only_list_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeReadOnlyList), 0, (GInstanceInitFunc) gee_read_only_list_instance_init, NULL };
		static const GInterfaceInfo gee_iterable_info = { (GInterfaceInitFunc) gee_read_only_list_gee_iterable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		static const GInterfaceInfo gee_collection_info = { (GInterfaceInitFunc) gee_read_only_list_gee_collection_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		static const GInterfaceInfo gee_list_info = { (GInterfaceInitFunc) gee_read_only_list_gee_list_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_read_only_list_type_id = g_type_register_static (GEE_TYPE_COLLECTION_OBJECT, "GeeReadOnlyList", &g_define_type_info, 0);
		g_type_add_interface_static (gee_read_only_list_type_id, GEE_TYPE_ITERABLE, &gee_iterable_info);
		g_type_add_interface_static (gee_read_only_list_type_id, GEE_TYPE_COLLECTION, &gee_collection_info);
		g_type_add_interface_static (gee_read_only_list_type_id, GEE_TYPE_LIST, &gee_list_info);
	}
	return gee_read_only_list_type_id;
}




