/* -*- mode: c; style: linux; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* location-list.c
 * Copyright (C) 2000-2001 Ximian, Inc.
 *
 * Written by Bradford Hovinen <hovinen@ximian.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "location-list.h"
#include "util.h"

typedef struct _pair_t pair_t;

struct _pair_t 
{
	gpointer a, b;
};

enum {
	ARG_0,
	ARG_ARCHIVE,
	ARG_LOCATION_TREE,
	ARG_LOCATION_MENU
};

enum {
	LOCATION_CHANGED_SIGNAL,
	LAST_SIGNAL
};

struct _LocationListPrivate 
{
	gchar         *selected_location_id;
	Location      *selected_location;

	Archive       *archive;

	GtkCTree      *location_tree;
	GtkMenu       *location_menu;
	GtkOptionMenu *location_option_menu;
};

static GtkObjectClass *parent_class;

static int location_num, selected_location_num;

static gint location_list_signals[LAST_SIGNAL] = { 0 };

static void location_list_init        (LocationList *location_list);
static void location_list_class_init  (LocationListClass *class);

static void location_list_set_arg     (GtkObject *object, 
				       GtkArg *arg, 
				       guint arg_id);
static void location_list_get_arg     (GtkObject *object, 
				       GtkArg *arg, 
				       guint arg_id);
static void location_list_finalize    (GtkObject *object);

static void tree_select_row_cb        (GtkCTree *tree,
				       GList *node, gint column,
				       LocationList *list);
static void menu_item_select_cb       (GtkMenuItem *item,
				       LocationList *list);

static gint populate_tree_cb          (Archive *archive,
				       Location *location,
				       pair_t *data);
static void populate_tree             (LocationList *list);

static gint populate_menu_cb          (Archive *archive,
				       Location *location,
				       LocationList *list);
static void populate_menu             (LocationList *list);

guint
location_list_get_type (void)
{
	static guint location_list_type = 0;

	if (!location_list_type) {
		GtkTypeInfo location_list_info = {
			"LocationList",
			sizeof (LocationList),
			sizeof (LocationListClass),
			(GtkClassInitFunc) location_list_class_init,
			(GtkObjectInitFunc) location_list_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};

		location_list_type = 
			gtk_type_unique (gtk_object_get_type (), 
					 &location_list_info);
	}

	return location_list_type;
}

static void
location_list_init (LocationList *location_list)
{
	location_list->p = g_new0 (LocationListPrivate, 1);
}

static void
location_list_class_init (LocationListClass *class) 
{
	GtkObjectClass *object_class;

	gtk_object_add_arg_type ("LocationList::archive",
				 GTK_TYPE_POINTER,
				 GTK_ARG_READWRITE,
				 ARG_ARCHIVE);

	gtk_object_add_arg_type ("LocationList::location-tree",
				 GTK_TYPE_POINTER,
				 GTK_ARG_READWRITE | GTK_ARG_CONSTRUCT_ONLY,
				 ARG_LOCATION_TREE);

	gtk_object_add_arg_type ("LocationList::location-menu",
				 GTK_TYPE_POINTER,
				 GTK_ARG_READWRITE | GTK_ARG_CONSTRUCT_ONLY,
				 ARG_LOCATION_MENU);

	object_class = GTK_OBJECT_CLASS (class);
	object_class->finalize = location_list_finalize;
	object_class->set_arg = location_list_set_arg;
	object_class->get_arg = location_list_get_arg;

	location_list_signals[LOCATION_CHANGED_SIGNAL] =
		gtk_signal_new ("location-changed", GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (LocationListClass,
						   location_changed),
				gtk_marshal_NONE__POINTER,
				GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);

	gtk_object_class_add_signals (object_class, location_list_signals,
				      LAST_SIGNAL);

	parent_class = GTK_OBJECT_CLASS
		(gtk_type_class (gtk_object_get_type ()));
}

static void
location_list_set_arg (GtkObject *object, GtkArg *arg, guint arg_id) 
{
	LocationList *location_list;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_LOCATION_LIST (object));

	location_list = LOCATION_LIST (object);

	switch (arg_id) {
	case ARG_ARCHIVE:
		g_return_if_fail (GTK_VALUE_POINTER (*arg) == NULL || 
				  IS_ARCHIVE (GTK_VALUE_POINTER (*arg)));

		if (GTK_VALUE_POINTER (*arg) == NULL) return;

		location_list->p->archive =
			ARCHIVE (GTK_VALUE_POINTER (*arg));

		gtk_object_ref (GTK_OBJECT (location_list->p->archive));

		location_list->p->selected_location =
			archive_get_current_location
			(location_list->p->archive);
		location_list->p->selected_location_id =
			location_get_id (location_list->p->selected_location);

		if (location_list->p->location_tree != NULL)
			populate_tree (location_list);

		if (location_list->p->location_option_menu != NULL)
			populate_menu (location_list);

		break;

	case ARG_LOCATION_TREE:
		g_return_if_fail (GTK_VALUE_POINTER (*arg) == NULL ||
				  GTK_IS_CTREE (GTK_VALUE_POINTER (*arg)));

		location_list->p->location_tree = GTK_VALUE_POINTER (*arg);

		if (location_list->p->location_tree != NULL) {
			gtk_signal_connect (GTK_OBJECT
					    (location_list->p->location_tree),
					    "tree-select-row",
					    GTK_SIGNAL_FUNC
					    (tree_select_row_cb),
					    location_list);

			if (location_list->p->archive != NULL)
				populate_tree (location_list);
		}

		break;

	case ARG_LOCATION_MENU:
		g_return_if_fail (GTK_VALUE_POINTER (*arg) == NULL ||
				  GTK_IS_OPTION_MENU
				  (GTK_VALUE_POINTER (*arg)));

		location_list->p->location_option_menu =
			GTK_VALUE_POINTER (*arg);

		if (location_list->p->location_option_menu != NULL &&
		    location_list->p->archive != NULL)
			populate_menu (location_list);

		break;

	default:
		g_warning ("Bad argument set");
		break;
	}
}

static void
location_list_get_arg (GtkObject *object, GtkArg *arg, guint arg_id) 
{
	LocationList *location_list;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_LOCATION_LIST (object));

	location_list = LOCATION_LIST (object);

	switch (arg_id) {
	case ARG_ARCHIVE:
		GTK_VALUE_POINTER (*arg) = location_list->p->archive;
		break;

	case ARG_LOCATION_TREE:
		GTK_VALUE_POINTER (*arg) = location_list->p->location_tree;
		break;

	case ARG_LOCATION_MENU:
		GTK_VALUE_POINTER (*arg) = 
			location_list->p->location_option_menu;
		break;

	default:
		g_warning ("Bad argument get");
		break;
	}
}

static void
location_list_finalize (GtkObject *object) 
{
	LocationList *location_list;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_LOCATION_LIST (object));

	location_list = LOCATION_LIST (object);

	if (location_list->p->archive != NULL)
		gtk_object_unref
			(GTK_OBJECT (location_list->p->archive));

	g_free (location_list->p);

	GTK_OBJECT_CLASS (parent_class)->finalize (object);
}

GtkObject *
location_list_new (Archive *archive, GtkWidget *widget) 
{
	GtkOptionMenu *menu = NULL;
	GtkCTree *ctree = NULL;

	if (GTK_IS_OPTION_MENU (widget)) menu = GTK_OPTION_MENU (widget);
	else if (GTK_IS_CTREE (widget)) ctree = GTK_CTREE (widget);

	return gtk_object_new (location_list_get_type (),
			       "archive", archive,
			       "location-tree", ctree,
			       "location-menu", menu,
			       NULL);
}

gchar *
location_list_get_selected_location_id (LocationList *list)
{
	g_return_val_if_fail (list != NULL, NULL);
	g_return_val_if_fail (IS_LOCATION_LIST (list), NULL);

	return list->p->selected_location_id;
}

Location *
location_list_get_selected_location (LocationList *list)
{
	g_return_val_if_fail (list != NULL, NULL);
	g_return_val_if_fail (IS_LOCATION_LIST (list), NULL);

	return list->p->selected_location;
}

void
location_list_select (LocationList *list, Location *location) 
{
	gchar *current_id, *locid;
	GList *node;
	GtkObject *current = NULL;

	g_return_if_fail (list != NULL);
	g_return_if_fail (IS_LOCATION_LIST (list));

	list->p->selected_location = location;

	if (list->p->location_menu != NULL) {
		node = GTK_MENU_SHELL (list->p->location_menu)->children;
		locid = location_get_id (location);

		while (node != NULL) {
			current = node->data;
			g_assert (GTK_IS_MENU_ITEM (current));
			current_id = location_get_id
				(gtk_object_get_data (current, "location"));
			if (!strcmp (current_id, locid))
				break;
			else
				current = NULL;

			node = node->next;
		}

		if (current != NULL)
			gtk_menu_item_activate (GTK_MENU_ITEM (current));
	}
}

void
location_list_reread (LocationList *list)
{
	DEBUG_MSG ("Enter");

	g_return_if_fail (list != NULL);
	g_return_if_fail (IS_LOCATION_LIST (list));

	if (list->p->location_tree != NULL) {
		gtk_clist_freeze (GTK_CLIST (list));
		gtk_clist_clear (GTK_CLIST (list));
		populate_tree (list);
		gtk_clist_thaw (GTK_CLIST (list));
	}

	if (list->p->location_option_menu != NULL) {
		populate_menu (list);
	}
}

static void
tree_select_row_cb (GtkCTree *tree, GList *node, gint column,
		    LocationList *list) 
{
	GtkCTreeRow *row;

	g_return_if_fail (list != NULL);
	g_return_if_fail (IS_LOCATION_LIST (list));

	row = GTK_CTREE_ROW (node);
	list->p->selected_location = row->row.data;
	list->p->selected_location_id =
		GTK_CELL_PIXTEXT (row->row.cell[0])->text;

	gtk_signal_emit (GTK_OBJECT (list),
			 location_list_signals[LOCATION_CHANGED_SIGNAL],
			 row->row.data);
}

static void
menu_item_select_cb (GtkMenuItem *item, LocationList *list) 
{
	Location *location;

	g_return_if_fail (list != NULL);
	g_return_if_fail (IS_LOCATION_LIST (list));

	location = gtk_object_get_data (GTK_OBJECT (item), "location");

	g_return_if_fail (location != NULL);
	g_return_if_fail (IS_LOCATION (location));

	list->p->selected_location = location;
	list->p->selected_location_id = location_get_id (location);

	gtk_signal_emit (GTK_OBJECT (list),
			 location_list_signals[LOCATION_CHANGED_SIGNAL],
			 location);
}

static gint
populate_tree_cb (Archive *archive, Location *location, pair_t *data)
{
	pair_t new_pair;
	char *label;

	label = g_strdup (location_get_label (location));

	new_pair.b = gtk_ctree_insert_node (GTK_CTREE (data->a), 
					    (GtkCTreeNode *) data->b, NULL,
					    &label, GNOME_PAD_SMALL, NULL,
					    NULL, NULL, NULL, FALSE, TRUE);
	gtk_ctree_node_set_row_data
		(GTK_CTREE (data->a), (GtkCTreeNode *) new_pair.b, location);

	new_pair.a = data->a;

	archive_foreach_child_location
		(archive, (LocationCB) populate_tree_cb, location, &new_pair);

	return 0;
}

static void
populate_tree (LocationList *list) 
{
	pair_t pair;

	pair.a = list->p->location_tree;
	pair.b = NULL;

	archive_foreach_child_location (list->p->archive,
					(LocationCB) populate_tree_cb,
					NULL, &pair);
}

static gint
populate_menu_cb (Archive *archive, Location *location, LocationList *list) 
{
	GtkWidget *item;

	item = gtk_menu_item_new_with_label (location_get_label (location));
	gtk_object_set_data (GTK_OBJECT (item), "location", location);
	gtk_signal_connect (GTK_OBJECT (item), "activate",
			    GTK_SIGNAL_FUNC (menu_item_select_cb),
			    list);
	gtk_widget_show (item);

	gtk_menu_append (list->p->location_menu, item);

	if (location == list->p->selected_location)
		selected_location_num = location_num;
	else
		location_num++;

	archive_foreach_child_location (list->p->archive,
					(LocationCB) populate_menu_cb,
					location, list);

	return 0;
}

static void
populate_menu (LocationList *list) 
{
	location_num = 0;

	list->p->location_menu = GTK_MENU (gtk_menu_new ());

	archive_foreach_child_location (list->p->archive,
					(LocationCB) populate_menu_cb,
					NULL, list);

	gtk_option_menu_set_menu (list->p->location_option_menu,
				  GTK_WIDGET (list->p->location_menu));
	gtk_option_menu_set_history (list->p->location_option_menu,
				     selected_location_num);
}
