/* This file is part of the KDE project

   Copyright (C) 2001 Lukas Tinkl <lukas@kde.org>
                      Andreas Schlapbach <schlpbch@iam.unibe.ch>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License version 2 as published by the Free Software Foundation.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include "imgalleryplugin.h"

#include <qdir.h>
#include <qtextstream.h>
#include <qfile.h>
#include <qfont.h>
#include <qdatetime.h>
#include <qpixmap.h>
#include <qimage.h>
#include <qprogressdialog.h>
#include <qvbox.h> 
#include <qgroupbox.h>
#include <qlayout.h> 
  
#include <kaction.h>
#include <kinstance.h>
#include <kglobal.h>
#include <klocale.h>
#include <kcharsets.h>
#include <kmessagebox.h>
#include <kfiledialog.h>
#include <kurl.h>
#include <kapp.h>
#include <kdebug.h>


KIGPDialog::KIGPDialog(QWidget *parent, const QString& path, const char *name )   
  : KDialogBase( parent, name, true, i18n("Configure"), Ok|Cancel, Ok, true ),
    m_Path(path), gen_gallery( false )
{
  setCaption(i18n("Create Image Gallery"));
  setMinimumWidth(300);

  QVBox *page = makeVBoxMainWidget ();

  ///

  QGroupBox *tgroup = new QGroupBox( i18n("Page &Look"), page ); 
  QVBoxLayout *tvlay = new QVBoxLayout( tgroup, spacingHint() );   
  tvlay->addSpacing( fontMetrics().lineSpacing());     

  QLabel* pt_label = new QLabel( i18n("Page Title:"), tgroup);
  tvlay->addWidget(pt_label);

  title_lineedit = new QLineEdit(i18n("Image Gallery for %1").arg(m_Path), tgroup);
  tvlay->addWidget(title_lineedit);

  QHBoxLayout *hlay = new QHBoxLayout( spacingHint() );  
  tvlay->addLayout( hlay );   
  QLabel* fg_label = new QLabel( i18n("Foreground color:"), tgroup);

  fg_colorbutton = new KColorButton(tgroup);
  fg_colorbutton->setColor(QColor("#d0ffd0"));

  hlay->addWidget(fg_label);  
  hlay->addStretch( 100 ); 
  hlay->addWidget(fg_colorbutton);
 
  QHBoxLayout *hlay2 = new QHBoxLayout( spacingHint() );  
  tvlay->addLayout( hlay2 );   
  QLabel * bg_label = new QLabel( i18n("Background color:"), tgroup);

  bg_colorbutton = new KColorButton(tgroup);
  bg_colorbutton->setColor(QColor(33,33,33));

  hlay2->addWidget(bg_label);  
  hlay2->addStretch( 100 ); 
  hlay2->addWidget(bg_colorbutton);

  /// 
  
  QGroupBox *dgroup= new QGroupBox( i18n("&Directories"), page ); 
  QVBoxLayout *dvlay = new QVBoxLayout( dgroup, spacingHint() );  
  dvlay->addSpacing( fontMetrics().lineSpacing());

  QLabel * saveloc_label = new QLabel(i18n("Save to:"), dgroup);
  dvlay->addWidget(saveloc_label);

  urlreq = new KURLRequester(m_Path + "images.html", dgroup);
  dvlay->addWidget(urlreq);
  
  cp_local = new QCheckBox(i18n("Copy &Original Files"), dgroup);
  cp_local->setChecked(false);
  dvlay->addWidget(cp_local);
}

const QString KIGPDialog::getURL()
{
  return urlreq->url();
}

QString KIGPDialog::getTitle()
{
  return title_lineedit->text();
}

void KIGPDialog::slotOk()
{
  gen_gallery = true;
  this->accept();
}

void KIGPDialog::slotCancel()
{
  gen_gallery = false;
  this->done(Cancel);
}

bool KIGPDialog::getDialogOk()
{
  return gen_gallery;
}

bool KIGPDialog::getCopyFiles()
{
  return cp_local->isChecked();
}

const QColor KIGPDialog::getBackgroundColor()
{
  return bg_colorbutton->color();
}

const QColor KIGPDialog::getForegroundColor()
{
  return fg_colorbutton->color();
}

KIGPDialog::~KIGPDialog()
{
}


///////////////////////////////////////////////////////////////////////////


KImGalleryPlugin::KImGalleryPlugin( QObject* parent, const char* name )
  : KParts::Plugin( parent, name )
{
  new KAction( i18n( "&Create Image Gallery" ), "imagegallery", CTRL+Key_I, this,
	       SLOT( slotExecute() ), actionCollection(), "create_img_gallery" );
}

void KImGalleryPlugin::slotExecute()
{
  KonqDirPart * part = dynamic_cast<KonqDirPart *>(parent());
  if ( !part )
    {
      KMessageBox::sorry( 0L, i18n("KImGalleryPlugin::slotCreateHtml: Program error, please report a bug."));
      return;
    }
  if (!part->url().isLocalFile())	//TODO support remote URLs too?
    {
      KMessageBox::sorry( part->widget(), i18n("Creating an image gallery works only on local directories."));
      return;
    }

  m_configDlg = new KIGPDialog(part->widget(), part->url().path(+1));
  m_configDlg->show();

  if (m_configDlg->getDialogOk())
    {
      m_copyFiles = m_configDlg->getCopyFiles();
      m_backColorName = m_configDlg->getBackgroundColor().name();
      m_foreColorName = m_configDlg->getForegroundColor().name();
      m_textTitle = m_configDlg->getTitle();
      
      KURL url(m_configDlg->getURL());
      if ( !url.isEmpty() && url.isValid())
	{
	  m_progressDlg = new QProgressDialog( part->widget(), "progressDlg", true );
	  QObject::connect(m_progressDlg, SIGNAL( cancelled() ),
			   this, SLOT( slotCancelled() ) );

	  m_progressDlg->setLabelText( i18n("Creating thumbnails") );
	  m_progressDlg->setCancelButtonText(i18n("&Cancel"));
	  m_cancelled = false;
	  m_progressDlg->show();
	  if ( createHtml( url ) ){
	    // Open a browser to show the result
	    kapp->invokeBrowser(url.url());
	  }
	  delete m_progressDlg;
	}
    }
}

bool KImGalleryPlugin::createHtml(const KURL& url)
{
  KonqDirPart * part = dynamic_cast<KonqDirPart *>(parent());
  const QString sourceDir = part->url().path(+1);
  kdDebug() << "sourceDir: " << sourceDir << endl;
  QDir img_dir( sourceDir, "*.png *.PNG *.gif *.GIF *.jpg *.JPG *.jpeg *.JPEG",
		QDir::Name|QDir::IgnoreCase, QDir::Files|QDir::Readable);

  const QString imgGalleryDir = url.directory();
  kdDebug() << "imgGalleryDir: " << imgGalleryDir << endl;
  QDir thumb_dir( imgGalleryDir + QString::fromLatin1("/thumbs/"));

  if (!thumb_dir.exists()) {
    thumb_dir.setPath( imgGalleryDir);
    if (!(thumb_dir.mkdir("thumbs", false))) {
      KMessageBox::sorry(part->widget(), i18n("Couldn't create directory: %1").arg(thumb_dir.path()));
      return false;
    } else {
      thumb_dir.setPath( imgGalleryDir + QString::fromLatin1("/thumbs/") );
    }
  }

  QDir images_dir( imgGalleryDir + QString::fromLatin1("/images/"));

  // Create the "images" subdirectory if necessary
  if (m_copyFiles) {
    if (!images_dir.exists()) {
      images_dir.setPath( imgGalleryDir);
      if (!(images_dir.mkdir("images", false))) {
        KMessageBox::sorry(part->widget(), i18n("Couldn't create directory: %1").arg(images_dir.path()));
        return false;
      } else {
        images_dir.setPath( imgGalleryDir + QString::fromLatin1("/images/") );
      }
    }
  }

  QFile file( url.path() );
  kdDebug() << "url.path(): " << url.path()  << ", thumb_dir: "<< thumb_dir.path()
	    << ", img_dir: "<< img_dir.path() << endl;

  if ( img_dir.exists() && file.open(IO_WriteOnly) ) {
    QTextStream stream(&file);
    stream.setEncoding(QTextStream::Locale);

    const QString today(KGlobal::locale()->formatDate(QDate::currentDate()));

    const QFont::CharSet chset(KGlobal::charsets()->charsetForLocale());
    const QString chsetName(KGlobal::charsets()->name(chset));

    //header
    stream << "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\">\n";
    stream << "<html><head>\n<title>";
    stream << m_textTitle;
    stream << "</title>\n";
    //encoding meta header
    stream << "<meta http-equiv=\"content-type\" content=\"text/html; charset="
	   << chsetName << "\">\n";

    //adding a touch of style
    stream << "<style type=\"text/css\">\n";
    stream << "<!--BODY { color: " << m_foreColorName << "; background: " 
	   << m_backColorName << "; margin: 8%;}\n";
    stream << "H1 {color: " << m_foreColorName << ";}\n";
    stream << "TABLE, TBODY, TD, TR { color: " << m_foreColorName << "; padding: 1em}\n";
    stream << "--></style>\n";
    stream << "</head>\n";

    //body
    stream << "<body>\n<h1>" << m_textTitle << "</h1>\n";
    stream << i18n("<i>Number of images</i>: %1").arg((int)img_dir.count()) << "<br>\n";
    stream << i18n("<i>Created on</i>: %1").arg(today) << "<br>\n";
    stream << "<hr>\n<table>\n";

    //table with images
    uint numOfImages = img_dir.count();
    m_progressDlg->setTotalSteps( numOfImages );

    uint imgIndex;
    for (imgIndex = 0; !m_cancelled && (imgIndex < numOfImages);) {
      stream << "<tr>\n";
      for (uint col=0; !m_cancelled && (col < 5) && (imgIndex < numOfImages); col++) {
	const QString imgName = img_dir[imgIndex];

        if (m_copyFiles) {
          stream << "<td align='center'><a href=\"images/" << imgName << "\">";
        } else {
 			 KURL imgUrl(sourceDir);
          stream << "<td align='center'><a href=\"" << imgUrl.path(1) << imgName << "\">";
        }

	if (createThumb(imgName, sourceDir, imgGalleryDir)) {
	  const QString imgNamePNG = imgName.left(imgName.findRev('.',-1));
	  const QString imgPath("thumbs/" + imgNamePNG + ".png");

	  stream << "<img src=\"" << imgPath << "\" width=\"" << m_imgWidth << "\" ";
	  stream << "height=\"" << m_imgHeight << "\" alt=\"" << imgPath << "\">\n";
	  stream << "</a><br>" << imgName << "</td>\n";

	  m_progressDlg->setLabelText( i18n("Created thumbnail for: \n%1").arg(imgName) );
	} else {
	  m_progressDlg->setLabelText( i18n("Creating thumbnail for: \n%1\n failed").arg(imgName) );
	}
	m_progressDlg->setProgress( imgIndex );
	kapp->processEvents();
	imgIndex++;
      }
      stream << "</tr>\n";
    }
    //close the HTML
    stream << "</table>\n</body>\n</html>\n";
    file.close();

    if (m_cancelled) {
      // Remove the image file ..
      file.remove();
      // ..all the thumbnails ..
      for (uint i=0; i < imgIndex; i++) {
	const QString imgName = img_dir[i];
	const QString imgNamePNG = imgName.left(imgName.findRev('.',-1)) + ".png";
	bool isRemoved = thumb_dir.remove(imgNamePNG);
	kdDebug() << "removing: " << thumb_dir.path() << "/" << imgNamePNG << "; "<< isRemoved << endl;
      }
      // ..and the thumb directory
      thumb_dir.rmdir(thumb_dir.path());

      // ..and the images directory if images were to be copied
      if (m_copyFiles) {
	for (uint i=0; i < imgIndex; i++) {
	  const QString imgName = img_dir[i];
	  bool isRemoved = images_dir.remove(imgName);
	  kdDebug() << "removing: " << images_dir.path() << "/" << imgName << "; "<< isRemoved << endl;
	}
        images_dir.rmdir(images_dir.path());
      }
      return false;
    } else {
      return true;
    }
  } else {
    KMessageBox::sorry(part->widget(),i18n("Couldn't open file: %1").arg(url.path(+1)));
    return false;
  }
}

bool KImGalleryPlugin::createThumb( const QString& imgName, const QString& sourceDir,
				    const QString& imgGalleryDir)
{
  QPixmap pix;
  const QString pixPath = sourceDir + QString::fromLatin1("/") + imgName;

  if (m_copyFiles) {
    KURL srcURL(pixPath);
    KURL destURL(imgGalleryDir + QString::fromLatin1("/images/") + imgName);
    KIO::FileCopyJob *file_copy = new KIO::FileCopyJob(srcURL,destURL,420,false,true,false,false);

    delete file_copy;
  }

  // We use PNG for the thumbnails, so we can store GIF images too
  const QString imgNamePNG = imgName.left( imgName.findRev('.',-1) ) + ".png";
  const QString thumbDir = imgGalleryDir + QString::fromLatin1("/thumbs/");
  int extent = 140;

  //this code stolen from kdebase/kioslave/thumbnail/imagecreator.cpp
  // (c) 2000 gis and malte

  m_imgWidth = 120; // Setting the size of the images is
  m_imgHeight = 90; // required to generate faster 'loading' pages
  if ( pix.load( pixPath ) )
    {
      int w = pix.width(), h = pix.height();
      // scale to pixie size
      // kdDebug() << "w: " << w << " h: " << h << endl;
      // Resizing if to big
      if(w > extent || h > extent)
        {
	  if(w > h)
            {
	      h = (int)( (double)( h * extent ) / w );
	      if ( h == 0 ) h = 1;
	      w = extent;
	      ASSERT( h <= extent );
            }
	  else
            {
	      w = (int)( (double)( w * extent ) / h );
	      if ( w == 0 ) w = 1;
	      h = extent;
	      ASSERT( w <= extent );
            }
	  const QImage img(pix.convertToImage().smoothScale( w, h ));
	  if ( img.width() != w || img.height() != h )
            {
	      kdDebug() << "Resizing failed. Aborting." << endl;
	      return false;
            }
	  pix.convertFromImage( img );
	}
      kdDebug() << "Saving thumbnail to: " << thumbDir + imgNamePNG  << endl;
      if (!pix.save(thumbDir + imgNamePNG, "PNG"))
	{
	  kdDebug() << "Saving failed. Aborting." << endl;
	  return false;
	}
      m_imgWidth = w;
      m_imgHeight = h;
      return true;
    }
  return false;
}

void KImGalleryPlugin::slotCancelled()
{
  // kdDebug() << "slotCancelled" << endl;
  m_cancelled = true;
}


KImGalleryPluginFactory::KImGalleryPluginFactory( QObject* parent, const char* name )
  : KLibFactory( parent, name )
{
  s_instance = new KInstance("imgalleryplugin");
}

KImGalleryPluginFactory::~KImGalleryPluginFactory()
{
  delete s_instance;
  s_instance = 0L;
}

QObject* KImGalleryPluginFactory::createObject( QObject* parent, const char* name, const char*, const QStringList & )
{
  return new KImGalleryPlugin( parent, name );
}

extern "C"
{
  void* init_libkimgallery()
  {
    KGlobal::locale()->insertCatalogue("imgalleryplugin");
    return new KImGalleryPluginFactory;
  }

}

KInstance* KImGalleryPluginFactory::s_instance = 0L;

#include "imgalleryplugin.moc"

