/* This file is part of KCachegrind.
   Copyright (C) 2002, 2003 Josef Weidendorfer <Josef.Weidendorfer@gmx.de>

   KCachegrind is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation, version 2.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

/*
 * QCachegrind top level window
 */

#define TRACE_UPDATES 0

#include "toplevel.h"

#include <stdlib.h> // for system()

#include <QDebug>
#include <QDockWidget>
#include <QTimer>
#include <QByteArray>
#include <QLabel>
#include <QStatusBar>
#include <QMenuBar>
#include <QLineEdit>
#include <QTextStream>
#include <QSizePolicy>
#include <QProgressBar>
#include <QFile>
#include <QFileDialog>
#include <QEventLoop>
#include <QToolBar>
#include <QComboBox>
#include <QMessageBox>
#include <QtDBus/QDBusConnection>

#include "partselection.h"
#include "functionselection.h"
#include "stackselection.h"
#include "stackbrowser.h"
#include "tracedata.h"
#include "config.h"
#include "globalconfig.h"
//#include "configdlg.h"
#include "multiview.h"
#include "callgraphview.h"

TopLevel::TopLevel()
{
#ifndef QT_NO_DBUS
    QDBusConnection con = QDBusConnection::sessionBus();
    con.registerObject("/QCachegrind", this,
		       QDBusConnection::ExportScriptableSlots);
#endif

    _progressBar = 0;
    _statusbar = statusBar();
    _statusLabel = new QLabel(_statusbar);
    _statusbar->addWidget(_statusLabel, 1);

    _layoutCount = 1;
    _layoutCurrent = 0;

    resetState();

    GlobalConfig::config()->readOptions();

    createActions();
    createDocks();
    createMenu();
    createToolbar();

    _multiView = new MultiView(this, this);
    _multiView->setObjectName("MultiView");
    setCentralWidget(_multiView);

    // restore current state settings (not configuration options)
    restoreCurrentState(QString::null);

    // restore docks & toolbars from config
    QByteArray state, geometry;
    ConfigGroup* topConfig = ConfigStorage::group("TopWindow");
    _forcePartDock = topConfig->value("ForcePartDockVisible", false).toBool();
    state = topConfig->value("State", QByteArray()).toByteArray();
    geometry = topConfig->value("Geometry", QByteArray()).toByteArray();
    delete topConfig;

    if (!geometry.isEmpty())
	restoreGeometry(geometry);
    if (!state.isEmpty())
	restoreState(state);

    setWindowIcon(QIcon(":/app.png"));
}

TopLevel::~TopLevel()
{
    delete _data;
}

// reset the visualization state, e.g. before loading new data
void TopLevel::resetState()
{
    _activeParts.clear();
    _hiddenParts.clear();

    _data = 0;
    _function = 0;
    _eventType = 0;
    _eventType2 = 0;
    _groupType = ProfileContext::InvalidType;
    _group = 0;

    // for delayed slots
    _traceItemDelayed = 0;
    _eventTypeDelayed = 0;
    _eventType2Delayed = 0;
    _groupTypeDelayed = ProfileContext::InvalidType;
    _groupDelayed = 0;
    _directionDelayed = TraceItemView::None;
    _lastSender = 0;
}


/**
 * This saves the current state of the main window and
 * sub widgets.
 *
 * No positions are saved. These is done automatically for
 * KToolbar, and manually in queryExit() for QT docks.
 */
void TopLevel::saveCurrentState(const QString& postfix)
{
    QString eventType, eventType2;
    if (_eventType) eventType = _eventType->name();
    if (_eventType2) eventType2 = _eventType2->name();

    ConfigGroup* stateConfig = ConfigStorage::group(QString("CurrentState") + postfix);
    stateConfig->setValue("EventType", eventType);
    stateConfig->setValue("EventType2", eventType2);
    stateConfig->setValue("GroupType", ProfileContext::typeName(_groupType));
    delete stateConfig;

    _partSelection->saveOptions(QString("PartOverview"), postfix);
    _multiView->saveLayout(QString("MainView"), postfix);
    _multiView->saveOptions(QString("MainView"), postfix);
}

/**
 * This function is called when a trace is closed.
 * Save browsing position for later restoring
 */
void TopLevel::saveTraceSettings()
{
    QString key = traceKey();

    ConfigGroup* lConfig = ConfigStorage::group("Layouts");
    lConfig->setValue(QString("Count%1").arg(key), _layoutCount);
    lConfig->setValue(QString("Current%1").arg(key), _layoutCurrent);
    delete lConfig;

    ConfigGroup* pConfig = ConfigStorage::group("TracePositions");
    if (_eventType)
	pConfig->setValue(QString("EventType%1").arg(key), _eventType->name());
    if (_eventType2)
	pConfig->setValue(QString("EventType2%1").arg(key), _eventType2->name());
    if (_groupType != ProfileContext::InvalidType)
	pConfig->setValue(QString("GroupType%1").arg(key),
			  ProfileContext::typeName(_groupType));

    if (_data) {
	if (_group)
	    pConfig->setValue(QString("Group%1").arg(key), _group->name());
	saveCurrentState(key);
    }
    delete pConfig;
}

/**
 * This restores the current visualization state of the main window and
 * of the profile views.
 */
void TopLevel::restoreCurrentState(const QString& postfix)
{
    _partSelection->restoreOptions(QString("PartOverview"), postfix);
    _multiView->restoreLayout(QString("MainView"), postfix);
    _multiView->restoreOptions(QString("MainView"), postfix);

    _splittedToggleAction->setChecked(_multiView->childCount()>1);
    _splitDirectionToggleAction->setEnabled(_multiView->childCount()>1);
    _splitDirectionToggleAction->setChecked(_multiView->orientation() ==
					    Qt::Horizontal);
}

void TopLevel::sidebarMenuAboutToShow()
{
    QAction* action;
    QMenu *popup = _sidebarMenuAction->menu();

    popup->clear();

    action = popup->addAction(tr("Parts Overview"));
    action->setCheckable(true);
    action->setChecked(_partDock->isVisible());
    connect(action, SIGNAL(triggered(bool)), this, SLOT(togglePartDock()));

    action = popup->addAction(tr("Top Cost Call Stack"));
    action->setCheckable(true);
    action->setChecked(_stackDock->isVisible());
    connect(action, SIGNAL(triggered(bool)), this, SLOT(toggleStackDock()));

    action = popup->addAction(tr("Flat Profile"));
    action->setCheckable(true);
    action->setChecked(_functionDock->isVisible());
    connect(action, SIGNAL(triggered(bool)), this, SLOT(toggleFunctionDock()));
}

void TopLevel::recentFilesMenuAboutToShow()
{
    QAction* action;
    QStringList recentFiles;
    QMenu *popup = _recentFilesMenuAction->menu();

    popup->clear();

    ConfigGroup* generalConfig = ConfigStorage::group("GeneralSettings");
    recentFiles = generalConfig->value("RecentFiles",
				       QStringList()).toStringList();
    delete generalConfig;

    if (recentFiles.count() == 0)
	popup->addAction(tr("(No recent files)"));
    else {
	QString file;
	foreach(file, recentFiles) {
	    // paths shown to user should use OS-native separators
	    action = popup->addAction(QDir::toNativeSeparators(file));
	}
    }
}

void TopLevel::recentFilesTriggered(QAction* action)
{
    if (action)
	loadTrace(QDir::fromNativeSeparators(action->text()));
}

void TopLevel::createDocks()
{
    // part visualization/selection side bar
    _partDock = new QDockWidget(this);
    _partDock->setObjectName("part-dock");
    _partDock->setWindowTitle(tr("Parts Overview"));
    _partSelection = new PartSelection(this, _partDock);
    _partDock->setWidget(_partSelection);

    connect(_partSelection, SIGNAL(partsHideSelected()),
	    this, SLOT(partsHideSelectedSlotDelayed()));
    connect(_partSelection, SIGNAL(partsUnhideAll()),
	    this, SLOT(partsUnhideAllSlotDelayed()));

    // stack selection side bar
    _stackDock = new QDockWidget(this);
    _stackDock->setObjectName("stack-dock");
    _stackSelection = new StackSelection(_stackDock);
    _stackDock->setWidget(_stackSelection);
    _stackDock->setWindowTitle(tr("Top Cost Call Stack"));
    _stackSelection->setWhatsThis( tr(
                   "<b>The Top Cost Call Stack</b>"
                   "<p>This is a purely fictional 'most probable' call stack. "
                   "It is built up by starting with the current selected "
                   "function and adds the callers/callees with highest cost "
                   "at the top and to bottom.</p>"
                   "<p>The <b>Cost</b> and <b>Calls</b> columns show the "
                   "cost used for all calls from the function in the line "
                   "above.</p>"));
    connect(_stackSelection, SIGNAL(functionSelected(CostItem*)),
	    this, SLOT(setTraceItemDelayed(CostItem*)));
    // actions are already created
    connect(_upAction, SIGNAL(triggered(bool)),
	    _stackSelection, SLOT(browserUp()) );
    connect(_backAction, SIGNAL(triggered(bool)),
	    _stackSelection, SLOT(browserBack()) );
    connect(_forwardAction, SIGNAL(triggered(bool)),
	    _stackSelection, SLOT(browserForward()));

    // flat function profile side bar
    _functionDock = new QDockWidget(this);
    _functionDock->setObjectName("function-dock");
    _functionDock->setWindowTitle(tr("Flat Profile"));
    _functionSelection = new FunctionSelection(this, _functionDock);
    _functionDock->setWidget(_functionSelection);
    // functionDock needs call to updateView() when getting visible
    connect(_functionDock, SIGNAL(visibilityChanged(bool)),
	    this, SLOT(functionVisibilityChanged(bool)));

    // defaults (later to be adjusted from stored state in config)
    addDockWidget(Qt::LeftDockWidgetArea, _partDock );
    addDockWidget(Qt::LeftDockWidgetArea, _stackDock );
    addDockWidget(Qt::LeftDockWidgetArea, _functionDock );
    _stackDock->hide();
    _partDock->hide();
}




void TopLevel::createActions()
{
    QString hint;
    QIcon icon;

    // file menu actions
    _newAction = new QAction(tr("&New"), this);
    _newAction->setShortcuts(QKeySequence::New);
    _newAction->setStatusTip(tr("Open new empty window"));
    connect(_newAction, SIGNAL(triggered()), this, SLOT(newWindow()));

    icon = QApplication::style()->standardIcon(QStyle::SP_DialogOpenButton);
    _openAction = new QAction(icon, tr("&Open..."), this);
    _openAction->setShortcuts(QKeySequence::Open);
    _openAction->setStatusTip(tr("Open profile data file"));
    connect(_openAction, SIGNAL(triggered()), this, SLOT(loadTrace()));

    _addAction = new QAction(tr( "&Add..." ), this);
    _addAction->setStatusTip(tr("Add profile data to current window"));
    connect(_addAction, SIGNAL(triggered(bool)), SLOT(addTrace()));

    _reloadAction = new QAction(tr("&Reload", "Reload a document"), this);
    _reloadAction->setStatusTip(tr("Reload profile data including new parts"));
    connect(_reloadAction, SIGNAL(triggered(bool)), SLOT(reload()));

    _exportAction = new QAction(tr("Export Graph"), this);
    _exportAction->setStatusTip(tr("Generate GraphViz file 'callgraph.dot'"));
    connect(_exportAction, SIGNAL(triggered(bool)), SLOT(exportGraph()));

    _dumpToggleAction = new QAction(tr("Callgrind Dump"), this);
    _dumpToggleAction->setCheckable(true);
    _dumpToggleAction->setStatusTip(
	tr("Trigger Callgrind to dump profile data"));
    hint = tr("<b>Force Dump</b>"
              "<p>This forces a dump for a Callgrind profile run "
              "in the current directory. This action is checked while "
              "KCachegrind looks for the dump. If the dump is "
              "finished, it automatically reloads the current trace. "
              "If this is the one from the running Callgrind, the new "
              "created trace part will be loaded, too.</p>"
              "<p>Force dump creates a file 'callgrind.cmd', and "
              "checks every second for its existence. A running "
              "Callgrind will detect this file, dump a trace part, "
              "and delete 'callgrind.cmd'. "
              "The deletion is detected by KCachegrind, "
              "and it does a Reload. If there is <em>no</em> Callgrind "
              "running, press 'Force Dump' again to cancel the dump "
              "request. This deletes 'callgrind.cmd' itself and "
              "stops polling for a new dump.</p>"
              "<p>Note: A Callgrind run <em>only</em> detects "
              "existence of 'callgrind.cmd' when actively running "
              "a few milliseconds, i.e. "
              "<em>not</em> sleeping. Tip: For a profiled GUI program, "
              "you can awake Callgrind e.g. by resizing a window "
              "of the program.</p>");
    _dumpToggleAction->setWhatsThis(hint);
    connect(_dumpToggleAction, SIGNAL(triggered(bool)),
	    this, SLOT(forceTrace()));

    _recentFilesMenuAction = new QAction(tr("Open &Recent"), this);
    _recentFilesMenuAction->setMenu(new QMenu(this));
    connect(_recentFilesMenuAction->menu(), SIGNAL(aboutToShow()),
	     this, SLOT(recentFilesMenuAboutToShow()));
    connect(_recentFilesMenuAction->menu(), SIGNAL(triggered(QAction*)),
	    this, SLOT(recentFilesTriggered(QAction*)));

    _exitAction = new QAction(tr("E&xit"), this);
    _exitAction->setShortcut(tr("Ctrl+Q"));
    _exitAction->setStatusTip(tr("Exit the application"));
    connect(_exitAction, SIGNAL(triggered()), this, SLOT(close()));

    // view menu actions
    icon = QApplication::style()->standardIcon(QStyle::SP_BrowserReload);
    _cyclesToggleAction = new QAction(icon, tr("Detect Cycles"), this);
    _cyclesToggleAction->setCheckable(true);
    _cyclesToggleAction->setStatusTip(tr("Do Cycle Detection"));
    hint = tr("<b>Detect recursive cycles</b>"
              "<p>If this is switched off, the treemap drawing will show "
              "black areas when a recursive call is made instead of drawing "
	      "the recursion ad infinitum. Note that "
              "the size of black areas often will be wrong, as inside "
	      "recursive cycles the cost of calls cannot be determined; "
	      "the error is small, "
              "however, for false cycles (see documentation).</p>"
              "<p>The correct handling for cycles is to detect them and "
	      "collapse all functions of a cycle into an artificial "
	      "function, which is done when this option is selected. "
	      "Unfortunately, with GUI applications, this often will "
              "lead to huge false cycles, making the analysis impossible; "
	      "therefore, there is the option to switch this off.</p>");
    _cyclesToggleAction->setWhatsThis(hint);
    connect(_cyclesToggleAction, SIGNAL(triggered(bool)),
	    this, SLOT(toggleCycles()));
    _cyclesToggleAction->setChecked(GlobalConfig::showCycles());

    _percentageToggleAction = new QAction(QIcon(":/percent.png"),
					  tr("Relative Cost"), this);
    _percentageToggleAction->setCheckable(true);
    _percentageToggleAction->setStatusTip(tr("Show Relative Costs"));
    connect(_percentageToggleAction, SIGNAL(triggered(bool)),
	    this, SLOT(togglePercentage()));
    _percentageToggleAction->setChecked(GlobalConfig::showPercentage());

    _expandedToggleAction = new QAction(QIcon(":/move.png"),
					tr("Relative to Parent"), this);
    _expandedToggleAction->setCheckable(true);
    _expandedToggleAction->setStatusTip(
	tr("Show Percentage relative to Parent"));
    hint = tr("<b>Show percentage costs relative to parent</b>"
              "<p>If this is switched off, percentage costs are always "
	      "shown relative to the total cost of the profile part(s) "
	      "that are currently browsed. By turning on this option, "
	      "percentage cost of shown cost items will be relative "
	      "to the parent cost item.</p>"
              "<ul><table>"
              "<tr><td><b>Cost Type</b></td><td><b>Parent Cost</b></td></tr>"
              "<tr><td>Function Inclusive</td><td>Total</td></tr>"
              "<tr><td>Function Self</td><td>Function Group (*)/Total</td></tr>"
              "<tr><td>Call</td><td>Function Inclusive</td></tr>"
              "<tr><td>Source Line</td><td>Function Inclusive</td></tr>"
              "</table></ul>"
              "<p>(*) Only if function grouping is switched on "
	      "(e.g. ELF object grouping).</p>");
    _expandedToggleAction->setWhatsThis( hint );
    connect(_expandedToggleAction, SIGNAL(triggered(bool)),
	    this, SLOT(toggleExpanded()));
    _expandedToggleAction->setChecked(GlobalConfig::showExpanded());

    _splittedToggleAction = new QAction(tr("Splitted Visualization"), this);
    _splittedToggleAction->setCheckable(true);
    _splittedToggleAction->setStatusTip(
	tr("Show visualization of two cost items"));
    connect(_splittedToggleAction, SIGNAL(triggered(bool)),
	    this, SLOT(toggleSplitted()));

    _splitDirectionToggleAction = new QAction(tr("Split Horizontal"), this);
    _splitDirectionToggleAction->setCheckable(true);
    _splitDirectionToggleAction->setStatusTip(
	tr("Split visualization area horizontally"));
    connect(_splitDirectionToggleAction, SIGNAL(triggered(bool)),
	    this, SLOT(toggleSplitDirection()));

    _sidebarMenuAction = new QAction(tr("Sidebars"), this);
    _sidebarMenuAction->setMenu(new QMenu(this));
    connect( _sidebarMenuAction->menu(), SIGNAL( aboutToShow() ),
	     this, SLOT( sidebarMenuAboutToShow() ));

    _layoutDup = new QAction(tr("&Duplicate"), this);
    connect(_layoutDup, SIGNAL(triggered()), SLOT(layoutDuplicate()));
    _layoutDup->setShortcut(Qt::CTRL + Qt::Key_Plus);
    _layoutDup->setStatusTip(tr("Duplicate current layout"));

    _layoutRemove = new QAction(tr("&Remove"), this);
    connect(_layoutRemove, SIGNAL(triggered()), SLOT(layoutRemove()));
    _layoutRemove->setStatusTip(tr("Remove current layout"));

    _layoutNext = new QAction(tr("Go to &Next"), this);
    connect(_layoutNext, SIGNAL(triggered()), SLOT(layoutNext()));
    _layoutNext->setShortcut(Qt::CTRL + Qt::Key_Right);
    _layoutNext->setStatusTip(tr("Switch to next layout"));

    _layoutPrev = new QAction(tr("Go to &Previous"), this);
    connect(_layoutPrev, SIGNAL(triggered()), SLOT(layoutPrevious()));
    _layoutPrev->setShortcut(Qt::CTRL + Qt::Key_Left);
    _layoutPrev->setStatusTip(tr("Switch to previous layout"));

    _layoutRestore = new QAction(tr("&Restore to Default"), this);
    connect(_layoutRestore, SIGNAL(triggered()), SLOT(layoutRestore()));
    _layoutRestore->setStatusTip(tr("Restore layouts to default"));

    _layoutSave = new QAction(tr("&Save as Default"), this);
    connect(_layoutSave, SIGNAL(triggered()), SLOT(layoutSave()));
    _layoutSave->setStatusTip(tr("Save layouts as default"));

    // go menu actions
    icon = QApplication::style()->standardIcon(QStyle::SP_ArrowUp);
    _upAction = new QAction(icon, tr( "Up" ), this );
    _upAction->setShortcut( QKeySequence(Qt::ALT+Qt::Key_Up) );
    _upAction->setStatusTip(tr("Go Up in Call Stack"));
    _upAction->setMenu(new QMenu(this));
    connect(_upAction->menu(), SIGNAL(aboutToShow()),
	    this, SLOT(upAboutToShow()) );
    connect(_upAction->menu(), SIGNAL(triggered(QAction*)),
	    this, SLOT(upTriggered(QAction*)) );
    hint = tr("Go to last selected caller of current function");
    _upAction->setToolTip(hint);

    icon = QApplication::style()->standardIcon(QStyle::SP_ArrowBack);
    _backAction = new QAction(icon, tr("Back"), this);
    _backAction->setShortcut( QKeySequence(Qt::ALT+Qt::Key_Left) );
    _backAction->setStatusTip(tr("Go Back"));
    _backAction->setMenu(new QMenu(this));
    connect(_backAction->menu(), SIGNAL(aboutToShow()),
           this, SLOT(backAboutToShow()) );
    connect(_backAction->menu(), SIGNAL(triggered(QAction*)),
	    this, SLOT(backTriggered(QAction*)) );
    hint = tr("Go back in function selection history");
    _backAction->setToolTip(hint);

    icon = QApplication::style()->standardIcon(QStyle::SP_ArrowForward);
    _forwardAction = new QAction(icon, tr("Forward"), this);
    _forwardAction->setShortcut( QKeySequence(Qt::ALT+Qt::Key_Right) );
    _forwardAction->setStatusTip(tr("Go Forward"));
    _forwardAction->setMenu(new QMenu(this));
    connect(_forwardAction->menu(), SIGNAL(aboutToShow()),
	    this, SLOT(forwardAboutToShow()) );
    connect(_forwardAction->menu(), SIGNAL(triggered(QAction*)),
	    this, SLOT(forwardTriggered(QAction*)) );
    hint = tr("Go forward in function selection history");
    _forwardAction->setToolTip( hint );

    // help menu actions
    _aboutAction = new QAction(tr("&About QCachegrind ..."), this);
    _aboutAction->setStatusTip(tr("Show the application's About box"));
    connect(_aboutAction, SIGNAL(triggered()), this, SLOT(about()));

    _aboutQtAction = new QAction(tr("About Qt ..."), this);
    connect(_aboutQtAction, SIGNAL(triggered()), qApp, SLOT(aboutQt()));

    // toolbar actions
    _eventTypeBox = new QComboBox(this);
    _eventTypeBox->setMinimumContentsLength(25);
    hint = tr("Select primary event type of costs");
    _eventTypeBox->setToolTip( hint );
    connect( _eventTypeBox, SIGNAL(activated(const QString&)),
	     this, SLOT(eventTypeSelected(const QString&)));
}

void TopLevel::createMenu()
{
    QMenuBar* mBar = menuBar();

    QMenu* fileMenu = mBar->addMenu(tr("&File"));
    fileMenu->addAction(_newAction);
    fileMenu->addAction(_openAction);
    fileMenu->addAction(_recentFilesMenuAction);
    fileMenu->addAction(_addAction);
    fileMenu->addSeparator();
    fileMenu->addAction(_reloadAction);
    fileMenu->addAction(_exportAction);
    fileMenu->addAction(_dumpToggleAction);
    fileMenu->addSeparator();
    fileMenu->addAction(_exitAction);

    QMenu* layoutMenu = new QMenu(tr("&Layout"), this);
    layoutMenu->addAction(_layoutDup);
    layoutMenu->addAction(_layoutRemove);
    layoutMenu->addSeparator();
    layoutMenu->addAction(_layoutPrev);
    layoutMenu->addAction(_layoutNext);
    layoutMenu->addSeparator();
    layoutMenu->addAction(_layoutSave);
    layoutMenu->addAction(_layoutRestore);

    QMenu* viewMenu = mBar->addMenu(tr("&View"));
    viewMenu->addAction(_cyclesToggleAction);
    viewMenu->addAction(_percentageToggleAction);
    viewMenu->addAction(_expandedToggleAction);
    viewMenu->addSeparator();
    viewMenu->addAction(_splittedToggleAction);
    viewMenu->addAction(_splitDirectionToggleAction);
    viewMenu->addMenu(layoutMenu);
    viewMenu->addSeparator();
    viewMenu->addAction(_sidebarMenuAction);

    QMenu* goMenu = mBar->addMenu("&Go");
    goMenu->addAction(_backAction);
    goMenu->addAction(_forwardAction);
    goMenu->addAction(_upAction);

    QMenu* helpMenu = mBar->addMenu("&Help");
    helpMenu->addAction(_aboutAction);
    helpMenu->addAction(_aboutQtAction);
}

void TopLevel::createToolbar()
{
    QToolBar* tb = new QToolBar(tr("Main Toolbar"), this);
    tb->setObjectName("main-toolbar");
    addToolBar(Qt::TopToolBarArea, tb);

    tb->addAction(_openAction);
    tb->addSeparator();

    tb->addAction(_cyclesToggleAction);
    tb->addAction(_percentageToggleAction);
    tb->addAction(_expandedToggleAction);
    tb->addSeparator();

    tb->addAction(_backAction);
    tb->addAction(_forwardAction);
    tb->addAction(_upAction);
    tb->addSeparator();

    tb->addWidget(_eventTypeBox);
}


void TopLevel::about()
{
    QString text, version;
    version = QLatin1String("0.5.1");
    text = QString("<h3>QCachegrind %1</h3>").arg(version);
    text += tr("<p>QCachegrind is a graphical user interface for analysing "
	      "profiling data, which helps in the performance optimization "
	      "phase of developing a computer program. "
	      "QCachegrind is open-source, and it is distributed under the "
	      "terms of the GPL v2. For details and source code, see the "
	      "<a href=\"http://kcachegrind.sf.net\">homepage</a> of the "
	      "KCachegrind project.</p>"
              "Main author and maintainer: "
	      "<a href=\"mailto:Josef.Weidendorfer@gmx.de\">"
              "Josef Weidendorfer</a><br>"
              "(with lots of bug fixes and porting to Qt4 by the KDE team)");
    QMessageBox::about(this, tr("About QCachegrind"), text);
}


void TopLevel::togglePartDock()
{
  if (!_partDock->isVisible())
    _partDock->show();
  else
    _partDock->hide();
}

void TopLevel::toggleStackDock()
{
  if (!_stackDock->isVisible())
    _stackDock->show();
  else
    _stackDock->hide();
}

void TopLevel::toggleFunctionDock()
{
  if (!_functionDock->isVisible())
    _functionDock->show();
  else
    _functionDock->hide();
}

void TopLevel::togglePercentage()
{
  setPercentage(_percentageToggleAction->isChecked());
}

void TopLevel::setAbsoluteCost()
{
  setPercentage(false);
}

void TopLevel::setRelativeCost()
{
  setPercentage(true);
}

void TopLevel::setPercentage(bool show)
{
    if (GlobalConfig::showPercentage() == show) return;
    if (_percentageToggleAction->isChecked() != show)
	_percentageToggleAction->setChecked(show);
    _expandedToggleAction->setEnabled(show);
    GlobalConfig::setShowPercentage(show);

    _partSelection->notifyChange(TraceItemView::configChanged);
    _partSelection->updateView();

    _stackSelection->refresh();

    _functionSelection->notifyChange(TraceItemView::configChanged);
    _functionSelection->updateView();

    _multiView->notifyChange(TraceItemView::configChanged);
    _multiView->updateView();
}

void TopLevel::toggleExpanded()
{
    bool show = _expandedToggleAction->isChecked();
    if (GlobalConfig::showExpanded() == show) return;
    GlobalConfig::setShowExpanded(show);

    _partSelection->notifyChange(TraceItemView::configChanged);
    _partSelection->updateView();

    _stackSelection->refresh();

    _functionSelection->notifyChange(TraceItemView::configChanged);
    _functionSelection->updateView();

    _multiView->notifyChange(TraceItemView::configChanged);
    _multiView->updateView();
}

void TopLevel::toggleCycles()
{
    bool show = _cyclesToggleAction->isChecked();
    if (GlobalConfig::showCycles() == show) return;
    GlobalConfig::setShowCycles(show);

    if (!_data) return;

    _data->invalidateDynamicCost();
    _data->updateFunctionCycles();

    _partSelection->notifyChange(TraceItemView::configChanged);
    _partSelection->updateView();

    _stackSelection->rebuildStackList();

    _functionSelection->notifyChange(TraceItemView::configChanged);
    _functionSelection->updateView();

    _multiView->notifyChange(TraceItemView::configChanged);
    _multiView->updateView();
}


void TopLevel::functionVisibilityChanged(bool v)
{
    if (v)
	_functionSelection->updateView();
}


void TopLevel::newWindow()
{
    TopLevel* t = new TopLevel();
    t->show();
}


void TopLevel::loadTrace()
{
    QString file;
    file = QFileDialog::getOpenFileName(this,
					tr("Open Callgrind Data"),
					QString::null,
					tr("Callgrind Files (callgrind.*)"));
    loadTrace(file);
}

void TopLevel::loadTrace(QString file, bool addToRecentFiles)
{
    if (file.isEmpty()) return;

    if (_data && _data->parts().count()>0) {

	// In new window
	TopLevel* t = new TopLevel();
	t->show();
	t->loadDelayed(file, addToRecentFiles);
	return;
    }

    // this constructor enables progress bar callbacks
    TraceData* d = new TraceData(this);
    int filesLoaded = d->load(file);
    if (filesLoaded >0)
	setData(d);

    if (!addToRecentFiles) return;

    // add to recent file list in config
    QStringList recentFiles;
    ConfigGroup* generalConfig = ConfigStorage::group("GeneralSettings");
    recentFiles = generalConfig->value("RecentFiles",
				       QStringList()).toStringList();
    recentFiles.removeAll(file);
    if (filesLoaded >0)
	recentFiles.prepend(file);
    if (recentFiles.count() >5)
	recentFiles.removeLast();
    generalConfig->setValue("RecentFiles", recentFiles);
    delete generalConfig;
}


void TopLevel::addTrace()
{
    QString file;
    file = QFileDialog::getOpenFileName(this,
					tr("Add Callgrind Data"),
					QString::null,
					tr("Callgrind Files (callgrind.*)"));
    addTrace(file);
}


void TopLevel::addTrace(QString file)
{
    if (file.isEmpty()) return;

    if (_data) {
	_data->load(file);

	// GUI update for added data
	configChanged();
	return;
    }

    // this constructor enables progress bar callbacks
    TraceData* d = new TraceData(this);
    int filesLoaded = d->load(file);
    if (filesLoaded >0)
	setData(d);
}



void TopLevel::loadDelayed(QString file, bool addToRecentFiles)
{
    _loadTraceDelayed = file;
    _addToRecentFiles = addToRecentFiles;
    QTimer::singleShot(0, this, SLOT(loadTraceDelayed()));
}

void TopLevel::loadTraceDelayed()
{
    if (_loadTraceDelayed.isEmpty()) return;

    loadTrace(_loadTraceDelayed, _addToRecentFiles);
    _loadTraceDelayed = QString();
}


void TopLevel::reload()
{
    QString trace;
    if (!_data || _data->parts().count()==0)
	trace = "."; // open first trace found in dir
    else
	trace = _data->traceName();

    // this also keeps sure we have the same browsing position...
    TraceData* d = new TraceData(this);
    d->load(trace);
    setData(d);
}

void TopLevel::exportGraph()
{
    if (!_data || !_function) return;

    QString n = QString("callgraph.dot");
    GraphExporter ge(_data, _function, _eventType, _groupType, n);
    ge.writeDot();

#ifdef Q_OS_UNIX
    // shell commands only work in UNIX
    QString cmd = QString("(dot %1 -Tps > %2.ps; kghostview %3.ps)&")
	.arg(n).arg(n).arg(n);
    ::system(QFile::encodeName( cmd ));
#endif
}


bool TopLevel::setEventType(QString s)
{
  EventType* ct;

  ct = (_data) ? _data->eventTypes()->type(s) : 0;

  // if costtype with given name not found, use first available
  if (!ct && _data) ct = _data->eventTypes()->type(0);

  return setEventType(ct);
}

bool TopLevel::setEventType2(QString s)
{
  EventType* ct;

  // Special type tr("(Hidden)") gives 0
  ct = (_data) ? _data->eventTypes()->type(s) : 0;

  return setEventType2(ct);
}

void TopLevel::eventTypeSelected(const QString& s)
{
  EventType* ct;

  ct = (_data) ? _data->eventTypes()->typeForLong(s) : 0;
  setEventType(ct);
}

void TopLevel::eventType2Selected(const QString& s)
{
  EventType* ct;

  ct = (_data) ? _data->eventTypes()->typeForLong(s) : 0;
  setEventType2(ct);
}

bool TopLevel::setEventType(EventType* ct)
{
  if (_eventType == ct) return false;
  _eventType = ct;

  if (ct) {
      int idx = _eventTypeBox->findText(ct->longName());
      _eventTypeBox->setCurrentIndex(idx);
  }

  _partSelection->setEventType(_eventType);
  _stackSelection->setEventType(_eventType);

  _functionSelection->setEventType(_eventType);
  _functionSelection->updateView();

  _multiView->setEventType(_eventType);
  _multiView->updateView();

  updateStatusBar();

  return true;
}

bool TopLevel::setEventType2(EventType* ct)
{
  if (_eventType2 == ct) return false;
  _eventType2 = ct;

  QString longName = ct ? ct->longName() : tr("(Hidden)");

  _partSelection->setEventType2(_eventType2);
  _stackSelection->setEventType2(_eventType2);

  _functionSelection->setEventType2(_eventType2);
  _functionSelection->updateView();

  _multiView->setEventType2(_eventType2);
  _multiView->updateView();

  updateStatusBar();

  return true;
}


void TopLevel::groupTypeSelected(int cg)
{
  switch(cg) {
  case 0: setGroupType( ProfileContext::Function ); break;
  case 1: setGroupType( ProfileContext::Object ); break;
  case 2: setGroupType( ProfileContext::File ); break;
  case 3: setGroupType( ProfileContext::Class ); break;
  case 4: setGroupType( ProfileContext::FunctionCycle ); break;
  default: break;
  }
}

bool TopLevel::setGroupType(QString s)
{
  ProfileContext::Type gt;

  gt = ProfileContext::type(s);
  // only allow Function/Object/File/Class as grouptype
  switch(gt) {
  case ProfileContext::Object:
  case ProfileContext::File:
  case ProfileContext::Class:
  case ProfileContext::FunctionCycle:
    break;
  default:
    gt = ProfileContext::Function;
  }

  return setGroupType(gt);
}

bool TopLevel::setGroupType(ProfileContext::Type gt)
{
  if (_groupType == gt) return false;
  _groupType = gt;

  int idx = -1;
  switch(gt) {
  case ProfileContext::Function: idx = 0; break;
  case ProfileContext::Object: idx = 1; break;
  case ProfileContext::File: idx = 2; break;
  case ProfileContext::Class: idx = 3; break;
  case ProfileContext::FunctionCycle: idx = 4; break;
  default:
    break;
  }

  if (idx==-1) return false;

#if 0
  if (saGroup->currentItem() != idx)
    saGroup->setCurrentItem(idx);
#endif

  _stackSelection->setGroupType(_groupType);

  _partSelection->set(_groupType);
  _partSelection->updateView();

  _functionSelection->set(_groupType);
  _functionSelection->updateView();

  _multiView->set(_groupType);
  _multiView->updateView();

  updateStatusBar();

  return true;
}

bool TopLevel::setGroup(QString s)
{
    TraceCostItem* ci = _functionSelection->group(s);
    if (!ci)
	return false;

    return setGroup(ci);
}


bool TopLevel::setGroup(TraceCostItem* g)
{
    if (_group == g) return false;
    _group = g;

    _functionSelection->setGroup(g);
    updateStatusBar();

    return true;
}

bool TopLevel::setFunction(QString s)
{
  if (!_data) return false;

  ProfileCostArray* f = _data->search(ProfileContext::Function, s, _eventType);
  if (!f) return false;

  return setFunction((TraceFunction*)f);
}

bool TopLevel::setFunction(TraceFunction* f)
{
  if (_function == f) return false;
  _function = f;

  _multiView->activate(f);
  _multiView->updateView();

  _functionSelection->activate(f);
  _functionSelection->updateView();

  _partSelection->activate(f);
  _partSelection->updateView();

  _stackSelection->setFunction(_function);

  StackBrowser* b = _stackSelection->browser();
  if (b) {
    // do not disable up: a press forces stack-up extending...
    _forwardAction->setEnabled(b->canGoForward());
    _backAction->setEnabled(b->canGoBack());
  }

#if TRACE_UPDATES
  qDebug("TopLevel::setFunction(%s), lastSender %s",
         f ? f->prettyName().ascii() : "0",
         _lastSender ? _lastSender->name() :"0" );
#endif

  return true;
}


/**
 * Delayed versions.
 * We always have a pair of slots: One receiver to start the
 * delay with a singleShot Timer. It stores the parameter into a
 * temporary variable. And one parameterless slot for
 * forwarding, using this temporary.
 */
void TopLevel::setEventTypeDelayed(EventType* ct)
{
  _eventTypeDelayed = ct;
  QTimer::singleShot (0, this, SLOT(setEventTypeDelayed()));
}

void TopLevel::setEventType2Delayed(EventType* ct)
{
  _eventType2Delayed = ct;
  QTimer::singleShot (0, this, SLOT(setEventType2Delayed()));
}

void TopLevel::setEventTypeDelayed()
{
  setEventType(_eventTypeDelayed);
}

void TopLevel::setEventType2Delayed()
{
  setEventType2(_eventType2Delayed);
}

void TopLevel::setGroupTypeDelayed(ProfileContext::Type gt)
{
  _groupTypeDelayed = gt;
  QTimer::singleShot (0, this, SLOT(setGroupTypeDelayed()));
}

void TopLevel::setGroupTypeDelayed()
{
  setGroupType(_groupTypeDelayed);
}

void TopLevel::setGroupDelayed(TraceCostItem* g)
{
#if TRACE_UPDATES
  qDebug("TopLevel::setGroupDelayed(%s), sender %s",
         g ? g->prettyName().ascii() : "0",
         _lastSender ? _lastSender->name() :"0" );
#endif

  _groupDelayed = g;
  QTimer::singleShot (0, this, SLOT(setGroupDelayed()));
}

void TopLevel::setGroupDelayed()
{
  setGroup(_groupDelayed);
}

void TopLevel::setDirectionDelayed(TraceItemView::Direction d)
{
  _directionDelayed = d;
  QTimer::singleShot (0, this, SLOT(setDirectionDelayed()));
}

void TopLevel::setDirectionDelayed()
{
    switch(_directionDelayed) {
    case TraceItemView::Back:
	_stackSelection->browserBack();
	break;

    case TraceItemView::Forward:
        _stackSelection->browserForward();
	break;

    case TraceItemView::Up:
	{
	    StackBrowser* b = _stackSelection ? _stackSelection->browser() : 0;
	    HistoryItem* hi = b ? b->current() : 0;
	    TraceFunction* f = hi ? hi->function() : 0;

	    if (!f) break;
	    f = hi->stack()->caller(f, false);
	    if (f) setFunction(f);
	}
    break;

    default: break;
    }

    _directionDelayed = TraceItemView::None;
}


void TopLevel::setTraceItemDelayed(CostItem* i)
{
  // no need to select same item a 2nd time...
  if (_traceItemDelayed == i) return;
  _traceItemDelayed = i;
  _lastSender = sender();

  qDebug() << "Selected " << (i ? i->prettyName() : "(none)");

#if TRACE_UPDATES
  qDebug("TopLevel::setTraceItemDelayed(%s), sender %s",
         i ? i->prettyName().ascii() : "0",
         _lastSender ? _lastSender->name() :"0" );
#endif

  QTimer::singleShot (0, this, SLOT(setTraceItemDelayed()));
}

void TopLevel::setTraceItemDelayed()
{
    if (!_traceItemDelayed) return;

    switch(_traceItemDelayed->type()) {
    case ProfileContext::Function:
    case ProfileContext::FunctionCycle:
	setFunction((TraceFunction*)_traceItemDelayed);
	break;

    case ProfileContext::Object:
    case ProfileContext::File:
    case ProfileContext::Class:
	_multiView->activate(_traceItemDelayed);
	_multiView->updateView();
	break;

#if 0
	// this conflicts with the selection policy of InstrView ?!?
    case ProfileContext::Instr:
    case ProfileContext::Line:
	// only for multiview
	_multiView->activate(_traceItemDelayed);
	_multiView->updateView();
	break;
#endif

    default: break;
    }

    _traceItemDelayed = 0;
    _lastSender = 0;
}

/**
 * A TraceData object cannot be viewed many times in different
 * toplevel windows. Thus, this toplevel window takes ownership
 * of the TraceData object: on closing the window or opening
 * another trace, the object is destroyed.
 */
void TopLevel::setData(TraceData* data)
{
  if (data == _data) return;

  _lastSender = 0;

  saveTraceSettings();

  if (_data) {
    _partSelection->setData(0);
    _stackSelection->setData(0);

    _functionSelection->setData(0);
    _functionSelection->updateView();

    _multiView->setData(0);
    _multiView->updateView();

    // we are the owner...
    delete _data;
  }

  // reset members
  resetState();

  _data = data;

  // fill cost type list
  QStringList types;

  if (_data) {
      /* add all supported virtual types */
      EventTypeSet* m = _data->eventTypes();
      m->addKnownDerivedTypes();

      /* first, fill selection list with available cost types */
      for (int i=0;i<m->realCount();i++)
	  types << m->realType(i)->longName();
      for (int i=0;i<m->derivedCount();i++)
	  types << m->derivedType(i)->longName();
  }
  _eventTypes = types;
  _eventTypeBox->addItems(types);

  _stackSelection->setData(_data);

  _partSelection->setData(_data);
  _partSelection->updateView();

  _functionSelection->setData(_data);
  _functionSelection->updateView();

  _multiView->setData(_data);
  _multiView->updateView();

  /* this is needed to let the other widgets know the types */
  restoreTraceTypes();

  restoreTraceSettings();

  QString caption;
  if (_data) {
    caption = QDir::toNativeSeparators(_data->traceName());
    if (!_data->command().isEmpty())
      caption += " [" + _data->command() + ']';
  }
  setWindowTitle(caption);

  if (!_data || (!_forcePartDock && _data->parts().count()<2))
    _partDock->hide();
  else
    _partDock->show();

  updateStatusBar();
}

void TopLevel::addEventTypeMenu(QMenu* popup, bool withCost2)
{
  if (_data) {
    QMenu *popup1, *popup2 = 0;
    QAction* action;

    popup1 = popup->addMenu(tr("Primary Event Type"));
    connect(popup1, SIGNAL(triggered(QAction*)),
	     this, SLOT(setEventType(QAction*)));

    if (withCost2) {
      popup2 = popup->addMenu(tr("Secondary Event Type"));
      connect(popup2, SIGNAL(triggered(QAction*)),
	       this, SLOT(setEventType2(QAction*)));

      if (_eventType2) {
	action = popup2->addAction(tr("Hide"));
	action->setData(199);
	popup2->addSeparator();
      }
    }

    EventTypeSet* m = _data->eventTypes();
    EventType* ct;
    for (int i=0;i<m->realCount();i++) {
      ct = m->realType(i);

      action = popup1->addAction(ct->longName());
      action->setCheckable(true);
      action->setData(100+i);
      if (_eventType == ct) action->setChecked(true);

      if (popup2) {
	action = popup2->addAction(ct->longName());
	action->setCheckable(true);
	action->setData(100+i);
	if (_eventType2 == ct) action->setChecked(true);
      }
    }

    for (int i=0;i<m->derivedCount();i++) {
      ct = m->derivedType(i);

      action = popup1->addAction(ct->longName());
      action->setCheckable(true);
      action->setData(200+i);
      if (_eventType == ct) action->setChecked(true);

      if (popup2) {
	action = popup2->addAction(ct->longName());
	action->setCheckable(true);
	action->setData(200+i);
	if (_eventType2 == ct) action->setChecked(true);
      }
    }
  }

  if (GlobalConfig::showPercentage())
    popup->addAction(tr("Show Absolute Cost"),
		      this, SLOT(setAbsoluteCost()));
  else
    popup->addAction(tr("Show Relative Cost"),
		      this, SLOT(setRelativeCost()));
}

bool TopLevel::setEventType(QAction* action)
{
  if (!_data) return false;
  int id = action->data().toInt(0);

  EventTypeSet* m = _data->eventTypes();
  EventType* ct=0;
  if (id >=100 && id<199) ct = m->realType(id-100);
  if (id >=200 && id<299) ct = m->derivedType(id-200);

  return ct ? setEventType(ct) : false;
}

bool TopLevel::setEventType2(QAction* action)
{
  if (!_data) return false;
  int id = action->data().toInt(0);

  EventTypeSet* m = _data->eventTypes();
  EventType* ct=0;
  if (id >=100 && id<199) ct = m->realType(id-100);
  if (id >=200 && id<299) ct = m->derivedType(id-200);

  return setEventType2(ct);
}

void TopLevel::addGoMenu(QMenu* popup)
{
  StackBrowser* b = _stackSelection->browser();
  if (b) {
      if (b->canGoBack())
	  popup->addAction(tr("Go Back"), this, SLOT(goBack()));
      if (b->canGoForward())
	  popup->addAction(tr("Go Forward"), this, SLOT(goForward()));
  }
  // do not disable up: a press forces stack-up extending...
  popup->addAction(tr("Go Up"), this, SLOT(goUp()));
}

void TopLevel::goBack()
{
  setDirectionDelayed(TraceItemView::Back);
}

void TopLevel::goForward()
{
  setDirectionDelayed(TraceItemView::Forward);
}

void TopLevel::goUp()
{
  setDirectionDelayed(TraceItemView::Up);
}

QString TopLevel::traceKey()
{
  if (!_data || _data->command().isEmpty()) return QString();

  QString name = _data->command();
  QString key;
  for (int l=0;l<name.length();l++)
    if (name[l].isLetterOrNumber()) key += name[l];

  return QString("-") + key;
}


void TopLevel::restoreTraceTypes()
{
    QString key = traceKey();
    QString groupType, eventType, eventType2;

    ConfigGroup* pConfig = ConfigStorage::group("TracePositions");
    groupType = pConfig->value(QString("GroupType%1").arg(key),QString()).toString();
    eventType = pConfig->value(QString("EventType%1").arg(key),QString()).toString();
    eventType2 = pConfig->value(QString("EventType2%1").arg(key),QString()).toString();
    delete pConfig;

    ConfigGroup* cConfig = ConfigStorage::group("CurrentState");
    if (groupType.isEmpty())
	groupType = cConfig->value("GroupType",QString()).toString();
    if (eventType.isEmpty())
	eventType = cConfig->value("EventType",QString()).toString();
    if (eventType2.isEmpty())
	eventType2 = cConfig->value("EventType2",QString()).toString();
    delete cConfig;

    setGroupType(groupType);
    setEventType(eventType);
    setEventType2(eventType2);

    // if still no event type set, use first available
    if (!_eventType && !_eventTypes.isEmpty())
	eventTypeSelected(_eventTypes.first());

    ConfigGroup* aConfig = ConfigStorage::group("Layouts");
    _layoutCount = aConfig->value(QString("Count%1").arg(key), 0).toInt();
    _layoutCurrent = aConfig->value(QString("Current%1").arg(key), 0).toInt();
    delete aConfig;

    if (_layoutCount == 0) layoutRestore();
    updateLayoutActions();
}


/**
 * This must be called after setting group/cost types in the function
 * selection widget, because the group/function choosing depends on
 * filled lists in the function selection widget
 */
void TopLevel::restoreTraceSettings()
{
  if (!_data) return;

  QString key = traceKey();

  restoreCurrentState(key);

  ConfigGroup* pConfig = ConfigStorage::group("TracePositions");
  QString group = pConfig->value(QString("Group%1").arg(key),QString()).toString();
  delete pConfig;
  if (!group.isEmpty()) setGroup(group);

  // restoreCurrentState() usually leads to a call to setTraceItemDelayed()
  // to restore last active item...
  if (!_traceItemDelayed) {
    // function not available any more.. try with "main"
      if (!setFunction("main")) {
#if 1
	_functionSelection->setTopFunction();
#else
	HighestCostList hc;
	hc.clear(50);
	TraceFunctionMap::Iterator it;
	for ( it = _data->functionMap().begin();
	      it != _data->functionMap().end(); ++it )
	    hc.addCost(&(*it), (*it).inclusive()->subCost(_eventType));

	setFunction( (TraceFunction*) hc[0]);
#endif
      }
  }
}


/* Layout */

void TopLevel::layoutDuplicate()
{
    // save current and allocate a new slot
    _multiView->saveLayout(QString("Layout%1-MainView").arg(_layoutCurrent),
			   traceKey());
    _layoutCurrent = _layoutCount;
    _layoutCount++;

    updateLayoutActions();

    qDebug() << "TopLevel::layoutDuplicate: count " << _layoutCount;
}

void TopLevel::layoutRemove()
{
    if (_layoutCount <2) return;

    int from = _layoutCount-1;
    if (_layoutCurrent == from) { _layoutCurrent--; from--; }

    // restore from last and decrement count
    _multiView->restoreLayout(QString("Layout%1-MainView").arg(from),
			      traceKey());
    _layoutCount--;

    updateLayoutActions();

    qDebug() << "TopLevel::layoutRemove: count " << _layoutCount;
}

void TopLevel::layoutNext()
{
    if (_layoutCount <2) return;

    QString key = traceKey();
    QString layoutPrefix = QString("Layout%1-MainView");

    _multiView->saveLayout(layoutPrefix.arg(_layoutCurrent), key);
    _layoutCurrent++;
    if (_layoutCurrent == _layoutCount) _layoutCurrent = 0;
    _multiView->restoreLayout(layoutPrefix.arg(_layoutCurrent), key);

    qDebug() << "TopLevel::layoutNext: current " << _layoutCurrent;
}

void TopLevel::layoutPrevious()
{
    if (_layoutCount <2) return;

    QString key = traceKey();
    QString layoutPrefix = QString("Layout%1-MainView");

    _multiView->saveLayout(layoutPrefix.arg(_layoutCurrent), key);
    _layoutCurrent--;
    if (_layoutCurrent <0) _layoutCurrent = _layoutCount-1;
    _multiView->restoreLayout(layoutPrefix.arg(_layoutCurrent), key);

    qDebug() << "TopLevel::layoutPrevious: current " << _layoutCurrent;
}

void TopLevel::layoutSave()
{
    QString key = traceKey();
    QString layoutPrefix = QString("Layout%1-MainView");

    _multiView->saveLayout(layoutPrefix.arg(_layoutCurrent), key);

    // save all layouts as defaults (ie. without any group name postfix)
    for(int i=0;i<_layoutCount;i++) {
	_multiView->restoreLayout(layoutPrefix.arg(i), key);
	_multiView->saveLayout(layoutPrefix.arg(i), QString());
    }
    // restore the previously saved current layout
    _multiView->restoreLayout(layoutPrefix.arg(_layoutCurrent), key);

    ConfigGroup* layoutConfig = ConfigStorage::group("Layouts");
    layoutConfig->setValue("DefaultCount", _layoutCount);
    layoutConfig->setValue("DefaultCurrent", _layoutCurrent);
    delete layoutConfig;
}

void TopLevel::layoutRestore()
{
    ConfigGroup* layoutConfig = ConfigStorage::group("Layouts");
    _layoutCount = layoutConfig->value("DefaultCount", 0).toInt();
    _layoutCurrent = layoutConfig->value("DefaultCurrent", 0).toInt();
    delete layoutConfig;

    if (_layoutCount == 0) {
	_layoutCount++;
	return;
    }

    QString layoutPrefix = QString("Layout%1-MainView");
    _multiView->restoreLayout( layoutPrefix.arg(_layoutCurrent), traceKey());

    updateLayoutActions();
}


void TopLevel::updateLayoutActions()
{
    if (_layoutNext)
	_layoutNext->setEnabled(_layoutCount>1);

    if (_layoutPrev)
	_layoutPrev->setEnabled(_layoutCount>1);

    if (_layoutRemove)
	_layoutRemove->setEnabled(_layoutCount>1);

    if (_statusbar)
	_statusbar->message(tr("Layout Count: %1").arg(_layoutCount),
			    1000);
}


void TopLevel::updateStatusBar()
{
  if (!_data || _data->parts().count()==0) {
    _statusLabel->setText(tr("No profile data file loaded."));
    return;
  }

  QString status = QString("%1 [%2] - ")
                   .arg(_data->shortTraceName())
                   .arg(_data->activePartRange());

  if (_eventType) {
      status += tr("Total %1 Cost: %2")
	  .arg(_eventType->longName())
	  .arg(_data->prettySubCost(_eventType));

    /* this gets too long...
    if (_eventType2 && (_eventType2 != _eventType))
      status += tr(", %1 Cost: %2")
	.arg(_eventType2->longName())
	.arg(_data->prettySubCost(_eventType2));
    */
  }
  else
    status += tr("No event type selected");

  /* Not working... should give group of selected function

  if (_groupType != ProfileContext::Function) {
    status += QString(" - %1 '%2'")
              .arg(ProfileContext::trTypeName(_groupType))
              .arg(_group ? _group->prettyName() : tr("(None)"));
  }
  */

  _statusLabel->setText(status);
}

#if 0
void TopLevel::configure()
{
    if (ConfigDlg::configure( (KConfiguration*) GlobalConfig::config(),
			      _data, this)) {
      GlobalConfig::config()->saveOptions();

    configChanged();
  }
  else
      GlobalConfig::config()->readOptions();
}
#endif

void TopLevel::closeEvent(QCloseEvent* event)
{
    GlobalConfig::config()->saveOptions();

    saveTraceSettings();
    saveCurrentState(QString::null);

    // if part dock was chosen visible even for only 1 part loaded,
    // keep this choice...
    _forcePartDock = false;
    if (_data && (_data->parts().count()<2) && _partDock->isVisible())
	_forcePartDock=true;

    ConfigGroup* topConfig = ConfigStorage::group("TopWindow");
    topConfig->setValue("ForcePartDockVisible", _forcePartDock, false);
    topConfig->setValue("State", saveState());
    topConfig->setValue("Geometry", saveGeometry());
    delete topConfig;

    event->accept();
}


void TopLevel::toggleSplitted()
{
    int count = _multiView->childCount();
    if (count<1) count = 1;
    if (count>2) count = 2;
    count = 3-count;
    _multiView->setChildCount(count);

    _splittedToggleAction->setChecked(count>1);
    _splitDirectionToggleAction->setEnabled(count>1);
    _splitDirectionToggleAction->setChecked(_multiView->orientation() ==
					    Qt::Horizontal);
}

void TopLevel::toggleSplitDirection()
{
  _multiView->setOrientation( _splitDirectionToggleAction->isChecked() ?
                              Qt::Horizontal : Qt::Vertical );
}



// this is called after a config change in the dialog
void TopLevel::configChanged()
{
  //qDebug("TopLevel::configChanged");
  //_showPercentage->setChecked(GlobalConfig::showPercentage());

  // invalidate found/cached dirs of source files
  _data->resetSourceDirs();

  _partSelection->notifyChange(TraceItemView::configChanged);
  _partSelection->updateView();

  _stackSelection->refresh();

  _functionSelection->notifyChange(TraceItemView::configChanged);
  _functionSelection->updateView();

  _multiView->notifyChange(TraceItemView::configChanged);
  _multiView->updateView();
}



void TopLevel::activePartsChangedSlot(const TracePartList& list)
{
  if (!_data) return;

  if (!_data->activateParts(list)) {
//    qDebug("TopLevel::activePartsChangedSlot: No Change!");
    return;
  }
  _activeParts = list;

  _partSelection->set(list);
  _partSelection->updateView();

  _stackSelection->refresh();

  _functionSelection->set(list);
  _functionSelection->updateView();

  _multiView->set(list);
  _multiView->updateView();

  updateStatusBar();
}

void TopLevel::partsHideSelectedSlotDelayed()
{
  QTimer::singleShot( 0, this, SLOT(partsHideSelectedSlot()) );
}

// this puts selected parts into hidden list,
// deselects them and makes the remaining parts selected
void TopLevel::partsHideSelectedSlot()
{
  if (!_data) return;

  TracePart* part;
  TracePartList newHidden, newActive;
  TracePartList l = _data->parts();
  for (part=l.first();part;part=l.next()) {
    if ((_activeParts.findRef(part)>=0) ||
        (_hiddenParts.findRef(part)>=0))
      newHidden.append(part);
    else
      newActive.append(part);
  }

  _hiddenParts = newHidden;
  _partSelection->hiddenPartsChangedSlot(_hiddenParts);

#if 0
  _mainWidget1->hiddenPartsChangedSlot(_hiddenParts);
  _mainWidget2->hiddenPartsChangedSlot(_hiddenParts);
#endif

  activePartsChangedSlot(newActive);
}

void TopLevel::partsUnhideAllSlotDelayed()
{
  QTimer::singleShot( 0, this, SLOT(partsUnhideAllSlot()) );
}

// this unhides all hidden parts. Does NOT change selection
void TopLevel::partsUnhideAllSlot()
{
  if (!_data) return;

  _hiddenParts.clear();
  _partSelection->hiddenPartsChangedSlot(_hiddenParts);

#if 0
  _mainWidget1->hiddenPartsChangedSlot(_hiddenParts);
  _mainWidget2->hiddenPartsChangedSlot(_hiddenParts);
#endif
}

void TopLevel::forceTrace()
{
//  qDebug("forceTrace");

  // Needs Callgrind now...
  QFile cmd("callgrind.cmd");
  if (!cmd.exists()) {
    cmd.open(QIODevice::WriteOnly);
    cmd.write("DUMP\n", 5);
    cmd.close();
  }
  if (_dumpToggleAction->isChecked())
    QTimer::singleShot( 1000, this, SLOT(forceTraceReload()) );
  else {
    // cancel request
    cmd.remove();
  }

}

void TopLevel::forceTraceReload()
{
//  qDebug("forceTraceReload");

  QFile cmd("callgrind.cmd");
  if (cmd.exists()) {
    if (_dumpToggleAction->isChecked())
      QTimer::singleShot( 1000, this, SLOT(forceTraceReload()) );
    return;
  }
  _dumpToggleAction->setChecked(false);
  reload();
}

void TopLevel::forwardAboutToShow()
{
  QMenu *popup = _forwardAction->menu();

  popup->clear();
  StackBrowser* b = _stackSelection ? _stackSelection->browser() : 0;
  HistoryItem* hi = b ? b->current() : 0;
  TraceFunction* f;
  QAction* action;

  if (!hi) {
    popup->addAction(tr("(No Stack)"));
    return;
  }

  hi = hi->next();
  if (!hi) {
    popup->addAction(tr("(No next function)"));
    return;
  }

  int count = 1;
  while (count<GlobalConfig::maxSymbolCount() && hi) {
    f = hi->function();
    if (!f) break;

    QString name = f->prettyName();
    if ((int)name.length()>GlobalConfig::maxSymbolLength())
      name = name.left(GlobalConfig::maxSymbolLength()) + "...";

    //qDebug("forward: Adding %s", name.ascii());
    action = popup->addAction(name);
    action->setData(count);

    hi = hi->next();
    count++;
  }
}

void TopLevel::backAboutToShow()
{
  QMenu *popup = _backAction->menu();

  popup->clear();
  StackBrowser* b = _stackSelection ? _stackSelection->browser() : 0;
  HistoryItem* hi = b ? b->current() : 0;
  TraceFunction* f;
  QAction* action;

  if (!hi) {
    popup->addAction(tr("(No Stack)"));
    return;
  }

  hi = hi->last();
  if (!hi) {
    popup->addAction(tr("(No previous function)"));
    return;
  }

  int count = 1;
  while (count<GlobalConfig::maxSymbolCount() && hi) {
    f = hi->function();
    if (!f) break;

    QString name = f->prettyName();
    if ((int)name.length()>GlobalConfig::maxSymbolLength())
      name = name.left(GlobalConfig::maxSymbolLength()) + "...";

    //qDebug("back: Adding %s", name.ascii());
    action = popup->addAction(name);
    action->setData(count);

    hi = hi->last();
    count++;
  }
}

void TopLevel::upAboutToShow()
{
  QMenu *popup = _upAction->menu();

  popup->clear();
  StackBrowser* b = _stackSelection ? _stackSelection->browser() : 0;
  HistoryItem* hi = b ? b->current() : 0;
  TraceFunction* f = hi ? hi->function() : 0;
  QAction* action;

  if (!f) {
    popup->addAction(tr("(No Stack)"));
    return;
  }
  f = hi->stack()->caller(f, false);
  if (!f) {
    popup->addAction(tr("(No Function Up)"));
    return;
  }

  int count = 1;
  while (count<GlobalConfig::maxSymbolCount() && f) {
    QString name = f->prettyName();
    if ((int)name.length()>GlobalConfig::maxSymbolLength())
      name = name.left(GlobalConfig::maxSymbolLength()) + "...";

    action = popup->addAction(name);
    action->setData(count);

    f = hi->stack()->caller(f, false);
    count++;
  }
}

void TopLevel::forwardTriggered(QAction* action)
{
  int count = action->data().toInt(0);
  //qDebug("forwardTriggered: %d", count);
  if( count <= 0)
      return;

  StackBrowser* b = _stackSelection ? _stackSelection->browser() : 0;
  if (!b) return;

  while (count>1) {
    b->goForward();
    count--;
  }
  _stackSelection->browserForward();
}

void TopLevel::backTriggered(QAction* action)
{
  int count = action->data().toInt(0);
  //qDebug("backTriggered: %d", count);
  if( count <= 0)
     return;

  StackBrowser* b = _stackSelection ? _stackSelection->browser() : 0;
  if (!b) return;

  while (count>1) {
    b->goBack();
    count--;
  }
  _stackSelection->browserBack();
}

void TopLevel::upTriggered(QAction* action)
{
  int count = action->data().toInt(0);
  //qDebug("upTriggered: %d", count);
  if( count <= 0)
     return;

  StackBrowser* b = _stackSelection ? _stackSelection->browser() : 0;
  HistoryItem* hi = b ? b->current() : 0;
  if (!hi) return;

  TraceFunction* f = hi->function();

  while (count>0 && f) {
    f = hi->stack()->caller(f, false);
    count--;
  }

  //qDebug("upActivated: %s", f ? f->prettyName().ascii() : "??" );
  if (f)
    setFunction(f);
}

void TopLevel::showMessage(const QString& msg, int ms)
{
	if (_statusbar)
		_statusbar->message(msg, ms);
}

void TopLevel::showStatus(const QString& msg, int progress)
{
	static bool msgUpdateNeeded = true;

	if (!_statusbar) return;

	if (msg.isEmpty()) {
	        //reset status
		if (_progressBar) {
		    _statusbar->removeWidget(_progressBar);
		    delete _progressBar;
		    _progressBar = 0;
		}
		_statusbar->clear();
		_progressMsg = msg;
		return;
	}

	if (_progressMsg.isEmpty())
		_progressStart.start();

	if (msg != _progressMsg) {
		_progressMsg = msg;
		msgUpdateNeeded = true;
	}

	// do nothing if last change was less than 0.5 seconds ago
	if (_progressStart.elapsed() < 500)
		return;

	if (!_progressBar) {
		_progressBar = new QProgressBar(_statusbar);
		_progressBar->setMaximumSize(200, _statusbar->height()-4);
		_statusbar->addWidget(_progressBar, 1, true);
		_progressBar->show();
		msgUpdateNeeded = true;
	}

	_progressStart.restart();

	if (msgUpdateNeeded) {
		_statusbar->message(msg);
		msgUpdateNeeded = false;
	}
	_progressBar->setValue(progress);

	// let the progress bar update itself
	qApp->processEvents(QEventLoop::ExcludeUserInputEvents);
}

void TopLevel::loadStart(const QString& filename)
{
    showStatus(QString("Loading %1").arg(filename), 0);
    Logger::_filename = filename;
}

void TopLevel::loadFinished(const QString& msg)
{
    showStatus(QString::null, 0);
    if (!msg.isEmpty())
	showMessage(QString("Error loading %1: %2").arg(_filename).arg(msg),
		    2000);
}

void TopLevel::loadProgress(int progress)
{
    showStatus(QString("Loading %1").arg(_filename), progress);
}

void TopLevel::loadError(int line, const QString& msg)
{
    qCritical() << "Loading" << _filename.ascii() << ":" << line << ": " << msg.ascii();
}

void TopLevel::loadWarning(int line, const QString& msg)
{
    qWarning() << "Loading" << _filename.ascii() << ":" << line << ": " << msg.ascii();
}

#include "toplevel.moc"
