/*
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#pragma once

#include <folly/concurrency/UnboundedQueue.h>
#include <folly/executors/task_queue/BlockingQueue.h>
#include <folly/synchronization/LifoSem.h>

namespace folly {

template <class T, class Semaphore = folly::LifoSem>
class UnboundedBlockingQueue : public BlockingQueue<T> {
 public:
  explicit UnboundedBlockingQueue(
      const typename Semaphore::Options& semaphoreOptions = {})
      : sem_(semaphoreOptions) {}

  BlockingQueueAddResult add(T item) override {
    queue_.enqueue(std::move(item));
    return sem_.post();
  }

  T take() override {
    sem_.wait();
    return queue_.dequeue();
  }

  folly::Optional<T> try_take_for(std::chrono::milliseconds time) override {
    if (!sem_.try_wait_for(time)) {
      return folly::none;
    }
    return queue_.dequeue();
  }

  size_t size() override { return sem_.valueGuess(); }

 private:
  Semaphore sem_;
  UMPMCQueue<T, false, 6> queue_;
};

} // namespace folly
