"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const node_path_1 = require("node:path");
const node_fs_1 = require("node:fs");
const expect_1 = __importDefault(require("expect"));
const findNvim_1 = require("./findNvim");
const parseVersion = findNvim_1.exportsForTesting.parseVersion;
const compareVersions = findNvim_1.exportsForTesting.compareVersions;
const normalizePath = findNvim_1.exportsForTesting.normalizePath;
describe('findNvim', () => {
    const testDir = (0, node_path_1.join)(process.cwd(), 'test-dir');
    const nvimExecutablePath = normalizePath((0, node_path_1.join)(testDir, process.platform === 'win32' ? 'nvim.exe' : 'nvim'));
    before(() => {
        (0, node_fs_1.mkdirSync)(testDir, { recursive: true });
        (0, node_fs_1.writeFileSync)(nvimExecutablePath, 'fake-nvim-executable');
    });
    after(() => {
        (0, node_fs_1.rmSync)(testDir, { recursive: true, force: true });
    });
    it('parseVersion()', () => {
        (0, expect_1.default)(parseVersion('0.5.0-dev+1357-g192f89ea1')).toEqual([0, 5, 0, 'dev+1357-g192f89ea1']);
        (0, expect_1.default)(parseVersion('0.5.0-dev+1357-g192f89ea1-Homebrew')).toEqual([
            0,
            5,
            0,
            'dev+1357-g192f89ea1-Homebrew',
        ]);
        (0, expect_1.default)(parseVersion('0.9.1')).toEqual([0, 9, 1, 'zzz']);
        // Failure modes:
        (0, expect_1.default)(() => parseVersion(42)).toThrow(TypeError);
        (0, expect_1.default)(parseVersion('x.y.z')).toEqual(undefined);
        (0, expect_1.default)(parseVersion('1.y.z')).toEqual(undefined);
        (0, expect_1.default)(parseVersion('1.2.z')).toEqual(undefined);
        (0, expect_1.default)(parseVersion('x.2.3')).toEqual(undefined);
        (0, expect_1.default)(parseVersion('1.y.3')).toEqual(undefined);
    });
    it('compareVersions()', () => {
        (0, expect_1.default)(compareVersions('0.3.0', '0.3.0')).toBe(0);
        (0, expect_1.default)(compareVersions('0.3.0', '0.3.1')).toBe(-1);
        (0, expect_1.default)(compareVersions('0.3.1', '0.3.0')).toBe(1);
        (0, expect_1.default)(compareVersions('0.3.0-abc', '0.3.0-dev-420')).toBe(-1);
        (0, expect_1.default)(compareVersions('0.3.0', '0.3.0-dev-658+g06694203e-Homebrew')).toBe(1);
        (0, expect_1.default)(compareVersions('0.3.0-dev-658+g06694203e-Homebrew', '0.3.0')).toBe(-1);
        (0, expect_1.default)(compareVersions('0.3.0-dev-658+g06694203e-Homebrew', '0.3.0-dev-658+g06694203e-Homebrew')).toBe(0);
        (0, expect_1.default)(compareVersions('0.3.0-dev-658+g06694203e-Homebrew', '0.3.0-dev-659+g06694203e-Homebrew')).toBe(-1);
        (0, expect_1.default)(compareVersions('0.3.0-dev-659+g06694203e-Homebrew', '0.3.0-dev-658+g06694203e-Homebrew')).toBe(1);
        // Failure modes:
        (0, expect_1.default)(compareVersions('0.3.0', 'nonsense')).toBe(1);
        (0, expect_1.default)(() => compareVersions('nonsense', '0.3.0')).toThrow('Invalid version: "nonsense"');
        (0, expect_1.default)(() => compareVersions('nonsense', 'nonsense')).toThrow('Invalid version: "nonsense"');
        (0, expect_1.default)(() => compareVersions(undefined, undefined)).toThrow('Invalid version format: not a string');
    });
    /** Asserts that >=1 nvim binaries were found. */
    function assertOneOrMore(nvimRes) {
        (0, expect_1.default)(nvimRes).toEqual({
            matches: expect_1.default.any(Array),
            invalid: expect_1.default.any(Array),
        });
        (0, expect_1.default)(nvimRes.matches.length).toBeGreaterThan(0);
        (0, expect_1.default)(nvimRes.matches[0]).toEqual({
            nvimVersion: expect_1.default.any(String),
            path: expect_1.default.any(String),
            buildType: expect_1.default.any(String),
            luaJitVersion: expect_1.default.any(String),
            error: undefined,
        });
        (0, expect_1.default)(nvimRes.invalid.length).toEqual(0);
    }
    it('gets Nvim satisfying given min version', () => {
        const nvimRes = (0, findNvim_1.findNvim)({ minVersion: '0.3.0', orderBy: 'desc' });
        assertOneOrMore(nvimRes);
    });
    it('gets Nvim without specified min version', () => {
        const nvimRes = (0, findNvim_1.findNvim)();
        assertOneOrMore(nvimRes);
    });
    it('collects invalid matches separately', () => {
        const nvimRes = (0, findNvim_1.findNvim)({ minVersion: '9999.0.0' });
        (0, expect_1.default)(nvimRes).toEqual({
            matches: [],
            invalid: expect_1.default.any(Array),
        });
        (0, expect_1.default)(nvimRes.matches.length).toEqual(0);
        (0, expect_1.default)(nvimRes.invalid.length).toBeGreaterThan(0);
        (0, expect_1.default)(nvimRes.invalid[0]).toEqual({
            nvimVersion: expect_1.default.any(String),
            path: expect_1.default.any(String),
            buildType: expect_1.default.any(String),
            luaJitVersion: expect_1.default.any(String),
            error: undefined,
        });
    });
    it('stops searching on first match when firstMatch is True', () => {
        const nvimRes = (0, findNvim_1.findNvim)({ minVersion: '0.3.0', firstMatch: true });
        (0, expect_1.default)(nvimRes).toEqual({
            matches: expect_1.default.any(Array),
            invalid: expect_1.default.any(Array),
        });
        (0, expect_1.default)(nvimRes.matches.length).toEqual(1);
        (0, expect_1.default)(nvimRes.matches[0]).toEqual({
            nvimVersion: expect_1.default.any(String),
            path: expect_1.default.any(String),
            buildType: expect_1.default.any(String),
            luaJitVersion: expect_1.default.any(String),
            error: undefined,
        });
    });
    it('searches in additional custom paths', () => {
        const customPaths = [
            (0, node_path_1.join)(process.cwd(), 'package.json'),
            '/custom/path/to/nvim',
            '/another/custom/path',
        ].map(normalizePath);
        const nvimRes = (0, findNvim_1.findNvim)({ paths: customPaths });
        (0, expect_1.default)(nvimRes.matches.length).toBeGreaterThanOrEqual(1);
        (0, expect_1.default)(nvimRes.invalid.length).toBe(3);
        const invalidPaths = nvimRes.invalid.map(i => i.path);
        (0, expect_1.default)(invalidPaths).toEqual(customPaths);
    });
    it('searches in additional custom dirs', () => {
        const customDirs = [testDir, '/non/existent/dir'].map(normalizePath);
        const nvimRes = (0, findNvim_1.findNvim)({ dirs: customDirs });
        (0, expect_1.default)(nvimRes.matches.length).toBeGreaterThanOrEqual(1);
        (0, expect_1.default)(nvimRes.invalid.length).toBe(1);
        (0, expect_1.default)(nvimRes.invalid[0].path).toBe(nvimExecutablePath);
    });
});
