--!A cross-platform build utility based on Lua
--
-- Licensed under the Apache License, Version 2.0 (the "License");
-- you may not use this file except in compliance with the License.
-- You may obtain a copy of the License at
--
--     http://www.apache.org/licenses/LICENSE-2.0
--
-- Unless required by applicable law or agreed to in writing, software
-- distributed under the License is distributed on an "AS IS" BASIS,
-- WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
-- See the License for the specific language governing permissions and
-- limitations under the License.
--
-- Copyright (C) 2015-present, TBOOX Open Source Group.
--
-- @author      ruki
-- @file        main.lua
--

-- imports
import("core.base.option")
import("private.action.clean.remove_files")
import("private.cache.build_cache")

-- the builtin clean main entry
function main(target)

    -- is phony?
    if target:is_phony() then
        return
    end

    -- remove the target file
    local targetfile = target:targetfile()
    remove_files(targetfile)

    -- remove import library of the target file
    -- @see https://github.com/xmake-io/xmake/issues/3052
    if target:is_shared() then
        if target:is_plat("windows") then
            local implibfile = target:artifactfile("implib")
            local expfile = path.join(path.directory(implibfile), path.basename(targetfile) .. ".exp")
            if os.isfile(implibfile) then
                remove_files(implibfile)
            end
            if os.isfile(expfile) then
                remove_files(expfile)
            end
        end

        if target:is_plat("mingw") then
            local implibfile = target:artifactfile("implib")
            if os.isfile(implibfile) then
                remove_files(implibfile)
            end
        end
    end

    -- remove the symbol file
    remove_files(target:symbolfile())

    -- remove the c/c++ precompiled header file
    remove_files(target:pcoutputfile("c"))
    remove_files(target:pcoutputfile("cxx"))

    -- remove the clean files
    remove_files(target:get("cleanfiles"))

    -- remove all?
    if option.get("all") then

        -- remove config files
        local _, configfiles = target:configfiles()
        remove_files(configfiles)

        -- remove all dependent files for each platform
        remove_files(target:dependir({root = true}))

        -- remove all object files for each platform
        remove_files(target:objectdir({root = true}))

        -- remove all autogen files for each platform
        remove_files(target:autogendir({root = true}))

        -- clean build cache
        build_cache.clean()
    else

        -- remove dependent files for the current platform
        remove_files(target:dependir())

        -- remove object files for the current platform
        remove_files(target:objectdir())

        -- remove autogen files for the current platform
        remove_files(target:autogendir())
    end
end

