#!/usr/bin/env python3
#BEGIN_LEGAL
#
#Copyright (c) 2024 Intel Corporation
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#  
#END_LEGAL
import argparse
from cffi import FFI
from pathlib import Path
import sys

# Define the root directory for the XED library
sys.path.append(str(Path(__file__).parents[1] / 'gen'))

# Import strings containing function and type definitions for XED
import xedpy_types  # Assumed to contain cdef_string for types (TBD autogenerated)
import xedpy_funcs  # Assumed to contain cdef_string for functions (TBD autogenerated)

xedEnum = int   # Python type hint for XED C enumerations

class XedPy:
    BUFFER_SIZE = 700  # Constant for string buffer size

    def __init__(self, lib_path: Path):
        self.ffi = FFI()
        # Load the XED library and initialize FFI XED library definitions
        self.lib = self.load_xed_library(lib_path)
        # Initialize the XED decoder
        self.xedd = self.ffi.new('xed_decoded_inst_t*')
        self.initialize_xed()
        self.disas_syntax_default: xedEnum = self.lib.XED_SYNTAX_INTEL # Default to Intel Syntax

    def load_xed_library(self, lib_path: Path) -> FFI.dlopen:
        """Load the XED shared library using FFI.
        
        Args:
            lib_path (Path): The path to the XED shared library.
        
        Returns:
            FFI.dlopen: The loaded library object.
        
        Raises:
            FileNotFoundError: If the shared library is not found at the given path.
        """
        def print_string_with_line_numbers(s: str):
            '''Useful for identifying the line number in error messages generated by ffi.cdef().
               Use to debug the `xed_cffi_def_string` string'''
            for index, line in enumerate(s.split('\n'), start=1):
                print(f"{index}: {line}")

        if not lib_path.exists():
            raise FileNotFoundError(f"XED library not found at {lib_path}")
        # Declare the XED functions, structs and enums
        xed_cffi_def_string = xedpy_types.cdef_string + xedpy_funcs.cdef_string
        # print_string_with_line_numbers(xed_cffi_def_string) # Uncomment for CFFI def string debugging
        self.ffi.cdef(xed_cffi_def_string)
        # Load the shared library and return it
        return self.ffi.dlopen(str(lib_path.resolve()))

    def initialize_xed(self) -> None:
        """Initialize XED and set up the decoding environment."""
        # Initialize XED tables (must be done before decoding instructions)
        self.lib.xed_tables_init_py()
        # Set the decoding mode to 64-bit
        dstate = self.ffi.new('xed_state_t*')
        self.lib.xed_state_init2_py(dstate, self.lib.XED_MACHINE_MODE_LONG_64, self.lib.XED_ADDRESS_WIDTH_64b)
        self.lib.xed_operand_values_set_mode_py(self.xedd, dstate)

    def get_xed_enum_val(self, enum: str) -> xedEnum:
        return getattr(self.lib, enum, None)

    def decode_instruction(self, instruction_bytes: bytes) -> None:
        """Decode a given instruction byte sequence using XED.
        
        Args:
            instruction_bytes (bytes): The raw instruction bytes to be decoded.
        
        Raises:
            RuntimeError: If decoding fails.
        """
        print(f'[DECODED BYTES] {instruction_bytes.hex()}')
        err: xedEnum = self.lib.xed_decode_py(self.xedd, instruction_bytes, len(instruction_bytes))
        if err != self.lib.XED_ERROR_NONE:
            error_string: str = self.ffi.string(self.lib.xed_error_enum_t2str_py(err)).decode('utf-8')
            raise ValueError(f"XED ERROR: {error_string}")

    def print_decoded_instruction_info(self) -> None:
        """Print detailed information about the decoded instruction."""
        PAD = 12
        # Get and print the instruction ICLASS
        iclass: xedEnum = self.lib.xed_decoded_inst_get_iclass_py(self.xedd)
        iclass_str: str = self.ffi.string(self.lib.xed_iclass_enum_t2str_py(iclass)).decode('utf-8')
        print(f'{"ICLASS":<{PAD}}: {iclass_str}')

        # Get and print the instruction category
        category: xedEnum = self.lib.xed_decoded_inst_get_category_py(self.xedd)
        category_str: str = self.ffi.string(self.lib.xed_category_enum_t2str_py(category)).decode('utf-8')
        print(f'{"CATEGORY":<{PAD}}: {category_str}')

        # Get and print the instruction extension
        extension: xedEnum = self.lib.xed_decoded_inst_get_extension_py(self.xedd)
        extension_str: str = self.ffi.string(self.lib.xed_extension_enum_t2str_py(extension)).decode('utf-8')
        print(f'{"EXTENSION":<{PAD}}: {extension_str}')

        # Get and print the instruction IFORM
        iform: xedEnum = self.lib.xed_decoded_inst_get_iform_enum_py(self.xedd)
        iform_str: str = self.ffi.string(self.lib.xed_iform_enum_t2str_py(iform)).decode('utf-8')
        print(f'{"IFORM":<{PAD}}: {iform_str}')

        # Get and print the instruction ISA set
        isa_set: xedEnum = self.lib.xed_decoded_inst_get_isa_set_py(self.xedd)
        isa_set_str: str = self.ffi.string(self.lib.xed_isa_set_enum_t2str_py(isa_set)).decode('utf-8')
        print(f'{"ISA_SET":<{PAD}}: {isa_set_str}')

        # Check if it's an APX instruction
        apx: bool = self.lib.xed_classify_apx_py(self.xedd)
        if apx:
            print('[APX instruction]')

        # Allocate a buffer for the instruction format dump
        buffer = self.ffi.new(f'char[{self.BUFFER_SIZE}]')
        self.lib.xed_decoded_inst_dump_xed_format_py(self.xedd, buffer, self.BUFFER_SIZE, 0)
        print('\n==== XED Operands: ====')
        print(self.ffi.string(buffer).decode(), end='\n\n')
    
    def disassemble(self, syntax: xedEnum = None) -> str:
        """return the assembly string of a decoded xedd instruction"""
        buffer = self.ffi.new(f'char[{self.BUFFER_SIZE}]')
        # Init the xed_print_info_t struct
        pi = self.ffi.new('xed_print_info_t*')
        self.lib.xed_init_print_info_py(pi)
        pi.p = self.xedd
        pi.blen = self.BUFFER_SIZE
        pi.buf = buffer
        if syntax is None:
            syntax = self.disas_syntax_default    
        pi.syntax = syntax

        # disassemble
        ok = self.lib.xed_format_generic_py(pi)
        if not ok:
            raise ValueError(f"XED disassemble ERROR")
        
        return self.ffi.string(buffer).decode()

def parse_arguments() -> argparse.Namespace:
    parser = argparse.ArgumentParser(description='Python example using XED via CFFI')
    parser.add_argument('--xed-lib', 
                        dest='lib_path', 
                        help='Path to XED shared object library.', 
                        type=Path,
                        default=None)
    args = parser.parse_args()

    # Default to the expected library path
    if args.lib_path is None:    
        xed_root: Path = Path(__file__).parents[2].resolve()
        args.lib_path = xed_root / 'obj/xed.dll'
        if not args.lib_path.exists():
            args.lib_path = args.lib_path.with_name('libxed.so')

    return args

def main() -> None:
    """Main function to handle library loading, instruction decoding, and output."""
    args = parse_arguments()
    try:
        xed = XedPy(args.lib_path)

        # Define the raw instruction bytes to be decoded
        instruction_bytes = bytes([0x66, 0xd5, 0x80, 0x12, 0x4C, 0x83, 0x00])

        # Decode the instruction
        xed.decode_instruction(instruction_bytes)

        # Print detailed information about the decoded instruction
        xed.print_decoded_instruction_info()
        
        for syntax in ['INTEL', 'ATT']:
            disasm = xed.disassemble(xed.get_xed_enum_val(f'XED_SYNTAX_{syntax}'))
            print(f'Disassembly ({syntax}): {disasm}')
    except Exception as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)

if __name__ == '__main__':
    main()