/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 */

#include "./follower.hpp"
#include "./ipcs/file-monitor.hpp"
#include "./ipcs/unix-socket-server.hpp"
#include "./writer.hpp"
#include <thread>
#include <utility>

namespace sh4lt {

Follower::Follower(const fs::path& sockpath,
                   Reader::onData cb,
                   Reader::onServerConnected osc,
                   Reader::onServerDisconnected osd,
                   logger::Logger::ptr log)
    : log_(log),
      path_(sockpath.string()),
      on_data_cb_(cb),
      osc_(osc),
      osd_(osd),
      reader_(fileMonitor::is_unix_socket(path_, log_.get())
                  ? std::make_unique<Reader>(
                        path_, on_data_cb_, osc_, [&]() { on_server_disconnected(); }, log_.get())
                  : nullptr) {
  if (!reader_ || !(*reader_.get()))
    monitor_ = std::async(std::launch::async, [this]() { monitor(); });
}

Follower::~Follower() {
  is_destructing_ = true;
  quit_.store(true);
  if (monitor_.valid()) monitor_.get();
}

void Follower::monitor() {
  auto do_sleep = true;
  // give the change the reader to fail twice before cleaning dead sh4lt:
  // auto successive_fail = 0;
  while (!quit_.load()) {
    if (fileMonitor::is_unix_socket(path_, log_.get())) {
      do_sleep = false;
      // log_->debug("file detected, creating reader");
      reader_ = std::make_unique<Reader>(
          path_, on_data_cb_, osc_, [&]() { on_server_disconnected(); }, log_.get());
      if (*reader_.get()) {
        quit_.store(true);
      } else {
        reader_.reset();
        // log_->debug("file % exists but reader failed", path_);
        // if (1 == successive_fail) {
        //   if(!force_sockserv_cleaning(path_, log_.get()))
        //     log_->warning("follower shmpath is not dead sh4lt that can be
        //     cleaned");
        //   else
        //     log_->debug("sh4lt follower detected and cleaned a possible
        //     dead sh4lt: %",
        //                 path_);
        //   successive_fail = 0;
        // } else { ++successive_fail; }
      }
    }
    if (do_sleep) std::this_thread::sleep_for(std::chrono::milliseconds(kMonitorTimeMs));
  }  // end while
  quit_.store(true);
}

void Follower::on_server_disconnected() {
  log_->debug("follower on_server_disconnected");
  // starting monitor
  if (!is_destructing_) {
    quit_.store(false);
    monitor_ = std::async(std::launch::async, [this]() { monitor(); });
  }
  // calling user callback
  if (osd_) osd_();
}

}  // namespace sh4lt
