# Annotated Doc

Document parameters, class attributes, return types, and variables inline, with `Annotated`.

<a href="https://github.com/fastapi/annotated-doc/actions?query=workflow%3ATest+event%3Apush+branch%3Amain" target="_blank">
    <img src="https://github.com/fastapi/annotated-doc/actions/workflows/test.yml/badge.svg?event=push&branch=main" alt="Test">
</a>
<a href="https://coverage-badge.samuelcolvin.workers.dev/redirect/fastapi/annotated-doc" target="_blank">
    <img src="https://coverage-badge.samuelcolvin.workers.dev/fastapi/annotated-doc.svg" alt="Coverage">
</a>
<a href="https://pypi.org/project/annotated-doc" target="_blank">
    <img src="https://img.shields.io/pypi/v/annotated-doc?color=%2334D058&label=pypi%20package" alt="Package version">
</a>
<a href="https://pypi.org/project/annotated-doc" target="_blank">
    <img src="https://img.shields.io/pypi/pyversions/annotated-doc.svg?color=%2334D058" alt="Supported Python versions">
</a>

## Installation

```bash
pip install annotated-doc
```

Or with `uv`:

```Python
uv add annotated-doc
```

## Usage

Import `Doc` and pass a single literal string with the documentation for the specific parameter, class attribute, return type, or variable.

For example, to document a parameter `name` in a function `hi` you could do:

```Python
from typing import Annotated

from annotated_doc import Doc

def hi(name: Annotated[str, Doc("Who to say hi to")]) -> None:
    print(f"Hi, {name}!")
```

You can also use it to document class attributes:

```Python
from typing import Annotated

from annotated_doc import Doc

class User:
    name: Annotated[str, Doc("The user's name")]
    age: Annotated[int, Doc("The user's age")]
```

The same way, you could document return types and variables, or anything that could have a type annotation with `Annotated`.

## Who Uses This

`annotated-doc` was made for:

* [FastAPI](https://fastapi.tiangolo.com/)
* [Typer](https://typer.tiangolo.com/)
* [SQLModel](https://sqlmodel.tiangolo.com/)
* [Asyncer](https://asyncer.tiangolo.com/)

`annotated-doc` is supported by [griffe-typingdoc](https://github.com/mkdocstrings/griffe-typingdoc), which powers reference documentation like the one in the [FastAPI Reference](https://fastapi.tiangolo.com/reference/).

## Reasons not to use `annotated-doc`

You are already comfortable with one of the existing docstring formats, like:

* Sphinx
* numpydoc
* Google
* Keras

Your team is already comfortable using them.

You prefer having the documentation about parameters all together in a docstring, separated from the code defining them.

You care about a specific set of users, using one specific editor, and that editor already has support for the specific docstring format you use.

## Reasons to use `annotated-doc`

* No micro-syntax to learn for newcomers, it’s **just Python** syntax.
* **Editing** would be already fully supported by default by any editor (current or future) supporting Python syntax, including syntax errors, syntax highlighting, etc.
* **Rendering** would be relatively straightforward to implement by static tools (tools that don't need runtime execution), as the information can be extracted from the AST they normally already create.
* **Deduplication of information**: the name of a parameter would be defined in a single place, not duplicated inside of a docstring.
* **Elimination** of the possibility of having **inconsistencies** when removing a parameter or class variable and **forgetting to remove** its documentation.
* **Minimization** of the probability of adding a new parameter or class variable and **forgetting to add its documentation**.
* **Elimination** of the possibility of having **inconsistencies** between the **name** of a parameter in the **signature** and the name in the docstring when it is renamed.
* **Access** to the documentation string for each symbol at **runtime**, including existing (older) Python versions.
* A more formalized way to document other symbols, like type aliases, that could use Annotated.
* **Support** for apps using FastAPI, Typer and others.
* **AI Accessibility**: AI tools will have an easier way understanding each parameter as the distance from documentation to parameter is much closer.

## History

I ([@tiangolo](https://github.com/tiangolo)) originally wanted for this to be part of the Python standard library (in [PEP 727](https://peps.python.org/pep-0727/)), but the proposal was withdrawn as there was a fair amount of negative feedback and opposition.

The conclusion was that this was better done as an external effort, in a third-party library.

So, here it is, with a simpler approach, as a third-party library, in a way that can be used by others, starting with FastAPI and friends.

## License

This project is licensed under the terms of the MIT license.
