/** @file
  Provides the functions to submit Scsi commands defined in SCSI-2 specification for SCSI devices.

  This library class provides the functions to submit SCSI commands defined in SCSI-2 specification
  for hard drive, CD and DVD devices that are the most common SCSI boot targets used by UEFI platforms.
  This library class depends on SCSI I/O Protocol defined in UEFI Specification and SCSI-2 industry standard.

Copyright (c) 2006 - 2019, Intel Corporation. All rights reserved.<BR>
SPDX-License-Identifier: BSD-2-Clause-Patent

**/

#ifndef __SCSI_LIB_H__
#define __SCSI_LIB_H__

#include <Protocol/ScsiIo.h>

/**
  Execute Test Unit Ready SCSI command on a specific SCSI target.

  Executes the Test Unit Ready command on the SCSI target specified by ScsiIo.
  If Timeout is zero, then this function waits indefinitely for the command to complete.
  If Timeout is greater than zero, then the command is executed and will timeout after Timeout 100 ns units.
  If ScsiIo is NULL, then ASSERT().
  If SenseDataLength is NULL, then ASSERT().
  If HostAdapterStatus is NULL, then ASSERT().
  If TargetStatus is NULL, then ASSERT().

  If SenseDataLength is non-zero and SenseData is not NULL, SenseData must meet buffer
  alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise EFI_INVALID_PARAMETER
  gets returned.

  @param[in]     ScsiIo             A pointer to the SCSI I/O Protocol instance
                                    for the specific SCSI target.
  @param[in]     Timeout            The timeout in 100 ns units to use for the execution
                                    of this SCSI Request Packet. A Timeout value of
                                    zero means that this function will wait indefinitely
                                    for the SCSI Request Packet to execute. If Timeout
                                    is greater than zero, then this function will return
                                    EFI_TIMEOUT if the time required to execute the SCSI
                                    Request Packet is greater than Timeout.
  @param[in, out] SenseData         A pointer to sense data that was generated by
                                    the execution of the SCSI Request Packet. This
                                    buffer must be allocated by the caller.
                                    If SenseDataLength is 0, then this parameter is
                                    optional and may be NULL.
  @param[in, out] SenseDataLength   On input, a pointer to the length in bytes of
                                    the SenseData buffer. On output, a pointer to
                                    the number of bytes written to the SenseData buffer.
  @param[out]     HostAdapterStatus The status of the SCSI Host Controller that produces
                                    the SCSI bus containing the SCSI target specified by
                                    ScsiIo when the SCSI Request Packet was executed.
                                    See the EFI SCSI I/O Protocol in the UEFI Specification
                                    for details on the possible return values.
  @param[out]     TargetStatus      The status returned by the SCSI target specified
                                    by ScsiIo when the SCSI Request Packet was executed
                                    on the SCSI Host Controller. See the EFI SCSI I/O
                                    Protocol in the UEFI Specification for details on
                                    the possible return values.

  @retval EFI_SUCCESS               The command was executed successfully.
                                    See HostAdapterStatus, TargetStatus, SenseDataLength,
                                    and SenseData in that order for additional status
                                    information.
  @retval EFI_NOT_READY             The SCSI Request Packet could not be sent because
                                    there are too many SCSI Command Packets already
                                    queued. The SCSI Request Packet was not sent, so
                                    no additional status information is available.
                                    The caller may retry again later.
  @retval EFI_DEVICE_ERROR          A device error occurred while attempting to send
                                    SCSI Request Packet.  See HostAdapterStatus,
                                    TargetStatus, SenseDataLength, and SenseData in that
                                    order for additional status information.
  @retval EFI_UNSUPPORTED           The command described by the SCSI Request Packet
                                    is not supported by the SCSI initiator(i.e., SCSI
                                    Host Controller). The SCSI Request Packet was not
                                    sent, so no additional status information is available.
  @retval EFI_TIMEOUT               A timeout occurred while waiting for the SCSI Request
                                    Packet to execute.  See HostAdapterStatus, TargetStatus,
                                    SenseDataLength, and SenseData in that order for
                                    additional status information.
  @retval EFI_INVALID_PARAMETER     The contents of the SCSI Request Packet are invalid.

**/
EFI_STATUS
EFIAPI
ScsiTestUnitReadyCommand (
  IN     EFI_SCSI_IO_PROTOCOL  *ScsiIo,
  IN     UINT64                Timeout,
  IN OUT VOID                  *SenseData   OPTIONAL,
  IN OUT UINT8                 *SenseDataLength,
  OUT UINT8                    *HostAdapterStatus,
  OUT UINT8                    *TargetStatus
  );

/**
  Execute Inquiry SCSI command on a specific SCSI target.

  Executes the Inquiry command on the SCSI target specified by ScsiIo.
  If Timeout is zero, then this function waits indefinitely for the command to complete.
  If Timeout is greater than zero, then the command is executed and will timeout after Timeout 100 ns units.
  If ScsiIo is NULL, then ASSERT().
  If SenseDataLength is NULL, then ASSERT().
  If HostAdapterStatus is NULL, then ASSERT().
  If TargetStatus is NULL, then ASSERT().
  If InquiryDataLength is NULL, then ASSERT().

  If SenseDataLength is non-zero and SenseData is not NULL, SenseData must meet buffer
  alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise EFI_INVALID_PARAMETER
  gets returned.

  If InquiryDataLength is non-zero and InquiryDataBuffer is not NULL, InquiryDataBuffer
  must meet buffer alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise
  EFI_INVALID_PARAMETER gets returned.

  @param[in]      ScsiIo                 A pointer to the SCSI I/O Protocol instance
                                         for the specific SCSI target.
  @param[in]      Timeout                The timeout in 100 ns units to use for the
                                         execution of this SCSI Request Packet. A Timeout
                                         value of zero means that this function will wait
                                         indefinitely for the SCSI Request Packet to execute.
                                         If Timeout is greater than zero, then this function
                                         will return EFI_TIMEOUT if the time required to
                                         execute the SCSI Request Packet is greater than Timeout.
  @param[in, out] SenseData              A pointer to sense data that was generated
                                         by the execution of the SCSI Request Packet.
                                         This buffer must be allocated by the caller.
                                         If SenseDataLength is 0, then this parameter
                                         is optional and may be NULL.
  @param[in, out] SenseDataLength        On input, the length in bytes of the SenseData buffer.
                                         On output, the number of bytes written to the SenseData buffer.
  @param[out]     HostAdapterStatus      The status of the SCSI Host Controller that
                                         produces the SCSI bus containing the SCSI
                                         target specified by ScsiIo when the SCSI
                                         Request Packet was executed.  See the EFI
                                         SCSI I/O Protocol in the UEFI Specification
                                         for details on the possible return values.
  @param[out]     TargetStatus           The status returned by the SCSI target specified
                                         by ScsiIo when the SCSI Request Packet was
                                         executed on the SCSI Host Controller.
                                         See the EFI SCSI I/O Protocol in the UEFI
                                         Specification for details on the possible
                                         return values.
  @param[in, out] InquiryDataBuffer      A pointer to inquiry data that was generated
                                         by the execution of the SCSI Request Packet.
                                         This buffer must be allocated by the caller.
                                         If InquiryDataLength is 0, then this parameter
                                         is optional and may be NULL.
  @param[in, out] InquiryDataLength      On input, a pointer to the length in bytes
                                         of the InquiryDataBuffer buffer.
                                         On output, a pointer to the number of bytes
                                         written to the InquiryDataBuffer buffer.
  @param[in]      EnableVitalProductData If TRUE, then the supported vital product
                                         data is returned in InquiryDataBuffer.
                                         If FALSE, then the standard inquiry data is
                                         returned in InquiryDataBuffer.

  @retval EFI_SUCCESS                    The command was executed successfully. See HostAdapterStatus,
                                         TargetStatus, SenseDataLength, and SenseData in that order
                                         for additional status information.
  @retval EFI_BAD_BUFFER_SIZE            The SCSI Request Packet was executed, but the entire
                                         InquiryDataBuffer could not be transferred. The actual
                                         number of bytes transferred is returned in InquiryDataLength.
  @retval EFI_NOT_READY                  The SCSI Request Packet could not be sent because there
                                         are too many SCSI Command Packets already queued.
                                         The SCSI Request Packet was not sent, so no additional
                                         status information is available. The caller may retry again later.
  @retval EFI_DEVICE_ERROR               A device error occurred while attempting to send SCSI
                                         Request Packet.  See HostAdapterStatus, TargetStatus,
                                         SenseDataLength, and SenseData in that order for additional
                                         status information.
  @retval EFI_UNSUPPORTED                The command described by the SCSI Request Packet is not
                                         supported by the SCSI initiator(i.e., SCSI  Host Controller).
                                         The SCSI Request Packet was not sent, so no additional
                                         status information is available.
  @retval EFI_TIMEOUT                    A timeout occurred while waiting for the SCSI Request
                                         Packet to execute.  See HostAdapterStatus, TargetStatus,
                                         SenseDataLength, and SenseData in that order for
                                         additional status information.
  @retval EFI_INVALID_PARAMETER          The contents of the SCSI Request Packet are invalid.

**/
EFI_STATUS
EFIAPI
ScsiInquiryCommand (
  IN     EFI_SCSI_IO_PROTOCOL  *ScsiIo,
  IN     UINT64                Timeout,
  IN OUT VOID                  *SenseData   OPTIONAL,
  IN OUT UINT8                 *SenseDataLength,
  OUT UINT8                    *HostAdapterStatus,
  OUT UINT8                    *TargetStatus,
  IN OUT VOID                  *InquiryDataBuffer     OPTIONAL,
  IN OUT UINT32                *InquiryDataLength,
  IN     BOOLEAN               EnableVitalProductData
  );

/**
  Execute Inquiry SCSI command on a specific SCSI target.

  Executes the Inquiry command on the SCSI target specified by ScsiIo.
  If Timeout is zero, then this function waits indefinitely for the command to complete.
  If Timeout is greater than zero, then the command is executed and will timeout after Timeout 100 ns units.
  If ScsiIo is NULL, then ASSERT().
  If SenseDataLength is NULL, then ASSERT().
  If HostAdapterStatus is NULL, then ASSERT().
  If TargetStatus is NULL, then ASSERT().
  If InquiryDataLength is NULL, then ASSERT().

  If SenseDataLength is non-zero and SenseData is not NULL, SenseData must meet buffer
  alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise EFI_INVALID_PARAMETER
  gets returned.

  If InquiryDataLength is non-zero and InquiryDataBuffer is not NULL, InquiryDataBuffer
  must meet buffer alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise
  EFI_INVALID_PARAMETER gets returned.

  @param[in]      ScsiIo                 A pointer to the SCSI I/O Protocol instance
                                         for the specific SCSI target.
  @param[in]      Timeout                The timeout in 100 ns units to use for the
                                         execution of this SCSI Request Packet. A Timeout
                                         value of zero means that this function will wait
                                         indefinitely for the SCSI Request Packet to execute.
                                         If Timeout is greater than zero, then this function
                                         will return EFI_TIMEOUT if the time required to
                                         execute the SCSI Request Packet is greater than Timeout.
  @param[in, out] SenseData              A pointer to sense data that was generated
                                         by the execution of the SCSI Request Packet.
                                         This buffer must be allocated by the caller.
                                         If SenseDataLength is 0, then this parameter
                                         is optional and may be NULL.
  @param[in, out] SenseDataLength        On input, the length in bytes of the SenseData buffer.
                                         On output, the number of bytes written to the SenseData buffer.
  @param[out]     HostAdapterStatus      The status of the SCSI Host Controller that
                                         produces the SCSI bus containing the SCSI
                                         target specified by ScsiIo when the SCSI
                                         Request Packet was executed.  See the EFI
                                         SCSI I/O Protocol in the UEFI Specification
                                         for details on the possible return values.
  @param[out]     TargetStatus           The status returned by the SCSI target specified
                                         by ScsiIo when the SCSI Request Packet was
                                         executed on the SCSI Host Controller.
                                         See the EFI SCSI I/O Protocol in the UEFI
                                         Specification for details on the possible
                                         return values.
  @param[in, out] InquiryDataBuffer      A pointer to inquiry data that was generated
                                         by the execution of the SCSI Request Packet.
                                         This buffer must be allocated by the caller.
                                         If InquiryDataLength is 0, then this parameter
                                         is optional and may be NULL.
  @param[in, out] InquiryDataLength      On input, a pointer to the length in bytes
                                         of the InquiryDataBuffer buffer.
                                         On output, a pointer to the number of bytes
                                         written to the InquiryDataBuffer buffer.
  @param[in]      EnableVitalProductData If TRUE, then the supported vital product
                                         data for the PageCode is returned in InquiryDataBuffer.
                                         If FALSE, then the standard inquiry data is
                                         returned in InquiryDataBuffer and PageCode is ignored.
  @param[in]      PageCode               The page code of the vital product data.
                                         It's ignored if EnableVitalProductData is FALSE.

  @retval EFI_SUCCESS                    The command executed successfully. See HostAdapterStatus,
                                         TargetStatus, SenseDataLength, and SenseData in that order
                                         for additional status information.
  @retval EFI_BAD_BUFFER_SIZE            The SCSI Request Packet was executed, but the entire
                                         InquiryDataBuffer could not be transferred. The actual
                                         number of bytes transferred is returned in InquiryDataLength.
  @retval EFI_NOT_READY                  The SCSI Request Packet could not be sent because there
                                         are too many SCSI Command Packets already queued.
                                         The SCSI Request Packet was not sent, so no additional
                                         status information is available. The caller may retry again later.
  @retval EFI_DEVICE_ERROR               A device error occurred while attempting to send SCSI
                                         Request Packet.  See HostAdapterStatus, TargetStatus,
                                         SenseDataLength, and SenseData in that order for additional
                                         status information.
  @retval EFI_UNSUPPORTED                The command described by the SCSI Request Packet is not
                                         supported by the SCSI initiator(i.e., SCSI  Host Controller).
                                         The SCSI Request Packet was not sent, so no additional
                                         status information is available.
  @retval EFI_TIMEOUT                    A timeout occurred while waiting for the SCSI Request
                                         Packet to execute.  See HostAdapterStatus, TargetStatus,
                                         SenseDataLength, and SenseData in that order for
                                         additional status information.
  @retval EFI_INVALID_PARAMETER          The contents of the SCSI Request Packet are invalid.

**/
EFI_STATUS
EFIAPI
ScsiInquiryCommandEx (
  IN     EFI_SCSI_IO_PROTOCOL  *ScsiIo,
  IN     UINT64                Timeout,
  IN OUT VOID                  *SenseData   OPTIONAL,
  IN OUT UINT8                 *SenseDataLength,
  OUT UINT8                    *HostAdapterStatus,
  OUT UINT8                    *TargetStatus,
  IN OUT VOID                  *InquiryDataBuffer     OPTIONAL,
  IN OUT UINT32                *InquiryDataLength,
  IN     BOOLEAN               EnableVitalProductData,
  IN     UINT8                 PageCode
  );

/**
  Execute Mode Sense(10) SCSI command on a specific SCSI target.

  Executes the SCSI Mode Sense(10) command on the SCSI target specified by ScsiIo.
  If Timeout is zero, then this function waits indefinitely for the command to complete.
  If Timeout is greater than zero, then the command is executed and will timeout
  after Timeout 100 ns units.  The DBDField, PageControl, and PageCode parameters
  are used to construct the CDB for this SCSI command.
  If ScsiIo is NULL, then ASSERT().
  If SenseDataLength is NULL, then ASSERT().
  If HostAdapterStatus is NULL, then ASSERT().
  If TargetStatus is NULL, then ASSERT().
  If DataLength is NULL, then ASSERT().

  If SenseDataLength is non-zero and SenseData is not NULL, SenseData must meet buffer
  alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise EFI_INVALID_PARAMETER
  gets returned.

  If DataLength is non-zero and DataBuffer is not NULL, DataBuffer must meet buffer
  alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise EFI_INVALID_PARAMETER
  gets returned.

  @param[in]      ScsiIo             A pointer to the SCSI I/O Protocol instance
                                     for the specific SCSI target.
  @param[in]      Timeout            The timeout in 100 ns units to use for the
                                     execution of this SCSI Request Packet. A Timeout
                                     value of zero means that this function will wait
                                     indefinitely for the SCSI Request Packet to execute.
                                     If Timeout is greater than zero, then this function
                                     will return EFI_TIMEOUT if the time required to
                                     execute the SCSI Request Packet is greater than Timeout.
  @param[in, out]  SenseData         A pointer to sense data that was generated
                                     by the execution of the SCSI Request Packet.
                                     This buffer must be allocated by the caller.
                                     If SenseDataLength is 0, then this parameter
                                     is optional and may be NULL.
  @param[in, out]  SenseDataLength   On input, the length in bytes of the SenseData buffer.
                                     On output, the number of bytes written to the SenseData buffer.
  @param[out]     HostAdapterStatus  The status of the SCSI Host Controller that
                                     produces the SCSI bus containing the SCSI target
                                     specified by ScsiIo when the SCSI Request Packet
                                     was executed. See the EFI SCSI I/O Protocol in the
                                     UEFI Specification for details on the possible
                                     return values.
  @param[out]     TargetStatus       The status returned by the SCSI target specified
                                     by ScsiIo when the SCSI Request Packet was executed
                                     on the SCSI Host Controller.  See the EFI SCSI
                                     I/O Protocol in the UEFI Specification for details
                                     on the possible return values.
  @param[in, out]  DataBuffer        A pointer to data that was generated by the
                                     execution of the SCSI Request Packet.  This
                                     buffer must be allocated by the caller. If
                                     DataLength is 0, then this parameter is optional
                                     and may be NULL.
  @param[in, out]  DataLength        On input, a pointer to the length in bytes of
                                     the DataBuffer buffer.  On output, a pointer
                                     to the number of bytes written to the DataBuffer
                                     buffer.
  @param[in]      DBDField           Specifies the DBD field of the CDB for this SCSI Command.
  @param[in]      PageControl        Specifies the PC field of the CDB for this SCSI Command.
  @param[in]      PageCode           Specifies the Page Control field of the CDB for this SCSI Command.

  @retval EFI_SUCCESS                The command was executed successfully.
                                     See HostAdapterStatus, TargetStatus, SenseDataLength,
                                     and SenseData in that order for additional status information.
  @retval EFI_BAD_BUFFER_SIZE        The SCSI Request Packet was executed, but the
                                     entire DataBuffer could not be transferred.
                                     The actual number of bytes transferred is returned
                                     in DataLength.
  @retval EFI_NOT_READY              The SCSI Request Packet could not be sent because
                                     there are too many SCSI Command Packets already queued.
                                     The SCSI Request Packet was not sent, so no additional
                                     status information is available.  The caller may retry
                                     again later.
  @retval EFI_DEVICE_ERROR           A device error occurred while attempting to send
                                     SCSI Request Packet.  See HostAdapterStatus, TargetStatus,
                                     SenseDataLength, and SenseData in that order for
                                     additional status information.
  @retval EFI_UNSUPPORTED            The command described by the SCSI Request Packet
                                     is not supported by the SCSI initiator(i.e., SCSI
                                     Host Controller). The SCSI Request Packet was not
                                     sent, so no additional status information is available.
  @retval EFI_TIMEOUT                A timeout occurred while waiting for the SCSI
                                     Request Packet to execute.  See HostAdapterStatus,
                                     TargetStatus, SenseDataLength, and SenseData in that
                                     order for additional status information.
  @retval EFI_INVALID_PARAMETER      The contents of the SCSI Request Packet are invalid.

**/
EFI_STATUS
EFIAPI
ScsiModeSense10Command (
  IN     EFI_SCSI_IO_PROTOCOL  *ScsiIo,
  IN     UINT64                Timeout,
  IN OUT VOID                  *SenseData   OPTIONAL,
  IN OUT UINT8                 *SenseDataLength,
  OUT UINT8                    *HostAdapterStatus,
  OUT UINT8                    *TargetStatus,
  IN OUT VOID                  *DataBuffer  OPTIONAL,
  IN OUT UINT32                *DataLength,
  IN     UINT8                 DBDField     OPTIONAL,
  IN     UINT8                 PageControl,
  IN     UINT8                 PageCode
  );

/**
  Execute Request Sense SCSI command on a specific SCSI target.

  Executes the Request Sense command on the SCSI target specified by ScsiIo.
  If Timeout is zero, then this function waits indefinitely for the command to complete.
  If Timeout is greater than zero, then the command is executed and will timeout after Timeout 100 ns units.
  If ScsiIo is NULL, then ASSERT().
  If SenseDataLength is NULL, then ASSERT().
  If HostAdapterStatus is NULL, then ASSERT().
  If TargetStatus is NULL, then ASSERT().

  If SenseDataLength is non-zero and SenseData is not NULL, SenseData must meet buffer
  alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise EFI_INVALID_PARAMETER
  gets returned.

  @param[in]       ScsiIo               A pointer to SCSI IO protocol.
  @param[in]       Timeout              The length of timeout period.
  @param[in, out]  SenseData            A pointer to output sense data.
  @param[in, out]  SenseDataLength      The length of output sense data.
  @param[out]      HostAdapterStatus    The status of Host Adapter.
  @param[out]      TargetStatus         The status of the target.

  @retval EFI_SUCCESS                   Command is executed successfully.
  @retval EFI_NOT_READY                 The SCSI Request Packet could not be sent because there are
                                        too many SCSI Command Packets already queued.
  @retval EFI_DEVICE_ERROR              A device error occurred while attempting to send SCSI Request Packet.
  @retval EFI_UNSUPPORTED               The command described by the SCSI Request Packet is not supported by
                                        the SCSI initiator(i.e., SCSI  Host Controller)
  @retval EFI_TIMEOUT                   A timeout occurred while waiting for the SCSI Request Packet to execute.
  @retval EFI_INVALID_PARAMETER         The contents of the SCSI Request Packet are invalid.

**/
EFI_STATUS
EFIAPI
ScsiRequestSenseCommand (
  IN     EFI_SCSI_IO_PROTOCOL  *ScsiIo,
  IN     UINT64                Timeout,
  IN OUT VOID                  *SenseData   OPTIONAL,
  IN OUT UINT8                 *SenseDataLength,
  OUT UINT8                    *HostAdapterStatus,
  OUT UINT8                    *TargetStatus
  );

/**
  Execute Read Capacity SCSI command on a specific SCSI target.

  Executes the SCSI Read Capacity command on the SCSI target specified by ScsiIo.
  If Timeout is zero, then this function waits indefinitely for the command to complete.
  If Timeout is greater than zero, then the command is executed and will timeout after
  Timeout 100 ns units.  The Pmi parameter is used to construct the CDB for this SCSI command.
  If ScsiIo is NULL, then ASSERT().
  If SenseDataLength is NULL, then ASSERT().
  If HostAdapterStatus is NULL, then ASSERT().
  If TargetStatus is NULL, then ASSERT().
  If DataLength is NULL, then ASSERT().

  If SenseDataLength is non-zero and SenseData is not NULL, SenseData must meet buffer
  alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise EFI_INVALID_PARAMETER
  gets returned.

  If DataLength is non-zero and DataBuffer is not NULL, DataBuffer must meet buffer
  alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise EFI_INVALID_PARAMETER
  gets returned.

  @param[in]      ScsiIo               A pointer to SCSI IO protocol.
  @param[in]      Timeout              The length of timeout period.
  @param[in, out] SenseData            A pointer to output sense data.
  @param[in, out] SenseDataLength      The length of output sense data.
  @param[out]     HostAdapterStatus    The status of Host Adapter.
  @param[out]     TargetStatus         The status of the target.
  @param[in, out] DataBuffer           A pointer to a data buffer.
  @param[in, out] DataLength           The length of data buffer.
  @param[in]      Pmi                  Partial medium indicator.

  @retval  EFI_SUCCESS                 Command is executed successfully.
  @retval  EFI_BAD_BUFFER_SIZE         The SCSI Request Packet was executed, but the entire
                                       DataBuffer could not be transferred. The actual
                                       number of bytes transferred is returned in DataLength.
  @retval  EFI_NOT_READY               The SCSI Request Packet could not be sent because
                                       there are too many SCSI Command Packets already queued.
  @retval  EFI_DEVICE_ERROR            A device error occurred while attempting to send SCSI Request Packet.
  @retval  EFI_UNSUPPORTED             The command described by the SCSI Request Packet
                                       is not supported by the SCSI initiator(i.e., SCSI  Host Controller)
  @retval  EFI_TIMEOUT                 A timeout occurred while waiting for the SCSI Request Packet to execute.
  @retval  EFI_INVALID_PARAMETER       The contents of the SCSI Request Packet are invalid.

**/
EFI_STATUS
EFIAPI
ScsiReadCapacityCommand (
  IN     EFI_SCSI_IO_PROTOCOL  *ScsiIo,
  IN     UINT64                Timeout,
  IN OUT VOID                  *SenseData     OPTIONAL,
  IN OUT UINT8                 *SenseDataLength,
  OUT UINT8                    *HostAdapterStatus,
  OUT UINT8                    *TargetStatus,
  IN OUT VOID                  *DataBuffer    OPTIONAL,
  IN OUT UINT32                *DataLength,
  IN     BOOLEAN               Pmi
  );

/**
  Execute Read Capacity SCSI 16 command on a specific SCSI target.

  Executes the SCSI Read Capacity 16 command on the SCSI target specified by ScsiIo.
  If Timeout is zero, then this function waits indefinitely for the command to complete.
  If Timeout is greater than zero, then the command is executed and will timeout after
  Timeout 100 ns units.  The Pmi parameter is used to construct the CDB for this SCSI command.
  If ScsiIo is NULL, then ASSERT().
  If SenseDataLength is NULL, then ASSERT().
  If HostAdapterStatus is NULL, then ASSERT().
  If TargetStatus is NULL, then ASSERT().
  If DataLength is NULL, then ASSERT().

  If SenseDataLength is non-zero and SenseData is not NULL, SenseData must meet buffer
  alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise EFI_INVALID_PARAMETER
  gets returned.

  If DataLength is non-zero and DataBuffer is not NULL, DataBuffer must meet buffer
  alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise EFI_INVALID_PARAMETER
  gets returned.

  @param[in]      ScsiIo               A pointer to SCSI IO protocol.
  @param[in]      Timeout              The length of timeout period.
  @param[in, out] SenseData            A pointer to output sense data.
  @param[in, out] SenseDataLength      The length of output sense data.
  @param[out]     HostAdapterStatus    The status of Host Adapter.
  @param[out]     TargetStatus         The status of the target.
  @param[in, out] DataBuffer           A pointer to a data buffer.
  @param[in, out] DataLength           The length of data buffer.
  @param[in]      Pmi                  Partial medium indicator.

  @retval  EFI_SUCCESS                 Command is executed successfully.
  @retval  EFI_BAD_BUFFER_SIZE         The SCSI Request Packet was executed, but the entire
                                       DataBuffer could not be transferred. The actual
                                       number of bytes transferred is returned in DataLength.
  @retval  EFI_NOT_READY               The SCSI Request Packet could not be sent because
                                       there are too many SCSI Command Packets already queued.
  @retval  EFI_DEVICE_ERROR            A device error occurred while attempting to send SCSI Request Packet.
  @retval  EFI_UNSUPPORTED             The command described by the SCSI Request Packet
                                       is not supported by the SCSI initiator(i.e., SCSI  Host Controller)
  @retval  EFI_TIMEOUT                 A timeout occurred while waiting for the SCSI Request Packet to execute.
  @retval  EFI_INVALID_PARAMETER       The contents of the SCSI Request Packet are invalid.

**/
EFI_STATUS
EFIAPI
ScsiReadCapacity16Command (
  IN     EFI_SCSI_IO_PROTOCOL  *ScsiIo,
  IN     UINT64                Timeout,
  IN OUT VOID                  *SenseData   OPTIONAL,
  IN OUT UINT8                 *SenseDataLength,
  OUT UINT8                    *HostAdapterStatus,
  OUT UINT8                    *TargetStatus,
  IN OUT VOID                  *DataBuffer  OPTIONAL,
  IN OUT UINT32                *DataLength,
  IN     BOOLEAN               Pmi
  );

/**
  Execute Read(10) SCSI command on a specific SCSI target.

  Executes the SCSI Read(10) command on the SCSI target specified by ScsiIo.
  If Timeout is zero, then this function waits indefinitely for the command to complete.
  If Timeout is greater than zero, then the command is executed and will timeout
  after Timeout 100 ns units.  The StartLba and SectorSize parameters are used to
  construct the CDB for this SCSI command.
  If ScsiIo is NULL, then ASSERT().
  If SenseDataLength is NULL, then ASSERT().
  If HostAdapterStatus is NULL, then ASSERT().
  If TargetStatus is NULL, then ASSERT().
  If DataLength is NULL, then ASSERT().

  If SenseDataLength is non-zero and SenseData is not NULL, SenseData must meet buffer
  alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise EFI_INVALID_PARAMETER
  gets returned.

  If DataLength is non-zero and DataBuffer is not NULL, DataBuffer must meet buffer
  alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise EFI_INVALID_PARAMETER
  gets returned.

  @param[in]      ScsiIo               A pointer to SCSI IO protocol.
  @param[in]      Timeout              The length of timeout period.
  @param[in, out] SenseData            A pointer to output sense data.
  @param[in, out] SenseDataLength      The length of output sense data.
  @param[out]     HostAdapterStatus    The status of Host Adapter.
  @param[out]     TargetStatus         The status of the target.
  @param[in, out] DataBuffer           Read 10 command data.
  @param[in, out] DataLength           The length of data buffer.
  @param[in]      StartLba             The start address of LBA.
  @param[in]      SectorSize           The number of contiguous logical blocks of data that shall be transferred.

  @retval  EFI_SUCCESS                 Command is executed successfully.
  @retval  EFI_BAD_BUFFER_SIZE         The SCSI Request Packet was executed, but the entire DataBuffer could
                                       not be transferred. The actual number of bytes transferred is returned in DataLength.
  @retval  EFI_NOT_READY               The SCSI Request Packet could not be sent because there are too many
                                       SCSI Command Packets already queued.
  @retval  EFI_DEVICE_ERROR            A device error occurred while attempting to send SCSI Request Packet.
  @retval  EFI_UNSUPPORTED             The command described by the SCSI Request Packet is not supported by
                                       the SCSI initiator(i.e., SCSI  Host Controller)
  @retval  EFI_TIMEOUT                 A timeout occurred while waiting for the SCSI Request Packet to execute.
  @retval  EFI_INVALID_PARAMETER       The contents of the SCSI Request Packet are invalid.

**/
EFI_STATUS
EFIAPI
ScsiRead10Command (
  IN     EFI_SCSI_IO_PROTOCOL  *ScsiIo,
  IN     UINT64                Timeout,
  IN OUT VOID                  *SenseData    OPTIONAL,
  IN OUT UINT8                 *SenseDataLength,
  OUT UINT8                    *HostAdapterStatus,
  OUT UINT8                    *TargetStatus,
  IN OUT VOID                  *DataBuffer   OPTIONAL,
  IN OUT UINT32                *DataLength,
  IN     UINT32                StartLba,
  IN     UINT32                SectorSize
  );

/**
  Execute Write(10) SCSI command on a specific SCSI target.

  Executes the SCSI Write(10) command on the SCSI target specified by ScsiIo.
  If Timeout is zero, then this function waits indefinitely for the command to complete.
  If Timeout is greater than zero, then the command is executed and will timeout after
  Timeout 100 ns units.  The StartLba and SectorSize parameters are used to construct
  the CDB for this SCSI command.
  If ScsiIo is NULL, then ASSERT().
  If SenseDataLength is NULL, then ASSERT().
  If HostAdapterStatus is NULL, then ASSERT().
  If TargetStatus is NULL, then ASSERT().
  If DataLength is NULL, then ASSERT().

  If SenseDataLength is non-zero and SenseData is not NULL, SenseData must meet buffer
  alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise EFI_INVALID_PARAMETER
  gets returned.

  If DataLength is non-zero and DataBuffer is not NULL, DataBuffer must meet buffer
  alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise EFI_INVALID_PARAMETER
  gets returned.

  @param[in]      ScsiIo               SCSI IO Protocol to use
  @param[in]      Timeout              The length of timeout period.
  @param[in, out] SenseData            A pointer to output sense data.
  @param[in, out] SenseDataLength      The length of output sense data.
  @param[out]     HostAdapterStatus    The status of Host Adapter.
  @param[out]     TargetStatus         The status of the target.
  @param[in, out] DataBuffer           A pointer to a data buffer.
  @param[in, out] DataLength           The length of data buffer.
  @param[in]      StartLba             The start address of LBA.
  @param[in]      SectorSize           The number of contiguous logical blocks of data that shall be transferred.

  @retval  EFI_SUCCESS                 Command is executed successfully.
  @retval  EFI_BAD_BUFFER_SIZE         The SCSI Request Packet was executed, but the entire DataBuffer could
                                       not be transferred. The actual number of bytes transferred is returned in DataLength.
  @retval  EFI_NOT_READY               The SCSI Request Packet could not be sent because there are too many
                                       SCSI Command Packets already queued.
  @retval  EFI_DEVICE_ERROR            A device error occurred while attempting to send SCSI Request Packet.
  @retval  EFI_UNSUPPORTED             The command described by the SCSI Request Packet is not supported by
                                       the SCSI initiator(i.e., SCSI  Host Controller)
  @retval  EFI_TIMEOUT                 A timeout occurred while waiting for the SCSI Request Packet to execute.
  @retval  EFI_INVALID_PARAMETER       The contents of the SCSI Request Packet are invalid.

**/
EFI_STATUS
EFIAPI
ScsiWrite10Command (
  IN     EFI_SCSI_IO_PROTOCOL  *ScsiIo,
  IN     UINT64                Timeout,
  IN OUT VOID                  *SenseData    OPTIONAL,
  IN OUT UINT8                 *SenseDataLength,
  OUT UINT8                    *HostAdapterStatus,
  OUT UINT8                    *TargetStatus,
  IN OUT VOID                  *DataBuffer   OPTIONAL,
  IN OUT UINT32                *DataLength,
  IN     UINT32                StartLba,
  IN     UINT32                SectorSize
  );

/**
  Execute Read(16) SCSI command on a specific SCSI target.

  Executes the SCSI Read(16) command on the SCSI target specified by ScsiIo.
  If Timeout is zero, then this function waits indefinitely for the command to complete.
  If Timeout is greater than zero, then the command is executed and will timeout
  after Timeout 100 ns units.  The StartLba and SectorSize parameters are used to
  construct the CDB for this SCSI command.
  If ScsiIo is NULL, then ASSERT().
  If SenseDataLength is NULL, then ASSERT().
  If HostAdapterStatus is NULL, then ASSERT().
  If TargetStatus is NULL, then ASSERT().
  If DataLength is NULL, then ASSERT().

  If SenseDataLength is non-zero and SenseData is not NULL, SenseData must meet buffer
  alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise EFI_INVALID_PARAMETER
  gets returned.

  If DataLength is non-zero and DataBuffer is not NULL, DataBuffer must meet buffer
  alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise EFI_INVALID_PARAMETER
  gets returned.

  @param[in]      ScsiIo               A pointer to SCSI IO protocol.
  @param[in]      Timeout              The length of timeout period.
  @param[in, out] SenseData            A pointer to output sense data.
  @param[in, out] SenseDataLength      The length of output sense data.
  @param[out]     HostAdapterStatus    The status of Host Adapter.
  @param[out]     TargetStatus         The status of the target.
  @param[in, out] DataBuffer           Read 16 command data.
  @param[in, out] DataLength           The length of data buffer.
  @param[in]      StartLba             The start address of LBA.
  @param[in]      SectorSize           The number of contiguous logical blocks of data that shall be transferred.

  @retval  EFI_SUCCESS                 Command is executed successfully.
  @retval  EFI_BAD_BUFFER_SIZE         The SCSI Request Packet was executed, but the entire DataBuffer could
                                       not be transferred. The actual number of bytes transferred is returned in DataLength.
  @retval  EFI_NOT_READY               The SCSI Request Packet could not be sent because there are too many
                                       SCSI Command Packets already queued.
  @retval  EFI_DEVICE_ERROR            A device error occurred while attempting to send SCSI Request Packet.
  @retval  EFI_UNSUPPORTED             The command described by the SCSI Request Packet is not supported by
                                       the SCSI initiator(i.e., SCSI  Host Controller)
  @retval  EFI_TIMEOUT                 A timeout occurred while waiting for the SCSI Request Packet to execute.
  @retval  EFI_INVALID_PARAMETER       The contents of the SCSI Request Packet are invalid.

**/
EFI_STATUS
EFIAPI
ScsiRead16Command (
  IN     EFI_SCSI_IO_PROTOCOL  *ScsiIo,
  IN     UINT64                Timeout,
  IN OUT VOID                  *SenseData    OPTIONAL,
  IN OUT UINT8                 *SenseDataLength,
  OUT UINT8                    *HostAdapterStatus,
  OUT UINT8                    *TargetStatus,
  IN OUT VOID                  *DataBuffer   OPTIONAL,
  IN OUT UINT32                *DataLength,
  IN     UINT64                StartLba,
  IN     UINT32                SectorSize
  );

/**
  Execute Write(16) SCSI command on a specific SCSI target.

  Executes the SCSI Write(16) command on the SCSI target specified by ScsiIo.
  If Timeout is zero, then this function waits indefinitely for the command to complete.
  If Timeout is greater than zero, then the command is executed and will timeout after
  Timeout 100 ns units.  The StartLba and SectorSize parameters are used to construct
  the CDB for this SCSI command.
  If ScsiIo is NULL, then ASSERT().
  If SenseDataLength is NULL, then ASSERT().
  If HostAdapterStatus is NULL, then ASSERT().
  If TargetStatus is NULL, then ASSERT().
  If DataLength is NULL, then ASSERT().

  If SenseDataLength is non-zero and SenseData is not NULL, SenseData must meet buffer
  alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise EFI_INVALID_PARAMETER
  gets returned.

  If DataLength is non-zero and DataBuffer is not NULL, DataBuffer must meet buffer
  alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise EFI_INVALID_PARAMETER
  gets returned.

  @param[in]      ScsiIo               SCSI IO Protocol to use
  @param[in]      Timeout              The length of timeout period.
  @param[in, out] SenseData            A pointer to output sense data.
  @param[in, out] SenseDataLength      The length of output sense data.
  @param[out]     HostAdapterStatus    The status of Host Adapter.
  @param[out]     TargetStatus         The status of the target.
  @param[in, out] DataBuffer           A pointer to a data buffer.
  @param[in, out] DataLength           The length of data buffer.
  @param[in]      StartLba             The start address of LBA.
  @param[in]      SectorSize           The number of contiguous logical blocks of data that shall be transferred.

  @retval  EFI_SUCCESS                 Command is executed successfully.
  @retval  EFI_BAD_BUFFER_SIZE         The SCSI Request Packet was executed, but the entire DataBuffer could
                                       not be transferred. The actual number of bytes transferred is returned in DataLength.
  @retval  EFI_NOT_READY               The SCSI Request Packet could not be sent because there are too many
                                       SCSI Command Packets already queued.
  @retval  EFI_DEVICE_ERROR            A device error occurred while attempting to send SCSI Request Packet.
  @retval  EFI_UNSUPPORTED             The command described by the SCSI Request Packet is not supported by
                                       the SCSI initiator(i.e., SCSI  Host Controller)
  @retval  EFI_TIMEOUT                 A timeout occurred while waiting for the SCSI Request Packet to execute.
  @retval  EFI_INVALID_PARAMETER       The contents of the SCSI Request Packet are invalid.

**/
EFI_STATUS
EFIAPI
ScsiWrite16Command (
  IN     EFI_SCSI_IO_PROTOCOL  *ScsiIo,
  IN     UINT64                Timeout,
  IN OUT VOID                  *SenseData    OPTIONAL,
  IN OUT UINT8                 *SenseDataLength,
  OUT UINT8                    *HostAdapterStatus,
  OUT UINT8                    *TargetStatus,
  IN OUT VOID                  *DataBuffer   OPTIONAL,
  IN OUT UINT32                *DataLength,
  IN     UINT64                StartLba,
  IN     UINT32                SectorSize
  );

/**
  Execute Security Protocol In SCSI command on a specific SCSI target.

  Executes the SCSI Security Protocol In command on the SCSI target specified by ScsiIo.
  If Timeout is zero, then this function waits indefinitely for the command to complete.
  If Timeout is greater than zero, then the command is executed and will timeout after
  Timeout 100 ns units.
  If ScsiIo is NULL, then ASSERT().
  If SenseDataLength is NULL, then ASSERT().
  If HostAdapterStatus is NULL, then ASSERT().
  If TargetStatus is NULL, then ASSERT().
  If TransferLength is NULL, then ASSERT().

  If SenseDataLength is non-zero and SenseData is not NULL, SenseData must meet buffer
  alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise EFI_INVALID_PARAMETER
  gets returned.

  If DataLength is non-zero and DataBuffer is not NULL, DataBuffer must meet buffer
  alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise EFI_INVALID_PARAMETER
  gets returned.

  @param[in]      ScsiIo                    SCSI IO Protocol to use.
  @param[in]      Timeout                   The length of timeout period.
  @param[in, out] SenseData                 A pointer to output sense data.
  @param[in, out] SenseDataLength           The length of output sense data.
  @param[out]     HostAdapterStatus         The status of Host Adapter.
  @param[out]     TargetStatus              The status of the target.
  @param[in]      SecurityProtocol          The Security Protocol to use.
  @param[in]      SecurityProtocolSpecific  The Security Protocol Specific data.
  @param[in]      Inc512                    If TRUE, 512 increment (INC_512) bit will be set for the
                                            SECURITY PROTOCOL IN command.
  @param[in]      DataLength                The size in bytes of the data buffer.
  @param[in, out] DataBuffer                A pointer to a data buffer.
  @param[out]     TransferLength            A pointer to a buffer to store the size in
                                            bytes of the data written to the data buffer.

  @retval  EFI_SUCCESS                 Command is executed successfully.
  @retval  EFI_BAD_BUFFER_SIZE         The SCSI Request Packet was executed, but the entire DataBuffer could
                                       not be transferred. The actual number of bytes transferred is returned in TransferLength.
  @retval  EFI_NOT_READY               The SCSI Request Packet could not be sent because there are too many
                                       SCSI Command Packets already queued.
  @retval  EFI_DEVICE_ERROR            A device error occurred while attempting to send SCSI Request Packet.
  @retval  EFI_UNSUPPORTED             The command described by the SCSI Request Packet is not supported by
                                       the SCSI initiator(i.e., SCSI  Host Controller)
  @retval  EFI_TIMEOUT                 A timeout occurred while waiting for the SCSI Request Packet to execute.
  @retval  EFI_INVALID_PARAMETER       The contents of the SCSI Request Packet are invalid.

**/
EFI_STATUS
EFIAPI
ScsiSecurityProtocolInCommand (
  IN     EFI_SCSI_IO_PROTOCOL  *ScsiIo,
  IN     UINT64                Timeout,
  IN OUT VOID                  *SenseData    OPTIONAL,
  IN OUT UINT8                 *SenseDataLength,
  OUT UINT8                    *HostAdapterStatus,
  OUT UINT8                    *TargetStatus,
  IN     UINT8                 SecurityProtocol,
  IN     UINT16                SecurityProtocolSpecific,
  IN     BOOLEAN               Inc512,
  IN     UINTN                 DataLength,
  IN OUT VOID                  *DataBuffer   OPTIONAL,
  OUT UINTN                    *TransferLength
  );

/**
  Execute Security Protocol Out SCSI command on a specific SCSI target.

  Executes the SCSI Security Protocol Out command on the SCSI target specified by ScsiIo.
  If Timeout is zero, then this function waits indefinitely for the command to complete.
  If Timeout is greater than zero, then the command is executed and will timeout after
  Timeout 100 ns units.
  If ScsiIo is NULL, then ASSERT().
  If SenseDataLength is NULL, then ASSERT().
  If HostAdapterStatus is NULL, then ASSERT().
  If TargetStatus is NULL, then ASSERT().

  If SenseDataLength is non-zero and SenseData is not NULL, SenseData must meet buffer
  alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise EFI_INVALID_PARAMETER
  gets returned.

  If DataLength is non-zero and DataBuffer is not NULL, DataBuffer must meet buffer
  alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise EFI_INVALID_PARAMETER
  gets returned.

  @param[in]      ScsiIo                    SCSI IO Protocol to use.
  @param[in]      Timeout                   The length of timeout period.
  @param[in, out] SenseData                 A pointer to output sense data.
  @param[in, out] SenseDataLength           The length of output sense data.
  @param[out]     HostAdapterStatus         The status of Host Adapter.
  @param[out]     TargetStatus              The status of the target.
  @param[in]      SecurityProtocol          The Security Protocol to use.
  @param[in]      SecurityProtocolSpecific  The Security Protocol Specific data.
  @param[in]      Inc512                    If TRUE, 512 increment (INC_512) bit will be set for the
                                            SECURITY PROTOCOL OUT command.
  @param[in]      DataLength                The size in bytes of the transfer data.
  @param[in, out] DataBuffer                A pointer to a data buffer.

  @retval  EFI_SUCCESS                 Command is executed successfully.
  @retval  EFI_BAD_BUFFER_SIZE         The SCSI Request Packet was executed, but the entire DataBuffer could
                                       not be transferred. The actual number of bytes transferred is returned in DataLength.
  @retval  EFI_NOT_READY               The SCSI Request Packet could not be sent because there are too many
                                       SCSI Command Packets already queued.
  @retval  EFI_DEVICE_ERROR            A device error occurred while attempting to send SCSI Request Packet.
  @retval  EFI_UNSUPPORTED             The command described by the SCSI Request Packet is not supported by
                                       the SCSI initiator(i.e., SCSI  Host Controller)
  @retval  EFI_TIMEOUT                 A timeout occurred while waiting for the SCSI Request Packet to execute.
  @retval  EFI_INVALID_PARAMETER       The contents of the SCSI Request Packet are invalid.

**/
EFI_STATUS
EFIAPI
ScsiSecurityProtocolOutCommand (
  IN     EFI_SCSI_IO_PROTOCOL  *ScsiIo,
  IN     UINT64                Timeout,
  IN OUT VOID                  *SenseData    OPTIONAL,
  IN OUT UINT8                 *SenseDataLength,
  OUT UINT8                    *HostAdapterStatus,
  OUT UINT8                    *TargetStatus,
  IN     UINT8                 SecurityProtocol,
  IN     UINT16                SecurityProtocolSpecific,
  IN     BOOLEAN               Inc512,
  IN     UINTN                 DataLength,
  IN OUT VOID                  *DataBuffer   OPTIONAL
  );

/**
  Execute blocking/non-blocking Read(10) SCSI command on a specific SCSI
  target.

  Executes the SCSI Read(10) command on the SCSI target specified by ScsiIo.
  When Event is NULL, blocking command will be executed. Otherwise non-blocking
  command will be executed.
  For blocking I/O, if Timeout is zero, this function will wait indefinitely
  for the command to complete. If Timeout is greater than zero, then the
  command is executed and will timeout after Timeout 100 ns units.
  For non-blocking I/O, if Timeout is zero, Event will be signaled only after
  the command to completes. If Timeout is greater than zero, Event will also be
  signaled after Timeout 100 ns units.
  The StartLba and SectorSize parameters are used to construct the CDB for this
  SCSI command.

  If ScsiIo is NULL, then ASSERT().
  If SenseDataLength is NULL, then ASSERT().
  If HostAdapterStatus is NULL, then ASSERT().
  If TargetStatus is NULL, then ASSERT().
  If DataLength is NULL, then ASSERT().

  If SenseDataLength is non-zero and SenseData is not NULL, SenseData must meet
  buffer alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise
  EFI_INVALID_PARAMETER gets returned.

  If DataLength is non-zero and DataBuffer is not NULL, DataBuffer must meet
  buffer alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise
  EFI_INVALID_PARAMETER gets returned.

  @param[in]      ScsiIo               A pointer to SCSI IO protocol.
  @param[in]      Timeout              The length of timeout period.
  @param[in, out] SenseData            A pointer to output sense data.
  @param[in, out] SenseDataLength      The length of output sense data.
  @param[out]     HostAdapterStatus    The status of Host Adapter.
  @param[out]     TargetStatus         The status of the target.
  @param[in, out] DataBuffer           Read 16 command data.
  @param[in, out] DataLength           The length of data buffer.
  @param[in]      StartLba             The start address of LBA.
  @param[in]      SectorSize           The number of contiguous logical blocks
                                       of data that shall be transferred.
  @param[in]      Event                If the SCSI target does not support
                                       non-blocking I/O, then Event is ignored,
                                       and blocking I/O is performed. If Event
                                       is NULL, then blocking I/O is performed.
                                       If Event is not NULL and non-blocking
                                       I/O is supported, then non-blocking I/O
                                       is performed, and Event will be signaled
                                       when the SCSI Read(10) command
                                       completes.

  @retval  EFI_SUCCESS                 Command is executed successfully.
  @retval  EFI_BAD_BUFFER_SIZE         The SCSI Request Packet was executed,
                                       but the entire DataBuffer could not be
                                       transferred. The actual number of bytes
                                       transferred is returned in DataLength.
  @retval  EFI_NOT_READY               The SCSI Request Packet could not be
                                       sent because there are too many SCSI
                                       Command Packets already queued.
  @retval  EFI_DEVICE_ERROR            A device error occurred while attempting
                                       to send SCSI Request Packet.
  @retval  EFI_UNSUPPORTED             The command described by the SCSI
                                       Request Packet is not supported by the
                                       SCSI initiator(i.e., SCSI  Host
                                       Controller)
  @retval  EFI_TIMEOUT                 A timeout occurred while waiting for the
                                       SCSI Request Packet to execute.
  @retval  EFI_INVALID_PARAMETER       The contents of the SCSI Request Packet
                                       are invalid.
  @retval  EFI_OUT_OF_RESOURCES        The request could not be completed due
                                       to a lack of resources.

**/
EFI_STATUS
EFIAPI
ScsiRead10CommandEx (
  IN     EFI_SCSI_IO_PROTOCOL  *ScsiIo,
  IN     UINT64                Timeout,
  IN OUT VOID                  *SenseData    OPTIONAL,
  IN OUT UINT8                 *SenseDataLength,
  OUT UINT8                    *HostAdapterStatus,
  OUT UINT8                    *TargetStatus,
  IN OUT VOID                  *DataBuffer   OPTIONAL,
  IN OUT UINT32                *DataLength,
  IN     UINT32                StartLba,
  IN     UINT32                SectorSize,
  IN     EFI_EVENT             Event         OPTIONAL
  );

/**
  Execute blocking/non-blocking Write(10) SCSI command on a specific SCSI
  target.

  Executes the SCSI Write(10) command on the SCSI target specified by ScsiIo.
  When Event is NULL, blocking command will be executed. Otherwise non-blocking
  command will be executed.
  For blocking I/O, if Timeout is zero, this function will wait indefinitely
  for the command to complete. If Timeout is greater than zero, then the
  command is executed and will timeout after Timeout 100 ns units.
  For non-blocking I/O, if Timeout is zero, Event will be signaled only after
  the command to completes. If Timeout is greater than zero, Event will also be
  signaled after Timeout 100 ns units.
  The StartLba and SectorSize parameters are used to construct the CDB for this
  SCSI command.

  If ScsiIo is NULL, then ASSERT().
  If SenseDataLength is NULL, then ASSERT().
  If HostAdapterStatus is NULL, then ASSERT().
  If TargetStatus is NULL, then ASSERT().
  If DataLength is NULL, then ASSERT().

  If SenseDataLength is non-zero and SenseData is not NULL, SenseData must meet
  buffer alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise
  EFI_INVALID_PARAMETER gets returned.

  If DataLength is non-zero and DataBuffer is not NULL, DataBuffer must meet
  buffer alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise
  EFI_INVALID_PARAMETER gets returned.

  @param[in]      ScsiIo               SCSI IO Protocol to use
  @param[in]      Timeout              The length of timeout period.
  @param[in, out] SenseData            A pointer to output sense data.
  @param[in, out] SenseDataLength      The length of output sense data.
  @param[out]     HostAdapterStatus    The status of Host Adapter.
  @param[out]     TargetStatus         The status of the target.
  @param[in, out] DataBuffer           A pointer to a data buffer.
  @param[in, out] DataLength           The length of data buffer.
  @param[in]      StartLba             The start address of LBA.
  @param[in]      SectorSize           The number of contiguous logical blocks
                                       of data that shall be transferred.
  @param[in]      Event                If the SCSI target does not support
                                       non-blocking I/O, then Event is ignored,
                                       and blocking I/O is performed. If Event
                                       is NULL, then blocking I/O is performed.
                                       If Event is not NULL and non-blocking
                                       I/O is supported, then non-blocking I/O
                                       is performed, and Event will be signaled
                                       when the SCSI Write(10) command
                                       completes.

  @retval  EFI_SUCCESS                 Command is executed successfully.
  @retval  EFI_BAD_BUFFER_SIZE         The SCSI Request Packet was executed,
                                       but the entire DataBuffer could not be
                                       transferred. The actual number of bytes
                                       transferred is returned in DataLength.
  @retval  EFI_NOT_READY               The SCSI Request Packet could not be
                                       sent because there are too many SCSI
                                       Command Packets already queued.
  @retval  EFI_DEVICE_ERROR            A device error occurred while attempting
                                       to send SCSI Request Packet.
  @retval  EFI_UNSUPPORTED             The command described by the SCSI
                                       Request Packet is not supported by the
                                       SCSI initiator(i.e., SCSI  Host
                                       Controller)
  @retval  EFI_TIMEOUT                 A timeout occurred while waiting for the
                                       SCSI Request Packet to execute.
  @retval  EFI_INVALID_PARAMETER       The contents of the SCSI Request Packet
                                       are invalid.
  @retval  EFI_OUT_OF_RESOURCES        The request could not be completed due
                                       to a lack of resources.

**/
EFI_STATUS
EFIAPI
ScsiWrite10CommandEx (
  IN     EFI_SCSI_IO_PROTOCOL  *ScsiIo,
  IN     UINT64                Timeout,
  IN OUT VOID                  *SenseData    OPTIONAL,
  IN OUT UINT8                 *SenseDataLength,
  OUT UINT8                    *HostAdapterStatus,
  OUT UINT8                    *TargetStatus,
  IN OUT VOID                  *DataBuffer   OPTIONAL,
  IN OUT UINT32                *DataLength,
  IN     UINT32                StartLba,
  IN     UINT32                SectorSize,
  IN     EFI_EVENT             Event         OPTIONAL
  );

/**
  Execute blocking/non-blocking Read(16) SCSI command on a specific SCSI
  target.

  Executes the SCSI Read(16) command on the SCSI target specified by ScsiIo.
  When Event is NULL, blocking command will be executed. Otherwise non-blocking
  command will be executed.
  For blocking I/O, if Timeout is zero, this function will wait indefinitely
  for the command to complete. If Timeout is greater than zero, then the
  command is executed and will timeout after Timeout 100 ns units.
  For non-blocking I/O, if Timeout is zero, Event will be signaled only after
  the command to completes. If Timeout is greater than zero, Event will also be
  signaled after Timeout 100 ns units.
  The StartLba and SectorSize parameters are used to construct the CDB for this
  SCSI command.

  If ScsiIo is NULL, then ASSERT().
  If SenseDataLength is NULL, then ASSERT().
  If HostAdapterStatus is NULL, then ASSERT().
  If TargetStatus is NULL, then ASSERT().
  If DataLength is NULL, then ASSERT().

  If SenseDataLength is non-zero and SenseData is not NULL, SenseData must meet
  buffer alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise
  EFI_INVALID_PARAMETER gets returned.

  If DataLength is non-zero and DataBuffer is not NULL, DataBuffer must meet
  buffer alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise
  EFI_INVALID_PARAMETER gets returned.

  @param[in]      ScsiIo               A pointer to SCSI IO protocol.
  @param[in]      Timeout              The length of timeout period.
  @param[in, out] SenseData            A pointer to output sense data.
  @param[in, out] SenseDataLength      The length of output sense data.
  @param[out]     HostAdapterStatus    The status of Host Adapter.
  @param[out]     TargetStatus         The status of the target.
  @param[in, out] DataBuffer           Read 16 command data.
  @param[in, out] DataLength           The length of data buffer.
  @param[in]      StartLba             The start address of LBA.
  @param[in]      SectorSize           The number of contiguous logical blocks
                                       of data that shall be transferred.
  @param[in]      Event                If the SCSI target does not support
                                       non-blocking I/O, then Event is ignored,
                                       and blocking I/O is performed. If Event
                                       is NULL, then blocking I/O is performed.
                                       If Event is not NULL and non-blocking
                                       I/O is supported, then non-blocking I/O
                                       is performed, and Event will be signaled
                                       when the SCSI Read(16) command
                                       completes.

  @retval  EFI_SUCCESS                 Command is executed successfully.
  @retval  EFI_BAD_BUFFER_SIZE         The SCSI Request Packet was executed,
                                       but the entire DataBuffer could not be
                                       transferred. The actual number of bytes
                                       transferred is returned in DataLength.
  @retval  EFI_NOT_READY               The SCSI Request Packet could not be
                                       sent because there are too many SCSI
                                       Command Packets already queued.
  @retval  EFI_DEVICE_ERROR            A device error occurred while attempting
                                       to send SCSI Request Packet.
  @retval  EFI_UNSUPPORTED             The command described by the SCSI
                                       Request Packet is not supported by the
                                       SCSI initiator(i.e., SCSI  Host
                                       Controller)
  @retval  EFI_TIMEOUT                 A timeout occurred while waiting for the
                                       SCSI Request Packet to execute.
  @retval  EFI_INVALID_PARAMETER       The contents of the SCSI Request Packet
                                       are invalid.
  @retval  EFI_OUT_OF_RESOURCES        The request could not be completed due
                                       to a lack of resources.

**/
EFI_STATUS
EFIAPI
ScsiRead16CommandEx (
  IN     EFI_SCSI_IO_PROTOCOL  *ScsiIo,
  IN     UINT64                Timeout,
  IN OUT VOID                  *SenseData    OPTIONAL,
  IN OUT UINT8                 *SenseDataLength,
  OUT UINT8                    *HostAdapterStatus,
  OUT UINT8                    *TargetStatus,
  IN OUT VOID                  *DataBuffer   OPTIONAL,
  IN OUT UINT32                *DataLength,
  IN     UINT64                StartLba,
  IN     UINT32                SectorSize,
  IN     EFI_EVENT             Event         OPTIONAL
  );

/**
  Execute blocking/non-blocking Write(16) SCSI command on a specific SCSI
  target.

  Executes the SCSI Write(16) command on the SCSI target specified by ScsiIo.
  When Event is NULL, blocking command will be executed. Otherwise non-blocking
  command will be executed.
  For blocking I/O, if Timeout is zero, this function will wait indefinitely
  for the command to complete. If Timeout is greater than zero, then the
  command is executed and will timeout after Timeout 100 ns units.
  For non-blocking I/O, if Timeout is zero, Event will be signaled only after
  the command to completes. If Timeout is greater than zero, Event will also be
  signaled after Timeout 100 ns units.
  The StartLba and SectorSize parameters are used to construct the CDB for this
  SCSI command.

  If ScsiIo is NULL, then ASSERT().
  If SenseDataLength is NULL, then ASSERT().
  If HostAdapterStatus is NULL, then ASSERT().
  If TargetStatus is NULL, then ASSERT().
  If DataLength is NULL, then ASSERT().

  If SenseDataLength is non-zero and SenseData is not NULL, SenseData must meet
  buffer alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise
  EFI_INVALID_PARAMETER gets returned.

  If DataLength is non-zero and DataBuffer is not NULL, DataBuffer must meet
  buffer alignment requirement defined in EFI_SCSI_IO_PROTOCOL. Otherwise
  EFI_INVALID_PARAMETER gets returned.

  @param[in]      ScsiIo               SCSI IO Protocol to use
  @param[in]      Timeout              The length of timeout period.
  @param[in, out] SenseData            A pointer to output sense data.
  @param[in, out] SenseDataLength      The length of output sense data.
  @param[out]     HostAdapterStatus    The status of Host Adapter.
  @param[out]     TargetStatus         The status of the target.
  @param[in, out] DataBuffer           A pointer to a data buffer.
  @param[in, out] DataLength           The length of data buffer.
  @param[in]      StartLba             The start address of LBA.
  @param[in]      SectorSize           The number of contiguous logical blocks
                                       of data that shall be transferred.
  @param[in]      Event                If the SCSI target does not support
                                       non-blocking I/O, then Event is ignored,
                                       and blocking I/O is performed. If Event
                                       is NULL, then blocking I/O is performed.
                                       If Event is not NULL and non-blocking
                                       I/O is supported, then non-blocking I/O
                                       is performed, and Event will be signaled
                                       when the SCSI Write(16) command
                                       completes.

  @retval  EFI_SUCCESS                 Command is executed successfully.
  @retval  EFI_BAD_BUFFER_SIZE         The SCSI Request Packet was executed,
                                       but the entire DataBuffer could not be
                                       transferred. The actual number of bytes
                                       transferred is returned in DataLength.
  @retval  EFI_NOT_READY               The SCSI Request Packet could not be
                                       sent because there are too many SCSI
                                       Command Packets already queued.
  @retval  EFI_DEVICE_ERROR            A device error occurred while attempting
                                       to send SCSI Request Packet.
  @retval  EFI_UNSUPPORTED             The command described by the SCSI
                                       Request Packet is not supported by the
                                       SCSI initiator(i.e., SCSI  Host
                                       Controller)
  @retval  EFI_TIMEOUT                 A timeout occurred while waiting for the
                                       SCSI Request Packet to execute.
  @retval  EFI_INVALID_PARAMETER       The contents of the SCSI Request Packet
                                       are invalid.
  @retval  EFI_OUT_OF_RESOURCES        The request could not be completed due
                                       to a lack of resources.

**/
EFI_STATUS
EFIAPI
ScsiWrite16CommandEx (
  IN     EFI_SCSI_IO_PROTOCOL  *ScsiIo,
  IN     UINT64                Timeout,
  IN OUT VOID                  *SenseData    OPTIONAL,
  IN OUT UINT8                 *SenseDataLength,
  OUT UINT8                    *HostAdapterStatus,
  OUT UINT8                    *TargetStatus,
  IN OUT VOID                  *DataBuffer   OPTIONAL,
  IN OUT UINT32                *DataLength,
  IN     UINT64                StartLba,
  IN     UINT32                SectorSize,
  IN     EFI_EVENT             Event         OPTIONAL
  );

#endif
