---
title: MFront interface for code_aster
author: Thomas Helfer
date: 2020
lang: en-EN
link-citations: true
colorlinks: true
figPrefixTemplate: "$$i$$"
tblPrefixTemplate: "$$i$$"
secPrefixTemplate: "$$i$$"
eqnPrefixTemplate: "($$i$$)"
bibliography: bibliography.bib
---

This document shows how to use `MFront` mechanical behaviour in EDF
[`code_aster`](http://www.code-aster.org) finite element solver. It is
extracted from the
[`MFront` tutorial](documents/tutoriel/tutoriel.pdf). A full
description is available in the
[`code_aster`](http://www.code-aster.org) reference documentation (see
@code_notice_2014).

Usage of `MFront` in [`code_aster`](http://www.code-aster.org) can be
made very easy, once a few things have been clarified. This is
precisely the purpose of this page.

Note that this page is focused on mechanical behaviours. *One can also
use material properties generated with the `python` interface*, the
description of which is out of the scope of this page.

Usage of mechanical behaviours generated by `MFront` is a two step
process:

- Generating the shared library containing the mechanical behaviour
  using `MFront`.
- Declaring the mechanical behaviour in the `COMPORTEMENT` field of
  the `STAT_NON_LINE` function.

Those two steps are detailed in this document.

> **A word of caution**
> 
> `MFront` is now part of the `code_aster` distribution. **The use of
> another version of `MFront` for generating mechanical behaviours is
> strongly discouraged** as there is no garantee that two versions of
> MFront are binary compatible: combining two versions of `MFront` can
> lead to an error in the best case, crashes of `code_aster` in the
> worst case and a wide variety of strange behaviours in between.

> **`MFront` behaviours officially integrated in `code_aster`**
>
> Some mechanical behaviours officially available in `code_aster` are
> natively generated with `MFront`. Those may be distinguished by
> their names which are lowercase (e.g. `Iwan`).
> This page only deals with user generated `MFront` mechanical behaviours.

# Overview of the `aster` interface 

The `aster` interface can be used to introduce:

- Small strain mechanical behaviours. Those behaviours can be used
  with the `GDEF_LOG` and `GROT_GDEP` finite strain strategies (see
  @edf_modeles_2013).
- Finite strain mechanical behaviours.
- Cohesive zone models.

## Finite strain behaviours

Support for `MFront` finite strain behaviours is only available for
`code_aster` version greater than `13`.

`code_aster` provides two distincts finite strain formulation:

- `SIMO_MIEHE` which is a finite strain formulation where the
  principle of virtual power is expressed in the current
  configuration (see @edf_loi_2013).
- `GROT_GDEP` is the name in `code_aster` of a finite strain
  formulation based on the principle of virtual work in the reference
  configuration expressed in term of the Green-Lagrange strain and the
  second Piola-Kirchhoff stress. Such a formulation is also called
  `Total Lagrangian` in the litterature (see @belytschko_nonlinear_2000)
  and in other finite element solvers.

From the behaviour point of view, using `SIMO_MIEHE` or `GROT_GDEP`
differs from:

- The choice of the output stress (the Cauchy stress in `SIMO_MIEHE`,
the second Piola-Kirchhoff stress in `GROT_GDEP`)
- The definition of the consistent tangent operator.

The `@AsterFiniteStrainFormulation` keyword can be used to choose one
of these finite strain formulation. This keyword must be followed by
one of the following choice:

- `SIMO_MIEHE`
- `GROT_GDEP` or `TotalLagrangian`

By default, finite strain behaviours must be used with the
`SIMO_MIEHE` finite strain formulation.

# Generating the shared library

The first step can be done as part of a `code_aster` simulation or
*before* running `code_aster`. These two approaches have their
advantages and their drawbacks.

The first one is used in `code_aster` verification tests
associated with `MFront` and for various examples delivered with the
code.

In practice, we consider the second approach to be easier and more
flexible.

In the following, we will consider the case of single mechanical
behaviour implemented in a file called `plasticity.mfront`.

## Generating the shared library as part of the `code_aster` simulation

The instructions for the generation of the shared library are given in
the `.comm` file by an instruction similar to:

~~~~{.python}
os.system("mfront --obuild plasticity.mfront --interface=aster")
~~~~

Such an instruction requires the `os` `python` module to be loaded at
the beginning of the `.comm` file.

The previous instruction calls the `mfront` executable which will:

- generate `C++` sources for the `aster` interface from the
  `plasticity.mfront` file
- build the shared library from the previous `C++` sources.

Those operations are performed in a temporary directory in which the
`code_aster` simulation is run. **For the `plasticity.mfront` to be
present in this directory, it must be declared in `astk` as an external
data file (e.g. with type `nom`)**.

The library is generated in the `src` subdirectory. For convenience,
this library is often copied in the current directory and often
renamed with an instruction similar to:

~~~~{.python}
os.system("cp src/libAsterBehaviour.so plasticity.so")
~~~~

### Advantages

The advantage of this first approach is that `as_run` automatically
set various environment variables for `mfront` to work.

Moreover, the library is generated in the current directory (or in the
`src` subdirectory if the library is not copied and renamed), which
means that it can directly be found when needed, typically when the
`STAT_NON_LINE` function is called.

### Drawbacks

This first approach however have however serious drawbacks:

- No error messages is thrown if the generation of the `C++` sources
  or the compilation of the shared library fail.
- The user must know *a priori* the name of the generated library:
  Although `libAsterBehaviour.so` is the default and most common name,
  this name can be affected by the use of the `@Material` or the
  `@Library` `mfront` keywords.
- The user must know *a priori* the name of function that implements
  the behaviour. This name is most of the time deduced from the
  behaviour name, as given by `@Behaviour` `mfront` keyword, but it
  can affected by use of the `@Material` keyword.
- The shared library is compiled at each run of the simulation. This
  can be a waste of time (the compilation of an unique behaviour can
  take several seconds).
- It precludes, or at least makes more difficult, the use of advanced
  `MFront` functionalities, such as calling other `MFront` files
  (declaring for example material properties) from the
  `plasticity.mfront` file.

## Second approach: generating the library before launching the simulation

### Usage of `MFront` outside the `code_aster` environment

As previously described, `as_run` sets up various environment
variables to enable the use of `mfront` and the use of the shared
libraries generated by `mfront` during the simulation.

To use `MFront` outside of a `code_aster` simulation, we have to set an
appropriate environment.

Let `ASTER_ROOT` be an environment variable containing the installation
directory of `code_aster` (`as_run` and `astk` are then located in the
`$ASTER_ROOT/bin` directory). In the examples below, the `ASTER_ROOT`
variable is supposed to have been defined by the user. Using the
`bash` shell, this is done by:

~~~~{.sh}
$ export ASTER_ROOT=/home/th202608/codes/aster/13.3.0/install/
~~~~

Of course, the previous instruction must be adapted for your specific
installation.

`MFront` is installed in `$ASTER_ROOT/public/mfront-xxx` where `xxx`
stands for the version of `MFront` delivered with `code_aster`.

To use `MFront`, one must set the `PATH` and `LD_LIBRARY_PATH` as
follows:

~~~~{.sh}
$ export PATH=$ASTER_ROOT/public/mfront-xxx/bin:$PATH
$ export LD_LIBRARY_PATH=$ASTER_ROOT/public/mfront-xxx/lib:$LD_LIBRARY_PATH
~~~~

To check that those environments have been properly set, just type:

~~~~{.sh}
$ mfront
MFront::exe: no file name specified and nothing to be done
Usage: mfront [options] [files]
~~~~

This shows that `mfront` has been found and is functional.

### Generation of the shared library

The following instruction will compile the `MFront` behaviour using
the `aster` interface :

~~~~ {.bash}
$ mfront --obuild --interface=aster plasticity.mfront
Treating target : all
The following library has been built :
- libAsterBehaviour.so :  asterplasticity
~~~~~~~~~~~~~~~~~~~

This shows that the `libAsterBehabviour.so` has been generated. It
contains a function called `asterplasticity`. This library is located
in the `src` subdirectory.

### Advantages

This second approach has the following advantages:

- Compilation errors are easier to detect, analyze and correct.
- The mechanical behaviours can be tested with `MTest` before their
  introduction in `code_aster`, which is a good pratice.
- All the functionnalities of `MFront` are available.
- The shared library is build only once (i.e. is not rebuilt for each
  simulation run)
- The shared library can be placed in a shared folder, allowing
  various members of a team to use the same mechanical behaviour.

### Drawbacks

The shared library is not in the temporary directory used by `as_run`
to run the simulation, so the user must specify
where it is located. This can be done in various ways:

- The library can be marked as a dependency of the computation in
  `astk` (external data file).
- The absolute path of the library must be given.
- The directory where the shared library is located must be added to
  the `LD_LIBRARY_PATH`.

# Declaration of the call

Starting from an existing input file, two things must be declared:

- a material with appropriate coefficients using the `DEFI_MATERIAU`
  operator.
- a behaviour in the `COMPORTEMENT` field of the mechanical operators
  (`STAT_NON_LINE`, `DYNA_NON_LINE`, `SIMU_POINT_MAT`, ...).

## Call to the `DEFI_MATERIAU`

### Version 12.x

In the `DEFI_MATERIAU` block, one must add for `code_aster` version
`12.x`:

~~~~ {.python}
.......=DEFI_MATERIAU( UMAT=_F( LISTE_COEF = (C1,C2,....),),)
~~~~~~~~~~~~~~~~

### Version 13.x

For version greater than `13`, the syntax has evolved:

~~~~ {.python}
.......=DEFI_MATERIAU( MFRONT=_F( LISTE_COEF = (C1,C2,....)),)
~~~~~~~~~~~~~~~~

In both cases, `C1`, ..., `CN` are the material properties declared by the
`MFront` behaviour, **in the same exact order**.

## Definition of the material in the `COMPORTEMENT` field of mechanical operators

In the `COMPORTEMENT` part of the main computation instructions
(`STAT_NON_LINE`, `DYNA_NON_LINE`, `SIMU_POINT_MAT`, ...), the
behaviour has the name `MFRONT`. Here is an example of such
declaration:

### Version 12.x

~~~~ {.python}
COMPORTEMENT=_F ( RELATION    = 'UMAT',
	              LIBRAIRIE   = 'libAsterBehaviour.so',
			      NOM_ROUTINE = 'asterplasticity',
	              NB_VARI     = 19,
			      DEFORMATION = 'GDEF_LOG', )
~~~~~~~~~~~~~~~~

### Version 13.x

~~~~ {.python}
COMPORTEMENT=_F ( RELATION    = 'MFRONT',
	              LIBRAIRIE   = 'libAsterBehaviour.so',
			      NOM_ROUTINE = 'asterplasticity',
			      DEFORMATION = 'GDEF_LOG', )
~~~~~~~~~~~~~~~~

# A series of tutorials on video by N. Grilli

Nicolò Grilli (University of Bristol) published a series of three videos
showing in details how to make single and polycrystal simulations with
`MFront` and `code_aster`:

- <https://www.youtube.com/watch?v=GXVZwzprhBI>
- <https://www.youtube.com/watch?v=evNhkEcntaU>
- <https://www.youtube.com/watch?v=OAnJmMTGhE4>

The series adresses several advanced topics regarding the interface
between code_ater and MFront:

- handling orthotropy,
- postprocessing of the internal state variables,
- etc...

which can be very handy for a lot of users.

# Various examples

<div id="slideshow">
  <ul class="slides">
  <li>
  <video style="display:block; margin: 0 auto;" width="640" height="320" controls>
  <source src="media/SignoriniStrip.mp4" type="video/mp4">
  Your browser does not support the video tag.
  </video>
  </li>
  <li>
  <video style="display:block; margin: 0 auto;" width="640" height="320" controls>
  <source src="media/SignoriniCompression.mp4" type="video/mp4">
  Your browser does not support the video tag.
  </video>
  </li>
  </ul>
  <span class="arrow previous"></span>
  <span class="arrow next"></span>
</div>
<script src="http://ajax.googleapis.com/ajax/libs/jquery/1.4.2/jquery.min.js"></script>
<script src="js/slideshow.js"></script>

#References

<!-- Local IspellDict: english -->
