// @HEADER
// ************************************************************************
//
//                           Intrepid2 Package
//                 Copyright (2007) Sandia Corporation
//
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the Corporation nor the names of the
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY SANDIA CORPORATION "AS IS" AND ANY
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SANDIA CORPORATION OR THE
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Questions? Contact Kyungjoo Kim  (kyukim@sandia.gov), or
//                    Mauro Perego  (mperego@sandia.gov)
//
// ************************************************************************
// @HEADER

/** \file   Intrepid2_ProjectionToolsDefHCURL.hpp
    \brief  Header file for the Intrepid2::Experimental::ProjectionTools
            containing definitions for HCURL projections.
    \author Created by Mauro Perego
 */

#ifndef __INTREPID2_PROJECTIONTOOLSDEFHCURL_HPP__
#define __INTREPID2_PROJECTIONTOOLSDEFHCURL_HPP__

#include "Intrepid2_DefaultCubatureFactory.hpp"
#include "Intrepid2_ArrayTools.hpp"
#include "Intrepid2_FunctionSpaceTools.hpp"
#include "Intrepid2_NodalBasisFamily.hpp"


namespace Intrepid2 {

namespace FunctorsProjectionTools {

template<typename ViewType1, typename ViewType2, typename ViewType3, typename ViewType4>
struct ComputeBasisCoeffsOnEdges_HCurl {
  const ViewType1 basisTanAtBasisEPoints_;
  const ViewType1 basisAtBasisEPoints_;
  const ViewType2 basisEWeights_;
  const ViewType1 wTanBasisAtBasisEPoints_;
  const ViewType2 targetEWeights_;
  const ViewType1 basisAtTargetEPoints_;
  const ViewType1 wTanBasisAtTargetEPoints_;
  const ViewType3 tagToOrdinal_;
  const ViewType4 targetAtTargetEPoints_;
  const ViewType1 targetTanAtTargetEPoints_;
  const ViewType1 refEdgesTangent_;
  ordinal_type edgeCardinality_;
  ordinal_type offsetBasis_;
  ordinal_type offsetTarget_;
  ordinal_type edgeDim_;
  ordinal_type dim_;
  ordinal_type iedge_;

  ComputeBasisCoeffsOnEdges_HCurl(const ViewType1 basisTanAtBasisEPoints,
      const ViewType1 basisAtBasisEPoints, const ViewType2 basisEWeights,  const ViewType1 wTanBasisAtBasisEPoints,   const ViewType2 targetEWeights,
      const ViewType1 basisAtTargetEPoints, const ViewType1 wTanBasisAtTargetEPoints, const ViewType3 tagToOrdinal,
      const ViewType4 targetAtTargetEPoints, const ViewType1 targetTanAtTargetEPoints,
      const ViewType1 refEdgesTangent, ordinal_type edgeCardinality, ordinal_type offsetBasis,
      ordinal_type offsetTarget, ordinal_type edgeDim,
      ordinal_type dim, ordinal_type iedge) :
        basisTanAtBasisEPoints_(basisTanAtBasisEPoints),
        basisAtBasisEPoints_(basisAtBasisEPoints), basisEWeights_(basisEWeights), wTanBasisAtBasisEPoints_(wTanBasisAtBasisEPoints), targetEWeights_(targetEWeights),
        basisAtTargetEPoints_(basisAtTargetEPoints), wTanBasisAtTargetEPoints_(wTanBasisAtTargetEPoints),
        tagToOrdinal_(tagToOrdinal), targetAtTargetEPoints_(targetAtTargetEPoints),
        targetTanAtTargetEPoints_(targetTanAtTargetEPoints),
        refEdgesTangent_(refEdgesTangent), edgeCardinality_(edgeCardinality), offsetBasis_(offsetBasis),
        offsetTarget_(offsetTarget), edgeDim_(edgeDim), dim_(dim), iedge_(iedge)
  {}

  void
  KOKKOS_INLINE_FUNCTION
  operator()(const ordinal_type ic) const {

    ordinal_type numBasisEPoints = basisEWeights_.extent(0);
    ordinal_type numTargetEPoints = targetEWeights_.extent(0);
    for(ordinal_type j=0; j <edgeCardinality_; ++j) {
      ordinal_type jdof = tagToOrdinal_(edgeDim_, iedge_, j);
      for(ordinal_type iq=0; iq <numBasisEPoints; ++iq) {
        for(ordinal_type d=0; d <dim_; ++d)
          basisTanAtBasisEPoints_(ic,j,iq) += refEdgesTangent_(iedge_,d)*basisAtBasisEPoints_(ic,jdof,offsetBasis_+iq,d);
        wTanBasisAtBasisEPoints_(ic,j,iq) = basisTanAtBasisEPoints_(ic,j,iq)*basisEWeights_(iq);
      }

      for(ordinal_type iq=0; iq <numTargetEPoints; ++iq) {
        typename ViewType2::value_type  tmp = 0;
        for(ordinal_type d=0; d <dim_; ++d)
          tmp += refEdgesTangent_(iedge_,d)*basisAtTargetEPoints_(ic,jdof,offsetTarget_+iq,d);
        wTanBasisAtTargetEPoints_(ic,j,iq) = tmp*targetEWeights_(iq);
      }
    }
    for(ordinal_type iq=0; iq <numTargetEPoints; ++iq)
      for(ordinal_type d=0; d <dim_; ++d)
        targetTanAtTargetEPoints_(ic,iq) += refEdgesTangent_(iedge_,d)*targetAtTargetEPoints_(ic,offsetTarget_+iq,d);
  }
};


template<typename ViewType1, typename ViewType2, typename ViewType3, typename ViewType4,
typename ViewType5, typename ViewType6, typename ViewType7, typename ViewType8>
struct ComputeBasisCoeffsOnFaces_HCurl {
  const ViewType1 basisCoeffs_;
  const ViewType2 orts_;
  const ViewType3 negPartialProjTan_;
  const ViewType3 negPartialProjCurlNormal_;
  const ViewType3 hgradBasisGradAtBasisEPoints_;
  const ViewType3 wHgradBasisGradAtBasisEPoints_;
  const ViewType3 basisCurlAtBasisCurlEPoints_;
  const ViewType3 basisCurlNormalAtBasisCurlEPoints_;
  const ViewType3 basisAtBasisEPoints_;
  const ViewType3 normalTargetCurlAtTargetEPoints_;
  const ViewType3 basisTanAtBasisEPoints_;
  const ViewType3 hgradBasisGradAtTargetEPoints_;
  const ViewType3 wHgradBasisGradAtTargetEPoints_;
  const ViewType3 wNormalBasisCurlAtBasisCurlEPoints_;
  const ViewType3 basisCurlAtTargetCurlEPoints_;
  const ViewType3 wNormalBasisCurlBasisAtTargetCurlEPoints_;
  const ViewType4 targetAtTargetEPoints_;
  const ViewType3 targetTanAtTargetEPoints_;
  const ViewType4 targetCurlAtTargetCurlEPoints_;
  const ViewType5 basisEWeights_;
  const ViewType5 targetEWeights_;
  const ViewType5 basisCurlEWeights_;
  const ViewType5 targetCurlEWeights_;
  const ViewType6 tagToOrdinal_;
  const ViewType6 hGradTagToOrdinal_;
  const ViewType7 faceParametrization_;
  const ViewType8 computedDofs_;
  unsigned refTopologyKey_;
  ordinal_type offsetBasis_;
  ordinal_type offsetBasisCurl_;
  ordinal_type offsetTarget_;
  ordinal_type offsetTargetCurl_;
  ordinal_type iface_;
  ordinal_type hgradCardinality_;
  ordinal_type numFaces_;
  ordinal_type numFaceDofs_;
  ordinal_type numEdgeDofs_;
  ordinal_type faceDim_;
  ordinal_type dim_;




  ComputeBasisCoeffsOnFaces_HCurl(const ViewType1 basisCoeffs,
      const ViewType2 orts, const ViewType3 negPartialProjTan, const ViewType3 negPartialProjCurlNormal,
      const ViewType3 hgradBasisGradAtBasisEPoints, const ViewType3 wHgradBasisGradAtBasisEPoints,
      const ViewType3 basisCurlAtBasisCurlEPoints, const ViewType3 basisCurlNormalAtBasisCurlEPoints,
      const ViewType3 basisAtBasisEPoints,
      const ViewType3 normalTargetCurlAtTargetEPoints,
      const ViewType3 basisTanAtBasisEPoints,
      const ViewType3 hgradBasisGradAtTargetEPoints, const ViewType3 wHgradBasisGradAtTargetEPoints,
      const ViewType3 wNormalBasisCurlAtBasisCurlEPoints, const ViewType3 basisCurlAtTargetCurlEPoints,
      const ViewType3 wNormalBasisCurlBasisAtTargetCurlEPoints, const ViewType4 targetAtTargetEPoints,
      const ViewType3 targetTanAtTargetEPoints, const ViewType4 targetCurlAtTargetCurlEPoints,
      const ViewType5 basisEWeights, const ViewType5 targetEWeights,
      const ViewType5 basisCurlEWeights, const ViewType5 targetCurlEWeights, const ViewType6 tagToOrdinal,
      const ViewType6 hGradTagToOrdinal, const ViewType7 faceParametrization,
      const ViewType8 computedDofs, unsigned refTopologyKey, ordinal_type offsetBasis,
      ordinal_type offsetBasisCurl, ordinal_type offsetTarget,
      ordinal_type offsetTargetCurl, ordinal_type iface,
      ordinal_type hgradCardinality, ordinal_type numFaces,
      ordinal_type numFaceDofs, ordinal_type numEdgeDofs,
      ordinal_type faceDim, ordinal_type dim):
        basisCoeffs_(basisCoeffs),
        orts_(orts),  negPartialProjTan_(negPartialProjTan),  negPartialProjCurlNormal_(negPartialProjCurlNormal),
        hgradBasisGradAtBasisEPoints_(hgradBasisGradAtBasisEPoints),  wHgradBasisGradAtBasisEPoints_(wHgradBasisGradAtBasisEPoints),
        basisCurlAtBasisCurlEPoints_(basisCurlAtBasisCurlEPoints),  basisCurlNormalAtBasisCurlEPoints_(basisCurlNormalAtBasisCurlEPoints),
        basisAtBasisEPoints_(basisAtBasisEPoints),
        normalTargetCurlAtTargetEPoints_(normalTargetCurlAtTargetEPoints),  basisTanAtBasisEPoints_(basisTanAtBasisEPoints),
        hgradBasisGradAtTargetEPoints_(hgradBasisGradAtTargetEPoints),  wHgradBasisGradAtTargetEPoints_(wHgradBasisGradAtTargetEPoints),
        wNormalBasisCurlAtBasisCurlEPoints_(wNormalBasisCurlAtBasisCurlEPoints),  basisCurlAtTargetCurlEPoints_(basisCurlAtTargetCurlEPoints),
        wNormalBasisCurlBasisAtTargetCurlEPoints_(wNormalBasisCurlBasisAtTargetCurlEPoints),  targetAtTargetEPoints_(targetAtTargetEPoints),
        targetTanAtTargetEPoints_(targetTanAtTargetEPoints),  targetCurlAtTargetCurlEPoints_(targetCurlAtTargetCurlEPoints),
        basisEWeights_(basisEWeights),  targetEWeights_(targetEWeights),
        basisCurlEWeights_(basisCurlEWeights), targetCurlEWeights_(targetCurlEWeights),  tagToOrdinal_(tagToOrdinal),
        hGradTagToOrdinal_(hGradTagToOrdinal), faceParametrization_(faceParametrization),
        computedDofs_(computedDofs), refTopologyKey_(refTopologyKey), offsetBasis_(offsetBasis),
        offsetBasisCurl_(offsetBasisCurl), offsetTarget_(offsetTarget),
        offsetTargetCurl_(offsetTargetCurl), iface_(iface),
        hgradCardinality_(hgradCardinality), numFaces_(numFaces),
        numFaceDofs_(numFaceDofs), numEdgeDofs_(numEdgeDofs),
        faceDim_(faceDim), dim_(dim){}

  void
  KOKKOS_INLINE_FUNCTION
  operator()(const ordinal_type ic) const {

    ordinal_type fOrt[6];
    orts_(ic).getFaceOrientation(fOrt, numFaces_);

    ordinal_type ort = fOrt[iface_];

    typename ViewType3::value_type data[3*3];
    auto tangentsAndNormal = ViewType3(data, dim_, dim_);
    Impl::OrientationTools::getRefSideTangentsAndNormal(tangentsAndNormal, faceParametrization_,refTopologyKey_, iface_, ort);
    typename ViewType3::value_type n[3] = {tangentsAndNormal(2,0), tangentsAndNormal(2,1), tangentsAndNormal(2,2)};

    ordinal_type numBasisEPoints = basisEWeights_.extent(0);
    ordinal_type numTargetEPoints = targetEWeights_.extent(0);
    for(ordinal_type j=0; j <hgradCardinality_; ++j) {
      ordinal_type face_dof = hGradTagToOrdinal_(faceDim_, iface_, j);
      for(ordinal_type iq=0; iq <numBasisEPoints; ++iq) {
        for(ordinal_type d=0; d <dim_; ++d) {
          ordinal_type dp1 = (d+1) % dim_;
          ordinal_type dp2 = (d+2) % dim_;
          // basis \times n
          wHgradBasisGradAtBasisEPoints_(ic,j,iq,d) = (hgradBasisGradAtBasisEPoints_(face_dof,iq,dp1)*n[dp2] - hgradBasisGradAtBasisEPoints_(face_dof,iq,dp2)*n[dp1]) * basisEWeights_(iq);
        }
      }
      
      for(ordinal_type iq=0; iq <numTargetEPoints; ++iq) {
        for(ordinal_type d=0; d <dim_; ++d) {
          ordinal_type dp1 = (d+1) % dim_;
          ordinal_type dp2 = (d+2) % dim_;
          wHgradBasisGradAtTargetEPoints_(ic,j,iq,d) = (hgradBasisGradAtTargetEPoints_(face_dof,iq,dp1)*n[dp2] - hgradBasisGradAtTargetEPoints_(face_dof,iq,dp2)*n[dp1]) * targetEWeights_(iq);
        }
      }
    }

    ordinal_type numBasisCurlEPoints = basisCurlEWeights_.extent(0);
    for(ordinal_type j=0; j <numFaceDofs_; ++j) {
      ordinal_type jdof = tagToOrdinal_(faceDim_, iface_, j);
      for(ordinal_type iq=0; iq <numBasisEPoints; ++iq) {
        for(ordinal_type d=0; d <dim_; ++d) {
          ordinal_type dp1 = (d+1) % dim_;
          ordinal_type dp2 = (d+2) % dim_;
          // basis \times n
          basisTanAtBasisEPoints_(ic,j,iq,d) = basisAtBasisEPoints_(ic,jdof,offsetBasis_+iq,dp1)*n[dp2] - basisAtBasisEPoints_(ic,jdof,offsetBasis_+iq,dp2)*n[dp1];
        }
      }
      //Note: we are not considering the jacobian of the orientation map for normals since it is simply a scalar term for the integrals and it does not affect the projection
      for(ordinal_type iq=0; iq <numBasisCurlEPoints; ++iq) {
        for(ordinal_type d=0; d <dim_; ++d)
          basisCurlNormalAtBasisCurlEPoints_(ic,j,iq) += n[d]*basisCurlAtBasisCurlEPoints_(ic,jdof,offsetBasisCurl_+iq,d);
        wNormalBasisCurlAtBasisCurlEPoints_(ic,j,iq) = basisCurlNormalAtBasisCurlEPoints_(ic,j,iq) * basisCurlEWeights_(iq);
      }

      ordinal_type numTargetCurlEPoints = targetCurlEWeights_.extent(0);
      for(ordinal_type iq=0; iq <numTargetCurlEPoints; ++iq) {
        typename ViewType3::value_type tmp=0;
        // target \times n  
        for(ordinal_type d=0; d <dim_; ++d)
          tmp += n[d]*basisCurlAtTargetCurlEPoints_(ic,jdof,offsetTargetCurl_+iq,d);
        wNormalBasisCurlBasisAtTargetCurlEPoints_(ic,j,iq) = tmp*targetCurlEWeights_(iq);
      }
    }

    for(ordinal_type j=0; j <numEdgeDofs_; ++j) {
      ordinal_type jdof = computedDofs_(j);
      for(ordinal_type iq=0; iq <numBasisEPoints; ++iq)
        for(ordinal_type d=0; d <dim_; ++d) {
          ordinal_type dp1 = (d+1) % dim_;
          ordinal_type dp2 = (d+2) % dim_;
          negPartialProjCurlNormal_(ic,iq) -=  n[d]*basisCoeffs_(ic,jdof)*basisCurlAtBasisCurlEPoints_(ic,jdof,offsetBasisCurl_+iq,d);
          // basis \times n
          negPartialProjTan_(ic,iq,d) -=  (basisAtBasisEPoints_(ic,jdof,offsetBasis_+iq,dp1)*n[dp2] - basisAtBasisEPoints_(ic,jdof,offsetBasis_+iq,dp2)*n[dp1])*basisCoeffs_(ic,jdof);
        }
    }

    ordinal_type numTargetCurlEPoints = targetCurlEWeights_.extent(0);
    for(ordinal_type iq=0; iq <numTargetEPoints; ++iq)
      for(ordinal_type d=0; d <dim_; ++d) {
        ordinal_type dp1 = (d+1) % dim_;
        ordinal_type dp2 = (d+2) % dim_;
        // target \times n
        targetTanAtTargetEPoints_(ic,iq,d) = (targetAtTargetEPoints_(ic,offsetTarget_+iq,dp1)*n[dp2] - targetAtTargetEPoints_(ic,offsetTarget_+iq,dp2)*n[dp1]);
      }

    for(ordinal_type iq=0; iq <numTargetCurlEPoints; ++iq)
      for(ordinal_type d=0; d <dim_; ++d) {
        // target \cdot n
        normalTargetCurlAtTargetEPoints_(ic,iq) += n[d]*targetCurlAtTargetCurlEPoints_(ic,offsetTargetCurl_+iq,d);
      }
  }
};


template<typename ViewType1, typename ViewType2, typename ViewType3,
typename ViewType4, typename ViewType5>
struct ComputeBasisCoeffsOnCell_HCurl {
  const ViewType1 basisCoeffs_;
  const ViewType2 negPartialProj_;
  const ViewType2 negPartialProjCurl_;
  const ViewType2 cellBasisAtBasisEPoints_;
  const ViewType2 cellBasisCurlAtBasisCurlEPoints_;
  const ViewType2 basisAtBasisEPoints_;
  const ViewType2 hgradBasisGradAtBasisEPoints_;
  const ViewType2 basisCurlAtBasisCurlEPoints_;
  const ViewType2 hgradBasisGradAtTargetEPoints_;
  const ViewType2 basisCurlAtTargetCurlEPoints_;
  const ViewType3 basisEWeights_;
  const ViewType3 basisCurlEWeights_;
  const ViewType2 wHgradBasisGradAtBasisEPoints_;
  const ViewType2 wBasisCurlAtBasisCurlEPoints_;
  const ViewType3 targetEWeights_;
  const ViewType3 targetCurlEWeights_;
  const ViewType2 wHgradBasisGradAtTargetEPoints_;
  const ViewType2 wBasisCurlAtTargetCurlEPoints_;
  const ViewType4 computedDofs_;
  const ViewType5 tagToOrdinal_;
  const ViewType5 hGradTagToOrdinal_;
  ordinal_type numCellDofs_;
  ordinal_type hgradCardinality_;
  ordinal_type offsetBasis_;
  ordinal_type offsetBasisCurl_;
  ordinal_type offsetTargetCurl_;
  ordinal_type numEdgeFaceDofs_;
  ordinal_type dim_;
  ordinal_type derDim_;

  ComputeBasisCoeffsOnCell_HCurl(const ViewType1 basisCoeffs, ViewType2 negPartialProj,  ViewType2 negPartialProjCurl,
      const ViewType2 cellBasisAtBasisEPoints, const ViewType2 cellBasisCurlAtBasisCurlEPoints,
      const ViewType2 basisAtBasisEPoints, const ViewType2 hgradBasisGradAtBasisEPoints, const ViewType2 basisCurlAtBasisCurlEPoints,
      const ViewType2 hgradBasisGradAtTargetEPoints,   const ViewType2 basisCurlAtTargetCurlEPoints,
      const ViewType3 basisEWeights,  const ViewType3 basisCurlEWeights,
      const ViewType2 wHgradBasisGradAtBasisEPoints, const ViewType2 wBasisCurlAtBasisCurlEPoints,
      const ViewType3 targetEWeights, const ViewType3 targetCurlEWeights,
      const ViewType2 wHgradBasisGradAtTargetEPoints,
      const ViewType2 wBasisCurlAtTargetCurlEPoints, const ViewType4 computedDofs,
      const ViewType5 tagToOrdinal, const ViewType5 hGradTagToOrdinal,
      ordinal_type numCellDofs, ordinal_type hgradCardinality,
      ordinal_type offsetBasis, ordinal_type offsetBasisCurl, ordinal_type offsetTargetCurl,
      ordinal_type numEdgeFaceDofs, ordinal_type dim, ordinal_type derDim) :
        basisCoeffs_(basisCoeffs), negPartialProj_(negPartialProj), negPartialProjCurl_(negPartialProjCurl),
        cellBasisAtBasisEPoints_(cellBasisAtBasisEPoints), cellBasisCurlAtBasisCurlEPoints_(cellBasisCurlAtBasisCurlEPoints),
        basisAtBasisEPoints_(basisAtBasisEPoints), hgradBasisGradAtBasisEPoints_(hgradBasisGradAtBasisEPoints),
        basisCurlAtBasisCurlEPoints_(basisCurlAtBasisCurlEPoints),
        hgradBasisGradAtTargetEPoints_(hgradBasisGradAtTargetEPoints),
        basisCurlAtTargetCurlEPoints_(basisCurlAtTargetCurlEPoints),
        basisEWeights_(basisEWeights), basisCurlEWeights_(basisCurlEWeights),
        wHgradBasisGradAtBasisEPoints_(wHgradBasisGradAtBasisEPoints),
        wBasisCurlAtBasisCurlEPoints_(wBasisCurlAtBasisCurlEPoints),
        targetEWeights_(targetEWeights), targetCurlEWeights_(targetCurlEWeights),
        wHgradBasisGradAtTargetEPoints_(wHgradBasisGradAtTargetEPoints),
        wBasisCurlAtTargetCurlEPoints_(wBasisCurlAtTargetCurlEPoints),
        computedDofs_(computedDofs), tagToOrdinal_(tagToOrdinal), hGradTagToOrdinal_(hGradTagToOrdinal),
        numCellDofs_(numCellDofs), hgradCardinality_(hgradCardinality),
        offsetBasis_(offsetBasis), offsetBasisCurl_(offsetBasisCurl), offsetTargetCurl_(offsetTargetCurl),
        numEdgeFaceDofs_(numEdgeFaceDofs), dim_(dim), derDim_(derDim) {}

  void
  KOKKOS_INLINE_FUNCTION
  operator()(const ordinal_type ic) const {

    ordinal_type numBasisPoints = basisEWeights_.extent(0);
    ordinal_type numBasisCurlPoints = basisCurlEWeights_.extent(0);
    ordinal_type numTargetPoints = targetEWeights_.extent(0);
    ordinal_type numTargetCurlPoints = targetCurlEWeights_.extent(0);
    for(ordinal_type j=0; j <hgradCardinality_; ++j) {
      ordinal_type idof = hGradTagToOrdinal_(dim_, 0, j);
      for(ordinal_type d=0; d <dim_; ++d) {
        for(ordinal_type iq=0; iq <numBasisPoints; ++iq)
          wHgradBasisGradAtBasisEPoints_(ic,j,iq,d) = hgradBasisGradAtBasisEPoints_(idof,iq,d)*basisEWeights_(iq);
        for(ordinal_type iq=0; iq <numTargetPoints; ++iq)
          wHgradBasisGradAtTargetEPoints_(ic,j,iq,d) = hgradBasisGradAtTargetEPoints_(idof,iq,d)*targetEWeights_(iq);
      }
    }
    for(ordinal_type j=0; j <numCellDofs_; ++j) {
      ordinal_type idof = tagToOrdinal_(dim_, 0, j);
      for(ordinal_type d=0; d <dim_; ++d)
        for(ordinal_type iq=0; iq <numBasisPoints; ++iq)
          cellBasisAtBasisEPoints_(ic,j,iq,d)=basisAtBasisEPoints_(ic,idof,offsetBasis_+iq,d);

      for(ordinal_type d=0; d <derDim_; ++d) {
        for(ordinal_type iq=0; iq <numBasisCurlPoints; ++iq) {
          cellBasisCurlAtBasisCurlEPoints_(ic,j,iq,d)=basisCurlAtBasisCurlEPoints_(ic,idof,offsetBasisCurl_+iq,d);
          wBasisCurlAtBasisCurlEPoints_(ic,j,iq,d)=cellBasisCurlAtBasisCurlEPoints_(ic,j,iq,d)*basisCurlEWeights_(iq);
        }
        for(ordinal_type iq=0; iq <numTargetCurlPoints; ++iq)
          wBasisCurlAtTargetCurlEPoints_(ic,j,iq,d) = basisCurlAtTargetCurlEPoints_(ic,idof,offsetTargetCurl_+iq,d)*targetCurlEWeights_(iq);
      }
    }
    for(ordinal_type j=0; j < numEdgeFaceDofs_; ++j) {
      ordinal_type jdof = computedDofs_(j);
      for(ordinal_type d=0; d <derDim_; ++d)
        for(ordinal_type iq=0; iq <numBasisCurlPoints; ++iq)
          negPartialProjCurl_(ic,iq,d) -=  basisCoeffs_(ic,jdof)*basisCurlAtBasisCurlEPoints_(ic,jdof,offsetBasisCurl_+iq,d);
      for(ordinal_type d=0; d <dim_; ++d)
        for(ordinal_type iq=0; iq <numBasisPoints; ++iq)
          negPartialProj_(ic,iq,d) -=  basisCoeffs_(ic,jdof)*basisAtBasisEPoints_(ic,jdof,offsetBasis_+iq,d);
    }
  }
};

} // FunctorsProjectionTools namespace

#ifdef HAVE_INTREPID2_EXPERIMENTAL_NAMESPACE
namespace Experimental {


template<typename DeviceType>
template<typename BasisType,
typename ortValueType,       class ...ortProperties>
void
ProjectionTools<DeviceType>::getHCurlEvaluationPoints(typename BasisType::ScalarViewType ePoints,
    typename BasisType::ScalarViewType curlEPoints,
    const Kokkos::DynRankView<ortValueType,   ortProperties...>, //  orts,
    const BasisType* cellBasis,
    ProjectionStruct<DeviceType, typename BasisType::scalarType> * projStruct,
    const EvalPointsType evalPointType) {    
      RealSpaceTools<DeviceType>::clone(ePoints, projStruct->getAllEvalPoints(evalPointType));
      RealSpaceTools<DeviceType>::clone(curlEPoints, projStruct->getAllDerivEvalPoints(evalPointType));
}

template<typename DeviceType>
template<typename basisCoeffsValueType, class ...basisCoeffsProperties,
typename funValsValueType, class ...funValsProperties,
typename BasisType,
typename ortValueType,class ...ortProperties>
void
ProjectionTools<DeviceType>::getHCurlBasisCoeffs(Kokkos::DynRankView<basisCoeffsValueType,basisCoeffsProperties...> basisCoeffs,
    const Kokkos::DynRankView<funValsValueType,funValsProperties...> targetAtTargetEPoints,
    const Kokkos::DynRankView<funValsValueType,funValsProperties...> targetCurlAtTargetCurlEPoints,
    const typename BasisType::ScalarViewType, // targetEPoints,
    const typename BasisType::ScalarViewType, // targetCurlEPoints,
    const Kokkos::DynRankView<ortValueType,   ortProperties...>  orts,
    const BasisType* cellBasis,
    ProjectionStruct<DeviceType, typename BasisType::scalarType> * projStruct){
    
    getHCurlBasisCoeffs(basisCoeffs, targetAtTargetEPoints, targetCurlAtTargetCurlEPoints, orts, cellBasis, projStruct);
}
#endif


template<typename DeviceType>
template<typename basisCoeffsValueType, class ...basisCoeffsProperties,
typename funValsValueType, class ...funValsProperties,
typename BasisType,
typename ortValueType,class ...ortProperties>
void
ProjectionTools<DeviceType>::getHCurlBasisCoeffs(Kokkos::DynRankView<basisCoeffsValueType,basisCoeffsProperties...> basisCoeffs,
    const Kokkos::DynRankView<funValsValueType,funValsProperties...> targetAtTargetEPoints,
    const Kokkos::DynRankView<funValsValueType,funValsProperties...> targetCurlAtTargetCurlEPoints,
    const Kokkos::DynRankView<ortValueType,   ortProperties...>  orts,
    const BasisType* cellBasis,
    ProjectionStruct<DeviceType, typename BasisType::scalarType> * projStruct){

  typedef typename BasisType::scalarType scalarType;
  typedef Kokkos::DynRankView<scalarType,DeviceType> ScalarViewType;
  typedef Kokkos::pair<ordinal_type,ordinal_type> range_type;
  const auto cellTopo = cellBasis->getBaseCellTopology();
  ordinal_type dim = cellTopo.getDimension();
  ordinal_type basisCardinality = cellBasis->getCardinality();
  ordinal_type numCells = targetAtTargetEPoints.extent(0);
  const ordinal_type edgeDim = 1;
  const ordinal_type faceDim = 2;
  const ordinal_type derDim = dim == 3 ? dim : 1;

  const Kokkos::RangePolicy<ExecSpaceType> policy(0, numCells);

  const std::string& name = cellBasis->getName();

  ordinal_type numEdges = (cellBasis->getDofCount(1, 0) > 0) ? cellTopo.getEdgeCount() : 0;
  ordinal_type numFaces = (cellBasis->getDofCount(2, 0) > 0) ? cellTopo.getFaceCount() : 0;

  ScalarViewType refEdgesTangent("refEdgesTangent", numEdges, dim);
  ScalarViewType refFacesTangents("refFaceTangents", numFaces, dim, 2);
  ScalarViewType refFacesNormal("refFaceNormal", numFaces, dim);

  ordinal_type numEdgeDofs(0);
  for(ordinal_type ie=0; ie<numEdges; ++ie)
    numEdgeDofs += cellBasis->getDofCount(edgeDim,ie);

  ordinal_type numTotalFaceDofs(0);
  for(ordinal_type iface=0; iface<numFaces; ++iface)
    numTotalFaceDofs += cellBasis->getDofCount(faceDim,iface);

  auto tagToOrdinal = Kokkos::create_mirror_view_and_copy(MemSpaceType(), cellBasis->getAllDofOrdinal());

  Kokkos::View<ordinal_type*, DeviceType> computedDofs("computedDofs",numEdgeDofs+numTotalFaceDofs);

  auto targetEPointsRange  = projStruct->getTargetPointsRange();
  auto targetCurlEPointsRange  = projStruct->getTargetDerivPointsRange();

  auto basisEPointsRange = projStruct->getBasisPointsRange();
  auto basisCurlEPointsRange = projStruct->getBasisDerivPointsRange();

  auto refTopologyKey = projStruct->getTopologyKey();

  ordinal_type numTotalBasisEPoints = projStruct->getNumBasisEvalPoints(), numTotalBasisCurlEPoints = projStruct->getNumBasisDerivEvalPoints();
  auto basisEPoints = projStruct->getAllEvalPoints(EvalPointsType::BASIS);
  auto basisCurlEPoints = projStruct->getAllDerivEvalPoints(EvalPointsType::BASIS);

  ordinal_type numTotalTargetEPoints = projStruct->getNumTargetEvalPoints(), numTotalTargetCurlEPoints = projStruct->getNumTargetDerivEvalPoints();
  auto targetEPoints = projStruct->getAllEvalPoints(EvalPointsType::TARGET);
  auto targetCurlEPoints = projStruct->getAllDerivEvalPoints(EvalPointsType::TARGET);

  ScalarViewType basisAtBasisEPoints("basisAtBasisEPoints",numCells,basisCardinality, numTotalBasisEPoints, dim);
  ScalarViewType basisAtTargetEPoints("basisAtTargetEPoints",numCells,basisCardinality, numTotalTargetEPoints, dim);
  {
    ScalarViewType nonOrientedBasisAtBasisEPoints("nonOrientedBasisAtEPoints",basisCardinality, numTotalBasisEPoints, dim);
    ScalarViewType nonOrientedBasisAtTargetEPoints("nonOrientedBasisAtTargetEPoints",basisCardinality, numTotalTargetEPoints, dim);
    cellBasis->getValues(nonOrientedBasisAtTargetEPoints, targetEPoints);
    cellBasis->getValues(nonOrientedBasisAtBasisEPoints, basisEPoints);

    OrientationTools<DeviceType>::modifyBasisByOrientation(basisAtBasisEPoints, nonOrientedBasisAtBasisEPoints, orts, cellBasis);
    OrientationTools<DeviceType>::modifyBasisByOrientation(basisAtTargetEPoints, nonOrientedBasisAtTargetEPoints, orts, cellBasis);
  }

  ScalarViewType basisCurlAtBasisCurlEPoints;
  ScalarViewType basisCurlAtTargetCurlEPoints;
  if(numTotalBasisCurlEPoints>0) {
    ScalarViewType nonOrientedBasisCurlAtTargetCurlEPoints, nonOrientedBasisCurlAtBasisCurlEPoints;
    if (dim == 3) {
      basisCurlAtBasisCurlEPoints = ScalarViewType ("basisCurlAtBasisCurlEPoints",numCells,basisCardinality, numTotalBasisCurlEPoints, dim);
      nonOrientedBasisCurlAtBasisCurlEPoints = ScalarViewType ("nonOrientedBasisCurlAtBasisCurlEPoints", basisCardinality, numTotalBasisCurlEPoints, dim);
      basisCurlAtTargetCurlEPoints = ScalarViewType("basisCurlAtTargetCurlEPoints",numCells,basisCardinality, numTotalTargetCurlEPoints, dim);
      nonOrientedBasisCurlAtTargetCurlEPoints = ScalarViewType("nonOrientedBasisCurlAtTargetCurlEPoints",basisCardinality, numTotalTargetCurlEPoints, dim);
    } else {
      basisCurlAtBasisCurlEPoints = ScalarViewType ("basisCurlAtBasisCurlEPoints",numCells,basisCardinality, numTotalBasisCurlEPoints);
      nonOrientedBasisCurlAtBasisCurlEPoints = ScalarViewType ("nonOrientedBasisCurlAtBasisCurlEPoints",basisCardinality, numTotalBasisCurlEPoints);
      basisCurlAtTargetCurlEPoints = ScalarViewType("basisCurlAtTargetCurlEPoints",numCells,basisCardinality, numTotalTargetCurlEPoints);
      nonOrientedBasisCurlAtTargetCurlEPoints = ScalarViewType("nonOrientedBasisCurlAtTargetCurlEPoints",basisCardinality, numTotalTargetCurlEPoints);
    }

    cellBasis->getValues(nonOrientedBasisCurlAtBasisCurlEPoints, basisCurlEPoints,OPERATOR_CURL);
    cellBasis->getValues(nonOrientedBasisCurlAtTargetCurlEPoints, targetCurlEPoints,OPERATOR_CURL);
    OrientationTools<DeviceType>::modifyBasisByOrientation(basisCurlAtBasisCurlEPoints, nonOrientedBasisCurlAtBasisCurlEPoints, orts, cellBasis);
    OrientationTools<DeviceType>::modifyBasisByOrientation(basisCurlAtTargetCurlEPoints, nonOrientedBasisCurlAtTargetCurlEPoints, orts, cellBasis);
  }

  ordinal_type computedDofsCount = 0;
  for(ordinal_type ie=0; ie<numEdges; ++ie)  {

    ordinal_type edgeCardinality = cellBasis->getDofCount(edgeDim,ie);
    ordinal_type numBasisEPoints = range_size(basisEPointsRange(edgeDim, ie));
    ordinal_type numTargetEPoints = range_size(targetEPointsRange(edgeDim, ie));

    {
      auto refEdgeTan = Kokkos::subview(refEdgesTangent, ie, Kokkos::ALL());
      CellTools<DeviceType>::getReferenceEdgeTangent(refEdgeTan, ie, cellTopo);
    }

    ScalarViewType basisTanAtBasisEPoints("basisTanAtBasisEPoints",numCells,edgeCardinality, numBasisEPoints);
    ScalarViewType weightedTanBasisAtBasisEPoints("weightedTanBasisAtBasisEPoints",numCells,edgeCardinality, numBasisEPoints);
    ScalarViewType weightedTanBasisAtTargetEPoints("weightedTanBasisAtTargetEPoints",numCells,edgeCardinality, numTargetEPoints);
    ScalarViewType targetTanAtTargetEPoints("normalTargetAtTargetEPoints",numCells, numTargetEPoints);

    auto targetEWeights = Kokkos::create_mirror_view_and_copy(MemSpaceType(),projStruct->getTargetEvalWeights(edgeDim,ie));
    auto basisEWeights = Kokkos::create_mirror_view_and_copy(MemSpaceType(),projStruct->getBasisEvalWeights(edgeDim,ie));

    //Note: we are not considering the jacobian of the orientation map since it is simply a scalar term for the integrals and it does not affect the projection
    ordinal_type offsetBasis = basisEPointsRange(edgeDim, ie).first;
    ordinal_type offsetTarget = targetEPointsRange(edgeDim, ie).first;

    using functorTypeEdge = FunctorsProjectionTools::ComputeBasisCoeffsOnEdges_HCurl<ScalarViewType,  decltype(basisEWeights), decltype(tagToOrdinal), decltype(targetAtTargetEPoints)>;
    Kokkos::parallel_for(policy, functorTypeEdge(basisTanAtBasisEPoints,basisAtBasisEPoints,basisEWeights,
        weightedTanBasisAtBasisEPoints, targetEWeights,
        basisAtTargetEPoints, weightedTanBasisAtTargetEPoints, tagToOrdinal,
        targetAtTargetEPoints, targetTanAtTargetEPoints,
        refEdgesTangent, edgeCardinality, offsetBasis,
        offsetTarget, edgeDim,
        dim, ie));

    ScalarViewType edgeMassMat_("edgeMassMat_", numCells, edgeCardinality+1, edgeCardinality+1),
        edgeRhsMat_("rhsMat_", numCells, edgeCardinality+1);

    ScalarViewType eWeights_("eWeights_", numCells, 1, basisEWeights.extent(0)), targetEWeights_("targetEWeights", numCells, 1, targetEWeights.extent(0));
    RealSpaceTools<DeviceType>::clone(eWeights_, basisEWeights);
    RealSpaceTools<DeviceType>::clone(targetEWeights_, targetEWeights);

    range_type range_H(0, edgeCardinality);
    range_type range_B(edgeCardinality, edgeCardinality+1);
    FunctionSpaceTools<DeviceType >::integrate(Kokkos::subview(edgeMassMat_,Kokkos::ALL(),range_H,range_H), basisTanAtBasisEPoints, weightedTanBasisAtBasisEPoints);
    FunctionSpaceTools<DeviceType >::integrate(Kokkos::subview(edgeMassMat_,Kokkos::ALL(),range_H,range_B), basisTanAtBasisEPoints, eWeights_);
    FunctionSpaceTools<DeviceType >::integrate(Kokkos::subview(edgeRhsMat_,Kokkos::ALL(),range_H), targetTanAtTargetEPoints, weightedTanBasisAtTargetEPoints);
    FunctionSpaceTools<DeviceType >::integrate(Kokkos::subview(edgeRhsMat_,Kokkos::ALL(),range_B), targetTanAtTargetEPoints, targetEWeights_);

    typedef Kokkos::DynRankView<scalarType, Kokkos::LayoutRight, DeviceType> WorkArrayViewType;
    ScalarViewType t_("t",numCells, edgeCardinality+1);
    WorkArrayViewType w_("w",numCells, edgeCardinality+1);

    auto edgeDofs = Kokkos::subview(tagToOrdinal, edgeDim, ie, range_type(0,edgeCardinality));
    ElemSystem edgeSystem("edgeSystem", false);
    edgeSystem.solve(basisCoeffs, edgeMassMat_, edgeRhsMat_, t_, w_, edgeDofs, edgeCardinality, 1);

    auto computedEdgeDofs = Kokkos::subview(computedDofs, range_type(computedDofsCount,computedDofsCount+edgeCardinality));
    deep_copy(computedEdgeDofs, edgeDofs);
    computedDofsCount += edgeCardinality;

  }

  typename RefSubcellParametrization<DeviceType>::ConstViewType  subcellParamFace;
  Basis<DeviceType,scalarType,scalarType> *hgradBasis = NULL;
  if(numFaces>0) {
    subcellParamFace = RefSubcellParametrization<DeviceType>::get(faceDim, cellBasis->getBaseCellTopology().getKey());
    if(cellTopo.getKey() == shards::getCellTopologyData<shards::Hexahedron<8> >()->key)
      hgradBasis = new Basis_HGRAD_HEX_Cn_FEM<DeviceType,scalarType,scalarType>(cellBasis->getDegree(),POINTTYPE_WARPBLEND);
    else if(cellTopo.getKey() == shards::getCellTopologyData<shards::Tetrahedron<4> >()->key)
      hgradBasis = new Basis_HGRAD_TET_Cn_FEM<DeviceType,scalarType,scalarType>(cellBasis->getDegree(),POINTTYPE_WARPBLEND);
    else if(cellTopo.getKey() == shards::getCellTopologyData<shards::Wedge<6> >()->key) {
      hgradBasis = new typename DerivedNodalBasisFamily<DeviceType,scalarType,scalarType>::HGRAD_WEDGE(cellBasis->getDegree(),POINTTYPE_WARPBLEND);
    }
    else {
      std::stringstream ss;
      ss << ">>> ERROR (Intrepid2::ProjectionTools::getHCurlBasisCoeffs): "
          << "Method not implemented for basis " << name;
      INTREPID2_TEST_FOR_EXCEPTION( true, std::runtime_error, ss.str().c_str() );
    }
  }
  for(ordinal_type iface=0; iface<numFaces; ++iface) {

    ordinal_type numTargetEPoints = range_size(targetEPointsRange(faceDim, iface));
    ordinal_type numTargetCurlEPoints = range_size(targetCurlEPointsRange(faceDim, iface));
    ordinal_type numBasisEPoints = range_size(basisEPointsRange(faceDim, iface));
    ordinal_type numBasisCurlEPoints = range_size(basisCurlEPointsRange(faceDim, iface));

    ordinal_type numFaceDofs = cellBasis->getDofCount(faceDim,iface);

    ScalarViewType hgradBasisGradAtBasisEPoints("hgradBasisGradAtBasisEPoints", hgradBasis->getCardinality(), numBasisEPoints, dim);
    ScalarViewType hgradBasisGradAtTargetEPoints("hgradBasisGradAtTargetEPoints", hgradBasis->getCardinality(), numTargetEPoints, dim);

    ordinal_type hgradCardinality = hgradBasis->getDofCount(faceDim,iface);

    hgradBasis->getValues(hgradBasisGradAtBasisEPoints, Kokkos::subview(basisEPoints, basisEPointsRange(faceDim, iface), Kokkos::ALL()), OPERATOR_GRAD);
    hgradBasis->getValues(hgradBasisGradAtTargetEPoints, Kokkos::subview(targetEPoints, targetEPointsRange(faceDim, iface), Kokkos::ALL()), OPERATOR_GRAD);
    
    //no need to orient these basis as they act locally as test functions.

    auto hGradTagToOrdinal = Kokkos::create_mirror_view_and_copy(MemSpaceType(), hgradBasis->getAllDofOrdinal());

    ScalarViewType basisTanAtBasisEPoints("basisTanAtBasisEPoints",numCells,numFaceDofs, numBasisEPoints,dim);
    ScalarViewType basisCurlNormalAtBasisCurlEPoints("normaBasisCurlAtBasisEPoints",numCells,numFaceDofs, numBasisCurlEPoints);
    ScalarViewType wNormalBasisCurlAtBasisCurlEPoints("weightedNormalBasisCurlAtBasisEPoints",numCells,numFaceDofs, numBasisCurlEPoints);

    ScalarViewType targetTanAtTargetEPoints("targetTanAtTargetEPoints",numCells, numTargetEPoints, dim);
    ScalarViewType normalTargetCurlAtTargetEPoints("normalTargetCurlAtTargetEPoints",numCells, numTargetCurlEPoints);
    ScalarViewType wNormalBasisCurlBasisAtTargetCurlEPoints("weightedNormalBasisCurlAtTargetCurlEPoints",numCells,numFaceDofs, numTargetCurlEPoints);

    ScalarViewType wHgradBasisGradAtBasisEPoints("wHgradBasisGradAtBasisEPoints",numCells, hgradCardinality, numBasisEPoints, dim);
    ScalarViewType wHgradBasisGradAtTargetEPoints("wHgradBasisGradAtTargetEPoints",numCells, hgradCardinality, numTargetEPoints, dim);
  
    ScalarViewType negPartialProjCurlNormal("mNormalComputedProjection", numCells,numBasisEPoints);
    ScalarViewType negPartialProjTan("negPartialProjTan", numCells,numBasisEPoints,dim);

    ordinal_type offsetBasis = basisEPointsRange(faceDim, iface).first;
    ordinal_type offsetBasisCurl = basisCurlEPointsRange(faceDim, iface).first;
    ordinal_type offsetTarget = targetEPointsRange(faceDim, iface).first;
    ordinal_type offsetTargetCurl = targetCurlEPointsRange(faceDim, iface).first;

    auto basisEWeights = Kokkos::create_mirror_view_and_copy(MemSpaceType(),projStruct->getBasisEvalWeights(faceDim,iface));
    auto targetEWeights = Kokkos::create_mirror_view_and_copy(MemSpaceType(),projStruct->getTargetEvalWeights(faceDim,iface));
    auto targetCurlEWeights = Kokkos::create_mirror_view_and_copy(MemSpaceType(),projStruct->getTargetDerivEvalWeights(faceDim,iface));
    auto basisCurlEWeights = Kokkos::create_mirror_view_and_copy(MemSpaceType(),projStruct->getBasisDerivEvalWeights(faceDim,iface));
    const auto topoKey = refTopologyKey(faceDim, iface);
    using functorTypeFaces = FunctorsProjectionTools::ComputeBasisCoeffsOnFaces_HCurl<decltype(basisCoeffs), decltype(orts), ScalarViewType,  decltype(targetAtTargetEPoints), decltype(basisEWeights),
        decltype(tagToOrdinal), decltype(subcellParamFace), decltype(computedDofs)>;
    Kokkos::parallel_for(policy, functorTypeFaces(basisCoeffs,
        orts, negPartialProjTan, negPartialProjCurlNormal,
        hgradBasisGradAtBasisEPoints, wHgradBasisGradAtBasisEPoints,
        basisCurlAtBasisCurlEPoints, basisCurlNormalAtBasisCurlEPoints,
        basisAtBasisEPoints,
        normalTargetCurlAtTargetEPoints, basisTanAtBasisEPoints,
        hgradBasisGradAtTargetEPoints, wHgradBasisGradAtTargetEPoints,
        wNormalBasisCurlAtBasisCurlEPoints, basisCurlAtTargetCurlEPoints,
        wNormalBasisCurlBasisAtTargetCurlEPoints, targetAtTargetEPoints,
        targetTanAtTargetEPoints, targetCurlAtTargetCurlEPoints,
        basisEWeights, targetEWeights,
        basisCurlEWeights, targetCurlEWeights, tagToOrdinal,
        hGradTagToOrdinal, subcellParamFace,
        computedDofs, topoKey, offsetBasis,
        offsetBasisCurl, offsetTarget,
        offsetTargetCurl, iface,
        hgradCardinality, numFaces,
        numFaceDofs, numEdgeDofs,
        faceDim, dim));


    ScalarViewType faceMassMat_("faceMassMat_", numCells, numFaceDofs+hgradCardinality, numFaceDofs+hgradCardinality),
        faceRhsMat_("rhsMat_", numCells, numFaceDofs+hgradCardinality);
    range_type range_H(0, numFaceDofs);
    range_type range_B(numFaceDofs, numFaceDofs+hgradCardinality);
    FunctionSpaceTools<DeviceType >::integrate(Kokkos::subview(faceMassMat_,Kokkos::ALL(),range_H,range_H), basisCurlNormalAtBasisCurlEPoints, wNormalBasisCurlAtBasisCurlEPoints);
    FunctionSpaceTools<DeviceType >::integrate(Kokkos::subview(faceMassMat_,Kokkos::ALL(),range_H,range_B), basisTanAtBasisEPoints, wHgradBasisGradAtBasisEPoints);

    FunctionSpaceTools<DeviceType >::integrate(Kokkos::subview(faceRhsMat_,Kokkos::ALL(),range_H), normalTargetCurlAtTargetEPoints, wNormalBasisCurlBasisAtTargetCurlEPoints);
    FunctionSpaceTools<DeviceType >::integrate(Kokkos::subview(faceRhsMat_,Kokkos::ALL(),range_H), negPartialProjCurlNormal, wNormalBasisCurlAtBasisCurlEPoints,true);

    FunctionSpaceTools<DeviceType >::integrate(Kokkos::subview(faceRhsMat_,Kokkos::ALL(),range_B), targetTanAtTargetEPoints, wHgradBasisGradAtTargetEPoints);
    FunctionSpaceTools<DeviceType >::integrate(Kokkos::subview(faceRhsMat_,Kokkos::ALL(),range_B), negPartialProjTan, wHgradBasisGradAtBasisEPoints,true);


    typedef Kokkos::DynRankView<scalarType, Kokkos::LayoutRight, DeviceType> WorkArrayViewType;
    ScalarViewType t_("t",numCells, numFaceDofs+hgradCardinality);
    WorkArrayViewType w_("w",numCells, numFaceDofs+hgradCardinality);

    auto faceDofs = Kokkos::subview(tagToOrdinal, faceDim, iface, range_type(0,numFaceDofs));
    ElemSystem faceSystem( "faceSystem", false);
    faceSystem.solve(basisCoeffs, faceMassMat_, faceRhsMat_, t_, w_, faceDofs, numFaceDofs, hgradCardinality);

    auto computedFaceDofs = Kokkos::subview(computedDofs, range_type(computedDofsCount,computedDofsCount+numFaceDofs));
    deep_copy(computedFaceDofs, faceDofs);
    computedDofsCount += numFaceDofs;

  }
  delete hgradBasis;

  ordinal_type numCellDofs = cellBasis->getDofCount(dim,0);
  if(numCellDofs>0) {
    if(cellTopo.getKey() == shards::getCellTopologyData<shards::Hexahedron<8> >()->key)
      hgradBasis = new Basis_HGRAD_HEX_Cn_FEM<DeviceType,scalarType,scalarType>(cellBasis->getDegree());
    else if(cellTopo.getKey() == shards::getCellTopologyData<shards::Tetrahedron<4> >()->key)
      hgradBasis = new Basis_HGRAD_TET_Cn_FEM<DeviceType,scalarType,scalarType>(cellBasis->getDegree(),POINTTYPE_WARPBLEND);
    else if(cellTopo.getKey() == shards::getCellTopologyData<shards::Wedge<6> >()->key)
      hgradBasis = new typename DerivedNodalBasisFamily<DeviceType,scalarType,scalarType>::HGRAD_WEDGE(cellBasis->getDegree(),POINTTYPE_WARPBLEND);
    else if(cellTopo.getKey() == shards::getCellTopologyData<shards::Triangle<3> >()->key)
      hgradBasis = new Basis_HGRAD_TRI_Cn_FEM<DeviceType,scalarType,scalarType>(cellBasis->getDegree(),POINTTYPE_WARPBLEND);
    else if(cellTopo.getKey() == shards::getCellTopologyData<shards::Quadrilateral<4> >()->key)
      hgradBasis = new Basis_HGRAD_QUAD_Cn_FEM<DeviceType,scalarType,scalarType>(cellBasis->getDegree(),POINTTYPE_WARPBLEND);
    else  {
      std::stringstream ss;
      ss << ">>> ERROR (Intrepid2::ProjectionTools::getHCurlBasisCoeffs): "
          << "Method not implemented for basis " << name;
      INTREPID2_TEST_FOR_EXCEPTION( true, std::runtime_error, ss.str().c_str() );
    }

    range_type cellPointsRange = targetEPointsRange(dim, 0);
    range_type cellCurlPointsRange = targetCurlEPointsRange(dim, 0);

    ordinal_type numTargetCurlEPoints = range_size(targetCurlEPointsRange(dim,0));
    ordinal_type numBasisCurlEPoints = range_size(basisCurlEPointsRange(dim,0));
    ordinal_type numBasisEPoints = range_size(basisEPointsRange(dim,0));
    ordinal_type numTargetEPoints = range_size(targetEPointsRange(dim,0));

    ScalarViewType hgradBasisGradAtBasisEPoints("hgradBasisGradAtBasisEPoints",hgradBasis->getCardinality(), numBasisEPoints, dim);
    ScalarViewType hgradBasisGradAtTargetEPoints("hgradBasisGradAtTargetEPoints",hgradBasis->getCardinality(), numTargetEPoints, dim);

    ordinal_type hgradCardinality = hgradBasis->getDofCount(dim,0);
    ScalarViewType wHgradBasisGradAtTargetEPoints("wHgradBasisGradAtTargetEPoints",numCells, hgradCardinality, numTargetEPoints, dim);
    ScalarViewType wHgradBasisGradAtBasisEPoints("wHgradBasisGradAtBasisEPoints",numCells, hgradCardinality, numBasisEPoints, dim);

    hgradBasis->getValues(hgradBasisGradAtBasisEPoints,Kokkos::subview(basisEPoints, basisEPointsRange(dim, 0), Kokkos::ALL()), OPERATOR_GRAD);
    hgradBasis->getValues(hgradBasisGradAtTargetEPoints,Kokkos::subview(targetEPoints, targetEPointsRange(dim, 0), Kokkos::ALL()),OPERATOR_GRAD);

    ScalarViewType cellBasisAtBasisEPoints("basisCellAtEPoints",numCells,numCellDofs, numBasisEPoints, dim);
    ScalarViewType cellBasisCurlAtCurlEPoints("cellBasisCurlAtCurlEPoints",numCells,numCellDofs, numBasisCurlEPoints, derDim);
    ScalarViewType negPartialProjCurl("negPartialProjCurl", numCells, numBasisEPoints, derDim);
    ScalarViewType negPartialProj("negPartialProj", numCells, numBasisEPoints, dim);
    ScalarViewType wBasisCurlAtCurlEPoints("weightedBasisCurlAtBasisEPoints",numCells,numCellDofs, numBasisCurlEPoints,derDim);
    ScalarViewType wBasisCurlBasisAtTargetCurlEPoints("weightedBasisCurlAtTargetCurlEPoints",numCells,numCellDofs, numTargetCurlEPoints,derDim);

    auto targetEWeights = Kokkos::create_mirror_view_and_copy(MemSpaceType(),projStruct->getTargetEvalWeights(dim,0));
    auto basisEWeights = Kokkos::create_mirror_view_and_copy(MemSpaceType(),projStruct->getBasisEvalWeights(dim,0));
    auto targetCurlEWeights = Kokkos::create_mirror_view_and_copy(MemSpaceType(),projStruct->getTargetDerivEvalWeights(dim,0));
    auto basisCurlEWeights = Kokkos::create_mirror_view_and_copy(MemSpaceType(),projStruct->getBasisDerivEvalWeights(dim,0));
    ordinal_type offsetBasis = basisEPointsRange(dim, 0).first;
    ordinal_type offsetBasisCurl = basisCurlEPointsRange(dim, 0).first;
    ordinal_type offsetTargetCurl = targetCurlEPointsRange(dim, 0).first;


    auto hGradTagToOrdinal = Kokkos::create_mirror_view_and_copy(MemSpaceType(), hgradBasis->getAllDofOrdinal());

    using functorTypeCell = FunctorsProjectionTools::ComputeBasisCoeffsOnCell_HCurl<decltype(basisCoeffs), ScalarViewType,  decltype(basisEWeights),
        decltype(computedDofs), decltype(tagToOrdinal)>;
    Kokkos::parallel_for(policy, functorTypeCell(basisCoeffs, negPartialProj, negPartialProjCurl,
        cellBasisAtBasisEPoints, cellBasisCurlAtCurlEPoints,
        basisAtBasisEPoints, hgradBasisGradAtBasisEPoints, basisCurlAtBasisCurlEPoints,
        hgradBasisGradAtTargetEPoints,  basisCurlAtTargetCurlEPoints,
        basisEWeights, basisCurlEWeights, wHgradBasisGradAtBasisEPoints,
        wBasisCurlAtCurlEPoints, targetEWeights, targetCurlEWeights,
        wHgradBasisGradAtTargetEPoints,
        wBasisCurlBasisAtTargetCurlEPoints, computedDofs,
        tagToOrdinal, hGradTagToOrdinal,
        numCellDofs, hgradCardinality,
        offsetBasis, offsetBasisCurl,  offsetTargetCurl,
        numEdgeDofs+numTotalFaceDofs, dim, derDim));

    ScalarViewType cellMassMat_("cellMassMat_", numCells, numCellDofs+hgradCardinality, numCellDofs+hgradCardinality),
        cellRhsMat_("rhsMat_", numCells, numCellDofs+hgradCardinality);

    range_type range_H(0, numCellDofs);
    range_type range_B(numCellDofs, numCellDofs+hgradCardinality);
    FunctionSpaceTools<DeviceType >::integrate(Kokkos::subview(cellMassMat_,Kokkos::ALL(),range_H,range_H), cellBasisCurlAtCurlEPoints, wBasisCurlAtCurlEPoints);
    FunctionSpaceTools<DeviceType >::integrate(Kokkos::subview(cellMassMat_,Kokkos::ALL(),range_H,range_B), cellBasisAtBasisEPoints, wHgradBasisGradAtBasisEPoints);
    if(dim==3)
      FunctionSpaceTools<DeviceType >::integrate(Kokkos::subview(cellRhsMat_,Kokkos::ALL(),range_H), Kokkos::subview(targetCurlAtTargetCurlEPoints,Kokkos::ALL(),cellCurlPointsRange,Kokkos::ALL()), wBasisCurlBasisAtTargetCurlEPoints);
    else
      FunctionSpaceTools<DeviceType >::integrate(Kokkos::subview(cellRhsMat_,Kokkos::ALL(),range_H), Kokkos::subview(targetCurlAtTargetCurlEPoints,Kokkos::ALL(),cellCurlPointsRange), Kokkos::subview(wBasisCurlBasisAtTargetCurlEPoints,Kokkos::ALL(),Kokkos::ALL(),Kokkos::ALL(),0));
    FunctionSpaceTools<DeviceType >::integrate(Kokkos::subview(cellRhsMat_,Kokkos::ALL(),range_H), negPartialProjCurl, wBasisCurlAtCurlEPoints, true);

    FunctionSpaceTools<DeviceType >::integrate(Kokkos::subview(cellRhsMat_,Kokkos::ALL(),range_B), Kokkos::subview(targetAtTargetEPoints,Kokkos::ALL(),cellPointsRange,Kokkos::ALL()), wHgradBasisGradAtTargetEPoints);
    FunctionSpaceTools<DeviceType >::integrate(Kokkos::subview(cellRhsMat_,Kokkos::ALL(),range_B), negPartialProj, wHgradBasisGradAtBasisEPoints, true);

    typedef Kokkos::DynRankView<scalarType, Kokkos::LayoutRight, DeviceType> WorkArrayViewType;
    ScalarViewType t_("t",numCells, numCellDofs+hgradCardinality);
    WorkArrayViewType w_("w",numCells, numCellDofs+hgradCardinality);

    auto cellDofs = Kokkos::subview(tagToOrdinal, dim, 0, Kokkos::ALL());
    ElemSystem cellSystem( "cellSystem", true);
    cellSystem.solve(basisCoeffs, cellMassMat_, cellRhsMat_, t_, w_, cellDofs, numCellDofs, hgradCardinality);

    delete hgradBasis;
  }
}
#ifdef HAVE_INTREPID2_EXPERIMENTAL_NAMESPACE
}
#endif
}   // Intrepid2 namespace

#endif

