--
--  Copyright (C) 2014-2022, AdaCore
--  SPDX-License-Identifier: Apache-2.0
--

with Ada.Containers;                  use Ada.Containers;
with Ada.Containers.Hashed_Maps;
with Ada.Containers.Vectors;
with Ada.Directories;
with Ada.Exceptions;
with Ada.Finalization;
with Ada.Strings.Unbounded; use Ada.Strings.Unbounded;
with Ada.Strings.Wide_Wide_Unbounded; use Ada.Strings.Wide_Wide_Unbounded;
pragma Warnings (Off, "internal");
with Ada.Strings.Wide_Wide_Unbounded.Aux;
use Ada.Strings.Wide_Wide_Unbounded.Aux;
pragma Warnings (On, "internal");
with Ada.Text_IO;                     use Ada.Text_IO;
with Ada.Unchecked_Conversion;
with Ada.Unchecked_Deallocation;
with System;
with GNAT.Task_Lock;
with GNAT.Traceback.Symbolic;
with GNATCOLL.Traces;
with Langkit_Support.Adalog.Debug;
with Langkit_Support.Hashes; use Langkit_Support.Hashes;
with Langkit_Support.Images; use Langkit_Support.Images;
with Langkit_Support.Relative_Get;
with Libadalang.Private_Converters;
use Libadalang.Private_Converters;
with Libadalang.Introspection_Implementation;
pragma Warnings (Off, "referenced");
          with Ada.Containers.Hashed_Sets;
          with Libadalang.Env_Hooks;
            use Libadalang.Env_Hooks;
          with Libadalang.Implementation.Extensions;
            use Libadalang.Implementation.Extensions;
          with Libadalang.Internal_Default_Provider;
          with Libadalang.Sources;
pragma Warnings (On, "referenced");
package body Libadalang.Implementation is
   use Precomputed_Symbols;
   pragma Warnings (Off, "has no effect");
   use Solver;
   pragma Warnings (On, "has no effect");
   package Context_Vectors is new Ada.Containers.Vectors
     (Index_Type   => Positive,
      Element_Type => Internal_Context);
   type Contexts_Destructor is limited
      new Ada.Finalization.Limited_Controlled with null record;
   overriding procedure Finalize (CD : in out Contexts_Destructor);
   --  Helper to destroy all contexts when terminating the process
   package Context_Pool is
      procedure Acquire (Context : out Internal_Context)
         with Post => Context /= null;
      --  If a context is free for reuse, increment its serial number and
      --  return it. Otherwise, allocate a new one. In any case, this does not
      --  initialize it, except for the Serial_Number field.
      procedure Release (Context : in out Internal_Context)
         with Pre  => Context /= null,
              Post => Context = null;
      --  Tag Context as free for reuse and set it to null
      procedure Free;
      --  Free all contexts in this pool. Intended to be called only when the
      --  process is terminating, to avoid reported memory leaks.
   private
      Available : Context_Vectors.Vector;
      --  List of allocated contexts that can be re-used right now
      CD : Contexts_Destructor with Unreferenced;
      --  Singleton whose only purpose is to free all contexts in Available
      --  when finalized.
   end Context_Pool;
   procedure Register_Destroyable_Helper
     (Unit    : Internal_Unit;
      Object  : System.Address;
      Destroy : Destroy_Procedure);
   --  Common underlying implementation for Register_Destroyable_Gen
   pragma Warnings (Off, "referenced");
   function Construct_Entity_Array
     (V : AST_Envs.Entity_Vectors.Vector) return Internal_Entity_Array_Access;
   pragma Warnings (On, "referenced");
   procedure Destroy (Env : in out Lexical_Env_Access);
   function Snaps_At_Start (Self : Bare_Ada_Node) return Boolean;
   function Snaps_At_End (Self : Bare_Ada_Node) return Boolean;
   --  Those maps are used to give unique ids to lexical envs while pretty
   --  printing them.
   package Address_To_Id_Maps is new Ada.Containers.Hashed_Maps
     (Lexical_Env, Integer, Hash, "=");
   type Dump_Lexical_Env_State is record
      Env_Ids : Address_To_Id_Maps.Map;
      --  Mapping: Lexical_Env -> Integer, used to remember which unique Ids we
      --  assigned to the lexical environments we found.
      Next_Id : Positive := 1;
      --  Id to assign to the next unknown lexical environment
      Root_Env : Lexical_Env;
      --  Lexical environment we consider a root (this is the Root_Scope from
      --  the current analysis context), or null if unknown.
   end record;
   --  Holder for the state of lexical environment dumpers
   function Get_Env_Id
     (E : Lexical_Env; State : in out Dump_Lexical_Env_State) return String;
   --  If E is known, return its unique Id from State. Otherwise, assign it a
   --  new unique Id and return it.
   ------------------------
   -- Precomputed_Symbol --
   ------------------------
   pragma Warnings (Off, "referenced");
   function Precomputed_Symbol
     (Index : Precomputed_Symbol_Index) return Text_Type is
   pragma Warnings (On, "referenced");
   begin
         declare
            Raw_Text : constant Text_Type := (case Index is
               when Precomputed_Sym => """&""",
               when Precomputed_Sym_1 => """&""""+""",
               when Precomputed_Sym_10 => """=""",
               when Precomputed_Sym_11 => """>""",
               when Precomputed_Sym_12 => """>=""",
               when Precomputed_Sym_2 => """*""",
               when Precomputed_Sym_20 => "<<>>",
               when Precomputed_Sym_3 => """**""",
               when Precomputed_Sym_4 => """+""",
               when Precomputed_Sym_5 => """-""",
               when Precomputed_Sym_6 => """/""",
               when Precomputed_Sym_7 => """/=""",
               when Precomputed_Sym_8 => """<""",
               when Precomputed_Sym_9 => """<=""",
               when Precomputed_Sym_Abort => "abort",
               when Precomputed_Sym_Abort_Signal => "Abort_Signal",
               when Precomputed_Sym_Abort_Signal_259 => "abort_signal_",
               when Precomputed_Sym_Abs => """abs""",
               when Precomputed_Sym_Abs_260 => "abs",
               when Precomputed_Sym_Abstract => "abstract",
               when Precomputed_Sym_Abstract_State => "Abstract_State",
               when Precomputed_Sym_Accept => "accept",
               when Precomputed_Sym_Access => "Access",
               when Precomputed_Sym_Access_263 => "access",
               when Precomputed_Sym_Ada => "Ada",
               when Precomputed_Sym_Ada_264 => "ada",
               when Precomputed_Sym_Address => "Address",
               when Precomputed_Sym_Address_Size => "Address_Size",
               when Precomputed_Sym_Adjacent => "Adjacent",
               when Precomputed_Sym_Adjacent_265 => "adjacent",
               when Precomputed_Sym_Aft => "Aft",
               when Precomputed_Sym_Aliased => "aliased",
               when Precomputed_Sym_Alignment => "Alignment",
               when Precomputed_Sym_All => "all",
               when Precomputed_Sym_And => """and""",
               when Precomputed_Sym_And_268 => "and",
               when Precomputed_Sym_Array => "array",
               when Precomputed_Sym_Asm_Input => "Asm_Input",
               when Precomputed_Sym_Asm_Input_270 => "asm_input",
               when Precomputed_Sym_Asm_Input_Operand => "Asm_Input_Operand",
               when Precomputed_Sym_Asm_Output => "Asm_Output",
               when Precomputed_Sym_Asm_Output_271 => "asm_output",
               when Precomputed_Sym_Asm_Output_Operand => "Asm_Output_Operand",
               when Precomputed_Sym_Assert => "Assert",
               when Precomputed_Sym_Assert_And_Cut => "Assert_And_Cut",
               when Precomputed_Sym_Assume => "Assume",
               when Precomputed_Sym_Asynchronous => "Asynchronous",
               when Precomputed_Sym_At => "at",
               when Precomputed_Sym_Atomic => "Atomic",
               when Precomputed_Sym_Atomic_Components => "Atomic_Components",
               when Precomputed_Sym_Attach_Handler => "Attach_Handler",
               when Precomputed_Sym_Aux_Dec => "Aux_DEC",
               when Precomputed_Sym_Base => "Base",
               when Precomputed_Sym_Begin => "begin",
               when Precomputed_Sym_Bit_Order => "Bit_Order",
               when Precomputed_Sym_Body => "body",
               when Precomputed_Sym_Boolean => "Boolean",
               when Precomputed_Sym_Buffer => "Buffer",
               when Precomputed_Sym_Callable => "Callable",
               when Precomputed_Sym_Caller => "Caller",
               when Precomputed_Sym_Case => "case",
               when Precomputed_Sym_Ceiling => "Ceiling",
               when Precomputed_Sym_Ceiling_276 => "ceiling",
               when Precomputed_Sym_Character => "Character",
               when Precomputed_Sym_Class => "Class",
               when Precomputed_Sym_Compile_Time_Error => "Compile_Time_Error",
               when Precomputed_Sym_Compile_Time_Warning => "Compile_Time_Warning",
               when Precomputed_Sym_Component_Size => "Component_Size",
               when Precomputed_Sym_Compose => "Compose",
               when Precomputed_Sym_Compose_277 => "compose",
               when Precomputed_Sym_Constant => "constant",
               when Precomputed_Sym_Constant_Indexing => "Constant_Indexing",
               when Precomputed_Sym_Constrained => "Constrained",
               when Precomputed_Sym_Contract_Cases => "Contract_Cases",
               when Precomputed_Sym_Convention => "Convention",
               when Precomputed_Sym_Copy_Sign => "Copy_Sign",
               when Precomputed_Sym_Copy_Sign_279 => "copy_sign",
               when Precomputed_Sym_Count => "Count",
               when Precomputed_Sym_Cursor => "Cursor",
               when Precomputed_Sym_Debug => "Debug",
               when Precomputed_Sym_Declare => "declare",
               when Precomputed_Sym_Default_Bit_Order => "Default_Bit_Order",
               when Precomputed_Sym_Default_Initial_Condition => "Default_Initial_Condition",
               when Precomputed_Sym_Definite => "Definite",
               when Precomputed_Sym_Delay => "delay",
               when Precomputed_Sym_Delta => "Delta",
               when Precomputed_Sym_Delta_282 => "delta",
               when Precomputed_Sym_Depends => "Depends",
               when Precomputed_Sym_Descriptor_Size => "Descriptor_Size",
               when Precomputed_Sym_Digits => "Digits",
               when Precomputed_Sym_Digits_283 => "digits",
               when Precomputed_Sym_Discard_Names => "Discard_Names",
               when Precomputed_Sym_Do => "do",
               when Precomputed_Sym_Dummy => "__dummy",
               when Precomputed_Sym_Duration => "Duration",
               when Precomputed_Sym_Dynamic_Predicate => "Dynamic_Predicate",
               when Precomputed_Sym_Elaborate_Body => "Elaborate_Body",
               when Precomputed_Sym_Element => "Element",
               when Precomputed_Sym_Else => "else",
               when Precomputed_Sym_Elsif => "elsif",
               when Precomputed_Sym_End => "end",
               when Precomputed_Sym_Ensures => "Ensures",
               when Precomputed_Sym_Entry => "entry",
               when Precomputed_Sym_Enum_Rep => "Enum_Rep",
               when Precomputed_Sym_Enum_Rep_289 => "enum_rep",
               when Precomputed_Sym_Enum_Val => "Enum_Val",
               when Precomputed_Sym_Enum_Val_290 => "enum_val",
               when Precomputed_Sym_Epsilon => "Epsilon",
               when Precomputed_Sym_Exception => "exception",
               when Precomputed_Sym_Exception_Id => "Exception_Id",
               when Precomputed_Sym_Exception_Occurrence => "Exception_Occurrence",
               when Precomputed_Sym_Exceptions => "Exceptions",
               when Precomputed_Sym_Exit => "exit",
               when Precomputed_Sym_Exponent => "Exponent",
               when Precomputed_Sym_Exponent_293 => "exponent",
               when Precomputed_Sym_Export => "Export",
               when Precomputed_Sym_External_Tag => "External_Tag",
               when Precomputed_Sym_Finalization_Size => "Finalization_Size",
               when Precomputed_Sym_First => "First",
               when Precomputed_Sym_First_Bit => "First_Bit",
               when Precomputed_Sym_Fixed_Value => "Fixed_Value",
               when Precomputed_Sym_Fixed_Value_294 => "fixed_value",
               when Precomputed_Sym_Floor => "Floor",
               when Precomputed_Sym_Floor_295 => "floor",
               when Precomputed_Sym_For => "for",
               when Precomputed_Sym_Fore => "Fore",
               when Precomputed_Sym_Fraction => "Fraction",
               when Precomputed_Sym_Fraction_297 => "fraction",
               when Precomputed_Sym_Function => "function",
               when Precomputed_Sym_Generic => "generic",
               when Precomputed_Sym_Ghost => "Ghost",
               when Precomputed_Sym_Global => "Global",
               when Precomputed_Sym_Goto => "goto",
               when Precomputed_Sym_Has_Access_Values => "Has_Access_Values",
               when Precomputed_Sym_Has_Discriminants => "Has_Discriminants",
               when Precomputed_Sym_Has_Tagged_Values => "Has_Tagged_Values",
               when Precomputed_Sym_Identity => "Identity",
               when Precomputed_Sym_If => "if",
               when Precomputed_Sym_Image => "Image",
               when Precomputed_Sym_Image_302 => "image",
               when Precomputed_Sym_Img => "Img",
               when Precomputed_Sym_Implicit_Dereference => "Implicit_Dereference",
               when Precomputed_Sym_Import => "Import",
               when Precomputed_Sym_Import_Function => "Import_Function",
               when Precomputed_Sym_Import_Procedure => "Import_Procedure",
               when Precomputed_Sym_In => "in",
               when Precomputed_Sym_Independent => "Independent",
               when Precomputed_Sym_Independent_Components => "Independent_Components",
               when Precomputed_Sym_Index => "Index",
               when Precomputed_Sym_Initialized => "Initialized",
               when Precomputed_Sym_Inline => "Inline",
               when Precomputed_Sym_Input => "Input",
               when Precomputed_Sym_Input_304 => "input",
               when Precomputed_Sym_Integer => "Integer",
               when Precomputed_Sym_Integer_Literal => "Integer_Literal",
               when Precomputed_Sym_Integer_Value => "Integer_Value",
               when Precomputed_Sym_Integer_Value_305 => "integer_value",
               when Precomputed_Sym_Interface => "Interface",
               when Precomputed_Sym_Interface_306 => "interface",
               when Precomputed_Sym_Interrupt_Handler => "Interrupt_Handler",
               when Precomputed_Sym_Invalid_Value => "Invalid_Value",
               when Precomputed_Sym_Invariant => "Invariant",
               when Precomputed_Sym_Invariant_Class => "Invariant'Class",
               when Precomputed_Sym_Is => "is",
               when Precomputed_Sym_Iterable => "Iterable",
               when Precomputed_Sym_Iterator_Element => "Iterator_Element",
               when Precomputed_Sym_Iterator_Interfaces => "Iterator_Interfaces",
               when Precomputed_Sym_Large => "Large",
               when Precomputed_Sym_Last => "Last",
               when Precomputed_Sym_Last_Bit => "Last_Bit",
               when Precomputed_Sym_Left => "left",
               when Precomputed_Sym_Length => "Length",
               when Precomputed_Sym_Limited => "limited",
               when Precomputed_Sym_Loop => "loop",
               when Precomputed_Sym_Loop_Entry => "Loop_Entry",
               when Precomputed_Sym_Loop_Invariant => "Loop_Invariant",
               when Precomputed_Sym_Machine => "Machine",
               when Precomputed_Sym_Machine_311 => "machine",
               when Precomputed_Sym_Machine_Code => "Machine_Code",
               when Precomputed_Sym_Machine_Emax => "Machine_Emax",
               when Precomputed_Sym_Machine_Emin => "Machine_Emin",
               when Precomputed_Sym_Machine_Mantissa => "Machine_Mantissa",
               when Precomputed_Sym_Machine_Overflows => "Machine_Overflows",
               when Precomputed_Sym_Machine_Radix => "Machine_Radix",
               when Precomputed_Sym_Machine_Rounding => "Machine_Rounding",
               when Precomputed_Sym_Machine_Rounding_312 => "machine_rounding",
               when Precomputed_Sym_Machine_Rounds => "Machine_Rounds",
               when Precomputed_Sym_Mantissa => "Mantissa",
               when Precomputed_Sym_Max => "Max",
               when Precomputed_Sym_Max_313 => "max",
               when Precomputed_Sym_Max_Integer_Size => "Max_Integer_Size",
               when Precomputed_Sym_Max_Size_In_Storage_Elements => "Max_Size_In_Storage_Elements",
               when Precomputed_Sym_Maximum_Alignment => "Maximum_Alignment",
               when Precomputed_Sym_Min => "Min",
               when Precomputed_Sym_Min_314 => "min",
               when Precomputed_Sym_Mod => """mod""",
               when Precomputed_Sym_Mod_142 => "Mod",
               when Precomputed_Sym_Mod_315 => "mod",
               when Precomputed_Sym_Model_Epsilon => "Model_Epsilon",
               when Precomputed_Sym_Model_Mantissa => "Model_Mantissa",
               when Precomputed_Sym_Model_Small => "Model_Small",
               when Precomputed_Sym_Modulus => "Modulus",
               when Precomputed_Sym_New => "new",
               when Precomputed_Sym_Nextpart => "__nextpart",
               when Precomputed_Sym_No_Elaboration_Code => "No_Elaboration_Code",
               when Precomputed_Sym_No_Return => "No_Return",
               when Precomputed_Sym_Not => """not""",
               when Precomputed_Sym_Not_317 => "not",
               when Precomputed_Sym_Null => "null",
               when Precomputed_Sym_Object_Size => "Object_Size",
               when Precomputed_Sym_Of => "of",
               when Precomputed_Sym_Old => "Old",
               when Precomputed_Sym_Or => """or""",
               when Precomputed_Sym_Or_320 => "or",
               when Precomputed_Sym_Others => "others",
               when Precomputed_Sym_Out => "out",
               when Precomputed_Sym_Output => "Output",
               when Precomputed_Sym_Output_323 => "output",
               when Precomputed_Sym_Overriding => "overriding",
               when Precomputed_Sym_Pack => "Pack",
               when Precomputed_Sym_Package => "package",
               when Precomputed_Sym_Pos => "Pos",
               when Precomputed_Sym_Pos_326 => "pos",
               when Precomputed_Sym_Position => "Position",
               when Precomputed_Sym_Post => "Post",
               when Precomputed_Sym_Post_Class => "Post'Class",
               when Precomputed_Sym_Postcondition => "Postcondition",
               when Precomputed_Sym_Postcondition_Class => "Postcondition'Class",
               when Precomputed_Sym_Pragma => "pragma",
               when Precomputed_Sym_Pre => "Pre",
               when Precomputed_Sym_Pre_Class => "Pre'Class",
               when Precomputed_Sym_Precondition => "Precondition",
               when Precomputed_Sym_Precondition_Class => "Precondition'Class",
               when Precomputed_Sym_Pred => "Pred",
               when Precomputed_Sym_Pred_328 => "pred",
               when Precomputed_Sym_Predicate => "Predicate",
               when Precomputed_Sym_Predicate_Failure => "Predicate_Failure",
               when Precomputed_Sym_Preelaborate => "Preelaborate",
               when Precomputed_Sym_Private => "private",
               when Precomputed_Sym_Privatepart => "__privatepart",
               when Precomputed_Sym_Procedure => "procedure",
               when Precomputed_Sym_Protected => "protected",
               when Precomputed_Sym_Pure => "Pure",
               when Precomputed_Sym_Put_Image => "Put_Image",
               when Precomputed_Sym_Put_Image_332 => "put_image",
               when Precomputed_Sym_Raise => "raise",
               when Precomputed_Sym_Range => "Range",
               when Precomputed_Sym_Range_334 => "range",
               when Precomputed_Sym_Range_Length => "Range_Length",
               when Precomputed_Sym_Read => "Read",
               when Precomputed_Sym_Read_335 => "read",
               when Precomputed_Sym_Real_Literal => "Real_Literal",
               when Precomputed_Sym_Record => "record",
               when Precomputed_Sym_Reduce => "Reduce",
               when Precomputed_Sym_Refined_Depends => "Refined_Depends",
               when Precomputed_Sym_Refined_Global => "Refined_Global",
               when Precomputed_Sym_Refined_Post => "Refined_Post",
               when Precomputed_Sym_Refined_State => "Refined_State",
               when Precomputed_Sym_Rem => """rem""",
               when Precomputed_Sym_Rem_337 => "rem",
               when Precomputed_Sym_Remainder => "Remainder",
               when Precomputed_Sym_Remainder_338 => "remainder",
               when Precomputed_Sym_Remote_Call_Interface => "Remote_Call_Interface",
               when Precomputed_Sym_Remote_Types => "Remote_Types",
               when Precomputed_Sym_Renames => "renames",
               when Precomputed_Sym_Requeue => "requeue",
               when Precomputed_Sym_Requires => "Requires",
               when Precomputed_Sym_Restrictions => "Restrictions",
               when Precomputed_Sym_Result => "Result",
               when Precomputed_Sym_Return => "return",
               when Precomputed_Sym_Reverse => "reverse",
               when Precomputed_Sym_Right => "right",
               when Precomputed_Sym_Root_Buffer_Type => "Root_Buffer_Type",
               when Precomputed_Sym_Root_Integer => "root_integer",
               when Precomputed_Sym_Root_Real => "root_real",
               when Precomputed_Sym_Root_Storage_Pool => "Root_Storage_Pool",
               when Precomputed_Sym_Root_Stream_Type => "Root_Stream_Type",
               when Precomputed_Sym_Root_Types => "root_types_",
               when Precomputed_Sym_Rounding => "Rounding",
               when Precomputed_Sym_Rounding_347 => "rounding",
               when Precomputed_Sym_S => "S",
               when Precomputed_Sym_Safe_Large => "Safe_Large",
               when Precomputed_Sym_Safe_Small => "Safe_Small",
               when Precomputed_Sym_Scalar_Storage_Order => "Scalar_Storage_Order",
               when Precomputed_Sym_Scaling => "Scaling",
               when Precomputed_Sym_Scaling_348 => "scaling",
               when Precomputed_Sym_Select => "select",
               when Precomputed_Sym_Separate => "separate",
               when Precomputed_Sym_Shared_Passive => "Shared_Passive",
               when Precomputed_Sym_Size => "Size",
               when Precomputed_Sym_Small => "Small",
               when Precomputed_Sym_Small_Denominator => "Small_Denominator",
               when Precomputed_Sym_Small_Numerator => "Small_Numerator",
               when Precomputed_Sym_Some => "some",
               when Precomputed_Sym_Stable_Properties => "Stable_Properties",
               when Precomputed_Sym_Standard => "Standard",
               when Precomputed_Sym_Standard_352 => "standard",
               when Precomputed_Sym_Static_Predicate => "Static_Predicate",
               when Precomputed_Sym_Storage_Elements => "Storage_Elements",
               when Precomputed_Sym_Storage_Pool => "Storage_Pool",
               when Precomputed_Sym_Storage_Pools => "Storage_Pools",
               when Precomputed_Sym_Storage_Size => "Storage_Size",
               when Precomputed_Sym_Storage_Unit => "Storage_Unit",
               when Precomputed_Sym_Streams => "Streams",
               when Precomputed_Sym_String => "String",
               when Precomputed_Sym_String_Literal => "String_Literal",
               when Precomputed_Sym_Strings => "Strings",
               when Precomputed_Sym_Style_Checks => "Style_Checks",
               when Precomputed_Sym_Subtype => "subtype",
               when Precomputed_Sym_Succ => "Succ",
               when Precomputed_Sym_Succ_354 => "succ",
               when Precomputed_Sym_Synchronized => "synchronized",
               when Precomputed_Sym_System => "System",
               when Precomputed_Sym_System_Allocator_Alignment => "System_Allocator_Alignment",
               when Precomputed_Sym_Tag => "Tag",
               when Precomputed_Sym_Tagged => "tagged",
               when Precomputed_Sym_Tags => "Tags",
               when Precomputed_Sym_Target_Name => "Target_Name",
               when Precomputed_Sym_Task => "task",
               when Precomputed_Sym_Task_Id => "Task_Id",
               when Precomputed_Sym_Task_Identification => "Task_Identification",
               when Precomputed_Sym_Terminate => "terminate",
               when Precomputed_Sym_Terminated => "Terminated",
               when Precomputed_Sym_Test_Case => "Test_Case",
               when Precomputed_Sym_Text_Buffers => "Text_Buffers",
               when Precomputed_Sym_Text_Io => "text_io",
               when Precomputed_Sym_Then => "then",
               when Precomputed_Sym_To_Address => "To_Address",
               when Precomputed_Sym_Truncation => "Truncation",
               when Precomputed_Sym_Truncation_361 => "truncation",
               when Precomputed_Sym_Type => "type",
               when Precomputed_Sym_Type_Class => "Type_Class",
               when Precomputed_Sym_Type_Invariant => "Type_Invariant",
               when Precomputed_Sym_Type_Invariant_Class => "Type_Invariant'Class",
               when Precomputed_Sym_Unchecked_Access => "Unchecked_Access",
               when Precomputed_Sym_Unchecked_Union => "Unchecked_Union",
               when Precomputed_Sym_Universal_Int_Type => "Universal_Int_Type_",
               when Precomputed_Sym_Universal_Real_Type => "Universal_Real_Type_",
               when Precomputed_Sym_Unreferenced => "Unreferenced",
               when Precomputed_Sym_Unrestricted_Access => "Unrestricted_Access",
               when Precomputed_Sym_Until => "until",
               when Precomputed_Sym_Update => "Update",
               when Precomputed_Sym_Use => "use",
               when Precomputed_Sym_Vads_Size => "VADS_Size",
               when Precomputed_Sym_Val => "Val",
               when Precomputed_Sym_Val_365 => "val",
               when Precomputed_Sym_Valid => "Valid",
               when Precomputed_Sym_Valid_Scalars => "Valid_Scalars",
               when Precomputed_Sym_Value => "Value",
               when Precomputed_Sym_Value_366 => "value",
               when Precomputed_Sym_Value_Size => "Value_Size",
               when Precomputed_Sym_Variable_Indexing => "Variable_Indexing",
               when Precomputed_Sym_Volatile => "Volatile",
               when Precomputed_Sym_Volatile_Components => "Volatile_Components",
               when Precomputed_Sym_Warnings => "Warnings",
               when Precomputed_Sym_When => "when",
               when Precomputed_Sym_While => "while",
               when Precomputed_Sym_Wide_Character => "Wide_Character",
               when Precomputed_Sym_Wide_Image => "Wide_Image",
               when Precomputed_Sym_Wide_Image_369 => "wide_image",
               when Precomputed_Sym_Wide_String => "Wide_String",
               when Precomputed_Sym_Wide_Text_Io => "wide_text_io",
               when Precomputed_Sym_Wide_Value => "Wide_Value",
               when Precomputed_Sym_Wide_Value_371 => "wide_value",
               when Precomputed_Sym_Wide_Wide_Character => "Wide_Wide_Character",
               when Precomputed_Sym_Wide_Wide_Image => "Wide_Wide_Image",
               when Precomputed_Sym_Wide_Wide_Image_372 => "wide_wide_image",
               when Precomputed_Sym_Wide_Wide_String => "Wide_Wide_String",
               when Precomputed_Sym_Wide_Wide_Text_Io => "wide_wide_text_io",
               when Precomputed_Sym_Wide_Wide_Value => "Wide_Wide_Value",
               when Precomputed_Sym_Wide_Wide_Value_374 => "wide_wide_value",
               when Precomputed_Sym_Width => "Width",
               when Precomputed_Sym_With => "with",
               when Precomputed_Sym_Word_Size => "Word_Size",
               when Precomputed_Sym_Write => "Write",
               when Precomputed_Sym_Write_376 => "write",
               when Precomputed_Sym_Xor => """xor""",
               when Precomputed_Sym_Xor_377 => "xor"
            );
            Symbol : constant Symbolization_Result :=
                  Libadalang.Sources.Canonicalize (Raw_Text)
            ;
         begin
            if Symbol.Success then
               return Symbol.Symbol;
            else
               raise Program_Error with
                 "Cannot canonicalize symbol literal: " & Image (Raw_Text);
            end if;
         end;
   end Precomputed_Symbol;
   ----------------------------
   -- Construct_Entity_Array --
   ----------------------------
   function Construct_Entity_Array
     (V : AST_Envs.Entity_Vectors.Vector) return Internal_Entity_Array_Access
   is
      Ret : Internal_Entity_Array_Access :=
        Create_Internal_Entity_Array (V.Length);
   begin
      for J in V.First_Index .. V.Last_Index loop
         Ret.Items (J) := V.Get (J);
      end loop;
      declare
         Tmp : AST_Envs.Entity_Vectors.Vector := V;
      begin
         Tmp.Destroy;
      end;
      return Ret;
   end Construct_Entity_Array;
   ----------------
   -- Enter_Call --
   ----------------
   procedure Enter_Call
     (Context : Internal_Context; Call_Depth : access Natural)
   is
      Max             : Natural renames Context.Max_Call_Depth;
      Current         : Natural renames Context.Current_Call_Depth;
      High_Water_Mark : Natural renames Context.Call_Depth_High_Water_Mark;
   begin
      Current := Current + 1;
      High_Water_Mark := Natural'Max (High_Water_Mark, Current);
      Call_Depth.all := Current;
      if Current > Max then
         raise Property_Error with "stack overflow";
      end if;
   end Enter_Call;
   ---------------
   -- Exit_Call --
   ---------------
   procedure Exit_Call (Context : Internal_Context; Call_Depth : Natural) is
      Current : Natural renames Context.Current_Call_Depth;
   begin
      if Call_Depth /= Current then
         raise Unexpected_Call_Depth with
            "Langkit code generation bug for call depth handling detected";
      end if;
      Current := Current - 1;
   end Exit_Call;
   -----------
   -- Image --
   -----------
   function Image (Self : Symbol_Type) return String_Type is
   begin
      return Create_String (Image (Self));
   end Image;
   ------------------
   -- Context_Pool --
   ------------------
   package body Context_Pool is
      -------------
      -- Acquire --
      -------------
      procedure Acquire (Context : out Internal_Context) is
      begin
         GNAT.Task_Lock.Lock;
         if Available.Is_Empty then
            Context := new Analysis_Context_Type;
            Context.Serial_Number := 1;
         else
            Context := Available.Last_Element;
            Available.Delete_Last;
         end if;
         GNAT.Task_Lock.Unlock;
      exception
         when others =>
            GNAT.Task_Lock.Unlock;
            raise;
      end Acquire;
      -------------
      -- Release --
      -------------
      procedure Release (Context : in out Internal_Context) is
      begin
         GNAT.Task_Lock.Lock;
         Available.Append (Context);
         Context.Serial_Number := Context.Serial_Number + 1;
         Context := null;
         GNAT.Task_Lock.Unlock;
      exception
         when others =>
            GNAT.Task_Lock.Unlock;
            raise;
      end Release;
      ----------
      -- Free --
      ----------
      procedure Free is
      begin
         GNAT.Task_Lock.Lock;
         for C of Available loop
            Free (C);
         end loop;
         GNAT.Task_Lock.Unlock;
      exception
         when others =>
            GNAT.Task_Lock.Unlock;
            raise;
      end Free;
   end Context_Pool;
   --------------
   -- Finalize --
   --------------
   overriding procedure Finalize (CD : in out Contexts_Destructor) is
      pragma Unreferenced (CD);
   begin
      Context_Pool.Free;
   end Finalize;
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (File_Reader : in out Internal_File_Reader_Access) is
      procedure Destroy is new Ada.Unchecked_Deallocation
        (Internal_File_Reader'Class, Internal_File_Reader_Access);
   begin
      if File_Reader /= null and then File_Reader.all.Dec_Ref then
         Destroy (File_Reader);
      end if;
   end Dec_Ref;
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (Self : in out Internal_Event_Handler_Access) is
      procedure Destroy is new Ada.Unchecked_Deallocation
        (Internal_Event_Handler'Class, Internal_Event_Handler_Access);
   begin
      if Self /= null and then Self.all.Dec_Ref then
         Destroy (Self);
      end if;
   end Dec_Ref;
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (Provider : in out Internal_Unit_Provider_Access) is
      procedure Destroy is new Ada.Unchecked_Deallocation
        (Internal_Unit_Provider'Class, Internal_Unit_Provider_Access);
   begin
      if Provider /= null and then Provider.all.Dec_Ref then
         Destroy (Provider);
      end if;
   end Dec_Ref;
   ----------------
   -- Get_Env_Id --
   ----------------
   function Get_Env_Id
     (E : Lexical_Env; State : in out Dump_Lexical_Env_State) return String
   is
      C        : Address_To_Id_Maps.Cursor;
      Inserted : Boolean;
   begin
      if E = Null_Lexical_Env then
         return "$null";
      elsif E = State.Root_Env then
         --  Insert root env with a special Id so that we only print it once
         State.Env_Ids.Insert (E, -1, C, Inserted);
         return "$root";
      end if;
      State.Env_Ids.Insert (E, State.Next_Id, C, Inserted);
      if Inserted then
         State.Next_Id := State.Next_Id + 1;
      end if;
      return '@' & Stripped_Image (Address_To_Id_Maps.Element (C));
   end Get_Env_Id;
   pragma Warnings (Off, "referenced");
   function To_Lookup_Kind_Type (K : Lookup_Kind) return Lookup_Kind_Type
   is
     (Lookup_Kind_Type'Val (Lookup_Kind'Pos (K)));
   pragma Warnings (On, "referenced");
   --------------------
   -- Create_Context --
   --------------------
   function Create_Context
     (Charset        : String;
      File_Reader    : Internal_File_Reader_Access;
      Unit_Provider  : Internal_Unit_Provider_Access;
      Event_Handler  : Internal_Event_Handler_Access;
      With_Trivia    : Boolean;
      Tab_Stop       : Positive;
      Max_Call_Depth : Natural := 1000)
      return Internal_Context
   is
      Actual_Charset : constant String :=
        (if Charset = "" then Default_Charset else Charset);
      Symbols        : constant Precomputed_Symbol_Table
        := Create_Symbol_Table;
      Context        : Internal_Context;
   begin
      Context_Pool.Acquire (Context);
      Context.Ref_Count := 1;
      Context.Symbols := Symbol_Table (Symbols);
      Context.Charset := To_Unbounded_String (Actual_Charset);
      Context.Tab_Stop := Tab_Stop;
      Context.With_Trivia := With_Trivia;
      Context.Root_Scope := Create_Static_Lexical_Env
        (Parent => Null_Lexical_Env,
         Node   => null);
      --  Create a new ownership share for Event_Handler so that it lives at
      --  least as long as this analysis context.
      Context.Event_Handler := Event_Handler;
      if Context.Event_Handler /= null then
         Context.Event_Handler.Inc_Ref;
      end if;
      --  Create a new ownership share for File_Reader so that it lives at
      --  least as long as this analysis context.
      Context.File_Reader := File_Reader;
      if Context.File_Reader /= null then
         Context.File_Reader.Inc_Ref;
      end if;
      --  Create a new ownership share for Unit_Provider so that it lives at
      --  least as long as this analysis context.
      Context.Unit_Provider := Unit_Provider;
      if Context.Unit_Provider /= null then
         Context.Unit_Provider.Inc_Ref;
      end if;
         if Context.Unit_Provider = null then
            Context.Unit_Provider := Libadalang.Internal_Default_Provider.Create;
         end if;
      Initialize (Context.Parser);
      Context.Discard_Errors_In_Populate_Lexical_Env := True;
      Context.Logic_Resolution_Timeout :=
        Langkit_Support.Adalog.Default_Timeout_Ticks_Number;
      Context.In_Populate_Lexical_Env := False;
      Context.Cache_Version := 0;
      Context.Reparse_Cache_Version := 0;
      Context.Rewriting_Handle := No_Rewriting_Handle_Pointer;
      Context.Templates_Unit := No_Analysis_Unit;
      Context.Max_Call_Depth := Max_Call_Depth;
      Context.Available_Rebindings := Env_Rebindings_Vectors.Empty_Vector;
Fetch_Standard (Context);
      return Context;
   end Create_Context;
   -----------------
   -- Create_Unit --
   -----------------
   function Create_Unit
     (Context             : Internal_Context;
      Normalized_Filename : Virtual_File;
      Charset             : String;
      Rule                : Grammar_Rule) return Internal_Unit
   is
      use Units_Maps;
      Unit : Internal_Unit;
   begin
      Unit := Create_Special_Unit
        (Context, Normalized_Filename, Charset, Rule);
      Context.Units.Insert (Normalized_Filename, Unit);
      return Unit;
   end Create_Unit;
   --------------
   -- Get_Unit --
   --------------
   function Get_Unit
     (Context           : Internal_Context;
      Filename, Charset : String;
      Reparse           : Boolean;
      Input             : Internal_Lexer_Input;
      Rule              : Grammar_Rule;
      Is_Internal       : Boolean := False) return Internal_Unit
   is
      use Units_Maps;
      Normalized_Filename : constant GNATCOLL.VFS.Virtual_File :=
         Normalized_Unit_Filename (Context, Filename);
      Cur     : constant Cursor :=
         Context.Units.Find (Normalized_Filename);
      Created : constant Boolean := Cur = No_Element;
      Unit    : Internal_Unit;
      Actual_Charset : Unbounded_String;
      Refined_Input  : Internal_Lexer_Input := Input;
   begin
      --  Determine which encoding to use. Use the Charset parameter (if
      --  provided), otherwise use the context-wide default.
      Actual_Charset := (if Charset'Length /= 0
                         then To_Unbounded_String (Charset)
                         else Context.Charset);
      if Refined_Input.Kind = File then
         Refined_Input.Filename := Normalized_Filename;
      end if;
      if Refined_Input.Kind in File | Bytes_Buffer then
         Refined_Input.Charset := Actual_Charset;
         --  Unless the caller requested a specific charset for this unit,
         --  allow the lexer to automatically discover the source file encoding
         --  before defaulting to the context-specific one. We do this trying
         --  to match a byte order mark.
         Refined_Input.Read_BOM := Charset'Length = 0;
      end if;
      --  Create the Internal_Unit if needed
      Unit :=
        (if Created
         then Create_Unit (Context, Normalized_Filename,
                           To_String (Actual_Charset), Rule)
         else Element (Cur));
      --  If an internal unit is requested, set the corresponding flag.
      --  Otherwise, make sure that the unit we return isn't internal.
      if Is_Internal then
         Unit.Is_Internal := True;
      end if;
      --  (Re)parse it if needed
      if Created or else Reparse then
         --  It is illegal to reparse an internal unit for public API users.
         --  Since public APIs do not allow to pass True to Is_Internal, we can
         --  check here that only the implementation can ask to reparse an
         --  internal unit.
         if Unit.Is_Internal and then not Is_Internal then
            raise Precondition_Failure with "cannot reparse an internal unit";
         end if;
         declare
            Reparsed : Reparsed_Unit;
         begin
            Do_Parsing (Unit, Refined_Input, Reparsed);
            Update_After_Reparse (Unit, Reparsed);
         end;
         --  Now that we have removed reparsed the unit, update its current
         --  charset.
         Unit.Charset := Actual_Charset;
      end if;
      if Context.Event_Handler /= null then
         Context.Event_Handler.Unit_Parsed_Callback (Context, Unit, Reparse);
      end if;
      return Unit;
   end Get_Unit;
   --------------
   -- Has_Unit --
   --------------
   function Has_Unit
     (Context : Internal_Context; Unit_Filename : String) return Boolean is
   begin
      return Context.Units.Contains
        (Normalized_Unit_Filename (Context, Unit_Filename));
   end Has_Unit;
   -------------------
   -- Get_From_File --
   -------------------
   function Get_From_File
     (Context  : Internal_Context;
      Filename : String;
      Charset  : String;
      Reparse  : Boolean;
      Rule     : Grammar_Rule) return Internal_Unit
   is
      Input : constant Internal_Lexer_Input :=
        (Kind     => File,
         Charset  => <>,
         Read_BOM => False,
         Filename => <>);
   begin
      if Reparse and then Has_Rewriting_Handle (Context) then
         raise Precondition_Failure with
            "cannot reparse during tree rewriting";
      end if;
      return Get_Unit (Context, Filename, Charset, Reparse, Input, Rule);
   end Get_From_File;
   ---------------------
   -- Get_From_Buffer --
   ---------------------
   function Get_From_Buffer
     (Context  : Internal_Context;
      Filename : String;
      Charset  : String;
      Buffer   : String;
      Rule     : Grammar_Rule) return Internal_Unit
   is
      Input : constant Internal_Lexer_Input :=
        (Kind        => Bytes_Buffer,
         Charset     => <>,
         Read_BOM    => False,
         Bytes       => Buffer'Address,
         Bytes_Count => Buffer'Length);
   begin
      if Has_Rewriting_Handle (Context) then
         raise Precondition_Failure with
            "cannot parse from buffer during tree rewriting";
      elsif Context.File_Reader /= null then
         raise Precondition_Failure with
            "cannot parse from buffer with a file reader";
      end if;
      return Get_Unit (Context, Filename, Charset, True, Input, Rule);
   end Get_From_Buffer;
   --------------------
   -- Get_With_Error --
   --------------------
   function Get_With_Error
     (Context  : Internal_Context;
      Filename : String;
      Error    : Text_Type;
      Charset  : String;
      Rule     : Grammar_Rule) return Internal_Unit
   is
      use Units_Maps;
      Normalized_Filename : constant Virtual_File :=
         Normalized_Unit_Filename (Context, Filename);
      Cur                 : constant Cursor :=
         Context.Units.Find (Normalized_Filename);
   begin
      if Cur = No_Element then
         declare
            Unit : constant Internal_Unit := Create_Unit
              (Context, Normalized_Filename, Charset, Rule);
         begin
            Append (Unit.Diagnostics, No_Source_Location_Range, Error);
            return Unit;
         end;
      else
         return Element (Cur);
      end if;
   end Get_With_Error;
   -----------------------
   -- Get_From_Provider --
   -----------------------
   function Get_From_Provider
     (Context : Internal_Context;
      Name    : Text_Type;
      Kind    : Analysis_Unit_Kind;
      Charset : String;
      Reparse : Boolean) return Internal_Unit is
   begin
      if Reparse and then Has_Rewriting_Handle (Context) then
         raise Precondition_Failure with
            "cannot reparse during tree rewriting";
      end if;
      begin
         return Context.Unit_Provider.Get_Unit
           (Context, Name, Kind, Charset, Reparse);
      exception
         when Precondition_Failure | Property_Error =>
            raise Invalid_Unit_Name_Error with
               "Invalid unit name: " & Image (Name, With_Quotes => True)
               & " (" & Analysis_Unit_Kind'Image (Kind) & ")";
      end;
   end Get_From_Provider;
   -------------------
   -- Unit_Provider --
   -------------------
   function Unit_Provider
     (Context : Internal_Context) return Internal_Unit_Provider_Access
   is (Context.Unit_Provider);
   ----------
   -- Hash --
   ----------
   function Hash (Context : Internal_Context) return Hash_Type is
      function H is new Hash_Access (Analysis_Context_Type, Internal_Context);
   begin
      return H (Context);
   end Hash;
   ---------------------
   -- Has_With_Trivia --
   ---------------------
   function Has_With_Trivia (Context : Internal_Context) return Boolean is
   begin
      return Context.With_Trivia;
   end Has_With_Trivia;
   --------------------------------------------
   -- Discard_Errors_In_Populate_Lexical_Env --
   --------------------------------------------
   procedure Discard_Errors_In_Populate_Lexical_Env
     (Context : Internal_Context; Discard : Boolean) is
   begin
      Context.Discard_Errors_In_Populate_Lexical_Env := Discard;
   end Discard_Errors_In_Populate_Lexical_Env;
   ----------------------------------
   -- Set_Logic_Resolution_Timeout --
   ----------------------------------
   procedure Set_Logic_Resolution_Timeout
     (Context : Internal_Context; Timeout : Natural) is
   begin
      Context.Logic_Resolution_Timeout := Timeout;
   end Set_Logic_Resolution_Timeout;
   --------------------------
   -- Has_Rewriting_Handle --
   --------------------------
   function Has_Rewriting_Handle (Context : Internal_Context) return Boolean is
   begin
      return Context.Rewriting_Handle /= No_Rewriting_Handle_Pointer;
   end Has_Rewriting_Handle;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (Context : Internal_Context) is
   begin
      if Context /= null then
         Context.Ref_Count := Context.Ref_Count + 1;
      end if;
   end Inc_Ref;
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (Context : in out Internal_Context) is
   begin
      if Context /= null then
         Context.Ref_Count := Context.Ref_Count - 1;
         if Context.Ref_Count = 0 then
            Destroy (Context);
         end if;
      end if;
   end Dec_Ref;
   -------------
   -- Destroy --
   -------------
   procedure Destroy (Context : in out Internal_Context) is
   begin
      --  Destroy all named environment data structures
      for Desc of Context.Named_Envs loop
         for V of Desc.Foreign_Nodes loop
            V.Destroy;
         end loop;
         Destroy (Desc);
      end loop;
      Context.Named_Envs.Clear;
      --  If we are asked to free this context, it means that no one else have
      --  references to its analysis units, so it's safe to destroy these.
      for Unit of Context.Units loop
         Destroy (Unit);
      end loop;
      Context.Units := Units_Maps.Empty_Map;
      Context.Filenames := Virtual_File_Maps.Empty_Map;
      declare
         procedure Destroy is new Ada.Unchecked_Deallocation
           (Env_Rebindings_Type, Env_Rebindings);
         AR : Env_Rebindings_Vectors.Vector renames
            Context.Available_Rebindings;
         R  : Env_Rebindings;
      begin
         for I in AR.First_Index .. AR.Last_Index loop
            R := AR.Get (I);
            Destroy (R);
         end loop;
         AR.Destroy;
      end;
      Destroy (Context.Templates_Unit);
      AST_Envs.Destroy (Context.Root_Scope);
      Destroy (Context.Symbols);
      Destroy (Context.Parser);
      Dec_Ref (Context.File_Reader);
      Dec_Ref (Context.Unit_Provider);
      Dec_Ref (Context.Event_Handler);
      Context_Pool.Release (Context);
   end Destroy;
   -------------
   -- Context --
   -------------
   function Context (Unit : Internal_Unit) return Internal_Context is
   begin
      return Unit.Context;
   end Context;
   ----------
   -- Hash --
   ----------
   function Hash (Unit : Internal_Unit) return Hash_Type is
      function H is new Hash_Access (Analysis_Unit_Type, Internal_Unit);
   begin
      return H (Unit);
   end Hash;
   -------------
   -- Reparse --
   -------------
   procedure Reparse (Unit : Internal_Unit; Charset : String) is
      Dummy : constant Internal_Unit := Get_From_File
        (Unit.Context, +Unit.Filename.Full_Name, Charset,
         Reparse => True,
         Rule    => Unit.Rule);
   begin
      null;
   end Reparse;
   -------------
   -- Reparse --
   -------------
   procedure Reparse (Unit : Internal_Unit; Charset : String; Buffer : String)
   is
      Dummy : constant Internal_Unit := Get_From_Buffer
        (Unit.Context, +Unit.Filename.Full_Name, Charset, Buffer, Unit.Rule);
   begin
      null;
   end Reparse;
   --------------------------
   -- Populate_Lexical_Env --
   --------------------------
   procedure Populate_Lexical_Env (Unit : Internal_Unit) is
      Context : constant Internal_Context := Unit.Context;
      Has_Errors : Boolean := False;
      --  Whether at least one Property_Error occurred during this PLE pass
      Saved_In_Populate_Lexical_Env : constant Boolean :=
         Unit.Context.In_Populate_Lexical_Env;
      procedure Reset_Envs_Caches (Unit : Internal_Unit) is
         procedure Internal (Node : Bare_Ada_Node) is
         begin
            if Node = null then
               return;
            end if;
            Reset_Caches (Node.Self_Env);
            for I in 1 .. Children_Count (Node) loop
               Internal (Child (Node, I));
            end loop;
         end Internal;
      begin
         Internal (Unit.Ast_Root);
      end Reset_Envs_Caches;
   begin
      --  TODO??? Handle env invalidation when reparsing a unit and when a
      --  previous call raised a Property_Error.
      if Unit.Is_Env_Populated then
         return;
      end if;
      Unit.Is_Env_Populated := True;
      if Unit.Ast_Root = null then
         return;
      end if;
      GNATCOLL.Traces.Trace (Main_Trace, "Populating lexical envs for unit: "
                                         & Basename (Unit));
      if GNATCOLL.Traces.Active (Main_Trace) then
         GNATCOLL.Traces.Increase_Indent (Main_Trace);
      end if;
      Context.In_Populate_Lexical_Env := True;
      Has_Errors := Populate_Lexical_Env (Unit.Ast_Root);
      Context.In_Populate_Lexical_Env := Saved_In_Populate_Lexical_Env;
      if GNATCOLL.Traces.Active (Main_Trace) then
         GNATCOLL.Traces.Decrease_Indent (Main_Trace);
         GNATCOLL.Traces.Trace
           (Main_Trace,
            "Finished populating lexical envs for unit: " & Basename (Unit));
      end if;
      Reset_Envs_Caches (Unit);
      if Has_Errors and then not Context.Discard_Errors_In_Populate_Lexical_Env
      then
         raise Property_Error with
            "errors occurred in Populate_Lexical_Env";
      end if;
   end Populate_Lexical_Env;
   ------------------
   -- Get_Filename --
   ------------------
   function Get_Filename (Unit : Internal_Unit) return String is
     (+Unit.Filename.Full_Name);
   -----------------
   -- Get_Charset --
   -----------------
   function Get_Charset (Unit : Internal_Unit) return String is
   begin
      return To_String (Unit.Charset);
   end Get_Charset;
   ---------------------
   -- Has_Diagnostics --
   ---------------------
   function Has_Diagnostics (Unit : Internal_Unit) return Boolean is
   begin
      return not Unit.Diagnostics.Is_Empty;
   end Has_Diagnostics;
   -----------------
   -- Diagnostics --
   -----------------
   function Diagnostics (Unit : Internal_Unit) return Diagnostics_Array is
      Result : Diagnostics_Array (1 .. Natural (Unit.Diagnostics.Length));
      I      : Natural := 1;
   begin
      for D of Unit.Diagnostics loop
         Result (I) := D;
         I := I + 1;
      end loop;
      return Result;
   end Diagnostics;
   ---------------------------
   -- Format_GNU_Diagnostic --
   ---------------------------
   function Format_GNU_Diagnostic
     (Unit : Internal_Unit; D : Diagnostic) return String
   is
      Filename : constant String := Basename (Unit);
      Sloc     : constant Source_Location := Start_Sloc (D.Sloc_Range);
      Msg      : constant String :=
         Image
           (Ada.Strings.Wide_Wide_Unbounded.To_Wide_Wide_String (D.Message));
   begin
      return (Filename
              & (if Sloc = No_Source_Location then "" else ":" & Image (Sloc))
              & ": " & Msg);
   end Format_GNU_Diagnostic;
   ----------
   -- Root --
   ----------
   function Root (Unit : Internal_Unit) return Bare_Ada_Node is
     (Unit.Ast_Root);
   -----------------
   -- First_Token --
   -----------------
   function First_Token (Unit : Internal_Unit) return Token_Reference is
     (Wrap_Token_Reference (Unit.Context,
                            Unit.TDH'Access,
                            First_Token_Or_Trivia (Unit.TDH)));
   ----------------
   -- Last_Token --
   ----------------
   function Last_Token (Unit : Internal_Unit) return Token_Reference is
     (Wrap_Token_Reference (Unit.Context,
                            Unit.TDH'Access,
                            Last_Token_Or_Trivia (Unit.TDH)));
   -----------------
   -- Token_Count --
   -----------------
   function Token_Count (Unit : Internal_Unit) return Natural is
     (Unit.TDH.Tokens.Length);
   ------------------
   -- Trivia_Count --
   ------------------
   function Trivia_Count (Unit : Internal_Unit) return Natural is
     (Unit.TDH.Trivias.Length);
   ----------
   -- Text --
   ----------
   function Text (Unit : Internal_Unit) return Text_Type is
   begin
      return Text (First_Token (Unit), Last_Token (Unit));
   end Text;
   ------------------
   -- Lookup_Token --
   ------------------
   function Lookup_Token
     (Unit : Internal_Unit; Sloc : Source_Location) return Token_Reference
   is
      Result : constant Token_Or_Trivia_Index := Lookup_Token (Unit.TDH, Sloc);
   begin
      return Wrap_Token_Reference (Unit.Context, Unit.TDH'Access, Result);
   end Lookup_Token;
   ----------------------
   -- Dump_Lexical_Env --
   ----------------------
   procedure Dump_Lexical_Env (Unit : Internal_Unit) is
      Node     : constant Bare_Ada_Node := Unit.Ast_Root;
      Root_Env : constant Lexical_Env := Unit.Context.Root_Scope;
      State    : Dump_Lexical_Env_State := (Root_Env => Root_Env, others => <>);
      function Get_Parent (Env : Lexical_Env) return Lexical_Env
      is (Unwrap (Env).Parent);
      --------------------------
      -- Explore_Parent_Chain --
      --------------------------
      procedure Explore_Parent_Chain (Env : Lexical_Env) is
         P : Lexical_Env;
      begin
         if Env /= Null_Lexical_Env then
            P := Get_Parent (Env);
            Dump_One_Lexical_Env
              (Env, Get_Env_Id (Env, State), Get_Env_Id (P, State));
            Explore_Parent_Chain (P);
         end if;
      end Explore_Parent_Chain;
      --------------
      -- Internal --
      --------------
      procedure Internal (Current : Bare_Ada_Node) is
         Explore_Parent : Boolean := False;
         Env, Parent    : Lexical_Env;
      begin
         if Current = null then
            return;
         end if;
         --  We only dump environments that we haven't dumped before. This way
         --  we'll only dump environments at the site of their creation, and
         --  not in any subsequent link. We use the Env_Ids map to check which
         --  envs we have already seen or not.
         if not State.Env_Ids.Contains (Current.Self_Env) then
            Env := Current.Self_Env;
            Parent := Get_Parent (Env);
            Explore_Parent := not State.Env_Ids.Contains (Parent);
            Dump_One_Lexical_Env
              (Env, Get_Env_Id (Env, State), Get_Env_Id (Parent, State));
            if Explore_Parent then
               Explore_Parent_Chain (Parent);
            end if;
         end if;
         for Child of Internal_Bare_Ada_Node_Array'(Children (Current))
         loop
            Internal (Child);
         end loop;
      end Internal;
      --  This procedure implements the main recursive logic of dumping the
      --  environments.
   begin
      Internal (Bare_Ada_Node (Node));
   end Dump_Lexical_Env;
   --------------
   -- Get_Line --
   --------------
   function Get_Line
     (Unit : Internal_Unit; Line_Number : Positive) return Text_Type
   is
   begin
      return Get_Line (Unit.TDH, Line_Number);
   end Get_Line;
   -----------
   -- Print --
   -----------
   procedure Print (Unit : Internal_Unit; Show_Slocs : Boolean) is
   begin
      if Unit.Ast_Root = null then
         Put_Line ("<empty analysis unit>");
      else
         Print (Unit.Ast_Root, Show_Slocs);
      end if;
   end Print;
   ---------------
   -- PP_Trivia --
   ---------------
   procedure PP_Trivia (Unit : Internal_Unit) is
      procedure Process (Trivia : Token_Index) is
         Data : constant Stored_Token_Data :=
            Unit.TDH.Trivias.Get (Natural (Trivia)).T;
      begin
         Put_Line (Image (Text (Unit.TDH, Data)));
      end Process;
      Last_Token : constant Token_Index :=
         Token_Index (Token_Vectors.Last_Index (Unit.TDH.Tokens) - 1);
      --  Index for the last token in Unit excluding the Termination token
      --  (hence the -1).
   begin
      for Tok of Get_Leading_Trivias (Unit.TDH) loop
         Process (Tok);
      end loop;
      PP_Trivia (Unit.Ast_Root);
      for Tok of Get_Trivias (Unit.TDH, Last_Token) loop
         Process (Tok);
      end loop;
   end PP_Trivia;
   -------------
   -- Destroy --
   -------------
   procedure Destroy (Unit : in out Internal_Unit) is
   begin
      if Unit = No_Analysis_Unit then
         return;
      end if;
      Unit.Exiled_Entries.Destroy;
      Unit.Foreign_Nodes.Destroy;
      Unit.Exiled_Entries_In_NED.Destroy;
      Unit.Exiled_Envs.Destroy;
      Unit.Named_Envs.Destroy;
      Analysis_Unit_Sets.Destroy (Unit.Referenced_Units);
         Destroy (Unit.Memoization_Map);
      Destroy_Rebindings (Unit.Rebindings'Access);
      Unit.Rebindings.Destroy;
      if Unit.Ast_Root /= null then
         Destroy (Unit.Ast_Root);
      end if;
      Free (Unit.TDH);
      Free (Unit.Ast_Mem_Pool);
      Destroy_Unit_Destroyables (Unit);
      Destroyable_Vectors.Destroy (Unit.Destroyables);
      Free (Unit);
   end Destroy;
   -------------------
   -- Is_Token_Node --
   -------------------
   function Is_Token_Node (Node : Bare_Ada_Node) return Boolean is
   begin
      return Is_Token_Node (Node.Kind);
   end Is_Token_Node;
   ------------------
   -- Is_Synthetic --
   ------------------
   function Is_Synthetic (Node : Bare_Ada_Node) return Boolean is
   begin
      return Node.Kind in Synthetic_Nodes;
   end Is_Synthetic;
   ------------------------------
   -- Raise_Property_Exception --
   ------------------------------
   procedure Raise_Property_Exception
     (Node    : Bare_Ada_Node;
      Exc     : Ada.Exceptions.Exception_Id;
      Message : String)
   is
      Sloc_Prefix : constant String :=
        (if Node = null
         then ""
         else Ada.Directories.Simple_Name (Get_Filename (Unit (Node)))
              & ":" & Image (Sloc_Range (Node)) & ": ");
   begin
      Ada.Exceptions.Raise_Exception (Exc, Sloc_Prefix & Message);
   end Raise_Property_Exception;
   ------------------------------
   -- Register_Destroyable_Gen --
   ------------------------------
   procedure Register_Destroyable_Gen
     (Unit : Internal_Unit; Object : T_Access)
   is
      function Convert is new Ada.Unchecked_Conversion
        (System.Address, Destroy_Procedure);
      procedure Destroy_Procedure (Object : in out T_Access) renames Destroy;
   begin
      Register_Destroyable_Helper
        (Unit,
         Object.all'Address,
         Convert (Destroy_Procedure'Address));
   end Register_Destroyable_Gen;
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Bare_Ada_Node_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Bare_Ada_Node
   is
      function Absolute_Get
        (T : Bare_Ada_Node_Array_Access; Index : Integer)
         return Bare_Ada_Node
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Bare_Ada_Node,
         Sequence_Type => Bare_Ada_Node_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Bare_Ada_Node;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Bare_Ada_Node;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Bare_Ada_Node_Array_Access) return Bare_Ada_Node_Array_Access is
      Ret : Bare_Ada_Node_Array_Access := Create_Bare_Ada_Node_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Bare_Ada_Node_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Bare_Ada_Node_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Bare_Ada_Node_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Bare_Ada_Node_Array (Items_Count : Natural) return Bare_Ada_Node_Array_Access
   is (if Items_Count = 0
       then No_Bare_Ada_Node_Array_Type
       else new Bare_Ada_Node_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Bare_Ada_Node_Array
     (Items : Internal_Bare_Ada_Node_Array) return Bare_Ada_Node_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Bare_Ada_Node_Array_Type;
      end if;
      return new Bare_Ada_Node_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Bare_Ada_Node_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Bare_Ada_Node_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Bare_Anonymous_Expr_Decl_Vectors is new Langkit_Support.Vectors (Bare_Anonymous_Expr_Decl);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Bare_Anonymous_Expr_Decl_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Bare_Anonymous_Expr_Decl
   is
      function Absolute_Get
        (T : Bare_Anonymous_Expr_Decl_Array_Access; Index : Integer)
         return Bare_Anonymous_Expr_Decl
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Bare_Anonymous_Expr_Decl,
         Sequence_Type => Bare_Anonymous_Expr_Decl_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Bare_Anonymous_Expr_Decl;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Bare_Ada_Node;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Bare_Anonymous_Expr_Decl_Array_Access) return Bare_Anonymous_Expr_Decl_Array_Access is
      Ret : Bare_Anonymous_Expr_Decl_Array_Access := Create_Bare_Anonymous_Expr_Decl_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Bare_Anonymous_Expr_Decl_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Bare_Anonymous_Expr_Decl_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Bare_Anonymous_Expr_Decl_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Bare_Anonymous_Expr_Decl_Array (Items_Count : Natural) return Bare_Anonymous_Expr_Decl_Array_Access
   is (if Items_Count = 0
       then No_Bare_Anonymous_Expr_Decl_Array_Type
       else new Bare_Anonymous_Expr_Decl_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Bare_Anonymous_Expr_Decl_Array
     (Items : Internal_Bare_Anonymous_Expr_Decl_Array) return Bare_Anonymous_Expr_Decl_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Bare_Anonymous_Expr_Decl_Array_Type;
      end if;
      return new Bare_Anonymous_Expr_Decl_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Bare_Anonymous_Expr_Decl_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Bare_Anonymous_Expr_Decl_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Bare_Compilation_Unit_Vectors is new Langkit_Support.Vectors (Bare_Compilation_Unit);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Bare_Compilation_Unit_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Bare_Compilation_Unit
   is
      function Absolute_Get
        (T : Bare_Compilation_Unit_Array_Access; Index : Integer)
         return Bare_Compilation_Unit
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Bare_Compilation_Unit,
         Sequence_Type => Bare_Compilation_Unit_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Bare_Compilation_Unit;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Bare_Ada_Node;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Bare_Compilation_Unit_Array_Access) return Bare_Compilation_Unit_Array_Access is
      Ret : Bare_Compilation_Unit_Array_Access := Create_Bare_Compilation_Unit_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Bare_Compilation_Unit_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Bare_Compilation_Unit_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Bare_Compilation_Unit_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Bare_Compilation_Unit_Array (Items_Count : Natural) return Bare_Compilation_Unit_Array_Access
   is (if Items_Count = 0
       then No_Bare_Compilation_Unit_Array_Type
       else new Bare_Compilation_Unit_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Bare_Compilation_Unit_Array
     (Items : Internal_Bare_Compilation_Unit_Array) return Bare_Compilation_Unit_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Bare_Compilation_Unit_Array_Type;
      end if;
      return new Bare_Compilation_Unit_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Bare_Compilation_Unit_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Bare_Compilation_Unit_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Bare_Expr_Vectors is new Langkit_Support.Vectors (Bare_Expr);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Bare_Expr_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Bare_Expr
   is
      function Absolute_Get
        (T : Bare_Expr_Array_Access; Index : Integer)
         return Bare_Expr
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Bare_Expr,
         Sequence_Type => Bare_Expr_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Bare_Expr;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Bare_Ada_Node;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Bare_Expr_Array_Access) return Bare_Expr_Array_Access is
      Ret : Bare_Expr_Array_Access := Create_Bare_Expr_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Bare_Expr_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Bare_Expr_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Bare_Expr_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Bare_Expr_Array (Items_Count : Natural) return Bare_Expr_Array_Access
   is (if Items_Count = 0
       then No_Bare_Expr_Array_Type
       else new Bare_Expr_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Bare_Expr_Array
     (Items : Internal_Bare_Expr_Array) return Bare_Expr_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Bare_Expr_Array_Type;
      end if;
      return new Bare_Expr_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Bare_Expr_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Bare_Expr_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Bare_Name_Vectors is new Langkit_Support.Vectors (Bare_Name);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Bare_Name_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Bare_Name
   is
      function Absolute_Get
        (T : Bare_Name_Array_Access; Index : Integer)
         return Bare_Name
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Bare_Name,
         Sequence_Type => Bare_Name_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Bare_Name;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Bare_Ada_Node;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Bare_Name_Array_Access) return Bare_Name_Array_Access is
      Ret : Bare_Name_Array_Access := Create_Bare_Name_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Bare_Name_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Bare_Name_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Bare_Name_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Bare_Name_Array (Items_Count : Natural) return Bare_Name_Array_Access
   is (if Items_Count = 0
       then No_Bare_Name_Array_Type
       else new Bare_Name_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Bare_Name_Array
     (Items : Internal_Bare_Name_Array) return Bare_Name_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Bare_Name_Array_Type;
      end if;
      return new Bare_Name_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Bare_Name_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Bare_Name_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Bare_Pragma_Node_Vectors is new Langkit_Support.Vectors (Bare_Pragma_Node);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Bare_Pragma_Node_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Bare_Pragma_Node
   is
      function Absolute_Get
        (T : Bare_Pragma_Node_Array_Access; Index : Integer)
         return Bare_Pragma_Node
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Bare_Pragma_Node,
         Sequence_Type => Bare_Pragma_Node_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Bare_Pragma_Node;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Bare_Ada_Node;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Bare_Pragma_Node_Array_Access) return Bare_Pragma_Node_Array_Access is
      Ret : Bare_Pragma_Node_Array_Access := Create_Bare_Pragma_Node_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Bare_Pragma_Node_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Bare_Pragma_Node_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Bare_Pragma_Node_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Bare_Pragma_Node_Array (Items_Count : Natural) return Bare_Pragma_Node_Array_Access
   is (if Items_Count = 0
       then No_Bare_Pragma_Node_Array_Type
       else new Bare_Pragma_Node_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Bare_Pragma_Node_Array
     (Items : Internal_Bare_Pragma_Node_Array) return Bare_Pragma_Node_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Bare_Pragma_Node_Array_Type;
      end if;
      return new Bare_Pragma_Node_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Bare_Pragma_Node_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Bare_Pragma_Node_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Bare_Single_Tok_Node_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Bare_Single_Tok_Node
   is
      function Absolute_Get
        (T : Bare_Single_Tok_Node_Array_Access; Index : Integer)
         return Bare_Single_Tok_Node
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Bare_Single_Tok_Node,
         Sequence_Type => Bare_Single_Tok_Node_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Bare_Single_Tok_Node;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Bare_Ada_Node;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Bare_Single_Tok_Node_Array_Access) return Bare_Single_Tok_Node_Array_Access is
      Ret : Bare_Single_Tok_Node_Array_Access := Create_Bare_Single_Tok_Node_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Bare_Single_Tok_Node_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Bare_Single_Tok_Node_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Bare_Single_Tok_Node_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Bare_Single_Tok_Node_Array (Items_Count : Natural) return Bare_Single_Tok_Node_Array_Access
   is (if Items_Count = 0
       then No_Bare_Single_Tok_Node_Array_Type
       else new Bare_Single_Tok_Node_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Bare_Single_Tok_Node_Array
     (Items : Internal_Bare_Single_Tok_Node_Array) return Bare_Single_Tok_Node_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Bare_Single_Tok_Node_Array_Type;
      end if;
      return new Bare_Single_Tok_Node_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Bare_Single_Tok_Node_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Bare_Single_Tok_Node_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Bare_Use_Clause_Vectors is new Langkit_Support.Vectors (Bare_Use_Clause);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Bare_Use_Clause_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Bare_Use_Clause
   is
      function Absolute_Get
        (T : Bare_Use_Clause_Array_Access; Index : Integer)
         return Bare_Use_Clause
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Bare_Use_Clause,
         Sequence_Type => Bare_Use_Clause_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Bare_Use_Clause;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Bare_Ada_Node;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Bare_Use_Clause_Array_Access) return Bare_Use_Clause_Array_Access is
      Ret : Bare_Use_Clause_Array_Access := Create_Bare_Use_Clause_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Bare_Use_Clause_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Bare_Use_Clause_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Bare_Use_Clause_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Bare_Use_Clause_Array (Items_Count : Natural) return Bare_Use_Clause_Array_Access
   is (if Items_Count = 0
       then No_Bare_Use_Clause_Array_Type
       else new Bare_Use_Clause_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Bare_Use_Clause_Array
     (Items : Internal_Bare_Use_Clause_Array) return Bare_Use_Clause_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Bare_Use_Clause_Array_Type;
      end if;
      return new Bare_Use_Clause_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Bare_Use_Clause_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Bare_Use_Clause_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Aspect_Vectors is new Langkit_Support.Vectors (Internal_Aspect);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Aspect_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Aspect
   is
      function Absolute_Get
        (T : Internal_Aspect_Array_Access; Index : Integer)
         return Internal_Aspect
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Aspect,
         Sequence_Type => Internal_Aspect_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Aspect;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Aspect;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Aspect_Array_Access) return Internal_Aspect_Array_Access is
      Ret : Internal_Aspect_Array_Access := Create_Internal_Aspect_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Aspect_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Aspect_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Aspect_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Aspect_Array (Items_Count : Natural) return Internal_Aspect_Array_Access
   is (if Items_Count = 0
       then No_Internal_Aspect_Array_Type
       else new Internal_Aspect_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Aspect_Array
     (Items : Internal_Internal_Aspect_Array) return Internal_Aspect_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Aspect_Array_Type;
      end if;
      return new Internal_Aspect_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Aspect_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Aspect_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Completion_Item_Vectors is new Langkit_Support.Vectors (Internal_Completion_Item);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Completion_Item_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Completion_Item
   is
      function Absolute_Get
        (T : Internal_Completion_Item_Array_Access; Index : Integer)
         return Internal_Completion_Item
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Completion_Item,
         Sequence_Type => Internal_Completion_Item_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Completion_Item;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Completion_Item;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Completion_Item_Array_Access) return Internal_Completion_Item_Array_Access is
      Ret : Internal_Completion_Item_Array_Access := Create_Internal_Completion_Item_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Completion_Item_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Completion_Item_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Completion_Item_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Completion_Item_Array (Items_Count : Natural) return Internal_Completion_Item_Array_Access
   is (if Items_Count = 0
       then No_Internal_Completion_Item_Array_Type
       else new Internal_Completion_Item_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Completion_Item_Array
     (Items : Internal_Internal_Completion_Item_Array) return Internal_Completion_Item_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Completion_Item_Array_Type;
      end if;
      return new Internal_Completion_Item_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Completion_Item_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- To_Iterator --
      -----------------
      function To_Iterator
        (Self    : Internal_Completion_Item_Array_Access;
         Context : Internal_Context) return Internal_Completion_Item_Iterator_Access
      is
      begin
         Inc_Ref (Self);
         return new Internal_Internal_Completion_Item_Iterator'
           (Ref_Count  => 1,
            Safety_Net => Create_Safety_Net (Context),
            Elements   => Self,
            Index      => 1);
      end To_Iterator;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Completion_Item_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Discriminant_Values_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Discriminant_Values
   is
      function Absolute_Get
        (T : Internal_Discriminant_Values_Array_Access; Index : Integer)
         return Internal_Discriminant_Values
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Discriminant_Values,
         Sequence_Type => Internal_Discriminant_Values_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Discriminant_Values;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Discriminant_Values;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Discriminant_Values_Array_Access) return Internal_Discriminant_Values_Array_Access is
      Ret : Internal_Discriminant_Values_Array_Access := Create_Internal_Discriminant_Values_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Discriminant_Values_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Discriminant_Values_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Discriminant_Values_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Discriminant_Values_Array (Items_Count : Natural) return Internal_Discriminant_Values_Array_Access
   is (if Items_Count = 0
       then No_Internal_Discriminant_Values_Array_Type
       else new Internal_Discriminant_Values_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Discriminant_Values_Array
     (Items : Internal_Internal_Discriminant_Values_Array) return Internal_Discriminant_Values_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Discriminant_Values_Array_Type;
      end if;
      return new Internal_Discriminant_Values_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Discriminant_Values_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Discriminant_Values_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Doc_Annotation_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Doc_Annotation
   is
      function Absolute_Get
        (T : Internal_Doc_Annotation_Array_Access; Index : Integer)
         return Internal_Doc_Annotation
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Doc_Annotation,
         Sequence_Type => Internal_Doc_Annotation_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Doc_Annotation;
   begin
      if Relative_Get (T, Index, Result) then
            Inc_Ref (Result);
         return Result;
      elsif Or_Null then
         return No_Doc_Annotation;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Doc_Annotation_Array_Access) return Internal_Doc_Annotation_Array_Access is
      Ret : Internal_Doc_Annotation_Array_Access := Create_Internal_Doc_Annotation_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
         for Item of Ret.Items loop
            Inc_Ref (Item);
         end loop;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Doc_Annotation_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Doc_Annotation_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Doc_Annotation_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
            for Item of T.Items loop
               Dec_Ref (Item);
            end loop;
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Doc_Annotation_Array (Items_Count : Natural) return Internal_Doc_Annotation_Array_Access
   is (if Items_Count = 0
       then No_Internal_Doc_Annotation_Array_Type
       else new Internal_Doc_Annotation_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Doc_Annotation_Array
     (Items : Internal_Internal_Doc_Annotation_Array) return Internal_Doc_Annotation_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Doc_Annotation_Array_Type;
      end if;
         for El of Items loop
            Inc_Ref (El);
         end loop;
      return new Internal_Doc_Annotation_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Doc_Annotation_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               not Equivalent (L.Items (I), R.Items (I))
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Doc_Annotation_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Entity_Accept_Stmt_Vectors is new Langkit_Support.Vectors (Internal_Entity_Accept_Stmt);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Accept_Stmt_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Accept_Stmt
   is
      function Absolute_Get
        (T : Internal_Entity_Accept_Stmt_Array_Access; Index : Integer)
         return Internal_Entity_Accept_Stmt
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Accept_Stmt,
         Sequence_Type => Internal_Entity_Accept_Stmt_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Accept_Stmt;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Accept_Stmt;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Accept_Stmt_Array_Access) return Internal_Entity_Accept_Stmt_Array_Access is
      Ret : Internal_Entity_Accept_Stmt_Array_Access := Create_Internal_Entity_Accept_Stmt_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Accept_Stmt_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Accept_Stmt_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Accept_Stmt_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Accept_Stmt_Array (Items_Count : Natural) return Internal_Entity_Accept_Stmt_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Accept_Stmt_Array_Type
       else new Internal_Entity_Accept_Stmt_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Accept_Stmt_Array
     (Items : Internal_Internal_Entity_Accept_Stmt_Array) return Internal_Entity_Accept_Stmt_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Accept_Stmt_Array_Type;
      end if;
      return new Internal_Entity_Accept_Stmt_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Accept_Stmt_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Accept_Stmt_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Entity_Vectors is new Langkit_Support.Vectors (Internal_Entity);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity
   is
      function Absolute_Get
        (T : Internal_Entity_Array_Access; Index : Integer)
         return Internal_Entity
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity,
         Sequence_Type => Internal_Entity_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Array_Access) return Internal_Entity_Array_Access is
      Ret : Internal_Entity_Array_Access := Create_Internal_Entity_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Array (Items_Count : Natural) return Internal_Entity_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Array_Type
       else new Internal_Entity_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Array
     (Items : AST_Envs.Entity_Array) return Internal_Entity_Array_Access
   is (if Items'Length = 0
       then No_Internal_Entity_Array_Type
       else new Internal_Entity_Array_Record'
         (N         => Items'Length,
          Items     => Implementation.Internal_Internal_Entity_Array (Items),
          Ref_Count => 1));
   function Create_Internal_Entity_Array
     (Items : Internal_Internal_Entity_Array) return Internal_Entity_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Array_Type;
      end if;
      return new Internal_Entity_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Entity_Base_Assoc_Vectors is new Langkit_Support.Vectors (Internal_Entity_Base_Assoc);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Base_Assoc_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Base_Assoc
   is
      function Absolute_Get
        (T : Internal_Entity_Base_Assoc_Array_Access; Index : Integer)
         return Internal_Entity_Base_Assoc
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Base_Assoc,
         Sequence_Type => Internal_Entity_Base_Assoc_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Base_Assoc;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Base_Assoc;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Base_Assoc_Array_Access) return Internal_Entity_Base_Assoc_Array_Access is
      Ret : Internal_Entity_Base_Assoc_Array_Access := Create_Internal_Entity_Base_Assoc_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Base_Assoc_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Base_Assoc_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Base_Assoc_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Base_Assoc_Array (Items_Count : Natural) return Internal_Entity_Base_Assoc_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Base_Assoc_Array_Type
       else new Internal_Entity_Base_Assoc_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Base_Assoc_Array
     (Items : Internal_Internal_Entity_Base_Assoc_Array) return Internal_Entity_Base_Assoc_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Base_Assoc_Array_Type;
      end if;
      return new Internal_Entity_Base_Assoc_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Base_Assoc_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Base_Assoc_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Entity_Base_Formal_Param_Decl_Vectors is new Langkit_Support.Vectors (Internal_Entity_Base_Formal_Param_Decl);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Base_Formal_Param_Decl_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Base_Formal_Param_Decl
   is
      function Absolute_Get
        (T : Internal_Entity_Base_Formal_Param_Decl_Array_Access; Index : Integer)
         return Internal_Entity_Base_Formal_Param_Decl
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Base_Formal_Param_Decl,
         Sequence_Type => Internal_Entity_Base_Formal_Param_Decl_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Base_Formal_Param_Decl;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Base_Formal_Param_Decl;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Base_Formal_Param_Decl_Array_Access) return Internal_Entity_Base_Formal_Param_Decl_Array_Access is
      Ret : Internal_Entity_Base_Formal_Param_Decl_Array_Access := Create_Internal_Entity_Base_Formal_Param_Decl_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Base_Formal_Param_Decl_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Base_Formal_Param_Decl_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Base_Formal_Param_Decl_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Base_Formal_Param_Decl_Array (Items_Count : Natural) return Internal_Entity_Base_Formal_Param_Decl_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Base_Formal_Param_Decl_Array_Type
       else new Internal_Entity_Base_Formal_Param_Decl_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Base_Formal_Param_Decl_Array
     (Items : Internal_Internal_Entity_Base_Formal_Param_Decl_Array) return Internal_Entity_Base_Formal_Param_Decl_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Base_Formal_Param_Decl_Array_Type;
      end if;
      return new Internal_Entity_Base_Formal_Param_Decl_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Base_Formal_Param_Decl_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Base_Formal_Param_Decl_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Entity_Base_Type_Decl_Vectors is new Langkit_Support.Vectors (Internal_Entity_Base_Type_Decl);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Base_Type_Decl_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Base_Type_Decl
   is
      function Absolute_Get
        (T : Internal_Entity_Base_Type_Decl_Array_Access; Index : Integer)
         return Internal_Entity_Base_Type_Decl
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Base_Type_Decl,
         Sequence_Type => Internal_Entity_Base_Type_Decl_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Base_Type_Decl;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Base_Type_Decl;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Base_Type_Decl_Array_Access) return Internal_Entity_Base_Type_Decl_Array_Access is
      Ret : Internal_Entity_Base_Type_Decl_Array_Access := Create_Internal_Entity_Base_Type_Decl_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Base_Type_Decl_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Base_Type_Decl_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Base_Type_Decl_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Base_Type_Decl_Array (Items_Count : Natural) return Internal_Entity_Base_Type_Decl_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Base_Type_Decl_Array_Type
       else new Internal_Entity_Base_Type_Decl_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Base_Type_Decl_Array
     (Items : Internal_Internal_Entity_Base_Type_Decl_Array) return Internal_Entity_Base_Type_Decl_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Base_Type_Decl_Array_Type;
      end if;
      return new Internal_Entity_Base_Type_Decl_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Base_Type_Decl_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Base_Type_Decl_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      function Make_Unique (A : Internal_Entity_Base_Type_Decl_Array_Access) return Internal_Entity_Base_Type_Decl_Array_Access is
         --  To keep the order of the result deterministic, this function works
         --  in two steps. First, determine for each item in A whether it
         --  duplicate a previous item. Only then, go through all items in A
         --  and add only non-duplicated ones to the result.
         Duplicates : array (A.Items'Range) of Boolean := (others => False);
         --  For each item in A, indicate whether the value is a duplicate of a
         --  previous one in the array.
         Unique_Elements : Natural;
         --  Number of unique elements in A
      begin
         --  Step 1: compute Duplicates
         declare
            package Sets is new Ada.Containers.Hashed_Sets
              (Element_Type        => Internal_Entity_Base_Type_Decl,
               "="                 => "=",
               Equivalent_Elements => "=",
               Hash                => Hash);
            Set : Sets.Set;
         begin
            for I in A.Items'Range loop
               declare
                  Item : Internal_Entity_Base_Type_Decl renames A.Items (I);
               begin
                  if Set.Contains (Item) then
                     Duplicates (I) := True;
                  else
                     Set.Insert (Item);
                  end if;
               end;
            end loop;
            Unique_Elements := Natural (Set.Length);
         end;
         --  Step 2: create the result
         declare
            Next   : Positive := 1;
            Result : constant Internal_Entity_Base_Type_Decl_Array_Access :=
               Create_Internal_Entity_Base_Type_Decl_Array (Unique_Elements);
         begin
            for I in A.Items'Range loop
               declare
                  Item : Internal_Entity_Base_Type_Decl renames A.Items (I);
               begin
                  if not Duplicates (I) then
                     Result.Items (Next) := Item;
                     Next := Next + 1;
                  end if;
               end;
            end loop;
            pragma Assert (Next = Result.N + 1);
            return Result;
         end;
      end Make_Unique;
      ----------
      -- Hash --
      ----------
      function Hash (R : Internal_Entity_Base_Type_Decl_Array_Access) return Hash_Type is
         Result : Hash_Type := Initial_Hash;
      begin
         for I in R.Items'Range loop
            Result := Combine (Result, Hash (R.Items (I)));
         end loop;
         return Result;
      end Hash;
      package Internal_Entity_Basic_Assoc_Vectors is new Langkit_Support.Vectors (Internal_Entity_Basic_Assoc);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Basic_Assoc_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Basic_Assoc
   is
      function Absolute_Get
        (T : Internal_Entity_Basic_Assoc_Array_Access; Index : Integer)
         return Internal_Entity_Basic_Assoc
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Basic_Assoc,
         Sequence_Type => Internal_Entity_Basic_Assoc_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Basic_Assoc;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Basic_Assoc;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Basic_Assoc_Array_Access) return Internal_Entity_Basic_Assoc_Array_Access is
      Ret : Internal_Entity_Basic_Assoc_Array_Access := Create_Internal_Entity_Basic_Assoc_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Basic_Assoc_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Basic_Assoc_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Basic_Assoc_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Basic_Assoc_Array (Items_Count : Natural) return Internal_Entity_Basic_Assoc_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Basic_Assoc_Array_Type
       else new Internal_Entity_Basic_Assoc_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Basic_Assoc_Array
     (Items : Internal_Internal_Entity_Basic_Assoc_Array) return Internal_Entity_Basic_Assoc_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Basic_Assoc_Array_Type;
      end if;
      return new Internal_Entity_Basic_Assoc_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Basic_Assoc_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Basic_Assoc_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Entity_Basic_Decl_Vectors is new Langkit_Support.Vectors (Internal_Entity_Basic_Decl);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Basic_Decl_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Basic_Decl
   is
      function Absolute_Get
        (T : Internal_Entity_Basic_Decl_Array_Access; Index : Integer)
         return Internal_Entity_Basic_Decl
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Basic_Decl,
         Sequence_Type => Internal_Entity_Basic_Decl_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Basic_Decl;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Basic_Decl;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Basic_Decl_Array_Access) return Internal_Entity_Basic_Decl_Array_Access is
      Ret : Internal_Entity_Basic_Decl_Array_Access := Create_Internal_Entity_Basic_Decl_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Basic_Decl_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Basic_Decl_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Basic_Decl_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Basic_Decl_Array (Items_Count : Natural) return Internal_Entity_Basic_Decl_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Basic_Decl_Array_Type
       else new Internal_Entity_Basic_Decl_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Basic_Decl_Array
     (Items : Internal_Internal_Entity_Basic_Decl_Array) return Internal_Entity_Basic_Decl_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Basic_Decl_Array_Type;
      end if;
      return new Internal_Entity_Basic_Decl_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Basic_Decl_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Basic_Decl_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      function Make_Unique (A : Internal_Entity_Basic_Decl_Array_Access) return Internal_Entity_Basic_Decl_Array_Access is
         --  To keep the order of the result deterministic, this function works
         --  in two steps. First, determine for each item in A whether it
         --  duplicate a previous item. Only then, go through all items in A
         --  and add only non-duplicated ones to the result.
         Duplicates : array (A.Items'Range) of Boolean := (others => False);
         --  For each item in A, indicate whether the value is a duplicate of a
         --  previous one in the array.
         Unique_Elements : Natural;
         --  Number of unique elements in A
      begin
         --  Step 1: compute Duplicates
         declare
            package Sets is new Ada.Containers.Hashed_Sets
              (Element_Type        => Internal_Entity_Basic_Decl,
               "="                 => "=",
               Equivalent_Elements => "=",
               Hash                => Hash);
            Set : Sets.Set;
         begin
            for I in A.Items'Range loop
               declare
                  Item : Internal_Entity_Basic_Decl renames A.Items (I);
               begin
                  if Set.Contains (Item) then
                     Duplicates (I) := True;
                  else
                     Set.Insert (Item);
                  end if;
               end;
            end loop;
            Unique_Elements := Natural (Set.Length);
         end;
         --  Step 2: create the result
         declare
            Next   : Positive := 1;
            Result : constant Internal_Entity_Basic_Decl_Array_Access :=
               Create_Internal_Entity_Basic_Decl_Array (Unique_Elements);
         begin
            for I in A.Items'Range loop
               declare
                  Item : Internal_Entity_Basic_Decl renames A.Items (I);
               begin
                  if not Duplicates (I) then
                     Result.Items (Next) := Item;
                     Next := Next + 1;
                  end if;
               end;
            end loop;
            pragma Assert (Next = Result.N + 1);
            return Result;
         end;
      end Make_Unique;
      package Internal_Entity_Compilation_Unit_Vectors is new Langkit_Support.Vectors (Internal_Entity_Compilation_Unit);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Compilation_Unit_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Compilation_Unit
   is
      function Absolute_Get
        (T : Internal_Entity_Compilation_Unit_Array_Access; Index : Integer)
         return Internal_Entity_Compilation_Unit
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Compilation_Unit,
         Sequence_Type => Internal_Entity_Compilation_Unit_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Compilation_Unit;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Compilation_Unit;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Compilation_Unit_Array_Access) return Internal_Entity_Compilation_Unit_Array_Access is
      Ret : Internal_Entity_Compilation_Unit_Array_Access := Create_Internal_Entity_Compilation_Unit_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Compilation_Unit_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Compilation_Unit_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Compilation_Unit_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Compilation_Unit_Array (Items_Count : Natural) return Internal_Entity_Compilation_Unit_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Compilation_Unit_Array_Type
       else new Internal_Entity_Compilation_Unit_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Compilation_Unit_Array
     (Items : Internal_Internal_Entity_Compilation_Unit_Array) return Internal_Entity_Compilation_Unit_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Compilation_Unit_Array_Type;
      end if;
      return new Internal_Entity_Compilation_Unit_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Compilation_Unit_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Compilation_Unit_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      function Make_Unique (A : Internal_Entity_Compilation_Unit_Array_Access) return Internal_Entity_Compilation_Unit_Array_Access is
         --  To keep the order of the result deterministic, this function works
         --  in two steps. First, determine for each item in A whether it
         --  duplicate a previous item. Only then, go through all items in A
         --  and add only non-duplicated ones to the result.
         Duplicates : array (A.Items'Range) of Boolean := (others => False);
         --  For each item in A, indicate whether the value is a duplicate of a
         --  previous one in the array.
         Unique_Elements : Natural;
         --  Number of unique elements in A
      begin
         --  Step 1: compute Duplicates
         declare
            package Sets is new Ada.Containers.Hashed_Sets
              (Element_Type        => Internal_Entity_Compilation_Unit,
               "="                 => "=",
               Equivalent_Elements => "=",
               Hash                => Hash);
            Set : Sets.Set;
         begin
            for I in A.Items'Range loop
               declare
                  Item : Internal_Entity_Compilation_Unit renames A.Items (I);
               begin
                  if Set.Contains (Item) then
                     Duplicates (I) := True;
                  else
                     Set.Insert (Item);
                  end if;
               end;
            end loop;
            Unique_Elements := Natural (Set.Length);
         end;
         --  Step 2: create the result
         declare
            Next   : Positive := 1;
            Result : constant Internal_Entity_Compilation_Unit_Array_Access :=
               Create_Internal_Entity_Compilation_Unit_Array (Unique_Elements);
         begin
            for I in A.Items'Range loop
               declare
                  Item : Internal_Entity_Compilation_Unit renames A.Items (I);
               begin
                  if not Duplicates (I) then
                     Result.Items (Next) := Item;
                     Next := Next + 1;
                  end if;
               end;
            end loop;
            pragma Assert (Next = Result.N + 1);
            return Result;
         end;
      end Make_Unique;
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Declarative_Part_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Declarative_Part
   is
      function Absolute_Get
        (T : Internal_Entity_Declarative_Part_Array_Access; Index : Integer)
         return Internal_Entity_Declarative_Part
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Declarative_Part,
         Sequence_Type => Internal_Entity_Declarative_Part_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Declarative_Part;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Declarative_Part;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Declarative_Part_Array_Access) return Internal_Entity_Declarative_Part_Array_Access is
      Ret : Internal_Entity_Declarative_Part_Array_Access := Create_Internal_Entity_Declarative_Part_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Declarative_Part_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Declarative_Part_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Declarative_Part_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Declarative_Part_Array (Items_Count : Natural) return Internal_Entity_Declarative_Part_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Declarative_Part_Array_Type
       else new Internal_Entity_Declarative_Part_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Declarative_Part_Array
     (Items : Internal_Internal_Entity_Declarative_Part_Array) return Internal_Entity_Declarative_Part_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Declarative_Part_Array_Type;
      end if;
      return new Internal_Entity_Declarative_Part_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Declarative_Part_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Declarative_Part_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Entity_Defining_Name_Vectors is new Langkit_Support.Vectors (Internal_Entity_Defining_Name);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Defining_Name_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Defining_Name
   is
      function Absolute_Get
        (T : Internal_Entity_Defining_Name_Array_Access; Index : Integer)
         return Internal_Entity_Defining_Name
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Defining_Name,
         Sequence_Type => Internal_Entity_Defining_Name_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Defining_Name;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Defining_Name;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Defining_Name_Array_Access) return Internal_Entity_Defining_Name_Array_Access is
      Ret : Internal_Entity_Defining_Name_Array_Access := Create_Internal_Entity_Defining_Name_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Defining_Name_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Defining_Name_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Defining_Name_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Defining_Name_Array (Items_Count : Natural) return Internal_Entity_Defining_Name_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Defining_Name_Array_Type
       else new Internal_Entity_Defining_Name_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Defining_Name_Array
     (Items : Internal_Internal_Entity_Defining_Name_Array) return Internal_Entity_Defining_Name_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Defining_Name_Array_Type;
      end if;
      return new Internal_Entity_Defining_Name_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Defining_Name_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Defining_Name_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Entity_Expr_Vectors is new Langkit_Support.Vectors (Internal_Entity_Expr);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Expr_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Expr
   is
      function Absolute_Get
        (T : Internal_Entity_Expr_Array_Access; Index : Integer)
         return Internal_Entity_Expr
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Expr,
         Sequence_Type => Internal_Entity_Expr_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Expr;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Expr;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Expr_Array_Access) return Internal_Entity_Expr_Array_Access is
      Ret : Internal_Entity_Expr_Array_Access := Create_Internal_Entity_Expr_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Expr_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Expr_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Expr_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Expr_Array (Items_Count : Natural) return Internal_Entity_Expr_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Expr_Array_Type
       else new Internal_Entity_Expr_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Expr_Array
     (Items : Internal_Internal_Entity_Expr_Array) return Internal_Entity_Expr_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Expr_Array_Type;
      end if;
      return new Internal_Entity_Expr_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Expr_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Expr_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Generic_Instantiation_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Generic_Instantiation
   is
      function Absolute_Get
        (T : Internal_Entity_Generic_Instantiation_Array_Access; Index : Integer)
         return Internal_Entity_Generic_Instantiation
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Generic_Instantiation,
         Sequence_Type => Internal_Entity_Generic_Instantiation_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Generic_Instantiation;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Generic_Instantiation;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Generic_Instantiation_Array_Access) return Internal_Entity_Generic_Instantiation_Array_Access is
      Ret : Internal_Entity_Generic_Instantiation_Array_Access := Create_Internal_Entity_Generic_Instantiation_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Generic_Instantiation_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Generic_Instantiation_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Generic_Instantiation_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Generic_Instantiation_Array (Items_Count : Natural) return Internal_Entity_Generic_Instantiation_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Generic_Instantiation_Array_Type
       else new Internal_Entity_Generic_Instantiation_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Generic_Instantiation_Array
     (Items : Internal_Internal_Entity_Generic_Instantiation_Array) return Internal_Entity_Generic_Instantiation_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Generic_Instantiation_Array_Type;
      end if;
      return new Internal_Entity_Generic_Instantiation_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Generic_Instantiation_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Generic_Instantiation_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Entity_Identifier_Vectors is new Langkit_Support.Vectors (Internal_Entity_Identifier);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Identifier_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Identifier
   is
      function Absolute_Get
        (T : Internal_Entity_Identifier_Array_Access; Index : Integer)
         return Internal_Entity_Identifier
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Identifier,
         Sequence_Type => Internal_Entity_Identifier_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Identifier;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Identifier;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Identifier_Array_Access) return Internal_Entity_Identifier_Array_Access is
      Ret : Internal_Entity_Identifier_Array_Access := Create_Internal_Entity_Identifier_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Identifier_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Identifier_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Identifier_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Identifier_Array (Items_Count : Natural) return Internal_Entity_Identifier_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Identifier_Array_Type
       else new Internal_Entity_Identifier_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Identifier_Array
     (Items : Internal_Internal_Entity_Identifier_Array) return Internal_Entity_Identifier_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Identifier_Array_Type;
      end if;
      return new Internal_Entity_Identifier_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Identifier_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Identifier_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Entity_Mode_Vectors is new Langkit_Support.Vectors (Internal_Entity_Mode);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Mode_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Mode
   is
      function Absolute_Get
        (T : Internal_Entity_Mode_Array_Access; Index : Integer)
         return Internal_Entity_Mode
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Mode,
         Sequence_Type => Internal_Entity_Mode_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Mode;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Mode;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Mode_Array_Access) return Internal_Entity_Mode_Array_Access is
      Ret : Internal_Entity_Mode_Array_Access := Create_Internal_Entity_Mode_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Mode_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Mode_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Mode_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Mode_Array (Items_Count : Natural) return Internal_Entity_Mode_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Mode_Array_Type
       else new Internal_Entity_Mode_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Mode_Array
     (Items : Internal_Internal_Entity_Mode_Array) return Internal_Entity_Mode_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Mode_Array_Type;
      end if;
      return new Internal_Entity_Mode_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Mode_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Mode_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Entity_Name_Vectors is new Langkit_Support.Vectors (Internal_Entity_Name);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Name_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Name
   is
      function Absolute_Get
        (T : Internal_Entity_Name_Array_Access; Index : Integer)
         return Internal_Entity_Name
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Name,
         Sequence_Type => Internal_Entity_Name_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Name;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Name;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Name_Array_Access) return Internal_Entity_Name_Array_Access is
      Ret : Internal_Entity_Name_Array_Access := Create_Internal_Entity_Name_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Name_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Name_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Name_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Name_Array (Items_Count : Natural) return Internal_Entity_Name_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Name_Array_Type
       else new Internal_Entity_Name_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Name_Array
     (Items : Internal_Internal_Entity_Name_Array) return Internal_Entity_Name_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Name_Array_Type;
      end if;
      return new Internal_Entity_Name_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Name_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Name_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Entity_Param_Spec_Vectors is new Langkit_Support.Vectors (Internal_Entity_Param_Spec);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Param_Spec_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Param_Spec
   is
      function Absolute_Get
        (T : Internal_Entity_Param_Spec_Array_Access; Index : Integer)
         return Internal_Entity_Param_Spec
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Param_Spec,
         Sequence_Type => Internal_Entity_Param_Spec_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Param_Spec;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Param_Spec;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Param_Spec_Array_Access) return Internal_Entity_Param_Spec_Array_Access is
      Ret : Internal_Entity_Param_Spec_Array_Access := Create_Internal_Entity_Param_Spec_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Param_Spec_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Param_Spec_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Param_Spec_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Param_Spec_Array (Items_Count : Natural) return Internal_Entity_Param_Spec_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Param_Spec_Array_Type
       else new Internal_Entity_Param_Spec_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Param_Spec_Array
     (Items : Internal_Internal_Entity_Param_Spec_Array) return Internal_Entity_Param_Spec_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Param_Spec_Array_Type;
      end if;
      return new Internal_Entity_Param_Spec_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Param_Spec_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Param_Spec_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Entity_Pragma_Node_Vectors is new Langkit_Support.Vectors (Internal_Entity_Pragma_Node);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Pragma_Node_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Pragma_Node
   is
      function Absolute_Get
        (T : Internal_Entity_Pragma_Node_Array_Access; Index : Integer)
         return Internal_Entity_Pragma_Node
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Pragma_Node,
         Sequence_Type => Internal_Entity_Pragma_Node_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Pragma_Node;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Pragma_Node;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Pragma_Node_Array_Access) return Internal_Entity_Pragma_Node_Array_Access is
      Ret : Internal_Entity_Pragma_Node_Array_Access := Create_Internal_Entity_Pragma_Node_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Pragma_Node_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Pragma_Node_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Pragma_Node_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Pragma_Node_Array (Items_Count : Natural) return Internal_Entity_Pragma_Node_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Pragma_Node_Array_Type
       else new Internal_Entity_Pragma_Node_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Pragma_Node_Array
     (Items : Internal_Internal_Entity_Pragma_Node_Array) return Internal_Entity_Pragma_Node_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Pragma_Node_Array_Type;
      end if;
      return new Internal_Entity_Pragma_Node_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Pragma_Node_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Pragma_Node_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Entity_Type_Decl_Vectors is new Langkit_Support.Vectors (Internal_Entity_Type_Decl);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Type_Decl_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Type_Decl
   is
      function Absolute_Get
        (T : Internal_Entity_Type_Decl_Array_Access; Index : Integer)
         return Internal_Entity_Type_Decl
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Type_Decl,
         Sequence_Type => Internal_Entity_Type_Decl_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Type_Decl;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Type_Decl;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Type_Decl_Array_Access) return Internal_Entity_Type_Decl_Array_Access is
      Ret : Internal_Entity_Type_Decl_Array_Access := Create_Internal_Entity_Type_Decl_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Type_Decl_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Type_Decl_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Type_Decl_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Type_Decl_Array (Items_Count : Natural) return Internal_Entity_Type_Decl_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Type_Decl_Array_Type
       else new Internal_Entity_Type_Decl_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Type_Decl_Array
     (Items : Internal_Internal_Entity_Type_Decl_Array) return Internal_Entity_Type_Decl_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Type_Decl_Array_Type;
      end if;
      return new Internal_Entity_Type_Decl_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Type_Decl_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Type_Decl_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Entity_Type_Expr_Vectors is new Langkit_Support.Vectors (Internal_Entity_Type_Expr);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Type_Expr_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Type_Expr
   is
      function Absolute_Get
        (T : Internal_Entity_Type_Expr_Array_Access; Index : Integer)
         return Internal_Entity_Type_Expr
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Type_Expr,
         Sequence_Type => Internal_Entity_Type_Expr_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Type_Expr;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Type_Expr;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Type_Expr_Array_Access) return Internal_Entity_Type_Expr_Array_Access is
      Ret : Internal_Entity_Type_Expr_Array_Access := Create_Internal_Entity_Type_Expr_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Type_Expr_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Type_Expr_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Type_Expr_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Type_Expr_Array (Items_Count : Natural) return Internal_Entity_Type_Expr_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Type_Expr_Array_Type
       else new Internal_Entity_Type_Expr_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Type_Expr_Array
     (Items : Internal_Internal_Entity_Type_Expr_Array) return Internal_Entity_Type_Expr_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Type_Expr_Array_Type;
      end if;
      return new Internal_Entity_Type_Expr_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Type_Expr_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Type_Expr_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Env_Assoc_Vectors is new Langkit_Support.Vectors (Internal_Env_Assoc);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Env_Assoc_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Env_Assoc
   is
      function Absolute_Get
        (T : Internal_Env_Assoc_Array_Access; Index : Integer)
         return Internal_Env_Assoc
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Env_Assoc,
         Sequence_Type => Internal_Env_Assoc_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Env_Assoc;
   begin
      if Relative_Get (T, Index, Result) then
            Inc_Ref (Result);
         return Result;
      elsif Or_Null then
         return No_Env_Assoc;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Env_Assoc_Array_Access) return Internal_Env_Assoc_Array_Access is
      Ret : Internal_Env_Assoc_Array_Access := Create_Internal_Env_Assoc_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
         for Item of Ret.Items loop
            Inc_Ref (Item);
         end loop;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Env_Assoc_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Env_Assoc_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Env_Assoc_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
            for Item of T.Items loop
               Dec_Ref (Item);
            end loop;
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Env_Assoc_Array (Items_Count : Natural) return Internal_Env_Assoc_Array_Access
   is (if Items_Count = 0
       then No_Internal_Env_Assoc_Array_Type
       else new Internal_Env_Assoc_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Env_Assoc_Array
     (Items : Internal_Internal_Env_Assoc_Array) return Internal_Env_Assoc_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Env_Assoc_Array_Type;
      end if;
         for El of Items loop
            Inc_Ref (El);
         end loop;
      return new Internal_Env_Assoc_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Env_Assoc_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               not Equivalent (L.Items (I), R.Items (I))
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Env_Assoc_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Expected_Type_For_Expr_Vectors is new Langkit_Support.Vectors (Internal_Expected_Type_For_Expr);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Expected_Type_For_Expr_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Expected_Type_For_Expr
   is
      function Absolute_Get
        (T : Internal_Expected_Type_For_Expr_Array_Access; Index : Integer)
         return Internal_Expected_Type_For_Expr
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Expected_Type_For_Expr,
         Sequence_Type => Internal_Expected_Type_For_Expr_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Expected_Type_For_Expr;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Expected_Type_For_Expr;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Expected_Type_For_Expr_Array_Access) return Internal_Expected_Type_For_Expr_Array_Access is
      Ret : Internal_Expected_Type_For_Expr_Array_Access := Create_Internal_Expected_Type_For_Expr_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Expected_Type_For_Expr_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Expected_Type_For_Expr_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Expected_Type_For_Expr_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Expected_Type_For_Expr_Array (Items_Count : Natural) return Internal_Expected_Type_For_Expr_Array_Access
   is (if Items_Count = 0
       then No_Internal_Expected_Type_For_Expr_Array_Type
       else new Internal_Expected_Type_For_Expr_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Expected_Type_For_Expr_Array
     (Items : Internal_Internal_Expected_Type_For_Expr_Array) return Internal_Expected_Type_For_Expr_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Expected_Type_For_Expr_Array_Type;
      end if;
      return new Internal_Expected_Type_For_Expr_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Expected_Type_For_Expr_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Expected_Type_For_Expr_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Inner_Env_Assoc_Vectors is new Langkit_Support.Vectors (Internal_Inner_Env_Assoc);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Inner_Env_Assoc_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Inner_Env_Assoc
   is
      function Absolute_Get
        (T : Internal_Inner_Env_Assoc_Array_Access; Index : Integer)
         return Internal_Inner_Env_Assoc
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Inner_Env_Assoc,
         Sequence_Type => Internal_Inner_Env_Assoc_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Inner_Env_Assoc;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Inner_Env_Assoc;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Inner_Env_Assoc_Array_Access) return Internal_Inner_Env_Assoc_Array_Access is
      Ret : Internal_Inner_Env_Assoc_Array_Access := Create_Internal_Inner_Env_Assoc_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Inner_Env_Assoc_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Inner_Env_Assoc_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Inner_Env_Assoc_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Inner_Env_Assoc_Array (Items_Count : Natural) return Internal_Inner_Env_Assoc_Array_Access
   is (if Items_Count = 0
       then No_Internal_Inner_Env_Assoc_Array_Type
       else new Internal_Inner_Env_Assoc_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Inner_Env_Assoc_Array
     (Items : Internal_Internal_Inner_Env_Assoc_Array) return Internal_Inner_Env_Assoc_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Inner_Env_Assoc_Array_Type;
      end if;
      return new Internal_Inner_Env_Assoc_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Inner_Env_Assoc_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Inner_Env_Assoc_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Param_Actual_Vectors is new Langkit_Support.Vectors (Internal_Param_Actual);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Param_Actual_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Param_Actual
   is
      function Absolute_Get
        (T : Internal_Param_Actual_Array_Access; Index : Integer)
         return Internal_Param_Actual
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Param_Actual,
         Sequence_Type => Internal_Param_Actual_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Param_Actual;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Param_Actual;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Param_Actual_Array_Access) return Internal_Param_Actual_Array_Access is
      Ret : Internal_Param_Actual_Array_Access := Create_Internal_Param_Actual_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Param_Actual_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Param_Actual_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Param_Actual_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Param_Actual_Array (Items_Count : Natural) return Internal_Param_Actual_Array_Access
   is (if Items_Count = 0
       then No_Internal_Param_Actual_Array_Type
       else new Internal_Param_Actual_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Param_Actual_Array
     (Items : Internal_Internal_Param_Actual_Array) return Internal_Param_Actual_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Param_Actual_Array_Type;
      end if;
      return new Internal_Param_Actual_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Param_Actual_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Param_Actual_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Param_Match_Vectors is new Langkit_Support.Vectors (Internal_Param_Match);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Param_Match_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Param_Match
   is
      function Absolute_Get
        (T : Internal_Param_Match_Array_Access; Index : Integer)
         return Internal_Param_Match
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Param_Match,
         Sequence_Type => Internal_Param_Match_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Param_Match;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Param_Match;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Param_Match_Array_Access) return Internal_Param_Match_Array_Access is
      Ret : Internal_Param_Match_Array_Access := Create_Internal_Param_Match_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Param_Match_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Param_Match_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Param_Match_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Param_Match_Array (Items_Count : Natural) return Internal_Param_Match_Array_Access
   is (if Items_Count = 0
       then No_Internal_Param_Match_Array_Type
       else new Internal_Param_Match_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Param_Match_Array
     (Items : Internal_Internal_Param_Match_Array) return Internal_Param_Match_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Param_Match_Array_Type;
      end if;
      return new Internal_Param_Match_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Param_Match_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Param_Match_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Ref_Result_Vectors is new Langkit_Support.Vectors (Internal_Ref_Result);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Ref_Result_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Ref_Result
   is
      function Absolute_Get
        (T : Internal_Ref_Result_Array_Access; Index : Integer)
         return Internal_Ref_Result
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Ref_Result,
         Sequence_Type => Internal_Ref_Result_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Ref_Result;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Ref_Result;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Ref_Result_Array_Access) return Internal_Ref_Result_Array_Access is
      Ret : Internal_Ref_Result_Array_Access := Create_Internal_Ref_Result_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Ref_Result_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Ref_Result_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Ref_Result_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Ref_Result_Array (Items_Count : Natural) return Internal_Ref_Result_Array_Access
   is (if Items_Count = 0
       then No_Internal_Ref_Result_Array_Type
       else new Internal_Ref_Result_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Ref_Result_Array
     (Items : Internal_Internal_Ref_Result_Array) return Internal_Ref_Result_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Ref_Result_Array_Type;
      end if;
      return new Internal_Ref_Result_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Ref_Result_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Ref_Result_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Shape_Vectors is new Langkit_Support.Vectors (Internal_Shape);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Shape_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Shape
   is
      function Absolute_Get
        (T : Internal_Shape_Array_Access; Index : Integer)
         return Internal_Shape
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Shape,
         Sequence_Type => Internal_Shape_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Shape;
   begin
      if Relative_Get (T, Index, Result) then
            Inc_Ref (Result);
         return Result;
      elsif Or_Null then
         return No_Shape;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Shape_Array_Access) return Internal_Shape_Array_Access is
      Ret : Internal_Shape_Array_Access := Create_Internal_Shape_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
         for Item of Ret.Items loop
            Inc_Ref (Item);
         end loop;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Shape_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Shape_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Shape_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
            for Item of T.Items loop
               Dec_Ref (Item);
            end loop;
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Shape_Array (Items_Count : Natural) return Internal_Shape_Array_Access
   is (if Items_Count = 0
       then No_Internal_Shape_Array_Type
       else new Internal_Shape_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Shape_Array
     (Items : Internal_Internal_Shape_Array) return Internal_Shape_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Shape_Array_Type;
      end if;
         for El of Items loop
            Inc_Ref (El);
         end loop;
      return new Internal_Shape_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Shape_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               not Equivalent (L.Items (I), R.Items (I))
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Shape_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Single_Actual_Vectors is new Langkit_Support.Vectors (Internal_Single_Actual);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Single_Actual_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Single_Actual
   is
      function Absolute_Get
        (T : Internal_Single_Actual_Array_Access; Index : Integer)
         return Internal_Single_Actual
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Single_Actual,
         Sequence_Type => Internal_Single_Actual_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Single_Actual;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Single_Actual;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Single_Actual_Array_Access) return Internal_Single_Actual_Array_Access is
      Ret : Internal_Single_Actual_Array_Access := Create_Internal_Single_Actual_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Single_Actual_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Single_Actual_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Single_Actual_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Single_Actual_Array (Items_Count : Natural) return Internal_Single_Actual_Array_Access
   is (if Items_Count = 0
       then No_Internal_Single_Actual_Array_Type
       else new Internal_Single_Actual_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Single_Actual_Array
     (Items : Internal_Internal_Single_Actual_Array) return Internal_Single_Actual_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Single_Actual_Array_Type;
      end if;
      return new Internal_Single_Actual_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Single_Actual_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Single_Actual_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Substitution_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Substitution
   is
      function Absolute_Get
        (T : Internal_Substitution_Array_Access; Index : Integer)
         return Internal_Substitution
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Substitution,
         Sequence_Type => Internal_Substitution_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Substitution;
   begin
      if Relative_Get (T, Index, Result) then
            Inc_Ref (Result);
         return Result;
      elsif Or_Null then
         return No_Substitution;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Substitution_Array_Access) return Internal_Substitution_Array_Access is
      Ret : Internal_Substitution_Array_Access := Create_Internal_Substitution_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
         for Item of Ret.Items loop
            Inc_Ref (Item);
         end loop;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Substitution_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Substitution_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Substitution_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
            for Item of T.Items loop
               Dec_Ref (Item);
            end loop;
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Substitution_Array (Items_Count : Natural) return Internal_Substitution_Array_Access
   is (if Items_Count = 0
       then No_Internal_Substitution_Array_Type
       else new Internal_Substitution_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Substitution_Array
     (Items : Internal_Internal_Substitution_Array) return Internal_Substitution_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Substitution_Array_Type;
      end if;
         for El of Items loop
            Inc_Ref (El);
         end loop;
      return new Internal_Substitution_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Substitution_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               not Equivalent (L.Items (I), R.Items (I))
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Substitution_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Unit_Vectors is new Langkit_Support.Vectors (Internal_Unit);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Unit_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Unit
   is
      function Absolute_Get
        (T : Internal_Unit_Array_Access; Index : Integer)
         return Internal_Unit
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Unit,
         Sequence_Type => Internal_Unit_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Unit;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return null;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Unit_Array_Access) return Internal_Unit_Array_Access is
      Ret : Internal_Unit_Array_Access := Create_Internal_Unit_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Unit_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Unit_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Unit_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Unit_Array (Items_Count : Natural) return Internal_Unit_Array_Access
   is (if Items_Count = 0
       then No_Internal_Unit_Array_Type
       else new Internal_Unit_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Unit_Array
     (Items : Internal_Internal_Unit_Array) return Internal_Unit_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Unit_Array_Type;
      end if;
      return new Internal_Unit_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Unit_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Unit_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      function Make_Unique (A : Internal_Unit_Array_Access) return Internal_Unit_Array_Access is
         --  To keep the order of the result deterministic, this function works
         --  in two steps. First, determine for each item in A whether it
         --  duplicate a previous item. Only then, go through all items in A
         --  and add only non-duplicated ones to the result.
         Duplicates : array (A.Items'Range) of Boolean := (others => False);
         --  For each item in A, indicate whether the value is a duplicate of a
         --  previous one in the array.
         Unique_Elements : Natural;
         --  Number of unique elements in A
      begin
         --  Step 1: compute Duplicates
         declare
            package Sets is new Ada.Containers.Hashed_Sets
              (Element_Type        => Internal_Unit,
               "="                 => "=",
               Equivalent_Elements => "=",
               Hash                => Hash);
            Set : Sets.Set;
         begin
            for I in A.Items'Range loop
               declare
                  Item : Internal_Unit renames A.Items (I);
               begin
                  if Set.Contains (Item) then
                     Duplicates (I) := True;
                  else
                     Set.Insert (Item);
                  end if;
               end;
            end loop;
            Unique_Elements := Natural (Set.Length);
         end;
         --  Step 2: create the result
         declare
            Next   : Positive := 1;
            Result : constant Internal_Unit_Array_Access :=
               Create_Internal_Unit_Array (Unique_Elements);
         begin
            for I in A.Items'Range loop
               declare
                  Item : Internal_Unit renames A.Items (I);
               begin
                  if not Duplicates (I) then
                     Result.Items (Next) := Item;
                     Next := Next + 1;
                  end if;
               end;
            end loop;
            pragma Assert (Next = Result.N + 1);
            return Result;
         end;
      end Make_Unique;
      package Lexical_Env_Vectors is new Langkit_Support.Vectors (Lexical_Env);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Lexical_Env_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Lexical_Env
   is
      function Absolute_Get
        (T : Lexical_Env_Array_Access; Index : Integer)
         return Lexical_Env
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Lexical_Env,
         Sequence_Type => Lexical_Env_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Lexical_Env;
   begin
      if Relative_Get (T, Index, Result) then
            Inc_Ref (Result);
         return Result;
      elsif Or_Null then
         return Empty_Env;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Lexical_Env_Array_Access) return Lexical_Env_Array_Access is
      Ret : Lexical_Env_Array_Access := Create_Lexical_Env_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
         for Item of Ret.Items loop
            Inc_Ref (Item);
         end loop;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Lexical_Env_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Lexical_Env_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Lexical_Env_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
            for Item of T.Items loop
               Dec_Ref (Item);
            end loop;
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Lexical_Env_Array (Items_Count : Natural) return Lexical_Env_Array_Access
   is (if Items_Count = 0
       then No_Lexical_Env_Array_Type
       else new Lexical_Env_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Lexical_Env_Array
     (Items : Internal_Lexical_Env_Array) return Lexical_Env_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Lexical_Env_Array_Type;
      end if;
         for El of Items loop
            Inc_Ref (El);
         end loop;
      return new Lexical_Env_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Lexical_Env_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               not Equivalent (L.Items (I), R.Items (I))
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Lexical_Env_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Logic_Equation_Vectors is new Langkit_Support.Vectors (Logic_Equation);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Logic_Equation_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Logic_Equation
   is
      function Absolute_Get
        (T : Logic_Equation_Array_Access; Index : Integer)
         return Logic_Equation
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Logic_Equation,
         Sequence_Type => Logic_Equation_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Logic_Equation;
   begin
      if Relative_Get (T, Index, Result) then
            Inc_Ref (Result);
         return Result;
      elsif Or_Null then
         return Null_Logic_Equation;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Logic_Equation_Array_Access) return Logic_Equation_Array_Access is
      Ret : Logic_Equation_Array_Access := Create_Logic_Equation_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
         for Item of Ret.Items loop
            Inc_Ref (Item);
         end loop;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Logic_Equation_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Logic_Equation_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Logic_Equation_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
            for Item of T.Items loop
               Dec_Ref (Item);
            end loop;
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Logic_Equation_Array (Items_Count : Natural) return Logic_Equation_Array_Access
   is (if Items_Count = 0
       then No_Logic_Equation_Array_Type
       else new Logic_Equation_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Logic_Equation_Array
     (Items : Internal_Logic_Equation_Array) return Logic_Equation_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Logic_Equation_Array_Type;
      end if;
         for El of Items loop
            Inc_Ref (El);
         end loop;
      return new Logic_Equation_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Logic_Equation_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Logic_Equation_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package String_Type_Vectors is new Langkit_Support.Vectors (String_Type);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : String_Type_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return String_Type
   is
      function Absolute_Get
        (T : String_Type_Array_Access; Index : Integer)
         return String_Type
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => String_Type,
         Sequence_Type => String_Type_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : String_Type;
   begin
      if Relative_Get (T, Index, Result) then
            Inc_Ref (Result);
         return Result;
      elsif Or_Null then
         return Empty_String;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : String_Type_Array_Access) return String_Type_Array_Access is
      Ret : String_Type_Array_Access := Create_String_Type_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
         for Item of Ret.Items loop
            Inc_Ref (Item);
         end loop;
      return Ret;
   end Concat;
      ------------------
      -- Join_Strings --
      ------------------
      function Join_Strings
        (Separator : String_Type;
         Strings   : String_Type_Array_Access) return String_Type
      is
         Separator_Length : constant Natural := Separator.Length;
         Length           : Natural := 0;
         First            : Boolean;
      begin
         --  First, compute the length of the result: the sum of all string
         --  lengths in Strings.
         First := True;
         for S of Strings.Items loop
            if First then
               First := False;
            else
               Length := Length + Separator_Length;
            end if;
            Length := Length + S.Length;
         end loop;
         --  Create the result string with the correct length. Do not use our
         --  constructor and initialize the content in-place, to avoid extra
         --  copies.
         return Result : constant String_Type :=
            new String_Record (Length)
         do
            Result.Ref_Count := 1;
            --  Now copy the content of all strings into the result
            declare
               Last : Natural := 0;
            begin
               First := True;
               for S of Strings.Items loop
                  if First then
                     First := False;
                  else
                     Result.Content (Last + 1 .. Last + Separator_Length) :=
                        Separator.Content;
                     Last := Last + Separator_Length;
                  end if;
                  Result.Content (Last + 1 .. Last + S.Length) := S.Content;
                  Last := Last + S.Length;
               end loop;
            end;
         end return;
      end Join_Strings;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : String_Type_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : String_Type_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out String_Type_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
            for Item of T.Items loop
               Dec_Ref (Item);
            end loop;
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_String_Type_Array (Items_Count : Natural) return String_Type_Array_Access
   is (if Items_Count = 0
       then No_String_Type_Array_Type
       else new String_Type_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_String_Type_Array
     (Items : Internal_String_Type_Array) return String_Type_Array_Access is
   begin
      if Items'Length = 0 then
         return No_String_Type_Array_Type;
      end if;
         for El of Items loop
            Inc_Ref (El);
         end loop;
      return new String_Type_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : String_Type_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               not Equivalent (L.Items (I), R.Items (I))
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : String_Type_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Symbol_Type_Vectors is new Langkit_Support.Vectors (Symbol_Type);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Symbol_Type_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Symbol_Type
   is
      function Absolute_Get
        (T : Symbol_Type_Array_Access; Index : Integer)
         return Symbol_Type
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Symbol_Type,
         Sequence_Type => Symbol_Type_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Symbol_Type;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return null;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Symbol_Type_Array_Access) return Symbol_Type_Array_Access is
      Ret : Symbol_Type_Array_Access := Create_Symbol_Type_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Symbol_Type_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Symbol_Type_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Symbol_Type_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Symbol_Type_Array (Items_Count : Natural) return Symbol_Type_Array_Access
   is (if Items_Count = 0
       then No_Symbol_Type_Array_Type
       else new Symbol_Type_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Symbol_Type_Array
     (Items : Internal_Symbol_Type_Array) return Symbol_Type_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Symbol_Type_Array_Type;
      end if;
      return new Symbol_Type_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Symbol_Type_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Symbol_Type_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
   ----------
   -- Next --
   ----------
   function Next
     (T       : Bare_Ada_Node_Iterator_Access;
      Element : out Bare_Ada_Node) return Boolean is
   begin
      if T = null then
         raise Property_Error with "null access dereference";
      end if;
      Check_Safety_Net (T.Safety_Net);
      if T.Index > T.Elements.Items'Last then
         return False;
      else
         Element := T.Elements.Items (T.Index);
         T.Index := T.Index + 1;
         return True;
      end if;
   end Next;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Bare_Ada_Node_Iterator_Access) is
   begin
      if T /= null and then T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Bare_Ada_Node_Iterator_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Dec_Ref (T.Elements);
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Bare_Ada_Node_Iterator_Access) return String is
      begin
         return "<Iterator of AdaNode, index="
                & A.Index'Image & ">";
      end Trace_Image;
   ----------
   -- Next --
   ----------
   function Next
     (T       : Internal_Completion_Item_Iterator_Access;
      Element : out Internal_Completion_Item) return Boolean is
   begin
      if T = null then
         raise Property_Error with "null access dereference";
      end if;
      Check_Safety_Net (T.Safety_Net);
      if T.Index > T.Elements.Items'Last then
         return False;
      else
         Element := T.Elements.Items (T.Index);
         T.Index := T.Index + 1;
         return True;
      end if;
   end Next;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Completion_Item_Iterator_Access) is
   begin
      if T /= null and then T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Completion_Item_Iterator_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Dec_Ref (T.Elements);
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Completion_Item_Iterator_Access) return String is
      begin
         return "<Iterator of CompletionItem, index="
                & A.Index'Image & ">";
      end Trace_Image;
   ----------
   -- Next --
   ----------
   function Next
     (T       : Internal_Entity_Iterator_Access;
      Element : out Internal_Entity) return Boolean is
   begin
      if T = null then
         raise Property_Error with "null access dereference";
      end if;
      Check_Safety_Net (T.Safety_Net);
      if T.Index > T.Elements.Items'Last then
         return False;
      else
         Element := T.Elements.Items (T.Index);
         T.Index := T.Index + 1;
         return True;
      end if;
   end Next;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Iterator_Access) is
   begin
      if T /= null and then T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Iterator_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Dec_Ref (T.Elements);
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Iterator_Access) return String is
      begin
         return "<Iterator of AdaNode.entity, index="
                & A.Index'Image & ">";
      end Trace_Image;
   ----------
   -- Next --
   ----------
   function Next
     (T       : Internal_Inner_Env_Assoc_Iterator_Access;
      Element : out Internal_Inner_Env_Assoc) return Boolean is
   begin
      if T = null then
         raise Property_Error with "null access dereference";
      end if;
      Check_Safety_Net (T.Safety_Net);
      if T.Index > T.Elements.Items'Last then
         return False;
      else
         Element := T.Elements.Items (T.Index);
         T.Index := T.Index + 1;
         return True;
      end if;
   end Next;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Inner_Env_Assoc_Iterator_Access) is
   begin
      if T /= null and then T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Inner_Env_Assoc_Iterator_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Dec_Ref (T.Elements);
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Inner_Env_Assoc_Iterator_Access) return String is
      begin
         return "<Iterator of InnerEnvAssoc, index="
                & A.Index'Image & ">";
      end Trace_Image;
   ---------
   -- "<" --
   ---------
   function "<" (Left, Right : Internal_Unit) return Boolean is
   begin
      return Left.Filename < Right.Filename;
   end "<";
function Hash (Key : Mmz_Key_Item) return Hash_Type;
function Equivalent (L, R : Mmz_Key_Item) return Boolean;
procedure Destroy (Key : in out Mmz_Key_Array_Access);
----------------
-- Equivalent --
----------------
function Equivalent (L, R : Mmz_Key_Item) return Boolean is
begin
   if L.Kind /= R.Kind then
      return False;
   end if;
   case L.Kind is
         when Mmz_Bare_Access_Def =>
               return L.As_Bare_Access_Def = R.As_Bare_Access_Def;
         when Mmz_Bare_Ada_Node =>
               return L.As_Bare_Ada_Node = R.As_Bare_Ada_Node;
         when Mmz_Bare_Anonymous_Expr_Decl =>
               return L.As_Bare_Anonymous_Expr_Decl = R.As_Bare_Anonymous_Expr_Decl;
         when Mmz_Bare_Array_Type_Def =>
               return L.As_Bare_Array_Type_Def = R.As_Bare_Array_Type_Def;
         when Mmz_Bare_Assoc_List =>
               return L.As_Bare_Assoc_List = R.As_Bare_Assoc_List;
         when Mmz_Bare_Base_Aggregate =>
               return L.As_Bare_Base_Aggregate = R.As_Bare_Base_Aggregate;
         when Mmz_Bare_Base_Id =>
               return L.As_Bare_Base_Id = R.As_Bare_Base_Id;
         when Mmz_Bare_Base_Subp_Spec =>
               return L.As_Bare_Base_Subp_Spec = R.As_Bare_Base_Subp_Spec;
         when Mmz_Bare_Base_Type_Decl =>
               return L.As_Bare_Base_Type_Decl = R.As_Bare_Base_Type_Decl;
         when Mmz_Bare_Basic_Decl =>
               return L.As_Bare_Basic_Decl = R.As_Bare_Basic_Decl;
         when Mmz_Bare_Body_Node =>
               return L.As_Bare_Body_Node = R.As_Bare_Body_Node;
         when Mmz_Bare_Classwide_Type_Decl =>
               return L.As_Bare_Classwide_Type_Decl = R.As_Bare_Classwide_Type_Decl;
         when Mmz_Bare_Compilation_Unit =>
               return L.As_Bare_Compilation_Unit = R.As_Bare_Compilation_Unit;
         when Mmz_Bare_Decimal_Fixed_Point_Def =>
               return L.As_Bare_Decimal_Fixed_Point_Def = R.As_Bare_Decimal_Fixed_Point_Def;
         when Mmz_Bare_Defining_Name =>
               return L.As_Bare_Defining_Name = R.As_Bare_Defining_Name;
         when Mmz_Bare_End_Name =>
               return L.As_Bare_End_Name = R.As_Bare_End_Name;
         when Mmz_Bare_Enum_Literal_Decl =>
               return L.As_Bare_Enum_Literal_Decl = R.As_Bare_Enum_Literal_Decl;
         when Mmz_Bare_Enum_Type_Def =>
               return L.As_Bare_Enum_Type_Def = R.As_Bare_Enum_Type_Def;
         when Mmz_Bare_For_Loop_Spec =>
               return L.As_Bare_For_Loop_Spec = R.As_Bare_For_Loop_Spec;
         when Mmz_Bare_For_Loop_Var_Decl =>
               return L.As_Bare_For_Loop_Var_Decl = R.As_Bare_For_Loop_Var_Decl;
         when Mmz_Bare_Formal_Discrete_Type_Def =>
               return L.As_Bare_Formal_Discrete_Type_Def = R.As_Bare_Formal_Discrete_Type_Def;
         when Mmz_Bare_Generic_Instantiation =>
               return L.As_Bare_Generic_Instantiation = R.As_Bare_Generic_Instantiation;
         when Mmz_Bare_Mod_Int_Type_Def =>
               return L.As_Bare_Mod_Int_Type_Def = R.As_Bare_Mod_Int_Type_Def;
         when Mmz_Bare_Name =>
               return L.As_Bare_Name = R.As_Bare_Name;
         when Mmz_Bare_Ordinary_Fixed_Point_Def =>
               return L.As_Bare_Ordinary_Fixed_Point_Def = R.As_Bare_Ordinary_Fixed_Point_Def;
         when Mmz_Bare_Private_Type_Def =>
               return L.As_Bare_Private_Type_Def = R.As_Bare_Private_Type_Def;
         when Mmz_Bare_Real_Type_Def =>
               return L.As_Bare_Real_Type_Def = R.As_Bare_Real_Type_Def;
         when Mmz_Bare_Record_Type_Def =>
               return L.As_Bare_Record_Type_Def = R.As_Bare_Record_Type_Def;
         when Mmz_Bare_Signed_Int_Type_Def =>
               return L.As_Bare_Signed_Int_Type_Def = R.As_Bare_Signed_Int_Type_Def;
         when Mmz_Bare_Type_Decl =>
               return L.As_Bare_Type_Decl = R.As_Bare_Type_Decl;
         when Mmz_Bare_Type_Expr =>
               return L.As_Bare_Type_Expr = R.As_Bare_Type_Expr;
         when Mmz_Boolean =>
               return L.As_Boolean = R.As_Boolean;
         when Mmz_Env_Rebindings =>
               return L.As_Env_Rebindings = R.As_Env_Rebindings;
         when Mmz_Integer =>
               return L.As_Integer = R.As_Integer;
         when Mmz_Internal_Entity_Base_Type_Decl_Array_Access =>
               return Equivalent (L.As_Internal_Entity_Base_Type_Decl_Array_Access, R.As_Internal_Entity_Base_Type_Decl_Array_Access);
         when Mmz_Internal_Entity_Info =>
               return L.As_Internal_Entity_Info = R.As_Internal_Entity_Info;
         when Mmz_Internal_Metadata =>
               return L.As_Internal_Metadata = R.As_Internal_Metadata;
         when Mmz_Lexical_Env =>
               return Equivalent (L.As_Lexical_Env, R.As_Lexical_Env);
         when Mmz_Symbol_Type =>
               return L.As_Symbol_Type = R.As_Symbol_Type;
   end case;
end Equivalent;
----------
-- Hash --
----------
function Hash (Key : Mmz_Key_Item) return Hash_Type is
begin
   case Key.Kind is
         when Mmz_Bare_Access_Def =>
            return Hash (Key.As_Bare_Access_Def);
         when Mmz_Bare_Ada_Node =>
            return Hash (Key.As_Bare_Ada_Node);
         when Mmz_Bare_Anonymous_Expr_Decl =>
            return Hash (Key.As_Bare_Anonymous_Expr_Decl);
         when Mmz_Bare_Array_Type_Def =>
            return Hash (Key.As_Bare_Array_Type_Def);
         when Mmz_Bare_Assoc_List =>
            return Hash (Key.As_Bare_Assoc_List);
         when Mmz_Bare_Base_Aggregate =>
            return Hash (Key.As_Bare_Base_Aggregate);
         when Mmz_Bare_Base_Id =>
            return Hash (Key.As_Bare_Base_Id);
         when Mmz_Bare_Base_Subp_Spec =>
            return Hash (Key.As_Bare_Base_Subp_Spec);
         when Mmz_Bare_Base_Type_Decl =>
            return Hash (Key.As_Bare_Base_Type_Decl);
         when Mmz_Bare_Basic_Decl =>
            return Hash (Key.As_Bare_Basic_Decl);
         when Mmz_Bare_Body_Node =>
            return Hash (Key.As_Bare_Body_Node);
         when Mmz_Bare_Classwide_Type_Decl =>
            return Hash (Key.As_Bare_Classwide_Type_Decl);
         when Mmz_Bare_Compilation_Unit =>
            return Hash (Key.As_Bare_Compilation_Unit);
         when Mmz_Bare_Decimal_Fixed_Point_Def =>
            return Hash (Key.As_Bare_Decimal_Fixed_Point_Def);
         when Mmz_Bare_Defining_Name =>
            return Hash (Key.As_Bare_Defining_Name);
         when Mmz_Bare_End_Name =>
            return Hash (Key.As_Bare_End_Name);
         when Mmz_Bare_Enum_Literal_Decl =>
            return Hash (Key.As_Bare_Enum_Literal_Decl);
         when Mmz_Bare_Enum_Type_Def =>
            return Hash (Key.As_Bare_Enum_Type_Def);
         when Mmz_Bare_For_Loop_Spec =>
            return Hash (Key.As_Bare_For_Loop_Spec);
         when Mmz_Bare_For_Loop_Var_Decl =>
            return Hash (Key.As_Bare_For_Loop_Var_Decl);
         when Mmz_Bare_Formal_Discrete_Type_Def =>
            return Hash (Key.As_Bare_Formal_Discrete_Type_Def);
         when Mmz_Bare_Generic_Instantiation =>
            return Hash (Key.As_Bare_Generic_Instantiation);
         when Mmz_Bare_Mod_Int_Type_Def =>
            return Hash (Key.As_Bare_Mod_Int_Type_Def);
         when Mmz_Bare_Name =>
            return Hash (Key.As_Bare_Name);
         when Mmz_Bare_Ordinary_Fixed_Point_Def =>
            return Hash (Key.As_Bare_Ordinary_Fixed_Point_Def);
         when Mmz_Bare_Private_Type_Def =>
            return Hash (Key.As_Bare_Private_Type_Def);
         when Mmz_Bare_Real_Type_Def =>
            return Hash (Key.As_Bare_Real_Type_Def);
         when Mmz_Bare_Record_Type_Def =>
            return Hash (Key.As_Bare_Record_Type_Def);
         when Mmz_Bare_Signed_Int_Type_Def =>
            return Hash (Key.As_Bare_Signed_Int_Type_Def);
         when Mmz_Bare_Type_Decl =>
            return Hash (Key.As_Bare_Type_Decl);
         when Mmz_Bare_Type_Expr =>
            return Hash (Key.As_Bare_Type_Expr);
         when Mmz_Boolean =>
            return Hash (Key.As_Boolean);
         when Mmz_Env_Rebindings =>
            return Hash (Key.As_Env_Rebindings);
         when Mmz_Integer =>
            return Hash (Key.As_Integer);
         when Mmz_Internal_Entity_Base_Type_Decl_Array_Access =>
            return Hash (Key.As_Internal_Entity_Base_Type_Decl_Array_Access);
         when Mmz_Internal_Entity_Info =>
            return Hash (Key.As_Internal_Entity_Info);
         when Mmz_Internal_Metadata =>
            return Hash (Key.As_Internal_Metadata);
         when Mmz_Lexical_Env =>
            return Hash (Key.As_Lexical_Env);
         when Mmz_Symbol_Type =>
            return Hash (Key.As_Symbol_Type);
   end case;
end Hash;
----------
-- Hash --
----------
function Hash (Key : Mmz_Key) return Hash_Type is
   Result : Hash_Type := Mmz_Property'Pos (Key.Property);
begin
   for K of Key.Items.all loop
      Result := Combine (Result, Hash (K));
   end loop;
   return Result;
end Hash;
----------------
-- Equivalent --
----------------
function Equivalent (L, R : Mmz_Key) return Boolean is
   L_Items : Mmz_Key_Array renames L.Items.all;
   R_Items : Mmz_Key_Array renames R.Items.all;
begin
   if L.Property /= R.Property or else L_Items'Length /= R_Items'Length then
      return False;
   end if;
   for I in L_Items'Range loop
      if not Equivalent (L_Items (I), R_Items (I)) then
         return False;
      end if;
   end loop;
   return True;
end Equivalent;
-------------
-- Destroy --
-------------
procedure Destroy (Map : in out Memoization_Maps.Map) is
   use Memoization_Maps;
   --  We need keys and values to be valid when clearing the memoization map,
   --  but on the other hand we need to free keys and values as well. To
   --  achieve both goals, we first copy key and values into arrays, then we
   --  clear the map, and then we free keys/values in the arrays. Allocate both
   --  arrays on the heap to avoid stack overflow, as they can be quite big.
   Length : constant Natural := Natural (Map.Length);
   type Key_Array is array (1 .. Length) of Mmz_Key_Array_Access;
   type Key_Array_Access is access Key_Array;
   procedure Free is new Ada.Unchecked_Deallocation
     (Key_Array, Key_Array_Access);
   type Value_Array is array (1 .. Length) of Mmz_Value;
   type Value_Array_Access is access Value_Array;
   procedure Free is new Ada.Unchecked_Deallocation
     (Value_Array, Value_Array_Access);
   Keys   : Key_Array_Access := new Key_Array;
   Values : Value_Array_Access := new Value_Array;
   I      : Positive := 1;
begin
   for Cur in Map.Iterate loop
      Keys (I) := Key (Cur).Items;
      Values (I) := Element (Cur);
      I := I + 1;
   end loop;
   Map.Clear;
   for K_Array of Keys.all loop
      Destroy (K_Array);
   end loop;
      for V of Values.all loop
         case V.Kind is
               when Mmz_Internal_Entity_Array_Access =>
                  Dec_Ref (V.As_Internal_Entity_Array_Access);
               when Mmz_Internal_Entity_Base_Formal_Param_Decl_Array_Access =>
                  Dec_Ref (V.As_Internal_Entity_Base_Formal_Param_Decl_Array_Access);
               when Mmz_Internal_Entity_Base_Type_Decl_Array_Access =>
                  Dec_Ref (V.As_Internal_Entity_Base_Type_Decl_Array_Access);
               when Mmz_Internal_Entity_Basic_Decl_Array_Access =>
                  Dec_Ref (V.As_Internal_Entity_Basic_Decl_Array_Access);
               when Mmz_Internal_Entity_Compilation_Unit_Array_Access =>
                  Dec_Ref (V.As_Internal_Entity_Compilation_Unit_Array_Access);
               when Mmz_Internal_Env_Assoc =>
                  Dec_Ref (V.As_Internal_Env_Assoc);
               when Mmz_Internal_Env_Assoc_Array_Access =>
                  Dec_Ref (V.As_Internal_Env_Assoc_Array_Access);
               when Mmz_Internal_Inner_Env_Assoc_Array_Access =>
                  Dec_Ref (V.As_Internal_Inner_Env_Assoc_Array_Access);
               when Mmz_Internal_Param_Match_Array_Access =>
                  Dec_Ref (V.As_Internal_Param_Match_Array_Access);
               when Mmz_Internal_Single_Actual_Array_Access =>
                  Dec_Ref (V.As_Internal_Single_Actual_Array_Access);
               when Mmz_Lexical_Env =>
                  Dec_Ref (V.As_Lexical_Env);
            when others => null;
         end case;
      end loop;
   Free (Keys);
   Free (Values);
end Destroy;
-------------
-- Destroy --
-------------
procedure Destroy (Key : in out Mmz_Key_Array_Access) is
   procedure Free is new Ada.Unchecked_Deallocation
     (Mmz_Key_Array, Mmz_Key_Array_Access);
begin
      for K of Key.all loop
         case K.Kind is
               when Mmz_Internal_Entity_Base_Type_Decl_Array_Access =>
                  Dec_Ref (K.As_Internal_Entity_Base_Type_Decl_Array_Access);
               when Mmz_Lexical_Env =>
                  Dec_Ref (K.As_Lexical_Env);
            when others => null;
         end case;
      end loop;
   Free (Key);
end Destroy;
-------------------------
-- Find_Memoized_Value --
-------------------------
function Find_Memoized_Value
  (Unit       : Internal_Unit;
   Handle     : out Memoization_Handle;
   Value      : out Mmz_Value;
   Create_Key : access function return Mmz_Key) return Boolean
is
   Inserted : Boolean;
begin
   --  Make sure that we don't lookup stale caches
   Reset_Caches (Unit);
   --  Initialize handle: create the key and create a cursor pointing to an
   --  existing entry.
   Handle.Key := Create_Key.all;
   Handle.Cache_Version := Unit.Cache_Version;
   Value := (Kind => Mmz_Evaluating);
   Unit.Memoization_Map.Insert (Handle.Key, Value, Handle.Cur, Inserted);
   --  No existing entry yet? The above just created one. Otherwise, destroy
   --  our key and reuse the existing entry's.
   if not Inserted then
      Destroy (Handle.Key.Items);
      Handle.Key := Memoization_Maps.Key (Handle.Cur);
      Value := Memoization_Maps.Element (Handle.Cur);
   end if;
   return not Inserted;
end Find_Memoized_Value;
------------------------
-- Add_Memoized_Value --
------------------------
procedure Add_Memoized_Value
  (Unit   : Internal_Unit;
   Handle : in out Memoization_Handle;
   Value  : Mmz_Value;
   Stored : out Boolean) is
begin
   --  If Handle was created using a memoization map that has been since then
   --  reset, do nothing: the result can be partly stale due to the event that
   --  triggered the memoization tables reset.
   Stored := Unit.Cache_Version <= Handle.Cache_Version;
   if Stored then
      Unit.Memoization_Map.Replace_Element (Handle.Cur, Value);
   end if;
end Add_Memoized_Value;
   -------------------
   -- Solve_Wrapper --
   -------------------
   function Solve_Wrapper
     (R            : Solver.Relation;
      Context_Node : Bare_Ada_Node) return Boolean is
   begin
      if Context_Node /= null and then Langkit_Support.Adalog.Debug.Debug then
         Assign_Names_To_Logic_Vars (Context_Node);
      end if;
      begin
         return Solver.Solve_First
           (R, Timeout => Context_Node.Unit.Context.Logic_Resolution_Timeout);
      exception
         when Langkit_Support.Adalog.Early_Binding_Error =>
            Raise_Property_Exception
              (Context_Node,
               Property_Error'Identity,
               "invalid equation for logic resolution");
         when Langkit_Support.Adalog.Timeout_Error =>
            Raise_Property_Exception
              (Context_Node,
               Property_Error'Identity,
               "logic resolution timed out");
      end;
   end Solve_Wrapper;
   -------------
   -- Destroy --
   -------------
   procedure Destroy (Env : in out Lexical_Env_Access) is
      Mutable_Env : Lexical_Env :=
        (Wrap (Env), 0, Env.Kind, No_Generic_Unit, 0);
   begin
      Destroy (Mutable_Env);
      Env := null;
   end Destroy;
   ----------------
   -- Initialize --
   ----------------
   procedure Initialize
     (Self              : Bare_Ada_Node;
      Kind              : Ada_Node_Kind_Type;
      Unit              : Internal_Unit;
      Token_Start_Index : Token_Index;
      Token_End_Index   : Token_Index;
      Parent            : Bare_Ada_Node := null;
      Self_Env          : Lexical_Env := AST_Envs.Empty_Env) is
   begin
      pragma Unreferenced (Kind);
      Self.Parent := Parent;
      Self.Unit := Unit;
      Self.Token_Start_Index := Token_Start_Index;
      Self.Token_End_Index := Token_End_Index;
      Self.Self_Env := Self_Env;
      Self.Last_Attempted_Child := -1;
   end Initialize;
   --------------------
   -- Use_Direct_Env --
   --------------------
   procedure Use_Direct_Env (State : in out PLE_Node_State; Env : Lexical_Env)
   is
   begin
      State.Current_Env := Env;
      State.Current_NED := null;
   end Use_Direct_Env;
   -------------------
   -- Use_Named_Env --
   -------------------
   procedure Use_Named_Env
     (State   : in out PLE_Node_State;
      Context : Internal_Context;
      Name    : Symbol_Type) is
   begin
      State.Current_NED := Get_Named_Env_Descriptor (Context, Name);
      State.Current_Env := State.Current_NED.Env_With_Precedence;
   end Use_Named_Env;
   ---------------------
   -- Set_Initial_Env --
   ---------------------
   procedure Set_Initial_Env
     (Self         : Bare_Ada_Node;
      State        : in out PLE_Node_State;
      Env          : Internal_Designated_Env;
      DSL_Location : String) is
   begin
      case Env.Kind is
         when None =>
            Use_Direct_Env (State, Empty_Env);
         when Current_Env =>
            null;
         when Named_Env =>
            Use_Named_Env (State, Self.Unit.Context, Env.Env_Name);
         when Direct_Env =>
            --  Sanitize this environment value: make sure it's a non-foreign
            --  and primary environment.
            if Env.Direct_Env.Kind /= Static_Primary then
               Raise_Property_Exception
                 (Self,
                  Property_Error'Identity,
                  "Cannot set an env that is not static-primary as the"
                  & " initial env");
            elsif Is_Foreign_Strict (Env.Direct_Env, Self) then
               Raise_Property_Exception
                 (Self,
                  Property_Error'Identity,
                  "unsound foreign environment in SetInitialEnv ("
                  & DSL_Location & ")");
            end if;
            Use_Direct_Env (State, Env.Direct_Env);
      end case;
   end Set_Initial_Env;
   ----------------
   -- Add_To_Env --
   ----------------
   procedure Add_To_Env
     (Self         : Bare_Ada_Node;
      State        : PLE_Node_State;
      Key          : Symbol_Type;
      Value        : Bare_Ada_Node;
      Md           : Internal_Metadata;
      Resolver     : Entity_Resolver;
      Dest_Env     : Internal_Designated_Env;
      DSL_Location : String)
   is
      Context    : constant Internal_Context := Self.Unit.Context;
      Root_Scope : Lexical_Env renames Context.Root_Scope;
      --  Shortcuts
      Actual_Dest_Env : Lexical_Env;
      Dest_NED        : Named_Env_Descriptor_Access;
      --  Description for the destination environment
   begin
      --  Skip the env addition if explicitly requested
      if Key = null
         or else Value = null
         or else (case Dest_Env.Kind is
                  when None        => True,
                  when Current_Env => False,
                  when Named_Env   => Dest_Env.Env_Name = null,
                  when Direct_Env  => Dest_Env.Direct_Env = Empty_Env)
      then
         return;
      end if;
      if Value.Unit /= Self.Unit then
         Raise_Property_Exception
           (Self,
            Property_Error'Identity,
            "Cannot add_to_env an AST node that comes from another analysis"
            & " unit");
      end if;
         --  Make sure metadata does not contain any foreign node
         if (Md.Primitive /= null and then Md.Primitive.Unit /= Self.Unit) or else (Md.Primitive_Real_Type /= null and then Md.Primitive_Real_Type.Unit /= Self.Unit)
         then
            Raise_Property_Exception
              (Self,
               Property_Error'Identity,
               "Cannot add metadata that contains foreign nodes");
         end if;
      --  Then determine the destination environment
      case Dest_Env.Kind is
         when None =>
            raise Program_Error with "unreachable code";
         when Current_Env =>
            --  Just use the current environment
            Dest_NED := State.Current_NED;
            Actual_Dest_Env := State.Current_Env;
         when Named_Env =>
            --  There is an environment name: just lookup the corresponding
            --  NED/env.
            Dest_NED := Get_Named_Env_Descriptor (Context, Dest_Env.Env_Name);
            Actual_Dest_Env := Dest_NED.Env_With_Precedence;
         when Direct_Env =>
            --  There is an explicit destination environment
            Dest_NED := null;
            Actual_Dest_Env := Dest_Env.Direct_Env;
      end case;
      --  Sanitize it
      if Actual_Dest_Env.Kind /= Static_Primary then
         Raise_Property_Exception
           (Self,
            Property_Error'Identity,
            "Cannot add elements to a lexical env that is not static-primary");
      elsif
         --  Since lexical envs need to sort the foreign nodes they contain,
         --  and that the total order on nodes is not defined for synthetic
         --  nodes, it is not possible to add a synthetic node to a foreign
         --  lexical environment.
         --
         --  This reasoning applies to environments that belong to foreign
         --  units, but also to the root environment.
         Is_Foreign (Actual_Dest_Env, Self) and then Is_Synthetic (Value)
      then
         Raise_Property_Exception
           (Self,
            Property_Error'Identity,
            "Cannot add a synthetic node to a lexical env from another"
            & " analysis unit");
      elsif
         --  Reject direct references to foreign destination environments.
         --
         --  This is an attempt at identifying uses of the unsound relocation
         --  mechanism (as opposed to named environments), so this applies to
         --  all foreign environments (root scope included).
         DSL_Location'Length > 0
         and then Dest_Env.Kind = Direct_Env
         and then Is_Foreign_Strict (Actual_Dest_Env, Self)
      then
         Raise_Property_Exception
           (Self,
            Property_Error'Identity,
            "unsound foreign environment in AddToEnv (" & DSL_Location & ")");
      end if;
      --  Now that everything is sanitized, we can proceed with the actual
      --  key/value pair addition. Note that this does nothing if
      --  Actual_Dest_Env ended up empty.
      Add (Actual_Dest_Env, Key, Value, Md, Resolver);
      --  If we're adding the element to an environment by env name, we must
      --  register this association in two places: in the target named env
      --  entry, and in Value's unit.
      if Dest_NED /= null then
         declare
            use NED_Assoc_Maps;
            FN    : Map renames Dest_NED.Foreign_Nodes;
            Dummy : Boolean;
            Cur   : Cursor;
         begin
            FN.Insert (Key      => Key,
                       New_Item => Internal_Map_Node_Vectors.Empty_Vector,
                       Position => Cur,
                       Inserted => Dummy);
            declare
               V : Internal_Map_Node_Vectors.Vector renames
                  FN.Reference (Cur);
            begin
               V.Append ((Value, Md, Resolver));
            end;
         end;
         Value.Unit.Exiled_Entries_In_NED.Append ((Dest_NED, Key, Value));
      --  Otherwise, if we're adding the element to an environment that belongs
      --  to a different unit, or to the root scope, then...
      elsif Is_Foreign_Not_Empty (Actual_Dest_Env, Self) then
         --  Add the Key/Value association to the list of entries contained in
         --  other units, so we can remove them when reparsing Value's unit.
         Value.Unit.Exiled_Entries.Append ((Actual_Dest_Env, Key, Value));
         if Actual_Dest_Env /= Root_Scope then
            --  Add Val to the list of foreign nodes that Actual_Dest_Env's
            --  unit contains, so that when that unit is reparsed, we can call
            --  Add_To_Env again on those nodes.
            Convert_Unit (Actual_Dest_Env.Owner).Foreign_Nodes.Append
              ((Value, Self.Unit));
         end if;
      end if;
   end Add_To_Env;
   -------------
   -- Ref_Env --
   -------------
   procedure Ref_Env
     (Self                : Bare_Ada_Node;
      Dest_Env            : Lexical_Env;
      Ref_Env_Nodes       : in out Bare_Ada_Node_Array_Access;
      Resolver            : Lexical_Env_Resolver;
      Kind                : Ref_Kind;
      Cats                : Ref_Categories;
      Shed_Rebindings     : Boolean) is
   begin
      for N of Ref_Env_Nodes.Items loop
         if N /= null then
            if N.Unit /= Self.Unit then
               Raise_Property_Exception
                 (Self,
                  Property_Error'Identity,
                  "attempt to add a referenced environment to a foreign unit");
            end if;
            Reference (Dest_Env, N, Resolver, Kind, Cats, Shed_Rebindings);
         end if;
      end loop;
      Dec_Ref (Ref_Env_Nodes);
   end Ref_Env;
   -------------
   -- Add_Env --
   -------------
   procedure Add_Env
     (Self              : Bare_Ada_Node;
      State             : in out PLE_Node_State;
      No_Parent         : Boolean;
      Transitive_Parent : Boolean;
      Names             : in out Symbol_Type_Array_Access)
   is
      Parent_From_Name : constant Boolean := State.Current_NED /= null;
      --  Does the parent environment comes from a named environment lookup?
      --  Determine the parent of this new environment:
      --
      --  (1) no parent if requested;
      --  (2) the current environment as the static parent if it comes from a
      --      named env lookup or if it is not foreign (or is the empty/root
      --      environment).
      Parent : constant Lexical_Env :=
        (if No_Parent
         then Null_Lexical_Env
         else State.Current_Env);
   begin
      --  Create the environment itself
      Self.Self_Env := Create_Static_Lexical_Env
        (Parent            => Parent,
         Node              => Self,
         Transitive_Parent => Transitive_Parent);
      --  If the parent of this new environment comes from a named environment
      --  lookup, register this new environment so that its parent is updated
      --  when the precence for this named environment changes.
      if Parent_From_Name then
         declare
            NED : constant Named_Env_Descriptor_Access := State.Current_NED;
         begin
            Self.Unit.Exiled_Envs.Append ((NED, Self.Self_Env));
            NED.Foreign_Envs.Insert (Self, Self.Self_Env);
         end;
      end if;
      --  From now on, the current environment is Self.Self_Env, with a direct
      --  access to it. It does not go through the env naming scheme, since
      --  only this node and its children (i.e. non-foreign nodes) will access
      --  it as a "current" environment during PLE.
      Use_Direct_Env (State, Self.Self_Env);
      --  Register the environment we just created on all the requested names
      if Names /= null then
         declare
            Context   : constant Internal_Context := Self.Unit.Context;
            Env       : constant Lexical_Env := Self.Self_Env;
            NENU      : NED_Maps.Map renames
               State.Unit_State.Named_Envs_Needing_Update;
         begin
            for N of Names.Items loop
               Register_Named_Env (Context, N, Env, NENU);
            end loop;
            Dec_Ref (Names);
         end;
      end if;
   end Add_Env;
   ---------------------
   -- Pre_Env_Actions --
   ---------------------
   procedure Pre_Env_Actions
     (Self            : Bare_Ada_Node;
      State           : in out PLE_Node_State;
      Add_To_Env_Only : Boolean := False) is
   begin
   case Self.Kind is
            when Ada_With_Clause =>
            With_Clause_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Use_Type_Clause =>
            Use_Type_Clause_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Use_Package_Clause =>
            Use_Package_Clause_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Access_To_Subp_Def =>
            Access_To_Subp_Def_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Named_Stmt =>
            Named_Stmt_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Extended_Return_Stmt =>
            Extended_Return_Stmt_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Begin_Block .. Ada_Decl_Block =>
            Block_Stmt_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_For_Loop_Stmt =>
            For_Loop_Stmt_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Accept_Stmt .. Ada_Accept_Stmt_With_Stmts =>
            Accept_Stmt_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Pragma_Node =>
            Pragma_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Decl_Expr =>
            Decl_Expr_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Private_Part =>
            Private_Part_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Single_Task_Decl =>
            Single_Task_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Single_Protected_Decl =>
            Single_Protected_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Package_Renaming_Decl =>
            Package_Renaming_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Object_Decl .. Ada_No_Type_Object_Renaming_Decl =>
            Object_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Number_Decl =>
            Number_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Label_Decl =>
            Label_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Generic_Subp_Renaming_Decl =>
            Generic_Subp_Renaming_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Generic_Package_Renaming_Decl =>
            Generic_Package_Renaming_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Generic_Subp_Instantiation =>
            Generic_Subp_Instantiation_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Generic_Package_Instantiation =>
            Generic_Package_Instantiation_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Generic_Subp_Decl =>
            Generic_Subp_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Generic_Package_Decl =>
            Generic_Package_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_For_Loop_Var_Decl =>
            For_Loop_Var_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Exception_Handler =>
            Exception_Handler_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Exception_Decl =>
            Exception_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Entry_Index_Spec =>
            Entry_Index_Spec_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Task_Body =>
            Task_Body_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Protected_Body =>
            Protected_Body_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Package_Body =>
            Package_Body_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Entry_Body =>
            Entry_Body_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Task_Body_Stub =>
            Task_Body_Stub_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Subp_Body_Stub =>
            Subp_Body_Stub_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Protected_Body_Stub =>
            Protected_Body_Stub_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Package_Body_Stub =>
            Package_Body_Stub_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Expr_Function .. Ada_Subp_Renaming_Decl =>
            Base_Subp_Body_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Generic_Subp_Internal =>
            Generic_Subp_Internal_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Enum_Literal_Decl .. Ada_Synthetic_Char_Enum_Lit =>
            Enum_Literal_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Entry_Decl =>
            Entry_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Abstract_Subp_Decl .. Ada_Subp_Decl | Ada_Synthetic_Subp_Decl =>
            Basic_Subp_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Anonymous_Type_Decl .. Ada_Synth_Anonymous_Type_Decl =>
            null;
            when Ada_Concrete_Type_Decl .. Ada_Formal_Type_Decl =>
            Type_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Single_Task_Type_Decl =>
            Single_Task_Type_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Task_Type_Decl =>
            Task_Type_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Protected_Type_Decl =>
            Protected_Type_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Incomplete_Type_Decl .. Ada_Incomplete_Tagged_Type_Decl =>
            Incomplete_Type_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Classwide_Type_Decl =>
            null;
            when Ada_Discrete_Base_Subtype_Decl .. Ada_Subtype_Decl =>
            Base_Type_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Package_Decl =>
            Package_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Generic_Package_Internal =>
            Generic_Package_Internal_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Param_Spec =>
            Param_Spec_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Discriminant_Spec =>
            Discriminant_Spec_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Component_Decl =>
            Component_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Abstract_State_Decl =>
            Abstract_State_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
      when others =>  null; 
   end case;
   end Pre_Env_Actions;
   ----------------------
   -- Post_Env_Actions --
   ----------------------
   procedure Post_Env_Actions
     (Self : Bare_Ada_Node; State : in out PLE_Node_State) is
   begin
   case Self.Kind is
            when Ada_With_Clause =>
            null;
            when Ada_Use_Type_Clause =>
            Use_Type_Clause_Post_Env_Actions (Self, State);
            when Ada_Use_Package_Clause =>
            Use_Package_Clause_Post_Env_Actions (Self, State);
            when Ada_Access_To_Subp_Def =>
            null;
            when Ada_Named_Stmt =>
            null;
            when Ada_Extended_Return_Stmt =>
            null;
            when Ada_Begin_Block .. Ada_Decl_Block =>
            null;
            when Ada_For_Loop_Stmt =>
            null;
            when Ada_Accept_Stmt .. Ada_Accept_Stmt_With_Stmts =>
            null;
            when Ada_Pragma_Node =>
            null;
            when Ada_Decl_Expr =>
            null;
            when Ada_Private_Part =>
            null;
            when Ada_Single_Task_Decl =>
            null;
            when Ada_Single_Protected_Decl =>
            null;
            when Ada_Package_Renaming_Decl =>
            null;
            when Ada_Object_Decl .. Ada_No_Type_Object_Renaming_Decl =>
            null;
            when Ada_Number_Decl =>
            null;
            when Ada_Label_Decl =>
            null;
            when Ada_Generic_Subp_Renaming_Decl =>
            null;
            when Ada_Generic_Package_Renaming_Decl =>
            null;
            when Ada_Generic_Subp_Instantiation =>
            null;
            when Ada_Generic_Package_Instantiation =>
            null;
            when Ada_Generic_Subp_Decl =>
            Generic_Subp_Decl_Post_Env_Actions (Self, State);
            when Ada_Generic_Package_Decl =>
            null;
            when Ada_For_Loop_Var_Decl =>
            null;
            when Ada_Exception_Handler =>
            null;
            when Ada_Exception_Decl =>
            null;
            when Ada_Entry_Index_Spec =>
            null;
            when Ada_Task_Body =>
            null;
            when Ada_Protected_Body =>
            null;
            when Ada_Package_Body =>
            null;
            when Ada_Entry_Body =>
            null;
            when Ada_Task_Body_Stub =>
            null;
            when Ada_Subp_Body_Stub =>
            null;
            when Ada_Protected_Body_Stub =>
            null;
            when Ada_Package_Body_Stub =>
            null;
            when Ada_Expr_Function .. Ada_Subp_Renaming_Decl =>
            null;
            when Ada_Generic_Subp_Internal =>
            null;
            when Ada_Enum_Literal_Decl .. Ada_Synthetic_Char_Enum_Lit =>
            null;
            when Ada_Entry_Decl =>
            null;
            when Ada_Abstract_Subp_Decl .. Ada_Subp_Decl | Ada_Synthetic_Subp_Decl =>
            Basic_Subp_Decl_Post_Env_Actions (Self, State);
            when Ada_Anonymous_Type_Decl .. Ada_Synth_Anonymous_Type_Decl =>
            null;
            when Ada_Concrete_Type_Decl .. Ada_Formal_Type_Decl =>
            Type_Decl_Post_Env_Actions (Self, State);
            when Ada_Single_Task_Type_Decl =>
            null;
            when Ada_Task_Type_Decl =>
            null;
            when Ada_Protected_Type_Decl =>
            null;
            when Ada_Incomplete_Type_Decl .. Ada_Incomplete_Tagged_Type_Decl =>
            null;
            when Ada_Classwide_Type_Decl =>
            null;
            when Ada_Discrete_Base_Subtype_Decl .. Ada_Subtype_Decl =>
            null;
            when Ada_Package_Decl =>
            null;
            when Ada_Generic_Package_Internal =>
            null;
            when Ada_Param_Spec =>
            null;
            when Ada_Discriminant_Spec =>
            null;
            when Ada_Component_Decl =>
            null;
            when Ada_Abstract_State_Decl =>
            null;
      when others =>  null; 
   end case;
   end Post_Env_Actions;
   ----------------
   -- Get_Symbol --
   ----------------
   function Get_Symbol
     (Node : Bare_Ada_Node) return Symbol_Type is
   begin
      if Node = null then
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "cannot get the symbol of a null node");
      end if;
      return Get_Symbol (Token (Node, Node.Token_Start_Index));
   end Get_Symbol;
   ----------
   -- Text --
   ----------
   function Text
     (Node : Bare_Ada_Node) return Text_Type
   is
   begin
      if Node = null then
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "cannot get the text of a null node");
      end if;
      declare
         Start_T : constant Token_Reference :=
            Token (Node, Node.Token_Start_Index);
         End_T   : constant Token_Reference :=
            Token (Node, Node.Token_End_Index);
      begin
         --  No text is associated to synthetic and ghost nodes
         if Is_Synthetic (Node) then
            return "";
         end if;
         if Is_Ghost (Node) then
            return "";
         end if;
         return Text (Start_T, End_T);
      end;
   end Text;
   ---------------------
   -- Is_Visible_From --
   ---------------------
   function Is_Visible_From
     (Self                     : Bare_Ada_Node;
      Referenced_Env, Base_Env : Lexical_Env) return Boolean
   is
      Referenced_Unit : constant Internal_Unit :=
         Convert_Unit (Referenced_Env.Owner);
      Base_Unit       : constant Internal_Unit :=
         Convert_Unit (Base_Env.Owner);
   begin
      if Referenced_Unit = null then
         Raise_Property_Exception
           (Self,
            Property_Error'Identity,
            "referenced environment does not belong to any analysis unit");
      elsif Base_Unit = null then
         Raise_Property_Exception
           (Self,
            Property_Error'Identity,
            "base environment does not belong to any analysis unit");
      end if;
      return Is_Referenced_From (Referenced_Unit, Base_Unit);
   end Is_Visible_From;
   ----------
   -- Unit --
   ----------
   function Unit (Node : Bare_Ada_Node) return Internal_Unit is
   begin
      return Node.Unit;
   end Unit;
   function Lookup_Internal
     (Node : Bare_Ada_Node;
      Sloc : Source_Location) return Bare_Ada_Node;
   procedure Lookup_Relative
     (Node       : Bare_Ada_Node;
      Sloc       : Source_Location;
      Position   : out Relative_Position;
      Node_Found : out Bare_Ada_Node);
   --  Implementation helpers for the looking up process
   -----------------
   -- Set_Parents --
   -----------------
   procedure Set_Parents
     (Node, Parent : Bare_Ada_Node)
   is
   begin
      if Node = null then
         return;
      end if;
      Node.Parent := Bare_Ada_Node (Parent);
      for I in 1 .. Children_Count (Node) loop
         Set_Parents (Child (Node, I), Node);
      end loop;
   end Set_Parents;
   -------------
   -- Destroy --
   -------------
   procedure Destroy (Node : Bare_Ada_Node) is
   begin
      if Node = null then
         return;
      end if;
      Free_User_Fields (Node);
      for I in 1 .. Children_Count (Node) loop
         Destroy (Child (Node, I));
      end loop;
   end Destroy;
   -----------
   -- Child --
   -----------
   function Child (Node  : Bare_Ada_Node;
                   Index : Positive) return Bare_Ada_Node
   is
      Result          : Bare_Ada_Node;
      Index_In_Bounds : Boolean;
   begin
      Get_Child (Node, Index, Index_In_Bounds, Result);
      return Result;
   end Child;
   --------------
   -- Traverse --
   --------------
   function Traverse
     (Node  : Bare_Ada_Node;
      Visit : access function (Node : Bare_Ada_Node)
              return Visit_Status)
     return Visit_Status
   is
      Status : Visit_Status := Into;
   begin
      if Node /= null then
         Status := Visit (Node);
         --  Skip processing the child nodes if the returned status is Over
         --  or Stop. In the former case the previous call to Visit has taken
         --  care of processing the needed childs, and in the latter case we
         --  must immediately stop processing the tree.
         if Status = Into then
            for I in 1 .. Children_Count (Node) loop
               declare
                  Cur_Child : constant Bare_Ada_Node :=
                     Child (Node, I);
               begin
                  if Cur_Child /= null then
                     Status := Traverse (Cur_Child, Visit);
                     exit when Status /= Into;
                  end if;
               end;
            end loop;
         end if;
      end if;
      if Status = Stop then
         return Stop;
      --  At this stage the Over status has no sense and we just continue
      --  processing the tree.
      else
         return Into;
      end if;
   end Traverse;
   --------------
   -- Traverse --
   --------------
   procedure Traverse
     (Node  : Bare_Ada_Node;
      Visit : access function (Node : Bare_Ada_Node)
                               return Visit_Status)
   is
      Result_Status : Visit_Status;
      pragma Unreferenced (Result_Status);
   begin
      Result_Status := Traverse (Node, Visit);
   end Traverse;
   ------------------------
   -- Traverse_With_Data --
   ------------------------
   function Traverse_With_Data
     (Node  : Bare_Ada_Node;
      Visit : access function (Node : Bare_Ada_Node;
                               Data : in out Data_Type)
                               return Visit_Status;
      Data  : in out Data_Type)
      return Visit_Status
   is
      function Helper (Node : Bare_Ada_Node) return Visit_Status;
      ------------
      -- Helper --
      ------------
      function Helper (Node : Bare_Ada_Node) return Visit_Status is
      begin
         return Visit (Node, Data);
      end Helper;
      Saved_Data : Data_Type;
      Result     : Visit_Status;
   begin
      if Reset_After_Traversal then
         Saved_Data := Data;
      end if;
      Result := Traverse (Node, Helper'Access);
      if Reset_After_Traversal then
         Data := Saved_Data;
      end if;
      return Result;
   end Traverse_With_Data;
   ----------------
   -- Sloc_Range --
   ----------------
   function Sloc_Range
     (Node : Bare_Ada_Node) return Source_Location_Range
   is
      type Token_Anchor is (T_Start, T_End);
      type Token_Pos is record
         Pos    : Token_Index;
         Anchor : Token_Anchor;
      end record;
      TDH                    : Token_Data_Handler renames Node.Unit.TDH;
      Token_Start, Token_End : Token_Pos;
      function Get (Index : Token_Index) return Stored_Token_Data is
        (Get_Token (TDH, Index));
      function Sloc (T : Token_Pos) return Source_Location is
        (if T.Anchor = T_Start
         then Sloc_Start (TDH, Get (T.Pos))
         else Sloc_End (TDH, Get (T.Pos)));
   begin
      if Is_Synthetic (Node) then
         return Sloc_Range (Node.Parent);
      end if;
      if Is_Ghost (Node) then
         Token_Start := (if Node.Token_Start_Index = 1
                         then (1, T_Start)
                         else (Node.Token_Start_Index - 1, T_End));
         Token_End := Token_Start;
      else
         Token_Start := (Node.Token_Start_Index, T_Start);
         Token_End := (Node.Token_End_Index, T_End);
      end if;
      if Snaps_At_Start (Node)
         and then not Is_Ghost (Node)
         and then Token_Start.Pos /= 1
      then
         Token_Start := (Token_Start.Pos - 1, T_End);
      end if;
      if Snaps_At_End (Node) and then Token_End.Pos /= Last_Token (TDH) then
         Token_End := (Token_End.Pos + 1, T_Start);
      end if;
      return Make_Range (Sloc (Token_Start), Sloc (Token_End));
   end Sloc_Range;
   ------------
   -- Lookup --
   ------------
   function Lookup
     (Node : Bare_Ada_Node;
      Sloc : Source_Location) return Bare_Ada_Node
   is
      Position : Relative_Position;
      Result   : Bare_Ada_Node;
   begin
      if Sloc = No_Source_Location then
         return null;
      end if;
      Lookup_Relative
        (Bare_Ada_Node (Node), Sloc, Position, Result);
      return Result;
   end Lookup;
   ---------------------
   -- Lookup_Internal --
   ---------------------
   function Lookup_Internal
     (Node : Bare_Ada_Node;
      Sloc : Source_Location) return Bare_Ada_Node
   is
      --  For this implementation helper (i.e. internal primitive), we can
      --  assume that all lookups fall into this node's sloc range.
      pragma Assert (Compare (Sloc_Range (Node), Sloc) = Inside);
      Children : constant Internal_Bare_Ada_Node_Array :=
         Implementation.Children (Node);
      Pos      : Relative_Position;
      Result   : Bare_Ada_Node;
   begin
      --  Look for a child node that contains Sloc (i.e. return the most
      --  precise result).
      for Child of Children loop
         --  Note that we assume here that child nodes are ordered so that the
         --  first one has a sloc range that is before the sloc range of the
         --  second child node, etc.
         if Child /= null then
            Lookup_Relative (Child, Sloc, Pos, Result);
            case Pos is
               when Before =>
                   --  If this is the first node, Sloc is before it, so we can
                   --  stop here.  Otherwise, Sloc is between the previous
                   --  child node and the next one...  so we can stop here,
                   --  too.
                   return Node;
               when Inside =>
                   return Result;
               when After =>
                   --  Sloc is after the current child node, so see with the
                   --  next one.
                   null;
            end case;
         end if;
      end loop;
      --  If we reach this point, we found no children that covers Sloc, but
      --  Node still covers it (see the assertion).
      return Node;
   end Lookup_Internal;
   -------------
   -- Compare --
   -------------
   function Compare
     (Node : Bare_Ada_Node;
      Sloc : Source_Location) return Relative_Position is
   begin
      return Compare (Sloc_Range (Node), Sloc);
   end Compare;
   ---------------------
   -- Lookup_Relative --
   ---------------------
   procedure Lookup_Relative
     (Node       : Bare_Ada_Node;
      Sloc       : Source_Location;
      Position   : out Relative_Position;
      Node_Found : out Bare_Ada_Node)
   is
      Result : constant Relative_Position :=
        Compare (Node, Sloc);
   begin
      Position := Result;
      Node_Found := (if Result = Inside
                     then Lookup_Internal (Node, Sloc)
                     else null);
   end Lookup_Relative;
   -------------
   -- Compare --
   -------------
   function Compare
     (Self, Left, Right : Bare_Ada_Node;
      Relation          : Comparison_Relation) return Boolean
   is
      LS, RS : Source_Location;
   begin
      if Left = null or else Right = null or else Left.Unit /= Right.Unit then
         Raise_Property_Exception
           (Self,
            Property_Error'Identity,
            "invalid node comparison");
      end if;
      LS := Start_Sloc (Sloc_Range (Left));
      RS := Start_Sloc (Sloc_Range (Right));
      return (case Relation is
              when Langkit_Support.Types.Less_Than        => LS < RS,
              when Langkit_Support.Types.Less_Or_Equal    => LS <= RS,
              when Langkit_Support.Types.Greater_Than     => LS > RS,
              when Langkit_Support.Types.Greater_Or_Equal => LS >= RS);
   end Compare;
   --------------
   -- Children --
   --------------
   function Children
     (Node : Bare_Ada_Node) return Internal_Bare_Ada_Node_Array
   is
      First : constant Integer := Bare_Ada_Node_Vectors.Index_Type'First;
      Last  : constant Integer := First + Children_Count (Node) - 1;
   begin
      return A : Internal_Bare_Ada_Node_Array (First .. Last)
      do
         for I in First .. Last loop
            A (I) := Child (Node, I);
         end loop;
      end return;
   end Children;
   function Children
     (Node : Bare_Ada_Node) return Bare_Ada_Node_Array_Access
   is
      C : Internal_Bare_Ada_Node_Array := Children (Node);
   begin
      return Ret : Bare_Ada_Node_Array_Access :=
         Create_Bare_Ada_Node_Array (C'Length)
      do
         Ret.Items := C;
      end return;
   end Children;
   ---------
   -- Get --
   ---------
   function Get
     (Self    : Bare_Ada_Node;
      Node    : Bare_Ada_List;
      Index   : Integer;
      Or_Null : Boolean := False) return Bare_Ada_Node
   is
      function Length (Node : Bare_Ada_List) return Natural
      is (Node.Count);
      --  Wrapper around the Length primitive to get the compiler happy for the
      --  the package instantiation below.
      function Absolute_Get
        (L     : Bare_Ada_List;
         Index : Integer) return Bare_Ada_Node
      is (L.Nodes.all (Index + 1));
      --  L.Nodes is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Bare_Ada_Node,
         Sequence_Type => Bare_Ada_List,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Bare_Ada_Node;
   begin
      if Node = null and then Or_Null then
         return null;
      elsif Relative_Get (Node, Index, Result) then
         return Result;
      elsif Or_Null then
         return null;
      else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "out-of-bounds AST list access");
      end if;
   end Get;
   ---------------
   -- PP_Trivia --
   ---------------
   procedure PP_Trivia
     (Node        : Bare_Ada_Node;
      Line_Prefix : String := "")
   is
      Children_Prefix : constant String := Line_Prefix & "|  ";
   begin
      if Node = null then
         Put_Line (Line_Prefix & "None");
         return;
      end if;
      Put_Line (Line_Prefix & Kind_Name (Node));
      for C of Children_And_Trivia (Node) loop
         case C.Kind is
            when Trivia =>
               Put_Line (Children_Prefix & Image (Text (C.Trivia)));
            when Child =>
               PP_Trivia (C.Node, Children_Prefix);
         end case;
      end loop;
   end PP_Trivia;
   --------------------------
   -- Populate_Lexical_Env --
   --------------------------
   function Populate_Lexical_Env (Node : Bare_Ada_Node) return Boolean is
      Context    : constant Internal_Context := Node.Unit.Context;
      Unit_State : aliased PLE_Unit_State := (Named_Envs_Needing_Update => <>);
      Root_State : constant PLE_Node_State :=
        (Unit_State  => Unit_State'Unchecked_Access,
         Current_Env => Context.Root_Scope,
         Current_NED => null);
      function Populate_Internal
        (Node         : Bare_Ada_Node;
         Parent_State : PLE_Node_State) return Boolean;
      --  Do the lexical env population on Node and recurse on its children
      procedure Register_Foreign_Env
        (Node : Bare_Ada_Node; State : PLE_Node_State);
      --  Given a node and its PLE state, register Node.Self_Env as being
      --  initialized through the named environment mechanism, if that's indeed
      --  the case. Do nothing otherwise.
      -----------------------
      -- Populate_Internal --
      -----------------------
      function Populate_Internal
        (Node         : Bare_Ada_Node;
         Parent_State : PLE_Node_State) return Boolean
      is
         Result : Boolean := False;
         State  : PLE_Node_State := Parent_State;
      begin
         if Node = null then
            return Result;
         end if;
         --  By default (i.e. unless env actions add a new env), the
         --  environment we store in Node is the current one.
         Node.Self_Env := State.Current_Env;
         --  Run pre/post actions, and run PLE on children in between. Make
         --  sure we register the potential foreign Node.Self_Env environment
         --  at the end, even when an exception interrupts PLE to keep the
         --  state consistent.
         begin
            Pre_Env_Actions (Node, State);
            if State.Current_Env /= Null_Lexical_Env then
               Node.Self_Env := State.Current_Env;
               Register_Foreign_Env (Node, State);
            end if;
            --  Call recursively on children
            for I in First_Child_Index (Node) .. Last_Child_Index (Node) loop
               Result := Populate_Internal
                 (Child (Node, I), State) or else Result;
            end loop;
            Post_Env_Actions (Node, State);
         exception
            when Exc : Precondition_Failure | Property_Error =>
               if PLE_Errors_Trace.Is_Active then
                   GNATCOLL.Traces.Trace
                     (PLE_Errors_Trace,
                      "Exception raised during PLE "
                      & Ada.Exceptions.Exception_Name (Exc) & " : "
                      & Ada.Exceptions.Exception_Message (Exc));
                   GNATCOLL.Traces.Trace
                     (PLE_Errors_Trace,
                      GNAT.Traceback.Symbolic.Symbolic_Traceback (Exc));
               end if;
               Register_Foreign_Env (Node, State);
               return True;
         end;
         return Result;
      end Populate_Internal;
      --------------------------
      -- Register_Foreign_Env --
      --------------------------
      procedure Register_Foreign_Env
        (Node : Bare_Ada_Node; State : PLE_Node_State) is
      begin
         if State.Current_NED /= null then
            State.Current_NED.Nodes_With_Foreign_Env.Insert (Node);
            Node.Unit.Nodes_With_Foreign_Env.Insert (Node, State.Current_NED);
         end if;
      end Register_Foreign_Env;
   begin
      --  This function is meant to be called during an existing PLE pass. If
      --  if is called outside of this context, run the PLE pass on Node's
      --  analysis unit. Likewise, if PLE has not run on the unit that owns
      --  this PLE unit yet, do a full run, which will in the end trigger the
      --  PLE on this PLE unit.
      --
      --  We do this so that as soon as PLE is required on a PLE unit: the
      --  whole unit end up with its lexical environments populated.
      if not Context.In_Populate_Lexical_Env then
         begin
            Populate_Lexical_Env (Node.Unit);
            return False;
         exception
            when Precondition_Failure | Property_Error =>
               return True;
         end;
      end if;
      --  This is intended to be called on the root node only
      if Node.Parent /= null then
         raise Program_Error;
      end if;
      return Result : constant Boolean :=
         Populate_Internal (Node, Root_State)
      do
         Update_Named_Envs (Context, Unit_State.Named_Envs_Needing_Update);
      end return;
   end Populate_Lexical_Env;
   ------------------------------
   -- AST_Envs_Node_Text_Image --
   ------------------------------
   function AST_Envs_Node_Text_Image
     (Node  : Bare_Ada_Node;
      Short : Boolean := True) return Text_Type is
   begin
      if Short then
         if Node = null then
            return "null";
         end if;
         return To_Text (Basename (Node.Unit))
           & ":" & To_Text (Image (Start_Sloc (Sloc_Range (Node))));
      else
         return Short_Text_Image (Node);
      end if;
   end AST_Envs_Node_Text_Image;
   -------------------
   -- Is_Rebindable --
   -------------------
   function Is_Rebindable (Node : Bare_Ada_Node) return Boolean is
   begin
         return Node.Kind in Ada_Generic_Package_Decl .. Ada_Generic_Subp_Decl;
   end Is_Rebindable;
   -----------------------
   -- Acquire_Rebinding --
   -----------------------
   function Acquire_Rebinding
     (Node             : Bare_Ada_Node;
      Parent           : Env_Rebindings;
      Old_Env, New_Env : Lexical_Env) return Env_Rebindings
   is
      Result    : Env_Rebindings;
      Available : Env_Rebindings_Vectors.Vector renames
         Node.Unit.Context.Available_Rebindings;
   begin
      --  Use an existing and available Env_Rebindings_Type record for Node's
      --  Context, otherwise allocate a new rebinding.
      Result := (if Available.Is_Empty
                 then new Env_Rebindings_Type'(Version => 0, others => <>)
                 else Available.Pop);
      Result.Parent := Parent;
      Result.Old_Env := Old_Env;
      Result.New_Env := New_Env;
      Result.Children := Env_Rebindings_Vectors.Empty_Vector;
      return Result;
   end Acquire_Rebinding;
   -----------------------
   -- Release_Rebinding --
   -----------------------
   procedure Release_Rebinding (Self : in out Env_Rebindings) is
      Available : Env_Rebindings_Vectors.Vector renames
         Unwrap (Self.Old_Env).Node.Unit.Context.Available_Rebindings;
   begin
      --  Bumping the version number, to invalidate existing references to
      --  Self.
      Self.Version := Self.Version + 1;
      Self.Children.Destroy;
      Available.Append (Self);
      Self := null;
   end Release_Rebinding;
   ------------------------
   -- Register_Rebinding --
   ------------------------
   procedure Register_Rebinding
     (Node : Bare_Ada_Node; Rebinding : Env_Rebindings) is
   begin
      Node.Unit.Rebindings.Append (Rebinding);
   end Register_Rebinding;
   --------------------
   -- Element_Parent --
   --------------------
   function Element_Parent
     (Node : Bare_Ada_Node) return Bare_Ada_Node
   is (Node.Parent);
   ---------------
   -- Node_Unit --
   ---------------
   function Node_Unit (Node : Bare_Ada_Node) return Generic_Unit_Ptr is
   begin
      return Convert_Unit (Node.Unit);
   end Node_Unit;
   ----------
   -- Hash --
   ----------
   function Hash (Node : Bare_Ada_Node) return Hash_Type
   is
      function H is new Hash_Access
        (Root_Node_Record, Bare_Ada_Node);
   begin
      return H (Node);
   end Hash;
      function Hash (B : Boolean) return Hash_Type is (Boolean'Pos (B));
      function Hash (I : Integer) return Hash_Type is (Hash_Type'Mod (I));
   ------------------------
   -- Named environments --
   ------------------------
   ---------
   -- Add --
   ---------
   procedure Add
     (Self : in out NED_Assoc_Maps.Map;
      Key  : Symbol_Type;
      Node : AST_Envs.Internal_Map_Node)
   is
      use NED_Assoc_Maps;
      Pos   : Cursor;
      Dummy : Boolean;
   begin
      --  Make sure there is a vector entry for Key
      Self.Insert (Key, Internal_Map_Node_Vectors.Empty_Vector, Pos, Dummy);
      --  Append Node to that vector
      declare
         V : Internal_Map_Node_Vectors.Vector renames Self.Reference (Pos);
      begin
         V.Append (Node);
      end;
   end Add;
   ------------
   -- Remove --
   ------------
   procedure Remove
     (Self : in out NED_Assoc_Maps.Map;
      Key  : Symbol_Type;
      Node : Bare_Ada_Node)
   is
      use NED_Assoc_Maps;
      V : Internal_Map_Node_Vectors.Vector renames Self.Reference (Key);
   begin
      --  Remove the (assumed unique) entry in V whose node is Node. The order
      --  of items in V is not significant, so we can use Pop for efficient
      --  removal. Do the traversal in reverse order for correctness.
      for I in reverse 1 .. V.Length loop
         if V.Get_Access (I).Node = Node then
            V.Pop (I);
            exit;
         end if;
      end loop;
   end Remove;
   ------------------------------
   -- Get_Named_Env_Descriptor --
   ------------------------------
   function Get_Named_Env_Descriptor
     (Context : Internal_Context;
      Name    : Symbol_Type) return Named_Env_Descriptor_Access
   is
      use NED_Maps;
      --  Look for an existing entry for Name
      Pos : constant Cursor := Context.Named_Envs.Find (Name);
   begin
      if Has_Element (Pos) then
         return Element (Pos);
      end if;
      --  There is no such entry: create one
      return Result : constant Named_Env_Descriptor_Access :=
         new Named_Env_Descriptor'
           (Name                   => Name,
            Envs                   => <>,
            Env_With_Precedence    => Empty_Env,
            Foreign_Nodes          => <>,
            Foreign_Envs           => <>,
            Nodes_With_Foreign_Env => <>)
      do
         Context.Named_Envs.Insert (Name, Result);
      end return;
   end Get_Named_Env_Descriptor;
   ------------------------
   -- Register_Named_Env --
   ------------------------
   procedure Register_Named_Env
     (Context                   : Internal_Context;
      Name                      : Symbol_Type;
      Env                       : Lexical_Env;
      Named_Envs_Needing_Update : in out NED_Maps.Map)
   is
      NED_Access : constant Named_Env_Descriptor_Access :=
         Get_Named_Env_Descriptor (Context, Name);
      NED        : Named_Env_Descriptor renames NED_Access.all;
      Node       : constant Bare_Ada_Node := Env_Node (Env);
   begin
      NED.Envs.Insert (Node, Env);
      Node.Unit.Named_Envs.Append ((Name, Env));
      --  If that insertion must change the env that has precedence, signal
      --  that NED requires an update.
      if NED.Envs.First_Element /= NED.Env_With_Precedence then
         Named_Envs_Needing_Update.Include (Name, NED_Access);
      end if;
   end Register_Named_Env;
   ----------------------
   -- Update_Named_Env --
   ----------------------
   procedure Update_Named_Envs
     (Context : Internal_Context; Named_Envs : NED_Maps.Map)
   is
      Require_Cache_Reset : Boolean := False;
   begin
      for Cur in Named_Envs.Iterate loop
         declare
            NE      : Named_Env_Descriptor renames NED_Maps.Element (Cur).all;
            New_Env : constant Lexical_Env :=
              (if NE.Envs.Is_Empty
               then Empty_Env
               else NE.Envs.First_Element);
         begin
            --  If there was an environment with precedence, remove its foreign
            --  nodes.
            if NE.Env_With_Precedence /= Empty_Env then
               for Cur in NE.Foreign_Nodes.Iterate loop
                  declare
                     Key   : constant Symbol_Type :=
                        NED_Assoc_Maps.Key (Cur);
                     Nodes : Internal_Map_Node_Vectors.Vector renames
                        NE.Foreign_Nodes.Reference (Cur);
                  begin
                     for N of Nodes loop
                        Remove (NE.Env_With_Precedence, Key, N.Node);
                     end loop;
                  end;
               end loop;
            end if;
            --  Now, set the new environment that has precedence
            NE.Env_With_Precedence := New_Env;
            --  Add the foreign nodes to the new environment with precedence,
            --  if any.
            for Cur in NE.Foreign_Nodes.Iterate loop
               declare
                  Key   : constant Symbol_Type :=
                     NED_Assoc_Maps.Key (Cur);
                  Nodes : Internal_Map_Node_Vectors.Vector renames
                     NE.Foreign_Nodes.Reference (Cur);
               begin
                  for N of Nodes loop
                     Add (New_Env, Key, N.Node, N.Md, N.Resolver);
                  end loop;
               end;
            end loop;
            --  Set the parent environment of all foreign environments
            for Cur in NE.Foreign_Envs.Iterate loop
               declare
                  Env : Lexical_Env_Record renames
                     Unwrap (Sorted_Env_Maps.Element (Cur)).all;
               begin
                  Env.Parent := New_Env;
                  --  We have updated the lexical env hierarchy (e.g. an env
                  --  which had no parent may have one now), so the cached
                  --  entries for queries that traveresed the old env hierarchy
                  --  need to be invalidated.
                  Require_Cache_Reset := True;
               end;
            end loop;
            --  Update nodes whose environment was the old env with precedence
            for N of NE.Nodes_With_Foreign_Env loop
               N.Self_Env := New_Env;
            end loop;
         end;
      end loop;
      if Require_Cache_Reset then
         Invalidate_Caches (Context, Invalidate_Envs => True);
      end if;
   end Update_Named_Envs;
   --------------------------
   -- Big integers wrapper --
   --------------------------
   ------------------------
   -- Create_Big_Integer --
   ------------------------
   function Create_Big_Integer
     (Image : String; Base : Integer := 10) return Big_Integer_Type
   is
      use GNATCOLL.GMP;
      use GNATCOLL.GMP.Integers;
   begin
      return new Big_Integer_Record'(Value     => Make (Image, Int (Base)),
                                     Ref_Count => 1);
   end Create_Big_Integer;
   ------------------------
   -- Create_Big_Integer --
   ------------------------
   function Create_Big_Integer
     (Big_Int : GNATCOLL.GMP.Integers.Big_Integer) return Big_Integer_Type
   is
      Result : constant Big_Integer_Type :=
         new Big_Integer_Record'(Value     => <>,
                                 Ref_Count => 1);
   begin
      Result.Value.Set (Big_Int);
      return Result;
   end Create_Big_Integer;
   ------------------------
   -- Create_Big_Integer --
   ------------------------
   function Create_Big_Integer (Int : Integer) return Big_Integer_Type is
      Result : constant Big_Integer_Type :=
         new Big_Integer_Record'(Value     => <>,
                                 Ref_Count => 1);
   begin
      Result.Value.Set (GNATCOLL.GMP.Long (Int));
      return Result;
   end Create_Big_Integer;
   -------------------------------
   -- Create_Public_Big_Integer --
   -------------------------------
   function Create_Public_Big_Integer
     (Big_Int : Big_Integer_Type) return GNATCOLL.GMP.Integers.Big_Integer is
   begin
      return Result : GNATCOLL.GMP.Integers.Big_Integer do
         Result.Set (Big_Int.Value);
      end return;
   end Create_Public_Big_Integer;
   -----------------
   -- Trace_Image --
   -----------------
   function Trace_Image (I : Big_Integer_Type) return String is
   begin
      return GNATCOLL.GMP.Integers.Image (I.Value);
   end Trace_Image;
   ----------------
   -- To_Integer --
   ----------------
   function To_Integer
     (Self    : Bare_Ada_Node;
      Big_Int : Big_Integer_Type) return Integer
   is
      Image : constant String := Big_Int.Value.Image;
   begin
      return Integer'Value (Image);
   exception
      when Constraint_Error =>
         Raise_Property_Exception
           (Self, Property_Error'Identity, "out of range big integer");
   end To_Integer;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (Big_Int : Big_Integer_Type) is
   begin
      if Big_Int.Ref_Count /= -1 then
         Big_Int.Ref_Count := Big_Int.Ref_Count + 1;
      end if;
   end Inc_Ref;
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (Big_Int : in out Big_Integer_Type) is
      procedure Destroy is new Ada.Unchecked_Deallocation
        (Big_Integer_Record, Big_Integer_Type);
   begin
      if Big_Int = null or else Big_Int.Ref_Count = -1 then
         return;
      end if;
      Big_Int.Ref_Count := Big_Int.Ref_Count - 1;
      if Big_Int.Ref_Count = 0 then
         Destroy (Big_Int);
      end if;
   end Dec_Ref;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (Left, Right : Big_Integer_Type) return Boolean is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Left.Value = Right.Value;
   end Equivalent;
   ---------
   -- "<" --
   ---------
   function "<" (Left, Right : Big_Integer_Type) return Boolean is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Left.Value < Right.Value;
   end "<";
   ----------
   -- "<=" --
   ----------
   function "<=" (Left, Right : Big_Integer_Type) return Boolean is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Left.Value <= Right.Value;
   end "<=";
   ---------
   -- ">" --
   ---------
   function ">" (Left, Right : Big_Integer_Type) return Boolean is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Left.Value > Right.Value;
   end ">";
   ----------
   -- ">=" --
   ----------
   function ">=" (Left, Right : Big_Integer_Type) return Boolean is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Left.Value >= Right.Value;
   end ">=";
   ---------
   -- "+" --
   ---------
   function "+" (Left, Right : Big_Integer_Type) return Big_Integer_Type is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Create_Big_Integer (Left.Value + Right.Value);
   end "+";
   ---------
   -- "-" --
   ---------
   function "-" (Left, Right : Big_Integer_Type) return Big_Integer_Type is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Create_Big_Integer (Left.Value - Right.Value);
   end "-";
   ------------------
   -- Unit_Version --
   ------------------
   function Unit_Version (Unit : Generic_Unit_Ptr) return Version_Number is
   begin
      return Convert_Unit (Unit).Unit_Version;
   end Unit_Version;
   -------------------------
   -- Get_Context_Version --
   -------------------------
   function Get_Context_Version
     (Node : Bare_Ada_Node) return Version_Number is
   begin
      return Node.Unit.Context.Cache_Version;
   end Get_Context_Version;
   --------------------------
   -- Properties_May_Raise --
   --------------------------
   function Properties_May_Raise
     (Exc : Ada.Exceptions.Exception_Occurrence) return Boolean is
   begin
      return Ada.Exceptions.Exception_Identity (Exc) in
            Precondition_Failure'Identity
               |
            Property_Error'Identity
      ;
   end Properties_May_Raise;
   ----------------------
   -- Short_Text_Image --
   ----------------------
   function Short_Text_Image (Self : Bare_Ada_Node) return Text_Type
   is
   begin
      if Self = null then
         return "None";
      end if;
   case Self.Kind is
            when Ada_Char_Literal .. Ada_String_Literal =>
         return Base_Id_Short_Image (Self);
            when Ada_Abstract_State_Decl .. Ada_Single_Task_Decl =>
         return Basic_Decl_Short_Image (Self);
      when others => 
         return "<" & To_Text (Kind_Name (Self))
                & " "
                & To_Text
                  (Ada.Directories.Simple_Name
                     (Get_Filename (Unit (Self))))
                & ":" & To_Text (Image (Sloc_Range (Self))) & ">";
   end case;
   end Short_Text_Image;
   --------------------
   -- Snaps_At_Start --
   --------------------
   function Snaps_At_Start (Self : Bare_Ada_Node) return Boolean is
   begin
   case Self.Kind is
            when Ada_Handled_Stmts =>
         return True;
            when Ada_Declarative_Part .. Ada_Public_Part =>
         return True;
      when others => 
         return False;
   end case;
   end Snaps_At_Start;
   ------------------
   -- Snaps_At_End --
   ------------------
   function Snaps_At_End (Self : Bare_Ada_Node) return Boolean is
   begin
   case Self.Kind is
            when Ada_Handled_Stmts =>
         return True;
            when Ada_Declarative_Part .. Ada_Public_Part =>
         return True;
      when others => 
         return Is_Incomplete (Self);
   end case;
   end Snaps_At_End;
   -------------
   -- Parents --
   -------------
   function Parents
     (Node      : Bare_Ada_Node;
      With_Self : Boolean := True)
      return Bare_Ada_Node_Array_Access
   is
      Count : Natural := 0;
      Start : Bare_Ada_Node :=
        (if With_Self then Node else Node.Parent);
      Cur   : Bare_Ada_Node := Start;
   begin
      while Cur /= null loop
         Count := Count + 1;
         Cur := Cur.Parent;
      end loop;
      declare
         Result : constant Bare_Ada_Node_Array_Access :=
            Create_Bare_Ada_Node_Array (Count);
      begin
         Cur := Start;
         for I in Result.Items'Range loop
            Result.Items (I) := Cur;
            Cur := Cur.Parent;
         end loop;
         return Result;
      end;
   end Parents;
   -----------------------
   -- First_Child_Index --
   -----------------------
   function First_Child_Index (Node : Bare_Ada_Node) return Natural
   is (1);
   ----------------------
   -- Last_Child_Index --
   ----------------------
   function Last_Child_Index (Node : Bare_Ada_Node) return Natural
   is (Children_Count (Node));
   ---------------
   -- Get_Child --
   ---------------
   procedure Get_Child
     (Node            : Bare_Ada_Node;
      Index           : Positive;
      Index_In_Bounds : out Boolean;
      Result          : out Bare_Ada_Node)
   is
      K : constant Ada_Node_Kind_Type := Node.Kind;
   begin
      Index_In_Bounds := True;
      Result := null;
      case Ada_Ada_Node (K) is
when Ada_Ada_List =>
declare
N_Bare_Ada_List : constant Bare_Ada_List := Node;
begin
                    if Index > N_Bare_Ada_List.Count then
                        Index_In_Bounds := False;
                    else
                        Result := N_Bare_Ada_List.Nodes (Index);
                    end if;
                    return;
end;
when Ada_Constrained_Array_Indices_Range =>
declare
N_Bare_Constrained_Array_Indices : constant Bare_Constrained_Array_Indices := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Constrained_Array_Indices.Constrained_Array_Indices_F_List;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Unconstrained_Array_Indices_Range =>
declare
N_Bare_Unconstrained_Array_Indices : constant Bare_Unconstrained_Array_Indices := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Unconstrained_Array_Indices.Unconstrained_Array_Indices_F_Types;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Aspect_Assoc_Range =>
declare
N_Bare_Aspect_Assoc : constant Bare_Aspect_Assoc := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Aspect_Assoc.Aspect_Assoc_F_Id;
                            return;
                        when 2 =>
                            Result := N_Bare_Aspect_Assoc.Aspect_Assoc_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_At_Clause_Range =>
declare
N_Bare_At_Clause : constant Bare_At_Clause := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_At_Clause.At_Clause_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_At_Clause.At_Clause_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Attribute_Def_Clause_Range =>
declare
N_Bare_Attribute_Def_Clause : constant Bare_Attribute_Def_Clause := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Attribute_Def_Clause.Attribute_Def_Clause_F_Attribute_Expr;
                            return;
                        when 2 =>
                            Result := N_Bare_Attribute_Def_Clause.Attribute_Def_Clause_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Enum_Rep_Clause_Range =>
declare
N_Bare_Enum_Rep_Clause : constant Bare_Enum_Rep_Clause := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Enum_Rep_Clause.Enum_Rep_Clause_F_Type_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Enum_Rep_Clause.Enum_Rep_Clause_F_Aggregate;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Record_Rep_Clause_Range =>
declare
N_Bare_Record_Rep_Clause : constant Bare_Record_Rep_Clause := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Record_Rep_Clause.Record_Rep_Clause_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Record_Rep_Clause.Record_Rep_Clause_F_At_Expr;
                            return;
                        when 3 =>
                            Result := N_Bare_Record_Rep_Clause.Record_Rep_Clause_F_Components;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Aspect_Spec_Range =>
declare
N_Bare_Aspect_Spec : constant Bare_Aspect_Spec := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Aspect_Spec.Aspect_Spec_F_Aspect_Assocs;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Contract_Case_Assoc_Range =>
declare
N_Bare_Contract_Case_Assoc : constant Bare_Contract_Case_Assoc := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Contract_Case_Assoc.Contract_Case_Assoc_F_Guard;
                            return;
                        when 2 =>
                            Result := N_Bare_Contract_Case_Assoc.Contract_Case_Assoc_F_Consequence;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Pragma_Argument_Assoc_Range =>
declare
N_Bare_Pragma_Argument_Assoc : constant Bare_Pragma_Argument_Assoc := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Pragma_Argument_Assoc.Pragma_Argument_Assoc_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Pragma_Argument_Assoc.Pragma_Argument_Assoc_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Entry_Spec_Range =>
declare
N_Bare_Entry_Spec : constant Bare_Entry_Spec := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Entry_Spec.Entry_Spec_F_Entry_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Entry_Spec.Entry_Spec_F_Family_Type;
                            return;
                        when 3 =>
                            Result := N_Bare_Entry_Spec.Entry_Spec_F_Entry_Params;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Subp_Spec_Range =>
declare
N_Bare_Subp_Spec : constant Bare_Subp_Spec := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Subp_Spec.Subp_Spec_F_Subp_Kind;
                            return;
                        when 2 =>
                            Result := N_Bare_Subp_Spec.Subp_Spec_F_Subp_Name;
                            return;
                        when 3 =>
                            Result := N_Bare_Subp_Spec.Subp_Spec_F_Subp_Params;
                            return;
                        when 4 =>
                            Result := N_Bare_Subp_Spec.Subp_Spec_F_Subp_Returns;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Synthetic_Binary_Spec_Range =>
declare
N_Bare_Synthetic_Binary_Spec : constant Bare_Synthetic_Binary_Spec := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Synthetic_Binary_Spec.Synthetic_Binary_Spec_F_Left_Param;
                            return;
                        when 2 =>
                            Result := N_Bare_Synthetic_Binary_Spec.Synthetic_Binary_Spec_F_Right_Param;
                            return;
                        when 3 =>
                            Result := N_Bare_Synthetic_Binary_Spec.Synthetic_Binary_Spec_F_Return_Type_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Synthetic_Unary_Spec_Range =>
declare
N_Bare_Synthetic_Unary_Spec : constant Bare_Synthetic_Unary_Spec := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Synthetic_Unary_Spec.Synthetic_Unary_Spec_F_Right_Param;
                            return;
                        when 2 =>
                            Result := N_Bare_Synthetic_Unary_Spec.Synthetic_Unary_Spec_F_Return_Type_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Component_List_Range =>
declare
N_Bare_Component_List : constant Bare_Component_List := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Component_List.Component_List_F_Components;
                            return;
                        when 2 =>
                            Result := N_Bare_Component_List.Component_List_F_Variant_Part;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Known_Discriminant_Part_Range =>
declare
N_Bare_Known_Discriminant_Part : constant Bare_Known_Discriminant_Part := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Known_Discriminant_Part.Known_Discriminant_Part_F_Discr_Specs;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Entry_Completion_Formal_Params_Range =>
declare
N_Bare_Entry_Completion_Formal_Params : constant Bare_Entry_Completion_Formal_Params := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Entry_Completion_Formal_Params.Entry_Completion_Formal_Params_F_Params;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Generic_Formal_Part_Range =>
declare
N_Bare_Generic_Formal_Part : constant Bare_Generic_Formal_Part := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Generic_Formal_Part.Generic_Formal_Part_F_Decls;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Base_Record_Def =>
declare
N_Bare_Base_Record_Def : constant Bare_Base_Record_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Base_Record_Def.Base_Record_Def_F_Components;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Aggregate_Assoc_Range =>
declare
N_Bare_Aggregate_Assoc : constant Bare_Aggregate_Assoc := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Aggregate_Assoc.Aggregate_Assoc_F_Designators;
                            return;
                        when 2 =>
                            Result := N_Bare_Aggregate_Assoc.Aggregate_Assoc_F_R_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Composite_Constraint_Assoc_Range =>
declare
N_Bare_Composite_Constraint_Assoc : constant Bare_Composite_Constraint_Assoc := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Composite_Constraint_Assoc.Composite_Constraint_Assoc_F_Ids;
                            return;
                        when 2 =>
                            Result := N_Bare_Composite_Constraint_Assoc.Composite_Constraint_Assoc_F_Constraint_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Iterated_Assoc_Range =>
declare
N_Bare_Iterated_Assoc : constant Bare_Iterated_Assoc := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Iterated_Assoc.Iterated_Assoc_F_Spec;
                            return;
                        when 2 =>
                            Result := N_Bare_Iterated_Assoc.Iterated_Assoc_F_R_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Param_Assoc_Range =>
declare
N_Bare_Param_Assoc : constant Bare_Param_Assoc := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Param_Assoc.Param_Assoc_F_Designator;
                            return;
                        when 2 =>
                            Result := N_Bare_Param_Assoc.Param_Assoc_F_R_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Abstract_State_Decl_Range =>
declare
N_Bare_Abstract_State_Decl : constant Bare_Abstract_State_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Abstract_State_Decl.Abstract_State_Decl_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Abstract_State_Decl.Abstract_State_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Anonymous_Expr_Decl_Range =>
declare
N_Bare_Anonymous_Expr_Decl : constant Bare_Anonymous_Expr_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Anonymous_Expr_Decl.Anonymous_Expr_Decl_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Component_Decl_Range =>
declare
N_Bare_Component_Decl : constant Bare_Component_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Component_Decl.Component_Decl_F_Ids;
                            return;
                        when 2 =>
                            Result := N_Bare_Component_Decl.Component_Decl_F_Component_Def;
                            return;
                        when 3 =>
                            Result := N_Bare_Component_Decl.Component_Decl_F_Default_Expr;
                            return;
                        when 4 =>
                            Result := N_Bare_Component_Decl.Component_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Discriminant_Spec_Range =>
declare
N_Bare_Discriminant_Spec : constant Bare_Discriminant_Spec := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Discriminant_Spec.Discriminant_Spec_F_Ids;
                            return;
                        when 2 =>
                            Result := N_Bare_Discriminant_Spec.Discriminant_Spec_F_Type_Expr;
                            return;
                        when 3 =>
                            Result := N_Bare_Discriminant_Spec.Discriminant_Spec_F_Default_Expr;
                            return;
                        when 4 =>
                            Result := N_Bare_Discriminant_Spec.Discriminant_Spec_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Generic_Formal =>
declare
N_Bare_Generic_Formal : constant Bare_Generic_Formal := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Generic_Formal.Generic_Formal_F_Decl;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Param_Spec_Range =>
declare
N_Bare_Param_Spec : constant Bare_Param_Spec := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Param_Spec.Param_Spec_F_Ids;
                            return;
                        when 2 =>
                            Result := N_Bare_Param_Spec.Param_Spec_F_Has_Aliased;
                            return;
                        when 3 =>
                            Result := N_Bare_Param_Spec.Param_Spec_F_Mode;
                            return;
                        when 4 =>
                            Result := N_Bare_Param_Spec.Param_Spec_F_Type_Expr;
                            return;
                        when 5 =>
                            Result := N_Bare_Param_Spec.Param_Spec_F_Default_Expr;
                            return;
                        when 6 =>
                            Result := N_Bare_Param_Spec.Param_Spec_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Synthetic_Formal_Param_Decl_Range =>
declare
N_Bare_Synthetic_Formal_Param_Decl : constant Bare_Synthetic_Formal_Param_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Synthetic_Formal_Param_Decl.Synthetic_Formal_Param_Decl_F_Param_Type;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Base_Package_Decl =>
declare
N_Bare_Base_Package_Decl : constant Bare_Base_Package_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Base_Package_Decl.Base_Package_Decl_F_Package_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Base_Package_Decl.Base_Package_Decl_F_Aspects;
                            return;
                        when 3 =>
                            Result := N_Bare_Base_Package_Decl.Base_Package_Decl_F_Public_Part;
                            return;
                        when 4 =>
                            Result := N_Bare_Base_Package_Decl.Base_Package_Decl_F_Private_Part;
                            return;
                        when 5 =>
                            Result := N_Bare_Base_Package_Decl.Base_Package_Decl_F_End_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Base_Type_Decl =>
declare
N_Bare_Base_Type_Decl : constant Bare_Base_Type_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Base_Type_Decl.Base_Type_Decl_F_Name;
                            return;
                        when others => null;
                    end case;
case Ada_Base_Type_Decl (K) is
when Ada_Subtype_Decl_Range =>
declare
N_Bare_Subtype_Decl : constant Bare_Subtype_Decl := N_Bare_Base_Type_Decl;
begin
case Index is
                        when 2 =>
                            Result := N_Bare_Subtype_Decl.Subtype_Decl_F_Subtype;
                            return;
                        when 3 =>
                            Result := N_Bare_Subtype_Decl.Subtype_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Incomplete_Type_Decl_Range =>
declare
N_Bare_Incomplete_Type_Decl : constant Bare_Incomplete_Type_Decl := N_Bare_Base_Type_Decl;
begin
case Index is
                        when 2 =>
                            Result := N_Bare_Incomplete_Type_Decl.Incomplete_Type_Decl_F_Discriminants;
                            return;
                        when others => null;
                    end case;
case Ada_Incomplete_Type_Decl_Range (K) is
when Ada_Incomplete_Formal_Type_Decl_Range =>
declare
N_Bare_Incomplete_Formal_Type_Decl : constant Bare_Incomplete_Formal_Type_Decl := N_Bare_Incomplete_Type_Decl;
begin
case Index is
                        when 3 =>
                            Result := N_Bare_Incomplete_Formal_Type_Decl.Incomplete_Formal_Type_Decl_F_Is_Tagged;
                            return;
                        when 4 =>
                            Result := N_Bare_Incomplete_Formal_Type_Decl.Incomplete_Formal_Type_Decl_F_Default_Type;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Incomplete_Tagged_Type_Decl_Range =>
declare
N_Bare_Incomplete_Tagged_Type_Decl : constant Bare_Incomplete_Tagged_Type_Decl := N_Bare_Incomplete_Type_Decl;
begin
case Index is
                        when 3 =>
                            Result := N_Bare_Incomplete_Tagged_Type_Decl.Incomplete_Tagged_Type_Decl_F_Has_Abstract;
                            return;
                        when others => null;
                    end case;
end;
when others => null;
end case;
end;
when Ada_Protected_Type_Decl_Range =>
declare
N_Bare_Protected_Type_Decl : constant Bare_Protected_Type_Decl := N_Bare_Base_Type_Decl;
begin
case Index is
                        when 2 =>
                            Result := N_Bare_Protected_Type_Decl.Protected_Type_Decl_F_Discriminants;
                            return;
                        when 3 =>
                            Result := N_Bare_Protected_Type_Decl.Protected_Type_Decl_F_Aspects;
                            return;
                        when 4 =>
                            Result := N_Bare_Protected_Type_Decl.Protected_Type_Decl_F_Interfaces;
                            return;
                        when 5 =>
                            Result := N_Bare_Protected_Type_Decl.Protected_Type_Decl_F_Definition;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Task_Type_Decl_Range =>
declare
N_Bare_Task_Type_Decl : constant Bare_Task_Type_Decl := N_Bare_Base_Type_Decl;
begin
case Index is
                        when 2 =>
                            Result := N_Bare_Task_Type_Decl.Task_Type_Decl_F_Discriminants;
                            return;
                        when 3 =>
                            Result := N_Bare_Task_Type_Decl.Task_Type_Decl_F_Aspects;
                            return;
                        when 4 =>
                            Result := N_Bare_Task_Type_Decl.Task_Type_Decl_F_Definition;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Type_Decl =>
declare
N_Bare_Type_Decl : constant Bare_Type_Decl := N_Bare_Base_Type_Decl;
begin
case Index is
                        when 2 =>
                            Result := N_Bare_Type_Decl.Type_Decl_F_Discriminants;
                            return;
                        when 3 =>
                            Result := N_Bare_Type_Decl.Type_Decl_F_Type_Def;
                            return;
                        when others => null;
                    end case;
case Ada_Type_Decl (K) is
when Ada_Concrete_Type_Decl_Range =>
declare
N_Bare_Concrete_Type_Decl : constant Bare_Concrete_Type_Decl := N_Bare_Type_Decl;
begin
case Index is
                        when 4 =>
                            Result := N_Bare_Concrete_Type_Decl.Concrete_Type_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Formal_Type_Decl_Range =>
declare
N_Bare_Formal_Type_Decl : constant Bare_Formal_Type_Decl := N_Bare_Type_Decl;
begin
case Index is
                        when 4 =>
                            Result := N_Bare_Formal_Type_Decl.Formal_Type_Decl_F_Default_Type;
                            return;
                        when 5 =>
                            Result := N_Bare_Formal_Type_Decl.Formal_Type_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when others => null;
end case;
end;
when others => null;
end case;
end;
when Ada_Classic_Subp_Decl =>
declare
N_Bare_Classic_Subp_Decl : constant Bare_Classic_Subp_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Classic_Subp_Decl.Classic_Subp_Decl_F_Overriding;
                            return;
                        when 2 =>
                            Result := N_Bare_Classic_Subp_Decl.Classic_Subp_Decl_F_Subp_Spec;
                            return;
                        when others => null;
                    end case;
case Ada_Classic_Subp_Decl (K) is
when Ada_Abstract_Subp_Decl_Range =>
declare
N_Bare_Abstract_Subp_Decl : constant Bare_Abstract_Subp_Decl := N_Bare_Classic_Subp_Decl;
begin
case Index is
                        when 3 =>
                            Result := N_Bare_Abstract_Subp_Decl.Abstract_Subp_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Formal_Subp_Decl =>
declare
N_Bare_Formal_Subp_Decl : constant Bare_Formal_Subp_Decl := N_Bare_Classic_Subp_Decl;
begin
case Index is
                        when 3 =>
                            Result := N_Bare_Formal_Subp_Decl.Formal_Subp_Decl_F_Default_Expr;
                            return;
                        when 4 =>
                            Result := N_Bare_Formal_Subp_Decl.Formal_Subp_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Subp_Decl_Range =>
declare
N_Bare_Subp_Decl : constant Bare_Subp_Decl := N_Bare_Classic_Subp_Decl;
begin
case Index is
                        when 3 =>
                            Result := N_Bare_Subp_Decl.Subp_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when others => null;
end case;
end;
when Ada_Entry_Decl_Range =>
declare
N_Bare_Entry_Decl : constant Bare_Entry_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Entry_Decl.Entry_Decl_F_Overriding;
                            return;
                        when 2 =>
                            Result := N_Bare_Entry_Decl.Entry_Decl_F_Spec;
                            return;
                        when 3 =>
                            Result := N_Bare_Entry_Decl.Entry_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Enum_Literal_Decl_Range =>
declare
N_Bare_Enum_Literal_Decl : constant Bare_Enum_Literal_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Enum_Literal_Decl.Enum_Literal_Decl_F_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Generic_Subp_Internal_Range =>
declare
N_Bare_Generic_Subp_Internal : constant Bare_Generic_Subp_Internal := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Generic_Subp_Internal.Generic_Subp_Internal_F_Subp_Spec;
                            return;
                        when 2 =>
                            Result := N_Bare_Generic_Subp_Internal.Generic_Subp_Internal_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Synthetic_Subp_Decl_Range =>
declare
N_Bare_Synthetic_Subp_Decl : constant Bare_Synthetic_Subp_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Synthetic_Subp_Decl.Synthetic_Subp_Decl_F_Spec;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Base_Subp_Body =>
declare
N_Bare_Base_Subp_Body : constant Bare_Base_Subp_Body := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Base_Subp_Body.Base_Subp_Body_F_Overriding;
                            return;
                        when 2 =>
                            Result := N_Bare_Base_Subp_Body.Base_Subp_Body_F_Subp_Spec;
                            return;
                        when others => null;
                    end case;
case Ada_Base_Subp_Body (K) is
when Ada_Expr_Function_Range =>
declare
N_Bare_Expr_Function : constant Bare_Expr_Function := N_Bare_Base_Subp_Body;
begin
case Index is
                        when 3 =>
                            Result := N_Bare_Expr_Function.Expr_Function_F_Expr;
                            return;
                        when 4 =>
                            Result := N_Bare_Expr_Function.Expr_Function_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Null_Subp_Decl_Range =>
declare
N_Bare_Null_Subp_Decl : constant Bare_Null_Subp_Decl := N_Bare_Base_Subp_Body;
begin
case Index is
                        when 3 =>
                            Result := N_Bare_Null_Subp_Decl.Null_Subp_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Subp_Body_Range =>
declare
N_Bare_Subp_Body : constant Bare_Subp_Body := N_Bare_Base_Subp_Body;
begin
case Index is
                        when 3 =>
                            Result := N_Bare_Subp_Body.Subp_Body_F_Aspects;
                            return;
                        when 4 =>
                            Result := N_Bare_Subp_Body.Subp_Body_F_Decls;
                            return;
                        when 5 =>
                            Result := N_Bare_Subp_Body.Subp_Body_F_Stmts;
                            return;
                        when 6 =>
                            Result := N_Bare_Subp_Body.Subp_Body_F_End_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Subp_Renaming_Decl_Range =>
declare
N_Bare_Subp_Renaming_Decl : constant Bare_Subp_Renaming_Decl := N_Bare_Base_Subp_Body;
begin
case Index is
                        when 3 =>
                            Result := N_Bare_Subp_Renaming_Decl.Subp_Renaming_Decl_F_Renames;
                            return;
                        when 4 =>
                            Result := N_Bare_Subp_Renaming_Decl.Subp_Renaming_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when others => null;
end case;
end;
when Ada_Package_Body_Stub_Range =>
declare
N_Bare_Package_Body_Stub : constant Bare_Package_Body_Stub := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Package_Body_Stub.Package_Body_Stub_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Package_Body_Stub.Package_Body_Stub_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Protected_Body_Stub_Range =>
declare
N_Bare_Protected_Body_Stub : constant Bare_Protected_Body_Stub := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Protected_Body_Stub.Protected_Body_Stub_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Protected_Body_Stub.Protected_Body_Stub_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Subp_Body_Stub_Range =>
declare
N_Bare_Subp_Body_Stub : constant Bare_Subp_Body_Stub := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Subp_Body_Stub.Subp_Body_Stub_F_Overriding;
                            return;
                        when 2 =>
                            Result := N_Bare_Subp_Body_Stub.Subp_Body_Stub_F_Subp_Spec;
                            return;
                        when 3 =>
                            Result := N_Bare_Subp_Body_Stub.Subp_Body_Stub_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Task_Body_Stub_Range =>
declare
N_Bare_Task_Body_Stub : constant Bare_Task_Body_Stub := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Task_Body_Stub.Task_Body_Stub_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Task_Body_Stub.Task_Body_Stub_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Entry_Body_Range =>
declare
N_Bare_Entry_Body : constant Bare_Entry_Body := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Entry_Body.Entry_Body_F_Entry_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Entry_Body.Entry_Body_F_Index_Spec;
                            return;
                        when 3 =>
                            Result := N_Bare_Entry_Body.Entry_Body_F_Params;
                            return;
                        when 4 =>
                            Result := N_Bare_Entry_Body.Entry_Body_F_Aspects;
                            return;
                        when 5 =>
                            Result := N_Bare_Entry_Body.Entry_Body_F_Barrier;
                            return;
                        when 6 =>
                            Result := N_Bare_Entry_Body.Entry_Body_F_Decls;
                            return;
                        when 7 =>
                            Result := N_Bare_Entry_Body.Entry_Body_F_Stmts;
                            return;
                        when 8 =>
                            Result := N_Bare_Entry_Body.Entry_Body_F_End_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Package_Body_Range =>
declare
N_Bare_Package_Body : constant Bare_Package_Body := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Package_Body.Package_Body_F_Package_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Package_Body.Package_Body_F_Aspects;
                            return;
                        when 3 =>
                            Result := N_Bare_Package_Body.Package_Body_F_Decls;
                            return;
                        when 4 =>
                            Result := N_Bare_Package_Body.Package_Body_F_Stmts;
                            return;
                        when 5 =>
                            Result := N_Bare_Package_Body.Package_Body_F_End_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Protected_Body_Range =>
declare
N_Bare_Protected_Body : constant Bare_Protected_Body := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Protected_Body.Protected_Body_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Protected_Body.Protected_Body_F_Aspects;
                            return;
                        when 3 =>
                            Result := N_Bare_Protected_Body.Protected_Body_F_Decls;
                            return;
                        when 4 =>
                            Result := N_Bare_Protected_Body.Protected_Body_F_End_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Task_Body_Range =>
declare
N_Bare_Task_Body : constant Bare_Task_Body := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Task_Body.Task_Body_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Task_Body.Task_Body_F_Aspects;
                            return;
                        when 3 =>
                            Result := N_Bare_Task_Body.Task_Body_F_Decls;
                            return;
                        when 4 =>
                            Result := N_Bare_Task_Body.Task_Body_F_Stmts;
                            return;
                        when 5 =>
                            Result := N_Bare_Task_Body.Task_Body_F_End_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Entry_Index_Spec_Range =>
declare
N_Bare_Entry_Index_Spec : constant Bare_Entry_Index_Spec := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Entry_Index_Spec.Entry_Index_Spec_F_Id;
                            return;
                        when 2 =>
                            Result := N_Bare_Entry_Index_Spec.Entry_Index_Spec_F_Subtype;
                            return;
                        when 3 =>
                            Result := N_Bare_Entry_Index_Spec.Entry_Index_Spec_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Exception_Decl_Range =>
declare
N_Bare_Exception_Decl : constant Bare_Exception_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Exception_Decl.Exception_Decl_F_Ids;
                            return;
                        when 2 =>
                            Result := N_Bare_Exception_Decl.Exception_Decl_F_Renames;
                            return;
                        when 3 =>
                            Result := N_Bare_Exception_Decl.Exception_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Exception_Handler_Range =>
declare
N_Bare_Exception_Handler : constant Bare_Exception_Handler := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Exception_Handler.Exception_Handler_F_Exception_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Exception_Handler.Exception_Handler_F_Handled_Exceptions;
                            return;
                        when 3 =>
                            Result := N_Bare_Exception_Handler.Exception_Handler_F_Stmts;
                            return;
                        when others => null;
                    end case;
end;
when Ada_For_Loop_Var_Decl_Range =>
declare
N_Bare_For_Loop_Var_Decl : constant Bare_For_Loop_Var_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_For_Loop_Var_Decl.For_Loop_Var_Decl_F_Id;
                            return;
                        when 2 =>
                            Result := N_Bare_For_Loop_Var_Decl.For_Loop_Var_Decl_F_Id_Type;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Generic_Decl =>
declare
N_Bare_Generic_Decl : constant Bare_Generic_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Generic_Decl.Generic_Decl_F_Formal_Part;
                            return;
                        when others => null;
                    end case;
case Ada_Generic_Decl (K) is
when Ada_Generic_Package_Decl_Range =>
declare
N_Bare_Generic_Package_Decl : constant Bare_Generic_Package_Decl := N_Bare_Generic_Decl;
begin
case Index is
                        when 2 =>
                            Result := N_Bare_Generic_Package_Decl.Generic_Package_Decl_F_Package_Decl;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Generic_Subp_Decl_Range =>
declare
N_Bare_Generic_Subp_Decl : constant Bare_Generic_Subp_Decl := N_Bare_Generic_Decl;
begin
case Index is
                        when 2 =>
                            Result := N_Bare_Generic_Subp_Decl.Generic_Subp_Decl_F_Subp_Decl;
                            return;
                        when others => null;
                    end case;
end;
when others => null;
end case;
end;
when Ada_Generic_Package_Instantiation_Range =>
declare
N_Bare_Generic_Package_Instantiation : constant Bare_Generic_Package_Instantiation := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Generic_Package_Instantiation.Generic_Package_Instantiation_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Generic_Package_Instantiation.Generic_Package_Instantiation_F_Generic_Pkg_Name;
                            return;
                        when 3 =>
                            Result := N_Bare_Generic_Package_Instantiation.Generic_Package_Instantiation_F_Params;
                            return;
                        when 4 =>
                            Result := N_Bare_Generic_Package_Instantiation.Generic_Package_Instantiation_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Generic_Subp_Instantiation_Range =>
declare
N_Bare_Generic_Subp_Instantiation : constant Bare_Generic_Subp_Instantiation := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Generic_Subp_Instantiation.Generic_Subp_Instantiation_F_Overriding;
                            return;
                        when 2 =>
                            Result := N_Bare_Generic_Subp_Instantiation.Generic_Subp_Instantiation_F_Kind;
                            return;
                        when 3 =>
                            Result := N_Bare_Generic_Subp_Instantiation.Generic_Subp_Instantiation_F_Subp_Name;
                            return;
                        when 4 =>
                            Result := N_Bare_Generic_Subp_Instantiation.Generic_Subp_Instantiation_F_Generic_Subp_Name;
                            return;
                        when 5 =>
                            Result := N_Bare_Generic_Subp_Instantiation.Generic_Subp_Instantiation_F_Params;
                            return;
                        when 6 =>
                            Result := N_Bare_Generic_Subp_Instantiation.Generic_Subp_Instantiation_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Generic_Package_Renaming_Decl_Range =>
declare
N_Bare_Generic_Package_Renaming_Decl : constant Bare_Generic_Package_Renaming_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Generic_Package_Renaming_Decl.Generic_Package_Renaming_Decl_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Generic_Package_Renaming_Decl.Generic_Package_Renaming_Decl_F_Renames;
                            return;
                        when 3 =>
                            Result := N_Bare_Generic_Package_Renaming_Decl.Generic_Package_Renaming_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Generic_Subp_Renaming_Decl_Range =>
declare
N_Bare_Generic_Subp_Renaming_Decl : constant Bare_Generic_Subp_Renaming_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Generic_Subp_Renaming_Decl.Generic_Subp_Renaming_Decl_F_Kind;
                            return;
                        when 2 =>
                            Result := N_Bare_Generic_Subp_Renaming_Decl.Generic_Subp_Renaming_Decl_F_Name;
                            return;
                        when 3 =>
                            Result := N_Bare_Generic_Subp_Renaming_Decl.Generic_Subp_Renaming_Decl_F_Renames;
                            return;
                        when 4 =>
                            Result := N_Bare_Generic_Subp_Renaming_Decl.Generic_Subp_Renaming_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Label_Decl_Range =>
declare
N_Bare_Label_Decl : constant Bare_Label_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Label_Decl.Label_Decl_F_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Named_Stmt_Decl_Range =>
declare
N_Bare_Named_Stmt_Decl : constant Bare_Named_Stmt_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Named_Stmt_Decl.Named_Stmt_Decl_F_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Number_Decl_Range =>
declare
N_Bare_Number_Decl : constant Bare_Number_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Number_Decl.Number_Decl_F_Ids;
                            return;
                        when 2 =>
                            Result := N_Bare_Number_Decl.Number_Decl_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Object_Decl_Range =>
declare
N_Bare_Object_Decl : constant Bare_Object_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Object_Decl.Object_Decl_F_Ids;
                            return;
                        when 2 =>
                            Result := N_Bare_Object_Decl.Object_Decl_F_Has_Aliased;
                            return;
                        when 3 =>
                            Result := N_Bare_Object_Decl.Object_Decl_F_Has_Constant;
                            return;
                        when 4 =>
                            Result := N_Bare_Object_Decl.Object_Decl_F_Mode;
                            return;
                        when 5 =>
                            Result := N_Bare_Object_Decl.Object_Decl_F_Type_Expr;
                            return;
                        when 6 =>
                            Result := N_Bare_Object_Decl.Object_Decl_F_Default_Expr;
                            return;
                        when 7 =>
                            Result := N_Bare_Object_Decl.Object_Decl_F_Renaming_Clause;
                            return;
                        when 8 =>
                            Result := N_Bare_Object_Decl.Object_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Package_Renaming_Decl_Range =>
declare
N_Bare_Package_Renaming_Decl : constant Bare_Package_Renaming_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Package_Renaming_Decl.Package_Renaming_Decl_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Package_Renaming_Decl.Package_Renaming_Decl_F_Renames;
                            return;
                        when 3 =>
                            Result := N_Bare_Package_Renaming_Decl.Package_Renaming_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Single_Protected_Decl_Range =>
declare
N_Bare_Single_Protected_Decl : constant Bare_Single_Protected_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Single_Protected_Decl.Single_Protected_Decl_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Single_Protected_Decl.Single_Protected_Decl_F_Aspects;
                            return;
                        when 3 =>
                            Result := N_Bare_Single_Protected_Decl.Single_Protected_Decl_F_Interfaces;
                            return;
                        when 4 =>
                            Result := N_Bare_Single_Protected_Decl.Single_Protected_Decl_F_Definition;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Single_Task_Decl_Range =>
declare
N_Bare_Single_Task_Decl : constant Bare_Single_Task_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Single_Task_Decl.Single_Task_Decl_F_Task_Type;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Case_Stmt_Alternative_Range =>
declare
N_Bare_Case_Stmt_Alternative : constant Bare_Case_Stmt_Alternative := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Case_Stmt_Alternative.Case_Stmt_Alternative_F_Choices;
                            return;
                        when 2 =>
                            Result := N_Bare_Case_Stmt_Alternative.Case_Stmt_Alternative_F_Stmts;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Compilation_Unit_Range =>
declare
N_Bare_Compilation_Unit : constant Bare_Compilation_Unit := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Compilation_Unit.Compilation_Unit_F_Prelude;
                            return;
                        when 2 =>
                            Result := N_Bare_Compilation_Unit.Compilation_Unit_F_Body;
                            return;
                        when 3 =>
                            Result := N_Bare_Compilation_Unit.Compilation_Unit_F_Pragmas;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Component_Clause_Range =>
declare
N_Bare_Component_Clause : constant Bare_Component_Clause := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Component_Clause.Component_Clause_F_Id;
                            return;
                        when 2 =>
                            Result := N_Bare_Component_Clause.Component_Clause_F_Position;
                            return;
                        when 3 =>
                            Result := N_Bare_Component_Clause.Component_Clause_F_Range;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Component_Def_Range =>
declare
N_Bare_Component_Def : constant Bare_Component_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Component_Def.Component_Def_F_Has_Aliased;
                            return;
                        when 2 =>
                            Result := N_Bare_Component_Def.Component_Def_F_Has_Constant;
                            return;
                        when 3 =>
                            Result := N_Bare_Component_Def.Component_Def_F_Type_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Composite_Constraint_Range =>
declare
N_Bare_Composite_Constraint : constant Bare_Composite_Constraint := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Composite_Constraint.Composite_Constraint_F_Constraints;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Delta_Constraint_Range =>
declare
N_Bare_Delta_Constraint : constant Bare_Delta_Constraint := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Delta_Constraint.Delta_Constraint_F_Digits;
                            return;
                        when 2 =>
                            Result := N_Bare_Delta_Constraint.Delta_Constraint_F_Range;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Digits_Constraint_Range =>
declare
N_Bare_Digits_Constraint : constant Bare_Digits_Constraint := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Digits_Constraint.Digits_Constraint_F_Digits;
                            return;
                        when 2 =>
                            Result := N_Bare_Digits_Constraint.Digits_Constraint_F_Range;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Range_Constraint_Range =>
declare
N_Bare_Range_Constraint : constant Bare_Range_Constraint := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Range_Constraint.Range_Constraint_F_Range;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Declarative_Part_Range =>
declare
N_Bare_Declarative_Part : constant Bare_Declarative_Part := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Declarative_Part.Declarative_Part_F_Decls;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Elsif_Expr_Part_Range =>
declare
N_Bare_Elsif_Expr_Part : constant Bare_Elsif_Expr_Part := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Elsif_Expr_Part.Elsif_Expr_Part_F_Cond_Expr;
                            return;
                        when 2 =>
                            Result := N_Bare_Elsif_Expr_Part.Elsif_Expr_Part_F_Then_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Elsif_Stmt_Part_Range =>
declare
N_Bare_Elsif_Stmt_Part : constant Bare_Elsif_Stmt_Part := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Elsif_Stmt_Part.Elsif_Stmt_Part_F_Cond_Expr;
                            return;
                        when 2 =>
                            Result := N_Bare_Elsif_Stmt_Part.Elsif_Stmt_Part_F_Stmts;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Abstract_State_Decl_Expr_Range =>
declare
N_Bare_Abstract_State_Decl_Expr : constant Bare_Abstract_State_Decl_Expr := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Abstract_State_Decl_Expr.Abstract_State_Decl_Expr_F_State_Decl;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Allocator_Range =>
declare
N_Bare_Allocator : constant Bare_Allocator := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Allocator.Allocator_F_Subpool;
                            return;
                        when 2 =>
                            Result := N_Bare_Allocator.Allocator_F_Type_Or_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Base_Aggregate =>
declare
N_Bare_Base_Aggregate : constant Bare_Base_Aggregate := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Base_Aggregate.Base_Aggregate_F_Ancestor_Expr;
                            return;
                        when 2 =>
                            Result := N_Bare_Base_Aggregate.Base_Aggregate_F_Assocs;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Bin_Op_Range =>
declare
N_Bare_Bin_Op : constant Bare_Bin_Op := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Bin_Op.Bin_Op_F_Left;
                            return;
                        when 2 =>
                            Result := N_Bare_Bin_Op.Bin_Op_F_Op;
                            return;
                        when 3 =>
                            Result := N_Bare_Bin_Op.Bin_Op_F_Right;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Case_Expr_Alternative_Range =>
declare
N_Bare_Case_Expr_Alternative : constant Bare_Case_Expr_Alternative := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Case_Expr_Alternative.Case_Expr_Alternative_F_Choices;
                            return;
                        when 2 =>
                            Result := N_Bare_Case_Expr_Alternative.Case_Expr_Alternative_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Concat_Op_Range =>
declare
N_Bare_Concat_Op : constant Bare_Concat_Op := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Concat_Op.Concat_Op_F_First_Operand;
                            return;
                        when 2 =>
                            Result := N_Bare_Concat_Op.Concat_Op_F_Other_Operands;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Concat_Operand_Range =>
declare
N_Bare_Concat_Operand : constant Bare_Concat_Operand := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Concat_Operand.Concat_Operand_F_Operator;
                            return;
                        when 2 =>
                            Result := N_Bare_Concat_Operand.Concat_Operand_F_Operand;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Case_Expr_Range =>
declare
N_Bare_Case_Expr : constant Bare_Case_Expr := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Case_Expr.Case_Expr_F_Expr;
                            return;
                        when 2 =>
                            Result := N_Bare_Case_Expr.Case_Expr_F_Cases;
                            return;
                        when others => null;
                    end case;
end;
when Ada_If_Expr_Range =>
declare
N_Bare_If_Expr : constant Bare_If_Expr := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_If_Expr.If_Expr_F_Cond_Expr;
                            return;
                        when 2 =>
                            Result := N_Bare_If_Expr.If_Expr_F_Then_Expr;
                            return;
                        when 3 =>
                            Result := N_Bare_If_Expr.If_Expr_F_Alternatives;
                            return;
                        when 4 =>
                            Result := N_Bare_If_Expr.If_Expr_F_Else_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Contract_Cases_Range =>
declare
N_Bare_Contract_Cases : constant Bare_Contract_Cases := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Contract_Cases.Contract_Cases_F_Contract_Cases;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Decl_Expr_Range =>
declare
N_Bare_Decl_Expr : constant Bare_Decl_Expr := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Decl_Expr.Decl_Expr_F_Decls;
                            return;
                        when 2 =>
                            Result := N_Bare_Decl_Expr.Decl_Expr_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Membership_Expr_Range =>
declare
N_Bare_Membership_Expr : constant Bare_Membership_Expr := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Membership_Expr.Membership_Expr_F_Expr;
                            return;
                        when 2 =>
                            Result := N_Bare_Membership_Expr.Membership_Expr_F_Op;
                            return;
                        when 3 =>
                            Result := N_Bare_Membership_Expr.Membership_Expr_F_Membership_Exprs;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Attribute_Ref_Range =>
declare
N_Bare_Attribute_Ref : constant Bare_Attribute_Ref := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Attribute_Ref.Attribute_Ref_F_Prefix;
                            return;
                        when 2 =>
                            Result := N_Bare_Attribute_Ref.Attribute_Ref_F_Attribute;
                            return;
                        when 3 =>
                            Result := N_Bare_Attribute_Ref.Attribute_Ref_F_Args;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Call_Expr_Range =>
declare
N_Bare_Call_Expr : constant Bare_Call_Expr := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Call_Expr.Call_Expr_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Call_Expr.Call_Expr_F_Suffix;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Defining_Name_Range =>
declare
N_Bare_Defining_Name : constant Bare_Defining_Name := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Defining_Name.Defining_Name_F_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Discrete_Subtype_Name_Range =>
declare
N_Bare_Discrete_Subtype_Name : constant Bare_Discrete_Subtype_Name := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Discrete_Subtype_Name.Discrete_Subtype_Name_F_Subtype;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Dotted_Name_Range =>
declare
N_Bare_Dotted_Name : constant Bare_Dotted_Name := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Dotted_Name.Dotted_Name_F_Prefix;
                            return;
                        when 2 =>
                            Result := N_Bare_Dotted_Name.Dotted_Name_F_Suffix;
                            return;
                        when others => null;
                    end case;
end;
when Ada_End_Name_Range =>
declare
N_Bare_End_Name : constant Bare_End_Name := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_End_Name.End_Name_F_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Explicit_Deref_Range =>
declare
N_Bare_Explicit_Deref : constant Bare_Explicit_Deref := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Explicit_Deref.Explicit_Deref_F_Prefix;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Qual_Expr_Range =>
declare
N_Bare_Qual_Expr : constant Bare_Qual_Expr := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Qual_Expr.Qual_Expr_F_Prefix;
                            return;
                        when 2 =>
                            Result := N_Bare_Qual_Expr.Qual_Expr_F_Suffix;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Reduce_Attribute_Ref_Range =>
declare
N_Bare_Reduce_Attribute_Ref : constant Bare_Reduce_Attribute_Ref := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Reduce_Attribute_Ref.Reduce_Attribute_Ref_F_Prefix;
                            return;
                        when 2 =>
                            Result := N_Bare_Reduce_Attribute_Ref.Reduce_Attribute_Ref_F_Attribute;
                            return;
                        when 3 =>
                            Result := N_Bare_Reduce_Attribute_Ref.Reduce_Attribute_Ref_F_Args;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Update_Attribute_Ref_Range =>
declare
N_Bare_Update_Attribute_Ref : constant Bare_Update_Attribute_Ref := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Update_Attribute_Ref.Update_Attribute_Ref_F_Prefix;
                            return;
                        when 2 =>
                            Result := N_Bare_Update_Attribute_Ref.Update_Attribute_Ref_F_Attribute;
                            return;
                        when 3 =>
                            Result := N_Bare_Update_Attribute_Ref.Update_Attribute_Ref_F_Values;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Paren_Expr_Range =>
declare
N_Bare_Paren_Expr : constant Bare_Paren_Expr := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Paren_Expr.Paren_Expr_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Quantified_Expr_Range =>
declare
N_Bare_Quantified_Expr : constant Bare_Quantified_Expr := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Quantified_Expr.Quantified_Expr_F_Quantifier;
                            return;
                        when 2 =>
                            Result := N_Bare_Quantified_Expr.Quantified_Expr_F_Loop_Spec;
                            return;
                        when 3 =>
                            Result := N_Bare_Quantified_Expr.Quantified_Expr_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Raise_Expr_Range =>
declare
N_Bare_Raise_Expr : constant Bare_Raise_Expr := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Raise_Expr.Raise_Expr_F_Exception_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Raise_Expr.Raise_Expr_F_Error_Message;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Un_Op_Range =>
declare
N_Bare_Un_Op : constant Bare_Un_Op := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Un_Op.Un_Op_F_Op;
                            return;
                        when 2 =>
                            Result := N_Bare_Un_Op.Un_Op_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Handled_Stmts_Range =>
declare
N_Bare_Handled_Stmts : constant Bare_Handled_Stmts := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Handled_Stmts.Handled_Stmts_F_Stmts;
                            return;
                        when 2 =>
                            Result := N_Bare_Handled_Stmts.Handled_Stmts_F_Exceptions;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Library_Item_Range =>
declare
N_Bare_Library_Item : constant Bare_Library_Item := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Library_Item.Library_Item_F_Has_Private;
                            return;
                        when 2 =>
                            Result := N_Bare_Library_Item.Library_Item_F_Item;
                            return;
                        when others => null;
                    end case;
end;
when Ada_For_Loop_Spec_Range =>
declare
N_Bare_For_Loop_Spec : constant Bare_For_Loop_Spec := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_For_Loop_Spec.For_Loop_Spec_F_Var_Decl;
                            return;
                        when 2 =>
                            Result := N_Bare_For_Loop_Spec.For_Loop_Spec_F_Loop_Type;
                            return;
                        when 3 =>
                            Result := N_Bare_For_Loop_Spec.For_Loop_Spec_F_Has_Reverse;
                            return;
                        when 4 =>
                            Result := N_Bare_For_Loop_Spec.For_Loop_Spec_F_Iter_Expr;
                            return;
                        when 5 =>
                            Result := N_Bare_For_Loop_Spec.For_Loop_Spec_F_Iter_Filter;
                            return;
                        when others => null;
                    end case;
end;
when Ada_While_Loop_Spec_Range =>
declare
N_Bare_While_Loop_Spec : constant Bare_While_Loop_Spec := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_While_Loop_Spec.While_Loop_Spec_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Multi_Abstract_State_Decl_Range =>
declare
N_Bare_Multi_Abstract_State_Decl : constant Bare_Multi_Abstract_State_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Multi_Abstract_State_Decl.Multi_Abstract_State_Decl_F_Decls;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Params_Range =>
declare
N_Bare_Params : constant Bare_Params := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Params.Params_F_Params;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Paren_Abstract_State_Decl_Range =>
declare
N_Bare_Paren_Abstract_State_Decl : constant Bare_Paren_Abstract_State_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Paren_Abstract_State_Decl.Paren_Abstract_State_Decl_F_Decl;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Pp_Elsif_Directive_Range =>
declare
N_Bare_Pp_Elsif_Directive : constant Bare_Pp_Elsif_Directive := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Pp_Elsif_Directive.Pp_Elsif_Directive_F_Expr;
                            return;
                        when 2 =>
                            Result := N_Bare_Pp_Elsif_Directive.Pp_Elsif_Directive_F_Then_Kw;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Pp_If_Directive_Range =>
declare
N_Bare_Pp_If_Directive : constant Bare_Pp_If_Directive := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Pp_If_Directive.Pp_If_Directive_F_Expr;
                            return;
                        when 2 =>
                            Result := N_Bare_Pp_If_Directive.Pp_If_Directive_F_Then_Kw;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Pragma_Node_Range =>
declare
N_Bare_Pragma_Node : constant Bare_Pragma_Node := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Pragma_Node.Pragma_Node_F_Id;
                            return;
                        when 2 =>
                            Result := N_Bare_Pragma_Node.Pragma_Node_F_Args;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Protected_Def_Range =>
declare
N_Bare_Protected_Def : constant Bare_Protected_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Protected_Def.Protected_Def_F_Public_Part;
                            return;
                        when 2 =>
                            Result := N_Bare_Protected_Def.Protected_Def_F_Private_Part;
                            return;
                        when 3 =>
                            Result := N_Bare_Protected_Def.Protected_Def_F_End_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Range_Spec_Range =>
declare
N_Bare_Range_Spec : constant Bare_Range_Spec := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Range_Spec.Range_Spec_F_Range;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Renaming_Clause_Range =>
declare
N_Bare_Renaming_Clause : constant Bare_Renaming_Clause := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Renaming_Clause.Renaming_Clause_F_Renamed_Object;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Select_When_Part_Range =>
declare
N_Bare_Select_When_Part : constant Bare_Select_When_Part := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Select_When_Part.Select_When_Part_F_Cond_Expr;
                            return;
                        when 2 =>
                            Result := N_Bare_Select_When_Part.Select_When_Part_F_Stmts;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Accept_Stmt_Range =>
declare
N_Bare_Accept_Stmt : constant Bare_Accept_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Accept_Stmt.Accept_Stmt_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Accept_Stmt.Accept_Stmt_F_Entry_Index_Expr;
                            return;
                        when 3 =>
                            Result := N_Bare_Accept_Stmt.Accept_Stmt_F_Params;
                            return;
                        when others => null;
                    end case;
case Ada_Accept_Stmt_Range (K) is
when Ada_Accept_Stmt_With_Stmts_Range =>
declare
N_Bare_Accept_Stmt_With_Stmts : constant Bare_Accept_Stmt_With_Stmts := N_Bare_Accept_Stmt;
begin
case Index is
                        when 4 =>
                            Result := N_Bare_Accept_Stmt_With_Stmts.Accept_Stmt_With_Stmts_F_Stmts;
                            return;
                        when 5 =>
                            Result := N_Bare_Accept_Stmt_With_Stmts.Accept_Stmt_With_Stmts_F_End_Name;
                            return;
                        when others => null;
                    end case;
end;
when others => null;
end case;
end;
when Ada_Base_Loop_Stmt =>
declare
N_Bare_Base_Loop_Stmt : constant Bare_Base_Loop_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Base_Loop_Stmt.Base_Loop_Stmt_F_Spec;
                            return;
                        when 2 =>
                            Result := N_Bare_Base_Loop_Stmt.Base_Loop_Stmt_F_Stmts;
                            return;
                        when 3 =>
                            Result := N_Bare_Base_Loop_Stmt.Base_Loop_Stmt_F_End_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Begin_Block_Range =>
declare
N_Bare_Begin_Block : constant Bare_Begin_Block := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Begin_Block.Begin_Block_F_Stmts;
                            return;
                        when 2 =>
                            Result := N_Bare_Begin_Block.Begin_Block_F_End_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Decl_Block_Range =>
declare
N_Bare_Decl_Block : constant Bare_Decl_Block := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Decl_Block.Decl_Block_F_Decls;
                            return;
                        when 2 =>
                            Result := N_Bare_Decl_Block.Decl_Block_F_Stmts;
                            return;
                        when 3 =>
                            Result := N_Bare_Decl_Block.Decl_Block_F_End_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Case_Stmt_Range =>
declare
N_Bare_Case_Stmt : constant Bare_Case_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Case_Stmt.Case_Stmt_F_Expr;
                            return;
                        when 2 =>
                            Result := N_Bare_Case_Stmt.Case_Stmt_F_Pragmas;
                            return;
                        when 3 =>
                            Result := N_Bare_Case_Stmt.Case_Stmt_F_Alternatives;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Extended_Return_Stmt_Range =>
declare
N_Bare_Extended_Return_Stmt : constant Bare_Extended_Return_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Extended_Return_Stmt.Extended_Return_Stmt_F_Decl;
                            return;
                        when 2 =>
                            Result := N_Bare_Extended_Return_Stmt.Extended_Return_Stmt_F_Stmts;
                            return;
                        when others => null;
                    end case;
end;
when Ada_If_Stmt_Range =>
declare
N_Bare_If_Stmt : constant Bare_If_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_If_Stmt.If_Stmt_F_Cond_Expr;
                            return;
                        when 2 =>
                            Result := N_Bare_If_Stmt.If_Stmt_F_Then_Stmts;
                            return;
                        when 3 =>
                            Result := N_Bare_If_Stmt.If_Stmt_F_Alternatives;
                            return;
                        when 4 =>
                            Result := N_Bare_If_Stmt.If_Stmt_F_Else_Stmts;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Named_Stmt_Range =>
declare
N_Bare_Named_Stmt : constant Bare_Named_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Named_Stmt.Named_Stmt_F_Decl;
                            return;
                        when 2 =>
                            Result := N_Bare_Named_Stmt.Named_Stmt_F_Stmt;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Select_Stmt_Range =>
declare
N_Bare_Select_Stmt : constant Bare_Select_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Select_Stmt.Select_Stmt_F_Guards;
                            return;
                        when 2 =>
                            Result := N_Bare_Select_Stmt.Select_Stmt_F_Else_Stmts;
                            return;
                        when 3 =>
                            Result := N_Bare_Select_Stmt.Select_Stmt_F_Abort_Stmts;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Abort_Stmt_Range =>
declare
N_Bare_Abort_Stmt : constant Bare_Abort_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Abort_Stmt.Abort_Stmt_F_Names;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Assign_Stmt_Range =>
declare
N_Bare_Assign_Stmt : constant Bare_Assign_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Assign_Stmt.Assign_Stmt_F_Dest;
                            return;
                        when 2 =>
                            Result := N_Bare_Assign_Stmt.Assign_Stmt_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Call_Stmt_Range =>
declare
N_Bare_Call_Stmt : constant Bare_Call_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Call_Stmt.Call_Stmt_F_Call;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Delay_Stmt_Range =>
declare
N_Bare_Delay_Stmt : constant Bare_Delay_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Delay_Stmt.Delay_Stmt_F_Has_Until;
                            return;
                        when 2 =>
                            Result := N_Bare_Delay_Stmt.Delay_Stmt_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Exit_Stmt_Range =>
declare
N_Bare_Exit_Stmt : constant Bare_Exit_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Exit_Stmt.Exit_Stmt_F_Loop_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Exit_Stmt.Exit_Stmt_F_Cond_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Goto_Stmt_Range =>
declare
N_Bare_Goto_Stmt : constant Bare_Goto_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Goto_Stmt.Goto_Stmt_F_Label_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Label_Range =>
declare
N_Bare_Label : constant Bare_Label := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Label.Label_F_Decl;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Raise_Stmt_Range =>
declare
N_Bare_Raise_Stmt : constant Bare_Raise_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Raise_Stmt.Raise_Stmt_F_Exception_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Raise_Stmt.Raise_Stmt_F_Error_Message;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Requeue_Stmt_Range =>
declare
N_Bare_Requeue_Stmt : constant Bare_Requeue_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Requeue_Stmt.Requeue_Stmt_F_Call_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Requeue_Stmt.Requeue_Stmt_F_Has_Abort;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Return_Stmt_Range =>
declare
N_Bare_Return_Stmt : constant Bare_Return_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Return_Stmt.Return_Stmt_F_Return_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Subunit_Range =>
declare
N_Bare_Subunit : constant Bare_Subunit := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Subunit.Subunit_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Subunit.Subunit_F_Body;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Task_Def_Range =>
declare
N_Bare_Task_Def : constant Bare_Task_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Task_Def.Task_Def_F_Interfaces;
                            return;
                        when 2 =>
                            Result := N_Bare_Task_Def.Task_Def_F_Public_Part;
                            return;
                        when 3 =>
                            Result := N_Bare_Task_Def.Task_Def_F_Private_Part;
                            return;
                        when 4 =>
                            Result := N_Bare_Task_Def.Task_Def_F_End_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Access_Def =>
declare
N_Bare_Access_Def : constant Bare_Access_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Access_Def.Access_Def_F_Has_Not_Null;
                            return;
                        when others => null;
                    end case;
case Ada_Access_Def (K) is
when Ada_Access_To_Subp_Def_Range =>
declare
N_Bare_Access_To_Subp_Def : constant Bare_Access_To_Subp_Def := N_Bare_Access_Def;
begin
case Index is
                        when 2 =>
                            Result := N_Bare_Access_To_Subp_Def.Access_To_Subp_Def_F_Has_Protected;
                            return;
                        when 3 =>
                            Result := N_Bare_Access_To_Subp_Def.Access_To_Subp_Def_F_Subp_Spec;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Anonymous_Type_Access_Def_Range =>
declare
N_Bare_Anonymous_Type_Access_Def : constant Bare_Anonymous_Type_Access_Def := N_Bare_Access_Def;
begin
case Index is
                        when 2 =>
                            Result := N_Bare_Anonymous_Type_Access_Def.Anonymous_Type_Access_Def_F_Type_Decl;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Type_Access_Def_Range =>
declare
N_Bare_Type_Access_Def : constant Bare_Type_Access_Def := N_Bare_Access_Def;
begin
case Index is
                        when 2 =>
                            Result := N_Bare_Type_Access_Def.Type_Access_Def_F_Has_All;
                            return;
                        when 3 =>
                            Result := N_Bare_Type_Access_Def.Type_Access_Def_F_Has_Constant;
                            return;
                        when 4 =>
                            Result := N_Bare_Type_Access_Def.Type_Access_Def_F_Subtype_Indication;
                            return;
                        when others => null;
                    end case;
end;
when others => null;
end case;
end;
when Ada_Array_Type_Def_Range =>
declare
N_Bare_Array_Type_Def : constant Bare_Array_Type_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Array_Type_Def.Array_Type_Def_F_Indices;
                            return;
                        when 2 =>
                            Result := N_Bare_Array_Type_Def.Array_Type_Def_F_Component_Type;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Derived_Type_Def_Range =>
declare
N_Bare_Derived_Type_Def : constant Bare_Derived_Type_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Derived_Type_Def.Derived_Type_Def_F_Has_Abstract;
                            return;
                        when 2 =>
                            Result := N_Bare_Derived_Type_Def.Derived_Type_Def_F_Has_Limited;
                            return;
                        when 3 =>
                            Result := N_Bare_Derived_Type_Def.Derived_Type_Def_F_Has_Synchronized;
                            return;
                        when 4 =>
                            Result := N_Bare_Derived_Type_Def.Derived_Type_Def_F_Subtype_Indication;
                            return;
                        when 5 =>
                            Result := N_Bare_Derived_Type_Def.Derived_Type_Def_F_Interfaces;
                            return;
                        when 6 =>
                            Result := N_Bare_Derived_Type_Def.Derived_Type_Def_F_Record_Extension;
                            return;
                        when 7 =>
                            Result := N_Bare_Derived_Type_Def.Derived_Type_Def_F_Has_With_Private;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Enum_Type_Def_Range =>
declare
N_Bare_Enum_Type_Def : constant Bare_Enum_Type_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Enum_Type_Def.Enum_Type_Def_F_Enum_Literals;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Interface_Type_Def_Range =>
declare
N_Bare_Interface_Type_Def : constant Bare_Interface_Type_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Interface_Type_Def.Interface_Type_Def_F_Interface_Kind;
                            return;
                        when 2 =>
                            Result := N_Bare_Interface_Type_Def.Interface_Type_Def_F_Interfaces;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Mod_Int_Type_Def_Range =>
declare
N_Bare_Mod_Int_Type_Def : constant Bare_Mod_Int_Type_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Mod_Int_Type_Def.Mod_Int_Type_Def_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Private_Type_Def_Range =>
declare
N_Bare_Private_Type_Def : constant Bare_Private_Type_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Private_Type_Def.Private_Type_Def_F_Has_Abstract;
                            return;
                        when 2 =>
                            Result := N_Bare_Private_Type_Def.Private_Type_Def_F_Has_Tagged;
                            return;
                        when 3 =>
                            Result := N_Bare_Private_Type_Def.Private_Type_Def_F_Has_Limited;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Decimal_Fixed_Point_Def_Range =>
declare
N_Bare_Decimal_Fixed_Point_Def : constant Bare_Decimal_Fixed_Point_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Decimal_Fixed_Point_Def.Decimal_Fixed_Point_Def_F_Delta;
                            return;
                        when 2 =>
                            Result := N_Bare_Decimal_Fixed_Point_Def.Decimal_Fixed_Point_Def_F_Digits;
                            return;
                        when 3 =>
                            Result := N_Bare_Decimal_Fixed_Point_Def.Decimal_Fixed_Point_Def_F_Range;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Floating_Point_Def_Range =>
declare
N_Bare_Floating_Point_Def : constant Bare_Floating_Point_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Floating_Point_Def.Floating_Point_Def_F_Num_Digits;
                            return;
                        when 2 =>
                            Result := N_Bare_Floating_Point_Def.Floating_Point_Def_F_Range;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Ordinary_Fixed_Point_Def_Range =>
declare
N_Bare_Ordinary_Fixed_Point_Def : constant Bare_Ordinary_Fixed_Point_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Ordinary_Fixed_Point_Def.Ordinary_Fixed_Point_Def_F_Delta;
                            return;
                        when 2 =>
                            Result := N_Bare_Ordinary_Fixed_Point_Def.Ordinary_Fixed_Point_Def_F_Range;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Record_Type_Def_Range =>
declare
N_Bare_Record_Type_Def : constant Bare_Record_Type_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Record_Type_Def.Record_Type_Def_F_Has_Abstract;
                            return;
                        when 2 =>
                            Result := N_Bare_Record_Type_Def.Record_Type_Def_F_Has_Tagged;
                            return;
                        when 3 =>
                            Result := N_Bare_Record_Type_Def.Record_Type_Def_F_Has_Limited;
                            return;
                        when 4 =>
                            Result := N_Bare_Record_Type_Def.Record_Type_Def_F_Record_Def;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Signed_Int_Type_Def_Range =>
declare
N_Bare_Signed_Int_Type_Def : constant Bare_Signed_Int_Type_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Signed_Int_Type_Def.Signed_Int_Type_Def_F_Range;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Anonymous_Type_Range =>
declare
N_Bare_Anonymous_Type : constant Bare_Anonymous_Type := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Anonymous_Type.Anonymous_Type_F_Type_Decl;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Subtype_Indication_Range =>
declare
N_Bare_Subtype_Indication : constant Bare_Subtype_Indication := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Subtype_Indication.Subtype_Indication_F_Has_Not_Null;
                            return;
                        when 2 =>
                            Result := N_Bare_Subtype_Indication.Subtype_Indication_F_Name;
                            return;
                        when 3 =>
                            Result := N_Bare_Subtype_Indication.Subtype_Indication_F_Constraint;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Synthetic_Type_Expr_Range =>
declare
N_Bare_Synthetic_Type_Expr : constant Bare_Synthetic_Type_Expr := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Synthetic_Type_Expr.Synthetic_Type_Expr_F_Target_Type;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Unconstrained_Array_Index_Range =>
declare
N_Bare_Unconstrained_Array_Index : constant Bare_Unconstrained_Array_Index := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Unconstrained_Array_Index.Unconstrained_Array_Index_F_Subtype_Indication;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Use_Package_Clause_Range =>
declare
N_Bare_Use_Package_Clause : constant Bare_Use_Package_Clause := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Use_Package_Clause.Use_Package_Clause_F_Packages;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Use_Type_Clause_Range =>
declare
N_Bare_Use_Type_Clause : constant Bare_Use_Type_Clause := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Use_Type_Clause.Use_Type_Clause_F_Has_All;
                            return;
                        when 2 =>
                            Result := N_Bare_Use_Type_Clause.Use_Type_Clause_F_Types;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Value_Sequence_Range =>
declare
N_Bare_Value_Sequence : constant Bare_Value_Sequence := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Value_Sequence.Value_Sequence_F_Iter_Assoc;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Variant_Range =>
declare
N_Bare_Variant : constant Bare_Variant := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Variant.Variant_F_Choices;
                            return;
                        when 2 =>
                            Result := N_Bare_Variant.Variant_F_Components;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Variant_Part_Range =>
declare
N_Bare_Variant_Part : constant Bare_Variant_Part := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Variant_Part.Variant_Part_F_Discr_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Variant_Part.Variant_Part_F_Variant;
                            return;
                        when others => null;
                    end case;
end;
when Ada_With_Clause_Range =>
declare
N_Bare_With_Clause : constant Bare_With_Clause := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_With_Clause.With_Clause_F_Has_Limited;
                            return;
                        when 2 =>
                            Result := N_Bare_With_Clause.With_Clause_F_Has_Private;
                            return;
                        when 3 =>
                            Result := N_Bare_With_Clause.With_Clause_F_Packages;
                            return;
                        when others => null;
                    end case;
end;
when others => null;
end case;
      --  Execution should reach this point iff nothing matched this index, so
      --  we must be out of bounds.
      Index_In_Bounds := False;
   end Get_Child;
   -----------
   -- Print --
   -----------
   procedure Print
     (Node        : Bare_Ada_Node;
      Show_Slocs  : Boolean;
      Line_Prefix : String := "")
   is
      K : Ada_Node_Kind_Type;
   begin
      if Node = null then
         Put_Line ("None");
         return;
      end if;
      K := Node.Kind;
      Put (Line_Prefix & Kind_Name (Node));
      if Show_Slocs then
         Put ("[" & Image (Sloc_Range (Node)) & "]");
      end if;
      if Is_Incomplete (Node) then
         Put (" <<INCOMPLETE>>");
      end if;
      if Is_Token_Node (Node.Kind) then
         Put_Line (": " & Image (Text (Node)));
      elsif Node.Kind not in Ada_Ada_List then
         New_Line;
      end if;
         --  List nodes are displayed in a special way (they have no field)
         if K in Ada_Ada_List then
            if Node.Count = 0 then
               Put_Line (": <empty list>");
               return;
            end if;
            New_Line;
            for Child of Node.Nodes (1 .. Node.Count) loop
               if Child /= null then
                  Print (Child, Show_Slocs, Line_Prefix & "|  ");
               end if;
            end loop;
            return;
         end if;
         --  This is for regular nodes: display each field
         declare
            use Libadalang.Introspection_Implementation;
            Attr_Prefix     : constant String := Line_Prefix & "|";
            Children_Prefix : constant String := Line_Prefix & "|  ";
            Field_List      : constant Syntax_Field_Reference_Array :=
              Syntax_Fields (K);
         begin
            for I in Field_List'Range loop
               declare
                  Child : constant Bare_Ada_Node :=
                     Implementation.Child (Node, I);
               begin
                  Put
                    (Attr_Prefix
                     & Image (Syntax_Field_Name (Field_List (I)))
                     & ":");
                  if Child /= null then
                     New_Line;
                     Print (Child, Show_Slocs, Children_Prefix);
                  else
                     Put_Line (" <null>");
                  end if;
               end;
            end loop;
         end;
   end Print;
   ------------
   -- Parent --
   ------------
   function Parent (Node : Bare_Ada_Node) return Bare_Ada_Node is
   begin
      return Node.Parent;
   end Parent;
   ------------------
   -- Stored_Token --
   ------------------
   function Stored_Token
     (Node  : Bare_Ada_Node;
      Token : Token_Reference) return Token_Index
   is
      Index : constant Token_Or_Trivia_Index := Get_Token_Index (Token);
   begin
      if Node.Unit.TDH'Access /= Get_Token_TDH (Token) then
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "Cannot associate a token and a node from different analysis"
            & " units");
      elsif Index.Trivia /= No_Token_Index then
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "A node cannot hold trivia");
      end if;
      return Index.Token;
   end Stored_Token;
   -------------------------
   -- Children_And_Trivia --
   -------------------------
   function Children_And_Trivia
     (Node : Bare_Ada_Node) return Bare_Children_Array
   is
      package Children_Vectors is new Ada.Containers.Vectors
        (Positive, Bare_Child_Record);
      use Children_Vectors;
      Ret_Vec : Vector;
      Ctx     : Internal_Context renames Node.Unit.Context;
      TDH     : Token_Data_Handler renames Node.Unit.TDH;
      procedure Append_Trivias (First, Last : Token_Index);
      --  Append all the trivias of tokens between indices First and Last to
      --  the returned vector.
      function Filter_Children
        (Parent : Bare_Ada_Node)
         return Internal_Bare_Ada_Node_Array;
      --  Return an array for all children in Parent that are not null
      --------------------
      -- Append_Trivias --
      --------------------
      procedure Append_Trivias (First, Last : Token_Index) is
      begin
         for I in First .. Last loop
            for D of Get_Trivias (TDH, I) loop
               Ret_Vec.Append
                 (Bare_Child_Record'
                    (Kind   => Trivia,
                     Trivia => Wrap_Token_Reference
                                 (Ctx, TDH'Access, (I, D))));
            end loop;
         end loop;
      end Append_Trivias;
      ---------------------
      -- Filter_Children --
      ---------------------
      function Filter_Children
        (Parent : Bare_Ada_Node)
         return Internal_Bare_Ada_Node_Array
      is
         Children : constant Internal_Bare_Ada_Node_Array :=
            Implementation.Children (Parent);
         Result   : Internal_Bare_Ada_Node_Array (Children'Range);
         Next     : Integer := Result'First;
      begin
         for I in Children'Range loop
            if Children (I) /= null then
               Result (Next) := Children (I);
               Next := Next + 1;
            end if;
         end loop;
         return Result (Result'First .. Next - 1);
      end Filter_Children;
      First_Child : constant Positive := 1;
      N_Children  : constant Internal_Bare_Ada_Node_Array :=
         Filter_Children (Node);
   begin
      if N_Children'Length > 0
        and then (Node.Token_Start_Index
                    /= N_Children (First_Child).Token_Start_Index)
      then
         Append_Trivias (Node.Token_Start_Index,
                         N_Children (First_Child).Token_Start_Index - 1);
      end if;
      --  Append each node to Ret_Vec, and append trivia that follow after each
      --  non-ghost nodes.
      for I in N_Children'Range loop
         Ret_Vec.Append (Bare_Child_Record'(Child, N_Children (I)));
         if not Is_Ghost (N_Children (I)) then
            Append_Trivias (N_Children (I).Token_End_Index,
                            (if I = N_Children'Last
                             then Node.Token_End_Index - 1
                             else N_Children (I + 1).Token_Start_Index - 1));
         end if;
      end loop;
      declare
         A : Bare_Children_Array (1 .. Natural (Ret_Vec.Length));
      begin
         for I in A'Range loop
            A (I) := Ret_Vec.Element (I);
         end loop;
         return A;
      end;
   end Children_And_Trivia;
   --------------
   -- Is_Ghost --
   --------------
   function Is_Ghost (Node : Bare_Ada_Node) return Boolean
   is (Node.Token_End_Index = No_Token_Index);
   -------------------
   -- Is_Incomplete --
   -------------------
   function Is_Incomplete (Node : Bare_Ada_Node) return Boolean
   is
      LGC : Bare_Ada_Node;
   begin
     if Is_List_Node (Node.Kind) then
        LGC := (if Last_Child_Index (Node) /= 0
                then Child (Node, Last_Child_Index (Node))
                else null);
        return LGC /= null and then Is_Incomplete (LGC);
      else
         return Node.Last_Attempted_Child > -1;
      end if;
   end;
   -----------------
   -- Token_Start --
   -----------------
   function Token_Start (Node : Bare_Ada_Node) return Token_Reference
   is (Token (Node, Node.Token_Start_Index));
   ---------------
   -- Token_End --
   ---------------
   function Token_End (Node : Bare_Ada_Node) return Token_Reference
   is
     (if Node.Token_End_Index = No_Token_Index
      then Token_Start (Node)
      else Token (Node, Node.Token_End_Index));
   -----------
   -- Token --
   -----------
   function Token
     (Node  : Bare_Ada_Node;
      Index : Token_Index) return Token_Reference
   is
      Unit    : constant Internal_Unit := Node.Unit;
      Context : constant Internal_Context := Unit.Context;
   begin
      return Wrap_Token_Reference
        (Context, Token_Data (Unit), (Index, No_Token_Index));
   end Token;
   ---------
   -- "<" --
   ---------
   function "<" (Left, Right : Bare_Ada_Node) return Boolean is
   begin
      --  Reject invalid inputs
      if Left /= null and Is_Synthetic (Left) then
         raise Property_Error with "left node is synthetic";
      elsif Right /= null and Is_Synthetic (Right) then
         raise Property_Error with "right node is synthetic";
      end if;
      --  Null nodes come first
      if Left = null then
         return Right /= null;
      elsif Right = null then
         return False;
      end if;
      --  So we have two non-null nodes. Sort by unit filename
      if Left.Unit < Right.Unit then
         return True;
      elsif Left.Unit /= Right.Unit then
         return False;
      end if;
      --  Both nodes come from the same unit: compare their token indexes
      if Left.Token_Start_Index < Right.Token_Start_Index then
         return True;
      elsif Left.Token_Start_Index > Right.Token_Start_Index then
         return False;
      else
         return Left.Token_End_Index < Right.Token_End_Index;
      end if;
   end "<";
   -------------
   -- Is_Null --
   -------------
   function Is_Null (Node : Bare_Ada_Node) return Boolean
   is (Node = null);
   ----------
   -- Kind --
   ----------
   function Kind (Node : Bare_Ada_Node) return Ada_Node_Kind_Type
   is (Node.Kind);
   -----------------
   -- Child_Index --
   -----------------
   function Child_Index (Node : Bare_Ada_Node) return Integer
   is
      N : Bare_Ada_Node := null;
   begin
      if Node.Parent = null then
         raise Property_Error with
            "Trying to get the child index of a root node";
      end if;
      for I in First_Child_Index (Node.Parent)
            .. Last_Child_Index (Node.Parent)
      loop
         N := Child (Node.Parent, I);
         if N = Node then
            return I - 1;
         end if;
      end loop;
      --  If we reach this point, then Node isn't a Child of Node.Parent. This
      --  is not supposed to happen.
      raise Program_Error;
   end Child_Index;
   -------------------
   -- Fetch_Sibling --
   -------------------
   function Fetch_Sibling
     (Node   : Bare_Ada_Node;
      Offset : Integer) return Bare_Ada_Node is
   begin
      --  Root nodes have no sibling: handle them now to avoid invalid requests
      --  in the code below.
      if Node.Parent = null then
         return null;
      end if;
      declare
         Node_Index : constant Positive := Child_Index (Node) + 1;
         --  Child_Index is 0-based, but the Child primitive expects a 1-based
         --  index.
         Sibling_Index : constant Integer := Node_Index + Offset;
      begin
         --  Child returns null for out-of-bound indexes
         return (if Sibling_Index >= 1
                 then Child (Node.Parent, Sibling_Index)
                 else null);
      end;
   end Fetch_Sibling;
   -------------------
   -- Fetch_Sibling --
   -------------------
   function Fetch_Sibling
     (Node   : Bare_Ada_Node;
      E_Info : Internal_Entity_Info;
      Offset : Integer) return Internal_Entity
   is
      Sibling : constant Bare_Ada_Node := Fetch_Sibling (Node, Offset);
   begin
      --  Don't forget to clear entity info if the result is nul
      return (if Sibling = null
              then No_Entity
              else (Sibling, E_Info));
   end Fetch_Sibling;
   ----------------------
   -- Previous_Sibling --
   ----------------------
   function Previous_Sibling
     (Node   : Bare_Ada_Node;
      E_Info : Internal_Entity_Info := No_Entity_Info)
      return Internal_Entity is
   begin
      return Fetch_Sibling (Node, E_Info, -1);
   end Previous_Sibling;
   ------------------
   -- Next_Sibling --
   ------------------
   function Next_Sibling
     (Node   : Bare_Ada_Node;
      E_Info : Internal_Entity_Info := No_Entity_Info)
      return Internal_Entity is
   begin
      return Fetch_Sibling (Node, E_Info, 1);
   end Next_Sibling;
   -------------
   -- Combine --
   -------------
   function Combine
     (L, R : Internal_Metadata) return Internal_Metadata
   is
      Ret : Internal_Metadata := No_Metadata;
   begin
         Ret.Dottable_Subp := L.Dottable_Subp or R.Dottable_Subp;
         if L.Primitive = No_Bare_Ada_Node then
            Ret.Primitive := R.Primitive;
         elsif R.Primitive = No_Bare_Ada_Node then
            Ret.Primitive := L.Primitive;
         else
            raise Property_Error with "Wrong combine for metadata field";
         end if;
         if L.Primitive_Real_Type = No_Bare_Ada_Node then
            Ret.Primitive_Real_Type := R.Primitive_Real_Type;
         elsif R.Primitive_Real_Type = No_Bare_Ada_Node then
            Ret.Primitive_Real_Type := L.Primitive_Real_Type;
         else
            raise Property_Error with "Wrong combine for metadata field";
         end if;
      return Ret;
   end Combine;
   -------------------------------
   -- Create_Static_Lexical_Env --
   -------------------------------
   function Create_Static_Lexical_Env
     (Parent            : Lexical_Env;
      Node              : Bare_Ada_Node;
      Transitive_Parent : Boolean := False) return Lexical_Env
   is
      Unit : constant Internal_Unit :=
        (if Node = null then null else Node.Unit);
   begin
      return Result : Lexical_Env := Create_Lexical_Env
        (Parent, Node, Transitive_Parent, Convert_Unit (Unit))
      do
         if Unit /= null then
            Register_Destroyable (Unit, Unwrap (Result.Env));
         end if;
      end return;
   end Create_Static_Lexical_Env;
   ---------
   -- Get --
   ---------
   function Get
     (Self  : Bare_Ada_Node;
      A     : AST_Envs.Entity_Array;
      Index : Integer) return Internal_Entity
   is
      function Length (A : AST_Envs.Entity_Array) return Natural
      is (A'Length);
      function Get
        (A     : AST_Envs.Entity_Array;
         Index : Integer) return Internal_Entity
      is (A (Index + 1)); --  A is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Entity,
         Sequence_Type => AST_Envs.Entity_Array,
         Length        => Length,
         Get           => Get);
      Result : Internal_Entity;
   begin
      if Relative_Get (A, Index, Result) then
         return Result;
      else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "out-of-bounds array access");
      end if;
   end Get;
   -----------
   -- Group --
   -----------
   function Group
     (Envs   : Lexical_Env_Array_Access;
      Env_Md : Internal_Metadata := No_Metadata) return Lexical_Env
   is (Group (Lexical_Env_Array (Envs.Items), Env_Md));
      --
      --  Untyped wrappers for Bare_Ada_Node
      --
function Ada_Node_P_Resolve_Generic_Actual
  (E : Entity
  ) return Internal_Entity;
function Ada_Node_P_Use_Clauses_In_Spec_Of_Subp_Body
  (E : Entity
  ) return Lexical_Env;
function Ada_Node_P_Use_Clauses_In_Generic_Formal_Part
  (E : Entity
  ) return Lexical_Env;
function Ada_Node_P_Nested_Generic_Formal_Part
  (E : Entity
  ) return Lexical_Env;
      --
      --  Untyped wrappers for Bare_Base_Type_Decl
      --
function Base_Type_Decl_P_Dottable_Subps
  (E : Entity
  ) return Internal_Inner_Env_Assoc_Array_Access;
      --
      --  Untyped wrappers for Bare_Type_Decl
      --
function Type_Decl_F_Direct_Primitive_Subps
  (E : Entity
  ) return Internal_Inner_Env_Assoc_Array_Access;
function Type_Decl_P_Refined_Parent_Primitives_Env
  (E : Entity
  ) return Lexical_Env;
      --
      --  Untyped wrappers for Bare_Body_Node
      --
function Body_Node_P_Subunit_Decl_Env
  (E : Entity
  ) return Lexical_Env;
function Body_Node_P_Body_Decl_Scope
  (E : Entity
  ) return Lexical_Env;
      --
      --  Untyped wrappers for Bare_Package_Body
      --
function Package_Body_P_Package_Decl_Uses_Clauses_Envs
  (E : Entity
  ) return Lexical_Env;
      --
      --  Untyped wrappers for Bare_Task_Body
      --
function Task_Body_P_Task_Type_Decl_Scope
  (E : Entity
  ) return Lexical_Env;
      --
      --  Untyped wrappers for Bare_Generic_Instantiation
      --
function Generic_Instantiation_P_Instantiation_Bindings
  (E : Entity
  ) return Internal_Inner_Env_Assoc_Array_Access;
      --
      --  Untyped wrappers for Bare_Name
      --
function Name_P_Name_Designated_Type_Env
  (E : Entity
  ) return Lexical_Env;
function Name_P_Use_Package_Name_Designated_Env
  (E : Entity
  ) return Lexical_Env;
   ------------------
   -- Children_Env --
   ------------------
   function Children_Env
     (Node   : Bare_Ada_Node;
      E_Info : Internal_Entity_Info := No_Entity_Info)
      return Lexical_Env
   is (Rebind_Env (Node.Self_Env, E_Info));
   --------------
   -- Node_Env --
   --------------
   function Node_Env
     (Node   : Bare_Ada_Node;
      E_Info : Internal_Entity_Info := No_Entity_Info)
      return Lexical_Env
   is
      function Get_Base_Env return Lexical_Env;
      --  Return the environment that we need to rebind before returning
      ------------------
      -- Get_Base_Env --
      ------------------
      function Get_Base_Env return Lexical_Env is
         pragma Warnings (Off, "referenced");
         function Get_Parent_Env return Lexical_Env;
         pragma Warnings (On, "referenced");
         --------------------
         -- Get_Parent_Env --
         --------------------
         function Get_Parent_Env return Lexical_Env is
            Parent : constant Lexical_Env := AST_Envs.Parent (Node.Self_Env);
         begin
            --  If Node is the root scope or the empty environment, Parent can
            --  be a wrapper around the null node. Turn this into the
            --  Empty_Env, as null envs are erroneous values in properties.
            return (if Unwrap (Parent) = null
                    then Empty_Env
                    else Parent);
         end Get_Parent_Env;
      begin
         return
           (if Node.Kind in Ada_Generic_Package_Internal | Ada_Package_Decl | Ada_Incomplete_Type_Decl | Ada_Incomplete_Formal_Type_Decl | Ada_Incomplete_Tagged_Type_Decl | Ada_Protected_Type_Decl | Ada_Task_Type_Decl | Ada_Single_Task_Type_Decl | Ada_Concrete_Type_Decl | Ada_Formal_Type_Decl | Ada_Abstract_Subp_Decl | Ada_Abstract_Formal_Subp_Decl | Ada_Concrete_Formal_Subp_Decl | Ada_Subp_Decl | Ada_Entry_Decl | Ada_Enum_Literal_Decl | Ada_Synthetic_Char_Enum_Lit | Ada_Generic_Subp_Internal | Ada_Synthetic_Subp_Decl | Ada_Expr_Function | Ada_Null_Subp_Decl | Ada_Subp_Body | Ada_Subp_Renaming_Decl | Ada_Package_Body_Stub | Ada_Protected_Body_Stub | Ada_Subp_Body_Stub | Ada_Task_Body_Stub | Ada_Entry_Body | Ada_Package_Body | Ada_Protected_Body | Ada_Task_Body | Ada_Exception_Handler | Ada_Generic_Package_Decl | Ada_Generic_Subp_Decl | Ada_Generic_Package_Instantiation | Ada_Generic_Subp_Instantiation | Ada_Generic_Package_Renaming_Decl | Ada_Generic_Subp_Renaming_Decl | Ada_Package_Renaming_Decl | Ada_Single_Protected_Decl | Ada_Single_Task_Decl | Ada_Private_Part | Ada_Decl_Expr | Ada_Accept_Stmt | Ada_Accept_Stmt_With_Stmts | Ada_For_Loop_Stmt | Ada_Begin_Block | Ada_Decl_Block | Ada_Extended_Return_Stmt | Ada_Named_Stmt | Ada_Access_To_Subp_Def
            then Get_Parent_Env
            else Node.Self_Env);
      end Get_Base_Env;
      Base_Env : Lexical_Env := Get_Base_Env;
      Result   : constant Lexical_Env := Rebind_Env (Base_Env, E_Info);
   begin
      Dec_Ref (Base_Env);
      return Result;
   end Node_Env;
   ------------
   -- Parent --
   ------------
   function Parent
     (Node   : Bare_Ada_Node;
      E_Info : Internal_Entity_Info := No_Entity_Info)
      return Internal_Entity is
   begin
      --  TODO: shed entity information as appropriate
      return (Node.Parent, E_Info);
   end Parent;
   -------------
   -- Parents --
   -------------
   function Parents
     (Node      : Bare_Ada_Node;
      With_Self : Boolean := True;
      E_Info    : Internal_Entity_Info := No_Entity_Info)
      return Internal_Entity_Array_Access
   is
      Bare_Parents : Bare_Ada_Node_Array_Access := Parents (Node, With_Self);
      Result       : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (Bare_Parents.N);
   begin
      --  TODO: shed entity information as appropriate
      for I in Bare_Parents.Items'Range loop
         Result.Items (I) := (Bare_Parents.Items (I), E_Info);
      end loop;
      Dec_Ref (Bare_Parents);
      return Result;
   end Parents;
   --------------
   -- Children --
   --------------
   function Children
     (Node   : Bare_Ada_Node;
      E_Info : Internal_Entity_Info := No_Entity_Info)
      return Internal_Entity_Array_Access
   is
      Bare_Children : Bare_Ada_Node_Array_Access := Children (Node);
      Result        : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (Bare_Children.N);
   begin
      --  TODO: shed entity information as appropriate
      for I in Bare_Children.Items'Range loop
         Result.Items (I) := (Bare_Children.Items (I), E_Info);
      end loop;
      Dec_Ref (Bare_Children);
      return Result;
   end Children;
   ---------------------
   -- New_Unit_String --
   ---------------------
   function New_Unit_String
     (Unit : Internal_Unit; Str : String) return String_Access
   is
      procedure Register_Destroyable_String is new Register_Destroyable_Gen
        (String, String_Access, Free);
   begin
      return Ret : String_Access := new String'(Str) do
         Register_Destroyable_String (Unit, Ret);
      end return;
   end New_Unit_String;
   --------------------------------
   -- Assign_Names_To_Logic_Vars --
   --------------------------------
   procedure Assign_Names_To_Logic_Vars (Node : Bare_Ada_Node) is
      pragma Warnings (Off, "referenced");
      procedure Assign
        (Node  : Bare_Ada_Node;
         LV    : in out Logic_Var_Record;
         Field : String);
      --  Assign a name to the LV logic variable. Node must be the node that
      --  owns LV, and Field must be the name of the field in Node that holds
      --  LV.
      ------------
      -- Assign --
      ------------
      procedure Assign
        (Node  : Bare_Ada_Node;
         LV    : in out Logic_Var_Record;
         Field : String) is
      begin
         LV.Dbg_Name :=
           new String'(Image (Short_Text_Image (Node)) & "." & Field);
      end Assign;
      K : constant Ada_Node_Kind_Type := Node.Kind;
      pragma Warnings (On, "referenced");
   begin
      case Ada_Ada_Node (K) is
when Ada_Attribute_Ref_Range =>
declare
N_Bare_Attribute_Ref : constant Bare_Attribute_Ref := Node;
begin
Assign (N_Bare_Attribute_Ref,        N_Bare_Attribute_Ref.Attribute_Ref_R_Ref_Var,        "r_ref_var");
end;
when Ada_Call_Expr_Range =>
declare
N_Bare_Call_Expr : constant Bare_Call_Expr := Node;
begin
Assign (N_Bare_Call_Expr,        N_Bare_Call_Expr.Call_Expr_R_Called_Spec,        "r_called_spec");
end;
when Ada_Explicit_Deref_Range =>
declare
N_Bare_Explicit_Deref : constant Bare_Explicit_Deref := Node;
begin
Assign (N_Bare_Explicit_Deref,        N_Bare_Explicit_Deref.Explicit_Deref_R_Called_Spec,        "r_called_spec");
end;
when Ada_Reduce_Attribute_Ref_Range =>
declare
N_Bare_Reduce_Attribute_Ref : constant Bare_Reduce_Attribute_Ref := Node;
begin
Assign (N_Bare_Reduce_Attribute_Ref,        N_Bare_Reduce_Attribute_Ref.Reduce_Attribute_Ref_R_Ref_Var,        "r_ref_var");
end;
when Ada_Target_Name_Range =>
declare
N_Bare_Target_Name : constant Bare_Target_Name := Node;
begin
Assign (N_Bare_Target_Name,        N_Bare_Target_Name.Target_Name_R_Ref_Var,        "r_ref_var");
end;
when Ada_Update_Attribute_Ref_Range =>
declare
N_Bare_Update_Attribute_Ref : constant Bare_Update_Attribute_Ref := Node;
begin
Assign (N_Bare_Update_Attribute_Ref,        N_Bare_Update_Attribute_Ref.Update_Attribute_Ref_R_Ref_Var,        "r_ref_var");
end;
when others => null;
end case;
      for Child of Internal_Bare_Ada_Node_Array'(Children (Node)) loop
         if Child /= null then
            Assign_Names_To_Logic_Vars (Child);
         end if;
      end loop;
   end Assign_Names_To_Logic_Vars;
   ----------------
   -- Text_Image --
   ----------------
   function Text_Image (Ent : Internal_Entity) return Text_Type is
   begin
      if Ent.Node /= null then
         declare
            Node_Image : constant Text_Type := Short_Text_Image (Ent.Node);
         begin
            return
            (if Ent.Info.Rebindings /= null
             then "<| "
             & Node_Image (Node_Image'First + 1 .. Node_Image'Last - 1) & " "
             & AST_Envs.Text_Image (Ent.Info.Rebindings) & " |>"
             else Node_Image);
         end;
      else
         return "None";
      end if;
   end Text_Image;
   ---------------------
   -- Full_Sloc_Image --
   ---------------------
   function Full_Sloc_Image (Node : Bare_Ada_Node) return String_Type
   is
      Res      : constant Text_Type :=
        To_Text
          (Ada.Directories.Simple_Name
             (Get_Filename (Unit (Node))))
           & ":" & To_Text (Image (Start_Sloc (Sloc_Range (Node)))) & ": ";
   begin
      return Create_String (Res);
   end Full_Sloc_Image;
   -----------
   -- Image --
   -----------
   function Image (Ent : Internal_Entity) return String is
      Result : constant Text_Type := Text_Image (Ent);
   begin
      return Image (Result);
   end Image;
   ---------------
   -- Can_Reach --
   ---------------
   function Can_Reach (El, From : Bare_Ada_Node) return Boolean is
   begin
      --  Since this function is only used to implement sequential semantics in
      --  envs, we consider that elements coming from different units are
      --  always visible for each other, and let the user implement language
      --  specific visibility rules in the DSL.
      if El = null or else From = null or else El.Unit /= From.Unit then
         return True;
      end if;
      return El.Token_Start_Index < From.Token_Start_Index;
   end Can_Reach;
   -----------------
   -- Hash_Entity --
   -----------------
   function Hash_Entity (Self : Internal_Entity) return Hash_Type is
   begin
      return Combine (Hash (Self.Node), Hash (Self.Info.Rebindings));
   end Hash_Entity;
   --------------------
   -- Compare_Entity --
   --------------------
   function Compare_Entity (Left, Right : Internal_Entity) return Boolean
   is
   begin
      return (Left.Node = Right.Node
              and then Left.Info.Rebindings = Right.Info.Rebindings);
   end Compare_Entity;
   --------------------------------
   -- Create_Dynamic_Lexical_Env --
   --------------------------------
   function Create_Dynamic_Lexical_Env
     (Self              : Bare_Ada_Node;
      Assocs_Getter     : Inner_Env_Assocs_Resolver;
      Assoc_Resolver    : Entity_Resolver;
      Transitive_Parent : Boolean) return Lexical_Env
   is
      Unit : constant Internal_Unit := Self.Unit;
   begin
      --  This restriction is necessary to avoid relocation issues when
      --  Self.Self_Env is terminated.
      if Is_Foreign_Strict (Self.Self_Env, Self) then
         Raise_Property_Exception
           (Self,
            Property_Error'Identity,
            "cannot create a dynamic lexical env when Self.Self_Env is"
            & " foreign");
      end if;
      return Result : constant Lexical_Env := Create_Dynamic_Lexical_Env
        (Parent            => Null_Lexical_Env,
         Node              => Self,
         Transitive_Parent => Transitive_Parent,
         Owner             => Convert_Unit (Unit),
         Assocs_Getter     => Assocs_Getter,
         Assoc_Resolver    => Assoc_Resolver)
      do
         --  Since dynamic lexical environments can only be created in lazy
         --  field initializers, it is fine to tie Result's lifetime to the
         --  its owning unit's lifetime.
         Register_Destroyable (Unit, Unwrap (Result));
      end return;
   end Create_Dynamic_Lexical_Env;
   procedure Destroy_Synthetic_Node (Node : in out Bare_Ada_Node);
   --  Helper for the Register_Destroyable above
   ------------
   -- Length --
   ------------
   function Length (Node : Bare_Ada_List) return Natural
   is (if Node = null then 0 else Children_Count (Node));
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (B : Boolean) return String is
      begin
         return (if B then "True" else "False");
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (I : Integer) return String is
      begin
         return Integer'Image (I);
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (S : Symbol_Type) return String is
      begin
         return (if S = null
                 then "None"
                 else Image (S.all, With_Quotes => True));
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (C : Character_Type) return String is
         C_Str : constant Text_Type := (1 => C);
      begin
         return "'" & Image (C_Str) & "'";
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (S : String_Type) return String is
      begin
         return Image (S.Content, With_Quotes => True);
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (Env : Lexical_Env) return String is
      begin
         case Env.Kind is
         when Static_Primary =>
            return "<LexicalEnv static-primary for "
                   & Trace_Image (Env_Node (Env)) & ">";
         when others =>
            return "<LexicalEnv synthetic>";
         end case;
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (R : Env_Rebindings) return String is
      begin
         return Image (Text_Image (R));
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (Unit : Internal_Unit) return String is
      begin
         return "Internal_Unit (""" & Basename (Unit) & """)";
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (Eq : Logic_Equation) return String is
         pragma Unreferenced (Eq);
      begin
         return "<LogicEquation>";
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (Var : Logic_Var) return String is
         pragma Unreferenced (Var);
      begin
         return "<LogicVariable>";
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (K : Analysis_Unit_Kind) return String is
      begin
         return Analysis_Unit_Kind'Image (K);
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (Self : Ref_Categories) return String is
         Result : Unbounded_String;
         First  : Boolean := True;
      begin
         Append (Result, "RefCategories(");
         for C in Ref_Category loop
            if Self (C) then
               if First then
                  First := False;
               else
                  Append (Result, ", ");
               end if;
               Append (Result, C'Image);
            end if;
         end loop;
         Append (Result, ")");
         return To_String (Result);
      end Trace_Image;
      ----------
      -- Hash --
      ----------
      pragma Warnings (Off, "referenced");
      function Hash (R : Internal_Metadata) return Hash_Type is
         pragma Warnings (On, "referenced");
      begin
            return Combine ((Hash (R.Dottable_Subp), Hash (R.Primitive), Hash (R.Primitive_Real_Type)));
      end Hash;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Metadata) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Dottable_Subp => " & Trace_Image (R.Dottable_Subp)
                        & ", "
                     & "Primitive => " & Trace_Image (R.Primitive)
                        & ", "
                     & "Primitive_Real_Type => " & Trace_Image (R.Primitive_Real_Type)
               & ")");
      end Trace_Image;
      ----------
      -- Hash --
      ----------
      pragma Warnings (Off, "referenced");
      function Hash (R : Internal_Entity_Info) return Hash_Type is
         pragma Warnings (On, "referenced");
      begin
            return Combine ((Hash (R.Md), Hash (R.Rebindings), Hash (R.From_Rebound)));
      end Hash;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Info) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Md => " & Trace_Image (R.Md)
                        & ", "
                     & "Rebindings => " & Trace_Image (R.Rebindings)
                        & ", "
                     & "From_Rebound => " & Trace_Image (R.From_Rebound)
               & ")");
      end Trace_Image;
      function Create_Internal_Entity
        (Node : Bare_Ada_Node; Info : Internal_Entity_Info)
         return Internal_Entity is
      begin
         if Node = null then
            return No_Entity;
         end if;
         return (Node => Node, Info => Info);
      end;
      ----------
      -- Hash --
      ----------
      pragma Warnings (Off, "referenced");
      function Hash (R : Internal_Entity) return Hash_Type is
         pragma Warnings (On, "referenced");
      begin
            return Combine
              (Hash (R.Node), Hash (R.Info));
      end Hash;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Expr
        (Node : Bare_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Expr is
      begin
         if Node = null then
            return No_Entity_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Aspect) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Exists => " & Trace_Image (R.Exists)
                        & ", "
                     & "Node => " & Trace_Image (R.Node)
                        & ", "
                     & "Value => " & Trace_Image (R.Value)
               & ")");
      end Trace_Image;
      function Create_Internal_Entity_Basic_Decl
        (Node : Bare_Basic_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Basic_Decl is
      begin
         if Node = null then
            return No_Entity_Basic_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      ----------
      -- Hash --
      ----------
      pragma Warnings (Off, "referenced");
      function Hash (R : Internal_Entity_Basic_Decl) return Hash_Type is
         pragma Warnings (On, "referenced");
      begin
            return Combine
              (Hash (R.Node), Hash (R.Info));
      end Hash;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Basic_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Completion_Item) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Decl => " & Trace_Image (R.Decl)
                        & ", "
                     & "Is_Dot_Call => " & Trace_Image (R.Is_Dot_Call)
                        & ", "
                     & "Is_Visible => " & Trace_Image (R.Is_Visible)
                        & ", "
                     & "Weight => " & Trace_Image (R.Weight)
               & ")");
      end Trace_Image;
      -------------
      -- Inc_Ref --
      -------------
      procedure Inc_Ref (R : Internal_Designated_Env) is
      begin
               Inc_Ref (R.Direct_Env);
      end Inc_Ref;
      -------------
      -- Dec_Ref --
      -------------
      procedure Dec_Ref (R : in out Internal_Designated_Env) is
      begin
               Dec_Ref (R.Direct_Env);
      end Dec_Ref;
      ----------------
      -- Equivalent --
      ----------------
      function Equivalent (L, R : Internal_Designated_Env) return Boolean is
      begin
         return L.Kind = R.Kind and then L.Env_Name = R.Env_Name and then Equivalent (L.Direct_Env, R.Direct_Env);
      end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Designated_Env) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Kind => " & Trace_Image (R.Kind)
                        & ", "
                     & "Env_Name => " & Trace_Image (R.Env_Name)
                        & ", "
                     & "Direct_Env => " & Trace_Image (R.Direct_Env)
               & ")");
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Discrete_Range) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Low_Bound => " & Trace_Image (R.Low_Bound)
                        & ", "
                     & "High_Bound => " & Trace_Image (R.High_Bound)
               & ")");
      end Trace_Image;
      function Create_Internal_Entity_Ada_List
        (Node : Bare_Ada_List; Info : Internal_Entity_Info)
         return Internal_Entity_Ada_List is
      begin
         if Node = null then
            return No_Entity_Ada_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Ada_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Ada_Node_List
        (Node : Bare_Ada_Node_List; Info : Internal_Entity_Info)
         return Internal_Entity_Ada_Node_List is
      begin
         if Node = null then
            return No_Entity_Ada_Node_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Ada_Node_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Alternatives_List
        (Node : Bare_Alternatives_List; Info : Internal_Entity_Info)
         return Internal_Entity_Alternatives_List is
      begin
         if Node = null then
            return No_Entity_Alternatives_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Alternatives_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Name
        (Node : Bare_Name; Info : Internal_Entity_Info)
         return Internal_Entity_Name is
      begin
         if Node = null then
            return No_Entity_Name;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Name) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Single_Tok_Node
        (Node : Bare_Single_Tok_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Single_Tok_Node is
      begin
         if Node = null then
            return No_Entity_Single_Tok_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Single_Tok_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Base_Id
        (Node : Bare_Base_Id; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Id is
      begin
         if Node = null then
            return No_Entity_Base_Id;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Id) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Identifier
        (Node : Bare_Identifier; Info : Internal_Entity_Info)
         return Internal_Entity_Identifier is
      begin
         if Node = null then
            return No_Entity_Identifier;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Identifier) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Discriminant_Values) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Discriminant => " & Trace_Image (R.Discriminant)
                        & ", "
                     & "Values => " & Trace_Image (R.Values)
               & ")");
      end Trace_Image;
      -------------
      -- Inc_Ref --
      -------------
      procedure Inc_Ref (R : Internal_Doc_Annotation) is
      begin
               Inc_Ref (R.Key);
               Inc_Ref (R.Value);
      end Inc_Ref;
      -------------
      -- Dec_Ref --
      -------------
      procedure Dec_Ref (R : in out Internal_Doc_Annotation) is
      begin
               Dec_Ref (R.Key);
               Dec_Ref (R.Value);
      end Dec_Ref;
      ----------------
      -- Equivalent --
      ----------------
      function Equivalent (L, R : Internal_Doc_Annotation) return Boolean is
      begin
         return Equivalent (L.Key, R.Key) and then Equivalent (L.Value, R.Value);
      end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Doc_Annotation) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Key => " & Trace_Image (R.Key)
                        & ", "
                     & "Value => " & Trace_Image (R.Value)
               & ")");
      end Trace_Image;
      function Create_Internal_Entity_Abort_Node
        (Node : Bare_Abort_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Abort_Node is
      begin
         if Node = null then
            return No_Entity_Abort_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Abort_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Abort_Absent
        (Node : Bare_Abort_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_Abort_Absent is
      begin
         if Node = null then
            return No_Entity_Abort_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Abort_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Abort_Present
        (Node : Bare_Abort_Present; Info : Internal_Entity_Info)
         return Internal_Entity_Abort_Present is
      begin
         if Node = null then
            return No_Entity_Abort_Present;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Abort_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Stmt
        (Node : Bare_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Stmt is
      begin
         if Node = null then
            return No_Entity_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Simple_Stmt
        (Node : Bare_Simple_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Simple_Stmt is
      begin
         if Node = null then
            return No_Entity_Simple_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Simple_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Abort_Stmt
        (Node : Bare_Abort_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Abort_Stmt is
      begin
         if Node = null then
            return No_Entity_Abort_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Abort_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Abstract_Node
        (Node : Bare_Abstract_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Abstract_Node is
      begin
         if Node = null then
            return No_Entity_Abstract_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Abstract_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Abstract_Absent
        (Node : Bare_Abstract_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_Abstract_Absent is
      begin
         if Node = null then
            return No_Entity_Abstract_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Abstract_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Basic_Subp_Decl
        (Node : Bare_Basic_Subp_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Basic_Subp_Decl is
      begin
         if Node = null then
            return No_Entity_Basic_Subp_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Basic_Subp_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Classic_Subp_Decl
        (Node : Bare_Classic_Subp_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Classic_Subp_Decl is
      begin
         if Node = null then
            return No_Entity_Classic_Subp_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Classic_Subp_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Formal_Subp_Decl
        (Node : Bare_Formal_Subp_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Formal_Subp_Decl is
      begin
         if Node = null then
            return No_Entity_Formal_Subp_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Formal_Subp_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Abstract_Formal_Subp_Decl
        (Node : Bare_Abstract_Formal_Subp_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Abstract_Formal_Subp_Decl is
      begin
         if Node = null then
            return No_Entity_Abstract_Formal_Subp_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Abstract_Formal_Subp_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Abstract_Present
        (Node : Bare_Abstract_Present; Info : Internal_Entity_Info)
         return Internal_Entity_Abstract_Present is
      begin
         if Node = null then
            return No_Entity_Abstract_Present;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Abstract_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Abstract_State_Decl
        (Node : Bare_Abstract_State_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Abstract_State_Decl is
      begin
         if Node = null then
            return No_Entity_Abstract_State_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Abstract_State_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Abstract_State_Decl_Expr
        (Node : Bare_Abstract_State_Decl_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Abstract_State_Decl_Expr is
      begin
         if Node = null then
            return No_Entity_Abstract_State_Decl_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Abstract_State_Decl_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Abstract_State_Decl_List
        (Node : Bare_Abstract_State_Decl_List; Info : Internal_Entity_Info)
         return Internal_Entity_Abstract_State_Decl_List is
      begin
         if Node = null then
            return No_Entity_Abstract_State_Decl_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Abstract_State_Decl_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Abstract_Subp_Decl
        (Node : Bare_Abstract_Subp_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Abstract_Subp_Decl is
      begin
         if Node = null then
            return No_Entity_Abstract_Subp_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Abstract_Subp_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Composite_Stmt
        (Node : Bare_Composite_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Composite_Stmt is
      begin
         if Node = null then
            return No_Entity_Composite_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Composite_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Accept_Stmt
        (Node : Bare_Accept_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Accept_Stmt is
      begin
         if Node = null then
            return No_Entity_Accept_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Accept_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Accept_Stmt_With_Stmts
        (Node : Bare_Accept_Stmt_With_Stmts; Info : Internal_Entity_Info)
         return Internal_Entity_Accept_Stmt_With_Stmts is
      begin
         if Node = null then
            return No_Entity_Accept_Stmt_With_Stmts;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Accept_Stmt_With_Stmts) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Type_Def
        (Node : Bare_Type_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Type_Def is
      begin
         if Node = null then
            return No_Entity_Type_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Type_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Access_Def
        (Node : Bare_Access_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Access_Def is
      begin
         if Node = null then
            return No_Entity_Access_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Access_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Access_To_Subp_Def
        (Node : Bare_Access_To_Subp_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Access_To_Subp_Def is
      begin
         if Node = null then
            return No_Entity_Access_To_Subp_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Access_To_Subp_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Base_Aggregate
        (Node : Bare_Base_Aggregate; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Aggregate is
      begin
         if Node = null then
            return No_Entity_Base_Aggregate;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Aggregate) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Aggregate
        (Node : Bare_Aggregate; Info : Internal_Entity_Info)
         return Internal_Entity_Aggregate is
      begin
         if Node = null then
            return No_Entity_Aggregate;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Aggregate) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Basic_Assoc
        (Node : Bare_Basic_Assoc; Info : Internal_Entity_Info)
         return Internal_Entity_Basic_Assoc is
      begin
         if Node = null then
            return No_Entity_Basic_Assoc;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Basic_Assoc) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Aggregate_Assoc
        (Node : Bare_Aggregate_Assoc; Info : Internal_Entity_Info)
         return Internal_Entity_Aggregate_Assoc is
      begin
         if Node = null then
            return No_Entity_Aggregate_Assoc;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Aggregate_Assoc) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Aliased_Node
        (Node : Bare_Aliased_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Aliased_Node is
      begin
         if Node = null then
            return No_Entity_Aliased_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Aliased_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Aliased_Absent
        (Node : Bare_Aliased_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_Aliased_Absent is
      begin
         if Node = null then
            return No_Entity_Aliased_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Aliased_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Aliased_Present
        (Node : Bare_Aliased_Present; Info : Internal_Entity_Info)
         return Internal_Entity_Aliased_Present is
      begin
         if Node = null then
            return No_Entity_Aliased_Present;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Aliased_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_All_Node
        (Node : Bare_All_Node; Info : Internal_Entity_Info)
         return Internal_Entity_All_Node is
      begin
         if Node = null then
            return No_Entity_All_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_All_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_All_Absent
        (Node : Bare_All_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_All_Absent is
      begin
         if Node = null then
            return No_Entity_All_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_All_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_All_Present
        (Node : Bare_All_Present; Info : Internal_Entity_Info)
         return Internal_Entity_All_Present is
      begin
         if Node = null then
            return No_Entity_All_Present;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_All_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Allocator
        (Node : Bare_Allocator; Info : Internal_Entity_Info)
         return Internal_Entity_Allocator is
      begin
         if Node = null then
            return No_Entity_Allocator;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Allocator) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Anonymous_Expr_Decl
        (Node : Bare_Anonymous_Expr_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Anonymous_Expr_Decl is
      begin
         if Node = null then
            return No_Entity_Anonymous_Expr_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Anonymous_Expr_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Type_Expr
        (Node : Bare_Type_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Type_Expr is
      begin
         if Node = null then
            return No_Entity_Type_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Type_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Anonymous_Type
        (Node : Bare_Anonymous_Type; Info : Internal_Entity_Info)
         return Internal_Entity_Anonymous_Type is
      begin
         if Node = null then
            return No_Entity_Anonymous_Type;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Anonymous_Type) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Base_Type_Access_Def
        (Node : Bare_Base_Type_Access_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Type_Access_Def is
      begin
         if Node = null then
            return No_Entity_Base_Type_Access_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Type_Access_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Anonymous_Type_Access_Def
        (Node : Bare_Anonymous_Type_Access_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Anonymous_Type_Access_Def is
      begin
         if Node = null then
            return No_Entity_Anonymous_Type_Access_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Anonymous_Type_Access_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Base_Type_Decl
        (Node : Bare_Base_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Base_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      ----------
      -- Hash --
      ----------
      pragma Warnings (Off, "referenced");
      function Hash (R : Internal_Entity_Base_Type_Decl) return Hash_Type is
         pragma Warnings (On, "referenced");
      begin
            return Combine
              (Hash (R.Node), Hash (R.Info));
      end Hash;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Type_Decl
        (Node : Bare_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Anonymous_Type_Decl
        (Node : Bare_Anonymous_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Anonymous_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Anonymous_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Anonymous_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Array_Indices
        (Node : Bare_Array_Indices; Info : Internal_Entity_Info)
         return Internal_Entity_Array_Indices is
      begin
         if Node = null then
            return No_Entity_Array_Indices;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Array_Indices) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Array_Type_Def
        (Node : Bare_Array_Type_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Array_Type_Def is
      begin
         if Node = null then
            return No_Entity_Array_Type_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Array_Type_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Aspect_Assoc
        (Node : Bare_Aspect_Assoc; Info : Internal_Entity_Info)
         return Internal_Entity_Aspect_Assoc is
      begin
         if Node = null then
            return No_Entity_Aspect_Assoc;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Aspect_Assoc) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Aspect_Assoc_List
        (Node : Bare_Aspect_Assoc_List; Info : Internal_Entity_Info)
         return Internal_Entity_Aspect_Assoc_List is
      begin
         if Node = null then
            return No_Entity_Aspect_Assoc_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Aspect_Assoc_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Aspect_Clause
        (Node : Bare_Aspect_Clause; Info : Internal_Entity_Info)
         return Internal_Entity_Aspect_Clause is
      begin
         if Node = null then
            return No_Entity_Aspect_Clause;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Aspect_Clause) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Aspect_Spec
        (Node : Bare_Aspect_Spec; Info : Internal_Entity_Info)
         return Internal_Entity_Aspect_Spec is
      begin
         if Node = null then
            return No_Entity_Aspect_Spec;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Aspect_Spec) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Assign_Stmt
        (Node : Bare_Assign_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Assign_Stmt is
      begin
         if Node = null then
            return No_Entity_Assign_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Assign_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Basic_Assoc_List
        (Node : Bare_Basic_Assoc_List; Info : Internal_Entity_Info)
         return Internal_Entity_Basic_Assoc_List is
      begin
         if Node = null then
            return No_Entity_Basic_Assoc_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Basic_Assoc_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Assoc_List
        (Node : Bare_Assoc_List; Info : Internal_Entity_Info)
         return Internal_Entity_Assoc_List is
      begin
         if Node = null then
            return No_Entity_Assoc_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Assoc_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_At_Clause
        (Node : Bare_At_Clause; Info : Internal_Entity_Info)
         return Internal_Entity_At_Clause is
      begin
         if Node = null then
            return No_Entity_At_Clause;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_At_Clause) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Attribute_Def_Clause
        (Node : Bare_Attribute_Def_Clause; Info : Internal_Entity_Info)
         return Internal_Entity_Attribute_Def_Clause is
      begin
         if Node = null then
            return No_Entity_Attribute_Def_Clause;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Attribute_Def_Clause) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Attribute_Ref
        (Node : Bare_Attribute_Ref; Info : Internal_Entity_Info)
         return Internal_Entity_Attribute_Ref is
      begin
         if Node = null then
            return No_Entity_Attribute_Ref;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Attribute_Ref) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Base_Assoc
        (Node : Bare_Base_Assoc; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Assoc is
      begin
         if Node = null then
            return No_Entity_Base_Assoc;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Assoc) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Base_Assoc_List
        (Node : Bare_Base_Assoc_List; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Assoc_List is
      begin
         if Node = null then
            return No_Entity_Base_Assoc_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Assoc_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Base_Formal_Param_Decl
        (Node : Bare_Base_Formal_Param_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Formal_Param_Decl is
      begin
         if Node = null then
            return No_Entity_Base_Formal_Param_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Formal_Param_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Base_Formal_Param_Holder
        (Node : Bare_Base_Formal_Param_Holder; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Formal_Param_Holder is
      begin
         if Node = null then
            return No_Entity_Base_Formal_Param_Holder;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Formal_Param_Holder) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Base_Loop_Stmt
        (Node : Bare_Base_Loop_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Loop_Stmt is
      begin
         if Node = null then
            return No_Entity_Base_Loop_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Loop_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Base_Package_Decl
        (Node : Bare_Base_Package_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Package_Decl is
      begin
         if Node = null then
            return No_Entity_Base_Package_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Package_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Base_Record_Def
        (Node : Bare_Base_Record_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Record_Def is
      begin
         if Node = null then
            return No_Entity_Base_Record_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Record_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Body_Node
        (Node : Bare_Body_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Body_Node is
      begin
         if Node = null then
            return No_Entity_Body_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Body_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Base_Subp_Body
        (Node : Bare_Base_Subp_Body; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Subp_Body is
      begin
         if Node = null then
            return No_Entity_Base_Subp_Body;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Subp_Body) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Base_Subp_Spec
        (Node : Bare_Base_Subp_Spec; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Subp_Spec is
      begin
         if Node = null then
            return No_Entity_Base_Subp_Spec;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Subp_Spec) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Base_Subtype_Decl
        (Node : Bare_Base_Subtype_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Subtype_Decl is
      begin
         if Node = null then
            return No_Entity_Base_Subtype_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Subtype_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Basic_Decl_List
        (Node : Bare_Basic_Decl_List; Info : Internal_Entity_Info)
         return Internal_Entity_Basic_Decl_List is
      begin
         if Node = null then
            return No_Entity_Basic_Decl_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Basic_Decl_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Block_Stmt
        (Node : Bare_Block_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Block_Stmt is
      begin
         if Node = null then
            return No_Entity_Block_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Block_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Begin_Block
        (Node : Bare_Begin_Block; Info : Internal_Entity_Info)
         return Internal_Entity_Begin_Block is
      begin
         if Node = null then
            return No_Entity_Begin_Block;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Begin_Block) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Bin_Op
        (Node : Bare_Bin_Op; Info : Internal_Entity_Info)
         return Internal_Entity_Bin_Op is
      begin
         if Node = null then
            return No_Entity_Bin_Op;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Bin_Op) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Body_Stub
        (Node : Bare_Body_Stub; Info : Internal_Entity_Info)
         return Internal_Entity_Body_Stub is
      begin
         if Node = null then
            return No_Entity_Body_Stub;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Body_Stub) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Box_Expr
        (Node : Bare_Box_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Box_Expr is
      begin
         if Node = null then
            return No_Entity_Box_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Box_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Bracket_Aggregate
        (Node : Bare_Bracket_Aggregate; Info : Internal_Entity_Info)
         return Internal_Entity_Bracket_Aggregate is
      begin
         if Node = null then
            return No_Entity_Bracket_Aggregate;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Bracket_Aggregate) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Delta_Aggregate
        (Node : Bare_Delta_Aggregate; Info : Internal_Entity_Info)
         return Internal_Entity_Delta_Aggregate is
      begin
         if Node = null then
            return No_Entity_Delta_Aggregate;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Delta_Aggregate) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Bracket_Delta_Aggregate
        (Node : Bare_Bracket_Delta_Aggregate; Info : Internal_Entity_Info)
         return Internal_Entity_Bracket_Delta_Aggregate is
      begin
         if Node = null then
            return No_Entity_Bracket_Delta_Aggregate;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Bracket_Delta_Aggregate) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Call_Expr
        (Node : Bare_Call_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Call_Expr is
      begin
         if Node = null then
            return No_Entity_Call_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Call_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Call_Stmt
        (Node : Bare_Call_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Call_Stmt is
      begin
         if Node = null then
            return No_Entity_Call_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Call_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Cond_Expr
        (Node : Bare_Cond_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Cond_Expr is
      begin
         if Node = null then
            return No_Entity_Cond_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Cond_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Case_Expr
        (Node : Bare_Case_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Case_Expr is
      begin
         if Node = null then
            return No_Entity_Case_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Case_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Case_Expr_Alternative
        (Node : Bare_Case_Expr_Alternative; Info : Internal_Entity_Info)
         return Internal_Entity_Case_Expr_Alternative is
      begin
         if Node = null then
            return No_Entity_Case_Expr_Alternative;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Case_Expr_Alternative) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Case_Expr_Alternative_List
        (Node : Bare_Case_Expr_Alternative_List; Info : Internal_Entity_Info)
         return Internal_Entity_Case_Expr_Alternative_List is
      begin
         if Node = null then
            return No_Entity_Case_Expr_Alternative_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Case_Expr_Alternative_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Case_Stmt
        (Node : Bare_Case_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Case_Stmt is
      begin
         if Node = null then
            return No_Entity_Case_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Case_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Case_Stmt_Alternative
        (Node : Bare_Case_Stmt_Alternative; Info : Internal_Entity_Info)
         return Internal_Entity_Case_Stmt_Alternative is
      begin
         if Node = null then
            return No_Entity_Case_Stmt_Alternative;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Case_Stmt_Alternative) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Case_Stmt_Alternative_List
        (Node : Bare_Case_Stmt_Alternative_List; Info : Internal_Entity_Info)
         return Internal_Entity_Case_Stmt_Alternative_List is
      begin
         if Node = null then
            return No_Entity_Case_Stmt_Alternative_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Case_Stmt_Alternative_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Char_Literal
        (Node : Bare_Char_Literal; Info : Internal_Entity_Info)
         return Internal_Entity_Char_Literal is
      begin
         if Node = null then
            return No_Entity_Char_Literal;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Char_Literal) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Classwide_Type_Decl
        (Node : Bare_Classwide_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Classwide_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Classwide_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Classwide_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Compilation_Unit
        (Node : Bare_Compilation_Unit; Info : Internal_Entity_Info)
         return Internal_Entity_Compilation_Unit is
      begin
         if Node = null then
            return No_Entity_Compilation_Unit;
         end if;
         return (Node => Node, Info => Info);
      end;
      ----------
      -- Hash --
      ----------
      pragma Warnings (Off, "referenced");
      function Hash (R : Internal_Entity_Compilation_Unit) return Hash_Type is
         pragma Warnings (On, "referenced");
      begin
            return Combine
              (Hash (R.Node), Hash (R.Info));
      end Hash;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Compilation_Unit) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Compilation_Unit_List
        (Node : Bare_Compilation_Unit_List; Info : Internal_Entity_Info)
         return Internal_Entity_Compilation_Unit_List is
      begin
         if Node = null then
            return No_Entity_Compilation_Unit_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Compilation_Unit_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Component_Clause
        (Node : Bare_Component_Clause; Info : Internal_Entity_Info)
         return Internal_Entity_Component_Clause is
      begin
         if Node = null then
            return No_Entity_Component_Clause;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Component_Clause) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Component_Decl
        (Node : Bare_Component_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Component_Decl is
      begin
         if Node = null then
            return No_Entity_Component_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Component_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Component_Def
        (Node : Bare_Component_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Component_Def is
      begin
         if Node = null then
            return No_Entity_Component_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Component_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Component_List
        (Node : Bare_Component_List; Info : Internal_Entity_Info)
         return Internal_Entity_Component_List is
      begin
         if Node = null then
            return No_Entity_Component_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Component_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Constraint
        (Node : Bare_Constraint; Info : Internal_Entity_Info)
         return Internal_Entity_Constraint is
      begin
         if Node = null then
            return No_Entity_Constraint;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Constraint) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Composite_Constraint
        (Node : Bare_Composite_Constraint; Info : Internal_Entity_Info)
         return Internal_Entity_Composite_Constraint is
      begin
         if Node = null then
            return No_Entity_Composite_Constraint;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Composite_Constraint) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Composite_Constraint_Assoc
        (Node : Bare_Composite_Constraint_Assoc; Info : Internal_Entity_Info)
         return Internal_Entity_Composite_Constraint_Assoc is
      begin
         if Node = null then
            return No_Entity_Composite_Constraint_Assoc;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Composite_Constraint_Assoc) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Concat_Op
        (Node : Bare_Concat_Op; Info : Internal_Entity_Info)
         return Internal_Entity_Concat_Op is
      begin
         if Node = null then
            return No_Entity_Concat_Op;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Concat_Op) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Concat_Operand
        (Node : Bare_Concat_Operand; Info : Internal_Entity_Info)
         return Internal_Entity_Concat_Operand is
      begin
         if Node = null then
            return No_Entity_Concat_Operand;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Concat_Operand) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Concat_Operand_List
        (Node : Bare_Concat_Operand_List; Info : Internal_Entity_Info)
         return Internal_Entity_Concat_Operand_List is
      begin
         if Node = null then
            return No_Entity_Concat_Operand_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Concat_Operand_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Concrete_Formal_Subp_Decl
        (Node : Bare_Concrete_Formal_Subp_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Concrete_Formal_Subp_Decl is
      begin
         if Node = null then
            return No_Entity_Concrete_Formal_Subp_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Concrete_Formal_Subp_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Concrete_Type_Decl
        (Node : Bare_Concrete_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Concrete_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Concrete_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Concrete_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Constant_Node
        (Node : Bare_Constant_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Constant_Node is
      begin
         if Node = null then
            return No_Entity_Constant_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Constant_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Constant_Absent
        (Node : Bare_Constant_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_Constant_Absent is
      begin
         if Node = null then
            return No_Entity_Constant_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Constant_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Constant_Present
        (Node : Bare_Constant_Present; Info : Internal_Entity_Info)
         return Internal_Entity_Constant_Present is
      begin
         if Node = null then
            return No_Entity_Constant_Present;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Constant_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Constrained_Array_Indices
        (Node : Bare_Constrained_Array_Indices; Info : Internal_Entity_Info)
         return Internal_Entity_Constrained_Array_Indices is
      begin
         if Node = null then
            return No_Entity_Constrained_Array_Indices;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Constrained_Array_Indices) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Subtype_Indication
        (Node : Bare_Subtype_Indication; Info : Internal_Entity_Info)
         return Internal_Entity_Subtype_Indication is
      begin
         if Node = null then
            return No_Entity_Subtype_Indication;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Subtype_Indication) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Constrained_Subtype_Indication
        (Node : Bare_Constrained_Subtype_Indication; Info : Internal_Entity_Info)
         return Internal_Entity_Constrained_Subtype_Indication is
      begin
         if Node = null then
            return No_Entity_Constrained_Subtype_Indication;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Constrained_Subtype_Indication) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Constraint_List
        (Node : Bare_Constraint_List; Info : Internal_Entity_Info)
         return Internal_Entity_Constraint_List is
      begin
         if Node = null then
            return No_Entity_Constraint_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Constraint_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Contract_Case_Assoc
        (Node : Bare_Contract_Case_Assoc; Info : Internal_Entity_Info)
         return Internal_Entity_Contract_Case_Assoc is
      begin
         if Node = null then
            return No_Entity_Contract_Case_Assoc;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Contract_Case_Assoc) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Contract_Case_Assoc_List
        (Node : Bare_Contract_Case_Assoc_List; Info : Internal_Entity_Info)
         return Internal_Entity_Contract_Case_Assoc_List is
      begin
         if Node = null then
            return No_Entity_Contract_Case_Assoc_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Contract_Case_Assoc_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Contract_Cases
        (Node : Bare_Contract_Cases; Info : Internal_Entity_Info)
         return Internal_Entity_Contract_Cases is
      begin
         if Node = null then
            return No_Entity_Contract_Cases;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Contract_Cases) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Real_Type_Def
        (Node : Bare_Real_Type_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Real_Type_Def is
      begin
         if Node = null then
            return No_Entity_Real_Type_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Real_Type_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Decimal_Fixed_Point_Def
        (Node : Bare_Decimal_Fixed_Point_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Decimal_Fixed_Point_Def is
      begin
         if Node = null then
            return No_Entity_Decimal_Fixed_Point_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Decimal_Fixed_Point_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Decl_Block
        (Node : Bare_Decl_Block; Info : Internal_Entity_Info)
         return Internal_Entity_Decl_Block is
      begin
         if Node = null then
            return No_Entity_Decl_Block;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Decl_Block) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Decl_Expr
        (Node : Bare_Decl_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Decl_Expr is
      begin
         if Node = null then
            return No_Entity_Decl_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Decl_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Decl_List
        (Node : Bare_Decl_List; Info : Internal_Entity_Info)
         return Internal_Entity_Decl_List is
      begin
         if Node = null then
            return No_Entity_Decl_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Decl_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Declarative_Part
        (Node : Bare_Declarative_Part; Info : Internal_Entity_Info)
         return Internal_Entity_Declarative_Part is
      begin
         if Node = null then
            return No_Entity_Declarative_Part;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Declarative_Part) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Defining_Name
        (Node : Bare_Defining_Name; Info : Internal_Entity_Info)
         return Internal_Entity_Defining_Name is
      begin
         if Node = null then
            return No_Entity_Defining_Name;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Defining_Name) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Defining_Name_List
        (Node : Bare_Defining_Name_List; Info : Internal_Entity_Info)
         return Internal_Entity_Defining_Name_List is
      begin
         if Node = null then
            return No_Entity_Defining_Name_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Defining_Name_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Delay_Stmt
        (Node : Bare_Delay_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Delay_Stmt is
      begin
         if Node = null then
            return No_Entity_Delay_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Delay_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Delta_Constraint
        (Node : Bare_Delta_Constraint; Info : Internal_Entity_Info)
         return Internal_Entity_Delta_Constraint is
      begin
         if Node = null then
            return No_Entity_Delta_Constraint;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Delta_Constraint) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Derived_Type_Def
        (Node : Bare_Derived_Type_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Derived_Type_Def is
      begin
         if Node = null then
            return No_Entity_Derived_Type_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Derived_Type_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Digits_Constraint
        (Node : Bare_Digits_Constraint; Info : Internal_Entity_Info)
         return Internal_Entity_Digits_Constraint is
      begin
         if Node = null then
            return No_Entity_Digits_Constraint;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Digits_Constraint) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Discrete_Base_Subtype_Decl
        (Node : Bare_Discrete_Base_Subtype_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Discrete_Base_Subtype_Decl is
      begin
         if Node = null then
            return No_Entity_Discrete_Base_Subtype_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Discrete_Base_Subtype_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Discrete_Subtype_Indication
        (Node : Bare_Discrete_Subtype_Indication; Info : Internal_Entity_Info)
         return Internal_Entity_Discrete_Subtype_Indication is
      begin
         if Node = null then
            return No_Entity_Discrete_Subtype_Indication;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Discrete_Subtype_Indication) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Discrete_Subtype_Name
        (Node : Bare_Discrete_Subtype_Name; Info : Internal_Entity_Info)
         return Internal_Entity_Discrete_Subtype_Name is
      begin
         if Node = null then
            return No_Entity_Discrete_Subtype_Name;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Discrete_Subtype_Name) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Identifier_List
        (Node : Bare_Identifier_List; Info : Internal_Entity_Info)
         return Internal_Entity_Identifier_List is
      begin
         if Node = null then
            return No_Entity_Identifier_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Identifier_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Discriminant_Choice_List
        (Node : Bare_Discriminant_Choice_List; Info : Internal_Entity_Info)
         return Internal_Entity_Discriminant_Choice_List is
      begin
         if Node = null then
            return No_Entity_Discriminant_Choice_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Discriminant_Choice_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Discriminant_Part
        (Node : Bare_Discriminant_Part; Info : Internal_Entity_Info)
         return Internal_Entity_Discriminant_Part is
      begin
         if Node = null then
            return No_Entity_Discriminant_Part;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Discriminant_Part) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Discriminant_Spec
        (Node : Bare_Discriminant_Spec; Info : Internal_Entity_Info)
         return Internal_Entity_Discriminant_Spec is
      begin
         if Node = null then
            return No_Entity_Discriminant_Spec;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Discriminant_Spec) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Discriminant_Spec_List
        (Node : Bare_Discriminant_Spec_List; Info : Internal_Entity_Info)
         return Internal_Entity_Discriminant_Spec_List is
      begin
         if Node = null then
            return No_Entity_Discriminant_Spec_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Discriminant_Spec_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Dotted_Name
        (Node : Bare_Dotted_Name; Info : Internal_Entity_Info)
         return Internal_Entity_Dotted_Name is
      begin
         if Node = null then
            return No_Entity_Dotted_Name;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Dotted_Name) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Elsif_Expr_Part
        (Node : Bare_Elsif_Expr_Part; Info : Internal_Entity_Info)
         return Internal_Entity_Elsif_Expr_Part is
      begin
         if Node = null then
            return No_Entity_Elsif_Expr_Part;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Elsif_Expr_Part) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Elsif_Expr_Part_List
        (Node : Bare_Elsif_Expr_Part_List; Info : Internal_Entity_Info)
         return Internal_Entity_Elsif_Expr_Part_List is
      begin
         if Node = null then
            return No_Entity_Elsif_Expr_Part_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Elsif_Expr_Part_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Elsif_Stmt_Part
        (Node : Bare_Elsif_Stmt_Part; Info : Internal_Entity_Info)
         return Internal_Entity_Elsif_Stmt_Part is
      begin
         if Node = null then
            return No_Entity_Elsif_Stmt_Part;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Elsif_Stmt_Part) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Elsif_Stmt_Part_List
        (Node : Bare_Elsif_Stmt_Part_List; Info : Internal_Entity_Info)
         return Internal_Entity_Elsif_Stmt_Part_List is
      begin
         if Node = null then
            return No_Entity_Elsif_Stmt_Part_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Elsif_Stmt_Part_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_End_Name
        (Node : Bare_End_Name; Info : Internal_Entity_Info)
         return Internal_Entity_End_Name is
      begin
         if Node = null then
            return No_Entity_End_Name;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_End_Name) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Entry_Body
        (Node : Bare_Entry_Body; Info : Internal_Entity_Info)
         return Internal_Entity_Entry_Body is
      begin
         if Node = null then
            return No_Entity_Entry_Body;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Entry_Body) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Entry_Completion_Formal_Params
        (Node : Bare_Entry_Completion_Formal_Params; Info : Internal_Entity_Info)
         return Internal_Entity_Entry_Completion_Formal_Params is
      begin
         if Node = null then
            return No_Entity_Entry_Completion_Formal_Params;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Entry_Completion_Formal_Params) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Entry_Decl
        (Node : Bare_Entry_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Entry_Decl is
      begin
         if Node = null then
            return No_Entity_Entry_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Entry_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Entry_Index_Spec
        (Node : Bare_Entry_Index_Spec; Info : Internal_Entity_Info)
         return Internal_Entity_Entry_Index_Spec is
      begin
         if Node = null then
            return No_Entity_Entry_Index_Spec;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Entry_Index_Spec) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Entry_Spec
        (Node : Bare_Entry_Spec; Info : Internal_Entity_Info)
         return Internal_Entity_Entry_Spec is
      begin
         if Node = null then
            return No_Entity_Entry_Spec;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Entry_Spec) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Enum_Lit_Synth_Type_Expr
        (Node : Bare_Enum_Lit_Synth_Type_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Enum_Lit_Synth_Type_Expr is
      begin
         if Node = null then
            return No_Entity_Enum_Lit_Synth_Type_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Enum_Lit_Synth_Type_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Enum_Literal_Decl
        (Node : Bare_Enum_Literal_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Enum_Literal_Decl is
      begin
         if Node = null then
            return No_Entity_Enum_Literal_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Enum_Literal_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Enum_Literal_Decl_List
        (Node : Bare_Enum_Literal_Decl_List; Info : Internal_Entity_Info)
         return Internal_Entity_Enum_Literal_Decl_List is
      begin
         if Node = null then
            return No_Entity_Enum_Literal_Decl_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Enum_Literal_Decl_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Enum_Rep_Clause
        (Node : Bare_Enum_Rep_Clause; Info : Internal_Entity_Info)
         return Internal_Entity_Enum_Rep_Clause is
      begin
         if Node = null then
            return No_Entity_Enum_Rep_Clause;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Enum_Rep_Clause) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Enum_Subp_Spec
        (Node : Bare_Enum_Subp_Spec; Info : Internal_Entity_Info)
         return Internal_Entity_Enum_Subp_Spec is
      begin
         if Node = null then
            return No_Entity_Enum_Subp_Spec;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Enum_Subp_Spec) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Enum_Type_Def
        (Node : Bare_Enum_Type_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Enum_Type_Def is
      begin
         if Node = null then
            return No_Entity_Enum_Type_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Enum_Type_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Error_Decl
        (Node : Bare_Error_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Error_Decl is
      begin
         if Node = null then
            return No_Entity_Error_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Error_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Error_Stmt
        (Node : Bare_Error_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Error_Stmt is
      begin
         if Node = null then
            return No_Entity_Error_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Error_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Exception_Decl
        (Node : Bare_Exception_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Exception_Decl is
      begin
         if Node = null then
            return No_Entity_Exception_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Exception_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Exception_Handler
        (Node : Bare_Exception_Handler; Info : Internal_Entity_Info)
         return Internal_Entity_Exception_Handler is
      begin
         if Node = null then
            return No_Entity_Exception_Handler;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Exception_Handler) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Exit_Stmt
        (Node : Bare_Exit_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Exit_Stmt is
      begin
         if Node = null then
            return No_Entity_Exit_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Exit_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Explicit_Deref
        (Node : Bare_Explicit_Deref; Info : Internal_Entity_Info)
         return Internal_Entity_Explicit_Deref is
      begin
         if Node = null then
            return No_Entity_Explicit_Deref;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Explicit_Deref) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Expr_List
        (Node : Bare_Expr_List; Info : Internal_Entity_Info)
         return Internal_Entity_Expr_List is
      begin
         if Node = null then
            return No_Entity_Expr_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Expr_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Expr_Alternatives_List
        (Node : Bare_Expr_Alternatives_List; Info : Internal_Entity_Info)
         return Internal_Entity_Expr_Alternatives_List is
      begin
         if Node = null then
            return No_Entity_Expr_Alternatives_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Expr_Alternatives_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Expr_Function
        (Node : Bare_Expr_Function; Info : Internal_Entity_Info)
         return Internal_Entity_Expr_Function is
      begin
         if Node = null then
            return No_Entity_Expr_Function;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Expr_Function) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Extended_Return_Stmt
        (Node : Bare_Extended_Return_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Extended_Return_Stmt is
      begin
         if Node = null then
            return No_Entity_Extended_Return_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Extended_Return_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Object_Decl
        (Node : Bare_Object_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Object_Decl is
      begin
         if Node = null then
            return No_Entity_Object_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Object_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Extended_Return_Stmt_Object_Decl
        (Node : Bare_Extended_Return_Stmt_Object_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Extended_Return_Stmt_Object_Decl is
      begin
         if Node = null then
            return No_Entity_Extended_Return_Stmt_Object_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Extended_Return_Stmt_Object_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Floating_Point_Def
        (Node : Bare_Floating_Point_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Floating_Point_Def is
      begin
         if Node = null then
            return No_Entity_Floating_Point_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Floating_Point_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Loop_Spec
        (Node : Bare_Loop_Spec; Info : Internal_Entity_Info)
         return Internal_Entity_Loop_Spec is
      begin
         if Node = null then
            return No_Entity_Loop_Spec;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Loop_Spec) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_For_Loop_Spec
        (Node : Bare_For_Loop_Spec; Info : Internal_Entity_Info)
         return Internal_Entity_For_Loop_Spec is
      begin
         if Node = null then
            return No_Entity_For_Loop_Spec;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_For_Loop_Spec) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_For_Loop_Stmt
        (Node : Bare_For_Loop_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_For_Loop_Stmt is
      begin
         if Node = null then
            return No_Entity_For_Loop_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_For_Loop_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_For_Loop_Var_Decl
        (Node : Bare_For_Loop_Var_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_For_Loop_Var_Decl is
      begin
         if Node = null then
            return No_Entity_For_Loop_Var_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_For_Loop_Var_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Formal_Discrete_Type_Def
        (Node : Bare_Formal_Discrete_Type_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Formal_Discrete_Type_Def is
      begin
         if Node = null then
            return No_Entity_Formal_Discrete_Type_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Formal_Discrete_Type_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Formal_Type_Decl
        (Node : Bare_Formal_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Formal_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Formal_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Formal_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Decl
        (Node : Bare_Generic_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Decl is
      begin
         if Node = null then
            return No_Entity_Generic_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Formal
        (Node : Bare_Generic_Formal; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Formal is
      begin
         if Node = null then
            return No_Entity_Generic_Formal;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Formal) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Formal_Obj_Decl
        (Node : Bare_Generic_Formal_Obj_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Formal_Obj_Decl is
      begin
         if Node = null then
            return No_Entity_Generic_Formal_Obj_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Formal_Obj_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Formal_Package
        (Node : Bare_Generic_Formal_Package; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Formal_Package is
      begin
         if Node = null then
            return No_Entity_Generic_Formal_Package;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Formal_Package) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Formal_Part
        (Node : Bare_Generic_Formal_Part; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Formal_Part is
      begin
         if Node = null then
            return No_Entity_Generic_Formal_Part;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Formal_Part) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Formal_Subp_Decl
        (Node : Bare_Generic_Formal_Subp_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Formal_Subp_Decl is
      begin
         if Node = null then
            return No_Entity_Generic_Formal_Subp_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Formal_Subp_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Formal_Type_Decl
        (Node : Bare_Generic_Formal_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Formal_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Generic_Formal_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Formal_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Instantiation
        (Node : Bare_Generic_Instantiation; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Instantiation is
      begin
         if Node = null then
            return No_Entity_Generic_Instantiation;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Instantiation) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Package_Decl
        (Node : Bare_Generic_Package_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Package_Decl is
      begin
         if Node = null then
            return No_Entity_Generic_Package_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Package_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Package_Instantiation
        (Node : Bare_Generic_Package_Instantiation; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Package_Instantiation is
      begin
         if Node = null then
            return No_Entity_Generic_Package_Instantiation;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Package_Instantiation) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Package_Internal
        (Node : Bare_Generic_Package_Internal; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Package_Internal is
      begin
         if Node = null then
            return No_Entity_Generic_Package_Internal;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Package_Internal) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Renaming_Decl
        (Node : Bare_Generic_Renaming_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Renaming_Decl is
      begin
         if Node = null then
            return No_Entity_Generic_Renaming_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Renaming_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Package_Renaming_Decl
        (Node : Bare_Generic_Package_Renaming_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Package_Renaming_Decl is
      begin
         if Node = null then
            return No_Entity_Generic_Package_Renaming_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Package_Renaming_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Subp_Decl
        (Node : Bare_Generic_Subp_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Subp_Decl is
      begin
         if Node = null then
            return No_Entity_Generic_Subp_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Subp_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Subp_Instantiation
        (Node : Bare_Generic_Subp_Instantiation; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Subp_Instantiation is
      begin
         if Node = null then
            return No_Entity_Generic_Subp_Instantiation;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Subp_Instantiation) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Subp_Internal
        (Node : Bare_Generic_Subp_Internal; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Subp_Internal is
      begin
         if Node = null then
            return No_Entity_Generic_Subp_Internal;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Subp_Internal) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Subp_Renaming_Decl
        (Node : Bare_Generic_Subp_Renaming_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Subp_Renaming_Decl is
      begin
         if Node = null then
            return No_Entity_Generic_Subp_Renaming_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Subp_Renaming_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Goto_Stmt
        (Node : Bare_Goto_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Goto_Stmt is
      begin
         if Node = null then
            return No_Entity_Goto_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Goto_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Handled_Stmts
        (Node : Bare_Handled_Stmts; Info : Internal_Entity_Info)
         return Internal_Entity_Handled_Stmts is
      begin
         if Node = null then
            return No_Entity_Handled_Stmts;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Handled_Stmts) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_If_Expr
        (Node : Bare_If_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_If_Expr is
      begin
         if Node = null then
            return No_Entity_If_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_If_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_If_Stmt
        (Node : Bare_If_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_If_Stmt is
      begin
         if Node = null then
            return No_Entity_If_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_If_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Incomplete_Type_Decl
        (Node : Bare_Incomplete_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Incomplete_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Incomplete_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Incomplete_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Incomplete_Formal_Type_Decl
        (Node : Bare_Incomplete_Formal_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Incomplete_Formal_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Incomplete_Formal_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Incomplete_Formal_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Incomplete_Tagged_Type_Decl
        (Node : Bare_Incomplete_Tagged_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Incomplete_Tagged_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Incomplete_Tagged_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Incomplete_Tagged_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Num_Literal
        (Node : Bare_Num_Literal; Info : Internal_Entity_Info)
         return Internal_Entity_Num_Literal is
      begin
         if Node = null then
            return No_Entity_Num_Literal;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Num_Literal) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Int_Literal
        (Node : Bare_Int_Literal; Info : Internal_Entity_Info)
         return Internal_Entity_Int_Literal is
      begin
         if Node = null then
            return No_Entity_Int_Literal;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Int_Literal) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Interface_Kind
        (Node : Bare_Interface_Kind; Info : Internal_Entity_Info)
         return Internal_Entity_Interface_Kind is
      begin
         if Node = null then
            return No_Entity_Interface_Kind;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Interface_Kind) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Interface_Kind_Limited
        (Node : Bare_Interface_Kind_Limited; Info : Internal_Entity_Info)
         return Internal_Entity_Interface_Kind_Limited is
      begin
         if Node = null then
            return No_Entity_Interface_Kind_Limited;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Interface_Kind_Limited) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Interface_Kind_Protected
        (Node : Bare_Interface_Kind_Protected; Info : Internal_Entity_Info)
         return Internal_Entity_Interface_Kind_Protected is
      begin
         if Node = null then
            return No_Entity_Interface_Kind_Protected;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Interface_Kind_Protected) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Interface_Kind_Synchronized
        (Node : Bare_Interface_Kind_Synchronized; Info : Internal_Entity_Info)
         return Internal_Entity_Interface_Kind_Synchronized is
      begin
         if Node = null then
            return No_Entity_Interface_Kind_Synchronized;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Interface_Kind_Synchronized) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Interface_Kind_Task
        (Node : Bare_Interface_Kind_Task; Info : Internal_Entity_Info)
         return Internal_Entity_Interface_Kind_Task is
      begin
         if Node = null then
            return No_Entity_Interface_Kind_Task;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Interface_Kind_Task) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Interface_Type_Def
        (Node : Bare_Interface_Type_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Interface_Type_Def is
      begin
         if Node = null then
            return No_Entity_Interface_Type_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Interface_Type_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Iter_Type
        (Node : Bare_Iter_Type; Info : Internal_Entity_Info)
         return Internal_Entity_Iter_Type is
      begin
         if Node = null then
            return No_Entity_Iter_Type;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Iter_Type) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Iter_Type_In
        (Node : Bare_Iter_Type_In; Info : Internal_Entity_Info)
         return Internal_Entity_Iter_Type_In is
      begin
         if Node = null then
            return No_Entity_Iter_Type_In;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Iter_Type_In) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Iter_Type_Of
        (Node : Bare_Iter_Type_Of; Info : Internal_Entity_Info)
         return Internal_Entity_Iter_Type_Of is
      begin
         if Node = null then
            return No_Entity_Iter_Type_Of;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Iter_Type_Of) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Iterated_Assoc
        (Node : Bare_Iterated_Assoc; Info : Internal_Entity_Info)
         return Internal_Entity_Iterated_Assoc is
      begin
         if Node = null then
            return No_Entity_Iterated_Assoc;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Iterated_Assoc) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Known_Discriminant_Part
        (Node : Bare_Known_Discriminant_Part; Info : Internal_Entity_Info)
         return Internal_Entity_Known_Discriminant_Part is
      begin
         if Node = null then
            return No_Entity_Known_Discriminant_Part;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Known_Discriminant_Part) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Label
        (Node : Bare_Label; Info : Internal_Entity_Info)
         return Internal_Entity_Label is
      begin
         if Node = null then
            return No_Entity_Label;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Label) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Label_Decl
        (Node : Bare_Label_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Label_Decl is
      begin
         if Node = null then
            return No_Entity_Label_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Label_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Library_Item
        (Node : Bare_Library_Item; Info : Internal_Entity_Info)
         return Internal_Entity_Library_Item is
      begin
         if Node = null then
            return No_Entity_Library_Item;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Library_Item) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Limited_Node
        (Node : Bare_Limited_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Limited_Node is
      begin
         if Node = null then
            return No_Entity_Limited_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Limited_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Limited_Absent
        (Node : Bare_Limited_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_Limited_Absent is
      begin
         if Node = null then
            return No_Entity_Limited_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Limited_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Limited_Present
        (Node : Bare_Limited_Present; Info : Internal_Entity_Info)
         return Internal_Entity_Limited_Present is
      begin
         if Node = null then
            return No_Entity_Limited_Present;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Limited_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Loop_Stmt
        (Node : Bare_Loop_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Loop_Stmt is
      begin
         if Node = null then
            return No_Entity_Loop_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Loop_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Membership_Expr
        (Node : Bare_Membership_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Membership_Expr is
      begin
         if Node = null then
            return No_Entity_Membership_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Membership_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Mod_Int_Type_Def
        (Node : Bare_Mod_Int_Type_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Mod_Int_Type_Def is
      begin
         if Node = null then
            return No_Entity_Mod_Int_Type_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Mod_Int_Type_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Mode
        (Node : Bare_Mode; Info : Internal_Entity_Info)
         return Internal_Entity_Mode is
      begin
         if Node = null then
            return No_Entity_Mode;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Mode) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Mode_Default
        (Node : Bare_Mode_Default; Info : Internal_Entity_Info)
         return Internal_Entity_Mode_Default is
      begin
         if Node = null then
            return No_Entity_Mode_Default;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Mode_Default) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Mode_In
        (Node : Bare_Mode_In; Info : Internal_Entity_Info)
         return Internal_Entity_Mode_In is
      begin
         if Node = null then
            return No_Entity_Mode_In;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Mode_In) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Mode_In_Out
        (Node : Bare_Mode_In_Out; Info : Internal_Entity_Info)
         return Internal_Entity_Mode_In_Out is
      begin
         if Node = null then
            return No_Entity_Mode_In_Out;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Mode_In_Out) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Mode_Out
        (Node : Bare_Mode_Out; Info : Internal_Entity_Info)
         return Internal_Entity_Mode_Out is
      begin
         if Node = null then
            return No_Entity_Mode_Out;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Mode_Out) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Multi_Abstract_State_Decl
        (Node : Bare_Multi_Abstract_State_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Multi_Abstract_State_Decl is
      begin
         if Node = null then
            return No_Entity_Multi_Abstract_State_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Multi_Abstract_State_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Multi_Dim_Array_Assoc
        (Node : Bare_Multi_Dim_Array_Assoc; Info : Internal_Entity_Info)
         return Internal_Entity_Multi_Dim_Array_Assoc is
      begin
         if Node = null then
            return No_Entity_Multi_Dim_Array_Assoc;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Multi_Dim_Array_Assoc) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Name_List
        (Node : Bare_Name_List; Info : Internal_Entity_Info)
         return Internal_Entity_Name_List is
      begin
         if Node = null then
            return No_Entity_Name_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Name_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Named_Stmt
        (Node : Bare_Named_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Named_Stmt is
      begin
         if Node = null then
            return No_Entity_Named_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Named_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Named_Stmt_Decl
        (Node : Bare_Named_Stmt_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Named_Stmt_Decl is
      begin
         if Node = null then
            return No_Entity_Named_Stmt_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Named_Stmt_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_No_Type_Object_Renaming_Decl
        (Node : Bare_No_Type_Object_Renaming_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_No_Type_Object_Renaming_Decl is
      begin
         if Node = null then
            return No_Entity_No_Type_Object_Renaming_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_No_Type_Object_Renaming_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Not_Null
        (Node : Bare_Not_Null; Info : Internal_Entity_Info)
         return Internal_Entity_Not_Null is
      begin
         if Node = null then
            return No_Entity_Not_Null;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Not_Null) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Not_Null_Absent
        (Node : Bare_Not_Null_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_Not_Null_Absent is
      begin
         if Node = null then
            return No_Entity_Not_Null_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Not_Null_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Not_Null_Present
        (Node : Bare_Not_Null_Present; Info : Internal_Entity_Info)
         return Internal_Entity_Not_Null_Present is
      begin
         if Node = null then
            return No_Entity_Not_Null_Present;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Not_Null_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Null_Component_Decl
        (Node : Bare_Null_Component_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Null_Component_Decl is
      begin
         if Node = null then
            return No_Entity_Null_Component_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Null_Component_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Null_Literal
        (Node : Bare_Null_Literal; Info : Internal_Entity_Info)
         return Internal_Entity_Null_Literal is
      begin
         if Node = null then
            return No_Entity_Null_Literal;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Null_Literal) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Null_Record_Aggregate
        (Node : Bare_Null_Record_Aggregate; Info : Internal_Entity_Info)
         return Internal_Entity_Null_Record_Aggregate is
      begin
         if Node = null then
            return No_Entity_Null_Record_Aggregate;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Null_Record_Aggregate) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Null_Record_Def
        (Node : Bare_Null_Record_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Null_Record_Def is
      begin
         if Node = null then
            return No_Entity_Null_Record_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Null_Record_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Null_Stmt
        (Node : Bare_Null_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Null_Stmt is
      begin
         if Node = null then
            return No_Entity_Null_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Null_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Null_Subp_Decl
        (Node : Bare_Null_Subp_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Null_Subp_Decl is
      begin
         if Node = null then
            return No_Entity_Null_Subp_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Null_Subp_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Number_Decl
        (Node : Bare_Number_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Number_Decl is
      begin
         if Node = null then
            return No_Entity_Number_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Number_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op
        (Node : Bare_Op; Info : Internal_Entity_Info)
         return Internal_Entity_Op is
      begin
         if Node = null then
            return No_Entity_Op;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Abs
        (Node : Bare_Op_Abs; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Abs is
      begin
         if Node = null then
            return No_Entity_Op_Abs;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Abs) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_And
        (Node : Bare_Op_And; Info : Internal_Entity_Info)
         return Internal_Entity_Op_And is
      begin
         if Node = null then
            return No_Entity_Op_And;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_And) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_And_Then
        (Node : Bare_Op_And_Then; Info : Internal_Entity_Info)
         return Internal_Entity_Op_And_Then is
      begin
         if Node = null then
            return No_Entity_Op_And_Then;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_And_Then) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Concat
        (Node : Bare_Op_Concat; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Concat is
      begin
         if Node = null then
            return No_Entity_Op_Concat;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Concat) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Div
        (Node : Bare_Op_Div; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Div is
      begin
         if Node = null then
            return No_Entity_Op_Div;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Div) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Double_Dot
        (Node : Bare_Op_Double_Dot; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Double_Dot is
      begin
         if Node = null then
            return No_Entity_Op_Double_Dot;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Double_Dot) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Eq
        (Node : Bare_Op_Eq; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Eq is
      begin
         if Node = null then
            return No_Entity_Op_Eq;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Eq) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Gt
        (Node : Bare_Op_Gt; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Gt is
      begin
         if Node = null then
            return No_Entity_Op_Gt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Gt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Gte
        (Node : Bare_Op_Gte; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Gte is
      begin
         if Node = null then
            return No_Entity_Op_Gte;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Gte) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_In
        (Node : Bare_Op_In; Info : Internal_Entity_Info)
         return Internal_Entity_Op_In is
      begin
         if Node = null then
            return No_Entity_Op_In;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_In) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Lt
        (Node : Bare_Op_Lt; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Lt is
      begin
         if Node = null then
            return No_Entity_Op_Lt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Lt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Lte
        (Node : Bare_Op_Lte; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Lte is
      begin
         if Node = null then
            return No_Entity_Op_Lte;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Lte) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Minus
        (Node : Bare_Op_Minus; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Minus is
      begin
         if Node = null then
            return No_Entity_Op_Minus;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Minus) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Mod
        (Node : Bare_Op_Mod; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Mod is
      begin
         if Node = null then
            return No_Entity_Op_Mod;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Mod) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Mult
        (Node : Bare_Op_Mult; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Mult is
      begin
         if Node = null then
            return No_Entity_Op_Mult;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Mult) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Neq
        (Node : Bare_Op_Neq; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Neq is
      begin
         if Node = null then
            return No_Entity_Op_Neq;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Neq) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Not
        (Node : Bare_Op_Not; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Not is
      begin
         if Node = null then
            return No_Entity_Op_Not;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Not) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Not_In
        (Node : Bare_Op_Not_In; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Not_In is
      begin
         if Node = null then
            return No_Entity_Op_Not_In;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Not_In) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Or
        (Node : Bare_Op_Or; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Or is
      begin
         if Node = null then
            return No_Entity_Op_Or;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Or) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Or_Else
        (Node : Bare_Op_Or_Else; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Or_Else is
      begin
         if Node = null then
            return No_Entity_Op_Or_Else;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Or_Else) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Plus
        (Node : Bare_Op_Plus; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Plus is
      begin
         if Node = null then
            return No_Entity_Op_Plus;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Plus) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Pow
        (Node : Bare_Op_Pow; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Pow is
      begin
         if Node = null then
            return No_Entity_Op_Pow;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Pow) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Rem
        (Node : Bare_Op_Rem; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Rem is
      begin
         if Node = null then
            return No_Entity_Op_Rem;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Rem) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Xor
        (Node : Bare_Op_Xor; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Xor is
      begin
         if Node = null then
            return No_Entity_Op_Xor;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Xor) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Ordinary_Fixed_Point_Def
        (Node : Bare_Ordinary_Fixed_Point_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Ordinary_Fixed_Point_Def is
      begin
         if Node = null then
            return No_Entity_Ordinary_Fixed_Point_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Ordinary_Fixed_Point_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Others_Designator
        (Node : Bare_Others_Designator; Info : Internal_Entity_Info)
         return Internal_Entity_Others_Designator is
      begin
         if Node = null then
            return No_Entity_Others_Designator;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Others_Designator) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Overriding_Node
        (Node : Bare_Overriding_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Overriding_Node is
      begin
         if Node = null then
            return No_Entity_Overriding_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Overriding_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Overriding_Not_Overriding
        (Node : Bare_Overriding_Not_Overriding; Info : Internal_Entity_Info)
         return Internal_Entity_Overriding_Not_Overriding is
      begin
         if Node = null then
            return No_Entity_Overriding_Not_Overriding;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Overriding_Not_Overriding) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Overriding_Overriding
        (Node : Bare_Overriding_Overriding; Info : Internal_Entity_Info)
         return Internal_Entity_Overriding_Overriding is
      begin
         if Node = null then
            return No_Entity_Overriding_Overriding;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Overriding_Overriding) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Overriding_Unspecified
        (Node : Bare_Overriding_Unspecified; Info : Internal_Entity_Info)
         return Internal_Entity_Overriding_Unspecified is
      begin
         if Node = null then
            return No_Entity_Overriding_Unspecified;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Overriding_Unspecified) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Package_Body
        (Node : Bare_Package_Body; Info : Internal_Entity_Info)
         return Internal_Entity_Package_Body is
      begin
         if Node = null then
            return No_Entity_Package_Body;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Package_Body) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Package_Body_Stub
        (Node : Bare_Package_Body_Stub; Info : Internal_Entity_Info)
         return Internal_Entity_Package_Body_Stub is
      begin
         if Node = null then
            return No_Entity_Package_Body_Stub;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Package_Body_Stub) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Package_Decl
        (Node : Bare_Package_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Package_Decl is
      begin
         if Node = null then
            return No_Entity_Package_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Package_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Package_Renaming_Decl
        (Node : Bare_Package_Renaming_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Package_Renaming_Decl is
      begin
         if Node = null then
            return No_Entity_Package_Renaming_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Package_Renaming_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Param_Assoc
        (Node : Bare_Param_Assoc; Info : Internal_Entity_Info)
         return Internal_Entity_Param_Assoc is
      begin
         if Node = null then
            return No_Entity_Param_Assoc;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Param_Assoc) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Param_Spec
        (Node : Bare_Param_Spec; Info : Internal_Entity_Info)
         return Internal_Entity_Param_Spec is
      begin
         if Node = null then
            return No_Entity_Param_Spec;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Param_Spec) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Param_Spec_List
        (Node : Bare_Param_Spec_List; Info : Internal_Entity_Info)
         return Internal_Entity_Param_Spec_List is
      begin
         if Node = null then
            return No_Entity_Param_Spec_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Param_Spec_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Params
        (Node : Bare_Params; Info : Internal_Entity_Info)
         return Internal_Entity_Params is
      begin
         if Node = null then
            return No_Entity_Params;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Params) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Paren_Abstract_State_Decl
        (Node : Bare_Paren_Abstract_State_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Paren_Abstract_State_Decl is
      begin
         if Node = null then
            return No_Entity_Paren_Abstract_State_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Paren_Abstract_State_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Paren_Expr
        (Node : Bare_Paren_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Paren_Expr is
      begin
         if Node = null then
            return No_Entity_Paren_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Paren_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Parent_List
        (Node : Bare_Parent_List; Info : Internal_Entity_Info)
         return Internal_Entity_Parent_List is
      begin
         if Node = null then
            return No_Entity_Parent_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Parent_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Pp_Directive
        (Node : Bare_Pp_Directive; Info : Internal_Entity_Info)
         return Internal_Entity_Pp_Directive is
      begin
         if Node = null then
            return No_Entity_Pp_Directive;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Pp_Directive) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Pp_Else_Directive
        (Node : Bare_Pp_Else_Directive; Info : Internal_Entity_Info)
         return Internal_Entity_Pp_Else_Directive is
      begin
         if Node = null then
            return No_Entity_Pp_Else_Directive;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Pp_Else_Directive) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Pp_Elsif_Directive
        (Node : Bare_Pp_Elsif_Directive; Info : Internal_Entity_Info)
         return Internal_Entity_Pp_Elsif_Directive is
      begin
         if Node = null then
            return No_Entity_Pp_Elsif_Directive;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Pp_Elsif_Directive) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Pp_End_If_Directive
        (Node : Bare_Pp_End_If_Directive; Info : Internal_Entity_Info)
         return Internal_Entity_Pp_End_If_Directive is
      begin
         if Node = null then
            return No_Entity_Pp_End_If_Directive;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Pp_End_If_Directive) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Pp_If_Directive
        (Node : Bare_Pp_If_Directive; Info : Internal_Entity_Info)
         return Internal_Entity_Pp_If_Directive is
      begin
         if Node = null then
            return No_Entity_Pp_If_Directive;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Pp_If_Directive) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Pp_Then_Kw
        (Node : Bare_Pp_Then_Kw; Info : Internal_Entity_Info)
         return Internal_Entity_Pp_Then_Kw is
      begin
         if Node = null then
            return No_Entity_Pp_Then_Kw;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Pp_Then_Kw) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Pragma_Argument_Assoc
        (Node : Bare_Pragma_Argument_Assoc; Info : Internal_Entity_Info)
         return Internal_Entity_Pragma_Argument_Assoc is
      begin
         if Node = null then
            return No_Entity_Pragma_Argument_Assoc;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Pragma_Argument_Assoc) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Pragma_Node
        (Node : Bare_Pragma_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Pragma_Node is
      begin
         if Node = null then
            return No_Entity_Pragma_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Pragma_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Pragma_Node_List
        (Node : Bare_Pragma_Node_List; Info : Internal_Entity_Info)
         return Internal_Entity_Pragma_Node_List is
      begin
         if Node = null then
            return No_Entity_Pragma_Node_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Pragma_Node_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Private_Node
        (Node : Bare_Private_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Private_Node is
      begin
         if Node = null then
            return No_Entity_Private_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Private_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Private_Absent
        (Node : Bare_Private_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_Private_Absent is
      begin
         if Node = null then
            return No_Entity_Private_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Private_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Private_Part
        (Node : Bare_Private_Part; Info : Internal_Entity_Info)
         return Internal_Entity_Private_Part is
      begin
         if Node = null then
            return No_Entity_Private_Part;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Private_Part) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Private_Present
        (Node : Bare_Private_Present; Info : Internal_Entity_Info)
         return Internal_Entity_Private_Present is
      begin
         if Node = null then
            return No_Entity_Private_Present;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Private_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Private_Type_Def
        (Node : Bare_Private_Type_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Private_Type_Def is
      begin
         if Node = null then
            return No_Entity_Private_Type_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Private_Type_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Protected_Node
        (Node : Bare_Protected_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Protected_Node is
      begin
         if Node = null then
            return No_Entity_Protected_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Protected_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Protected_Absent
        (Node : Bare_Protected_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_Protected_Absent is
      begin
         if Node = null then
            return No_Entity_Protected_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Protected_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Protected_Body
        (Node : Bare_Protected_Body; Info : Internal_Entity_Info)
         return Internal_Entity_Protected_Body is
      begin
         if Node = null then
            return No_Entity_Protected_Body;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Protected_Body) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Protected_Body_Stub
        (Node : Bare_Protected_Body_Stub; Info : Internal_Entity_Info)
         return Internal_Entity_Protected_Body_Stub is
      begin
         if Node = null then
            return No_Entity_Protected_Body_Stub;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Protected_Body_Stub) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Protected_Def
        (Node : Bare_Protected_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Protected_Def is
      begin
         if Node = null then
            return No_Entity_Protected_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Protected_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Protected_Present
        (Node : Bare_Protected_Present; Info : Internal_Entity_Info)
         return Internal_Entity_Protected_Present is
      begin
         if Node = null then
            return No_Entity_Protected_Present;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Protected_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Protected_Type_Decl
        (Node : Bare_Protected_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Protected_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Protected_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Protected_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Public_Part
        (Node : Bare_Public_Part; Info : Internal_Entity_Info)
         return Internal_Entity_Public_Part is
      begin
         if Node = null then
            return No_Entity_Public_Part;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Public_Part) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Qual_Expr
        (Node : Bare_Qual_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Qual_Expr is
      begin
         if Node = null then
            return No_Entity_Qual_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Qual_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Quantified_Expr
        (Node : Bare_Quantified_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Quantified_Expr is
      begin
         if Node = null then
            return No_Entity_Quantified_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Quantified_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Quantifier
        (Node : Bare_Quantifier; Info : Internal_Entity_Info)
         return Internal_Entity_Quantifier is
      begin
         if Node = null then
            return No_Entity_Quantifier;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Quantifier) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Quantifier_All
        (Node : Bare_Quantifier_All; Info : Internal_Entity_Info)
         return Internal_Entity_Quantifier_All is
      begin
         if Node = null then
            return No_Entity_Quantifier_All;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Quantifier_All) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Quantifier_Some
        (Node : Bare_Quantifier_Some; Info : Internal_Entity_Info)
         return Internal_Entity_Quantifier_Some is
      begin
         if Node = null then
            return No_Entity_Quantifier_Some;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Quantifier_Some) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Raise_Expr
        (Node : Bare_Raise_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Raise_Expr is
      begin
         if Node = null then
            return No_Entity_Raise_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Raise_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Raise_Stmt
        (Node : Bare_Raise_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Raise_Stmt is
      begin
         if Node = null then
            return No_Entity_Raise_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Raise_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Range_Constraint
        (Node : Bare_Range_Constraint; Info : Internal_Entity_Info)
         return Internal_Entity_Range_Constraint is
      begin
         if Node = null then
            return No_Entity_Range_Constraint;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Range_Constraint) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Range_Spec
        (Node : Bare_Range_Spec; Info : Internal_Entity_Info)
         return Internal_Entity_Range_Spec is
      begin
         if Node = null then
            return No_Entity_Range_Spec;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Range_Spec) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Real_Literal
        (Node : Bare_Real_Literal; Info : Internal_Entity_Info)
         return Internal_Entity_Real_Literal is
      begin
         if Node = null then
            return No_Entity_Real_Literal;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Real_Literal) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Record_Def
        (Node : Bare_Record_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Record_Def is
      begin
         if Node = null then
            return No_Entity_Record_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Record_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Record_Rep_Clause
        (Node : Bare_Record_Rep_Clause; Info : Internal_Entity_Info)
         return Internal_Entity_Record_Rep_Clause is
      begin
         if Node = null then
            return No_Entity_Record_Rep_Clause;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Record_Rep_Clause) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Record_Type_Def
        (Node : Bare_Record_Type_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Record_Type_Def is
      begin
         if Node = null then
            return No_Entity_Record_Type_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Record_Type_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Reduce_Attribute_Ref
        (Node : Bare_Reduce_Attribute_Ref; Info : Internal_Entity_Info)
         return Internal_Entity_Reduce_Attribute_Ref is
      begin
         if Node = null then
            return No_Entity_Reduce_Attribute_Ref;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Reduce_Attribute_Ref) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Relation_Op
        (Node : Bare_Relation_Op; Info : Internal_Entity_Info)
         return Internal_Entity_Relation_Op is
      begin
         if Node = null then
            return No_Entity_Relation_Op;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Relation_Op) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Renaming_Clause
        (Node : Bare_Renaming_Clause; Info : Internal_Entity_Info)
         return Internal_Entity_Renaming_Clause is
      begin
         if Node = null then
            return No_Entity_Renaming_Clause;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Renaming_Clause) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Requeue_Stmt
        (Node : Bare_Requeue_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Requeue_Stmt is
      begin
         if Node = null then
            return No_Entity_Requeue_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Requeue_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Return_Stmt
        (Node : Bare_Return_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Return_Stmt is
      begin
         if Node = null then
            return No_Entity_Return_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Return_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Reverse_Node
        (Node : Bare_Reverse_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Reverse_Node is
      begin
         if Node = null then
            return No_Entity_Reverse_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Reverse_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Reverse_Absent
        (Node : Bare_Reverse_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_Reverse_Absent is
      begin
         if Node = null then
            return No_Entity_Reverse_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Reverse_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Reverse_Present
        (Node : Bare_Reverse_Present; Info : Internal_Entity_Info)
         return Internal_Entity_Reverse_Present is
      begin
         if Node = null then
            return No_Entity_Reverse_Present;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Reverse_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Select_Stmt
        (Node : Bare_Select_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Select_Stmt is
      begin
         if Node = null then
            return No_Entity_Select_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Select_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Select_When_Part
        (Node : Bare_Select_When_Part; Info : Internal_Entity_Info)
         return Internal_Entity_Select_When_Part is
      begin
         if Node = null then
            return No_Entity_Select_When_Part;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Select_When_Part) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Select_When_Part_List
        (Node : Bare_Select_When_Part_List; Info : Internal_Entity_Info)
         return Internal_Entity_Select_When_Part_List is
      begin
         if Node = null then
            return No_Entity_Select_When_Part_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Select_When_Part_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Signed_Int_Type_Def
        (Node : Bare_Signed_Int_Type_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Signed_Int_Type_Def is
      begin
         if Node = null then
            return No_Entity_Signed_Int_Type_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Signed_Int_Type_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Single_Protected_Decl
        (Node : Bare_Single_Protected_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Single_Protected_Decl is
      begin
         if Node = null then
            return No_Entity_Single_Protected_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Single_Protected_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Single_Task_Decl
        (Node : Bare_Single_Task_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Single_Task_Decl is
      begin
         if Node = null then
            return No_Entity_Single_Task_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Single_Task_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Task_Type_Decl
        (Node : Bare_Task_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Task_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Task_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Task_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Single_Task_Type_Decl
        (Node : Bare_Single_Task_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Single_Task_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Single_Task_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Single_Task_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Stmt_List
        (Node : Bare_Stmt_List; Info : Internal_Entity_Info)
         return Internal_Entity_Stmt_List is
      begin
         if Node = null then
            return No_Entity_Stmt_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Stmt_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_String_Literal
        (Node : Bare_String_Literal; Info : Internal_Entity_Info)
         return Internal_Entity_String_Literal is
      begin
         if Node = null then
            return No_Entity_String_Literal;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_String_Literal) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Subp_Body
        (Node : Bare_Subp_Body; Info : Internal_Entity_Info)
         return Internal_Entity_Subp_Body is
      begin
         if Node = null then
            return No_Entity_Subp_Body;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Subp_Body) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Subp_Body_Stub
        (Node : Bare_Subp_Body_Stub; Info : Internal_Entity_Info)
         return Internal_Entity_Subp_Body_Stub is
      begin
         if Node = null then
            return No_Entity_Subp_Body_Stub;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Subp_Body_Stub) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Subp_Decl
        (Node : Bare_Subp_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Subp_Decl is
      begin
         if Node = null then
            return No_Entity_Subp_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Subp_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Subp_Kind
        (Node : Bare_Subp_Kind; Info : Internal_Entity_Info)
         return Internal_Entity_Subp_Kind is
      begin
         if Node = null then
            return No_Entity_Subp_Kind;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Subp_Kind) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Subp_Kind_Function
        (Node : Bare_Subp_Kind_Function; Info : Internal_Entity_Info)
         return Internal_Entity_Subp_Kind_Function is
      begin
         if Node = null then
            return No_Entity_Subp_Kind_Function;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Subp_Kind_Function) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Subp_Kind_Procedure
        (Node : Bare_Subp_Kind_Procedure; Info : Internal_Entity_Info)
         return Internal_Entity_Subp_Kind_Procedure is
      begin
         if Node = null then
            return No_Entity_Subp_Kind_Procedure;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Subp_Kind_Procedure) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Subp_Renaming_Decl
        (Node : Bare_Subp_Renaming_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Subp_Renaming_Decl is
      begin
         if Node = null then
            return No_Entity_Subp_Renaming_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Subp_Renaming_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Subp_Spec
        (Node : Bare_Subp_Spec; Info : Internal_Entity_Info)
         return Internal_Entity_Subp_Spec is
      begin
         if Node = null then
            return No_Entity_Subp_Spec;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Subp_Spec) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Subtype_Decl
        (Node : Bare_Subtype_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Subtype_Decl is
      begin
         if Node = null then
            return No_Entity_Subtype_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Subtype_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Subunit
        (Node : Bare_Subunit; Info : Internal_Entity_Info)
         return Internal_Entity_Subunit is
      begin
         if Node = null then
            return No_Entity_Subunit;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Subunit) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Synchronized_Node
        (Node : Bare_Synchronized_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Synchronized_Node is
      begin
         if Node = null then
            return No_Entity_Synchronized_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synchronized_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Synchronized_Absent
        (Node : Bare_Synchronized_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_Synchronized_Absent is
      begin
         if Node = null then
            return No_Entity_Synchronized_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synchronized_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Synchronized_Present
        (Node : Bare_Synchronized_Present; Info : Internal_Entity_Info)
         return Internal_Entity_Synchronized_Present is
      begin
         if Node = null then
            return No_Entity_Synchronized_Present;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synchronized_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Synth_Anonymous_Type_Decl
        (Node : Bare_Synth_Anonymous_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Synth_Anonymous_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Synth_Anonymous_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synth_Anonymous_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Synthetic_Binary_Spec
        (Node : Bare_Synthetic_Binary_Spec; Info : Internal_Entity_Info)
         return Internal_Entity_Synthetic_Binary_Spec is
      begin
         if Node = null then
            return No_Entity_Synthetic_Binary_Spec;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synthetic_Binary_Spec) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Synthetic_Char_Enum_Lit
        (Node : Bare_Synthetic_Char_Enum_Lit; Info : Internal_Entity_Info)
         return Internal_Entity_Synthetic_Char_Enum_Lit is
      begin
         if Node = null then
            return No_Entity_Synthetic_Char_Enum_Lit;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synthetic_Char_Enum_Lit) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Synthetic_Defining_Name
        (Node : Bare_Synthetic_Defining_Name; Info : Internal_Entity_Info)
         return Internal_Entity_Synthetic_Defining_Name is
      begin
         if Node = null then
            return No_Entity_Synthetic_Defining_Name;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synthetic_Defining_Name) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Synthetic_Formal_Param_Decl
        (Node : Bare_Synthetic_Formal_Param_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Synthetic_Formal_Param_Decl is
      begin
         if Node = null then
            return No_Entity_Synthetic_Formal_Param_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synthetic_Formal_Param_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Synthetic_Identifier
        (Node : Bare_Synthetic_Identifier; Info : Internal_Entity_Info)
         return Internal_Entity_Synthetic_Identifier is
      begin
         if Node = null then
            return No_Entity_Synthetic_Identifier;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synthetic_Identifier) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Synthetic_Renaming_Clause
        (Node : Bare_Synthetic_Renaming_Clause; Info : Internal_Entity_Info)
         return Internal_Entity_Synthetic_Renaming_Clause is
      begin
         if Node = null then
            return No_Entity_Synthetic_Renaming_Clause;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synthetic_Renaming_Clause) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Synthetic_Subp_Decl
        (Node : Bare_Synthetic_Subp_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Synthetic_Subp_Decl is
      begin
         if Node = null then
            return No_Entity_Synthetic_Subp_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synthetic_Subp_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Synthetic_Type_Expr
        (Node : Bare_Synthetic_Type_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Synthetic_Type_Expr is
      begin
         if Node = null then
            return No_Entity_Synthetic_Type_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synthetic_Type_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Synthetic_Unary_Spec
        (Node : Bare_Synthetic_Unary_Spec; Info : Internal_Entity_Info)
         return Internal_Entity_Synthetic_Unary_Spec is
      begin
         if Node = null then
            return No_Entity_Synthetic_Unary_Spec;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synthetic_Unary_Spec) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Tagged_Node
        (Node : Bare_Tagged_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Tagged_Node is
      begin
         if Node = null then
            return No_Entity_Tagged_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Tagged_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Tagged_Absent
        (Node : Bare_Tagged_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_Tagged_Absent is
      begin
         if Node = null then
            return No_Entity_Tagged_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Tagged_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Tagged_Present
        (Node : Bare_Tagged_Present; Info : Internal_Entity_Info)
         return Internal_Entity_Tagged_Present is
      begin
         if Node = null then
            return No_Entity_Tagged_Present;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Tagged_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Target_Name
        (Node : Bare_Target_Name; Info : Internal_Entity_Info)
         return Internal_Entity_Target_Name is
      begin
         if Node = null then
            return No_Entity_Target_Name;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Target_Name) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Task_Body
        (Node : Bare_Task_Body; Info : Internal_Entity_Info)
         return Internal_Entity_Task_Body is
      begin
         if Node = null then
            return No_Entity_Task_Body;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Task_Body) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Task_Body_Stub
        (Node : Bare_Task_Body_Stub; Info : Internal_Entity_Info)
         return Internal_Entity_Task_Body_Stub is
      begin
         if Node = null then
            return No_Entity_Task_Body_Stub;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Task_Body_Stub) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Task_Def
        (Node : Bare_Task_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Task_Def is
      begin
         if Node = null then
            return No_Entity_Task_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Task_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Terminate_Alternative
        (Node : Bare_Terminate_Alternative; Info : Internal_Entity_Info)
         return Internal_Entity_Terminate_Alternative is
      begin
         if Node = null then
            return No_Entity_Terminate_Alternative;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Terminate_Alternative) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Type_Access_Def
        (Node : Bare_Type_Access_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Type_Access_Def is
      begin
         if Node = null then
            return No_Entity_Type_Access_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Type_Access_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Type_Attributes_Repository
        (Node : Bare_Type_Attributes_Repository; Info : Internal_Entity_Info)
         return Internal_Entity_Type_Attributes_Repository is
      begin
         if Node = null then
            return No_Entity_Type_Attributes_Repository;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Type_Attributes_Repository) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Un_Op
        (Node : Bare_Un_Op; Info : Internal_Entity_Info)
         return Internal_Entity_Un_Op is
      begin
         if Node = null then
            return No_Entity_Un_Op;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Un_Op) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Unconstrained_Array_Index
        (Node : Bare_Unconstrained_Array_Index; Info : Internal_Entity_Info)
         return Internal_Entity_Unconstrained_Array_Index is
      begin
         if Node = null then
            return No_Entity_Unconstrained_Array_Index;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Unconstrained_Array_Index) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Unconstrained_Array_Index_List
        (Node : Bare_Unconstrained_Array_Index_List; Info : Internal_Entity_Info)
         return Internal_Entity_Unconstrained_Array_Index_List is
      begin
         if Node = null then
            return No_Entity_Unconstrained_Array_Index_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Unconstrained_Array_Index_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Unconstrained_Array_Indices
        (Node : Bare_Unconstrained_Array_Indices; Info : Internal_Entity_Info)
         return Internal_Entity_Unconstrained_Array_Indices is
      begin
         if Node = null then
            return No_Entity_Unconstrained_Array_Indices;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Unconstrained_Array_Indices) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Unknown_Discriminant_Part
        (Node : Bare_Unknown_Discriminant_Part; Info : Internal_Entity_Info)
         return Internal_Entity_Unknown_Discriminant_Part is
      begin
         if Node = null then
            return No_Entity_Unknown_Discriminant_Part;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Unknown_Discriminant_Part) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Until_Node
        (Node : Bare_Until_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Until_Node is
      begin
         if Node = null then
            return No_Entity_Until_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Until_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Until_Absent
        (Node : Bare_Until_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_Until_Absent is
      begin
         if Node = null then
            return No_Entity_Until_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Until_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Until_Present
        (Node : Bare_Until_Present; Info : Internal_Entity_Info)
         return Internal_Entity_Until_Present is
      begin
         if Node = null then
            return No_Entity_Until_Present;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Until_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Update_Attribute_Ref
        (Node : Bare_Update_Attribute_Ref; Info : Internal_Entity_Info)
         return Internal_Entity_Update_Attribute_Ref is
      begin
         if Node = null then
            return No_Entity_Update_Attribute_Ref;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Update_Attribute_Ref) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Use_Clause
        (Node : Bare_Use_Clause; Info : Internal_Entity_Info)
         return Internal_Entity_Use_Clause is
      begin
         if Node = null then
            return No_Entity_Use_Clause;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Use_Clause) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Use_Package_Clause
        (Node : Bare_Use_Package_Clause; Info : Internal_Entity_Info)
         return Internal_Entity_Use_Package_Clause is
      begin
         if Node = null then
            return No_Entity_Use_Package_Clause;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Use_Package_Clause) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Use_Type_Clause
        (Node : Bare_Use_Type_Clause; Info : Internal_Entity_Info)
         return Internal_Entity_Use_Type_Clause is
      begin
         if Node = null then
            return No_Entity_Use_Type_Clause;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Use_Type_Clause) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Value_Sequence
        (Node : Bare_Value_Sequence; Info : Internal_Entity_Info)
         return Internal_Entity_Value_Sequence is
      begin
         if Node = null then
            return No_Entity_Value_Sequence;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Value_Sequence) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Variant
        (Node : Bare_Variant; Info : Internal_Entity_Info)
         return Internal_Entity_Variant is
      begin
         if Node = null then
            return No_Entity_Variant;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Variant) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Variant_List
        (Node : Bare_Variant_List; Info : Internal_Entity_Info)
         return Internal_Entity_Variant_List is
      begin
         if Node = null then
            return No_Entity_Variant_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Variant_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Variant_Part
        (Node : Bare_Variant_Part; Info : Internal_Entity_Info)
         return Internal_Entity_Variant_Part is
      begin
         if Node = null then
            return No_Entity_Variant_Part;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Variant_Part) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_While_Loop_Spec
        (Node : Bare_While_Loop_Spec; Info : Internal_Entity_Info)
         return Internal_Entity_While_Loop_Spec is
      begin
         if Node = null then
            return No_Entity_While_Loop_Spec;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_While_Loop_Spec) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_While_Loop_Stmt
        (Node : Bare_While_Loop_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_While_Loop_Stmt is
      begin
         if Node = null then
            return No_Entity_While_Loop_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_While_Loop_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_With_Clause
        (Node : Bare_With_Clause; Info : Internal_Entity_Info)
         return Internal_Entity_With_Clause is
      begin
         if Node = null then
            return No_Entity_With_Clause;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_With_Clause) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_With_Private
        (Node : Bare_With_Private; Info : Internal_Entity_Info)
         return Internal_Entity_With_Private is
      begin
         if Node = null then
            return No_Entity_With_Private;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_With_Private) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_With_Private_Absent
        (Node : Bare_With_Private_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_With_Private_Absent is
      begin
         if Node = null then
            return No_Entity_With_Private_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_With_Private_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_With_Private_Present
        (Node : Bare_With_Private_Present; Info : Internal_Entity_Info)
         return Internal_Entity_With_Private_Present is
      begin
         if Node = null then
            return No_Entity_With_Private_Present;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_With_Private_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      -------------
      -- Inc_Ref --
      -------------
      procedure Inc_Ref (R : Internal_Env_Assoc) is
      begin
               Inc_Ref (R.Dest_Env);
      end Inc_Ref;
      -------------
      -- Dec_Ref --
      -------------
      procedure Dec_Ref (R : in out Internal_Env_Assoc) is
      begin
               Dec_Ref (R.Dest_Env);
      end Dec_Ref;
      ----------------
      -- Equivalent --
      ----------------
      function Equivalent (L, R : Internal_Env_Assoc) return Boolean is
      begin
         return L.Key = R.Key and then L.Value = R.Value and then Equivalent (L.Dest_Env, R.Dest_Env) and then L.Metadata = R.Metadata;
      end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Env_Assoc) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Key => " & Trace_Image (R.Key)
                        & ", "
                     & "Value => " & Trace_Image (R.Value)
                        & ", "
                     & "Dest_Env => " & Trace_Image (R.Dest_Env)
                        & ", "
                     & "Metadata => " & Trace_Image (R.Metadata)
               & ")");
      end Trace_Image;
      -------------
      -- Inc_Ref --
      -------------
      procedure Inc_Ref (R : Internal_Eval_Discrete_Range) is
      begin
               Inc_Ref (R.Low_Bound);
               Inc_Ref (R.High_Bound);
      end Inc_Ref;
      -------------
      -- Dec_Ref --
      -------------
      procedure Dec_Ref (R : in out Internal_Eval_Discrete_Range) is
      begin
               Dec_Ref (R.Low_Bound);
               Dec_Ref (R.High_Bound);
      end Dec_Ref;
      ----------------
      -- Equivalent --
      ----------------
      function Equivalent (L, R : Internal_Eval_Discrete_Range) return Boolean is
      begin
         return Equivalent (L.Low_Bound, R.Low_Bound) and then Equivalent (L.High_Bound, R.High_Bound);
      end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Eval_Discrete_Range) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Low_Bound => " & Trace_Image (R.Low_Bound)
                        & ", "
                     & "High_Bound => " & Trace_Image (R.High_Bound)
               & ")");
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Expected_Type_For_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Expected_Type => " & Trace_Image (R.Expected_Type)
                        & ", "
                     & "Expr => " & Trace_Image (R.Expr)
               & ")");
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Inner_Env_Assoc) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Key => " & Trace_Image (R.Key)
                        & ", "
                     & "Value => " & Trace_Image (R.Value)
                        & ", "
                     & "Metadata => " & Trace_Image (R.Metadata)
               & ")");
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Logic_Val_Result) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Success => " & Trace_Image (R.Success)
                        & ", "
                     & "Value => " & Trace_Image (R.Value)
               & ")");
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Multidim_Aggregate_Info) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Agg => " & Trace_Image (R.Agg)
                        & ", "
                     & "Typ => " & Trace_Image (R.Typ)
                        & ", "
                     & "Rank => " & Trace_Image (R.Rank)
               & ")");
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Param_Actual) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Param => " & Trace_Image (R.Param)
                        & ", "
                     & "Actual => " & Trace_Image (R.Actual)
               & ")");
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Single_Actual) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Name => " & Trace_Image (R.Name)
                        & ", "
                     & "Assoc => " & Trace_Image (R.Assoc)
               & ")");
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Param_Match) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Has_Matched => " & Trace_Image (R.Has_Matched)
                        & ", "
                     & "Actual => " & Trace_Image (R.Actual)
                        & ", "
                     & "Formal => " & Trace_Image (R.Formal)
               & ")");
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Ref_Result) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Ref => " & Trace_Image (R.Ref)
                        & ", "
                     & "Kind => " & Trace_Image (R.Kind)
               & ")");
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Refd_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Decl => " & Trace_Image (R.Decl)
                        & ", "
                     & "Kind => " & Trace_Image (R.Kind)
               & ")");
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Refd_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Def_Name => " & Trace_Image (R.Def_Name)
                        & ", "
                     & "Kind => " & Trace_Image (R.Kind)
               & ")");
      end Trace_Image;
      -------------
      -- Inc_Ref --
      -------------
      procedure Inc_Ref (R : Internal_Shape) is
      begin
               Inc_Ref (R.Components);
               Inc_Ref (R.Discriminants_Values);
      end Inc_Ref;
      -------------
      -- Dec_Ref --
      -------------
      procedure Dec_Ref (R : in out Internal_Shape) is
      begin
               Dec_Ref (R.Components);
               Dec_Ref (R.Discriminants_Values);
      end Dec_Ref;
      ----------------
      -- Equivalent --
      ----------------
      function Equivalent (L, R : Internal_Shape) return Boolean is
      begin
         return Equivalent (L.Components, R.Components) and then Equivalent (L.Discriminants_Values, R.Discriminants_Values);
      end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Shape) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Components => " & Trace_Image (R.Components)
                        & ", "
                     & "Discriminants_Values => " & Trace_Image (R.Discriminants_Values)
               & ")");
      end Trace_Image;
      -------------
      -- Inc_Ref --
      -------------
      procedure Inc_Ref (R : Internal_Substitution) is
      begin
               Inc_Ref (R.To_Value);
      end Inc_Ref;
      -------------
      -- Dec_Ref --
      -------------
      procedure Dec_Ref (R : in out Internal_Substitution) is
      begin
               Dec_Ref (R.To_Value);
      end Dec_Ref;
      ----------------
      -- Equivalent --
      ----------------
      function Equivalent (L, R : Internal_Substitution) return Boolean is
      begin
         return L.From_Decl = R.From_Decl and then Equivalent (L.To_Value, R.To_Value) and then L.Value_Type = R.Value_Type;
      end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Substitution) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "From_Decl => " & Trace_Image (R.From_Decl)
                        & ", "
                     & "To_Value => " & Trace_Image (R.To_Value)
                        & ", "
                     & "Value_Type => " & Trace_Image (R.Value_Type)
               & ")");
      end Trace_Image;
   type Ada_Node_P_Is_Not_Null_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         null;
   end record;
      overriding function Call
        (Self : Ada_Node_P_Is_Not_Null_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Ada_Node_P_Is_Not_Null_0_Predicate) return String;
      overriding procedure Destroy (Self : in out Ada_Node_P_Is_Not_Null_0_Predicate);
   function Create_Ada_Node_P_Is_Not_Null_0_Predicate
      return Ada_Node_P_Is_Not_Null_0_Predicate is
   begin
      return Ada_Node_P_Is_Not_Null_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Ada_Node_P_Is_Not_Null_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
         pragma Unreferenced (Self);
      Node : constant Bare_Ada_Node := Entity.Node;
   begin
      return Ada_Node_P_Is_Not_Null (Node, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Ada_Node_P_Is_Not_Null_0_Predicate) return String is
   begin
      return "AdaNode.is_not_null";
   end Image;
      -------------
      -- Destroy --
      -------------
      overriding procedure Destroy (Self : in out Ada_Node_P_Is_Not_Null_0_Predicate) is
      begin
         null;
      end Destroy;
   type Basic_Decl_P_Subp_Decl_Match_Signature_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Internal_Entity_Basic_Decl;
   end record;
      overriding function Call
        (Self : Basic_Decl_P_Subp_Decl_Match_Signature_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Basic_Decl_P_Subp_Decl_Match_Signature_0_Predicate) return String;
      overriding procedure Destroy (Self : in out Basic_Decl_P_Subp_Decl_Match_Signature_0_Predicate);
   function Create_Basic_Decl_P_Subp_Decl_Match_Signature_0_Predicate
   (
         Field_0 : Internal_Entity_Basic_Decl
   )
      return Basic_Decl_P_Subp_Decl_Match_Signature_0_Predicate is
   begin
      return Basic_Decl_P_Subp_Decl_Match_Signature_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Basic_Decl_P_Subp_Decl_Match_Signature_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
      Node : constant Bare_Basic_Decl := Entity.Node;
   begin
      return Basic_Decl_P_Subp_Decl_Match_Signature (Node, Self.Field_0, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Basic_Decl_P_Subp_Decl_Match_Signature_0_Predicate) return String is
   begin
      return "BasicDecl.subp_decl_match_signature";
   end Image;
      -------------
      -- Destroy --
      -------------
      overriding procedure Destroy (Self : in out Basic_Decl_P_Subp_Decl_Match_Signature_0_Predicate) is
      begin
         null;
      end Destroy;
   type Basic_Decl_P_Is_Subprogram_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         null;
   end record;
      overriding function Call
        (Self : Basic_Decl_P_Is_Subprogram_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Basic_Decl_P_Is_Subprogram_0_Predicate) return String;
      overriding procedure Destroy (Self : in out Basic_Decl_P_Is_Subprogram_0_Predicate);
   function Create_Basic_Decl_P_Is_Subprogram_0_Predicate
      return Basic_Decl_P_Is_Subprogram_0_Predicate is
   begin
      return Basic_Decl_P_Is_Subprogram_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Basic_Decl_P_Is_Subprogram_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
         pragma Unreferenced (Self);
      Node : constant Bare_Basic_Decl := Entity.Node;
   begin
      return Basic_Decl_P_Is_Subprogram (Node);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Basic_Decl_P_Is_Subprogram_0_Predicate) return String is
   begin
      return "BasicDecl.is_subprogram";
   end Image;
      -------------
      -- Destroy --
      -------------
      overriding procedure Destroy (Self : in out Basic_Decl_P_Is_Subprogram_0_Predicate) is
      begin
         null;
      end Destroy;
   type Basic_Decl_P_Is_Stream_Subprogram_For_Type_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Internal_Entity_Base_Type_Decl;
         Field_1 : Boolean;
   end record;
      overriding function Call
        (Self : Basic_Decl_P_Is_Stream_Subprogram_For_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Basic_Decl_P_Is_Stream_Subprogram_For_Type_0_Predicate) return String;
      overriding procedure Destroy (Self : in out Basic_Decl_P_Is_Stream_Subprogram_For_Type_0_Predicate);
   function Create_Basic_Decl_P_Is_Stream_Subprogram_For_Type_0_Predicate
   (
         Field_0 : Internal_Entity_Base_Type_Decl;
         Field_1 : Boolean
   )
      return Basic_Decl_P_Is_Stream_Subprogram_For_Type_0_Predicate is
   begin
      return Basic_Decl_P_Is_Stream_Subprogram_For_Type_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0, Field_1 => Field_1);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Basic_Decl_P_Is_Stream_Subprogram_For_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
      Node : constant Bare_Basic_Decl := Entity.Node;
   begin
      return Basic_Decl_P_Is_Stream_Subprogram_For_Type (Node, Self.Field_0, Self.Field_1, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Basic_Decl_P_Is_Stream_Subprogram_For_Type_0_Predicate) return String is
   begin
      return "BasicDecl.is_stream_subprogram_for_type";
   end Image;
      -------------
      -- Destroy --
      -------------
      overriding procedure Destroy (Self : in out Basic_Decl_P_Is_Stream_Subprogram_For_Type_0_Predicate) is
      begin
         null;
      end Destroy;
   type Basic_Decl_P_Is_Put_Image_Subprogram_For_Type_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Internal_Entity_Base_Type_Decl;
   end record;
      overriding function Call
        (Self : Basic_Decl_P_Is_Put_Image_Subprogram_For_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Basic_Decl_P_Is_Put_Image_Subprogram_For_Type_0_Predicate) return String;
      overriding procedure Destroy (Self : in out Basic_Decl_P_Is_Put_Image_Subprogram_For_Type_0_Predicate);
   function Create_Basic_Decl_P_Is_Put_Image_Subprogram_For_Type_0_Predicate
   (
         Field_0 : Internal_Entity_Base_Type_Decl
   )
      return Basic_Decl_P_Is_Put_Image_Subprogram_For_Type_0_Predicate is
   begin
      return Basic_Decl_P_Is_Put_Image_Subprogram_For_Type_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Basic_Decl_P_Is_Put_Image_Subprogram_For_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
      Node : constant Bare_Basic_Decl := Entity.Node;
   begin
      return Basic_Decl_P_Is_Put_Image_Subprogram_For_Type (Node, Self.Field_0, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Basic_Decl_P_Is_Put_Image_Subprogram_For_Type_0_Predicate) return String is
   begin
      return "BasicDecl.is_put_image_subprogram_for_type";
   end Image;
      -------------
      -- Destroy --
      -------------
      overriding procedure Destroy (Self : in out Basic_Decl_P_Is_Put_Image_Subprogram_For_Type_0_Predicate) is
      begin
         null;
      end Destroy;
   type Base_Type_Decl_P_Is_Array_Or_Rec_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Array_Or_Rec_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Is_Array_Or_Rec_0_Predicate) return String;
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Is_Array_Or_Rec_0_Predicate);
   function Create_Base_Type_Decl_P_Is_Array_Or_Rec_0_Predicate
   (
         Field_0 : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Is_Array_Or_Rec_0_Predicate is
   begin
      return Base_Type_Decl_P_Is_Array_Or_Rec_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Array_Or_Rec_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
      Node : constant Bare_Base_Type_Decl := Entity.Node;
   begin
      return Base_Type_Decl_P_Is_Array_Or_Rec (Node, Self.Field_0, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Is_Array_Or_Rec_0_Predicate) return String is
   begin
      return "BaseTypeDecl.is_array_or_rec";
   end Image;
      -------------
      -- Destroy --
      -------------
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Is_Array_Or_Rec_0_Predicate) is
      begin
         null;
      end Destroy;
   type Base_Type_Decl_P_Is_Task_Type_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         null;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Task_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Is_Task_Type_0_Predicate) return String;
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Is_Task_Type_0_Predicate);
   function Create_Base_Type_Decl_P_Is_Task_Type_0_Predicate
      return Base_Type_Decl_P_Is_Task_Type_0_Predicate is
   begin
      return Base_Type_Decl_P_Is_Task_Type_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Task_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
         pragma Unreferenced (Self);
      Node : constant Bare_Base_Type_Decl := Entity.Node;
   begin
      return Dispatcher_Base_Type_Decl_P_Is_Task_Type (Node);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Is_Task_Type_0_Predicate) return String is
   begin
      return "BaseTypeDecl.is_task_type";
   end Image;
      -------------
      -- Destroy --
      -------------
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Is_Task_Type_0_Predicate) is
      begin
         null;
      end Destroy;
   type Base_Type_Decl_P_One_Non_Universal_0_Predicate is
   new Solver_Ifc.N_Predicate_Type
   with record
         null;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_One_Non_Universal_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_One_Non_Universal_0_Predicate) return String;
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_One_Non_Universal_0_Predicate);
   function Create_Base_Type_Decl_P_One_Non_Universal_0_Predicate
      return Base_Type_Decl_P_One_Non_Universal_0_Predicate is
   begin
      return Base_Type_Decl_P_One_Non_Universal_0_Predicate'(N => 2, Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_One_Non_Universal_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   is
         pragma Unreferenced (Self);
         Entity : Internal_Entity := Entities (1);
      Node : constant Bare_Base_Type_Decl := Entity.Node;
   begin
      return Base_Type_Decl_P_One_Non_Universal (Node, (Node => Entities (2).Node, Info => Entities (2).Info), E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_One_Non_Universal_0_Predicate) return String is
   begin
      return "BaseTypeDecl.one_non_universal";
   end Image;
      -------------
      -- Destroy --
      -------------
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_One_Non_Universal_0_Predicate) is
      begin
         null;
      end Destroy;
   type Base_Type_Decl_P_Is_Non_Null_Char_Type_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Non_Null_Char_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Is_Non_Null_Char_Type_0_Predicate) return String;
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Is_Non_Null_Char_Type_0_Predicate);
   function Create_Base_Type_Decl_P_Is_Non_Null_Char_Type_0_Predicate
   (
         Field_0 : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Is_Non_Null_Char_Type_0_Predicate is
   begin
      return Base_Type_Decl_P_Is_Non_Null_Char_Type_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Non_Null_Char_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
      Node : constant Bare_Base_Type_Decl := Entity.Node;
   begin
      return Base_Type_Decl_P_Is_Non_Null_Char_Type (Node, Self.Field_0, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Is_Non_Null_Char_Type_0_Predicate) return String is
   begin
      return "BaseTypeDecl.is_non_null_char_type";
   end Image;
      -------------
      -- Destroy --
      -------------
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Is_Non_Null_Char_Type_0_Predicate) is
      begin
         null;
      end Destroy;
   type Base_Type_Decl_P_Is_Universal_Type_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         null;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Universal_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Is_Universal_Type_0_Predicate) return String;
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Is_Universal_Type_0_Predicate);
   function Create_Base_Type_Decl_P_Is_Universal_Type_0_Predicate
      return Base_Type_Decl_P_Is_Universal_Type_0_Predicate is
   begin
      return Base_Type_Decl_P_Is_Universal_Type_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Universal_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
         pragma Unreferenced (Self);
      Node : constant Bare_Base_Type_Decl := Entity.Node;
   begin
      return Base_Type_Decl_P_Is_Universal_Type (Node, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Is_Universal_Type_0_Predicate) return String is
   begin
      return "BaseTypeDecl.is_universal_type";
   end Image;
      -------------
      -- Destroy --
      -------------
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Is_Universal_Type_0_Predicate) is
      begin
         null;
      end Destroy;
   type Base_Type_Decl_P_Is_Not_Universal_Type_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         null;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Not_Universal_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Is_Not_Universal_Type_0_Predicate) return String;
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Is_Not_Universal_Type_0_Predicate);
   function Create_Base_Type_Decl_P_Is_Not_Universal_Type_0_Predicate
      return Base_Type_Decl_P_Is_Not_Universal_Type_0_Predicate is
   begin
      return Base_Type_Decl_P_Is_Not_Universal_Type_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Not_Universal_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
         pragma Unreferenced (Self);
      Node : constant Bare_Base_Type_Decl := Entity.Node;
   begin
      return Base_Type_Decl_P_Is_Not_Universal_Type (Node, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Is_Not_Universal_Type_0_Predicate) return String is
   begin
      return "BaseTypeDecl.is_not_universal_type";
   end Image;
      -------------
      -- Destroy --
      -------------
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Is_Not_Universal_Type_0_Predicate) is
      begin
         null;
      end Destroy;
   type Base_Type_Decl_P_Is_Access_Type_Predicate_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Access_Type_Predicate_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Is_Access_Type_Predicate_0_Predicate) return String;
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Is_Access_Type_Predicate_0_Predicate);
   function Create_Base_Type_Decl_P_Is_Access_Type_Predicate_0_Predicate
   (
         Field_0 : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Is_Access_Type_Predicate_0_Predicate is
   begin
      return Base_Type_Decl_P_Is_Access_Type_Predicate_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Access_Type_Predicate_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
      Node : constant Bare_Base_Type_Decl := Entity.Node;
   begin
      return Base_Type_Decl_P_Is_Access_Type_Predicate (Node, Self.Field_0, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Is_Access_Type_Predicate_0_Predicate) return String is
   begin
      return "BaseTypeDecl.is_access_type_predicate";
   end Image;
      -------------
      -- Destroy --
      -------------
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Is_Access_Type_Predicate_0_Predicate) is
      begin
         null;
      end Destroy;
   type Base_Type_Decl_P_Is_Iterator_Type_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Iterator_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Is_Iterator_Type_0_Predicate) return String;
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Is_Iterator_Type_0_Predicate);
   function Create_Base_Type_Decl_P_Is_Iterator_Type_0_Predicate
   (
         Field_0 : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Is_Iterator_Type_0_Predicate is
   begin
      return Base_Type_Decl_P_Is_Iterator_Type_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Iterator_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
      Node : constant Bare_Base_Type_Decl := Entity.Node;
   begin
      return Base_Type_Decl_P_Is_Iterator_Type (Node, Self.Field_0, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Is_Iterator_Type_0_Predicate) return String is
   begin
      return "BaseTypeDecl.is_iterator_type";
   end Image;
      -------------
      -- Destroy --
      -------------
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Is_Iterator_Type_0_Predicate) is
      begin
         null;
      end Destroy;
   type Base_Type_Decl_P_Is_Discrete_Type_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Discrete_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Is_Discrete_Type_0_Predicate) return String;
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Is_Discrete_Type_0_Predicate);
   function Create_Base_Type_Decl_P_Is_Discrete_Type_0_Predicate
   (
         Field_0 : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Is_Discrete_Type_0_Predicate is
   begin
      return Base_Type_Decl_P_Is_Discrete_Type_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Discrete_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
      Node : constant Bare_Base_Type_Decl := Entity.Node;
   begin
      return Dispatcher_Base_Type_Decl_P_Is_Discrete_Type (Node, Self.Field_0, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Is_Discrete_Type_0_Predicate) return String is
   begin
      return "BaseTypeDecl.is_discrete_type";
   end Image;
      -------------
      -- Destroy --
      -------------
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Is_Discrete_Type_0_Predicate) is
      begin
         null;
      end Destroy;
   type Base_Type_Decl_P_Is_Not_Root_Int_Type_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         null;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Not_Root_Int_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Is_Not_Root_Int_Type_0_Predicate) return String;
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Is_Not_Root_Int_Type_0_Predicate);
   function Create_Base_Type_Decl_P_Is_Not_Root_Int_Type_0_Predicate
      return Base_Type_Decl_P_Is_Not_Root_Int_Type_0_Predicate is
   begin
      return Base_Type_Decl_P_Is_Not_Root_Int_Type_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Not_Root_Int_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
         pragma Unreferenced (Self);
      Node : constant Bare_Base_Type_Decl := Entity.Node;
   begin
      return Base_Type_Decl_P_Is_Not_Root_Int_Type (Node, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Is_Not_Root_Int_Type_0_Predicate) return String is
   begin
      return "BaseTypeDecl.is_not_root_int_type";
   end Image;
      -------------
      -- Destroy --
      -------------
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Is_Not_Root_Int_Type_0_Predicate) is
      begin
         null;
      end Destroy;
   type Base_Type_Decl_P_Allows_String_Literal_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Allows_String_Literal_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Allows_String_Literal_0_Predicate) return String;
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Allows_String_Literal_0_Predicate);
   function Create_Base_Type_Decl_P_Allows_String_Literal_0_Predicate
   (
         Field_0 : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Allows_String_Literal_0_Predicate is
   begin
      return Base_Type_Decl_P_Allows_String_Literal_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Allows_String_Literal_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
      Node : constant Bare_Base_Type_Decl := Entity.Node;
   begin
      return Base_Type_Decl_P_Allows_String_Literal (Node, Self.Field_0, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Allows_String_Literal_0_Predicate) return String is
   begin
      return "BaseTypeDecl.allows_string_literal";
   end Image;
      -------------
      -- Destroy --
      -------------
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Allows_String_Literal_0_Predicate) is
      begin
         null;
      end Destroy;
   type Base_Type_Decl_P_Is_Subp_Access_Of_0_Predicate is
   new Solver_Ifc.N_Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Subp_Access_Of_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Is_Subp_Access_Of_0_Predicate) return String;
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Is_Subp_Access_Of_0_Predicate);
   function Create_Base_Type_Decl_P_Is_Subp_Access_Of_0_Predicate
   (
         Field_0 : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Is_Subp_Access_Of_0_Predicate is
   begin
      return Base_Type_Decl_P_Is_Subp_Access_Of_0_Predicate'(N => 2, Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Subp_Access_Of_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   is
         Entity : Internal_Entity := Entities (1);
      Node : constant Bare_Base_Type_Decl := Entity.Node;
   begin
      return Base_Type_Decl_P_Is_Subp_Access_Of (Node, (Node => Entities (2).Node, Info => Entities (2).Info), Self.Field_0, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Is_Subp_Access_Of_0_Predicate) return String is
   begin
      return "BaseTypeDecl.is_subp_access_of";
   end Image;
      -------------
      -- Destroy --
      -------------
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Is_Subp_Access_Of_0_Predicate) is
      begin
         null;
      end Destroy;
   type Base_Type_Decl_P_Is_Tagged_Type_With_Deref_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Tagged_Type_With_Deref_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Is_Tagged_Type_With_Deref_0_Predicate) return String;
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Is_Tagged_Type_With_Deref_0_Predicate);
   function Create_Base_Type_Decl_P_Is_Tagged_Type_With_Deref_0_Predicate
   (
         Field_0 : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Is_Tagged_Type_With_Deref_0_Predicate is
   begin
      return Base_Type_Decl_P_Is_Tagged_Type_With_Deref_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Tagged_Type_With_Deref_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
      Node : constant Bare_Base_Type_Decl := Entity.Node;
   begin
      return Base_Type_Decl_P_Is_Tagged_Type_With_Deref (Node, Self.Field_0, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Is_Tagged_Type_With_Deref_0_Predicate) return String is
   begin
      return "BaseTypeDecl.is_tagged_type_with_deref";
   end Image;
      -------------
      -- Destroy --
      -------------
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Is_Tagged_Type_With_Deref_0_Predicate) is
      begin
         null;
      end Destroy;
   type Base_Type_Decl_P_Is_Array_Def_With_Deref_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Array_Def_With_Deref_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Is_Array_Def_With_Deref_0_Predicate) return String;
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Is_Array_Def_With_Deref_0_Predicate);
   function Create_Base_Type_Decl_P_Is_Array_Def_With_Deref_0_Predicate
   (
         Field_0 : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Is_Array_Def_With_Deref_0_Predicate is
   begin
      return Base_Type_Decl_P_Is_Array_Def_With_Deref_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Array_Def_With_Deref_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
      Node : constant Bare_Base_Type_Decl := Entity.Node;
   begin
      return Base_Type_Decl_P_Is_Array_Def_With_Deref (Node, Self.Field_0, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Is_Array_Def_With_Deref_0_Predicate) return String is
   begin
      return "BaseTypeDecl.is_array_def_with_deref";
   end Image;
      -------------
      -- Destroy --
      -------------
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Is_Array_Def_With_Deref_0_Predicate) is
      begin
         null;
      end Destroy;
   type Base_Type_Decl_P_Is_Array_Def_With_Deref_Or_Null_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Array_Def_With_Deref_Or_Null_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Is_Array_Def_With_Deref_Or_Null_0_Predicate) return String;
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Is_Array_Def_With_Deref_Or_Null_0_Predicate);
   function Create_Base_Type_Decl_P_Is_Array_Def_With_Deref_Or_Null_0_Predicate
   (
         Field_0 : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Is_Array_Def_With_Deref_Or_Null_0_Predicate is
   begin
      return Base_Type_Decl_P_Is_Array_Def_With_Deref_Or_Null_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Array_Def_With_Deref_Or_Null_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
      Node : constant Bare_Base_Type_Decl := Entity.Node;
   begin
      return Base_Type_Decl_P_Is_Array_Def_With_Deref_Or_Null (Node, Self.Field_0, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Is_Array_Def_With_Deref_Or_Null_0_Predicate) return String is
   begin
      return "BaseTypeDecl.is_array_def_with_deref_or_null";
   end Image;
      -------------
      -- Destroy --
      -------------
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Is_Array_Def_With_Deref_Or_Null_0_Predicate) is
      begin
         null;
      end Destroy;
   type Base_Type_Decl_P_Matching_Prefix_Type_0_Predicate is
   new Solver_Ifc.N_Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Matching_Prefix_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Matching_Prefix_Type_0_Predicate) return String;
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Matching_Prefix_Type_0_Predicate);
   function Create_Base_Type_Decl_P_Matching_Prefix_Type_0_Predicate
   (
         Field_0 : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Matching_Prefix_Type_0_Predicate is
   begin
      return Base_Type_Decl_P_Matching_Prefix_Type_0_Predicate'(N => 2, Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Matching_Prefix_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   is
         Entity : Internal_Entity := Entities (1);
      Node : constant Bare_Base_Type_Decl := Entity.Node;
   begin
      return Base_Type_Decl_P_Matching_Prefix_Type (Node, (Node => Entities (2).Node, Info => Entities (2).Info), Self.Field_0, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Matching_Prefix_Type_0_Predicate) return String is
   begin
      return "BaseTypeDecl.matching_prefix_type";
   end Image;
      -------------
      -- Destroy --
      -------------
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Matching_Prefix_Type_0_Predicate) is
      begin
         null;
      end Destroy;
   type Base_Type_Decl_P_Matching_Formal_Prim_Type_0_Predicate is
   new Solver_Ifc.N_Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Matching_Formal_Prim_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Matching_Formal_Prim_Type_0_Predicate) return String;
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Matching_Formal_Prim_Type_0_Predicate);
   function Create_Base_Type_Decl_P_Matching_Formal_Prim_Type_0_Predicate
   (
         Field_0 : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Matching_Formal_Prim_Type_0_Predicate is
   begin
      return Base_Type_Decl_P_Matching_Formal_Prim_Type_0_Predicate'(N => 2, Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Matching_Formal_Prim_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   is
         Entity : Internal_Entity := Entities (1);
      Node : constant Bare_Base_Type_Decl := Entity.Node;
   begin
      return Base_Type_Decl_P_Matching_Formal_Prim_Type (Node, (Node => Entities (2).Node, Info => Entities (2).Info), Self.Field_0, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Matching_Formal_Prim_Type_0_Predicate) return String is
   begin
      return "BaseTypeDecl.matching_formal_prim_type";
   end Image;
      -------------
      -- Destroy --
      -------------
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Matching_Formal_Prim_Type_0_Predicate) is
      begin
         null;
      end Destroy;
   type Base_Type_Decl_P_Matching_Formal_Type_0_Predicate is
   new Solver_Ifc.N_Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Matching_Formal_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Matching_Formal_Type_0_Predicate) return String;
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Matching_Formal_Type_0_Predicate);
   function Create_Base_Type_Decl_P_Matching_Formal_Type_0_Predicate
   (
         Field_0 : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Matching_Formal_Type_0_Predicate is
   begin
      return Base_Type_Decl_P_Matching_Formal_Type_0_Predicate'(N => 2, Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Matching_Formal_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   is
         Entity : Internal_Entity := Entities (1);
      Node : constant Bare_Base_Type_Decl := Entity.Node;
   begin
      return Base_Type_Decl_P_Matching_Formal_Type (Node, (Node => Entities (2).Node, Info => Entities (2).Info), Self.Field_0, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Matching_Formal_Type_0_Predicate) return String is
   begin
      return "BaseTypeDecl.matching_formal_type";
   end Image;
      -------------
      -- Destroy --
      -------------
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Matching_Formal_Type_0_Predicate) is
      begin
         null;
      end Destroy;
   type Base_Type_Decl_P_Matching_Assign_Type_0_Predicate is
   new Solver_Ifc.N_Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Matching_Assign_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Matching_Assign_Type_0_Predicate) return String;
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Matching_Assign_Type_0_Predicate);
   function Create_Base_Type_Decl_P_Matching_Assign_Type_0_Predicate
   (
         Field_0 : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Matching_Assign_Type_0_Predicate is
   begin
      return Base_Type_Decl_P_Matching_Assign_Type_0_Predicate'(N => 2, Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Matching_Assign_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   is
         Entity : Internal_Entity := Entities (1);
      Node : constant Bare_Base_Type_Decl := Entity.Node;
   begin
      return Base_Type_Decl_P_Matching_Assign_Type (Node, (Node => Entities (2).Node, Info => Entities (2).Info), Self.Field_0, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Matching_Assign_Type_0_Predicate) return String is
   begin
      return "BaseTypeDecl.matching_assign_type";
   end Image;
      -------------
      -- Destroy --
      -------------
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Matching_Assign_Type_0_Predicate) is
      begin
         null;
      end Destroy;
   type Base_Type_Decl_P_Matching_Type_0_Predicate is
   new Solver_Ifc.N_Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Matching_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Matching_Type_0_Predicate) return String;
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Matching_Type_0_Predicate);
   function Create_Base_Type_Decl_P_Matching_Type_0_Predicate
   (
         Field_0 : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Matching_Type_0_Predicate is
   begin
      return Base_Type_Decl_P_Matching_Type_0_Predicate'(N => 2, Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Matching_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   is
         Entity : Internal_Entity := Entities (1);
      Node : constant Bare_Base_Type_Decl := Entity.Node;
   begin
      return Base_Type_Decl_P_Matching_Type (Node, (Node => Entities (2).Node, Info => Entities (2).Info), Self.Field_0, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Matching_Type_0_Predicate) return String is
   begin
      return "BaseTypeDecl.matching_type";
   end Image;
      -------------
      -- Destroy --
      -------------
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Matching_Type_0_Predicate) is
      begin
         null;
      end Destroy;
   type Base_Type_Decl_P_Matching_Allocator_Type_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Internal_Entity_Base_Type_Decl;
         Field_1 : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Matching_Allocator_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Matching_Allocator_Type_0_Predicate) return String;
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Matching_Allocator_Type_0_Predicate);
   function Create_Base_Type_Decl_P_Matching_Allocator_Type_0_Predicate
   (
         Field_0 : Internal_Entity_Base_Type_Decl;
         Field_1 : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Matching_Allocator_Type_0_Predicate is
   begin
      return Base_Type_Decl_P_Matching_Allocator_Type_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0, Field_1 => Field_1);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Matching_Allocator_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
      Node : constant Bare_Base_Type_Decl := Entity.Node;
   begin
      return Base_Type_Decl_P_Matching_Allocator_Type (Node, Self.Field_0, Self.Field_1, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Matching_Allocator_Type_0_Predicate) return String is
   begin
      return "BaseTypeDecl.matching_allocator_type";
   end Image;
      -------------
      -- Destroy --
      -------------
      overriding procedure Destroy (Self : in out Base_Type_Decl_P_Matching_Allocator_Type_0_Predicate) is
      begin
         null;
      end Destroy;
   type Logic_Functor_656 is new Solver_Ifc.Converter_Type with
      record
            Origin : Bare_Ada_Node;
      end record;
   overriding function Convert (Self : Logic_Functor_656; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Logic_Functor_656) return String;
   overriding function Convert (Self : Logic_Functor_656; From : Internal_Entity) return Internal_Entity is
      Ret : Internal_Entity_Base_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in Ada_Base_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;
      Ret := Base_Type_Decl_P_Accessed_Type_No_Call (Node => From.Node, Origin => Self.Origin, E_Info => From.Info);
      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Logic_Functor_656) return String is
   begin
      return ("BaseTypeDecl.accessed_type_no_call");
   end Image;
   type Logic_Functor_604 is new Solver_Ifc.Converter_Type with
      null record;
   overriding function Convert (Self : Logic_Functor_604; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Logic_Functor_604) return String;
   overriding function Convert (Self : Logic_Functor_604; From : Internal_Entity) return Internal_Entity is
         pragma Unreferenced (Self);
      Ret : Internal_Entity_Base_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in Ada_Base_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;
      Ret := Base_Type_Decl_P_Anonymous_Access_Type_Or_Null (Node => From.Node, E_Info => From.Info);
      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Logic_Functor_604) return String is
   begin
      return ("BaseTypeDecl.anonymous_access_type_or_null");
   end Image;
   type Logic_Functor_633 is new Solver_Ifc.Combiner_Type with
      record
            Origin : Bare_Ada_Node;
      end record;
   overriding function Combine (Self : Logic_Functor_633; Vals : Entity_Vars.Value_Array) return Internal_Entity with Inline;
   overriding function Image (Self : Logic_Functor_633) return String;
   overriding function Combine (Self : Logic_Functor_633; Vals : Entity_Vars.Value_Array) return Internal_Entity is
         From : constant Internal_Entity := Vals (1);
      Ret : Internal_Entity_Base_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in Ada_Base_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for comb_prop");
            end if;
            if Vals (2).Node /= null
               and then Vals (2).Node.Kind not in Ada_Base_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for comb_prop");
            end if;
      Ret := Base_Type_Decl_P_Array_Concat_Result_Type (Node => From.Node, Other => (Node => Vals (2).Node,  Info => Vals (2).Info), Origin => Self.Origin, E_Info => From.Info);
      return (Node => Ret.Node, Info => Ret.Info);
   end Combine;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Logic_Functor_633) return String is
   begin
      return ("BaseTypeDecl.array_concat_result_type");
   end Image;
   type Logic_Functor_602 is new Solver_Ifc.Converter_Type with
      record
            Origin : Bare_Ada_Node;
      end record;
   overriding function Convert (Self : Logic_Functor_602; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Logic_Functor_602) return String;
   overriding function Convert (Self : Logic_Functor_602; From : Internal_Entity) return Internal_Entity is
      Ret : Internal_Entity_Base_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in Ada_Base_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;
      Ret := Base_Type_Decl_P_Base_Subtype (Node => From.Node, Origin => Self.Origin, E_Info => From.Info);
      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Logic_Functor_602) return String is
   begin
      return ("BaseTypeDecl.base_subtype");
   end Image;
   type Logic_Functor_672 is new Solver_Ifc.Converter_Type with
      record
            Origin : Bare_Ada_Node;
      end record;
   overriding function Convert (Self : Logic_Functor_672; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Logic_Functor_672) return String;
   overriding function Convert (Self : Logic_Functor_672; From : Internal_Entity) return Internal_Entity is
      Ret : Internal_Entity_Base_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in Ada_Base_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;
      Ret := Base_Type_Decl_P_Comp_Type (Node => From.Node, Origin => Self.Origin, E_Info => From.Info);
      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Logic_Functor_672) return String is
   begin
      return ("BaseTypeDecl.comp_type");
   end Image;
   type Logic_Functor_697 is new Solver_Ifc.Converter_Type with
      record
            Origin : Bare_Ada_Node;
      end record;
   overriding function Convert (Self : Logic_Functor_697; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Logic_Functor_697) return String;
   overriding function Convert (Self : Logic_Functor_697; From : Internal_Entity) return Internal_Entity is
      Ret : Internal_Entity_Basic_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in Ada_Base_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;
      Ret := Base_Type_Decl_P_Corresponding_Char_Literal (Node => From.Node, Origin => Self.Origin, E_Info => From.Info);
      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Logic_Functor_697) return String is
   begin
      return ("BaseTypeDecl.corresponding_char_literal");
   end Image;
   type Logic_Functor_630 is new Solver_Ifc.Converter_Type with
      record
            Origin : Bare_Ada_Node;
      end record;
   overriding function Convert (Self : Logic_Functor_630; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Logic_Functor_630) return String;
   overriding function Convert (Self : Logic_Functor_630; From : Internal_Entity) return Internal_Entity is
      Ret : Internal_Entity_Base_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in Ada_Base_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;
      Ret := Base_Type_Decl_P_Derefed_Base_Subtype (Node => From.Node, Origin => Self.Origin, E_Info => From.Info);
      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Logic_Functor_630) return String is
   begin
      return ("BaseTypeDecl.derefed_base_subtype");
   end Image;
   type Logic_Functor_629 is new Solver_Ifc.Converter_Type with
      record
            Origin : Bare_Ada_Node;
      end record;
   overriding function Convert (Self : Logic_Functor_629; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Logic_Functor_629) return String;
   overriding function Convert (Self : Logic_Functor_629; From : Internal_Entity) return Internal_Entity is
      Ret : Internal_Entity_Base_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in Ada_Base_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;
      Ret := Base_Type_Decl_P_Derefed_Type (Node => From.Node, Origin => Self.Origin, E_Info => From.Info);
      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Logic_Functor_629) return String is
   begin
      return ("BaseTypeDecl.derefed_type");
   end Image;
   type Logic_Functor_634 is new Solver_Ifc.Combiner_Type with
      record
            Origin : Bare_Ada_Node;
      end record;
   overriding function Combine (Self : Logic_Functor_634; Vals : Entity_Vars.Value_Array) return Internal_Entity with Inline;
   overriding function Image (Self : Logic_Functor_634) return String;
   overriding function Combine (Self : Logic_Functor_634; Vals : Entity_Vars.Value_Array) return Internal_Entity is
         From : constant Internal_Entity := Vals (1);
      Ret : Internal_Entity_Base_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in Ada_Base_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for comb_prop");
            end if;
            if Vals (2).Node /= null
               and then Vals (2).Node.Kind not in Ada_Base_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for comb_prop");
            end if;
      Ret := Base_Type_Decl_P_Expected_Array_Concat_Operand_Type (Node => From.Node, Operand_Type => (Node => Vals (2).Node,  Info => Vals (2).Info), Origin => Self.Origin, E_Info => From.Info);
      return (Node => Ret.Node, Info => Ret.Info);
   end Combine;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Logic_Functor_634) return String is
   begin
      return ("BaseTypeDecl.expected_array_concat_operand_type");
   end Image;
   type Logic_Functor_632 is new Solver_Ifc.Combiner_Type with
      record
            Origin : Bare_Ada_Node;
      end record;
   overriding function Combine (Self : Logic_Functor_632; Vals : Entity_Vars.Value_Array) return Internal_Entity with Inline;
   overriding function Image (Self : Logic_Functor_632) return String;
   overriding function Combine (Self : Logic_Functor_632; Vals : Entity_Vars.Value_Array) return Internal_Entity is
         From : constant Internal_Entity := Vals (1);
      Ret : Internal_Entity_Base_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in Ada_Base_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for comb_prop");
            end if;
            if Vals (2).Node /= null
               and then Vals (2).Node.Kind not in Ada_Base_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for comb_prop");
            end if;
      Ret := Base_Type_Decl_P_Non_Universal_Base_Subtype (Node => From.Node, Second => (Node => Vals (2).Node,  Info => Vals (2).Info), Origin => Self.Origin, E_Info => From.Info);
      return (Node => Ret.Node, Info => Ret.Info);
   end Combine;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Logic_Functor_632) return String is
   begin
      return ("BaseTypeDecl.non_universal_base_subtype");
   end Image;
   type Logic_Functor_639 is new Solver_Ifc.Converter_Type with
      null record;
   overriding function Convert (Self : Logic_Functor_639; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Logic_Functor_639) return String;
   overriding function Convert (Self : Logic_Functor_639; From : Internal_Entity) return Internal_Entity is
         pragma Unreferenced (Self);
      Ret : Internal_Entity_Discrete_Base_Subtype_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in Ada_Base_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;
      Ret := Base_Type_Decl_P_Scalar_Base_Type (Node => From.Node, E_Info => From.Info);
      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Logic_Functor_639) return String is
   begin
      return ("BaseTypeDecl.scalar_base_type");
   end Image;
   type Logic_Functor_208 is new Solver_Ifc.Converter_Type with
      record
            Origin : Bare_Ada_Node;
      end record;
   overriding function Convert (Self : Logic_Functor_208; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Logic_Functor_208) return String;
   overriding function Convert (Self : Logic_Functor_208; From : Internal_Entity) return Internal_Entity is
      Ret : Internal_Entity_Base_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in Ada_Basic_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;
      Ret := Basic_Decl_P_Identity_Type (Node => From.Node, Origin => Self.Origin, E_Info => From.Info);
      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Logic_Functor_208) return String is
   begin
      return ("BasicDecl.identity_type");
   end Image;
   type Logic_Functor_213 is new Solver_Ifc.Converter_Type with
      null record;
   overriding function Convert (Self : Logic_Functor_213; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Logic_Functor_213) return String;
   overriding function Convert (Self : Logic_Functor_213; From : Internal_Entity) return Internal_Entity is
         pragma Unreferenced (Self);
      Ret : Internal_Entity_Base_Subp_Spec;
   begin
            if From.Node /= null
               and then From.Node.Kind not in Ada_Basic_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;
      Ret := Basic_Decl_P_Subp_Spec_Or_Null (Node => From.Node, E_Info => From.Info);
      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Logic_Functor_213) return String is
   begin
      return ("BasicDecl.subp_spec_or_null");
   end Image;
   type Logic_Functor_638 is new Solver_Ifc.Converter_Type with
      null record;
   overriding function Convert (Self : Logic_Functor_638; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Logic_Functor_638) return String;
   overriding function Convert (Self : Logic_Functor_638; From : Internal_Entity) return Internal_Entity is
         pragma Unreferenced (Self);
      Ret : Internal_Entity_Classwide_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in Ada_Base_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;
      Ret := Dispatcher_Base_Type_Decl_P_Classwide_Type (Node => From.Node, E_Info => From.Info);
      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Logic_Functor_638) return String is
   begin
      return ("BaseTypeDecl.classwide_type");
   end Image;
   type Logic_Functor_211 is new Solver_Ifc.Converter_Type with
      record
            Origin : Bare_Ada_Node;
      end record;
   overriding function Convert (Self : Logic_Functor_211; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Logic_Functor_211) return String;
   overriding function Convert (Self : Logic_Functor_211; From : Internal_Entity) return Internal_Entity is
      Ret : Internal_Entity_Base_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in Ada_Basic_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;
      Ret := Dispatcher_Basic_Decl_P_Expr_Type (Node => From.Node, Origin => Self.Origin, E_Info => From.Info);
      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Logic_Functor_211) return String is
   begin
      return ("BasicDecl.expr_type");
   end Image;
   --
   --  Primitives for Bare_Ada_Node
   --
--# property-start AdaNode.declarative_scope /tmp/lal-23.0.0/ada/ast.py:128
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Declarative_Scope
  (Node : Bare_Ada_Node
  )
   return Bare_Declarative_Part
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   --# bind self Self
   Property_Result : Bare_Declarative_Part;
            procedure Finalizer_Scope_2 with Inline_Always;
      Node_Parents : Bare_Ada_Node_Array_Access;
Is_A : Boolean;
Find_Result : Bare_Ada_Node;
Cast_Result : Bare_Declarative_Part;
            procedure Finalizer_Scope_2 is
            begin
                     Dec_Ref (Node_Parents);
            end Finalizer_Scope_2;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
   --# scope-start
      --# expr-start 4 '<Cast to DeclarativePart at ast.py:131>' Cast_Result /tmp/lal-23.0.0/ada/ast.py:131
--# expr-start 3 '<Find at ast.py:129>' Find_Result /tmp/lal-23.0.0/ada/ast.py:129
   --# expr-start 1 '<FieldAccess for parents at ast.py:129>' Node_Parents /tmp/lal-23.0.0/ada/ast.py:129
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Node_Parents := Parents (Self, True); 
--# expr-done 1
      Find_Result := No_Bare_Ada_Node;
      declare
         Collection : constant Bare_Ada_Node_Array_Access := Node_Parents;
      begin
         for Item_0 of
               Collection.Items
         loop
   --# scope-start
               --# bind p Item_0
      --# expr-start 2 '<IsA DeclarativePart at ast.py:130>' Is_A /tmp/lal-23.0.0/ada/ast.py:130
Is_A := Item_0 /= null 
and then Item_0.Kind in Ada_Declarative_Part_Range; 
--# expr-done 2
      if Is_A then
         Find_Result := Item_0;
      end if;
   --# end
               exit when Is_A;
         end loop;
      end;
--# expr-done 3
   if Find_Result = null
      or else Find_Result.Kind in Ada_Declarative_Part_Range
   then
      Cast_Result := Find_Result;
   else
         Cast_Result := No_Bare_Ada_Node;
   end if;
--# expr-done 4
      Property_Result := Cast_Result;
   --# end
      Finalizer_Scope_2;
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when Exc : Precondition_Failure | Property_Error =>
               Finalizer_Scope_2;
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Ada_Node_P_Declarative_Scope;
--# end
--# property-start AdaNode.owning_unit_kind /tmp/lal-23.0.0/ada/ast.py:138
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Owning_Unit_Kind
  (Node : Bare_Ada_Node
  )
   return Analysis_Unit_Kind
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   --# bind self Self
   Property_Result : Analysis_Unit_Kind;
      Fld : Internal_Unit;
Fld_1 : Bare_Ada_Node;
Cast_Result : Bare_Compilation_Unit;
Fld_2 : Analysis_Unit_Kind;
Let_Result : Analysis_Unit_Kind;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
   --# scope-start
      --# expr-start 9 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 8 '<FieldAccess for unit_kind at ast.py:142>' Fld_2 /tmp/lal-23.0.0/ada/ast.py:142
--# expr-start 7 '<Cast to CompilationUnit at ast.py:142>' Cast_Result /tmp/lal-23.0.0/ada/ast.py:142
--# expr-start 6 '<FieldAccess for root at ast.py:142>' Fld_1 /tmp/lal-23.0.0/ada/ast.py:142
--# expr-start 5 '<FieldAccess for unit at ast.py:142>' Fld /tmp/lal-23.0.0/ada/ast.py:142
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Fld := Libadalang.Implementation.Unit (Node => Self);
--# expr-done 5
      if Fld = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Fld_1 := Fld.Ast_Root;
--# expr-done 6
   if Fld_1 = null
      or else Fld_1.Kind in Ada_Compilation_Unit_Range
   then
      Cast_Result := Fld_1;
   else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "invalid object cast");
   end if;
--# expr-done 7
      if Cast_Result = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start CompilationUnit.unit_kind
Fld_2 := Libadalang.Implementation.Compilation_Unit_P_Unit_Kind (Node => Cast_Result);
--# end
--# expr-done 8
Let_Result := Fld_2; 
--# end
--# expr-done 9
      Property_Result := Let_Result;
   --# end
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Ada_Node_P_Owning_Unit_Kind;
--# end
--# property-start AdaNode.withed_unit_helper /tmp/lal-23.0.0/ada/ast.py:145
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Withed_Unit_Helper
  (Node : Bare_Ada_Node
      ; Unit_Name : Bare_Name
  )
   return Bare_Compilation_Unit
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   --# bind self Self
   --# bind unit_name Unit_Name
   Property_Result : Bare_Compilation_Unit;
            procedure Finalizer_Scope_4 with Inline_Always;
      Block_Var : Symbol_Type_Array_Access;
Block_Var_1 : Bare_Compilation_Unit;
Block_Var_2 : Bare_Compilation_Unit;
Block_Var_3 : Bare_Compilation_Unit;
Var_Expr : Bare_Compilation_Unit;
Var_Expr_1 : Bare_Compilation_Unit;
Fld : Symbol_Type_Array_Access;
Fld_1 : Bare_Compilation_Unit;
Fld_2 : Bare_Compilation_Unit;
Is_Null : Boolean;
If_Result : Bare_Compilation_Unit;
Fld_3 : Bare_Compilation_Unit;
Fld_4 : Bare_Basic_Decl;
Scope_Result : Bare_Basic_Decl;
Result_Var : Bare_Basic_Decl;
Is_A : Boolean;
Is_Null_1 : Boolean;
If_Result_1 : Boolean;
Is_Null_2 : Boolean;
Is_Null_3 : Boolean;
If_Result_2 : Boolean;
If_Result_3 : Boolean;
If_Result_4 : Bare_Compilation_Unit;
Scope_Result_1 : Bare_Compilation_Unit;
Result_Var_1 : Bare_Compilation_Unit;
Let_Result : Bare_Compilation_Unit;
            procedure Finalizer_Scope_4 is
            begin
                     Dec_Ref (Fld);
                     Dec_Ref (Block_Var);
            end Finalizer_Scope_4;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
      if Self /= null then
         Reset_Caches (Self.Unit);
      end if;
   --# scope-start
      --# expr-start 22 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 10 '<FieldAccess for as_symbol_array at ast.py:156>' Fld /tmp/lal-23.0.0/ada/ast.py:156
      if Unit_Name = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start '[dispatcher]Name.as_symbol_array'
Fld := Libadalang.Implementation.Dispatcher_Name_P_As_Symbol_Array (Node => Unit_Name);
--# end
--# expr-done 10
Block_Var := Fld; Inc_Ref (Block_Var);
--# bind unit_name_array Block_Var
--# expr-start 11 '<FieldAccess for designated_compilation_unit at ast.py:157>' Fld_1 /tmp/lal-23.0.0/ada/ast.py:157
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start AdaNode.designated_compilation_unit
Fld_1 := Libadalang.Implementation.Ada_Node_P_Designated_Compilation_Unit (Node => Self, Name => Block_Var, Kind => Unit_Specification, Not_Found_Is_Error => False);
--# end
--# expr-done 11
Block_Var_1 := Fld_1; 
--# bind spec Block_Var_1
--# expr-start 14 '<If at ast.py:162>' If_Result /tmp/lal-23.0.0/ada/ast.py:162
--# expr-start 12 '<IsNull at ast.py:163>' Is_Null /tmp/lal-23.0.0/ada/ast.py:163
Is_Null := Block_Var_1 = null; 
--# expr-done 12
if Is_Null then
   --# expr-start 13 '<FieldAccess for designated_compilation_unit at ast.py:164>' Fld_2 /tmp/lal-23.0.0/ada/ast.py:164
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start AdaNode.designated_compilation_unit
Fld_2 := Libadalang.Implementation.Ada_Node_P_Designated_Compilation_Unit (Node => Self, Name => Block_Var, Kind => Unit_Body, Not_Found_Is_Error => False);
--# end
--# expr-done 13
   If_Result := Fld_2;
else
   If_Result := No_Bare_Ada_Node;
end if;
--# expr-done 14
Block_Var_2 := If_Result; 
--# bind body Block_Var_2
--# expr-start 21 '<If at ast.py:176>' If_Result_4 /tmp/lal-23.0.0/ada/ast.py:176
--# expr-start 16 '<IsA PackageBody at ast.py:177>' Is_A /tmp/lal-23.0.0/ada/ast.py:177
Var_Expr := Block_Var_2;
if Var_Expr /= No_Bare_Ada_Node then
   --# scope-start
--# expr-start 15 '<FieldAccess for decl at ast.py:177>' Fld_4 /tmp/lal-23.0.0/ada/ast.py:177
      if Var_Expr = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start CompilationUnit.decl
Fld_4 := Libadalang.Implementation.Compilation_Unit_P_Decl (Node => Var_Expr);
--# end
--# expr-done 15
Scope_Result := Fld_4;
   --# end
   Result_Var := Scope_Result;
else
   Result_Var := No_Bare_Ada_Node;
end if;
Is_A := Result_Var /= null 
and then Result_Var.Kind in Ada_Package_Body_Range; 
--# expr-done 16
if Is_A then
   --# expr-start 17 '<IsNull at ast.py:177>' Is_Null_1 /tmp/lal-23.0.0/ada/ast.py:177
Is_Null_1 := Block_Var_1 = null; 
--# expr-done 17
   If_Result_1 := Is_Null_1;
else
   If_Result_1 := False;
end if;
if If_Result_1 then
   If_Result_3 := True;
else
--# expr-start 18 '<IsNull at ast.py:178>' Is_Null_2 /tmp/lal-23.0.0/ada/ast.py:178
Is_Null_2 := Block_Var_1 = null; 
--# expr-done 18
if Is_Null_2 then
   --# expr-start 19 '<IsNull at ast.py:178>' Is_Null_3 /tmp/lal-23.0.0/ada/ast.py:178
Is_Null_3 := Block_Var_2 = null; 
--# expr-done 19
   If_Result_2 := Is_Null_3;
else
   If_Result_2 := False;
end if;
   If_Result_3 := If_Result_2;
end if;
if If_Result_3 then
   --# expr-start 20 '<FieldAccess for designated_compilation_unit at ast.py:180>' Fld_3 /tmp/lal-23.0.0/ada/ast.py:180
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start AdaNode.designated_compilation_unit
Fld_3 := Libadalang.Implementation.Ada_Node_P_Designated_Compilation_Unit (Node => Self, Name => Block_Var, Kind => Unit_Specification, Not_Found_Is_Error => True);
--# end
--# expr-done 20
   If_Result_4 := Fld_3;
else
   If_Result_4 := No_Bare_Ada_Node;
end if;
--# expr-done 21
Block_Var_3 := If_Result_4; 
Var_Expr_1 := Block_Var_1;
if Var_Expr_1 /= No_Bare_Ada_Node then
   --# scope-start
   --# bind e Var_Expr_1
Scope_Result_1 := Var_Expr_1;
   --# end
   Result_Var_1 := Scope_Result_1;
else
   Result_Var_1 := Block_Var_2;
end if;
Let_Result := Result_Var_1; 
--# end
--# expr-done 22
      Property_Result := Let_Result;
   --# end
      Finalizer_Scope_4;
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when Exc : Precondition_Failure | Property_Error =>
               Finalizer_Scope_4;
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Ada_Node_P_Withed_Unit_Helper;
--# end
--# property-start '[dispatcher]AdaNode.custom_id_text' dispatcher
pragma Warnings (Off, "is not referenced");
 function Dispatcher_Ada_Node_P_Custom_Id_Text
  (Node : Bare_Ada_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )
   return String_Type
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   --# bind self Self
   Property_Result : String_Type;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
      if Self /= null then
         Reset_Caches (Self.Unit);
      end if;
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dispatching on null node");
      end if;
      case Ada_Ada_Node (Self.Kind) is
               when Ada_Abort_Absent .. Ada_Param_Assoc | Ada_Case_Stmt_Alternative .. Ada_Signed_Int_Type_Def | Ada_Synthetic_Type_Expr .. Ada_With_Private_Present =>
                  --# property-call-start AdaNode.custom_id_text
                  Property_Result := Ada_Node_P_Custom_Id_Text
                    (Self
                        , E_Info
                    );
                  --# end
               when Ada_Abstract_State_Decl .. Ada_Single_Task_Decl =>
                  --# property-call-start BasicDecl.custom_id_text
                  Property_Result := Basic_Decl_P_Custom_Id_Text
                    (Self
                        , E_Info
                    );
                  --# end
               when Ada_Anonymous_Type =>
                  --# property-call-start AnonymousType.custom_id_text
                  Property_Result := Anonymous_Type_P_Custom_Id_Text
                    (Self
                        , E_Info
                    );
                  --# end
               when Ada_Enum_Lit_Synth_Type_Expr =>
                  --# property-call-start EnumLitSynthTypeExpr.custom_id_text
                  Property_Result := Enum_Lit_Synth_Type_Expr_P_Custom_Id_Text
                    (Self
                        , E_Info
                    );
                  --# end
               when Ada_Subtype_Indication .. Ada_Discrete_Subtype_Indication =>
                  --# property-call-start SubtypeIndication.custom_id_text
                  Property_Result := Subtype_Indication_P_Custom_Id_Text
                    (Self
                        , E_Info
                    );
                  --# end
      end case;
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Dispatcher_Ada_Node_P_Custom_Id_Text;
--# end
--# property-start AdaNode.is_contract_aspect /tmp/lal-23.0.0/ada/ast.py:201
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Is_Contract_Aspect
  (Node : Bare_Ada_Node
      ; Name : Symbol_Type
  )
   return Boolean
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   --# bind self Self
   --# bind name Name
   Property_Result : Boolean;
      Sym : Symbol_Type;
Is_Equal : Boolean;
Sym_1 : Symbol_Type;
Is_Equal_1 : Boolean;
If_Result : Boolean;
Sym_2 : Symbol_Type;
Is_Equal_2 : Boolean;
If_Result_1 : Boolean;
Sym_3 : Symbol_Type;
Is_Equal_3 : Boolean;
If_Result_2 : Boolean;
Sym_4 : Symbol_Type;
Is_Equal_4 : Boolean;
If_Result_3 : Boolean;
Sym_5 : Symbol_Type;
Is_Equal_5 : Boolean;
If_Result_4 : Boolean;
Sym_6 : Symbol_Type;
Is_Equal_6 : Boolean;
If_Result_5 : Boolean;
Sym_7 : Symbol_Type;
Is_Equal_7 : Boolean;
If_Result_6 : Boolean;
Sym_8 : Symbol_Type;
Is_Equal_8 : Boolean;
If_Result_7 : Boolean;
Sym_9 : Symbol_Type;
Is_Equal_9 : Boolean;
If_Result_8 : Boolean;
Sym_10 : Symbol_Type;
Is_Equal_10 : Boolean;
If_Result_9 : Boolean;
Sym_11 : Symbol_Type;
Is_Equal_11 : Boolean;
If_Result_10 : Boolean;
Sym_12 : Symbol_Type;
Is_Equal_12 : Boolean;
If_Result_11 : Boolean;
Sym_13 : Symbol_Type;
Is_Equal_13 : Boolean;
If_Result_12 : Boolean;
Sym_14 : Symbol_Type;
Is_Equal_14 : Boolean;
If_Result_13 : Boolean;
Sym_15 : Symbol_Type;
Is_Equal_15 : Boolean;
If_Result_14 : Boolean;
Sym_16 : Symbol_Type;
Is_Equal_16 : Boolean;
If_Result_15 : Boolean;
Sym_17 : Symbol_Type;
Is_Equal_17 : Boolean;
If_Result_16 : Boolean;
Sym_18 : Symbol_Type;
Is_Equal_18 : Boolean;
If_Result_17 : Boolean;
Sym_19 : Symbol_Type;
Is_Equal_19 : Boolean;
If_Result_18 : Boolean;
Sym_20 : Symbol_Type;
Is_Equal_20 : Boolean;
If_Result_19 : Boolean;
Sym_21 : Symbol_Type;
Is_Equal_21 : Boolean;
If_Result_20 : Boolean;
Sym_22 : Symbol_Type;
Is_Equal_22 : Boolean;
If_Result_21 : Boolean;
Sym_23 : Symbol_Type;
Is_Equal_23 : Boolean;
If_Result_22 : Boolean;
Sym_24 : Symbol_Type;
Is_Equal_24 : Boolean;
If_Result_23 : Boolean;
Sym_25 : Symbol_Type;
Is_Equal_25 : Boolean;
If_Result_24 : Boolean;
Let_Result : Boolean;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
   --# scope-start
      --# expr-start 75 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 24 '<Eq at ast.py:202>' Is_Equal /tmp/lal-23.0.0/ada/ast.py:202
--# expr-start 23 '<Symbol Pre at ???>' Sym None
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Pre); 
--# expr-done 23
Is_Equal := Name = Sym; 
--# expr-done 24
if Is_Equal then
   If_Result := True;
else
   --# expr-start 26 '<Eq at ast.py:202>' Is_Equal_1 /tmp/lal-23.0.0/ada/ast.py:202
--# expr-start 25 '<Symbol Pre'"'"'Class at ???>' Sym_1 None
Sym_1 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Pre_Class); 
--# expr-done 25
Is_Equal_1 := Name = Sym_1; 
--# expr-done 26
   If_Result := Is_Equal_1;
end if;
if If_Result then
   If_Result_1 := True;
else
   --# expr-start 28 '<Eq at ast.py:202>' Is_Equal_2 /tmp/lal-23.0.0/ada/ast.py:202
--# expr-start 27 '<Symbol Post at ???>' Sym_2 None
Sym_2 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Post); 
--# expr-done 27
Is_Equal_2 := Name = Sym_2; 
--# expr-done 28
   If_Result_1 := Is_Equal_2;
end if;
if If_Result_1 then
   If_Result_2 := True;
else
   --# expr-start 30 '<Eq at ast.py:202>' Is_Equal_3 /tmp/lal-23.0.0/ada/ast.py:202
--# expr-start 29 '<Symbol Post'"'"'Class at ???>' Sym_3 None
Sym_3 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Post_Class); 
--# expr-done 29
Is_Equal_3 := Name = Sym_3; 
--# expr-done 30
   If_Result_2 := Is_Equal_3;
end if;
if If_Result_2 then
   If_Result_3 := True;
else
   --# expr-start 32 '<Eq at ast.py:202>' Is_Equal_4 /tmp/lal-23.0.0/ada/ast.py:202
--# expr-start 31 '<Symbol Refined_Post at ???>' Sym_4 None
Sym_4 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Refined_Post); 
--# expr-done 31
Is_Equal_4 := Name = Sym_4; 
--# expr-done 32
   If_Result_3 := Is_Equal_4;
end if;
if If_Result_3 then
   If_Result_4 := True;
else
   --# expr-start 34 '<Eq at ast.py:202>' Is_Equal_5 /tmp/lal-23.0.0/ada/ast.py:202
--# expr-start 33 '<Symbol Precondition at ???>' Sym_5 None
Sym_5 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Precondition); 
--# expr-done 33
Is_Equal_5 := Name = Sym_5; 
--# expr-done 34
   If_Result_4 := Is_Equal_5;
end if;
if If_Result_4 then
   If_Result_5 := True;
else
   --# expr-start 36 '<Eq at ast.py:202>' Is_Equal_6 /tmp/lal-23.0.0/ada/ast.py:202
--# expr-start 35 '<Symbol Postcondition at ???>' Sym_6 None
Sym_6 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Postcondition); 
--# expr-done 35
Is_Equal_6 := Name = Sym_6; 
--# expr-done 36
   If_Result_5 := Is_Equal_6;
end if;
if If_Result_5 then
   If_Result_6 := True;
else
   --# expr-start 38 '<Eq at ast.py:202>' Is_Equal_7 /tmp/lal-23.0.0/ada/ast.py:202
--# expr-start 37 '<Symbol Precondition'"'"'Class at ???>' Sym_7 None
Sym_7 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Precondition_Class); 
--# expr-done 37
Is_Equal_7 := Name = Sym_7; 
--# expr-done 38
   If_Result_6 := Is_Equal_7;
end if;
if If_Result_6 then
   If_Result_7 := True;
else
   --# expr-start 40 '<Eq at ast.py:202>' Is_Equal_8 /tmp/lal-23.0.0/ada/ast.py:202
--# expr-start 39 '<Symbol Postcondition'"'"'Class at ???>' Sym_8 None
Sym_8 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Postcondition_Class); 
--# expr-done 39
Is_Equal_8 := Name = Sym_8; 
--# expr-done 40
   If_Result_7 := Is_Equal_8;
end if;
if If_Result_7 then
   If_Result_8 := True;
else
   --# expr-start 42 '<Eq at ast.py:202>' Is_Equal_9 /tmp/lal-23.0.0/ada/ast.py:202
--# expr-start 41 '<Symbol Invariant at ???>' Sym_9 None
Sym_9 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Invariant); 
--# expr-done 41
Is_Equal_9 := Name = Sym_9; 
--# expr-done 42
   If_Result_8 := Is_Equal_9;
end if;
if If_Result_8 then
   If_Result_9 := True;
else
   --# expr-start 44 '<Eq at ast.py:202>' Is_Equal_10 /tmp/lal-23.0.0/ada/ast.py:202
--# expr-start 43 '<Symbol Invariant'"'"'Class at ???>' Sym_10 None
Sym_10 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Invariant_Class); 
--# expr-done 43
Is_Equal_10 := Name = Sym_10; 
--# expr-done 44
   If_Result_9 := Is_Equal_10;
end if;
if If_Result_9 then
   If_Result_10 := True;
else
   --# expr-start 46 '<Eq at ast.py:202>' Is_Equal_11 /tmp/lal-23.0.0/ada/ast.py:202
--# expr-start 45 '<Symbol Type_Invariant at ???>' Sym_11 None
Sym_11 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Type_Invariant); 
--# expr-done 45
Is_Equal_11 := Name = Sym_11; 
--# expr-done 46
   If_Result_10 := Is_Equal_11;
end if;
if If_Result_10 then
   If_Result_11 := True;
else
   --# expr-start 48 '<Eq at ast.py:202>' Is_Equal_12 /tmp/lal-23.0.0/ada/ast.py:202
--# expr-start 47 '<Symbol Type_Invariant'"'"'Class at ???>' Sym_12 None
Sym_12 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Type_Invariant_Class); 
--# expr-done 47
Is_Equal_12 := Name = Sym_12; 
--# expr-done 48
   If_Result_11 := Is_Equal_12;
end if;
if If_Result_11 then
   If_Result_12 := True;
else
   --# expr-start 50 '<Eq at ast.py:202>' Is_Equal_13 /tmp/lal-23.0.0/ada/ast.py:202
--# expr-start 49 '<Symbol Predicate at ???>' Sym_13 None
Sym_13 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Predicate); 
--# expr-done 49
Is_Equal_13 := Name = Sym_13; 
--# expr-done 50
   If_Result_12 := Is_Equal_13;
end if;
if If_Result_12 then
   If_Result_13 := True;
else
   --# expr-start 52 '<Eq at ast.py:202>' Is_Equal_14 /tmp/lal-23.0.0/ada/ast.py:202
--# expr-start 51 '<Symbol Static_Predicate at ???>' Sym_14 None
Sym_14 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Static_Predicate); 
--# expr-done 51
Is_Equal_14 := Name = Sym_14; 
--# expr-done 52
   If_Result_13 := Is_Equal_14;
end if;
if If_Result_13 then
   If_Result_14 := True;
else
   --# expr-start 54 '<Eq at ast.py:202>' Is_Equal_15 /tmp/lal-23.0.0/ada/ast.py:202
--# expr-start 53 '<Symbol Dynamic_Predicate at ???>' Sym_15 None
Sym_15 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Dynamic_Predicate); 
--# expr-done 53
Is_Equal_15 := Name = Sym_15; 
--# expr-done 54
   If_Result_14 := Is_Equal_15;
end if;
if If_Result_14 then
   If_Result_15 := True;
else
   --# expr-start 56 '<Eq at ast.py:202>' Is_Equal_16 /tmp/lal-23.0.0/ada/ast.py:202
--# expr-start 55 '<Symbol Default_Initial_Condition at ???>' Sym_16 None
Sym_16 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Default_Initial_Condition); 
--# expr-done 55
Is_Equal_16 := Name = Sym_16; 
--# expr-done 56
   If_Result_15 := Is_Equal_16;
end if;
if If_Result_15 then
   If_Result_16 := True;
else
   --# expr-start 58 '<Eq at ast.py:202>' Is_Equal_17 /tmp/lal-23.0.0/ada/ast.py:202
--# expr-start 57 '<Symbol Contract_Cases at ???>' Sym_17 None
Sym_17 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Contract_Cases); 
--# expr-done 57
Is_Equal_17 := Name = Sym_17; 
--# expr-done 58
   If_Result_16 := Is_Equal_17;
end if;
if If_Result_16 then
   If_Result_17 := True;
else
   --# expr-start 60 '<Eq at ast.py:202>' Is_Equal_18 /tmp/lal-23.0.0/ada/ast.py:202
--# expr-start 59 '<Symbol Test_Case at ???>' Sym_18 None
Sym_18 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Test_Case); 
--# expr-done 59
Is_Equal_18 := Name = Sym_18; 
--# expr-done 60
   If_Result_17 := Is_Equal_18;
end if;
if If_Result_17 then
   If_Result_18 := True;
else
   --# expr-start 62 '<Eq at ast.py:202>' Is_Equal_19 /tmp/lal-23.0.0/ada/ast.py:202
--# expr-start 61 '<Symbol Global at ???>' Sym_19 None
Sym_19 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Global); 
--# expr-done 61
Is_Equal_19 := Name = Sym_19; 
--# expr-done 62
   If_Result_18 := Is_Equal_19;
end if;
if If_Result_18 then
   If_Result_19 := True;
else
   --# expr-start 64 '<Eq at ast.py:202>' Is_Equal_20 /tmp/lal-23.0.0/ada/ast.py:202
--# expr-start 63 '<Symbol Refined_Global at ???>' Sym_20 None
Sym_20 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Refined_Global); 
--# expr-done 63
Is_Equal_20 := Name = Sym_20; 
--# expr-done 64
   If_Result_19 := Is_Equal_20;
end if;
if If_Result_19 then
   If_Result_20 := True;
else
   --# expr-start 66 '<Eq at ast.py:202>' Is_Equal_21 /tmp/lal-23.0.0/ada/ast.py:202
--# expr-start 65 '<Symbol Refined_State at ???>' Sym_21 None
Sym_21 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Refined_State); 
--# expr-done 65
Is_Equal_21 := Name = Sym_21; 
--# expr-done 66
   If_Result_20 := Is_Equal_21;
end if;
if If_Result_20 then
   If_Result_21 := True;
else
   --# expr-start 68 '<Eq at ast.py:202>' Is_Equal_22 /tmp/lal-23.0.0/ada/ast.py:202
--# expr-start 67 '<Symbol Stable_Properties at ???>' Sym_22 None
Sym_22 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Stable_Properties); 
--# expr-done 67
Is_Equal_22 := Name = Sym_22; 
--# expr-done 68
   If_Result_21 := Is_Equal_22;
end if;
if If_Result_21 then
   If_Result_22 := True;
else
   --# expr-start 70 '<Eq at ast.py:202>' Is_Equal_23 /tmp/lal-23.0.0/ada/ast.py:202
--# expr-start 69 '<Symbol Depends at ???>' Sym_23 None
Sym_23 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Depends); 
--# expr-done 69
Is_Equal_23 := Name = Sym_23; 
--# expr-done 70
   If_Result_22 := Is_Equal_23;
end if;
if If_Result_22 then
   If_Result_23 := True;
else
   --# expr-start 72 '<Eq at ast.py:202>' Is_Equal_24 /tmp/lal-23.0.0/ada/ast.py:202
--# expr-start 71 '<Symbol Refined_Depends at ???>' Sym_24 None
Sym_24 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Refined_Depends); 
--# expr-done 71
Is_Equal_24 := Name = Sym_24; 
--# expr-done 72
   If_Result_23 := Is_Equal_24;
end if;
if If_Result_23 then
   If_Result_24 := True;
else
   --# expr-start 74 '<Eq at ast.py:202>' Is_Equal_25 /tmp/lal-23.0.0/ada/ast.py:202
--# expr-start 73 '<Symbol Predicate_Failure at ???>' Sym_25 None
Sym_25 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Predicate_Failure); 
--# expr-done 73
Is_Equal_25 := Name = Sym_25; 
--# expr-done 74
   If_Result_24 := Is_Equal_25;
end if;
Let_Result := If_Result_24; 
--# end
--# expr-done 75
      Property_Result := Let_Result;
   --# end
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Ada_Node_P_Is_Contract_Aspect;
--# end
--# property-start AdaNode.in_contract /tmp/lal-23.0.0/ada/ast.py:216
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_In_Contract
  (Node : Bare_Ada_Node
  )
   return Boolean
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   --# bind self Self
   Property_Result : Boolean;
            procedure Finalizer_Scope_7 with Inline_Always;
      Var_Expr : Boolean;
Var_Expr_1 : Bare_Aspect_Assoc;
Var_Expr_2 : Bare_Pragma_Node;
Node_Parents : Bare_Ada_Node_Array_Access;
Cast_Expr : Bare_Ada_Node;
Cast_Result : Bare_Aspect_Assoc;
Fld : Bare_Name;
As_Entity : Internal_Entity_Name;
Is_Null : Boolean;
If_Result : Internal_Entity_Name;
Fld_1 : Symbol_Type;
Fld_2 : Boolean;
Scope_Result : Boolean;
Result_Var : Boolean;
Scope_Result_1 : Boolean;
Cast_Expr_1 : Bare_Ada_Node;
Cast_Result_1 : Bare_Pragma_Node;
Fld_3 : Bare_Identifier;
As_Entity_1 : Internal_Entity_Identifier;
Is_Null_1 : Boolean;
If_Result_1 : Internal_Entity_Identifier;
Fld_4 : Symbol_Type;
Fld_5 : Boolean;
Scope_Result_2 : Boolean;
Result_Var_1 : Boolean;
Result_Var_2 : Boolean;
Find_Result : Bare_Ada_Node;
Is_Null_2 : Boolean;
Not_Val : Boolean;
            procedure Finalizer_Scope_7 is
            begin
                     Dec_Ref (Node_Parents);
            end Finalizer_Scope_7;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
   --# scope-start
      --# expr-start 89 '<Not at ast.py:216>' Not_Val /tmp/lal-23.0.0/ada/ast.py:216
--# expr-start 88 '<IsNull at ast.py:222>' Is_Null_2 /tmp/lal-23.0.0/ada/ast.py:222
--# expr-start 87 '<Find at ast.py:216>' Find_Result /tmp/lal-23.0.0/ada/ast.py:216
   --# expr-start 76 '<FieldAccess for parents at ast.py:216>' Node_Parents /tmp/lal-23.0.0/ada/ast.py:216
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Node_Parents := Parents (Self, True); 
--# expr-done 76
      Find_Result := No_Bare_Ada_Node;
      declare
         Collection : constant Bare_Ada_Node_Array_Access := Node_Parents;
      begin
         for Item_1 of
               Collection.Items
         loop
   --# scope-start
               --# bind p Item_1
--# expr-start 77 '<Cast to AspectAssoc at ast.py:217>' Cast_Result /tmp/lal-23.0.0/ada/ast.py:217
Cast_Expr := Item_1; 
   if Cast_Expr = null
      or else Cast_Expr.Kind in Ada_Aspect_Assoc_Range
   then
      Cast_Result := Cast_Expr;
   else
         Cast_Result := No_Bare_Ada_Node;
   end if;
--# expr-done 77
Var_Expr_1 := Cast_Result;
if Var_Expr_1 /= No_Bare_Ada_Node then
   --# scope-start
   --# bind a Var_Expr_1
--# expr-start 81 '<FieldAccess for is_contract_aspect at ast.py:218>' Fld_2 /tmp/lal-23.0.0/ada/ast.py:218
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# expr-start 80 '<FieldAccess for name_symbol at ast.py:218>' Fld_1 /tmp/lal-23.0.0/ada/ast.py:218
--# expr-start 79 '<as_bare_entity at ast.py:218>' If_Result /tmp/lal-23.0.0/ada/ast.py:218
--# expr-start 78 '<FieldAccess for id at ast.py:218>' Fld /tmp/lal-23.0.0/ada/ast.py:218
      if Var_Expr_1 = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Fld := Var_Expr_1.Aspect_Assoc_F_Id;
--# expr-done 78
Is_Null := Fld = null; 
if Is_Null then
   If_Result := No_Entity_Name;
else
As_Entity := (Info => No_Entity_Info, Node => Fld); 
   If_Result := As_Entity;
end if;
--# expr-done 79
      if If_Result.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start '[dispatcher]Name.name_symbol'
Fld_1 := Libadalang.Implementation.Dispatcher_Name_P_Name_Symbol (Node => If_Result.Node);
--# end
--# expr-done 80
--# property-call-start AdaNode.is_contract_aspect
Fld_2 := Libadalang.Implementation.Ada_Node_P_Is_Contract_Aspect (Node => Self, Name => Fld_1);
--# end
--# expr-done 81
Scope_Result := Fld_2;
   --# end
   Result_Var := Scope_Result;
else
   Result_Var := False;
end if;
Var_Expr := Result_Var;
if Var_Expr /= False then
   --# scope-start
   --# bind e Var_Expr
Scope_Result_1 := Var_Expr;
   --# end
   Result_Var_2 := Scope_Result_1;
else
--# expr-start 82 '<Cast to Pragma at ast.py:219>' Cast_Result_1 /tmp/lal-23.0.0/ada/ast.py:219
Cast_Expr_1 := Item_1; 
   if Cast_Expr_1 = null
      or else Cast_Expr_1.Kind in Ada_Pragma_Node_Range
   then
      Cast_Result_1 := Cast_Expr_1;
   else
         Cast_Result_1 := No_Bare_Ada_Node;
   end if;
--# expr-done 82
Var_Expr_2 := Cast_Result_1;
if Var_Expr_2 /= No_Bare_Ada_Node then
   --# scope-start
   --# bind p Var_Expr_2
--# expr-start 86 '<FieldAccess for is_contract_aspect at ast.py:220>' Fld_5 /tmp/lal-23.0.0/ada/ast.py:220
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# expr-start 85 '<FieldAccess for name_symbol at ast.py:220>' Fld_4 /tmp/lal-23.0.0/ada/ast.py:220
--# expr-start 84 '<as_bare_entity at ast.py:220>' If_Result_1 /tmp/lal-23.0.0/ada/ast.py:220
--# expr-start 83 '<FieldAccess for id at ast.py:220>' Fld_3 /tmp/lal-23.0.0/ada/ast.py:220
      if Var_Expr_2 = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Fld_3 := Var_Expr_2.Pragma_Node_F_Id;
--# expr-done 83
Is_Null_1 := Fld_3 = null; 
if Is_Null_1 then
   If_Result_1 := No_Entity_Identifier;
else
As_Entity_1 := (Info => No_Entity_Info, Node => Fld_3); 
   If_Result_1 := As_Entity_1;
end if;
--# expr-done 84
      if If_Result_1.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start '[dispatcher]Name.name_symbol'
Fld_4 := Libadalang.Implementation.Dispatcher_Name_P_Name_Symbol (Node => If_Result_1.Node);
--# end
--# expr-done 85
--# property-call-start AdaNode.is_contract_aspect
Fld_5 := Libadalang.Implementation.Ada_Node_P_Is_Contract_Aspect (Node => Self, Name => Fld_4);
--# end
--# expr-done 86
Scope_Result_2 := Fld_5;
   --# end
   Result_Var_1 := Scope_Result_2;
else
   Result_Var_1 := False;
end if;
   Result_Var_2 := Result_Var_1;
end if;
      if Result_Var_2 then
         Find_Result := Item_1;
      end if;
   --# end
               exit when Result_Var_2;
         end loop;
      end;
--# expr-done 87
Is_Null_2 := Find_Result = null; 
--# expr-done 88
Not_Val := not (Is_Null_2); 
--# expr-done 89
      Property_Result := Not_Val;
   --# end
      Finalizer_Scope_7;
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when Exc : Precondition_Failure | Property_Error =>
               Finalizer_Scope_7;
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Ada_Node_P_In_Contract;
--# end
--# property-start AdaNode.in_aspect /tmp/lal-23.0.0/ada/ast.py:225
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_In_Aspect
  (Node : Bare_Ada_Node
      ; Name : Symbol_Type
  )
   return Boolean
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   --# bind self Self
   --# bind name Name
   Property_Result : Boolean;
            procedure Finalizer_Scope_8 with Inline_Always;
      Var_Expr : Bare_Aspect_Assoc;
Node_Parents : Bare_Ada_Node_Array_Access;
Cast_Expr : Bare_Ada_Node;
Cast_Result : Bare_Aspect_Assoc;
Fld : Bare_Name;
Fld_1 : Boolean;
Scope_Result : Boolean;
Result_Var : Boolean;
Quantifier_Result : Boolean;
Let_Result : Boolean;
            procedure Finalizer_Scope_8 is
            begin
                     Dec_Ref (Node_Parents);
            end Finalizer_Scope_8;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
   --# scope-start
      --# expr-start 95 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 94 '<AnyQuantifier at ast.py:229>' Quantifier_Result /tmp/lal-23.0.0/ada/ast.py:229
   --# expr-start 90 '<FieldAccess for parents at ast.py:229>' Node_Parents /tmp/lal-23.0.0/ada/ast.py:229
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Node_Parents := Parents (Self, True); 
--# expr-done 90
      Quantifier_Result := False;
      declare
         Collection : constant Bare_Ada_Node_Array_Access := Node_Parents;
      begin
         for Item_2 of
               Collection.Items
         loop
   --# scope-start
               --# bind p Item_2
--# expr-start 91 '<Cast to AspectAssoc at ast.py:230>' Cast_Result /tmp/lal-23.0.0/ada/ast.py:230
Cast_Expr := Item_2; 
   if Cast_Expr = null
      or else Cast_Expr.Kind in Ada_Aspect_Assoc_Range
   then
      Cast_Result := Cast_Expr;
   else
         Cast_Result := No_Bare_Ada_Node;
   end if;
--# expr-done 91
Var_Expr := Cast_Result;
if Var_Expr /= No_Bare_Ada_Node then
   --# scope-start
   --# bind a Var_Expr
--# expr-start 93 '<FieldAccess for name_is at ast.py:230>' Fld_1 /tmp/lal-23.0.0/ada/ast.py:230
--# expr-start 92 '<FieldAccess for id at ast.py:230>' Fld /tmp/lal-23.0.0/ada/ast.py:230
      if Var_Expr = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Fld := Var_Expr.Aspect_Assoc_F_Id;
--# expr-done 92
      if Fld = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start Name.name_is
Fld_1 := Libadalang.Implementation.Name_P_Name_Is (Node => Fld, Sym => Name);
--# end
--# expr-done 93
Scope_Result := Fld_1;
   --# end
   Result_Var := Scope_Result;
else
   Result_Var := False;
end if;
      Quantifier_Result := Result_Var;
   --# end
               exit when Quantifier_Result;
         end loop;
      end;
--# expr-done 94
Let_Result := Quantifier_Result; 
--# end
--# expr-done 95
      Property_Result := Let_Result;
   --# end
      Finalizer_Scope_8;
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when Exc : Precondition_Failure | Property_Error =>
               Finalizer_Scope_8;
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Ada_Node_P_In_Aspect;
--# end
--# property-start AdaNode.empty_env /tmp/lal-23.0.0/ada/ast.py:233
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Empty_Env
  (Node : Bare_Ada_Node
  )
   return Lexical_Env
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   --# bind self Self
   Property_Result : Lexical_Env;
            procedure Finalizer_Scope_9 with Inline_Always;
      Node_Parents : Bare_Ada_Node_Array_Access;
Is_A : Boolean;
Find_Result : Bare_Ada_Node;
Cast_Result : Bare_Compilation_Unit;
Fld : Lexical_Env := Empty_Env;
            procedure Finalizer_Scope_9 is
            begin
                     Dec_Ref (Node_Parents);
                     Dec_Ref (Fld);
            end Finalizer_Scope_9;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
      if Self /= null then
         Reset_Caches (Self.Unit);
      end if;
   --# scope-start
      --# expr-start 100 '<FieldAccess for get_empty_env at ast.py:235>' Fld /tmp/lal-23.0.0/ada/ast.py:235
--# expr-start 99 '<Cast to CompilationUnit at ast.py:235>' Cast_Result /tmp/lal-23.0.0/ada/ast.py:235
--# expr-start 98 '<Find at ast.py:234>' Find_Result /tmp/lal-23.0.0/ada/ast.py:234
   --# expr-start 96 '<FieldAccess for parents at ast.py:234>' Node_Parents /tmp/lal-23.0.0/ada/ast.py:234
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Node_Parents := Parents (Self, True); 
--# expr-done 96
      Find_Result := No_Bare_Ada_Node;
      declare
         Collection : constant Bare_Ada_Node_Array_Access := Node_Parents;
      begin
         for Item_3 of
               Collection.Items
         loop
   --# scope-start
               --# bind p Item_3
      --# expr-start 97 '<IsA CompilationUnit at ast.py:234>' Is_A /tmp/lal-23.0.0/ada/ast.py:234
Is_A := Item_3 /= null 
and then Item_3.Kind in Ada_Compilation_Unit_Range; 
--# expr-done 97
      if Is_A then
         Find_Result := Item_3;
      end if;
   --# end
               exit when Is_A;
         end loop;
      end;
--# expr-done 98
   if Find_Result = null
      or else Find_Result.Kind in Ada_Compilation_Unit_Range
   then
      Cast_Result := Find_Result;
   else
         Cast_Result := No_Bare_Ada_Node;
   end if;
--# expr-done 99
      if Cast_Result = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Fld := Libadalang.Implementation.Extensions.Compilation_Unit_P_Get_Empty_Env (Node => Cast_Result);
--# expr-done 100
      Property_Result := Fld;
         Inc_Ref (Property_Result);
   --# end
      Finalizer_Scope_9;
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when Exc : Precondition_Failure | Property_Error =>
               Finalizer_Scope_9;
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Ada_Node_P_Empty_Env;
--# end
--# property-start AdaNode.is_not_null /tmp/lal-23.0.0/ada/ast.py:239
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Is_Not_Null
  (Node : Bare_Ada_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )
   return Boolean
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   Ent : Internal_Entity :=
     Internal_Entity'(Node => Self, Info => E_Info);
   --# bind entity Ent
   Property_Result : Boolean;
      Is_Null : Boolean;
Not_Val : Boolean;
Let_Result : Boolean;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
   --# scope-start
      --# expr-start 103 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 102 '<Not at ast.py:247>' Not_Val /tmp/lal-23.0.0/ada/ast.py:247
--# expr-start 101 '<IsNull at ast.py:247>' Is_Null /tmp/lal-23.0.0/ada/ast.py:247
Is_Null := Ent.Node = null; 
--# expr-done 101
Not_Val := not (Is_Null); 
--# expr-done 102
Let_Result := Not_Val; 
--# end
--# expr-done 103
      Property_Result := Let_Result;
   --# end
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Ada_Node_P_Is_Not_Null;
--# end
--# property-start AdaNode.eval_discrete_range /tmp/lal-23.0.0/ada/ast.py:250
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Eval_Discrete_Range
  (Node : Bare_Ada_Node
      ; Dr : Internal_Discrete_Range
  )
   return Internal_Eval_Discrete_Range
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   --# bind self Self
   --# bind dr Dr
   Property_Result : Internal_Eval_Discrete_Range;
            procedure Finalizer_Scope_11 with Inline_Always;
            procedure Finalizer_Scope_1743 with Inline_Always;
      Var_Expr : Internal_Entity_Expr;
Fld : Internal_Entity_Expr;
Fld_1 : Big_Integer_Type;
Scope_Result : Big_Integer_Type;
Big_Int : Big_Integer_Type;
Result_Var : Big_Integer_Type;
Fld_2 : Internal_Entity_Expr;
Fld_3 : Big_Integer_Type;
New_Struct : Internal_Eval_Discrete_Range := No_Eval_Discrete_Range;
Is_Equal : Boolean;
If_Result : Internal_Eval_Discrete_Range := No_Eval_Discrete_Range;
Let_Result : Internal_Eval_Discrete_Range := No_Eval_Discrete_Range;
            procedure Finalizer_Scope_11 is
            begin
                     Dec_Ref (Scope_Result);
                     Dec_Ref (Big_Int);
                     Dec_Ref (Result_Var);
                     Dec_Ref (Fld_3);
                     Dec_Ref (New_Struct);
                     Dec_Ref (If_Result);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_11;
            procedure Finalizer_Scope_1743 is
            begin
                     Dec_Ref (Fld_1);
            end Finalizer_Scope_1743;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
   --# scope-start
      --# expr-start 112 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 111 '<If at ast.py:254>' If_Result /tmp/lal-23.0.0/ada/ast.py:254
--# expr-start 104 '<Eq at ast.py:255>' Is_Equal /tmp/lal-23.0.0/ada/ast.py:255
Is_Equal := Dr = No_Discrete_Range; 
--# expr-done 104
if Is_Equal then
   If_Result := raise Property_Error with "Attempting to evaluate a null discrete range";
else
   --# expr-start 110 '<New EvalDiscreteRange at ast.py:260>' New_Struct /tmp/lal-23.0.0/ada/ast.py:260
--# expr-start 106 '<FieldAccess for eval_as_int at ast.py:265>' Fld_3 /tmp/lal-23.0.0/ada/ast.py:265
--# expr-start 105 '<FieldAccess for high_bound at ast.py:265>' Fld_2 /tmp/lal-23.0.0/ada/ast.py:265
Fld_2 := Dr.High_Bound;
--# expr-done 105
      if Fld_2.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start Expr.eval_as_int
Fld_3 := Libadalang.Implementation.Expr_P_Eval_As_Int (Node => Fld_2.Node, E_Info => Fld_2.Info);
--# end
--# expr-done 106
--# expr-start 107 '<FieldAccess for low_bound at ast.py:261>' Fld /tmp/lal-23.0.0/ada/ast.py:261
Fld := Dr.Low_Bound;
--# expr-done 107
Var_Expr := Fld;
if Var_Expr /= No_Entity_Expr then
   --# scope-start
   --# bind lb Var_Expr
--# expr-start 108 '<FieldAccess for eval_as_int at ast.py:262>' Fld_1 /tmp/lal-23.0.0/ada/ast.py:262
      if Var_Expr.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start Expr.eval_as_int
Fld_1 := Libadalang.Implementation.Expr_P_Eval_As_Int (Node => Var_Expr.Node, E_Info => Var_Expr.Info);
--# end
--# expr-done 108
Scope_Result := Fld_1;
   Inc_Ref (Scope_Result);
   --# end
      Finalizer_Scope_1743;
   Result_Var := Scope_Result;
else
   --# expr-start 109 '<BigInteger 0 at ast.py:263>' Big_Int /tmp/lal-23.0.0/ada/ast.py:263
Big_Int := Create_Big_Integer ("0"); 
--# expr-done 109
   Result_Var := Big_Int;
end if;
      Inc_Ref (Result_Var);
Inc_Ref (Fld_3);
Inc_Ref (Result_Var);
New_Struct := (High_Bound => Fld_3, Low_Bound => Result_Var); 
--# expr-done 110
   If_Result := New_Struct;
end if;
      Inc_Ref (If_Result);
--# expr-done 111
Let_Result := If_Result; Inc_Ref (Let_Result);
--# end
--# expr-done 112
      Property_Result := Let_Result;
         Inc_Ref (Property_Result);
   --# end
      Finalizer_Scope_11;
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when Exc : Precondition_Failure | Property_Error =>
               Finalizer_Scope_11;
               Finalizer_Scope_1743;
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Ada_Node_P_Eval_Discrete_Range;
--# end
--# property-start AdaNode.sym_join /tmp/lal-23.0.0/ada/ast.py:270
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Sym_Join
  (Node : Bare_Ada_Node
      ; Syms : Symbol_Type_Array_Access
      ; Sep : String_Type
  )
   return String_Type
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   --# bind self Self
   --# bind syms Syms
   --# bind sep Sep
   Property_Result : String_Type;
            procedure Finalizer_Scope_12 with Inline_Always;
            procedure Finalizer_Scope_1744 with Inline_Always;
      Fld : String_Type;
Map_Result : String_Type_Array_Access;
Join_Result : String_Type;
Let_Result : String_Type;
            procedure Finalizer_Scope_12 is
            begin
                     Dec_Ref (Map_Result);
                     Dec_Ref (Join_Result);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_12;
            procedure Finalizer_Scope_1744 is
            begin
                     Dec_Ref (Fld);
            end Finalizer_Scope_1744;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
      if Self /= null then
         Reset_Caches (Self.Unit);
      end if;
   --# scope-start
      --# expr-start 116 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 115 '<Join at ast.py:274>' Join_Result /tmp/lal-23.0.0/ada/ast.py:274
--# expr-start 114 '<Map at ast.py:274>' Map_Result /tmp/lal-23.0.0/ada/ast.py:274
      declare
         Map_Result_Vec : String_Type_Vectors.Vector;
      begin
      declare
         Collection : constant Symbol_Type_Array_Access := Syms;
      begin
         for Item_4 of
               Collection.Items
         loop
   --# scope-start
               --# bind s Item_4
      --# expr-start 113 '<FieldAccess for image at ast.py:274>' Fld /tmp/lal-23.0.0/ada/ast.py:274
      if Item_4 = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Fld := Libadalang.Implementation.Image (Self => Item_4);
--# expr-done 113
         declare
            Item_To_Append : constant String_Type := Fld;
         begin
               Inc_Ref (Item_To_Append);
            String_Type_Vectors.Append (Map_Result_Vec, Item_To_Append);
         end;
   --# end
      Finalizer_Scope_1744;
         end loop;
      end;
         Map_Result := Create_String_Type_Array
           (Items_Count => Natural (String_Type_Vectors.Length (Map_Result_Vec)));
         for I in Map_Result.Items'Range loop
            Map_Result.Items (I) := String_Type_Vectors.Get
              (Map_Result_Vec,
               I + String_Type_Vectors.Index_Type'First - Map_Result.Items'First);
         end loop;
         String_Type_Vectors.Destroy (Map_Result_Vec);
      end;
--# expr-done 114
Join_Result := Join_Strings (Sep, Map_Result); 
--# expr-done 115
Let_Result := Join_Result; Inc_Ref (Let_Result);
--# end
--# expr-done 116
      Property_Result := Let_Result;
         Inc_Ref (Property_Result);
   --# end
      Finalizer_Scope_12;
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when Exc : Precondition_Failure | Property_Error =>
               Finalizer_Scope_12;
               Finalizer_Scope_1744;
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Ada_Node_P_Sym_Join;
--# end
--# property-start AdaNode.enclosing_compilation_unit /tmp/lal-23.0.0/ada/ast.py:278
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Enclosing_Compilation_Unit
  (Node : Bare_Ada_Node
  )
   return Bare_Compilation_Unit
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   --# bind self Self
   Property_Result : Bare_Compilation_Unit;
            procedure Finalizer_Scope_13 with Inline_Always;
      Node_Parents : Bare_Ada_Node_Array_Access;
Is_A : Boolean;
Find_Result : Bare_Ada_Node;
Cast_Result : Bare_Compilation_Unit;
Let_Result : Bare_Compilation_Unit;
            procedure Finalizer_Scope_13 is
            begin
                     Dec_Ref (Node_Parents);
            end Finalizer_Scope_13;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
   --# scope-start
      --# expr-start 121 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 120 '<Cast to CompilationUnit at ast.py:288>' Cast_Result /tmp/lal-23.0.0/ada/ast.py:288
--# expr-start 119 '<Find at ast.py:286>' Find_Result /tmp/lal-23.0.0/ada/ast.py:286
   --# expr-start 117 '<FieldAccess for parents at ast.py:286>' Node_Parents /tmp/lal-23.0.0/ada/ast.py:286
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Node_Parents := Parents (Self, True); 
--# expr-done 117
      Find_Result := No_Bare_Ada_Node;
      declare
         Collection : constant Bare_Ada_Node_Array_Access := Node_Parents;
      begin
         for Item_5 of
               Collection.Items
         loop
   --# scope-start
               --# bind n Item_5
      --# expr-start 118 '<IsA CompilationUnit at ast.py:287>' Is_A /tmp/lal-23.0.0/ada/ast.py:287
Is_A := Item_5 /= null 
and then Item_5.Kind in Ada_Compilation_Unit_Range; 
--# expr-done 118
      if Is_A then
         Find_Result := Item_5;
      end if;
   --# end
               exit when Is_A;
         end loop;
      end;
--# expr-done 119
   if Find_Result = null
      or else Find_Result.Kind in Ada_Compilation_Unit_Range
   then
      Cast_Result := Find_Result;
   else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "invalid object cast");
   end if;
--# expr-done 120
Let_Result := Cast_Result; 
--# end
--# expr-done 121
      Property_Result := Let_Result;
   --# end
      Finalizer_Scope_13;
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when Exc : Precondition_Failure | Property_Error =>
               Finalizer_Scope_13;
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Ada_Node_P_Enclosing_Compilation_Unit;
--# end
--# property-start AdaNode.is_children_env /tmp/lal-23.0.0/ada/ast.py:291
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Is_Children_Env
  (Node : Bare_Ada_Node
      ; Parent : Lexical_Env
      ; Current_Env : Lexical_Env
  )
   return Boolean
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   --# bind self Self
   --# bind parent Parent
   --# bind current_env Current_Env
   Property_Result : Boolean;
            procedure Finalizer_Scope_14 with Inline_Always;
      Is_Equal : Boolean;
Is_Equal_1 : Boolean;
Is_Equal_2 : Boolean;
Env_Parent : Lexical_Env := Empty_Env;
Fld : Boolean;
If_Result : Boolean;
If_Result_1 : Boolean;
If_Result_2 : Boolean;
Let_Result : Boolean;
            procedure Finalizer_Scope_14 is
            begin
                     Dec_Ref (Env_Parent);
            end Finalizer_Scope_14;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
   --# scope-start
      --# expr-start 128 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 127 '<Cond at ast.py:296>' If_Result_2 /tmp/lal-23.0.0/ada/ast.py:296
--# expr-start 122 '<Eq at ast.py:297>' Is_Equal /tmp/lal-23.0.0/ada/ast.py:297
Is_Equal := Equivalent (Parent, Empty_Env); 
--# expr-done 122
if Is_Equal then
   If_Result_2 := False;
else
--# expr-start 123 '<Eq at ast.py:298>' Is_Equal_1 /tmp/lal-23.0.0/ada/ast.py:298
Is_Equal_1 := Equivalent (Current_Env, Parent); 
--# expr-done 123
if Is_Equal_1 then
   If_Result_1 := True;
else
--# expr-start 124 '<IsNull at ast.py:299>' Is_Equal_2 /tmp/lal-23.0.0/ada/ast.py:299
Is_Equal_2 := Equivalent (Current_Env, Empty_Env); 
--# expr-done 124
if Is_Equal_2 then
   If_Result := False;
else
   --# expr-start 126 '<FieldAccess for is_children_env at ast.py:300>' Fld /tmp/lal-23.0.0/ada/ast.py:300
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# expr-start 125 '<env_parent at ast.py:300>' Env_Parent /tmp/lal-23.0.0/ada/ast.py:300
Env_Parent := AST_Envs.Parent (Current_Env); 
--# expr-done 125
--# property-call-start AdaNode.is_children_env
Fld := Libadalang.Implementation.Ada_Node_P_Is_Children_Env (Node => Self, Parent => Parent, Current_Env => Env_Parent);
--# end
--# expr-done 126
   If_Result := Fld;
end if;
   If_Result_1 := If_Result;
end if;
   If_Result_2 := If_Result_1;
end if;
--# expr-done 127
Let_Result := If_Result_2; 
--# end
--# expr-done 128
      Property_Result := Let_Result;
   --# end
      Finalizer_Scope_14;
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when Exc : Precondition_Failure | Property_Error =>
               Finalizer_Scope_14;
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Ada_Node_P_Is_Children_Env;
--# end
--# property-start AdaNode.without_md /tmp/lal-23.0.0/ada/ast.py:304
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Without_Md
  (Node : Bare_Ada_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )
   return Internal_Entity
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   Ent : Internal_Entity :=
     Internal_Entity'(Node => Self, Info => E_Info);
   --# bind entity Ent
   Property_Result : Internal_Entity;
      Fld : Bare_Ada_Node;
Fld_1 : Internal_Entity_Info;
Fld_2 : Env_Rebindings;
Fld_3 : Internal_Entity_Info;
Fld_4 : Boolean;
New_Struct : Internal_Entity_Info;
New_Struct_1 : Internal_Entity;
Let_Result : Internal_Entity;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
   --# scope-start
      --# expr-start 136 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 135 '<New AdaNode.entity at ???>' New_Struct_1 None
--# expr-start 133 '<New EntityInfo at ast.py:309>' New_Struct /tmp/lal-23.0.0/ada/ast.py:309
--# expr-start 130 '<FieldAccess for from_rebound at ast.py:312>' Fld_4 /tmp/lal-23.0.0/ada/ast.py:312
--# expr-start 129 '<FieldAccess for info at ast.py:312>' Fld_3 /tmp/lal-23.0.0/ada/ast.py:312
Fld_3 := Ent.Info;
--# expr-done 129
Fld_4 := Fld_3.From_Rebound;
--# expr-done 130
--# expr-start 132 '<FieldAccess for rebindings at ast.py:310>' Fld_2 /tmp/lal-23.0.0/ada/ast.py:310
--# expr-start 131 '<FieldAccess for info at ast.py:310>' Fld_1 /tmp/lal-23.0.0/ada/ast.py:310
Fld_1 := Ent.Info;
--# expr-done 131
Fld_2 := Fld_1.Rebindings;
--# expr-done 132
New_Struct := (From_Rebound => Fld_4, Md => No_Metadata, Rebindings => Fld_2); 
--# expr-done 133
--# expr-start 134 '<FieldAccess for node at ast.py:309>' Fld /tmp/lal-23.0.0/ada/ast.py:309
Fld := Ent.Node;
--# expr-done 134
New_Struct_1 := (Info => New_Struct, Node => Fld); 
--# expr-done 135
Let_Result := New_Struct_1; 
--# end
--# expr-done 136
      Property_Result := Let_Result;
   --# end
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Ada_Node_P_Without_Md;
--# end
--# property-start AdaNode.get_uninstantiated_node /tmp/lal-23.0.0/ada/ast.py:317
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Get_Uninstantiated_Node
  (Node : Bare_Ada_Node
  )
   return Internal_Entity
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   --# bind self Self
   Property_Result : Internal_Entity;
      As_Entity : Internal_Entity;
Is_Null : Boolean;
If_Result : Internal_Entity;
Let_Result : Internal_Entity;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
   --# scope-start
      --# expr-start 138 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 137 '<as_bare_entity at ast.py:323>' If_Result /tmp/lal-23.0.0/ada/ast.py:323
Is_Null := Self = null; 
if Is_Null then
   If_Result := No_Entity;
else
As_Entity := (Info => No_Entity_Info, Node => Self); 
   If_Result := As_Entity;
end if;
--# expr-done 137
Let_Result := If_Result; 
--# end
--# expr-done 138
      Property_Result := Let_Result;
   --# end
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Ada_Node_P_Get_Uninstantiated_Node;
--# end
--# property-start AdaNode.complete /tmp/lal-23.0.0/ada/ast.py:327
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Complete
  (Node : Bare_Ada_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )
   return Internal_Completion_Item_Iterator_Access
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   Ent : Internal_Entity :=
     Internal_Entity'(Node => Self, Info => E_Info);
   --# bind entity Ent
   Property_Result : Internal_Completion_Item_Iterator_Access;
            procedure Finalizer_Scope_17 with Inline_Always;
      Fld : Bare_Ada_Node;
Bound_Origin : Bare_Ada_Node;
Fld_1 : Internal_Completion_Item_Array_Access;
Fld_2 : Internal_Entity_Basic_Decl;
Cast_Result : Internal_Entity_Synthetic_Subp_Decl;
Is_Null : Boolean;
Map_Result : Internal_Completion_Item_Array_Access;
Iter : Internal_Completion_Item_Iterator_Access;
Dyn_Var_Bind_Result : Internal_Completion_Item_Iterator_Access;
Let_Result : Internal_Completion_Item_Iterator_Access;
            procedure Finalizer_Scope_17 is
            begin
                     Dec_Ref (Fld_1);
                     Dec_Ref (Map_Result);
                     Dec_Ref (Iter);
                     Dec_Ref (Dyn_Var_Bind_Result);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_17;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
      if Self /= null then
         Reset_Caches (Self.Unit);
            Populate_Lexical_Env (Self.Unit);
      end if;
   --# scope-start
      --# expr-start 147 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 146 '<bind at ast.py:331>' Dyn_Var_Bind_Result /tmp/lal-23.0.0/ada/ast.py:331
--# expr-start 139 '<FieldAccess for origin_node at ast.py:332>' Fld /tmp/lal-23.0.0/ada/ast.py:332
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start AdaNode.origin_node
Fld := Libadalang.Implementation.Ada_Node_P_Origin_Node (Node => Self);
--# end
--# expr-done 139
Bound_Origin := Fld; 
--# expr-start 145 '<FieldAccess for to_iterator at ast.py:338>' Iter /tmp/lal-23.0.0/ada/ast.py:338
--# expr-start 144 '<Filter at ast.py:333>' Map_Result /tmp/lal-23.0.0/ada/ast.py:333
   --# expr-start 140 '<FieldAccess for complete_items at ast.py:333>' Fld_1 /tmp/lal-23.0.0/ada/ast.py:333
      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start '[dispatcher]AdaNode.complete_items'
Fld_1 := Libadalang.Implementation.Dispatcher_Ada_Node_P_Complete_Items (Node => Ent.Node, Origin => Bound_Origin, E_Info => Ent.Info);
--# end
--# expr-done 140
      declare
         Map_Result_Vec : Internal_Completion_Item_Vectors.Vector;
      begin
      declare
         Collection : constant Internal_Completion_Item_Array_Access := Fld_1;
      begin
         for Item_6 of
               Collection.Items
         loop
   --# scope-start
               --# bind n Item_6
         --# expr-start 143 '<IsNull at ast.py:337>' Is_Null /tmp/lal-23.0.0/ada/ast.py:337
--# expr-start 142 '<Cast to SyntheticSubpDecl at ast.py:337>' Cast_Result /tmp/lal-23.0.0/ada/ast.py:337
--# expr-start 141 '<FieldAccess for decl at ast.py:337>' Fld_2 /tmp/lal-23.0.0/ada/ast.py:337
Fld_2 := Item_6.Decl;
--# expr-done 141
   if Fld_2.Node = null
      or else Fld_2.Node.Kind in Ada_Synthetic_Subp_Decl_Range
   then
      Cast_Result := Create_Internal_Entity_Synthetic_Subp_Decl
        (Node => Fld_2.Node,
         Info => Fld_2.Info);
   else
         Cast_Result := No_Entity_Synthetic_Subp_Decl;
   end if;
--# expr-done 142
Is_Null := Cast_Result.Node = null; 
--# expr-done 143
         if Is_Null then
         declare
            Item_To_Append : constant Internal_Completion_Item := Item_6;
         begin
            Internal_Completion_Item_Vectors.Append (Map_Result_Vec, Item_To_Append);
         end;
         end if;
   --# end
         end loop;
      end;
         Map_Result := Create_Internal_Completion_Item_Array
           (Items_Count => Natural (Internal_Completion_Item_Vectors.Length (Map_Result_Vec)));
         for I in Map_Result.Items'Range loop
            Map_Result.Items (I) := Internal_Completion_Item_Vectors.Get
              (Map_Result_Vec,
               I + Internal_Completion_Item_Vectors.Index_Type'First - Map_Result.Items'First);
         end loop;
         Internal_Completion_Item_Vectors.Destroy (Map_Result_Vec);
      end;
--# expr-done 144
Iter := To_Iterator (Map_Result, Self.Unit.Context); 
--# expr-done 145
Dyn_Var_Bind_Result := Iter; Inc_Ref (Dyn_Var_Bind_Result);
--# expr-done 146
Let_Result := Dyn_Var_Bind_Result; Inc_Ref (Let_Result);
--# end
--# expr-done 147
      Property_Result := Let_Result;
         Inc_Ref (Property_Result);
   --# end
      Finalizer_Scope_17;
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when Exc : Precondition_Failure | Property_Error =>
               Finalizer_Scope_17;
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Ada_Node_P_Complete;
--# end
--# property-start '[dispatcher]AdaNode.complete_items' dispatcher
pragma Warnings (Off, "is not referenced");
 function Dispatcher_Ada_Node_P_Complete_Items
  (Node : Bare_Ada_Node
      ; Origin : Bare_Ada_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )
   return Internal_Completion_Item_Array_Access
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   --# bind self Self
   --# bind origin Origin
   Property_Result : Internal_Completion_Item_Array_Access;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
      if Self /= null then
         Reset_Caches (Self.Unit);
      end if;
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dispatching on null node");
      end if;
      case Ada_Ada_Node (Self.Kind) is
               when Ada_Abort_Absent .. Ada_Abstract_State_Decl_List | Ada_Constraint_List .. Ada_Discrete_Subtype_Name | Ada_End_Name .. Ada_Char_Literal | Ada_Op_Abs .. Ada_Enum_Lit_Synth_Type_Expr | Ada_Synthetic_Type_Expr .. Ada_With_Private_Present =>
                  --# property-call-start AdaNode.complete_items
                  Property_Result := Ada_Node_P_Complete_Items
                    (Self
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Alternatives_List =>
                  --# property-call-start AlternativesList.complete_items
                  Property_Result := Alternatives_List_P_Complete_Items
                    (Self
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Dotted_Name =>
                  --# property-call-start DottedName.complete_items
                  Property_Result := Dotted_Name_P_Complete_Items
                    (Self
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Identifier =>
                  --# property-call-start Identifier.complete_items
                  Property_Result := Identifier_P_Complete_Items
                    (Self
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Subtype_Indication .. Ada_Discrete_Subtype_Indication =>
                  --# property-call-start SubtypeIndication.complete_items
                  Property_Result := Subtype_Indication_P_Complete_Items
                    (Self
                        , Origin
                        , E_Info
                    );
                  --# end
      end case;
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Dispatcher_Ada_Node_P_Complete_Items;
--# end
--# property-start '[dispatcher]AdaNode.complete_item_weight' dispatcher
pragma Warnings (Off, "is not referenced");
 function Dispatcher_Ada_Node_P_Complete_Item_Weight
  (Node : Bare_Ada_Node
      ; Item : Internal_Entity_Basic_Decl
      ; Origin : Bare_Ada_Node
  )
   return Integer
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   --# bind self Self
   --# bind item Item
   --# bind origin Origin
   Property_Result : Integer;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
      if Self /= null then
         Reset_Caches (Self.Unit);
      end if;
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dispatching on null node");
      end if;
      case Ada_Ada_Node (Self.Kind) is
               when Ada_Abort_Absent .. Ada_Abort_Stmt | Ada_Call_Stmt .. Ada_With_Private_Present =>
                  --# property-call-start AdaNode.complete_item_weight
                  Property_Result := Ada_Node_P_Complete_Item_Weight
                    (Self
                        , Item
                        , Origin
                    );
                  --# end
               when Ada_Assign_Stmt =>
                  --# property-call-start AssignStmt.complete_item_weight
                  Property_Result := Assign_Stmt_P_Complete_Item_Weight
                    (Self
                        , Item
                        , Origin
                    );
                  --# end
      end case;
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Dispatcher_Ada_Node_P_Complete_Item_Weight;
--# end
--# property-start AdaNode.valid_keywords /tmp/lal-23.0.0/ada/ast.py:371
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Valid_Keywords
  (Node : Bare_Ada_Node
  )
   return Symbol_Type_Array_Access
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   --# bind self Self
   Property_Result : Symbol_Type_Array_Access;
            procedure Finalizer_Scope_22 with Inline_Always;
      Sym : Symbol_Type;
Sym_1 : Symbol_Type;
Sym_2 : Symbol_Type;
Sym_3 : Symbol_Type;
Sym_4 : Symbol_Type;
Sym_5 : Symbol_Type;
Sym_6 : Symbol_Type;
Sym_7 : Symbol_Type;
Sym_8 : Symbol_Type;
Sym_9 : Symbol_Type;
Sym_10 : Symbol_Type;
Sym_11 : Symbol_Type;
Sym_12 : Symbol_Type;
Sym_13 : Symbol_Type;
Sym_14 : Symbol_Type;
Sym_15 : Symbol_Type;
Sym_16 : Symbol_Type;
Sym_17 : Symbol_Type;
Sym_18 : Symbol_Type;
Sym_19 : Symbol_Type;
Sym_20 : Symbol_Type;
Sym_21 : Symbol_Type;
Sym_22 : Symbol_Type;
Sym_23 : Symbol_Type;
Sym_24 : Symbol_Type;
Sym_25 : Symbol_Type;
Sym_26 : Symbol_Type;
Sym_27 : Symbol_Type;
Sym_28 : Symbol_Type;
Sym_29 : Symbol_Type;
Sym_30 : Symbol_Type;
Sym_31 : Symbol_Type;
Sym_32 : Symbol_Type;
Sym_33 : Symbol_Type;
Sym_34 : Symbol_Type;
Sym_35 : Symbol_Type;
Sym_36 : Symbol_Type;
Sym_37 : Symbol_Type;
Sym_38 : Symbol_Type;
Sym_39 : Symbol_Type;
Sym_40 : Symbol_Type;
Sym_41 : Symbol_Type;
Sym_42 : Symbol_Type;
Sym_43 : Symbol_Type;
Sym_44 : Symbol_Type;
Sym_45 : Symbol_Type;
Sym_46 : Symbol_Type;
Sym_47 : Symbol_Type;
Sym_48 : Symbol_Type;
Sym_49 : Symbol_Type;
Sym_50 : Symbol_Type;
Sym_51 : Symbol_Type;
Sym_52 : Symbol_Type;
Sym_53 : Symbol_Type;
Sym_54 : Symbol_Type;
Sym_55 : Symbol_Type;
Sym_56 : Symbol_Type;
Sym_57 : Symbol_Type;
Sym_58 : Symbol_Type;
Sym_59 : Symbol_Type;
Sym_60 : Symbol_Type;
Sym_61 : Symbol_Type;
Sym_62 : Symbol_Type;
Sym_63 : Symbol_Type;
Sym_64 : Symbol_Type;
Sym_65 : Symbol_Type;
Sym_66 : Symbol_Type;
Sym_67 : Symbol_Type;
Sym_68 : Symbol_Type;
Sym_69 : Symbol_Type;
Sym_70 : Symbol_Type;
Sym_71 : Symbol_Type;
Sym_72 : Symbol_Type;
Array_Lit : Symbol_Type_Array_Access;
Let_Result : Symbol_Type_Array_Access;
            procedure Finalizer_Scope_22 is
            begin
                     Dec_Ref (Array_Lit);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_22;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
   --# scope-start
      --# expr-start 222 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 221 '<ArrayLiteral at ast.py:379>' Array_Lit /tmp/lal-23.0.0/ada/ast.py:379
--# expr-start 148 '<Symbol abort at ???>' Sym None
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Abort); 
--# expr-done 148
--# expr-start 149 '<Symbol abs at ???>' Sym_1 None
Sym_1 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Abs_260); 
--# expr-done 149
--# expr-start 150 '<Symbol abstract at ???>' Sym_2 None
Sym_2 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Abstract); 
--# expr-done 150
--# expr-start 151 '<Symbol accept at ???>' Sym_3 None
Sym_3 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Accept); 
--# expr-done 151
--# expr-start 152 '<Symbol access at ???>' Sym_4 None
Sym_4 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Access_263); 
--# expr-done 152
--# expr-start 153 '<Symbol aliased at ???>' Sym_5 None
Sym_5 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Aliased); 
--# expr-done 153
--# expr-start 154 '<Symbol all at ???>' Sym_6 None
Sym_6 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_All); 
--# expr-done 154
--# expr-start 155 '<Symbol and at ???>' Sym_7 None
Sym_7 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_And_268); 
--# expr-done 155
--# expr-start 156 '<Symbol array at ???>' Sym_8 None
Sym_8 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Array); 
--# expr-done 156
--# expr-start 157 '<Symbol at at ???>' Sym_9 None
Sym_9 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_At); 
--# expr-done 157
--# expr-start 158 '<Symbol begin at ???>' Sym_10 None
Sym_10 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Begin); 
--# expr-done 158
--# expr-start 159 '<Symbol body at ???>' Sym_11 None
Sym_11 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Body); 
--# expr-done 159
--# expr-start 160 '<Symbol case at ???>' Sym_12 None
Sym_12 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Case); 
--# expr-done 160
--# expr-start 161 '<Symbol constant at ???>' Sym_13 None
Sym_13 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Constant); 
--# expr-done 161
--# expr-start 162 '<Symbol declare at ???>' Sym_14 None
Sym_14 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Declare); 
--# expr-done 162
--# expr-start 163 '<Symbol delay at ???>' Sym_15 None
Sym_15 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Delay); 
--# expr-done 163
--# expr-start 164 '<Symbol delta at ???>' Sym_16 None
Sym_16 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Delta_282); 
--# expr-done 164
--# expr-start 165 '<Symbol digits at ???>' Sym_17 None
Sym_17 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Digits_283); 
--# expr-done 165
--# expr-start 166 '<Symbol do at ???>' Sym_18 None
Sym_18 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Do); 
--# expr-done 166
--# expr-start 167 '<Symbol else at ???>' Sym_19 None
Sym_19 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Else); 
--# expr-done 167
--# expr-start 168 '<Symbol elsif at ???>' Sym_20 None
Sym_20 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Elsif); 
--# expr-done 168
--# expr-start 169 '<Symbol end at ???>' Sym_21 None
Sym_21 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_End); 
--# expr-done 169
--# expr-start 170 '<Symbol entry at ???>' Sym_22 None
Sym_22 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Entry); 
--# expr-done 170
--# expr-start 171 '<Symbol exception at ???>' Sym_23 None
Sym_23 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Exception); 
--# expr-done 171
--# expr-start 172 '<Symbol exit at ???>' Sym_24 None
Sym_24 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Exit); 
--# expr-done 172
--# expr-start 173 '<Symbol for at ???>' Sym_25 None
Sym_25 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_For); 
--# expr-done 173
--# expr-start 174 '<Symbol function at ???>' Sym_26 None
Sym_26 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Function); 
--# expr-done 174
--# expr-start 175 '<Symbol generic at ???>' Sym_27 None
Sym_27 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Generic); 
--# expr-done 175
--# expr-start 176 '<Symbol goto at ???>' Sym_28 None
Sym_28 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Goto); 
--# expr-done 176
--# expr-start 177 '<Symbol if at ???>' Sym_29 None
Sym_29 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_If); 
--# expr-done 177
--# expr-start 178 '<Symbol in at ???>' Sym_30 None
Sym_30 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_In); 
--# expr-done 178
--# expr-start 179 '<Symbol interface at ???>' Sym_31 None
Sym_31 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Interface_306); 
--# expr-done 179
--# expr-start 180 '<Symbol is at ???>' Sym_32 None
Sym_32 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Is); 
--# expr-done 180
--# expr-start 181 '<Symbol limited at ???>' Sym_33 None
Sym_33 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Limited); 
--# expr-done 181
--# expr-start 182 '<Symbol loop at ???>' Sym_34 None
Sym_34 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Loop); 
--# expr-done 182
--# expr-start 183 '<Symbol mod at ???>' Sym_35 None
Sym_35 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Mod_315); 
--# expr-done 183
--# expr-start 184 '<Symbol new at ???>' Sym_36 None
Sym_36 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_New); 
--# expr-done 184
--# expr-start 185 '<Symbol not at ???>' Sym_37 None
Sym_37 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Not_317); 
--# expr-done 185
--# expr-start 186 '<Symbol null at ???>' Sym_38 None
Sym_38 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Null); 
--# expr-done 186
--# expr-start 187 '<Symbol others at ???>' Sym_39 None
Sym_39 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Others); 
--# expr-done 187
--# expr-start 188 '<Symbol out at ???>' Sym_40 None
Sym_40 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Out); 
--# expr-done 188
--# expr-start 189 '<Symbol of at ???>' Sym_41 None
Sym_41 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Of); 
--# expr-done 189
--# expr-start 190 '<Symbol or at ???>' Sym_42 None
Sym_42 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Or_320); 
--# expr-done 190
--# expr-start 191 '<Symbol overriding at ???>' Sym_43 None
Sym_43 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Overriding); 
--# expr-done 191
--# expr-start 192 '<Symbol package at ???>' Sym_44 None
Sym_44 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Package); 
--# expr-done 192
--# expr-start 193 '<Symbol pragma at ???>' Sym_45 None
Sym_45 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Pragma); 
--# expr-done 193
--# expr-start 194 '<Symbol private at ???>' Sym_46 None
Sym_46 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Private); 
--# expr-done 194
--# expr-start 195 '<Symbol procedure at ???>' Sym_47 None
Sym_47 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Procedure); 
--# expr-done 195
--# expr-start 196 '<Symbol protected at ???>' Sym_48 None
Sym_48 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Protected); 
--# expr-done 196
--# expr-start 197 '<Symbol raise at ???>' Sym_49 None
Sym_49 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Raise); 
--# expr-done 197
--# expr-start 198 '<Symbol range at ???>' Sym_50 None
Sym_50 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Range_334); 
--# expr-done 198
--# expr-start 199 '<Symbol record at ???>' Sym_51 None
Sym_51 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Record); 
--# expr-done 199
--# expr-start 200 '<Symbol rem at ???>' Sym_52 None
Sym_52 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Rem_337); 
--# expr-done 200
--# expr-start 201 '<Symbol renames at ???>' Sym_53 None
Sym_53 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Renames); 
--# expr-done 201
--# expr-start 202 '<Symbol requeue at ???>' Sym_54 None
Sym_54 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Requeue); 
--# expr-done 202
--# expr-start 203 '<Symbol return at ???>' Sym_55 None
Sym_55 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Return); 
--# expr-done 203
--# expr-start 204 '<Symbol reverse at ???>' Sym_56 None
Sym_56 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Reverse); 
--# expr-done 204
--# expr-start 205 '<Symbol select at ???>' Sym_57 None
Sym_57 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Select); 
--# expr-done 205
--# expr-start 206 '<Symbol separate at ???>' Sym_58 None
Sym_58 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Separate); 
--# expr-done 206
--# expr-start 207 '<Symbol some at ???>' Sym_59 None
Sym_59 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Some); 
--# expr-done 207
--# expr-start 208 '<Symbol subtype at ???>' Sym_60 None
Sym_60 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Subtype); 
--# expr-done 208
--# expr-start 209 '<Symbol synchronized at ???>' Sym_61 None
Sym_61 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Synchronized); 
--# expr-done 209
--# expr-start 210 '<Symbol tagged at ???>' Sym_62 None
Sym_62 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Tagged); 
--# expr-done 210
--# expr-start 211 '<Symbol task at ???>' Sym_63 None
Sym_63 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Task); 
--# expr-done 211
--# expr-start 212 '<Symbol terminate at ???>' Sym_64 None
Sym_64 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Terminate); 
--# expr-done 212
--# expr-start 213 '<Symbol then at ???>' Sym_65 None
Sym_65 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Then); 
--# expr-done 213
--# expr-start 214 '<Symbol type at ???>' Sym_66 None
Sym_66 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Type); 
--# expr-done 214
--# expr-start 215 '<Symbol until at ???>' Sym_67 None
Sym_67 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Until); 
--# expr-done 215
--# expr-start 216 '<Symbol use at ???>' Sym_68 None
Sym_68 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Use); 
--# expr-done 216
--# expr-start 217 '<Symbol when at ???>' Sym_69 None
Sym_69 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_When); 
--# expr-done 217
--# expr-start 218 '<Symbol while at ???>' Sym_70 None
Sym_70 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_While); 
--# expr-done 218
--# expr-start 219 '<Symbol with at ???>' Sym_71 None
Sym_71 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_With); 
--# expr-done 219
--# expr-start 220 '<Symbol xor at ???>' Sym_72 None
Sym_72 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Xor_377); 
--# expr-done 220
Array_Lit := Create_Symbol_Type_Array (Internal_Symbol_Type_Array'(1 => Sym, 2 => Sym_1, 3 => Sym_2, 4 => Sym_3, 5 => Sym_4, 6 => Sym_5, 7 => Sym_6, 8 => Sym_7, 9 => Sym_8, 10 => Sym_9, 11 => Sym_10, 12 => Sym_11, 13 => Sym_12, 14 => Sym_13, 15 => Sym_14, 16 => Sym_15, 17 => Sym_16, 18 => Sym_17, 19 => Sym_18, 20 => Sym_19, 21 => Sym_20, 22 => Sym_21, 23 => Sym_22, 24 => Sym_23, 25 => Sym_24, 26 => Sym_25, 27 => Sym_26, 28 => Sym_27, 29 => Sym_28, 30 => Sym_29, 31 => Sym_30, 32 => Sym_31, 33 => Sym_32, 34 => Sym_33, 35 => Sym_34, 36 => Sym_35, 37 => Sym_36, 38 => Sym_37, 39 => Sym_38, 40 => Sym_39, 41 => Sym_40, 42 => Sym_41, 43 => Sym_42, 44 => Sym_43, 45 => Sym_44, 46 => Sym_45, 47 => Sym_46, 48 => Sym_47, 49 => Sym_48, 50 => Sym_49, 51 => Sym_50, 52 => Sym_51, 53 => Sym_52, 54 => Sym_53, 55 => Sym_54, 56 => Sym_55, 57 => Sym_56, 58 => Sym_57, 59 => Sym_58, 60 => Sym_59, 61 => Sym_60, 62 => Sym_61, 63 => Sym_62, 64 => Sym_63, 65 => Sym_64, 66 => Sym_65, 67 => Sym_66, 68 => Sym_67, 69 => Sym_68, 70 => Sym_69, 71 => Sym_70, 72 => Sym_71, 73 => Sym_72)); 
--# expr-done 221
Let_Result := Array_Lit; Inc_Ref (Let_Result);
--# end
--# expr-done 222
      Property_Result := Let_Result;
         Inc_Ref (Property_Result);
   --# end
      Finalizer_Scope_22;
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when Exc : Precondition_Failure | Property_Error =>
               Finalizer_Scope_22;
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Ada_Node_P_Valid_Keywords;
--# end
--# property-start AdaNode.generic_instantiations /tmp/lal-23.0.0/ada/ast.py:394
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Generic_Instantiations
  (Node : Bare_Ada_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )
   return Internal_Entity_Generic_Instantiation_Array_Access
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   Ent : Internal_Entity :=
     Internal_Entity'(Node => Self, Info => E_Info);
   --# bind entity Ent
   Property_Result : Internal_Entity_Generic_Instantiation_Array_Access;
            procedure Finalizer_Scope_23 with Inline_Always;
      Fld : Internal_Entity_Info;
Fld_1 : Env_Rebindings;
Fld_2 : Internal_Entity_Generic_Instantiation_Array_Access;
Let_Result : Internal_Entity_Generic_Instantiation_Array_Access;
            procedure Finalizer_Scope_23 is
            begin
                     Dec_Ref (Fld_2);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_23;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
   --# scope-start
      --# expr-start 226 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 225 '<FieldAccess for generic_instantiations_internal at ast.py:400>' Fld_2 /tmp/lal-23.0.0/ada/ast.py:400
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# expr-start 224 '<FieldAccess for rebindings at ast.py:400>' Fld_1 /tmp/lal-23.0.0/ada/ast.py:400
--# expr-start 223 '<FieldAccess for info at ast.py:400>' Fld /tmp/lal-23.0.0/ada/ast.py:400
Fld := Ent.Info;
--# expr-done 223
Fld_1 := Fld.Rebindings;
--# expr-done 224
--# property-call-start AdaNode.generic_instantiations_internal
Fld_2 := Libadalang.Implementation.Ada_Node_P_Generic_Instantiations_Internal (Node => Self, R => Fld_1);
--# end
--# expr-done 225
Let_Result := Fld_2; Inc_Ref (Let_Result);
--# end
--# expr-done 226
      Property_Result := Let_Result;
         Inc_Ref (Property_Result);
   --# end
      Finalizer_Scope_23;
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when Exc : Precondition_Failure | Property_Error =>
               Finalizer_Scope_23;
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Ada_Node_P_Generic_Instantiations;
--# end
--# property-start AdaNode.generic_instantiations_internal /tmp/lal-23.0.0/ada/ast.py:403
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Generic_Instantiations_Internal
  (Node : Bare_Ada_Node
      ; R : Env_Rebindings
  )
   return Internal_Entity_Generic_Instantiation_Array_Access
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   --# bind self Self
   --# bind r R
   Property_Result : Internal_Entity_Generic_Instantiation_Array_Access;
            procedure Finalizer_Scope_24 with Inline_Always;
      Head : Internal_Entity_Generic_Instantiation;
Tail : Internal_Entity_Generic_Instantiation_Array_Access;
Fld : Lexical_Env := Empty_Env;
Env_Node : Bare_Ada_Node;
Cast_Result : Bare_Generic_Instantiation;
As_Entity : Internal_Entity_Generic_Instantiation;
Is_Null : Boolean;
If_Result : Internal_Entity_Generic_Instantiation;
Fld_1 : Env_Rebindings;
Fld_2 : Internal_Entity_Generic_Instantiation_Array_Access;
Singleton : Internal_Entity_Generic_Instantiation_Array_Access;
Concat_Result : Internal_Entity_Generic_Instantiation_Array_Access;
Let_Result : Internal_Entity_Generic_Instantiation_Array_Access;
Is_Equal : Boolean;
If_Result_1 : Internal_Entity_Generic_Instantiation_Array_Access;
Let_Result_1 : Internal_Entity_Generic_Instantiation_Array_Access;
            procedure Finalizer_Scope_24 is
            begin
                     Dec_Ref (Fld);
                     Dec_Ref (Fld_2);
                     Dec_Ref (Tail);
                     Dec_Ref (Singleton);
                     Dec_Ref (Concat_Result);
                     Dec_Ref (Let_Result);
                     Dec_Ref (If_Result_1);
                     Dec_Ref (Let_Result_1);
            end Finalizer_Scope_24;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
   --# scope-start
      --# expr-start 237 '<Block at ???>' Let_Result_1 None
--# scope-start
--# expr-start 236 '<If at ast.py:404>' If_Result_1 /tmp/lal-23.0.0/ada/ast.py:404
--# expr-start 227 '<Eq at ast.py:405>' Is_Equal /tmp/lal-23.0.0/ada/ast.py:405
Is_Equal := R = null; 
--# expr-done 227
if Is_Equal then
   If_Result_1 := No_Internal_Entity_Generic_Instantiation_Array_Type;
else
   --# expr-start 235 '<Let at ast.py:408>' Let_Result /tmp/lal-23.0.0/ada/ast.py:408
--# scope-start
--# expr-start 231 '<as_bare_entity at ast.py:410>' If_Result /tmp/lal-23.0.0/ada/ast.py:410
--# expr-start 230 '<Cast to GenericInstantiation at ast.py:410>' Cast_Result /tmp/lal-23.0.0/ada/ast.py:410
--# expr-start 229 '<env_node at ast.py:409>' Env_Node /tmp/lal-23.0.0/ada/ast.py:409
--# expr-start 228 '<FieldAccess for new_env at ast.py:409>' Fld /tmp/lal-23.0.0/ada/ast.py:409
      if R = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Fld := R.New_Env;
Inc_Ref (Fld);
--# expr-done 228
Env_Node := AST_Envs.Env_Node (Fld); 
--# expr-done 229
   if Env_Node = null
      or else Env_Node.Kind in Ada_Generic_Instantiation
   then
      Cast_Result := Env_Node;
   else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "invalid object cast");
   end if;
--# expr-done 230
Is_Null := Cast_Result = null; 
if Is_Null then
   If_Result := No_Entity_Generic_Instantiation;
else
As_Entity := (Info => No_Entity_Info, Node => Cast_Result); 
   If_Result := As_Entity;
end if;
--# expr-done 231
Head := If_Result; 
--# bind head Head
--# expr-start 233 '<FieldAccess for generic_instantiations_internal at ast.py:411>' Fld_2 /tmp/lal-23.0.0/ada/ast.py:411
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# expr-start 232 '<FieldAccess for get_parent at ast.py:411>' Fld_1 /tmp/lal-23.0.0/ada/ast.py:411
      if R = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Fld_1 := R.Parent;
--# expr-done 232
--# property-call-start AdaNode.generic_instantiations_internal
Fld_2 := Libadalang.Implementation.Ada_Node_P_Generic_Instantiations_Internal (Node => Self, R => Fld_1);
--# end
--# expr-done 233
Tail := Fld_2; Inc_Ref (Tail);
--# bind tail Tail
--# expr-start 234 '<Concat at ast.py:412>' Concat_Result /tmp/lal-23.0.0/ada/ast.py:412
                Singleton := Create_Internal_Entity_Generic_Instantiation_Array (Items_Count => 1);
                Singleton.Items (1) := Head;
Concat_Result := Concat (Singleton, Tail); 
--# expr-done 234
Let_Result := Concat_Result; Inc_Ref (Let_Result);
--# end
--# expr-done 235
   If_Result_1 := Let_Result;
end if;
      Inc_Ref (If_Result_1);
--# expr-done 236
Let_Result_1 := If_Result_1; Inc_Ref (Let_Result_1);
--# end
--# expr-done 237
      Property_Result := Let_Result_1;
         Inc_Ref (Property_Result);
   --# end
      Finalizer_Scope_24;
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when Exc : Precondition_Failure | Property_Error =>
               Finalizer_Scope_24;
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Ada_Node_P_Generic_Instantiations_Internal;
--# end
--# property-start AdaNode.remove_rebindings /tmp/lal-23.0.0/ada/ast.py:416
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Remove_Rebindings
  (Node : Bare_Ada_Node
      ; Base : Env_Rebindings
      ; Suffix : Env_Rebindings
  )
   return Env_Rebindings
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   --# bind self Self
   --# bind base Base
   --# bind suffix Suffix
   Property_Result : Env_Rebindings;
            procedure Finalizer_Scope_25 with Inline_Always;
      Is_Equal : Boolean;
Is_Equal_1 : Boolean;
If_Result : Boolean;
Fld : Lexical_Env := Empty_Env;
Fld_1 : Lexical_Env := Empty_Env;
Is_Equal_2 : Boolean;
Fld_2 : Lexical_Env := Empty_Env;
Fld_3 : Lexical_Env := Empty_Env;
Is_Equal_3 : Boolean;
If_Result_1 : Boolean;
Fld_4 : Env_Rebindings;
Fld_5 : Env_Rebindings;
Fld_6 : Env_Rebindings;
If_Result_2 : Env_Rebindings;
If_Result_3 : Env_Rebindings;
Let_Result : Env_Rebindings;
            procedure Finalizer_Scope_25 is
            begin
                     Dec_Ref (Fld);
                     Dec_Ref (Fld_1);
                     Dec_Ref (Fld_2);
                     Dec_Ref (Fld_3);
            end Finalizer_Scope_25;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
   --# scope-start
      --# expr-start 250 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 249 '<Cond at ast.py:421>' If_Result_3 /tmp/lal-23.0.0/ada/ast.py:421
--# expr-start 238 '<IsNull at ast.py:422>' Is_Equal /tmp/lal-23.0.0/ada/ast.py:422
Is_Equal := Base = null; 
--# expr-done 238
if Is_Equal then
   If_Result := True;
else
   --# expr-start 239 '<IsNull at ast.py:422>' Is_Equal_1 /tmp/lal-23.0.0/ada/ast.py:422
Is_Equal_1 := Suffix = null; 
--# expr-done 239
   If_Result := Is_Equal_1;
end if;
if If_Result then
   If_Result_3 := Base;
else
--# expr-start 242 '<Eq at ast.py:425>' Is_Equal_2 /tmp/lal-23.0.0/ada/ast.py:425
--# expr-start 240 '<FieldAccess for old_env at ast.py:425>' Fld /tmp/lal-23.0.0/ada/ast.py:425
      if Base = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Fld := Base.Old_Env;
Inc_Ref (Fld);
--# expr-done 240
--# expr-start 241 '<FieldAccess for old_env at ast.py:425>' Fld_1 /tmp/lal-23.0.0/ada/ast.py:425
      if Suffix = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Fld_1 := Suffix.Old_Env;
Inc_Ref (Fld_1);
--# expr-done 241
Is_Equal_2 := Equivalent (Fld, Fld_1); 
--# expr-done 242
if Is_Equal_2 then
   --# expr-start 245 '<Eq at ast.py:426>' Is_Equal_3 /tmp/lal-23.0.0/ada/ast.py:426
--# expr-start 243 '<FieldAccess for new_env at ast.py:426>' Fld_2 /tmp/lal-23.0.0/ada/ast.py:426
      if Base = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Fld_2 := Base.New_Env;
Inc_Ref (Fld_2);
--# expr-done 243
--# expr-start 244 '<FieldAccess for new_env at ast.py:426>' Fld_3 /tmp/lal-23.0.0/ada/ast.py:426
      if Suffix = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Fld_3 := Suffix.New_Env;
Inc_Ref (Fld_3);
--# expr-done 244
Is_Equal_3 := Equivalent (Fld_2, Fld_3); 
--# expr-done 245
   If_Result_1 := Is_Equal_3;
else
   If_Result_1 := False;
end if;
if If_Result_1 then
   --# expr-start 248 '<FieldAccess for remove_rebindings at ast.py:427>' Fld_6 /tmp/lal-23.0.0/ada/ast.py:427
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# expr-start 246 '<FieldAccess for get_parent at ast.py:427>' Fld_4 /tmp/lal-23.0.0/ada/ast.py:427
      if Base = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Fld_4 := Base.Parent;
--# expr-done 246
--# expr-start 247 '<FieldAccess for get_parent at ast.py:427>' Fld_5 /tmp/lal-23.0.0/ada/ast.py:427
      if Suffix = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Fld_5 := Suffix.Parent;
--# expr-done 247
--# property-call-start AdaNode.remove_rebindings
Fld_6 := Libadalang.Implementation.Ada_Node_P_Remove_Rebindings (Node => Self, Base => Fld_4, Suffix => Fld_5);
--# end
--# expr-done 248
   If_Result_2 := Fld_6;
else
   If_Result_2 := Base;
end if;
   If_Result_3 := If_Result_2;
end if;
--# expr-done 249
Let_Result := If_Result_3; 
--# end
--# expr-done 250
      Property_Result := Let_Result;
   --# end
      Finalizer_Scope_25;
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when Exc : Precondition_Failure | Property_Error =>
               Finalizer_Scope_25;
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Ada_Node_P_Remove_Rebindings;
--# end
--# property-start AdaNode.logic_val /tmp/lal-23.0.0/ada/ast.py:437
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Logic_Val
  (Node : Bare_Ada_Node
      ; From_Node : Internal_Entity
      ; Lvar : Logic_Var
  )
   return Internal_Logic_Val_Result
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   --# bind self Self
   --# bind from_node From_Node
   --# bind lvar Lvar
   Property_Result : Internal_Logic_Val_Result;
      Block_Var : Boolean;
Fld : Boolean;
Is_Logic_Var_Defined : Boolean;
Eq_Solution : Internal_Entity;
If_Result : Internal_Entity;
If_Result_1 : Internal_Entity;
New_Struct : Internal_Logic_Val_Result;
Let_Result : Internal_Logic_Val_Result;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
      if Self /= null then
         Reset_Caches (Self.Unit);
      end if;
   --# scope-start
      --# expr-start 255 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 251 '<FieldAccess for resolve_names_from_closest_entry_point at ast.py:438>' Fld /tmp/lal-23.0.0/ada/ast.py:438
      if From_Node.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start AdaNode.resolve_names_from_closest_entry_point
Fld := Libadalang.Implementation.Ada_Node_P_Resolve_Names_From_Closest_Entry_Point (Node => From_Node.Node, E_Info => From_Node.Info);
--# end
--# expr-done 251
Block_Var := Fld; 
--# bind success Block_Var
--# expr-start 254 '<New LogicValResult at ast.py:440>' New_Struct /tmp/lal-23.0.0/ada/ast.py:440
--# expr-start 253 '<If at ast.py:440>' If_Result_1 /tmp/lal-23.0.0/ada/ast.py:440
if Block_Var then
   --# expr-start 252 '<get_value at ast.py:441>' If_Result /tmp/lal-23.0.0/ada/ast.py:441
Is_Logic_Var_Defined := Entity_Vars.Is_Defined (Lvar); 
if Is_Logic_Var_Defined then
Eq_Solution := Entity_Vars.Get_Value (Lvar); 
   If_Result := Eq_Solution;
else
   If_Result := No_Entity;
end if;
--# expr-done 252
   If_Result_1 := If_Result;
else
   If_Result_1 := No_Entity;
end if;
--# expr-done 253
New_Struct := (Success => Block_Var, Value => If_Result_1); 
--# expr-done 254
Let_Result := New_Struct; 
--# end
--# expr-done 255
      Property_Result := Let_Result;
   --# end
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Ada_Node_P_Logic_Val;
--# end
--# property-start AdaNode.semantic_parent_helper /tmp/lal-23.0.0/ada/ast.py:445
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Semantic_Parent_Helper
  (Node : Bare_Ada_Node
      ; Env : Lexical_Env
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )
   return Internal_Entity
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   Ent : Internal_Entity :=
     Internal_Entity'(Node => Self, Info => E_Info);
   --# bind entity Ent
   --# bind env Env
   Property_Result : Internal_Entity;
            procedure Finalizer_Scope_1756 with Inline_Always;
      Var_Expr : Lexical_Env := Empty_Env;
Var_Expr_1 : Internal_Entity;
Env_Node : Bare_Ada_Node;
As_Entity : Internal_Entity;
Is_Null : Boolean;
If_Result : Internal_Entity;
Scope_Result : Internal_Entity;
Env_Parent : Lexical_Env := Empty_Env;
Fld : Internal_Entity;
Result_Var : Internal_Entity;
Scope_Result_1 : Internal_Entity;
Result_Var_1 : Internal_Entity;
Let_Result : Internal_Entity;
            procedure Finalizer_Scope_1756 is
            begin
                     Dec_Ref (Var_Expr);
                     Dec_Ref (Env_Parent);
            end Finalizer_Scope_1756;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
   --# scope-start
      --# expr-start 260 '<Block at ???>' Let_Result None
--# scope-start
Var_Expr := Env;
      Inc_Ref (Var_Expr);
if Var_Expr /= Empty_Env then
   --# scope-start
   --# bind env Var_Expr
--# expr-start 257 '<as_entity at ast.py:446>' If_Result /tmp/lal-23.0.0/ada/ast.py:446
--# expr-start 256 '<env_node at ast.py:446>' Env_Node /tmp/lal-23.0.0/ada/ast.py:446
Env_Node := AST_Envs.Env_Node (Var_Expr); 
--# expr-done 256
Is_Null := Env_Node = null; 
if Is_Null then
   If_Result := No_Entity;
else
As_Entity := (Info => E_Info, Node => Env_Node); 
   If_Result := As_Entity;
end if;
--# expr-done 257
Var_Expr_1 := If_Result;
if Var_Expr_1 /= No_Entity then
   --# scope-start
   --# bind e Var_Expr_1
Scope_Result := Var_Expr_1;
   --# end
   Result_Var := Scope_Result;
else
   --# expr-start 259 '<FieldAccess for semantic_parent_helper at ast.py:447>' Fld /tmp/lal-23.0.0/ada/ast.py:447
      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# expr-start 258 '<env_parent at ast.py:447>' Env_Parent /tmp/lal-23.0.0/ada/ast.py:447
Env_Parent := AST_Envs.Parent (Var_Expr); 
--# expr-done 258
--# property-call-start AdaNode.semantic_parent_helper
Fld := Libadalang.Implementation.Ada_Node_P_Semantic_Parent_Helper (Node => Ent.Node, Env => Env_Parent, E_Info => Ent.Info);
--# end
--# expr-done 259
   Result_Var := Fld;
end if;
Scope_Result_1 := Result_Var;
   --# end
      Finalizer_Scope_1756;
   Result_Var_1 := Scope_Result_1;
else
   Result_Var_1 := No_Entity;
end if;
Let_Result := Result_Var_1; 
--# end
--# expr-done 260
      Property_Result := Let_Result;
   --# end
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when Exc : Precondition_Failure | Property_Error =>
               Finalizer_Scope_1756;
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Ada_Node_P_Semantic_Parent_Helper;
--# end
--# property-start AdaNode.semantic_parent /tmp/lal-23.0.0/ada/ast.py:451
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Semantic_Parent
  (Node : Bare_Ada_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )
   return Internal_Entity
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   Ent : Internal_Entity :=
     Internal_Entity'(Node => Self, Info => E_Info);
   --# bind entity Ent
   Property_Result : Internal_Entity;
            procedure Finalizer_Scope_28 with Inline_Always;
      Fld : Lexical_Env := Empty_Env;
Fld_1 : Internal_Entity;
Let_Result : Internal_Entity;
            procedure Finalizer_Scope_28 is
            begin
                     Dec_Ref (Fld);
            end Finalizer_Scope_28;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
      if Self /= null then
         Reset_Caches (Self.Unit);
            Populate_Lexical_Env (Self.Unit);
      end if;
   --# scope-start
      --# expr-start 263 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 262 '<FieldAccess for semantic_parent_helper at ast.py:460>' Fld_1 /tmp/lal-23.0.0/ada/ast.py:460
      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# expr-start 261 '<FieldAccess for node_env at ast.py:460>' Fld /tmp/lal-23.0.0/ada/ast.py:460
      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Fld := Libadalang.Implementation.Node_Env (Node => Ent.Node, E_Info => Ent.Info);
--# expr-done 261
--# property-call-start AdaNode.semantic_parent_helper
Fld_1 := Libadalang.Implementation.Ada_Node_P_Semantic_Parent_Helper (Node => Ent.Node, Env => Fld, E_Info => Ent.Info);
--# end
--# expr-done 262
Let_Result := Fld_1; 
--# end
--# expr-done 263
      Property_Result := Let_Result;
   --# end
      Finalizer_Scope_28;
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when Exc : Precondition_Failure | Property_Error =>
               Finalizer_Scope_28;
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Ada_Node_P_Semantic_Parent;
--# end
--# property-start AdaNode.is_in_top_level_public_part /tmp/lal-23.0.0/ada/ast.py:463
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Is_In_Top_Level_Public_Part
  (Node : Bare_Ada_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )
   return Boolean
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   Ent : Internal_Entity :=
     Internal_Entity'(Node => Self, Info => E_Info);
   --# bind entity Ent
   Property_Result : Boolean;
      Block_Var : Internal_Entity;
Var_Expr : Internal_Entity_Base_Package_Decl;
Fld : Internal_Entity;
Is_Null : Boolean;
Is_A : Boolean;
Fld_1 : Internal_Entity;
Cast_Result : Internal_Entity_Base_Package_Decl;
Fld_2 : Boolean;
Scope_Result : Boolean;
Result_Var : Boolean;
If_Result : Boolean;
Is_A_1 : Boolean;
Cast_Expr : Internal_Entity;
Cast_Result_1 : Internal_Entity_Package_Body;
Fld_3 : Boolean;
If_Result_1 : Boolean;
Is_A_2 : Boolean;
Cast_Expr_1 : Internal_Entity;
Cast_Result_2 : Internal_Entity_Base_Package_Decl;
Fld_4 : Boolean;
If_Result_2 : Boolean;
Fld_5 : Boolean;
If_Result_3 : Boolean;
If_Result_4 : Boolean;
If_Result_5 : Boolean;
If_Result_6 : Boolean;
Let_Result : Boolean;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
      if Self /= null then
         Reset_Caches (Self.Unit);
      end if;
   --# scope-start
      --# expr-start 278 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 264 '<FieldAccess for semantic_parent at ast.py:468>' Fld /tmp/lal-23.0.0/ada/ast.py:468
      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start AdaNode.semantic_parent
Fld := Libadalang.Implementation.Ada_Node_P_Semantic_Parent (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 264
Block_Var := Fld; 
--# bind node Block_Var
--# expr-start 277 '<Cond at ast.py:469>' If_Result_6 /tmp/lal-23.0.0/ada/ast.py:469
--# expr-start 265 '<IsNull at ast.py:470>' Is_Null /tmp/lal-23.0.0/ada/ast.py:470
Is_Null := Block_Var.Node = null; 
--# expr-done 265
if Is_Null then
   If_Result_6 := False;
else
--# expr-start 266 '<IsA PrivatePart at ast.py:473>' Is_A /tmp/lal-23.0.0/ada/ast.py:473
Is_A := Block_Var.Node /= null 
and then Block_Var.Node.Kind in Ada_Private_Part_Range; 
--# expr-done 266
if Is_A then
--# expr-start 268 '<Cast to BasePackageDecl at ast.py:474>' Cast_Result /tmp/lal-23.0.0/ada/ast.py:474
--# expr-start 267 '<parent at ast.py:474>' Fld_1 /tmp/lal-23.0.0/ada/ast.py:474
      if Block_Var.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Fld_1 := Libadalang.Implementation.Parent (Node => Block_Var.Node, E_Info => Block_Var.Info);
--# expr-done 267
   if Fld_1.Node = null
      or else Fld_1.Node.Kind in Ada_Base_Package_Decl
   then
      Cast_Result := Create_Internal_Entity_Base_Package_Decl
        (Node => Fld_1.Node,
         Info => Fld_1.Info);
   else
         Cast_Result := No_Entity_Base_Package_Decl;
   end if;
--# expr-done 268
Var_Expr := Cast_Result;
if Var_Expr /= No_Entity_Base_Package_Decl then
   --# scope-start
--# expr-start 269 '<FieldAccess for is_compilation_unit_root at ast.py:474>' Fld_2 /tmp/lal-23.0.0/ada/ast.py:474
      if Var_Expr.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start BasicDecl.is_compilation_unit_root
Fld_2 := Libadalang.Implementation.Basic_Decl_P_Is_Compilation_Unit_Root (Node => Var_Expr.Node);
--# end
--# expr-done 269
Scope_Result := Fld_2;
   --# end
   Result_Var := Scope_Result;
else
   Result_Var := False;
end if;
   If_Result := Result_Var;
else
   If_Result := False;
end if;
if If_Result then
   If_Result_5 := False;
else
--# expr-start 270 '<IsA PackageBody at ast.py:477>' Is_A_1 /tmp/lal-23.0.0/ada/ast.py:477
Is_A_1 := Block_Var.Node /= null 
and then Block_Var.Node.Kind in Ada_Package_Body_Range; 
--# expr-done 270
if Is_A_1 then
   --# expr-start 272 '<FieldAccess for is_compilation_unit_root at ast.py:478>' Fld_3 /tmp/lal-23.0.0/ada/ast.py:478
--# expr-start 271 '<Cast to PackageBody at ast.py:478>' Cast_Result_1 /tmp/lal-23.0.0/ada/ast.py:478
Cast_Expr := Block_Var; 
   if Cast_Expr.Node = null
      or else Cast_Expr.Node.Kind in Ada_Package_Body_Range
   then
      Cast_Result_1 := Create_Internal_Entity_Package_Body
        (Node => Cast_Expr.Node,
         Info => Cast_Expr.Info);
   else
         Cast_Result_1 := No_Entity_Package_Body;
   end if;
--# expr-done 271
      if Cast_Result_1.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start BasicDecl.is_compilation_unit_root
Fld_3 := Libadalang.Implementation.Basic_Decl_P_Is_Compilation_Unit_Root (Node => Cast_Result_1.Node);
--# end
--# expr-done 272
   If_Result_1 := Fld_3;
else
   If_Result_1 := False;
end if;
if If_Result_1 then
   If_Result_4 := False;
else
--# expr-start 273 '<IsA BasePackageDecl at ast.py:481>' Is_A_2 /tmp/lal-23.0.0/ada/ast.py:481
Is_A_2 := Block_Var.Node /= null 
and then Block_Var.Node.Kind in Ada_Base_Package_Decl; 
--# expr-done 273
if Is_A_2 then
   --# expr-start 275 '<FieldAccess for is_compilation_unit_root at ast.py:482>' Fld_4 /tmp/lal-23.0.0/ada/ast.py:482
--# expr-start 274 '<Cast to BasePackageDecl at ast.py:482>' Cast_Result_2 /tmp/lal-23.0.0/ada/ast.py:482
Cast_Expr_1 := Block_Var; 
   if Cast_Expr_1.Node = null
      or else Cast_Expr_1.Node.Kind in Ada_Base_Package_Decl
   then
      Cast_Result_2 := Create_Internal_Entity_Base_Package_Decl
        (Node => Cast_Expr_1.Node,
         Info => Cast_Expr_1.Info);
   else
         Cast_Result_2 := No_Entity_Base_Package_Decl;
   end if;
--# expr-done 274
      if Cast_Result_2.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start BasicDecl.is_compilation_unit_root
Fld_4 := Libadalang.Implementation.Basic_Decl_P_Is_Compilation_Unit_Root (Node => Cast_Result_2.Node);
--# end
--# expr-done 275
   If_Result_2 := Fld_4;
else
   If_Result_2 := False;
end if;
if If_Result_2 then
   If_Result_3 := True;
else
   --# expr-start 276 '<FieldAccess for is_in_top_level_public_part at ast.py:485>' Fld_5 /tmp/lal-23.0.0/ada/ast.py:485
      if Block_Var.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start AdaNode.is_in_top_level_public_part
Fld_5 := Libadalang.Implementation.Ada_Node_P_Is_In_Top_Level_Public_Part (Node => Block_Var.Node, E_Info => Block_Var.Info);
--# end
--# expr-done 276
   If_Result_3 := Fld_5;
end if;
   If_Result_4 := If_Result_3;
end if;
   If_Result_5 := If_Result_4;
end if;
   If_Result_6 := If_Result_5;
end if;
--# expr-done 277
Let_Result := If_Result_6; 
--# end
--# expr-done 278
      Property_Result := Let_Result;
   --# end
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Ada_Node_P_Is_In_Top_Level_Public_Part;
--# end
--# property-start AdaNode.parent_basic_decl /tmp/lal-23.0.0/ada/ast.py:489
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Parent_Basic_Decl
  (Node : Bare_Ada_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )
   return Internal_Entity_Basic_Decl
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   Ent : Internal_Entity :=
     Internal_Entity'(Node => Self, Info => E_Info);
   --# bind entity Ent
   Property_Result : Internal_Entity_Basic_Decl;
      Var_Expr : Internal_Entity;
Var_Expr_1 : Internal_Entity_Basic_Decl;
Fld : Internal_Entity;
Fld_1 : Internal_Entity_Basic_Decl;
Fld_2 : Internal_Entity;
Cast_Expr : Internal_Entity;
Cast_Result : Internal_Entity_Basic_Decl;
Fld_3 : Internal_Entity_Generic_Instantiation;
Cast_Expr_1 : Internal_Entity;
Cast_Result_1 : Internal_Entity_Basic_Decl;
Cast_Result_2 : Internal_Entity_Basic_Decl;
Is_A : Boolean;
If_Result : Internal_Entity_Basic_Decl;
Scope_Result : Internal_Entity_Basic_Decl;
Fld_4 : Internal_Entity_Basic_Decl;
Result_Var : Internal_Entity_Basic_Decl;
Scope_Result_1 : Internal_Entity_Basic_Decl;
Result_Var_1 : Internal_Entity_Basic_Decl;
Is_A_1 : Boolean;
If_Result_1 : Internal_Entity_Basic_Decl;
Let_Result : Internal_Entity_Basic_Decl;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
      if Self /= null then
         Reset_Caches (Self.Unit);
            Populate_Lexical_Env (Self.Unit);
      end if;
   --# scope-start
      --# expr-start 290 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 289 '<If at ast.py:504>' If_Result_1 /tmp/lal-23.0.0/ada/ast.py:504
--# expr-start 279 '<IsA ClasswideTypeDecl, DiscreteBaseSubtypeDecl, SynthAnonymousTypeDecl at ast.py:509>' Is_A_1 /tmp/lal-23.0.0/ada/ast.py:509
Is_A_1 := Ent.Node /= null 
and then Ent.Node.Kind in Ada_Classwide_Type_Decl_Range | Ada_Discrete_Base_Subtype_Decl_Range | Ada_Synth_Anonymous_Type_Decl_Range; 
--# expr-done 279
if Is_A_1 then
   --# expr-start 281 '<FieldAccess for parent_basic_decl at ast.py:511>' Fld_1 /tmp/lal-23.0.0/ada/ast.py:511
--# expr-start 280 '<FieldAccess for semantic_parent at ast.py:511>' Fld /tmp/lal-23.0.0/ada/ast.py:511
      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start AdaNode.semantic_parent
Fld := Libadalang.Implementation.Ada_Node_P_Semantic_Parent (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 280
      if Fld.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start AdaNode.parent_basic_decl
Fld_1 := Libadalang.Implementation.Ada_Node_P_Parent_Basic_Decl (Node => Fld.Node, E_Info => Fld.Info);
--# end
--# expr-done 281
   If_Result_1 := Fld_1;
else
--# expr-start 282 '<FieldAccess for semantic_parent at ast.py:512>' Fld_2 /tmp/lal-23.0.0/ada/ast.py:512
      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start AdaNode.semantic_parent
Fld_2 := Libadalang.Implementation.Ada_Node_P_Semantic_Parent (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 282
Var_Expr := Fld_2;
if Var_Expr /= No_Entity then
   --# scope-start
   --# bind sp Var_Expr
--# expr-start 287 '<If at ast.py:513>' If_Result /tmp/lal-23.0.0/ada/ast.py:513
--# expr-start 283 '<IsA GenericSubpInternal, GenericPackageInternal at ast.py:514>' Is_A /tmp/lal-23.0.0/ada/ast.py:514
Is_A := Var_Expr.Node /= null 
and then Var_Expr.Node.Kind in Ada_Generic_Subp_Internal_Range | Ada_Generic_Package_Internal_Range; 
--# expr-done 283
if Is_A then
--# expr-start 285 '<FieldAccess for get_instantiation at ast.py:515>' Fld_3 /tmp/lal-23.0.0/ada/ast.py:515
--# expr-start 284 '<Cast to BasicDecl at ast.py:515>' Cast_Result /tmp/lal-23.0.0/ada/ast.py:515
Cast_Expr := Var_Expr; 
   if Cast_Expr.Node = null
      or else Cast_Expr.Node.Kind in Ada_Basic_Decl
   then
      Cast_Result := Create_Internal_Entity_Basic_Decl
        (Node => Cast_Expr.Node,
         Info => Cast_Expr.Info);
   else
         Cast_Result := No_Entity_Basic_Decl;
   end if;
--# expr-done 284
      if Cast_Result.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start BasicDecl.get_instantiation
Fld_3 := Libadalang.Implementation.Basic_Decl_P_Get_Instantiation (Node => Cast_Result.Node, E_Info => Cast_Result.Info);
--# end
--# expr-done 285
      Cast_Result_2 := Create_Internal_Entity_Basic_Decl
        (Node => Fld_3.Node,
         Info => Fld_3.Info);
   If_Result := Cast_Result_2;
else
   --# expr-start 286 '<Cast to BasicDecl at ast.py:516>' Cast_Result_1 /tmp/lal-23.0.0/ada/ast.py:516
Cast_Expr_1 := Var_Expr; 
   if Cast_Expr_1.Node = null
      or else Cast_Expr_1.Node.Kind in Ada_Basic_Decl
   then
      Cast_Result_1 := Create_Internal_Entity_Basic_Decl
        (Node => Cast_Expr_1.Node,
         Info => Cast_Expr_1.Info);
   else
         Cast_Result_1 := No_Entity_Basic_Decl;
   end if;
--# expr-done 286
   If_Result := Cast_Result_1;
end if;
--# expr-done 287
Var_Expr_1 := If_Result;
if Var_Expr_1 /= No_Entity_Basic_Decl then
   --# scope-start
   --# bind e Var_Expr_1
Scope_Result := Var_Expr_1;
   --# end
   Result_Var := Scope_Result;
else
   --# expr-start 288 '<FieldAccess for parent_basic_decl at ast.py:517>' Fld_4 /tmp/lal-23.0.0/ada/ast.py:517
      if Var_Expr.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start AdaNode.parent_basic_decl
Fld_4 := Libadalang.Implementation.Ada_Node_P_Parent_Basic_Decl (Node => Var_Expr.Node, E_Info => Var_Expr.Info);
--# end
--# expr-done 288
   Result_Var := Fld_4;
end if;
Scope_Result_1 := Result_Var;
   --# end
   Result_Var_1 := Scope_Result_1;
else
   Result_Var_1 := No_Entity_Basic_Decl;
end if;
   If_Result_1 := Result_Var_1;
end if;
--# expr-done 289
Let_Result := If_Result_1; 
--# end
--# expr-done 290
      Property_Result := Let_Result;
   --# end
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Ada_Node_P_Parent_Basic_Decl;
--# end
--# property-start '[dispatcher]AdaNode.immediate_declarative_region' dispatcher
pragma Warnings (Off, "is not referenced");
 function Dispatcher_Ada_Node_P_Immediate_Declarative_Region
  (Node : Bare_Ada_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )
   return Lexical_Env
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   --# bind self Self
   Property_Result : Lexical_Env;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
      if Self /= null then
         Reset_Caches (Self.Unit);
      end if;
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dispatching on null node");
      end if;
      case Ada_Ada_Node (Self.Kind) is
               when Ada_Abort_Absent .. Ada_Param_Assoc | Ada_Case_Stmt_Alternative .. Ada_Declarative_Part | Ada_Public_Part .. Ada_Begin_Block | Ada_Case_Stmt .. Ada_With_Private_Present =>
                  --# property-call-start AdaNode.immediate_declarative_region
                  Property_Result := Ada_Node_P_Immediate_Declarative_Region
                    (Self
                        , E_Info
                    );
                  --# end
               when Ada_Abstract_State_Decl .. Ada_Single_Task_Decl =>
                  --# property-call-start BasicDecl.immediate_declarative_region
                  Property_Result := Basic_Decl_P_Immediate_Declarative_Region
                    (Self
                        , E_Info
                    );
                  --# end
               when Ada_Private_Part =>
                  --# property-call-start PrivatePart.immediate_declarative_region
                  Property_Result := Private_Part_P_Immediate_Declarative_Region
                    (Self
                        , E_Info
                    );
                  --# end
               when Ada_Decl_Block =>
                  --# property-call-start DeclBlock.immediate_declarative_region
                  Property_Result := Decl_Block_P_Immediate_Declarative_Region
                    (Self
                        , E_Info
                    );
                  --# end
      end case;
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Dispatcher_Ada_Node_P_Immediate_Declarative_Region;
--# end
--# property-start AdaNode.designated_compilation_unit /tmp/lal-23.0.0/ada/ast.py:565
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Designated_Compilation_Unit
  (Node : Bare_Ada_Node
      ; Name : Symbol_Type_Array_Access
      ; Kind : Analysis_Unit_Kind
      ; Load_If_Needed : Boolean
         := True
      ; Not_Found_Is_Error : Boolean
         := True
      ; Process_Parents : Boolean
         := True
  )
   return Bare_Compilation_Unit
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   --# bind self Self
   --# bind name Name
   --# bind kind Kind
   --# bind load_if_needed Load_If_Needed
   --# bind not_found_is_error Not_Found_Is_Error
   --# bind process_parents Process_Parents
   Property_Result : Bare_Compilation_Unit;
      Block_Var : Internal_Unit;
Fld : Internal_Unit;
Fld_1 : Bare_Compilation_Unit;
Let_Result : Bare_Compilation_Unit;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
      if Self /= null then
         Reset_Caches (Self.Unit);
      end if;
   --# scope-start
      --# expr-start 293 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 291 '<FieldAccess for get_unit at ast.py:575>' Fld /tmp/lal-23.0.0/ada/ast.py:575
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Fld := Libadalang.Implementation.Extensions.Ada_Node_P_Get_Unit (Node => Self, Name => Name, Kind => Kind, Load_If_Needed => Load_If_Needed, Not_Found_Is_Error => Not_Found_Is_Error, Process_Parents => Process_Parents);
--# expr-done 291
Block_Var := Fld; 
--# bind designated_analysis_unit Block_Var
--# expr-start 292 '<FieldAccess for compilation_unit_with_name at ast.py:579>' Fld_1 /tmp/lal-23.0.0/ada/ast.py:579
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start AdaNode.compilation_unit_with_name
Fld_1 := Libadalang.Implementation.Ada_Node_P_Compilation_Unit_With_Name (Node => Self, Unit => Block_Var, Name => Name);
--# end
--# expr-done 292
Let_Result := Fld_1; 
--# end
--# expr-done 293
      Property_Result := Let_Result;
   --# end
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Ada_Node_P_Designated_Compilation_Unit;
--# end
--# property-start AdaNode.compilation_unit_with_name /tmp/lal-23.0.0/ada/ast.py:582
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Compilation_Unit_With_Name
  (Node : Bare_Ada_Node
      ; Unit : Internal_Unit
      ; Name : Symbol_Type_Array_Access
  )
   return Bare_Compilation_Unit
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   --# bind self Self
   --# bind unit Unit
   --# bind name Name
   Property_Result : Bare_Compilation_Unit;
            procedure Finalizer_Scope_1770 with Inline_Always;
      Var_Expr : Bare_Ada_Node;
Match_0 : Bare_Compilation_Unit;
Match_1 : Bare_Compilation_Unit_List;
Match_2 : Bare_Ada_Node;
Fld : Bare_Ada_Node;
Fld_1 : Symbol_Type_Array_Access;
Is_Equal : Boolean;
Item_8 : Bare_Compilation_Unit;
Find_Result : Bare_Compilation_Unit;
Match_Prefix : Bare_Ada_Node;
Cast_Result : Bare_Compilation_Unit;
Let_Result : Bare_Compilation_Unit;
Scope_Result : Bare_Compilation_Unit;
Cast_Result_1 : Bare_Compilation_Unit_List;
Let_Result_1 : Bare_Compilation_Unit;
Scope_Result_1 : Bare_Compilation_Unit;
Cast_Result_2 : Bare_Ada_Node;
Let_Result_2 : Bare_Compilation_Unit;
Scope_Result_2 : Bare_Compilation_Unit;
Match_Result : Bare_Compilation_Unit;
Scope_Result_3 : Bare_Compilation_Unit;
Result_Var : Bare_Compilation_Unit;
Let_Result_3 : Bare_Compilation_Unit;
            procedure Finalizer_Scope_1770 is
            begin
                     Dec_Ref (Fld_1);
            end Finalizer_Scope_1770;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
      if Self /= null then
         Reset_Caches (Self.Unit);
      end if;
   --# scope-start
      --# expr-start 299 '<Block at ???>' Let_Result_3 None
--# scope-start
--# expr-start 294 '<FieldAccess for root at ast.py:588>' Fld /tmp/lal-23.0.0/ada/ast.py:588
      if Unit = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Fld := Unit.Ast_Root;
--# expr-done 294
Var_Expr := Fld;
if Var_Expr /= No_Bare_Ada_Node then
   --# scope-start
--# expr-start 298 '<Match at ast.py:588>' Match_Result /tmp/lal-23.0.0/ada/ast.py:588
      if Var_Expr = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Match_Prefix := Var_Expr; 
case Ada_Ada_Node (Match_Prefix.Kind) is
         when Ada_Compilation_Unit =>
   --# scope-start
--# scope-start
      Cast_Result := Match_Prefix;
Match_0 := Cast_Result; 
--# bind single Match_0
Let_Result := Match_0; 
--# end
Scope_Result := Let_Result;
   --# end
            Match_Result := Scope_Result; 
         when Ada_Compilation_Unit_List =>
   --# scope-start
--# scope-start
      Cast_Result_1 := Match_Prefix;
Match_1 := Cast_Result_1; 
--# bind multi Match_1
--# expr-start 297 '<Find at ast.py:595>' Find_Result /tmp/lal-23.0.0/ada/ast.py:595
      Find_Result := No_Bare_Ada_Node;
      if Match_1 = null then
      Find_Result := No_Bare_Ada_Node;
      else
      declare
         Collection : constant Bare_Compilation_Unit_List := Match_1;
      begin
         for Untyped_Item_8 of
               Collection.Nodes (1 .. Children_Count (Collection))
         loop
                  Item_8 := Untyped_Item_8; 
   --# scope-start
               --# bind c Item_8
      --# expr-start 296 '<Eq at ast.py:596>' Is_Equal /tmp/lal-23.0.0/ada/ast.py:596
--# expr-start 295 '<FieldAccess for syntactic_fully_qualified_name at ast.py:596>' Fld_1 /tmp/lal-23.0.0/ada/ast.py:596
      if Item_8 = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start CompilationUnit.syntactic_fully_qualified_name
Fld_1 := Libadalang.Implementation.Compilation_Unit_P_Syntactic_Fully_Qualified_Name (Node => Item_8);
--# end
--# expr-done 295
Is_Equal := Equivalent (Fld_1, Name); 
--# expr-done 296
      if Is_Equal then
         Find_Result := Item_8;
      end if;
   --# end
      Finalizer_Scope_1770;
               exit when Is_Equal;
         end loop;
      end;
      end if;
--# expr-done 297
Let_Result_1 := Find_Result; 
--# end
Scope_Result_1 := Let_Result_1;
   --# end
            Match_Result := Scope_Result_1; 
         when Ada_Abort_Absent .. Ada_Case_Stmt_Alternative_List | Ada_Concat_Operand_List .. Ada_Case_Stmt_Alternative | Ada_Component_Clause .. Ada_With_Private_Present =>
   --# scope-start
--# scope-start
      Cast_Result_2 := Match_Prefix;
Match_2 := Cast_Result_2; 
--# bind _ Match_2
Let_Result_2 := raise Property_Error with "Unexpected analysis unit root"; 
--# end
Scope_Result_2 := Let_Result_2;
   --# end
            Match_Result := Scope_Result_2; 
end case;
--# expr-done 298
Scope_Result_3 := Match_Result;
   --# end
   Result_Var := Scope_Result_3;
else
   Result_Var := No_Bare_Ada_Node;
end if;
Let_Result_3 := Result_Var; 
--# end
--# expr-done 299
      Property_Result := Let_Result_3;
   --# end
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when Exc : Precondition_Failure | Property_Error =>
               Finalizer_Scope_1770;
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Ada_Node_P_Compilation_Unit_With_Name;
--# end
--# property-start AdaNode.get_unit_root_decl /tmp/lal-23.0.0/ada/ast.py:605
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Get_Unit_Root_Decl
  (Node : Bare_Ada_Node
      ; Name : Symbol_Type_Array_Access
      ; Kind : Analysis_Unit_Kind
      ; Load_If_Needed : Boolean
         := True
      ; Not_Found_Is_Error : Boolean
         := True
      ; Process_Parents : Boolean
         := True
  )
   return Bare_Basic_Decl
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   --# bind self Self
   --# bind name Name
   --# bind kind Kind
   --# bind load_if_needed Load_If_Needed
   --# bind not_found_is_error Not_Found_Is_Error
   --# bind process_parents Process_Parents
   Property_Result : Bare_Basic_Decl;
      Block_Var : Bare_Compilation_Unit;
Var_Expr : Bare_Compilation_Unit;
Fld : Bare_Compilation_Unit;
Fld_1 : Bare_Basic_Decl;
Scope_Result : Bare_Basic_Decl;
Result_Var : Bare_Basic_Decl;
Let_Result : Bare_Basic_Decl;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
      if Self /= null then
         Reset_Caches (Self.Unit);
      end if;
   --# scope-start
      --# expr-start 302 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 300 '<FieldAccess for designated_compilation_unit at ast.py:614>' Fld /tmp/lal-23.0.0/ada/ast.py:614
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start AdaNode.designated_compilation_unit
Fld := Libadalang.Implementation.Ada_Node_P_Designated_Compilation_Unit (Node => Self, Name => Name, Kind => Kind, Load_If_Needed => Load_If_Needed, Not_Found_Is_Error => Not_Found_Is_Error, Process_Parents => Process_Parents);
--# end
--# expr-done 300
Block_Var := Fld; 
--# bind cu Block_Var
Var_Expr := Block_Var;
if Var_Expr /= No_Bare_Ada_Node then
   --# scope-start
--# expr-start 301 '<FieldAccess for decl at ast.py:617>' Fld_1 /tmp/lal-23.0.0/ada/ast.py:617
      if Var_Expr = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start CompilationUnit.decl
Fld_1 := Libadalang.Implementation.Compilation_Unit_P_Decl (Node => Var_Expr);
--# end
--# expr-done 301
Scope_Result := Fld_1;
   --# end
   Result_Var := Scope_Result;
else
   Result_Var := No_Bare_Ada_Node;
end if;
Let_Result := Result_Var; 
--# end
--# expr-done 302
      Property_Result := Let_Result;
   --# end
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Ada_Node_P_Get_Unit_Root_Decl;
--# end
--# property-start '[dispatcher]AdaNode.xref_equation' dispatcher
pragma Warnings (Off, "is not referenced");
 function Dispatcher_Ada_Node_P_Xref_Equation
  (Node : Bare_Ada_Node
      ; Env : Lexical_Env
      ; Origin : Bare_Ada_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )
   return Logic_Equation
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   --# bind self Self
   --# bind env Env
   --# bind origin Origin
   Property_Result : Logic_Equation;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
      if Self /= null then
         Reset_Caches (Self.Unit);
      end if;
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dispatching on null node");
      end if;
      case Ada_Ada_Node (Self.Kind) is
               when Ada_Abort_Absent .. Ada_All_Present | Ada_Aspect_Spec .. Ada_Pragma_Argument_Assoc | Ada_Component_List .. Ada_Record_Def | Ada_Composite_Constraint_Assoc | Ada_Abstract_State_Decl .. Ada_Anonymous_Expr_Decl | Ada_Synthetic_Formal_Param_Decl .. Ada_Discrete_Base_Subtype_Decl | Ada_Classwide_Type_Decl .. Ada_Incomplete_Tagged_Type_Decl | Ada_Abstract_Subp_Decl | Ada_Subp_Decl .. Ada_Synthetic_Subp_Decl | Ada_Null_Subp_Decl .. Ada_Subp_Body | Ada_Package_Body_Stub .. Ada_Task_Body_Stub | Ada_Package_Body .. Ada_Task_Body | Ada_Error_Decl | Ada_Generic_Package_Decl .. Ada_Generic_Subp_Decl | Ada_Label_Decl .. Ada_Named_Stmt_Decl | Ada_Single_Task_Decl | Ada_Compilation_Unit | Ada_Constant_Absent .. Ada_Constant_Present | Ada_Declarative_Part .. Ada_Elsif_Expr_Part | Ada_Case_Expr_Alternative | Ada_Concat_Operand | Ada_Contract_Cases | Ada_Discrete_Subtype_Name | Ada_Synthetic_Identifier | Ada_Handled_Stmts .. Ada_Limited_Present | Ada_Mode_Default .. Ada_Null_Component_Decl | Ada_Overriding_Not_Overriding .. Ada_Pp_Then_Kw | Ada_Private_Absent .. Ada_Quantifier_Some | Ada_Renaming_Clause .. Ada_Reverse_Present | Ada_Error_Stmt | Ada_Subp_Kind_Function .. Ada_Subp_Kind_Procedure | Ada_Synchronized_Absent .. Ada_Tagged_Present | Ada_Type_Attributes_Repository | Ada_Anonymous_Type_Access_Def | Ada_Enum_Lit_Synth_Type_Expr | Ada_Synthetic_Type_Expr .. Ada_Until_Present | Ada_Variant | Ada_With_Private_Absent .. Ada_With_Private_Present =>
                  --# property-call-start AdaNode.xref_equation
                  Property_Result := Ada_Node_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Constrained_Array_Indices =>
                  --# property-call-start ConstrainedArrayIndices.xref_equation
                  Property_Result := Constrained_Array_Indices_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Unconstrained_Array_Indices =>
                  --# property-call-start UnconstrainedArrayIndices.xref_equation
                  Property_Result := Unconstrained_Array_Indices_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Aspect_Assoc =>
                  --# property-call-start AspectAssoc.xref_equation
                  Property_Result := Aspect_Assoc_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_At_Clause =>
                  --# property-call-start AtClause.xref_equation
                  Property_Result := At_Clause_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Attribute_Def_Clause =>
                  --# property-call-start AttributeDefClause.xref_equation
                  Property_Result := Attribute_Def_Clause_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Enum_Rep_Clause =>
                  --# property-call-start EnumRepClause.xref_equation
                  Property_Result := Enum_Rep_Clause_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Record_Rep_Clause =>
                  --# property-call-start RecordRepClause.xref_equation
                  Property_Result := Record_Rep_Clause_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Enum_Subp_Spec .. Ada_Synthetic_Unary_Spec =>
                  --# property-call-start BaseSubpSpec.xref_equation
                  Property_Result := Base_Subp_Spec_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Entry_Spec =>
                  --# property-call-start EntrySpec.xref_equation
                  Property_Result := Entry_Spec_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Aggregate_Assoc .. Ada_Multi_Dim_Array_Assoc =>
                  --# property-call-start AggregateAssoc.xref_equation
                  Property_Result := Aggregate_Assoc_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Iterated_Assoc =>
                  --# property-call-start IteratedAssoc.xref_equation
                  Property_Result := Iterated_Assoc_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Param_Assoc =>
                  --# property-call-start ParamAssoc.xref_equation
                  Property_Result := Param_Assoc_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Component_Decl =>
                  --# property-call-start ComponentDecl.xref_equation
                  Property_Result := Component_Decl_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Discriminant_Spec =>
                  --# property-call-start DiscriminantSpec.xref_equation
                  Property_Result := Discriminant_Spec_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Generic_Formal_Obj_Decl .. Ada_Generic_Formal_Type_Decl =>
                  --# property-call-start GenericFormal.xref_equation
                  Property_Result := Generic_Formal_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Param_Spec =>
                  --# property-call-start ParamSpec.xref_equation
                  Property_Result := Param_Spec_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Subtype_Decl =>
                  --# property-call-start SubtypeDecl.xref_equation
                  Property_Result := Subtype_Decl_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Protected_Type_Decl =>
                  --# property-call-start ProtectedTypeDecl.xref_equation
                  Property_Result := Protected_Type_Decl_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Task_Type_Decl .. Ada_Single_Task_Type_Decl =>
                  --# property-call-start TaskTypeDecl.xref_equation
                  Property_Result := Task_Type_Decl_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Anonymous_Type_Decl .. Ada_Concrete_Type_Decl =>
                  --# property-call-start TypeDecl.xref_equation
                  Property_Result := Type_Decl_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Formal_Type_Decl =>
                  --# property-call-start FormalTypeDecl.xref_equation
                  Property_Result := Formal_Type_Decl_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Abstract_Formal_Subp_Decl .. Ada_Concrete_Formal_Subp_Decl =>
                  --# property-call-start FormalSubpDecl.xref_equation
                  Property_Result := Formal_Subp_Decl_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Expr_Function =>
                  --# property-call-start ExprFunction.xref_equation
                  Property_Result := Expr_Function_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Subp_Renaming_Decl =>
                  --# property-call-start SubpRenamingDecl.xref_equation
                  Property_Result := Subp_Renaming_Decl_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Entry_Body =>
                  --# property-call-start EntryBody.xref_equation
                  Property_Result := Entry_Body_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Entry_Index_Spec =>
                  --# property-call-start EntryIndexSpec.xref_equation
                  Property_Result := Entry_Index_Spec_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Exception_Decl =>
                  --# property-call-start ExceptionDecl.xref_equation
                  Property_Result := Exception_Decl_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Exception_Handler =>
                  --# property-call-start ExceptionHandler.xref_equation
                  Property_Result := Exception_Handler_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_For_Loop_Var_Decl =>
                  --# property-call-start ForLoopVarDecl.xref_equation
                  Property_Result := For_Loop_Var_Decl_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Generic_Package_Instantiation .. Ada_Generic_Subp_Instantiation =>
                  --# property-call-start GenericInstantiation.xref_equation
                  Property_Result := Generic_Instantiation_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Generic_Package_Renaming_Decl .. Ada_Generic_Subp_Renaming_Decl =>
                  --# property-call-start GenericRenamingDecl.xref_equation
                  Property_Result := Generic_Renaming_Decl_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Number_Decl =>
                  --# property-call-start NumberDecl.xref_equation
                  Property_Result := Number_Decl_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Object_Decl .. Ada_Extended_Return_Stmt_Object_Decl =>
                  --# property-call-start ObjectDecl.xref_equation
                  Property_Result := Object_Decl_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_No_Type_Object_Renaming_Decl =>
                  --# property-call-start NoTypeObjectRenamingDecl.xref_equation
                  Property_Result := No_Type_Object_Renaming_Decl_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Package_Renaming_Decl =>
                  --# property-call-start PackageRenamingDecl.xref_equation
                  Property_Result := Package_Renaming_Decl_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Single_Protected_Decl =>
                  --# property-call-start SingleProtectedDecl.xref_equation
                  Property_Result := Single_Protected_Decl_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Case_Stmt_Alternative =>
                  --# property-call-start CaseStmtAlternative.xref_equation
                  Property_Result := Case_Stmt_Alternative_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Component_Clause =>
                  --# property-call-start ComponentClause.xref_equation
                  Property_Result := Component_Clause_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Component_Def =>
                  --# property-call-start ComponentDef.xref_equation
                  Property_Result := Component_Def_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Composite_Constraint =>
                  --# property-call-start CompositeConstraint.xref_equation
                  Property_Result := Composite_Constraint_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Delta_Constraint =>
                  --# property-call-start DeltaConstraint.xref_equation
                  Property_Result := Delta_Constraint_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Digits_Constraint =>
                  --# property-call-start DigitsConstraint.xref_equation
                  Property_Result := Digits_Constraint_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Range_Constraint =>
                  --# property-call-start RangeConstraint.xref_equation
                  Property_Result := Range_Constraint_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Elsif_Stmt_Part =>
                  --# property-call-start ElsifStmtPart.xref_equation
                  Property_Result := Elsif_Stmt_Part_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Abstract_State_Decl_Expr =>
                  --# property-call-start AbstractStateDeclExpr.xref_equation
                  Property_Result := Abstract_State_Decl_Expr_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Allocator =>
                  --# property-call-start Allocator.xref_equation
                  Property_Result := Allocator_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Aggregate .. Ada_Null_Record_Aggregate =>
                  --# property-call-start BaseAggregate.xref_equation
                  Property_Result := Base_Aggregate_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Bin_Op .. Ada_Relation_Op =>
                  --# property-call-start BinOp.xref_equation
                  Property_Result := Bin_Op_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Box_Expr =>
                  --# property-call-start BoxExpr.xref_equation
                  Property_Result := Box_Expr_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Concat_Op =>
                  --# property-call-start ConcatOp.xref_equation
                  Property_Result := Concat_Op_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Case_Expr =>
                  --# property-call-start CaseExpr.xref_equation
                  Property_Result := Case_Expr_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_If_Expr =>
                  --# property-call-start IfExpr.xref_equation
                  Property_Result := If_Expr_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Decl_Expr =>
                  --# property-call-start DeclExpr.xref_equation
                  Property_Result := Decl_Expr_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Membership_Expr =>
                  --# property-call-start MembershipExpr.xref_equation
                  Property_Result := Membership_Expr_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Attribute_Ref =>
                  --# property-call-start AttributeRef.xref_equation
                  Property_Result := Attribute_Ref_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Call_Expr =>
                  --# property-call-start CallExpr.xref_equation
                  Property_Result := Call_Expr_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Defining_Name .. Ada_Synthetic_Defining_Name =>
                  --# property-call-start DefiningName.xref_equation
                  Property_Result := Defining_Name_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Dotted_Name =>
                  --# property-call-start DottedName.xref_equation
                  Property_Result := Dotted_Name_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_End_Name =>
                  --# property-call-start EndName.xref_equation
                  Property_Result := End_Name_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Explicit_Deref =>
                  --# property-call-start ExplicitDeref.xref_equation
                  Property_Result := Explicit_Deref_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Qual_Expr =>
                  --# property-call-start QualExpr.xref_equation
                  Property_Result := Qual_Expr_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Reduce_Attribute_Ref =>
                  --# property-call-start ReduceAttributeRef.xref_equation
                  Property_Result := Reduce_Attribute_Ref_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Identifier =>
                  --# property-call-start BaseId.xref_equation
                  Property_Result := Base_Id_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Char_Literal =>
                  --# property-call-start CharLiteral.xref_equation
                  Property_Result := Char_Literal_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Op_Abs .. Ada_Op_Xor =>
                  --# property-call-start Op.xref_equation
                  Property_Result := Op_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_String_Literal =>
                  --# property-call-start StringLiteral.xref_equation
                  Property_Result := String_Literal_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Null_Literal =>
                  --# property-call-start NullLiteral.xref_equation
                  Property_Result := Null_Literal_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Int_Literal =>
                  --# property-call-start IntLiteral.xref_equation
                  Property_Result := Int_Literal_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Real_Literal =>
                  --# property-call-start RealLiteral.xref_equation
                  Property_Result := Real_Literal_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Target_Name =>
                  --# property-call-start TargetName.xref_equation
                  Property_Result := Target_Name_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Update_Attribute_Ref =>
                  --# property-call-start UpdateAttributeRef.xref_equation
                  Property_Result := Update_Attribute_Ref_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Paren_Expr =>
                  --# property-call-start ParenExpr.xref_equation
                  Property_Result := Paren_Expr_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Quantified_Expr =>
                  --# property-call-start QuantifiedExpr.xref_equation
                  Property_Result := Quantified_Expr_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Raise_Expr =>
                  --# property-call-start RaiseExpr.xref_equation
                  Property_Result := Raise_Expr_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Un_Op =>
                  --# property-call-start UnOp.xref_equation
                  Property_Result := Un_Op_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_For_Loop_Spec =>
                  --# property-call-start ForLoopSpec.xref_equation
                  Property_Result := For_Loop_Spec_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_While_Loop_Spec =>
                  --# property-call-start WhileLoopSpec.xref_equation
                  Property_Result := While_Loop_Spec_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Others_Designator =>
                  --# property-call-start OthersDesignator.xref_equation
                  Property_Result := Others_Designator_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Pragma_Node =>
                  --# property-call-start Pragma.xref_equation
                  Property_Result := Pragma_Node_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Range_Spec =>
                  --# property-call-start RangeSpec.xref_equation
                  Property_Result := Range_Spec_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Select_When_Part =>
                  --# property-call-start SelectWhenPart.xref_equation
                  Property_Result := Select_When_Part_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Accept_Stmt .. Ada_Accept_Stmt_With_Stmts =>
                  --# property-call-start AcceptStmt.xref_equation
                  Property_Result := Accept_Stmt_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_For_Loop_Stmt .. Ada_While_Loop_Stmt =>
                  --# property-call-start BaseLoopStmt.xref_equation
                  Property_Result := Base_Loop_Stmt_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Begin_Block .. Ada_Decl_Block =>
                  --# property-call-start BlockStmt.xref_equation
                  Property_Result := Block_Stmt_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Case_Stmt =>
                  --# property-call-start CaseStmt.xref_equation
                  Property_Result := Case_Stmt_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Extended_Return_Stmt =>
                  --# property-call-start ExtendedReturnStmt.xref_equation
                  Property_Result := Extended_Return_Stmt_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_If_Stmt =>
                  --# property-call-start IfStmt.xref_equation
                  Property_Result := If_Stmt_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Named_Stmt =>
                  --# property-call-start NamedStmt.xref_equation
                  Property_Result := Named_Stmt_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Select_Stmt =>
                  --# property-call-start SelectStmt.xref_equation
                  Property_Result := Select_Stmt_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Abort_Stmt =>
                  --# property-call-start AbortStmt.xref_equation
                  Property_Result := Abort_Stmt_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Assign_Stmt =>
                  --# property-call-start AssignStmt.xref_equation
                  Property_Result := Assign_Stmt_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Call_Stmt =>
                  --# property-call-start CallStmt.xref_equation
                  Property_Result := Call_Stmt_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Delay_Stmt =>
                  --# property-call-start DelayStmt.xref_equation
                  Property_Result := Delay_Stmt_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Exit_Stmt =>
                  --# property-call-start ExitStmt.xref_equation
                  Property_Result := Exit_Stmt_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Goto_Stmt =>
                  --# property-call-start GotoStmt.xref_equation
                  Property_Result := Goto_Stmt_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Label =>
                  --# property-call-start Label.xref_equation
                  Property_Result := Label_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Null_Stmt =>
                  --# property-call-start NullStmt.xref_equation
                  Property_Result := Null_Stmt_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Raise_Stmt =>
                  --# property-call-start RaiseStmt.xref_equation
                  Property_Result := Raise_Stmt_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Requeue_Stmt =>
                  --# property-call-start RequeueStmt.xref_equation
                  Property_Result := Requeue_Stmt_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Return_Stmt =>
                  --# property-call-start ReturnStmt.xref_equation
                  Property_Result := Return_Stmt_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Terminate_Alternative =>
                  --# property-call-start TerminateAlternative.xref_equation
                  Property_Result := Terminate_Alternative_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Subunit =>
                  --# property-call-start Subunit.xref_equation
                  Property_Result := Subunit_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Task_Def =>
                  --# property-call-start TaskDef.xref_equation
                  Property_Result := Task_Def_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Access_To_Subp_Def =>
                  --# property-call-start AccessToSubpDef.xref_equation
                  Property_Result := Access_To_Subp_Def_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Type_Access_Def =>
                  --# property-call-start TypeAccessDef.xref_equation
                  Property_Result := Type_Access_Def_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Array_Type_Def =>
                  --# property-call-start ArrayTypeDef.xref_equation
                  Property_Result := Array_Type_Def_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Derived_Type_Def =>
                  --# property-call-start DerivedTypeDef.xref_equation
                  Property_Result := Derived_Type_Def_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Enum_Type_Def =>
                  --# property-call-start EnumTypeDef.xref_equation
                  Property_Result := Enum_Type_Def_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Formal_Discrete_Type_Def =>
                  --# property-call-start FormalDiscreteTypeDef.xref_equation
                  Property_Result := Formal_Discrete_Type_Def_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Interface_Type_Def =>
                  --# property-call-start InterfaceTypeDef.xref_equation
                  Property_Result := Interface_Type_Def_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Mod_Int_Type_Def =>
                  --# property-call-start ModIntTypeDef.xref_equation
                  Property_Result := Mod_Int_Type_Def_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Private_Type_Def =>
                  --# property-call-start PrivateTypeDef.xref_equation
                  Property_Result := Private_Type_Def_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Decimal_Fixed_Point_Def =>
                  --# property-call-start DecimalFixedPointDef.xref_equation
                  Property_Result := Decimal_Fixed_Point_Def_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Floating_Point_Def =>
                  --# property-call-start FloatingPointDef.xref_equation
                  Property_Result := Floating_Point_Def_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Ordinary_Fixed_Point_Def =>
                  --# property-call-start OrdinaryFixedPointDef.xref_equation
                  Property_Result := Ordinary_Fixed_Point_Def_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Record_Type_Def =>
                  --# property-call-start RecordTypeDef.xref_equation
                  Property_Result := Record_Type_Def_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Signed_Int_Type_Def =>
                  --# property-call-start SignedIntTypeDef.xref_equation
                  Property_Result := Signed_Int_Type_Def_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Anonymous_Type =>
                  --# property-call-start AnonymousType.xref_equation
                  Property_Result := Anonymous_Type_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Subtype_Indication .. Ada_Discrete_Subtype_Indication =>
                  --# property-call-start SubtypeIndication.xref_equation
                  Property_Result := Subtype_Indication_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Use_Package_Clause =>
                  --# property-call-start UsePackageClause.xref_equation
                  Property_Result := Use_Package_Clause_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Use_Type_Clause =>
                  --# property-call-start UseTypeClause.xref_equation
                  Property_Result := Use_Type_Clause_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Value_Sequence =>
                  --# property-call-start ValueSequence.xref_equation
                  Property_Result := Value_Sequence_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Variant_Part =>
                  --# property-call-start VariantPart.xref_equation
                  Property_Result := Variant_Part_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_With_Clause =>
                  --# property-call-start WithClause.xref_equation
                  Property_Result := With_Clause_P_Xref_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
      end case;
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Dispatcher_Ada_Node_P_Xref_Equation;
--# end
--# property-start '[dispatcher]AdaNode.xref_stop_resolution' dispatcher
pragma Warnings (Off, "is not referenced");
 function Dispatcher_Ada_Node_P_Xref_Stop_Resolution
  (Node : Bare_Ada_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )
   return Boolean
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   --# bind self Self
   Property_Result : Boolean;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
      if Self /= null then
         Reset_Caches (Self.Unit);
      end if;
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dispatching on null node");
      end if;
      case Ada_Ada_Node (Self.Kind) is
               when Ada_Abort_Absent .. Ada_Record_Def | Ada_Composite_Constraint_Assoc | Ada_Param_Assoc .. Ada_Elsif_Stmt_Part | Ada_Handled_Stmts .. Ada_Quantifier_Some | Ada_Renaming_Clause .. Ada_With_Private_Present =>
                  --# property-call-start AdaNode.xref_stop_resolution
                  Property_Result := Ada_Node_P_Xref_Stop_Resolution
                    (Self
                        , E_Info
                    );
                  --# end
               when Ada_Aggregate_Assoc .. Ada_Multi_Dim_Array_Assoc =>
                  --# property-call-start AggregateAssoc.xref_stop_resolution
                  Property_Result := Aggregate_Assoc_P_Xref_Stop_Resolution
                    (Self
                        , E_Info
                    );
                  --# end
               when Ada_Iterated_Assoc =>
                  --# property-call-start IteratedAssoc.xref_stop_resolution
                  Property_Result := Iterated_Assoc_P_Xref_Stop_Resolution
                    (Self
                        , E_Info
                    );
                  --# end
               when Ada_Abstract_State_Decl_Expr .. Ada_Attribute_Ref | Ada_Defining_Name .. Ada_Un_Op =>
                  --# property-call-start Expr.xref_stop_resolution
                  Property_Result := Expr_P_Xref_Stop_Resolution
                    (Self
                        , E_Info
                    );
                  --# end
               when Ada_Call_Expr =>
                  --# property-call-start CallExpr.xref_stop_resolution
                  Property_Result := Call_Expr_P_Xref_Stop_Resolution
                    (Self
                        , E_Info
                    );
                  --# end
               when Ada_Range_Spec =>
                  --# property-call-start RangeSpec.xref_stop_resolution
                  Property_Result := Range_Spec_P_Xref_Stop_Resolution
                    (Self
                        , E_Info
                    );
                  --# end
      end case;
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Dispatcher_Ada_Node_P_Xref_Stop_Resolution;
--# end
--# property-start '[dispatcher]AdaNode.stop_resolution_equation' dispatcher
pragma Warnings (Off, "is not referenced");
 function Dispatcher_Ada_Node_P_Stop_Resolution_Equation
  (Node : Bare_Ada_Node
      ; Env : Lexical_Env
      ; Origin : Bare_Ada_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )
   return Logic_Equation
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   --# bind self Self
   --# bind env Env
   --# bind origin Origin
   Property_Result : Logic_Equation;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
      if Self /= null then
         Reset_Caches (Self.Unit);
      end if;
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dispatching on null node");
      end if;
      case Ada_Ada_Node (Self.Kind) is
               when Ada_Abort_Absent .. Ada_Attribute_Ref | Ada_Defining_Name .. Ada_With_Private_Present =>
                  --# property-call-start AdaNode.stop_resolution_equation
                  Property_Result := Ada_Node_P_Stop_Resolution_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
               when Ada_Call_Expr =>
                  --# property-call-start CallExpr.stop_resolution_equation
                  Property_Result := Call_Expr_P_Stop_Resolution_Equation
                    (Self
                        , Env
                        , Origin
                        , E_Info
                    );
                  --# end
      end case;
   if Self /= null then
      Exit_Call (Self.Unit.Context, Call_Depth);
   end if;
   return Property_Result;
exception
   when others =>
      if Self /= null then
         Exit_Call (Self.Unit.Context, Call_Depth);
      end if;
      raise;
end Dispatcher_Ada_Node_P_Stop_Resolution_Equation;
--# end
--# property-start '[dispatcher]AdaNode.xref_initial_env' dispatcher
pragma Warnings (Off, "is not referenced");
 function Dispatcher_Ada_Node_P_Xref_Initial_Env
  (Node : Bare_Ada_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )
   return Lexical_Env
is
   Self : Bare_Ada_Node := Bare_Ada_Node (Node);
   Call_Depth : aliased Natural;
   Ent : Internal_Entity :=
     Internal_Entity'(Node => Self, Info => E_Info);
   --# bind entity Ent
   Property_Result : Lexical_Env;
begin
   --# property-body-start
   if Self /= null then
      Enter_Call (Self.Unit.Context, Call_Depth'Access);
   end if;
      if Self /= null then
         Reset_Caches (Self.Unit);
      end if;
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dispatching on null node");
      end if;
      case Ada_Ada_Node (Self.Kind) is
               when Ada_Abort_Absent .. Ada_Pp_Then_Kw | Ada_Private_Absent .. Ada_With_Private_Present =>
                  --# property-call-start AdaNode.xref_initial_env
                  Property_Result := Ada_Node_P_Xref_Initial_Env
   