/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// TGUI - Texus' Graphical User Interface
// Copyright (C) 2012-2024 Bruno Van de Velde (vdv_b@tgui.eu)
//
// This software is provided 'as-is', without any express or implied warranty.
// In no event will the authors be held liable for any damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it freely,
// subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented;
//    you must not claim that you wrote the original software.
//    If you use this software in a product, an acknowledgment
//    in the product documentation would be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such,
//    and must not be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source distribution.
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef TGUI_CONFIG_HPP
#define TGUI_CONFIG_HPP

// Config.hpp is generated by CMake, it should not be edited directly.

// Version of the library
#define TGUI_VERSION_MAJOR 1
#define TGUI_VERSION_MINOR 4
#define TGUI_VERSION_PATCH 1

// Detect the platform, to enable platform-specific code
#if defined(_WIN32)
    #define TGUI_SYSTEM_WINDOWS // Windows
#elif defined(__APPLE__) && defined(__MACH__)
    #include "TargetConditionals.h"
    #if TARGET_OS_IPHONE || TARGET_IPHONE_SIMULATOR
        #define TGUI_SYSTEM_IOS // iOS
    #elif TARGET_OS_MAC
        #define TGUI_SYSTEM_MACOS // macOS
    #endif
#elif defined(__unix__)
    #if defined(__ANDROID__)
        #define TGUI_SYSTEM_ANDROID // Android
    #else //if defined(__linux__) || defined(__FreeBSD__) || defined(__FreeBSD_kernel__) || defined(__OpenBSD__)
        #define TGUI_SYSTEM_LINUX // Linux or BSD
    #endif
#endif

// TGUI will link in the same way as SFML, unless TGUI_DYNAMIC or TGUI_STATIC is defined
#if !defined(TGUI_DYNAMIC) && !defined(TGUI_STATIC)
    #ifdef SFML_STATIC
        #define TGUI_STATIC
    #endif
#endif

#ifndef TGUI_STATIC
    #ifdef TGUI_SYSTEM_WINDOWS
        // Windows compilers need specific (and different) keywords for export and import
        #ifdef TGUI_EXPORTS
            #define TGUI_API __declspec(dllexport)
        #else
            #define TGUI_API __declspec(dllimport)
        #endif

		// For Visual C++ compilers, we also need to turn off the annoying C4251 warning
        #ifdef _MSC_VER
            #pragma warning(disable: 4251)
        #endif
    #else
        #define TGUI_API __attribute__ ((__visibility__ ("default")))
    #endif
#else
    // Static build doesn't need import/export macros
    #define TGUI_API
#endif

// Enables code that relies on a specific backend
#define TGUI_HAS_WINDOW_BACKEND_SFML 1
#define TGUI_HAS_WINDOW_BACKEND_SDL 0
#define TGUI_HAS_WINDOW_BACKEND_GLFW 0
#define TGUI_HAS_WINDOW_BACKEND_RAYLIB 0

#define TGUI_HAS_RENDERER_BACKEND_SFML_GRAPHICS 1
#define TGUI_HAS_RENDERER_BACKEND_SDL_RENDERER 0
#define TGUI_HAS_RENDERER_BACKEND_OPENGL3 0
#define TGUI_HAS_RENDERER_BACKEND_GLES2 0
#define TGUI_HAS_RENDERER_BACKEND_RAYLIB 0

#define TGUI_HAS_FONT_BACKEND_SFML_GRAPHICS 1
#define TGUI_HAS_FONT_BACKEND_SDL_TTF 0
#define TGUI_HAS_FONT_BACKEND_FREETYPE 0
#define TGUI_HAS_FONT_BACKEND_RAYLIB 0

#define TGUI_HAS_BACKEND_SFML_GRAPHICS 1
#define TGUI_HAS_BACKEND_SFML_OPENGL3 0
#define TGUI_HAS_BACKEND_SDL_RENDERER 0
#define TGUI_HAS_BACKEND_SDL_OPENGL3 0
#define TGUI_HAS_BACKEND_SDL_GLES2 0
#define TGUI_HAS_BACKEND_SDL_TTF_OPENGL3 0
#define TGUI_HAS_BACKEND_SDL_TTF_GLES2 0
#define TGUI_HAS_BACKEND_GLFW_OPENGL3 0
#define TGUI_HAS_BACKEND_GLFW_GLES2 0
#define TGUI_HAS_BACKEND_RAYLIB 0

#define TGUI_USE_SDL3 0

// Some window backends (SFML < 2.6 and SDL < 3) don't support diagonal mouse cursors on Linux in which case TGUI loads them directly with X11.
// Settig this option to FALSE in CMake will remove the X11 dependency but will cause those cursors to not show up when using them.
#define TGUI_USE_X11 0

// Options to use system-wide dependencies instead of version TGUI ships in its extlibs folder.
// This might be useful for package managers.
#define TGUI_USE_SYSTEM_STB 0
#define TGUI_USE_SYSTEM_GLAD 0
#define TGUI_USE_SYSTEM_NANOSVG 0
#define TGUI_USE_SYSTEM_AURORA 0

// Define that specifies the mininmum c++ support in both the TGUI code and user code.
// This constant can be lower than the actual c++ standard version used to compile with,
// as long as this constant is the same when compiling TGUI and when using the TGUI libs.
#define TGUI_COMPILED_WITH_CPP_VER 17

#define TGUI_BUILD_AS_CXX_MODULE 0

#if TGUI_EXPERIMENTAL_USE_STD_MODULE
    import std;
#endif

// When TGUI_BUILD_AS_CXX_MODULE is set to 1, TGUI_MODULE_EXPORT will be defined as "export" while
// TGUI is being build. This define takes place before this header is included.
#ifndef TGUI_MODULE_EXPORT
    #define TGUI_MODULE_EXPORT
#endif

// Include the version header when c++20 is available to use the library feature-testing macros
#if TGUI_COMPILED_WITH_CPP_VER >= 20
    #include <version>
#elif defined(__has_include)
    // Try including the header even when TGUI itself wasn't compiled with c++20
    #if __has_include(<version>)
        #include <version>
    #endif
#endif

// Enable the use of std::filesystem if TGUI is built with c++17 with a new enough compiler.
// Although GCC and clang supported it before version 9, this is the first version where no
// additional library has to be linked in order to use std::filesystem. This is also the
// reason why we can't rely on __cpp_lib_filesystem for this.
// Xcode 11 supports it on macOS 10.15 only, so we require Xcode 12 which doesn't run on macOS 10.14.
#if TGUI_COMPILED_WITH_CPP_VER >= 17
    #if (defined(__apple_build_version__) && defined(__clang_major__) && (__clang_major__ >= 12)) \
     || (!defined(__apple_build_version__) && defined(__clang_major__) && (__clang_major__ >= 9)) \
     || (defined(__GNUC__) && (__GNUC__ >= 9)) \
     || (defined(_MSC_VER) && (_MSC_VER >= 1914))
        #define TGUI_USE_STD_FILESYSTEM

        // Use clock_cast to convert file time to system time when clock_cast is available
        #if __cpp_lib_chrono >= 201907L
            #define TGUI_USE_STD_FILESYSTEM_FILE_TIME
        #endif
    #endif
#endif

#if __cplusplus >= 201703L
    #define TGUI_EMPLACE_BACK(object, vector) auto& object = vector.emplace_back();
#else
    #define TGUI_EMPLACE_BACK(object, vector) vector.emplace_back(); auto& object = vector.back();
#endif

#ifndef TGUI_NO_DEPRECATED_WARNINGS
    #define TGUI_DEPRECATED(msg) [[deprecated(msg)]]
#else
    #define TGUI_DEPRECATED(msg)
#endif

#if defined(__has_cpp_attribute) && (__cplusplus >= 201703L)
#   if __has_cpp_attribute(nodiscard)
#       define TGUI_NODISCARD [[nodiscard]]
#   else
#        define TGUI_NODISCARD
#   endif
#else
#   define TGUI_NODISCARD
#endif

#if defined(__GNUC__)
    #define TGUI_IGNORE_DEPRECATED_WARNINGS_START \
        _Pragma("GCC diagnostic push") \
        _Pragma("GCC diagnostic ignored \"-Wdeprecated-declarations\"")

    #define TGUI_IGNORE_DEPRECATED_WARNINGS_END _Pragma("GCC diagnostic pop")
#elif defined (_MSC_VER) && (_MSC_VER >= 1927)
    #define TGUI_IGNORE_DEPRECATED_WARNINGS_START \
        _Pragma("warning(push)") \
        _Pragma("warning(disable: 4996)")

    #define TGUI_IGNORE_DEPRECATED_WARNINGS_END _Pragma("warning(pop)")
#elif defined (_MSC_VER)
    #define TGUI_IGNORE_DEPRECATED_WARNINGS_START \
        __pragma(warning(push)) \
        __pragma(warning(disable: 4996))

    #define TGUI_IGNORE_DEPRECATED_WARNINGS_END __pragma(warning(pop))
#endif

#if !defined(NDEBUG) && !defined(TGUI_NO_RUNTIME_WARNINGS)
    #if !TGUI_EXPERIMENTAL_USE_STD_MODULE
        #include <iostream>
    #endif
    #define TGUI_PRINT_WARNING(msg) { std::cerr << "TGUI warning: " << msg << "\n"; }
#else
    #define TGUI_PRINT_WARNING(msg)
#endif

#if !defined(NDEBUG) && !defined(TGUI_DISABLE_ASSERTS)
    #include <cassert>
    #if !TGUI_EXPERIMENTAL_USE_STD_MODULE
        #include <iostream>
        #include <exception>
    #endif
    #define TGUI_ASSERT(condition, msg) { \
        if (!(condition)) { \
            std::cerr << "TGUI assertion: " << msg << "\n"; \
            assert(condition); \
            std::terminate(); \
        } \
    }
#else
    #define TGUI_ASSERT(condition, msg)
#endif

// Using [=] gives a warning in c++20, but using [=,this] may not compile with older c++ versions
#if __cplusplus > 201703L
    #define TGUI_LAMBDA_CAPTURE_EQ_THIS [=,this]
#else
    #define TGUI_LAMBDA_CAPTURE_EQ_THIS [=]
#endif

#if defined(__cpp_if_constexpr) && (__cpp_if_constexpr >= 201606L)
    #define TGUI_IF_CONSTEXPR if constexpr
#else
    #define TGUI_IF_CONSTEXPR if

    // Disable warning in some Visual Studio versions about being able to use "if constexpr",
    // which we aren't doing because these VS versions didn't define __cpp_if_constexpr yet.
    #if defined TGUI_SYSTEM_WINDOWS && defined _MSC_VER
        #pragma warning(disable:4127)
    #endif
#endif

#endif // TGUI_CONFIG_HPP
