%feature("docstring") OT::CalibrationResult
"Calibration result.

Returned by calibration algorithms, see :class:`~openturns.CalibrationAlgorithm`.

Parameters
----------
parameterPrior : :class:`~openturns.Distribution`
    The prior distribution of the parameter.
parameterPosterior : :class:`~openturns.Distribution`
    The posterior distribution of the parameter.
parameterMap : sequence of float
    The maximum a posteriori estimate of the parameter.
observationsError : :class:`~openturns.Distribution`
    The distribution of the observations error.
inputObservations : :class:`~openturns.Sample`
    The sample of input observations.
outputObservations : :class:`~openturns.Sample`
    The sample of output observations.
residualFunction : :class:`~openturns.Function`
    The residual function.
bayesian : bool
    Whether the method is Bayesian

Notes
-----
The residual function returns `model(inputObservations) - outputObservations`.

Examples
--------
>>> import openturns as ot

# assume we obtained a result from CalibrationAlgorithm

>>> result = ot.CalibrationResult()
>>> pmap = result.getParameterMAP()
>>> prior = result.getParameterPrior()
>>> posterior = result.getParameterPosterior()
>>> graph1 = result.drawParameterDistributions()  # doctest: +SKIP
>>> graph2 = result.drawResiduals()  # doctest: +SKIP
>>> graph3 = result.drawObservationsVsInputs()  # doctest: +SKIP
>>> graph4 = result.drawObservationsVsPredictions()  # doctest: +SKIP
"

// ---------------------------------------------------------------------

%feature("docstring") OT::CalibrationResult::getParameterPrior
"Accessor to the parameter prior distribution.

Returns
-------
parameterPrior : :class:`~openturns.Distribution`
    The prior distribution of the parameter."

// ---------------------------------------------------------------------

%feature("docstring") OT::CalibrationResult::setParameterPrior
"Accessor to the parameter prior distribution.

Parameters
----------
parameterPrior : : :class:`~openturns.Distribution`
    The prior distribution of the parameter."

// ---------------------------------------------------------------------

%feature("docstring") OT::CalibrationResult::getParameterPosterior
"Accessor to the parameter posterior distribution.

The content of the posterior distribution depends of the class that
created this distribution.
The next table presents this distribution depending on the
algorithm.

+------------------------------------------------------+-----------------------------------------------------------------------+
| Class                                                | Distribution                                                          |
+======================================================+=======================================================================+
| :class:`~openturns.LinearLeastSquaresCalibration`    | Distribution of the parameters with respect to randomness in the data |
+------------------------------------------------------+-----------------------------------------------------------------------+
| :class:`~openturns.NonLinearLeastSquaresCalibration` | Distribution of the parameters with respect to randomness in the data |
+------------------------------------------------------+-----------------------------------------------------------------------+
| :class:`~openturns.GaussianLinearCalibration`        | Posterior distribution of the parameters given the observations       |
+------------------------------------------------------+-----------------------------------------------------------------------+
| :class:`~openturns.GaussianNonLinearCalibration`     | Distribution of the MAP with respect to randomness in the data        |
+------------------------------------------------------+-----------------------------------------------------------------------+

**Table 1.** Content of the distribution returned by  :meth:`~openturns.CalibrationResult.getParameterPosterior()` different classes returning a :class:`~openturns.CalibrationResult`.


Returns
-------
parameterPosterior : :class:`~openturns.Distribution`
    The posterior distribution of the parameter."

// ---------------------------------------------------------------------

%feature("docstring") OT::CalibrationResult::setParameterPosterior
"Accessor to the parameter posterior distribution.

Parameters
----------
parameterPosterior : : :class:`~openturns.Distribution`
    The posterior distribution of the parameter."

// ---------------------------------------------------------------------

%feature("docstring") OT::CalibrationResult::getObservationsError
"Accessor to the observations error distribution.

Returns
-------
observationsError : :class:`~openturns.Distribution`
    The observations error distribution."

// ---------------------------------------------------------------------

%feature("docstring") OT::CalibrationResult::setObservationsError
"Accessor to the observations error distribution.

Parameters
----------
observationsError : :class:`~openturns.Distribution`
    The observations error distribution."

// ---------------------------------------------------------------------

%feature("docstring") OT::CalibrationResult::getParameterMAP
"Accessor to the maximum a posteriori parameter estimate.

Returns
-------
parameterPosterior : :class:`~openturns.Point`
    The maximum a posteriori parameter estimate."

// ---------------------------------------------------------------------

%feature("docstring") OT::CalibrationResult::setParameterMAP
"Accessor to the maximum a posteriori parameter estimate.

Parameters
----------
parameterPosterior : sequence of float
    The maximum a posteriori parameter estimate."

// ---------------------------------------------------------------------

%feature("docstring") OT::CalibrationResult::getInputObservations
"Accessor to the input observations.

Returns
-------
inputObservations : :class:`~openturns.Sample`
    The sample of input observations."

// ---------------------------------------------------------------------

%feature("docstring") OT::CalibrationResult::setInputObservations
"Accessor to the input observations.

Parameters
----------
inputObservations : :class:`~openturns.Sample`
    The sample of input observations."

// ---------------------------------------------------------------------

%feature("docstring") OT::CalibrationResult::getOutputObservations
"Accessor to the output observations.

Returns
-------
outputObservations : :class:`~openturns.Sample`
    The sample of output observations."

// ---------------------------------------------------------------------

%feature("docstring") OT::CalibrationResult::setOutputObservations
"Accessor to the output observations.

Parameters
----------
outputObservations : :class:`~openturns.Sample`
    The sample of output observations."

// ---------------------------------------------------------------------

%feature("docstring") OT::CalibrationResult::getResidualFunction
"Accessor to the residual function.

Returns
-------
residualFunction : :class:`~openturns.Function`
    The residual function."

// ---------------------------------------------------------------------

%feature("docstring") OT::CalibrationResult::setResidualFunction
"Accessor to the residual function.

Parameters
----------
residualFunction : :class:`~openturns.Function`
    The residual function."

// ---------------------------------------------------------------------

%feature("docstring") OT::CalibrationResult::setOutputAtPriorAndPosteriorMean
"Accessor to the output at prior/posterior mean.

Parameters
----------
outputAtPrior : :class:`~openturns.Sample`
    The sample of output at prior mean.
outputAtPosterior : :class:`~openturns.Sample`
    The sample of output at posterior mean."

// ---------------------------------------------------------------------

%feature("docstring") OT::CalibrationResult::getOutputAtPriorMean
"Accessor to the output observations.

Returns
-------
outputAtPrior : :class:`~openturns.Sample`
    Output at prior mean."

// ---------------------------------------------------------------------

%feature("docstring") OT::CalibrationResult::getOutputAtPosteriorMean
"Accessor to the output observations.

Returns
-------
outputAtPosterior : :class:`~openturns.Sample`
    Output at posterior mean."

// ---------------------------------------------------------------------

%feature("docstring") OT::CalibrationResult::drawParameterDistributions
"Draw parameter prior/posterior.

Plots the prior and posterior distribution of the calibrated parameter theta.

Returns
-------
grid : :class:`~openturns.GridLayout`
    Graph array."

// ---------------------------------------------------------------------

%feature("docstring") OT::CalibrationResult::drawResiduals
"Draw residuals.

Plot the distribution of the sample residuals before and after calibration 
using kernel smoothing and the distribution of the observation errors.

Returns
-------
grid : :class:`~openturns.GridLayout`
    Graph array."

// ---------------------------------------------------------------------

%feature("docstring") OT::CalibrationResult::drawObservationsVsInputs
"Draw observations/inputs.

Plot the observed output of the model depending on the observed input before
and after calibration.

Returns
-------
grid : :class:`~openturns.GridLayout`
    Graph array."

// ---------------------------------------------------------------------

%feature("docstring") OT::CalibrationResult::drawObservationsVsPredictions
"Draw observations/predictions.

Plots the output of the model depending on the output observations before and
after calibration.

Returns
-------
grid : :class:`~openturns.GridLayout`
    Graph array."

// ---------------------------------------------------------------------

%feature("docstring") OT::CalibrationResult::drawResidualsNormalPlot
"Draw residuals normal plot.

Plots the quantile-quantile graphs of the empirical residual 
after calibration vs the Gaussian distribution.

Returns
-------
grid : :class:`~openturns.GridLayout`
    Graph array."

// ---------------------------------------------------------------------

%feature("docstring") OT::CalibrationResult::isBayesian
"Bayesian method accessor.

Returns
-------
bayesian : bool
    Whether the method is Bayesian"
