/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { describe, it, expect, vi, beforeEach } from 'vitest';
import { Config, ApprovalMode } from './config.js';
import * as path from 'node:path';
import { setGeminiMdFilename as mockSetGeminiMdFilename } from '../tools/memoryTool.js';
import { DEFAULT_TELEMETRY_TARGET, DEFAULT_OTLP_ENDPOINT, } from '../telemetry/index.js';
import { AuthType, createContentGeneratorConfig, } from '../core/contentGenerator.js';
import { GeminiClient } from '../core/client.js';
import { GitService } from '../services/gitService.js';
vi.mock('fs', async (importOriginal) => {
    const actual = await importOriginal();
    return {
        ...actual,
        existsSync: vi.fn().mockReturnValue(true),
        statSync: vi.fn().mockReturnValue({
            isDirectory: vi.fn().mockReturnValue(true),
        }),
        realpathSync: vi.fn((path) => path),
    };
});
import { ShellTool } from '../tools/shell.js';
import { ReadFileTool } from '../tools/read-file.js';
vi.mock('fs', async (importOriginal) => {
    const actual = await importOriginal();
    return {
        ...actual,
        existsSync: vi.fn().mockReturnValue(true),
        statSync: vi.fn().mockReturnValue({
            isDirectory: vi.fn().mockReturnValue(true),
        }),
        realpathSync: vi.fn((path) => path),
    };
});
// Mock dependencies that might be called during Config construction or createServerConfig
vi.mock('../tools/tool-registry', () => {
    const ToolRegistryMock = vi.fn();
    ToolRegistryMock.prototype.registerTool = vi.fn();
    ToolRegistryMock.prototype.discoverAllTools = vi.fn();
    ToolRegistryMock.prototype.getAllTools = vi.fn(() => []); // Mock methods if needed
    ToolRegistryMock.prototype.getTool = vi.fn();
    ToolRegistryMock.prototype.getFunctionDeclarations = vi.fn(() => []);
    return { ToolRegistry: ToolRegistryMock };
});
vi.mock('../utils/memoryDiscovery.js', () => ({
    loadServerHierarchicalMemory: vi.fn(),
}));
// Mock individual tools if their constructors are complex or have side effects
vi.mock('../tools/ls');
vi.mock('../tools/read-file');
vi.mock('../tools/grep');
vi.mock('../tools/glob');
vi.mock('../tools/edit');
vi.mock('../tools/shell');
vi.mock('../tools/write-file');
vi.mock('../tools/web-fetch');
vi.mock('../tools/read-many-files');
vi.mock('../tools/memoryTool', () => ({
    MemoryTool: vi.fn(),
    setGeminiMdFilename: vi.fn(),
    getCurrentGeminiMdFilename: vi.fn(() => 'QWEN.md'), // Mock the original filename
    DEFAULT_CONTEXT_FILENAME: 'QWEN.md',
    GEMINI_CONFIG_DIR: '.gemini',
}));
vi.mock('../core/contentGenerator.js', async (importOriginal) => {
    const actual = await importOriginal();
    return {
        ...actual,
        createContentGeneratorConfig: vi.fn(),
    };
});
vi.mock('../core/client.js', () => ({
    GeminiClient: vi.fn().mockImplementation(() => ({
        initialize: vi.fn().mockResolvedValue(undefined),
    })),
}));
vi.mock('../telemetry/index.js', async (importOriginal) => {
    const actual = await importOriginal();
    return {
        ...actual,
        initializeTelemetry: vi.fn(),
    };
});
vi.mock('../services/gitService.js', () => {
    const GitServiceMock = vi.fn();
    GitServiceMock.prototype.initialize = vi.fn();
    return { GitService: GitServiceMock };
});
vi.mock('../ide/ide-client.js', () => ({
    IdeClient: {
        getInstance: vi.fn().mockResolvedValue({
            getConnectionStatus: vi.fn(),
            initialize: vi.fn(),
            shutdown: vi.fn(),
        }),
    },
}));
describe('Server Config (config.ts)', () => {
    const MODEL = 'gemini-pro';
    const SANDBOX = {
        command: 'docker',
        image: 'qwen-code-sandbox',
    };
    const TARGET_DIR = '/path/to/target';
    const DEBUG_MODE = false;
    const QUESTION = 'test question';
    const FULL_CONTEXT = false;
    const USER_MEMORY = 'Test User Memory';
    const TELEMETRY_SETTINGS = { enabled: false };
    const EMBEDDING_MODEL = 'gemini-embedding';
    const SESSION_ID = 'test-session-id';
    const baseParams = {
        cwd: '/tmp',
        embeddingModel: EMBEDDING_MODEL,
        sandbox: SANDBOX,
        targetDir: TARGET_DIR,
        debugMode: DEBUG_MODE,
        question: QUESTION,
        fullContext: FULL_CONTEXT,
        userMemory: USER_MEMORY,
        telemetry: TELEMETRY_SETTINGS,
        sessionId: SESSION_ID,
        model: MODEL,
        usageStatisticsEnabled: false,
    };
    beforeEach(() => {
        // Reset mocks if necessary
        vi.clearAllMocks();
    });
    describe('initialize', () => {
        it('should throw an error if checkpointing is enabled and GitService fails', async () => {
            const gitError = new Error('Git is not installed');
            GitService.prototype.initialize.mockRejectedValue(gitError);
            const config = new Config({
                ...baseParams,
                checkpointing: true,
            });
            await expect(config.initialize()).rejects.toThrow(gitError);
        });
        it('should not throw an error if checkpointing is disabled and GitService fails', async () => {
            const gitError = new Error('Git is not installed');
            GitService.prototype.initialize.mockRejectedValue(gitError);
            const config = new Config({
                ...baseParams,
                checkpointing: false,
            });
            await expect(config.initialize()).resolves.toBeUndefined();
        });
        it('should throw an error if initialized more than once', async () => {
            const config = new Config({
                ...baseParams,
                checkpointing: false,
            });
            await expect(config.initialize()).resolves.toBeUndefined();
            await expect(config.initialize()).rejects.toThrow('Config was already initialized');
        });
    });
    describe('refreshAuth', () => {
        it('should refresh auth and update config', async () => {
            const config = new Config(baseParams);
            const authType = AuthType.USE_GEMINI;
            const newModel = 'gemini-flash';
            const mockContentConfig = {
                model: newModel,
                apiKey: 'test-key',
            };
            createContentGeneratorConfig.mockReturnValue(mockContentConfig);
            // Set fallback mode to true to ensure it gets reset
            config.setFallbackMode(true);
            expect(config.isInFallbackMode()).toBe(true);
            await config.refreshAuth(authType);
            expect(createContentGeneratorConfig).toHaveBeenCalledWith(config, authType);
            // Verify that contentGeneratorConfig is updated with the new model
            expect(config.getContentGeneratorConfig()).toEqual(mockContentConfig);
            expect(config.getContentGeneratorConfig().model).toBe(newModel);
            expect(config.getModel()).toBe(newModel); // getModel() should return the updated model
            expect(GeminiClient).toHaveBeenCalledWith(config);
            // Verify that fallback mode is reset
            expect(config.isInFallbackMode()).toBe(false);
        });
        it('should preserve conversation history when refreshing auth', async () => {
            const config = new Config(baseParams);
            const authType = AuthType.USE_GEMINI;
            const mockContentConfig = {
                model: 'gemini-pro',
                apiKey: 'test-key',
            };
            createContentGeneratorConfig.mockReturnValue(mockContentConfig);
            // Mock the existing client with some history
            const mockExistingHistory = [
                { role: 'user', parts: [{ text: 'Hello' }] },
                { role: 'model', parts: [{ text: 'Hi there!' }] },
                { role: 'user', parts: [{ text: 'How are you?' }] },
            ];
            const mockExistingClient = {
                isInitialized: vi.fn().mockReturnValue(true),
                getHistory: vi.fn().mockReturnValue(mockExistingHistory),
            };
            const mockNewClient = {
                isInitialized: vi.fn().mockReturnValue(true),
                getHistory: vi.fn().mockReturnValue([]),
                setHistory: vi.fn(),
                initialize: vi.fn().mockResolvedValue(undefined),
            };
            // Set the existing client
            config.geminiClient = mockExistingClient;
            GeminiClient.mockImplementation(() => mockNewClient);
            await config.refreshAuth(authType);
            // Verify that existing history was retrieved
            expect(mockExistingClient.getHistory).toHaveBeenCalled();
            // Verify that new client was created and initialized
            expect(GeminiClient).toHaveBeenCalledWith(config);
            expect(mockNewClient.initialize).toHaveBeenCalledWith(mockContentConfig);
            // Verify that history was restored to the new client
            expect(mockNewClient.setHistory).toHaveBeenCalledWith(mockExistingHistory, { stripThoughts: false });
        });
        it('should handle case when no existing client is initialized', async () => {
            const config = new Config(baseParams);
            const authType = AuthType.USE_GEMINI;
            const mockContentConfig = {
                model: 'gemini-pro',
                apiKey: 'test-key',
            };
            createContentGeneratorConfig.mockReturnValue(mockContentConfig);
            const mockNewClient = {
                isInitialized: vi.fn().mockReturnValue(true),
                getHistory: vi.fn().mockReturnValue([]),
                setHistory: vi.fn(),
                initialize: vi.fn().mockResolvedValue(undefined),
            };
            // No existing client
            config.geminiClient = null;
            GeminiClient.mockImplementation(() => mockNewClient);
            await config.refreshAuth(authType);
            // Verify that new client was created and initialized
            expect(GeminiClient).toHaveBeenCalledWith(config);
            expect(mockNewClient.initialize).toHaveBeenCalledWith(mockContentConfig);
            // Verify that setHistory was not called since there was no existing history
            expect(mockNewClient.setHistory).not.toHaveBeenCalled();
        });
        it('should strip thoughts when switching from GenAI to Vertex', async () => {
            const config = new Config(baseParams);
            const mockContentConfig = {
                model: 'gemini-pro',
                apiKey: 'test-key',
                authType: AuthType.USE_GEMINI,
            };
            config.contentGeneratorConfig = mockContentConfig;
            createContentGeneratorConfig.mockReturnValue({
                ...mockContentConfig,
                authType: AuthType.LOGIN_WITH_GOOGLE,
            });
            const mockExistingHistory = [
                { role: 'user', parts: [{ text: 'Hello' }] },
            ];
            const mockExistingClient = {
                isInitialized: vi.fn().mockReturnValue(true),
                getHistory: vi.fn().mockReturnValue(mockExistingHistory),
            };
            const mockNewClient = {
                isInitialized: vi.fn().mockReturnValue(true),
                getHistory: vi.fn().mockReturnValue([]),
                setHistory: vi.fn(),
                initialize: vi.fn().mockResolvedValue(undefined),
            };
            config.geminiClient = mockExistingClient;
            GeminiClient.mockImplementation(() => mockNewClient);
            await config.refreshAuth(AuthType.LOGIN_WITH_GOOGLE);
            expect(mockNewClient.setHistory).toHaveBeenCalledWith(mockExistingHistory, { stripThoughts: true });
        });
        it('should not strip thoughts when switching from Vertex to GenAI', async () => {
            const config = new Config(baseParams);
            const mockContentConfig = {
                model: 'gemini-pro',
                apiKey: 'test-key',
                authType: AuthType.LOGIN_WITH_GOOGLE,
            };
            config.contentGeneratorConfig = mockContentConfig;
            createContentGeneratorConfig.mockReturnValue({
                ...mockContentConfig,
                authType: AuthType.USE_GEMINI,
            });
            const mockExistingHistory = [
                { role: 'user', parts: [{ text: 'Hello' }] },
            ];
            const mockExistingClient = {
                isInitialized: vi.fn().mockReturnValue(true),
                getHistory: vi.fn().mockReturnValue(mockExistingHistory),
            };
            const mockNewClient = {
                isInitialized: vi.fn().mockReturnValue(true),
                getHistory: vi.fn().mockReturnValue([]),
                setHistory: vi.fn(),
                initialize: vi.fn().mockResolvedValue(undefined),
            };
            config.geminiClient = mockExistingClient;
            GeminiClient.mockImplementation(() => mockNewClient);
            await config.refreshAuth(AuthType.USE_GEMINI);
            expect(mockNewClient.setHistory).toHaveBeenCalledWith(mockExistingHistory, { stripThoughts: false });
        });
    });
    it('Config constructor should store userMemory correctly', () => {
        const config = new Config(baseParams);
        expect(config.getUserMemory()).toBe(USER_MEMORY);
        // Verify other getters if needed
        expect(config.getTargetDir()).toBe(path.resolve(TARGET_DIR)); // Check resolved path
    });
    it('Config constructor should default userMemory to empty string if not provided', () => {
        const paramsWithoutMemory = { ...baseParams };
        delete paramsWithoutMemory.userMemory;
        const config = new Config(paramsWithoutMemory);
        expect(config.getUserMemory()).toBe('');
    });
    it('Config constructor should call setGeminiMdFilename with contextFileName if provided', () => {
        const contextFileName = 'CUSTOM_AGENTS.md';
        const paramsWithContextFile = {
            ...baseParams,
            contextFileName,
        };
        new Config(paramsWithContextFile);
        expect(mockSetGeminiMdFilename).toHaveBeenCalledWith(contextFileName);
    });
    it('Config constructor should not call setGeminiMdFilename if contextFileName is not provided', () => {
        new Config(baseParams); // baseParams does not have contextFileName
        expect(mockSetGeminiMdFilename).not.toHaveBeenCalled();
    });
    it('should set default file filtering settings when not provided', () => {
        const config = new Config(baseParams);
        expect(config.getFileFilteringRespectGitIgnore()).toBe(true);
    });
    it('should set custom file filtering settings when provided', () => {
        const paramsWithFileFiltering = {
            ...baseParams,
            fileFiltering: {
                respectGitIgnore: false,
            },
        };
        const config = new Config(paramsWithFileFiltering);
        expect(config.getFileFilteringRespectGitIgnore()).toBe(false);
    });
    it('should initialize WorkspaceContext with includeDirectories', () => {
        const includeDirectories = ['/path/to/dir1', '/path/to/dir2'];
        const paramsWithIncludeDirs = {
            ...baseParams,
            includeDirectories,
        };
        const config = new Config(paramsWithIncludeDirs);
        const workspaceContext = config.getWorkspaceContext();
        const directories = workspaceContext.getDirectories();
        // Should include the target directory plus the included directories
        expect(directories).toHaveLength(3);
        expect(directories).toContain(path.resolve(baseParams.targetDir));
        expect(directories).toContain('/path/to/dir1');
        expect(directories).toContain('/path/to/dir2');
    });
    it('Config constructor should set telemetry to true when provided as true', () => {
        const paramsWithTelemetry = {
            ...baseParams,
            telemetry: { enabled: true },
        };
        const config = new Config(paramsWithTelemetry);
        expect(config.getTelemetryEnabled()).toBe(true);
    });
    it('Config constructor should set telemetry to false when provided as false', () => {
        const paramsWithTelemetry = {
            ...baseParams,
            telemetry: { enabled: false },
        };
        const config = new Config(paramsWithTelemetry);
        expect(config.getTelemetryEnabled()).toBe(false);
    });
    it('Config constructor should default telemetry to default value if not provided', () => {
        const paramsWithoutTelemetry = { ...baseParams };
        delete paramsWithoutTelemetry.telemetry;
        const config = new Config(paramsWithoutTelemetry);
        expect(config.getTelemetryEnabled()).toBe(TELEMETRY_SETTINGS.enabled);
    });
    it('should have a getFileService method that returns FileDiscoveryService', () => {
        const config = new Config(baseParams);
        const fileService = config.getFileService();
        expect(fileService).toBeDefined();
    });
    describe('Usage Statistics', () => {
        it('defaults usage statistics to enabled if not specified', () => {
            const config = new Config({
                ...baseParams,
                usageStatisticsEnabled: undefined,
            });
            expect(config.getUsageStatisticsEnabled()).toBe(true);
        });
        it.each([{ enabled: true }, { enabled: false }])('sets usage statistics based on the provided value (enabled: $enabled)', ({ enabled }) => {
            const config = new Config({
                ...baseParams,
                usageStatisticsEnabled: enabled,
            });
            expect(config.getUsageStatisticsEnabled()).toBe(enabled);
        });
    });
    describe('Telemetry Settings', () => {
        it('should return default telemetry target if not provided', () => {
            const params = {
                ...baseParams,
                telemetry: { enabled: true },
            };
            const config = new Config(params);
            expect(config.getTelemetryTarget()).toBe(DEFAULT_TELEMETRY_TARGET);
        });
        it('should return provided OTLP endpoint', () => {
            const endpoint = 'http://custom.otel.collector:4317';
            const params = {
                ...baseParams,
                telemetry: { enabled: true, otlpEndpoint: endpoint },
            };
            const config = new Config(params);
            expect(config.getTelemetryOtlpEndpoint()).toBe(endpoint);
        });
        it('should return default OTLP endpoint if not provided', () => {
            const params = {
                ...baseParams,
                telemetry: { enabled: true },
            };
            const config = new Config(params);
            expect(config.getTelemetryOtlpEndpoint()).toBe(DEFAULT_OTLP_ENDPOINT);
        });
        it('should return provided logPrompts setting', () => {
            const params = {
                ...baseParams,
                telemetry: { enabled: true, logPrompts: false },
            };
            const config = new Config(params);
            expect(config.getTelemetryLogPromptsEnabled()).toBe(false);
        });
        it('should return default logPrompts setting (true) if not provided', () => {
            const params = {
                ...baseParams,
                telemetry: { enabled: true },
            };
            const config = new Config(params);
            expect(config.getTelemetryLogPromptsEnabled()).toBe(true);
        });
        it('should return default logPrompts setting (true) if telemetry object is not provided', () => {
            const paramsWithoutTelemetry = { ...baseParams };
            delete paramsWithoutTelemetry.telemetry;
            const config = new Config(paramsWithoutTelemetry);
            expect(config.getTelemetryLogPromptsEnabled()).toBe(true);
        });
        it('should return default telemetry target if telemetry object is not provided', () => {
            const paramsWithoutTelemetry = { ...baseParams };
            delete paramsWithoutTelemetry.telemetry;
            const config = new Config(paramsWithoutTelemetry);
            expect(config.getTelemetryTarget()).toBe(DEFAULT_TELEMETRY_TARGET);
        });
        it('should return default OTLP endpoint if telemetry object is not provided', () => {
            const paramsWithoutTelemetry = { ...baseParams };
            delete paramsWithoutTelemetry.telemetry;
            const config = new Config(paramsWithoutTelemetry);
            expect(config.getTelemetryOtlpEndpoint()).toBe(DEFAULT_OTLP_ENDPOINT);
        });
        it('should return provided OTLP protocol', () => {
            const params = {
                ...baseParams,
                telemetry: { enabled: true, otlpProtocol: 'http' },
            };
            const config = new Config(params);
            expect(config.getTelemetryOtlpProtocol()).toBe('http');
        });
        it('should return default OTLP protocol if not provided', () => {
            const params = {
                ...baseParams,
                telemetry: { enabled: true },
            };
            const config = new Config(params);
            expect(config.getTelemetryOtlpProtocol()).toBe('grpc');
        });
        it('should return default OTLP protocol if telemetry object is not provided', () => {
            const paramsWithoutTelemetry = { ...baseParams };
            delete paramsWithoutTelemetry.telemetry;
            const config = new Config(paramsWithoutTelemetry);
            expect(config.getTelemetryOtlpProtocol()).toBe('grpc');
        });
    });
    describe('UseRipgrep Configuration', () => {
        it('should default useRipgrep to false when not provided', () => {
            const config = new Config(baseParams);
            expect(config.getUseRipgrep()).toBe(false);
        });
        it('should set useRipgrep to true when provided as true', () => {
            const paramsWithRipgrep = {
                ...baseParams,
                useRipgrep: true,
            };
            const config = new Config(paramsWithRipgrep);
            expect(config.getUseRipgrep()).toBe(true);
        });
        it('should set useRipgrep to false when explicitly provided as false', () => {
            const paramsWithRipgrep = {
                ...baseParams,
                useRipgrep: false,
            };
            const config = new Config(paramsWithRipgrep);
            expect(config.getUseRipgrep()).toBe(false);
        });
        it('should default useRipgrep to false when undefined', () => {
            const paramsWithUndefinedRipgrep = {
                ...baseParams,
                useRipgrep: undefined,
            };
            const config = new Config(paramsWithUndefinedRipgrep);
            expect(config.getUseRipgrep()).toBe(false);
        });
    });
    describe('createToolRegistry', () => {
        it('should register a tool if coreTools contains an argument-specific pattern', async () => {
            const params = {
                ...baseParams,
                coreTools: ['ShellTool(git status)'],
            };
            const config = new Config(params);
            await config.initialize();
            // The ToolRegistry class is mocked, so we can inspect its prototype's methods.
            const registerToolMock = (await vi.importMock('../tools/tool-registry')).ToolRegistry.prototype.registerTool;
            // Check that registerTool was called for ShellTool
            const wasShellToolRegistered = registerToolMock.mock.calls.some((call) => call[0] instanceof vi.mocked(ShellTool));
            expect(wasShellToolRegistered).toBe(true);
            // Check that registerTool was NOT called for ReadFileTool
            const wasReadFileToolRegistered = registerToolMock.mock.calls.some((call) => call[0] instanceof vi.mocked(ReadFileTool));
            expect(wasReadFileToolRegistered).toBe(false);
        });
    });
});
describe('setApprovalMode with folder trust', () => {
    it('should throw an error when setting YOLO mode in an untrusted folder', () => {
        const config = new Config({
            sessionId: 'test',
            targetDir: '.',
            debugMode: false,
            model: 'test-model',
            cwd: '.',
            trustedFolder: false, // Untrusted
        });
        expect(() => config.setApprovalMode(ApprovalMode.YOLO)).toThrow('Cannot enable privileged approval modes in an untrusted folder.');
    });
    it('should throw an error when setting AUTO_EDIT mode in an untrusted folder', () => {
        const config = new Config({
            sessionId: 'test',
            targetDir: '.',
            debugMode: false,
            model: 'test-model',
            cwd: '.',
            trustedFolder: false, // Untrusted
        });
        expect(() => config.setApprovalMode(ApprovalMode.AUTO_EDIT)).toThrow('Cannot enable privileged approval modes in an untrusted folder.');
    });
    it('should NOT throw an error when setting DEFAULT mode in an untrusted folder', () => {
        const config = new Config({
            sessionId: 'test',
            targetDir: '.',
            debugMode: false,
            model: 'test-model',
            cwd: '.',
            trustedFolder: false, // Untrusted
        });
        expect(() => config.setApprovalMode(ApprovalMode.DEFAULT)).not.toThrow();
    });
    it('should NOT throw an error when setting PLAN mode in an untrusted folder', () => {
        const config = new Config({
            sessionId: 'test',
            targetDir: '.',
            debugMode: false,
            model: 'test-model',
            cwd: '.',
            trustedFolder: false, // Untrusted
        });
        expect(() => config.setApprovalMode(ApprovalMode.PLAN)).not.toThrow();
    });
    it('should NOT throw an error when setting any mode in a trusted folder', () => {
        const config = new Config({
            sessionId: 'test',
            targetDir: '.',
            debugMode: false,
            model: 'test-model',
            cwd: '.',
            trustedFolder: true, // Trusted
        });
        expect(() => config.setApprovalMode(ApprovalMode.YOLO)).not.toThrow();
        expect(() => config.setApprovalMode(ApprovalMode.AUTO_EDIT)).not.toThrow();
        expect(() => config.setApprovalMode(ApprovalMode.DEFAULT)).not.toThrow();
        expect(() => config.setApprovalMode(ApprovalMode.PLAN)).not.toThrow();
    });
    it('should NOT throw an error when setting any mode if trustedFolder is undefined', () => {
        const config = new Config({
            sessionId: 'test',
            targetDir: '.',
            debugMode: false,
            model: 'test-model',
            cwd: '.',
            trustedFolder: undefined, // Undefined
        });
        expect(() => config.setApprovalMode(ApprovalMode.YOLO)).not.toThrow();
        expect(() => config.setApprovalMode(ApprovalMode.AUTO_EDIT)).not.toThrow();
        expect(() => config.setApprovalMode(ApprovalMode.DEFAULT)).not.toThrow();
        expect(() => config.setApprovalMode(ApprovalMode.PLAN)).not.toThrow();
    });
    describe('Model Switch Logging', () => {
        it('should log model switch when setModel is called with different model', async () => {
            const config = new Config({
                sessionId: 'test-model-switch',
                targetDir: '.',
                debugMode: false,
                model: 'qwen3-coder-plus',
                cwd: '.',
            });
            // Initialize the config to set up content generator
            await config.initialize();
            // Mock the logger's logModelSwitch method
            const logModelSwitchSpy = vi.spyOn(config['logger'], 'logModelSwitch');
            // Change the model
            await config.setModel('qwen-vl-max-latest', {
                reason: 'vision_auto_switch',
                context: 'Test model switch',
            });
            // Verify that logModelSwitch was called with correct parameters
            expect(logModelSwitchSpy).toHaveBeenCalledWith({
                fromModel: 'qwen3-coder-plus',
                toModel: 'qwen-vl-max-latest',
                reason: 'vision_auto_switch',
                context: 'Test model switch',
            });
        });
        it('should not log when setModel is called with same model', async () => {
            const config = new Config({
                sessionId: 'test-same-model',
                targetDir: '.',
                debugMode: false,
                model: 'qwen3-coder-plus',
                cwd: '.',
            });
            // Initialize the config to set up content generator
            await config.initialize();
            // Mock the logger's logModelSwitch method
            const logModelSwitchSpy = vi.spyOn(config['logger'], 'logModelSwitch');
            // Set the same model
            await config.setModel('qwen3-coder-plus');
            // Verify that logModelSwitch was not called
            expect(logModelSwitchSpy).not.toHaveBeenCalled();
        });
        it('should use default reason when no options provided', async () => {
            const config = new Config({
                sessionId: 'test-default-reason',
                targetDir: '.',
                debugMode: false,
                model: 'qwen3-coder-plus',
                cwd: '.',
            });
            // Initialize the config to set up content generator
            await config.initialize();
            // Mock the logger's logModelSwitch method
            const logModelSwitchSpy = vi.spyOn(config['logger'], 'logModelSwitch');
            // Change the model without options
            await config.setModel('qwen-vl-max-latest');
            // Verify that logModelSwitch was called with default reason
            expect(logModelSwitchSpy).toHaveBeenCalledWith({
                fromModel: 'qwen3-coder-plus',
                toModel: 'qwen-vl-max-latest',
                reason: 'manual',
                context: undefined,
            });
        });
    });
});
//# sourceMappingURL=config.test.js.map