/*
 * Copyright (C) 2019 Peter Ross
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/**
 * @file
 * VP4 video decoder
 */

#ifndef AVCODEC_VP4DATA_H
#define AVCODEC_VP4DATA_H

#include <stdint.h>

static const uint8_t vp4_generic_dequant[64] = {
    16, 17, 18, 20, 22, 24, 26, 28,
    17, 18, 20, 22, 24, 26, 28, 32,
    18, 20, 22, 24, 26, 28, 32, 36,
    20, 22, 24, 26, 28, 32, 36, 40,
    22, 24, 26, 28, 32, 36, 40, 44,
    24, 26, 28, 32, 36, 40, 44, 48,
    26, 28, 32, 36, 40, 44, 48, 52,
    28, 32, 36, 40, 44, 48, 52, 56
};

static const uint8_t vp4_y_dc_scale_factor[64] = {
    180, 180, 180, 180, 180, 180, 175, 170,
    165, 160, 157, 155, 152, 150, 147, 145,
    142, 140, 137, 135, 132, 130, 127, 125,
    122, 120, 117, 115, 112, 110, 107, 105,
    102, 100,  97,  95,  92,  90,  87,  85,
     82,  80,  77,  75,  72,  70,  67,  65,
     62,  60,  57,  55,  52,  50,  47,  45,
     42,  40,  37,  35,  32,  30,  27,  25
};

static const uint8_t vp4_uv_dc_scale_factor[64] = {
    150, 150, 150, 150, 150, 150, 150, 150,
    150, 150, 150, 150, 150, 150, 147, 145,
    142, 140, 137, 135, 132, 130, 127, 125,
    122, 120, 117, 115, 112, 110, 107, 105,
    102, 100,  97,  95,  92,  90,  87,  85,
     82,  80,  77,  75,  72,  70,  67,  65,
     62,  60,  57,  55,  52,  50,  47,  45,
     42,  40,  37,  35,  32,  30,  27,  25
};

static const uint16_t vp4_ac_scale_factor[64] = {
    500, 475, 450, 430, 410, 390, 370, 350,
    330, 315, 300, 285, 270, 260, 250, 240,
    230, 220, 210, 200, 190, 185, 180, 170,
    160, 150, 143, 135, 128, 120, 113, 106,
    100,  94,  90,  85,  80,  75,  70,  66,
     62,  57,  52,  49,  45,  41,  38,  35,
     33,  30,  27,  24,  22,  20,  18,  16,
     14,  12,  10,   9,   7,   6,   4,   1
};

static const uint8_t vp4_filter_limit_values[64] = {
    30, 25, 20, 20, 15, 15, 14, 14,
    13, 13, 12, 12, 11, 11, 10, 10,
     9,  9,  8,  8,  7,  7,  7,  7,
     6,  6,  6,  6,  5,  5,  5,  5,
     4,  4,  4,  4,  3,  3,  3,  3,
     2,  2,  2,  2,  2,  2,  2,  2,
     2,  2,  2,  2,  2,  2,  2,  2,
     1,  1,  1,  1,  1,  1,  1,  1
};

static const uint8_t vp4_block_pattern_table_selector[14] = {
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 1, 1
};

static const uint8_t vp4_block_pattern_vlc[2][14][2] = {
    {
        { 0x0, 3 }, { 0xf, 4 }, { 0x9, 4 }, { 0x2, 3 },
        { 0xd, 4 }, { 0xe, 5 }, { 0xb, 4 }, { 0x1, 3 },
        { 0xf, 5 }, { 0x8, 4 }, { 0x6, 4 }, { 0xe, 4 },
        { 0xc, 4 }, { 0xa, 4 }
    },
    {
        { 0x7, 4 }, { 0xa, 4 }, { 0x9, 4 }, { 0xc, 4 },
        { 0x8, 4 }, { 0xd, 5 }, { 0x0, 3 }, { 0xe, 4 },
        { 0xc, 5 }, { 0xd, 4 }, { 0x1, 3 }, { 0xb, 4 },
        { 0xf, 4 }, { 0x2, 3 }
    }
};

static const uint8_t vp4_mv_table_selector[32] = {
    0, 1, 2, 2, 3, 3, 3, 3,
    4, 4, 4, 4, 4, 4, 4, 4,
    5, 5, 5, 5, 5, 5, 5, 5,
    6, 6, 6, 6, 6, 6, 6, 6,
};

static const uint8_t vp4_mv_vlc[2][7][63][2] = {
    { /* x-axis tables */
        {
            { 30,  2 }, { 36,  5 }, { 19,  8 }, { 55,  9 }, { 61,  9 },
            {  4,  8 }, { 43,  8 }, { 22,  6 }, { 29,  4 }, { 50,  7 },
            { 44,  7 }, { 35,  6 }, { 49,  8 }, { 17,  8 }, { 60,  8 },
            {  7,  9 }, { 11,  9 }, { 27,  6 }, { 24,  6 }, { 38,  6 },
            { 14,  7 }, {  6,  8 }, { 56,  8 }, { 47,  7 }, {  8,  8 },
            { 59, 10 }, {  9, 10 }, { 57,  9 }, { 18,  7 }, { 42,  7 },
            { 25,  7 }, { 21,  8 }, { 12,  8 }, { 48,  7 }, { 37,  7 },
            { 10,  8 }, {  1,  9 }, {  5, 10 }, {  3, 10 }, { 16,  7 },
            { 41,  8 }, { 54,  8 }, { 20,  7 }, { 26,  6 }, { 28,  5 },
            { 32,  3 }, { 52,  8 }, { 51,  9 }, { 13,  9 }, { 15,  7 },
            { 46,  7 }, { 39,  8 }, {  2,  9 }, { 45,  9 }, { 34,  5 },
            { 40,  7 }, { 23,  8 }, { 58,  9 }, { 53,  9 }, {  0,  7 },
            { 62,  7 }, { 33,  5 }, { 31,  3 },
        },
        {
            { 29,  4 }, { 27,  6 }, { 54,  8 }, { 19,  8 }, { 53,  9 },
            { 13,  9 }, { 60,  8 }, { 25,  7 }, {  6,  8 }, {  2,  8 },
            { 11,  9 }, { 61,  9 }, {  8,  8 }, { 48,  7 }, { 28,  4 },
            { 46,  7 }, { 16,  7 }, { 24,  6 }, { 40,  6 }, { 20,  7 },
            { 62,  7 }, { 33,  3 }, { 36,  5 }, { 52,  8 }, { 41,  8 },
            { 44,  7 }, { 10,  8 }, { 59, 10 }, {  7, 10 }, { 21,  9 },
            { 18,  7 }, { 23,  8 }, { 12,  8 }, {  0,  7 }, {  5, 10 },
            {  3, 10 }, { 17,  9 }, { 15,  8 }, { 37,  7 }, { 38,  6 },
            { 49,  9 }, { 45,  9 }, { 14,  8 }, { 50,  8 }, { 47,  8 },
            { 34,  4 }, { 35,  6 }, { 42,  7 }, { 55, 10 }, {  1, 10 },
            { 43,  9 }, { 58,  9 }, {  9, 11 }, { 57, 11 }, { 51, 10 },
            { 22,  7 }, { 39,  8 }, { 56,  9 }, {  4,  9 }, { 26,  6 },
            { 30,  3 }, { 32,  3 }, { 31,  3 },
        },
        {
            { 32,  2 }, { 35,  4 }, {  6,  8 }, { 12,  8 }, {  0,  7 },
            { 20,  7 }, { 44,  7 }, { 29,  5 }, { 34,  3 }, { 37,  6 },
            { 57,  9 }, { 10,  9 }, { 21,  8 }, { 39,  7 }, { 50,  8 },
            { 47,  8 }, { 62,  7 }, { 26,  6 }, { 28,  5 }, {  7, 10 },
            { 55, 10 }, { 49, 10 }, {  3, 10 }, { 17,  9 }, { 58,  9 },
            { 15,  8 }, { 56,  9 }, { 45,  9 }, { 42,  7 }, { 54,  8 },
            { 41,  8 }, { 33,  3 }, { 60,  9 }, { 43,  9 }, { 18,  8 },
            { 22,  7 }, { 40,  7 }, { 48,  8 }, { 25,  8 }, { 36,  5 },
            { 31,  4 }, { 30,  4 }, { 24,  7 }, { 51, 10 }, { 13, 11 },
            { 59, 11 }, { 53, 10 }, {  1, 10 }, { 14,  8 }, { 19,  9 },
            { 11, 10 }, {  9, 10 }, { 46,  8 }, {  8,  9 }, {  5, 10 },
            { 61, 10 }, { 16,  8 }, { 27,  7 }, {  4,  9 }, {  2,  9 },
            { 52,  9 }, { 23,  9 }, { 38,  6 },
        },
        {
            { 10,  7 }, { 15,  7 }, {  0,  6 }, { 41,  6 }, { 22,  6 },
            { 33,  4 }, { 31,  4 }, { 55,  8 }, { 19,  8 }, { 60,  7 },
            { 56,  7 }, { 54,  7 }, { 28,  5 }, { 37,  4 }, { 16,  7 },
            { 50,  7 }, { 44,  6 }, { 14,  7 }, {  7,  9 }, {  3,  9 },
            { 59,  9 }, { 13,  9 }, { 24,  6 }, { 46,  6 }, { 47,  7 },
            {  4,  8 }, {  2,  8 }, { 39,  5 }, { 30,  4 }, { 35,  4 },
            { 61,  9 }, { 53,  9 }, { 21,  8 }, { 52,  7 }, { 43,  7 },
            { 18,  7 }, {  9,  9 }, { 11,  9 }, {  6,  8 }, { 25,  7 },
            { 17,  9 }, {  5,  9 }, { 23,  8 }, { 12,  8 }, { 58,  8 },
            { 38,  4 }, { 32,  4 }, { 20,  7 }, { 45,  8 }, {  8,  8 },
            { 26,  6 }, { 40,  5 }, { 36,  4 }, { 27,  7 }, { 57,  9 },
            { 51,  9 }, { 49,  9 }, {  1,  9 }, { 48,  7 }, { 62,  7 },
            { 29,  6 }, { 42,  6 }, { 34,  4 },
        },
        {
            { 37,  5 }, { 62,  5 }, { 28,  5 }, {  4,  7 }, { 19,  8 },
            { 13,  8 }, { 51,  7 }, {  8,  7 }, { 43,  5 }, { 54,  6 },
            { 22,  6 }, { 42,  4 }, { 25,  7 }, {  9,  8 }, {  2,  8 },
            {  6,  7 }, { 10,  7 }, { 48,  5 }, { 29,  6 }, { 12,  7 },
            { 60,  7 }, { 41,  5 }, { 38,  4 }, { 40,  4 }, { 39,  5 },
            {  3,  9 }, {  7,  9 }, { 17,  8 }, { 15,  7 }, {  0,  6 },
            { 52,  6 }, { 24,  6 }, { 57,  8 }, { 55,  8 }, { 27,  7 },
            { 33,  6 }, { 35,  6 }, { 26,  6 }, { 53,  8 }, { 59,  9 },
            {  5,  9 }, { 49,  7 }, { 47,  6 }, { 45,  6 }, { 56,  7 },
            { 21,  8 }, { 61,  8 }, { 31,  5 }, { 34,  5 }, { 46,  5 },
            { 30,  5 }, { 18,  7 }, { 58,  7 }, { 16,  7 }, { 20,  7 },
            { 50,  6 }, {  1,  9 }, { 11,  9 }, { 23,  8 }, { 14,  7 },
            { 44,  5 }, { 32,  5 }, { 36,  5 },
        },
        {
            { 50,  4 }, { 56,  5 }, { 22,  6 }, { 27,  7 }, { 10,  7 },
            { 46,  4 }, { 20,  6 }, { 43,  6 }, { 18,  6 }, {  8,  7 },
            { 11,  8 }, {  7,  8 }, { 38,  5 }, { 40,  5 }, { 48,  4 },
            { 16,  6 }, { 24,  6 }, { 28,  6 }, { 57,  7 }, {  6,  7 },
            { 31,  5 }, { 30,  5 }, { 54,  5 }, { 60,  6 }, { 58,  6 },
            { 42,  5 }, { 53,  6 }, { 26,  6 }, { 32,  5 }, { 44,  5 },
            { 17,  8 }, {  1,  8 }, { 59,  8 }, { 19,  8 }, { 45,  6 },
            { 62,  5 }, { 37,  7 }, { 14,  7 }, {  5,  9 }, {  9,  9 },
            {  4,  8 }, { 29,  7 }, { 52,  5 }, { 51,  6 }, { 12,  7 },
            { 35,  7 }, { 34,  6 }, { 23,  8 }, {  2,  8 }, { 25,  8 },
            { 21,  8 }, { 47,  5 }, { 39,  7 }, { 55,  7 }, { 49,  6 },
            { 41,  7 }, { 15,  7 }, {  0,  6 }, { 33,  7 }, { 61,  8 },
            {  3,  9 }, { 13,  9 }, { 36,  6 },
        },
        {
            { 47,  5 }, { 14,  6 }, { 53,  6 }, { 43,  7 }, {  5,  8 },
            { 23,  8 }, { 37,  6 }, { 50,  5 }, { 24,  6 }, { 27,  7 },
            { 35,  7 }, { 31,  5 }, { 16,  6 }, { 22,  6 }, { 41,  7 },
            { 12,  7 }, { 57,  6 }, { 62,  3 }, { 48,  5 }, { 30,  5 },
            { 52,  5 }, { 55,  6 }, {  2,  7 }, { 33,  7 }, { 46,  5 },
            { 61,  6 }, { 59,  6 }, { 32,  5 }, {  4,  7 }, { 21,  8 },
            { 19,  8 }, { 49,  7 }, { 29,  7 }, { 54,  5 }, {  0,  5 },
            { 26,  6 }, { 39,  7 }, {  7,  8 }, {  3,  9 }, { 13,  9 },
            { 45,  7 }, {  6,  7 }, { 38,  6 }, { 58,  5 }, { 34,  6 },
            { 28,  6 }, { 18,  7 }, { 10,  7 }, { 51,  7 }, { 15,  7 },
            { 44,  6 }, { 17,  8 }, {  1,  8 }, {  8,  7 }, { 56,  5 },
            { 42,  6 }, { 36,  6 }, { 60,  5 }, { 11,  9 }, {  9,  9 },
            { 25,  8 }, { 20,  7 }, { 40,  6 },
        },
    },
    { /* y-axis tables */
        {
            { 32,  2 }, { 31,  2 }, { 48,  7 }, {  4,  8 }, { 12,  8 },
            { 44,  7 }, { 10,  8 }, { 23,  8 }, { 20,  7 }, { 11,  9 },
            { 51,  9 }, { 41,  8 }, { 46,  7 }, {  6,  8 }, { 52,  8 },
            { 42,  7 }, {  3,  9 }, { 57, 10 }, {  7, 10 }, { 50,  8 },
            { 38,  6 }, { 26,  6 }, {  8,  8 }, { 49,  9 }, { 17,  9 },
            { 15,  7 }, { 62,  7 }, { 22,  7 }, {  0,  7 }, {  1,  9 },
            { 61,  9 }, { 45,  9 }, {  5, 10 }, { 55, 10 }, { 29,  5 },
            { 28,  5 }, { 54,  9 }, { 43,  9 }, { 19,  9 }, { 21,  9 },
            { 25,  8 }, { 39,  8 }, { 27,  7 }, { 24,  7 }, { 30,  3 },
            { 33,  5 }, { 60,  9 }, { 56,  9 }, { 59, 10 }, { 13, 10 },
            { 58,  9 }, { 35,  7 }, { 36,  6 }, { 34,  5 }, { 40,  7 },
            { 14,  8 }, { 16,  8 }, { 18,  8 }, {  2,  9 }, { 53, 10 },
            {  9, 10 }, { 47,  8 }, { 37,  8 },
        },
        {
            { 31,  2 }, { 32,  2 }, { 29,  5 }, { 17,  9 }, { 11, 10 },
            {  1, 10 }, { 39,  8 }, { 27,  7 }, { 35,  6 }, { 22,  7 },
            { 23,  8 }, { 56,  9 }, { 45,  9 }, { 51, 11 }, {  5, 11 },
            { 13, 10 }, { 60,  9 }, { 14,  8 }, { 12,  8 }, { 21,  9 },
            {  4,  9 }, { 25,  8 }, { 58,  9 }, {  8,  9 }, { 18,  8 },
            {  2,  9 }, { 54,  9 }, { 26,  6 }, { 30,  3 }, { 34,  4 },
            { 28,  5 }, { 19, 10 }, { 57, 11 }, { 59, 11 }, { 52,  9 },
            { 48,  8 }, { 16,  8 }, { 46,  8 }, { 10,  9 }, { 61, 10 },
            { 53, 11 }, { 49, 11 }, { 20,  8 }, {  0,  8 }, { 41,  9 },
            { 50,  9 }, { 33,  4 }, { 24,  7 }, { 62,  8 }, { 44,  8 },
            { 37,  8 }, { 47,  9 }, {  7, 11 }, { 55, 11 }, {  6, 10 },
            { 40,  7 }, { 15,  9 }, {  9, 11 }, {  3, 11 }, { 43, 10 },
            { 42,  8 }, { 38,  7 }, { 36,  6 },
        },
        {
            { 31,  3 }, { 16,  7 }, { 61,  9 }, { 17,  9 }, { 23,  8 },
            { 18,  7 }, { 10,  8 }, {  9,  9 }, {  3, 10 }, { 59, 10 },
            { 48,  7 }, { 54,  8 }, { 43,  8 }, {  0,  7 }, { 13,  9 },
            {  5, 10 }, { 55, 10 }, { 15,  8 }, { 35,  4 }, { 38,  5 },
            { 20,  7 }, { 39,  7 }, { 62,  7 }, {  1,  9 }, { 58,  9 },
            { 41,  8 }, { 36,  4 }, { 34,  3 }, { 37,  6 }, {  8,  8 },
            { 52,  8 }, { 46,  7 }, { 28,  5 }, { 26,  6 }, { 50,  8 },
            { 49, 10 }, { 57, 10 }, {  4,  9 }, { 27,  7 }, { 40,  6 },
            { 44,  7 }, {  2,  9 }, { 21,  9 }, { 47,  8 }, { 33,  3 },
            { 12,  9 }, {  6,  9 }, { 14,  8 }, { 42,  7 }, { 51, 10 },
            { 53, 10 }, { 19,  9 }, { 45,  9 }, { 11, 10 }, {  7, 10 },
            { 25,  8 }, { 60,  9 }, { 56,  9 }, { 29,  6 }, { 22,  7 },
            { 24,  7 }, { 30,  4 }, { 32,  3 },
        },
        {
            { 39,  5 }, { 48,  6 }, { 13,  9 }, { 11,  9 }, {  2,  8 },
            { 43,  7 }, { 28,  5 }, { 25,  7 }, { 27,  7 }, { 46,  6 },
            { 34,  3 }, { 36,  3 }, { 30,  4 }, { 37,  4 }, { 16,  7 },
            { 47,  7 }, {  7, 10 }, {  9, 10 }, {  5,  9 }, { 12,  8 },
            { 52,  7 }, { 10,  8 }, {  4,  8 }, { 18,  7 }, { 55,  9 },
            { 17,  9 }, { 58,  8 }, { 50,  7 }, { 44,  6 }, { 23,  8 },
            {  1,  9 }, { 53,  9 }, { 20,  7 }, { 29,  6 }, { 26,  6 },
            { 35,  4 }, { 38,  4 }, { 45,  8 }, { 60,  8 }, {  0,  7 },
            { 22,  7 }, { 56,  8 }, {  6,  9 }, { 49,  9 }, { 42,  6 },
            { 62,  7 }, { 15,  8 }, { 54,  8 }, { 32,  4 }, { 40,  5 },
            { 14,  8 }, { 19,  9 }, {  8,  9 }, { 24,  7 }, { 41,  7 },
            { 51,  9 }, { 21,  9 }, { 57, 10 }, {  3, 10 }, { 61, 10 },
            { 59, 10 }, { 31,  5 }, { 33,  5 },
        },
        {
            { 34,  4 }, { 52,  6 }, { 29,  6 }, { 22,  6 }, {  0,  6 },
            { 32,  4 }, { 42,  4 }, { 37,  5 }, { 49,  7 }, { 12,  7 },
            { 10,  7 }, { 21,  8 }, {  3,  9 }, { 59,  9 }, { 36,  4 },
            { 48,  5 }, { 41,  5 }, { 24,  6 }, { 58,  7 }, { 60,  7 },
            { 50,  6 }, { 18,  7 }, { 14,  7 }, { 38,  4 }, { 40,  4 },
            {  2,  8 }, {  6,  8 }, { 27,  7 }, { 56,  7 }, { 11,  9 },
            {  1,  9 }, { 53,  8 }, { 54,  7 }, { 23,  8 }, {  8,  8 },
            { 47,  6 }, { 39,  5 }, { 45,  6 }, { 43,  6 }, { 31,  5 },
            { 26,  6 }, { 16,  7 }, { 13,  9 }, { 57,  9 }, {  4,  8 },
            { 46,  5 }, { 35,  6 }, { 62,  6 }, { 51,  8 }, { 15,  8 },
            { 19,  9 }, {  9,  9 }, { 61,  9 }, { 55,  9 }, { 20,  7 },
            { 25,  8 }, {  5, 10 }, {  7, 10 }, { 17,  9 }, { 33,  6 },
            { 28,  6 }, { 44,  5 }, { 30,  5 },
        },
        {
            { 53,  6 }, { 22,  6 }, { 54,  5 }, { 32,  4 }, { 48,  4 },
            { 29,  6 }, { 57,  7 }, { 11,  8 }, { 17,  8 }, {  9,  8 },
            {  1,  8 }, { 55,  7 }, { 12,  7 }, { 18,  7 }, { 60,  6 },
            { 45,  6 }, {  4,  7 }, {  2,  7 }, { 20,  6 }, { 38,  5 },
            { 16,  6 }, { 19,  8 }, { 59,  8 }, {  6,  7 }, { 36,  5 },
            { 52,  5 }, { 39,  6 }, { 58,  6 }, { 42,  5 }, { 33,  6 },
            { 24,  6 }, { 25,  7 }, {  7,  9 }, {  3,  9 }, { 21,  8 },
            { 56,  6 }, { 40,  5 }, { 62,  5 }, { 44,  5 }, {  0,  6 },
            { 26,  6 }, { 34,  5 }, { 47,  5 }, { 51,  6 }, { 49,  6 },
            { 31,  5 }, { 15,  7 }, { 41,  7 }, { 43,  7 }, { 27,  7 },
            { 50,  5 }, { 30,  5 }, { 37,  7 }, {  8,  7 }, { 28,  6 },
            { 46,  5 }, { 35,  7 }, { 61,  8 }, { 23,  8 }, { 10,  8 },
            {  5,  9 }, { 13,  9 }, { 14,  7 },
        },
        {
            { 62,  3 }, { 25,  7 }, {  3,  8 }, { 19,  8 }, { 59,  6 },
            { 29,  6 }, { 22,  6 }, { 14,  6 }, {  4,  7 }, { 23,  7 },
            { 48,  5 }, { 60,  5 }, { 61,  6 }, {  1,  7 }, { 39,  7 },
            { 46,  5 }, { 52,  5 }, { 40,  5 }, { 34,  5 }, { 58,  5 },
            { 21,  8 }, { 13,  8 }, { 49,  7 }, { 16,  6 }, { 12,  7 },
            {  2,  7 }, { 47,  6 }, { 56,  5 }, {  0,  5 }, {  6,  7 },
            { 45,  7 }, { 55,  6 }, { 54,  5 }, {  9,  8 }, {  5,  8 },
            { 15,  7 }, { 33,  6 }, { 10,  7 }, { 53,  7 }, { 24,  6 },
            {  8,  7 }, { 37,  7 }, { 43,  7 }, { 20,  7 }, { 44,  6 },
            { 36,  6 }, { 28,  6 }, { 42,  6 }, { 30,  5 }, { 17,  8 },
            {  7,  9 }, { 11,  9 }, { 27,  7 }, { 26,  6 }, { 38,  6 },
            { 50,  6 }, { 51,  8 }, { 41,  8 }, { 57,  7 }, { 18,  7 },
            { 35,  7 }, { 31,  5 }, { 32,  5 },
        },
    }
};

static const uint8_t vp4_bias[5 * 16][32][2] = {
    { /* DC bias table 0 */
        { 20,  3 }, { 12,  4 }, { 14,  4 }, { 19,  3 }, {  0,  5 }, { 23,  6 },
        { 30,  7 }, { 31,  8 }, { 25,  8 }, { 17,  4 }, { 22,  4 }, { 13,  4 },
        { 21,  4 }, {  9,  4 }, { 16,  5 }, { 11,  5 }, { 18,  4 }, {  1,  7 },
        {  7,  7 }, { 29, 10 }, {  4, 10 }, { 28, 10 }, { 26, 12 }, {  8, 14 },
        {  6, 14 }, {  5, 13 }, { 27, 11 }, {  2,  9 }, {  3,  9 }, { 24,  7 },
        { 15,  5 }, { 10,  4 },
    },
    { /* DC bias table 1 */
        { 19,  3 }, { 11,  4 }, { 21,  4 }, { 10,  3 }, { 12,  4 }, { 14,  4 },
        {  3,  7 }, {  6, 11 }, {  8, 12 }, { 29, 12 }, { 26, 11 }, {  5, 11 },
        {  4,  9 }, {  2,  8 }, { 24,  6 }, {  0,  5 }, { 17,  4 }, { 13,  4 },
        { 23,  6 }, { 27, 10 }, { 28, 10 }, { 25,  9 }, { 31,  8 }, { 30,  7 },
        { 16,  5 }, { 20,  4 }, { 18,  4 }, {  9,  4 }, { 22,  6 }, {  1,  7 },
        {  7,  7 }, { 15,  5 },
    },
    { /* DC bias table 2 */
        {  9,  3 }, { 15,  4 }, { 11,  4 }, {  1,  6 }, {  3,  7 }, { 22,  7 },
        { 21,  5 }, { 14,  4 }, { 10,  3 }, { 17,  4 }, { 12,  4 }, { 20,  4 },
        { 24,  6 }, {  5, 10 }, { 28, 10 }, {  6, 12 }, {  8, 13 }, { 29, 13 },
        { 26, 11 }, { 27, 10 }, {  4,  9 }, { 25,  9 }, {  2,  8 }, { 31,  8 },
        {  0,  5 }, { 13,  4 }, { 18,  4 }, { 16,  5 }, { 23,  6 }, { 30,  7 },
        {  7,  7 }, { 19,  4 },
    },
    { /* DC bias table 3 */
        { 15,  4 }, { 20,  4 }, {  9,  3 }, { 30,  6 }, { 31,  7 }, {  4,  8 },
        { 27,  9 }, { 22,  9 }, {  7,  6 }, {  1,  6 }, { 11,  4 }, { 17,  4 },
        { 14,  4 }, { 10,  3 }, { 12,  4 }, { 21,  6 }, { 24,  6 }, { 16,  5 },
        { 19,  4 }, { 18,  4 }, { 13,  4 }, {  0,  5 }, {  3,  7 }, { 25,  9 },
        {  8, 13 }, { 29, 13 }, {  6, 12 }, { 26, 11 }, {  5, 11 }, { 28, 11 },
        {  2,  8 }, { 23,  6 },
    },
    { /* DC bias table 4 */
        { 15,  4 }, {  0,  4 }, { 30,  6 }, {  2,  7 }, { 29, 12 }, {  8, 12 },
        { 22, 11 }, { 26, 10 }, { 27,  9 }, {  4,  8 }, { 23,  5 }, { 17,  4 },
        {  9,  3 }, { 19,  4 }, { 11,  4 }, { 14,  4 }, { 18,  4 }, { 20,  5 },
        { 21,  7 }, {  3,  7 }, {  1,  6 }, { 12,  4 }, { 10,  3 }, { 16,  5 },
        { 24,  6 }, {  5, 10 }, {  6, 11 }, { 28, 11 }, { 25,  9 }, { 31,  8 },
        {  7,  7 }, { 13,  4 },
    },
    { /* DC bias table 5 */
        { 15,  4 }, { 20,  5 }, { 23,  5 }, { 19,  4 }, { 17,  4 }, {  0,  4 },
        { 18,  4 }, { 14,  4 }, { 30,  6 }, { 31,  7 }, { 28, 10 }, { 26, 10 },
        { 27,  9 }, { 21,  8 }, { 16,  5 }, {  9,  3 }, { 11,  4 }, { 12,  4 },
        {  2,  7 }, {  4,  8 }, { 22, 12 }, {  8, 13 }, { 29, 13 }, {  6, 11 },
        {  5, 10 }, { 25,  9 }, { 24,  6 }, {  7,  7 }, {  3,  7 }, {  1,  6 },
        { 13,  4 }, { 10,  3 },
    },
    { /* DC bias table 6 */
        { 13,  3 }, { 17,  4 }, { 23,  5 }, { 20,  6 }, {  4,  7 }, { 31,  7 },
        { 18,  4 }, {  1,  5 }, {  3,  6 }, { 24,  6 }, { 14,  4 }, { 11,  4 },
        {  0,  4 }, { 16,  5 }, { 19,  5 }, {  9,  3 }, { 12,  4 }, {  7,  7 },
        { 27,  9 }, {  8, 12 }, { 29, 13 }, { 22, 13 }, { 28, 11 }, { 21, 10 },
        { 25,  9 }, {  5, 10 }, { 26, 11 }, {  6, 11 }, {  2,  7 }, { 30,  7 },
        { 15,  5 }, { 10,  3 },
    },
    { /* DC bias table 7 */
        { 10,  2 }, { 15,  4 }, { 19,  5 }, { 24,  6 }, {  2,  6 }, { 14,  4 },
        { 16,  5 }, {  1,  5 }, { 11,  4 }, { 30,  7 }, { 25,  9 }, { 21, 11 },
        {  8, 12 }, { 22, 13 }, { 29, 13 }, {  6, 10 }, {  7,  8 }, {  3,  6 },
        { 23,  6 }, {  5,  9 }, { 28, 11 }, { 26, 11 }, { 27, 10 }, { 31,  8 },
        { 20,  8 }, {  4,  8 }, {  9,  3 }, { 18,  5 }, { 17,  5 }, { 12,  4 },
        {  0,  4 }, { 13,  4 },
    },
    { /* DC bias table 8 */
        { 15,  5 }, { 19,  5 }, { 17,  5 }, {  8,  6 }, { 26,  6 }, { 24,  4 },
        { 18,  5 }, {  6, 10 }, {  5, 10 }, {  4,  9 }, { 22,  8 }, {  7,  7 },
        {  3,  7 }, { 21,  7 }, {  0,  4 }, { 12,  4 }, { 13,  4 }, { 11,  4 },
        { 25,  5 }, { 30,  6 }, { 20,  6 }, { 23,  4 }, {  2,  7 }, { 31,  7 },
        { 16,  6 }, { 14,  5 }, {  1,  6 }, { 27,  6 }, { 29,  6 }, { 28,  6 },
        { 10,  3 }, {  9,  3 },
    },
    { /* DC bias table 9 */
        { 30,  6 }, { 26,  6 }, { 19,  5 }, { 24,  4 }, { 25,  5 }, { 15,  5 },
        { 17,  5 }, { 31,  7 }, { 21,  7 }, { 20,  6 }, { 23,  4 }, { 18,  5 },
        { 28,  5 }, { 12,  4 }, { 13,  4 }, {  0,  4 }, { 11,  4 }, {  3,  7 },
        {  2,  7 }, { 29,  6 }, { 27,  6 }, { 16,  6 }, { 14,  5 }, {  8,  7 },
        {  7,  8 }, {  6, 11 }, {  5, 11 }, {  4, 10 }, { 22,  9 }, {  1,  6 },
        { 10,  3 }, {  9,  3 },
    },
    { /* DC bias table 10 */
        {  9,  2 }, { 15,  5 }, { 26,  6 }, { 29,  6 }, { 17,  5 }, { 18,  5 },
        { 13,  4 }, { 12,  4 }, { 11,  4 }, { 20,  6 }, { 27,  6 }, {  8,  7 },
        {  7,  8 }, {  6, 11 }, {  5, 11 }, {  4, 10 }, { 22,  9 }, {  3,  7 },
        { 30,  7 }, {  0,  4 }, { 14,  5 }, { 16,  6 }, { 28,  6 }, { 10,  3 },
        { 24,  5 }, {  2,  7 }, { 31,  8 }, { 21,  8 }, { 25,  6 }, {  1,  6 },
        { 19,  6 }, { 23,  5 },
    },
    { /* DC bias table 11 */
        { 17,  5 }, {  6, 10 }, {  5, 10 }, { 22,  9 }, {  4,  8 }, { 31,  7 },
        {  2,  6 }, { 15,  5 }, { 18,  5 }, { 24,  4 }, { 25,  5 }, { 28,  5 },
        { 13,  4 }, {  1,  5 }, {  3,  6 }, { 26,  6 }, { 12,  4 }, { 11,  4 },
        { 23,  4 }, { 29,  6 }, { 16,  6 }, { 14,  5 }, {  0,  4 }, {  8,  7 },
        { 21,  8 }, {  7,  8 }, { 19,  6 }, { 30,  7 }, { 20,  7 }, { 27,  6 },
        { 10,  3 }, {  9,  3 },
    },
    { /* DC bias table 12 */
        {  9,  2 }, { 31,  7 }, {  4,  7 }, { 26,  6 }, { 28,  5 }, { 12,  4 },
        { 11,  4 }, { 19,  6 }, { 16,  6 }, {  8,  7 }, { 20,  7 }, {  2,  6 },
        { 23,  4 }, { 14,  5 }, {  1,  5 }, { 29,  6 }, {  7,  8 }, {  6, 11 },
        { 22, 11 }, {  5, 10 }, { 21,  9 }, { 30,  7 }, { 27,  6 }, { 17,  6 },
        {  0,  4 }, { 18,  6 }, {  3,  6 }, { 15,  6 }, { 25,  6 }, { 24,  5 },
        { 13,  5 }, { 10,  3 },
    },
    { /* DC bias table 13 */
        {  9,  2 }, { 13,  4 }, { 12,  4 }, { 11,  4 }, { 26,  6 }, { 16,  6 },
        { 14,  5 }, { 23,  4 }, {  8,  7 }, {  7,  8 }, { 31,  8 }, { 29,  6 },
        { 30,  7 }, {  4,  7 }, { 18,  6 }, { 17,  6 }, {  2,  6 }, {  1,  5 },
        { 27,  6 }, { 25,  6 }, { 15,  6 }, {  6, 11 }, { 22, 11 }, {  5, 10 },
        { 21,  9 }, { 20,  8 }, { 19,  7 }, { 10,  3 }, {  0,  4 }, { 24,  5 },
        { 28,  6 }, {  3,  6 },
    },
    { /* DC bias table 14 */
        {  9,  2 }, { 13,  4 }, { 23,  4 }, { 12,  4 }, { 11,  4 }, { 29,  6 },
        {  4,  6 }, { 14,  5 }, { 18,  6 }, { 17,  6 }, {  3,  5 }, { 27,  6 },
        { 25,  6 }, { 15,  6 }, { 31,  8 }, {  5,  8 }, { 19,  7 }, { 24,  5 },
        {  1,  5 }, {  0,  4 }, { 28,  6 }, {  2,  6 }, { 20,  8 }, {  8,  8 },
        { 26,  7 }, { 16,  7 }, { 22, 11 }, {  6, 11 }, { 21, 10 }, {  7,  9 },
        { 30,  8 }, { 10,  3 },
    },
    { /* DC bias table 15 */
        {  0,  3 }, { 13,  4 }, {  2,  5 }, { 18,  6 }, { 26,  7 }, { 29,  7 },
        {  9,  2 }, { 14,  5 }, { 23,  5 }, { 12,  4 }, { 11,  4 }, { 17,  6 },
        {  5,  7 }, { 30,  8 }, { 22, 11 }, { 21, 11 }, {  7, 10 }, { 31,  9 },
        { 15,  6 }, { 19,  7 }, { 25,  7 }, {  4,  6 }, { 20,  8 }, {  6,  9 },
        {  8,  9 }, { 27,  7 }, {  3,  5 }, {  1,  5 }, { 24,  6 }, { 16,  7 },
        { 28,  7 }, { 10,  3 },
    },
    { /* AC bias group 1, table 0 */
        {  9,  3 }, { 15,  4 }, {  0,  5 }, { 25,  6 }, {  1,  7 }, { 27,  7 },
        { 23,  4 }, { 17,  4 }, { 11,  4 }, { 12,  4 }, { 14,  4 }, { 30,  5 },
        { 26,  7 }, { 28,  7 }, { 21,  6 }, { 19,  4 }, { 18,  4 }, {  7,  5 },
        { 16,  5 }, { 13,  4 }, { 24,  6 }, { 31,  7 }, {  2,  9 }, {  4, 11 },
        {  6, 13 }, {  5, 13 }, {  8, 12 }, { 29, 10 }, {  3,  9 }, { 22,  9 },
        { 20,  5 }, { 10,  4 },
    },
    { /* AC bias group 1, table 1 */
        {  9,  3 }, { 15,  4 }, { 17,  4 }, { 21,  6 }, { 25,  6 }, {  0,  5 },
        { 19,  4 }, { 23,  4 }, { 12,  4 }, { 11,  4 }, {  2,  8 }, {  3,  8 },
        { 27,  7 }, { 26,  7 }, {  1,  7 }, { 30,  5 }, { 18,  4 }, { 14,  4 },
        { 20,  5 }, {  7,  5 }, { 13,  4 }, { 16,  5 }, { 22,  9 }, {  6, 13 },
        {  5, 13 }, {  8, 12 }, {  4, 11 }, { 29, 10 }, { 28,  8 }, { 31,  7 },
        { 24,  6 }, { 10,  4 },
    },
    { /* AC bias group 1, table 2 */
        { 17,  4 }, { 24,  5 }, { 31,  6 }, { 29,  9 }, {  8, 11 }, {  5, 12 },
        {  6, 13 }, { 22, 13 }, {  4, 10 }, {  2,  8 }, { 28,  7 }, { 10,  3 },
        {  9,  3 }, { 18,  4 }, { 11,  4 }, { 12,  4 }, { 14,  4 }, { 23,  4 },
        { 25,  6 }, { 27,  7 }, { 26,  7 }, { 30,  5 }, { 16,  5 }, {  0,  5 },
        { 13,  4 }, {  7,  5 }, {  1,  7 }, {  3,  8 }, { 21,  8 }, { 20,  6 },
        { 19,  5 }, { 15,  5 },
    },
    { /* AC bias group 1, table 3 */
        { 17,  4 }, { 18,  4 }, { 10,  3 }, {  9,  3 }, { 21,  8 }, { 29,  9 },
        {  8, 11 }, { 22, 13 }, {  6, 13 }, {  5, 12 }, {  4, 10 }, { 28,  7 },
        { 31,  6 }, { 24,  5 }, { 11,  4 }, { 12,  4 }, { 14,  4 }, { 16,  5 },
        { 20,  6 }, { 27,  7 }, {  2,  8 }, {  3,  8 }, { 23,  4 }, { 30,  5 },
        { 19,  5 }, {  7,  5 }, { 25,  6 }, { 26,  7 }, {  1,  7 }, { 13,  4 },
        {  0,  5 }, { 15,  5 },
    },
    { /* AC bias group 1, table 4 */
        { 25,  5 }, {  4,  8 }, { 21, 10 }, {  8, 11 }, { 22, 13 }, {  6, 13 },
        {  5, 12 }, { 29,  9 }, {  2,  7 }, { 26,  6 }, {  0,  4 }, { 19,  5 },
        { 28,  7 }, {  3,  7 }, { 31,  6 }, { 24,  5 }, { 16,  5 }, { 10,  3 },
        {  9,  3 }, { 14,  4 }, { 11,  4 }, { 12,  4 }, {  1,  6 }, { 20,  7 },
        { 27,  7 }, {  7,  5 }, { 23,  4 }, { 30,  5 }, { 18,  5 }, { 13,  4 },
        { 17,  5 }, { 15,  5 },
    },
    { /* AC bias group 1, table 5 */
        {  1,  5 }, { 27,  6 }, { 19,  6 }, { 25,  5 }, { 16,  5 }, { 18,  5 },
        { 26,  6 }, { 28,  7 }, {  2,  7 }, { 14,  4 }, {  0,  4 }, { 17,  5 },
        { 24,  5 }, { 10,  3 }, {  9,  3 }, { 11,  4 }, { 12,  4 }, { 31,  6 },
        {  5, 10 }, {  6, 12 }, { 22, 13 }, { 21, 13 }, {  8, 11 }, { 29,  9 },
        { 20,  9 }, {  4,  9 }, {  3,  7 }, {  7,  5 }, { 13,  4 }, { 15,  5 },
        { 30,  5 }, { 23,  4 },
    },
    { /* AC bias group 1, table 6 */
        { 30,  4 }, { 17,  5 }, {  1,  5 }, { 14,  4 }, { 27,  6 }, {  3,  6 },
        { 25,  5 }, { 10,  3 }, {  9,  3 }, { 19,  7 }, {  4,  8 }, {  6, 12 },
        { 22, 14 }, { 21, 14 }, { 20, 13 }, {  8, 11 }, {  5, 10 }, { 29,  9 },
        { 18,  6 }, { 15,  5 }, { 12,  4 }, { 11,  4 }, { 13,  4 }, { 24,  5 },
        {  7,  5 }, {  0,  4 }, { 26,  6 }, { 16,  6 }, { 31,  6 }, { 28,  7 },
        {  2,  7 }, { 23,  4 },
    },
    { /* AC bias group 1, table 7 */
        { 30,  4 }, { 13,  4 }, { 23,  3 }, {  3,  5 }, {  7,  5 }, { 12,  4 },
        { 10,  3 }, { 11,  4 }, {  8,  9 }, { 20, 12 }, { 19, 12 }, { 22, 12 },
        { 21, 12 }, {  6, 10 }, {  5,  8 }, { 29,  8 }, { 18,  8 }, { 27,  6 },
        { 14,  5 }, {  9,  3 }, { 17,  7 }, {  4,  7 }, { 15,  6 }, { 25,  5 },
        { 24,  5 }, {  2,  6 }, { 31,  6 }, {  1,  5 }, { 26,  6 }, { 16,  7 },
        { 28,  7 }, {  0,  4 },
    },
    { /* AC bias group 1, table 8 */
        { 26,  5 }, { 15,  5 }, { 24,  4 }, { 23,  3 }, { 19,  6 }, {  8,  7 },
        { 21,  9 }, {  6, 12 }, {  5, 12 }, { 22, 11 }, {  4, 10 }, { 20,  8 },
        { 28,  5 }, { 12,  4 }, { 13,  4 }, { 11,  4 }, { 10,  3 }, {  9,  3 },
        { 27,  6 }, { 29,  6 }, { 25,  5 }, { 31,  6 }, { 16,  6 }, { 30,  5 },
        { 14,  5 }, { 18,  6 }, {  7,  6 }, {  3,  8 }, {  2,  8 }, {  1,  7 },
        { 17,  6 }, {  0,  5 },
    },
    { /* AC bias group 1, table 9 */
        { 12,  4 }, { 13,  4 }, { 11,  4 }, {  0,  4 }, { 23,  3 }, { 26,  5 },
        { 16,  6 }, {  1,  6 }, { 24,  4 }, { 10,  3 }, { 28,  5 }, { 31,  6 },
        { 18,  6 }, { 30,  5 }, { 14,  5 }, {  9,  3 }, { 27,  6 }, { 17,  6 },
        { 25,  5 }, { 19,  7 }, {  5, 12 }, {  6, 13 }, { 22, 13 }, { 21, 11 },
        {  4, 10 }, { 20,  9 }, {  2,  8 }, {  7,  6 }, { 15,  6 }, {  8,  8 },
        {  3,  8 }, { 29,  7 },
    },
    { /* AC bias group 1, table 10 */
        { 12,  4 }, { 13,  4 }, { 11,  4 }, { 26,  5 }, { 28,  5 }, { 23,  3 },
        { 24,  4 }, {  0,  4 }, { 10,  3 }, { 19,  7 }, {  2,  7 }, { 29,  6 },
        { 31,  6 }, { 17,  6 }, { 30,  5 }, { 14,  5 }, {  9,  3 }, { 27,  6 },
        {  1,  6 }, { 25,  5 }, {  3,  7 }, { 20,  9 }, { 22, 13 }, {  6, 13 },
        {  5, 12 }, { 21, 11 }, {  4, 10 }, {  8,  8 }, {  7,  6 }, { 15,  6 },
        { 18,  7 }, { 16,  7 },
    },
    { /* AC bias group 1, table 11 */
        { 13,  4 }, { 12,  4 }, { 11,  4 }, { 28,  5 }, {  8,  7 }, {  2,  7 },
        { 29,  6 }, { 23,  3 }, { 24,  4 }, { 26,  5 }, { 17,  6 }, { 31,  6 },
        {  0,  4 }, { 14,  5 }, { 30,  5 }, { 10,  3 }, {  9,  3 }, { 27,  6 },
        {  3,  7 }, {  4,  9 }, {  5, 11 }, { 22, 13 }, {  6, 13 }, { 21, 12 },
        { 20, 10 }, { 19,  8 }, {  7,  6 }, {  1,  6 }, { 25,  5 }, { 15,  6 },
        { 18,  7 }, { 16,  7 },
    },
    { /* AC bias group 1, table 12 */
        { 13,  4 }, { 12,  4 }, { 11,  4 }, {  3,  6 }, {  8,  7 }, { 18,  7 },
        { 14,  5 }, { 23,  3 }, { 24,  4 }, { 26,  5 }, { 31,  6 }, { 29,  6 },
        { 10,  3 }, {  9,  3 }, { 30,  5 }, { 28,  5 }, {  0,  4 }, { 15,  6 },
        {  4,  8 }, { 20, 10 }, {  6, 12 }, { 22, 13 }, { 21, 13 }, {  5, 11 },
        { 19,  9 }, {  2,  7 }, {  7,  6 }, { 16,  7 }, { 17,  7 }, { 25,  5 },
        { 27,  6 }, {  1,  6 },
    },
    { /* AC bias group 1, table 13 */
        { 27,  5 }, { 14,  5 }, { 12,  4 }, { 11,  4 }, {  2,  6 }, { 15,  6 },
        {  1,  5 }, { 23,  3 }, { 24,  4 }, {  3,  6 }, { 16,  7 }, { 17,  7 },
        { 31,  6 }, {  7,  6 }, { 10,  3 }, { 30,  5 }, { 28,  5 }, { 26,  5 },
        { 29,  6 }, { 19,  9 }, { 22, 13 }, { 21, 13 }, {  6, 12 }, { 20, 11 },
        {  5, 10 }, {  4,  8 }, { 18,  8 }, {  8,  8 }, {  9,  3 }, {  0,  4 },
        { 13,  5 }, { 25,  5 },
    },
    { /* AC bias group 1, table 14 */
        {  0,  3 }, { 11,  4 }, { 15,  6 }, {  8,  7 }, { 18,  8 }, { 20, 12 },
        { 22, 13 }, { 21, 13 }, {  6, 11 }, { 19, 10 }, {  5,  9 }, {  7,  6 },
        { 29,  6 }, { 23,  3 }, { 10,  3 }, { 24,  4 }, { 28,  5 }, { 30,  5 },
        {  9,  3 }, {  4,  7 }, { 17,  8 }, { 16,  8 }, {  2,  6 }, { 26,  5 },
        { 13,  5 }, {  1,  5 }, { 31,  6 }, { 14,  6 }, {  3,  6 }, { 27,  6 },
        { 25,  5 }, { 12,  5 },
    },
    { /* AC bias group 1, table 15 */
        {  1,  4 }, {  7,  6 }, { 31,  6 }, {  2,  5 }, { 24,  4 }, { 30,  5 },
        {  5,  7 }, { 20, 12 }, { 19, 12 }, { 22, 12 }, { 21, 12 }, { 18, 10 },
        { 17,  9 }, { 15,  8 }, { 29,  6 }, {  0,  3 }, { 10,  3 }, {  9,  3 },
        { 23,  3 }, { 28,  5 }, {  4,  6 }, { 16,  9 }, {  6,  9 }, {  8,  8 },
        { 14,  7 }, { 12,  5 }, { 11,  5 }, {  3,  5 }, { 26,  5 }, { 27,  6 },
        { 13,  6 }, { 25,  5 },
    },
    { /* AC bias group 2, table 0 */
        { 18,  4 }, { 15,  4 }, {  0,  5 }, { 26,  6 }, { 31,  6 }, { 19,  5 },
        { 24,  5 }, { 10,  3 }, {  9,  3 }, { 14,  4 }, { 16,  5 }, {  7,  5 },
        { 12,  4 }, { 11,  4 }, {  8,  8 }, {  2,  9 }, { 21, 10 }, {  5, 13 },
        {  4, 13 }, { 22, 13 }, {  6, 13 }, {  3, 11 }, { 27,  7 }, { 25,  6 },
        { 30,  5 }, { 23,  4 }, { 13,  4 }, { 28,  6 }, { 29,  8 }, {  1,  8 },
        { 20,  7 }, { 17,  5 },
    },
    { /* AC bias group 2, table 1 */
        { 13,  3 }, { 28,  5 }, { 16,  5 }, { 26,  6 }, {  1,  7 }, {  8,  8 },
        { 20,  8 }, { 18,  5 }, { 14,  4 }, {  0,  5 }, { 31,  6 }, { 19,  6 },
        { 10,  3 }, {  9,  3 }, {  7,  5 }, { 17,  5 }, { 12,  4 }, { 11,  4 },
        { 24,  5 }, { 30,  5 }, { 15,  5 }, { 25,  6 }, {  2,  9 }, { 21, 11 },
        {  4, 12 }, {  5, 13 }, { 22, 14 }, {  6, 14 }, {  3, 10 }, { 29,  8 },
        { 27,  7 }, { 23,  4 },
    },
    { /* AC bias group 2, table 2 */
        { 23,  3 }, { 18,  5 }, { 16,  5 }, { 27,  6 }, { 26,  6 }, { 17,  5 },
        { 14,  4 }, { 28,  5 }, {  2,  8 }, {  8,  8 }, {  1,  7 }, { 31,  6 },
        { 10,  3 }, {  9,  3 }, {  7,  5 }, { 15,  5 }, { 11,  4 }, { 12,  4 },
        {  0,  5 }, { 24,  5 }, { 13,  4 }, { 30,  5 }, { 19,  7 }, {  3,  9 },
        {  4, 11 }, {  6, 13 }, {  5, 13 }, { 22, 13 }, { 21, 13 }, { 20, 10 },
        { 29,  8 }, { 25,  6 },
    },
    { /* AC bias group 2, table 3 */
        {  0,  4 }, { 14,  4 }, { 23,  3 }, { 25,  5 }, { 29,  7 }, {  2,  8 },
        { 20, 10 }, {  6, 13 }, {  5, 13 }, { 22, 13 }, { 21, 13 }, {  4, 11 },
        {  3,  9 }, { 27,  6 }, { 15,  5 }, {  7,  5 }, { 18,  6 }, { 26,  6 },
        { 28,  5 }, { 11,  4 }, { 10,  3 }, {  9,  3 }, { 12,  4 }, { 13,  4 },
        { 31,  6 }, { 16,  6 }, {  8,  8 }, { 19,  8 }, {  1,  7 }, { 17,  6 },
        { 30,  5 }, { 24,  5 },
    },
    { /* AC bias group 2, table 4 */
        { 28,  4 }, { 15,  5 }, { 17,  6 }, { 16,  6 }, { 23,  3 }, {  0,  4 },
        {  1,  6 }, { 18,  7 }, { 22, 12 }, { 21, 12 }, {  5, 12 }, { 20, 13 },
        {  6, 13 }, {  4, 10 }, { 19,  9 }, {  3,  8 }, {  7,  5 }, { 25,  5 },
        { 27,  6 }, { 29,  7 }, {  2,  8 }, {  8,  8 }, { 13,  4 }, { 11,  4 },
        { 12,  4 }, { 10,  3 }, {  9,  3 }, { 31,  6 }, { 26,  6 }, { 30,  5 },
        { 14,  5 }, { 24,  5 },
    },
    { /* AC bias group 2, table 5 */
        { 24,  4 }, {  7,  5 }, { 17,  7 }, {  3,  7 }, {  2,  7 }, { 16,  7 },
        { 23,  3 }, { 13,  4 }, { 28,  4 }, { 14,  5 }, { 15,  6 }, {  1,  6 },
        {  0,  4 }, { 11,  4 }, { 12,  4 }, {  9,  3 }, { 10,  3 }, { 25,  5 },
        { 30,  5 }, { 31,  6 }, { 27,  6 }, { 19, 11 }, { 22, 13 }, { 21, 13 },
        {  5, 13 }, { 20, 14 }, {  6, 14 }, {  4, 10 }, { 18,  9 }, {  8,  8 },
        { 29,  7 }, { 26,  6 },
    },
    { /* AC bias group 2, table 6 */
        { 27,  5 }, { 26,  5 }, {  7,  5 }, {  3,  6 }, {  8,  7 }, { 15,  7 },
        { 23,  3 }, { 24,  4 }, {  1,  5 }, { 29,  6 }, { 14,  6 }, { 11,  4 },
        { 12,  4 }, { 10,  3 }, {  9,  3 }, { 25,  5 }, { 30,  5 }, { 28,  4 },
        { 18, 10 }, {  5, 11 }, { 22, 13 }, { 21, 13 }, {  6, 13 }, { 20, 14 },
        { 19, 14 }, { 17,  9 }, {  4,  9 }, { 16,  9 }, {  2,  7 }, { 31,  6 },
        { 13,  5 }, {  0,  4 },
    },
    { /* AC bias group 2, table 7 */
        { 24,  4 }, { 11,  4 }, { 23,  3 }, {  0,  3 }, { 30,  5 }, {  3,  5 },
        { 12,  4 }, {  9,  3 }, { 10,  3 }, { 14,  7 }, {  4,  7 }, {  7,  6 },
        { 25,  5 }, { 28,  4 }, { 29,  6 }, { 27,  6 }, { 13,  6 }, {  2,  6 },
        {  1,  5 }, { 26,  6 }, {  8,  8 }, {  5,  9 }, { 16, 11 }, { 22, 14 },
        { 21, 14 }, {  6, 13 }, { 18, 14 }, { 20, 15 }, { 19, 15 }, { 17, 13 },
        { 15, 10 }, { 31,  7 },
    },
    { /* AC bias group 2, table 8 */
        { 29,  5 }, {  7,  5 }, { 13,  4 }, { 28,  4 }, { 26,  5 }, { 14,  5 },
        { 12,  4 }, { 11,  4 }, { 23,  3 }, { 24,  4 }, { 18,  7 }, { 20, 10 },
        {  4, 11 }, {  6, 13 }, {  5, 13 }, { 22, 13 }, { 21, 13 }, { 19,  9 },
        {  3,  9 }, {  2,  9 }, { 15,  6 }, { 30,  5 }, { 10,  3 }, {  9,  3 },
        {  1,  7 }, { 16,  7 }, { 31,  6 }, { 25,  5 }, {  0,  5 }, { 27,  6 },
        { 17,  7 }, {  8,  7 },
    },
    { /* AC bias group 2, table 9 */
        {  8,  6 }, {  1,  6 }, { 29,  5 }, {  0,  4 }, { 14,  5 }, { 26,  5 },
        { 28,  4 }, { 12,  4 }, { 11,  4 }, { 23,  3 }, { 24,  4 }, {  4, 10 },
        { 20, 11 }, {  6, 13 }, {  5, 13 }, { 22, 13 }, { 21, 13 }, { 19,  9 },
        {  3,  8 }, { 17,  7 }, { 15,  6 }, { 30,  5 }, { 10,  3 }, {  9,  3 },
        { 31,  6 }, { 16,  7 }, { 18,  8 }, {  2,  8 }, { 25,  5 }, {  7,  6 },
        { 27,  6 }, { 13,  5 },
    },
    { /* AC bias group 2, table 10 */
        { 14,  5 }, { 27,  5 }, { 12,  4 }, { 11,  4 }, { 29,  5 }, {  8,  6 },
        {  1,  6 }, {  0,  4 }, { 26,  5 }, { 16,  7 }, { 19, 10 }, { 22, 13 },
        { 21, 13 }, {  5, 13 }, { 20, 14 }, {  6, 14 }, {  4, 11 }, { 18,  9 },
        {  3,  8 }, { 31,  6 }, { 23,  3 }, { 28,  4 }, { 24,  4 }, { 10,  3 },
        {  9,  3 }, { 30,  5 }, { 13,  5 }, {  7,  6 }, {  2,  8 }, { 17,  8 },
        { 15,  7 }, { 25,  5 },
    },
    { /* AC bias group 2, table 11 */
        { 15,  6 }, { 16,  7 }, {  3,  7 }, { 27,  5 }, { 12,  4 }, { 11,  4 },
        {  8,  6 }, {  2,  7 }, { 18,  9 }, { 19, 11 }, { 22, 13 }, { 21, 13 },
        {  5, 13 }, { 20, 14 }, {  6, 14 }, {  4, 10 }, { 17,  8 }, { 26,  5 },
        { 23,  3 }, { 30,  5 }, { 29,  5 }, { 24,  4 }, {  0,  4 }, { 28,  4 },
        { 10,  3 }, {  9,  3 }, { 31,  6 }, {  1,  6 }, { 13,  5 }, {  7,  6 },
        { 14,  6 }, { 25,  5 },
    },
    { /* AC bias group 2, table 12 */
        { 12,  4 }, { 11,  4 }, {  1,  5 }, { 15,  7 }, {  4,  9 }, { 22, 12 },
        { 21, 12 }, {  6, 13 }, {  5, 13 }, { 20, 13 }, { 19, 13 }, { 18, 10 },
        { 17,  9 }, { 16,  9 }, { 14,  6 }, { 30,  5 }, {  8,  6 }, { 31,  6 },
        { 27,  5 }, { 13,  5 }, { 26,  5 }, {  7,  6 }, {  3,  7 }, {  2,  7 },
        { 23,  3 }, { 24,  4 }, { 29,  5 }, { 25,  5 }, { 10,  3 }, {  9,  3 },
        { 28,  4 }, {  0,  4 },
    },
    { /* AC bias group 2, table 13 */
        { 25,  4 }, {  3,  6 }, { 31,  6 }, {  2,  6 }, {  7,  6 }, {  0,  3 },
        { 23,  3 }, { 27,  5 }, {  8,  6 }, { 15,  8 }, {  4,  9 }, { 16, 10 },
        { 17, 11 }, { 22, 14 }, { 21, 14 }, { 18, 13 }, {  5, 13 }, {  6, 14 },
        { 20, 15 }, { 19, 15 }, { 14,  7 }, {  1,  5 }, { 26,  5 }, { 10,  3 },
        {  9,  3 }, { 24,  4 }, { 29,  5 }, { 12,  5 }, { 11,  5 }, { 13,  6 },
        { 30,  6 }, { 28,  4 },
    },
    { /* AC bias group 2, table 14 */
        { 28,  3 }, { 23,  3 }, { 24,  4 }, { 27,  5 }, { 26,  5 }, {  9,  3 },
        { 10,  3 }, {  0,  3 }, {  8,  6 }, { 30,  6 }, { 12,  5 }, { 11,  5 },
        { 29,  5 }, { 25,  5 }, {  2,  6 }, { 31,  7 }, {  7,  7 }, {  1,  5 },
        {  3,  6 }, {  4,  8 }, { 15, 10 }, { 22, 14 }, { 21, 14 }, { 17, 13 },
        { 18, 14 }, {  6, 14 }, { 20, 14 }, { 19, 14 }, { 16, 12 }, {  5, 12 },
        { 14,  9 }, { 13,  7 },
    },
    { /* AC bias group 2, table 15 */
        { 27,  5 }, { 26,  5 }, {  8,  6 }, { 15, 11 }, { 16, 12 }, { 18, 13 },
        { 17, 13 }, { 20, 13 }, { 19, 13 }, { 22, 13 }, { 21, 13 }, {  6, 11 },
        { 14,  9 }, {  5,  8 }, { 13,  7 }, { 12,  5 }, {  9,  3 }, { 10,  3 },
        { 28,  3 }, {  3,  4 }, { 11,  5 }, { 25,  5 }, {  0,  3 }, {  1,  4 },
        { 29,  5 }, {  2,  5 }, {  4,  6 }, { 30,  7 }, {  7,  8 }, { 31,  8 },
        { 24,  5 }, { 23,  4 },
    },
    { /* AC bias group 3, table 0 */
        { 23,  3 }, { 17,  5 }, {  2,  8 }, {  3,  9 }, { 22, 12 }, { 21, 12 },
        {  4, 12 }, {  6, 13 }, {  5, 13 }, { 20, 10 }, {  1,  7 }, { 27,  6 },
        { 14,  4 }, { 25,  5 }, {  0,  5 }, {  7,  5 }, { 28,  5 }, { 10,  3 },
        {  9,  3 }, { 15,  5 }, {  8,  7 }, { 19,  7 }, { 26,  6 }, { 12,  4 },
        { 11,  4 }, { 13,  4 }, { 18,  6 }, { 31,  6 }, { 16,  6 }, { 29,  6 },
        { 24,  5 }, { 30,  5 },
    },
    { /* AC bias group 3, table 1 */
        { 19,  7 }, {  2,  8 }, { 20, 11 }, {  4, 11 }, {  6, 12 }, {  5, 12 },
        { 22, 12 }, { 21, 12 }, {  3,  9 }, { 18,  6 }, { 29,  5 }, { 24,  4 },
        { 23,  3 }, { 15,  5 }, {  7,  5 }, { 25,  5 }, { 27,  6 }, {  8,  7 },
        {  1,  7 }, { 16,  6 }, { 17,  6 }, { 28,  5 }, { 13,  4 }, { 12,  4 },
        { 11,  4 }, { 10,  3 }, {  9,  3 }, { 31,  6 }, { 26,  6 }, {  0,  5 },
        { 30,  5 }, { 14,  5 },
    },
    { /* AC bias group 3, table 2 */
        { 26,  5 }, {  7,  5 }, {  0,  4 }, { 23,  3 }, { 24,  4 }, { 13,  4 },
        { 18,  7 }, {  8,  7 }, { 27,  6 }, { 25,  5 }, { 29,  5 }, { 14,  5 },
        { 12,  4 }, { 11,  4 }, { 10,  3 }, {  9,  3 }, {  2,  8 }, {  4, 11 },
        { 22, 13 }, { 21, 13 }, {  5, 13 }, { 20, 14 }, {  6, 14 }, { 19, 10 },
        {  3,  9 }, {  1,  7 }, { 15,  6 }, { 31,  6 }, { 17,  7 }, { 16,  7 },
        { 30,  5 }, { 28,  5 },
    },
    { /* AC bias group 3, table 3 */
        { 28,  4 }, { 13,  4 }, { 24,  4 }, {  1,  6 }, { 15,  6 }, { 14,  5 },
        { 23,  3 }, {  0,  4 }, { 25,  5 }, { 16,  7 }, {  4, 10 }, {  5, 12 },
        { 20, 13 }, {  6, 13 }, { 22, 13 }, { 21, 13 }, { 19, 12 }, { 18,  9 },
        {  3,  8 }, { 31,  6 }, { 11,  4 }, { 12,  4 }, { 10,  3 }, { 30,  5 },
        { 29,  5 }, { 27,  6 }, {  8,  7 }, {  2,  8 }, { 17,  8 }, {  7,  6 },
        { 26,  6 }, {  9,  3 },
    },
    { /* AC bias group 3, table 4 */
        { 29,  4 }, { 27,  5 }, { 26,  5 }, { 24,  4 }, { 28,  4 }, { 23,  3 },
        { 11,  4 }, { 12,  4 }, { 30,  5 }, { 16,  8 }, {  4, 10 }, { 18, 11 },
        { 22, 13 }, { 21, 13 }, {  6, 14 }, {  5, 14 }, { 20, 14 }, { 19, 14 },
        { 17,  9 }, { 15,  7 }, { 31,  6 }, { 25,  5 }, {  7,  6 }, {  1,  6 },
        {  0,  4 }, { 13,  5 }, { 14,  6 }, {  8,  7 }, {  3,  8 }, {  2,  8 },
        { 10,  3 }, {  9,  3 },
    },
    { /* AC bias group 3, table 5 */
        { 14,  6 }, { 15,  8 }, {  4,  9 }, { 16, 10 }, { 18, 13 }, {  6, 13 },
        { 20, 13 }, { 19, 13 }, { 17, 12 }, { 22, 14 }, { 21, 14 }, {  5, 13 },
        {  8,  7 }, { 30,  5 }, { 11,  4 }, { 12,  4 }, {  1,  5 }, { 13,  5 },
        { 23,  3 }, { 24,  4 }, { 29,  4 }, { 26,  5 }, {  7,  6 }, { 31,  6 },
        { 28,  4 }, {  9,  3 }, { 10,  3 }, { 25,  5 }, {  3,  7 }, {  2,  7 },
        { 27,  6 }, {  0,  4 },
    },
    { /* AC bias group 3, table 6 */
        { 11,  4 }, { 12,  4 }, {  0,  3 }, { 23,  3 }, { 31,  6 }, {  7,  6 },
        { 26,  5 }, { 29,  4 }, {  9,  3 }, { 10,  3 }, { 28,  4 }, { 25,  5 },
        {  2,  6 }, { 13,  6 }, {  1,  5 }, {  8,  7 }, {  4,  8 }, { 20, 13 },
        { 19, 13 }, { 22, 13 }, { 21, 13 }, { 16, 13 }, {  6, 13 }, { 18, 13 },
        { 17, 13 }, { 15, 11 }, {  5, 11 }, { 14,  9 }, {  3,  6 }, { 30,  6 },
        { 27,  6 }, { 24,  5 },
    },
    { /* AC bias group 3, table 7 */
        {  3,  4 }, {  1,  4 }, { 23,  3 }, {  0,  3 }, {  9,  3 }, { 10,  3 },
        {  2,  5 }, { 25,  5 }, { 11,  5 }, { 13,  8 }, {  6, 10 }, { 18, 13 },
        { 17, 13 }, { 20, 13 }, { 19, 13 }, { 22, 13 }, { 21, 13 }, { 14, 13 },
        { 16, 14 }, { 15, 14 }, { 31,  9 }, {  8,  8 }, {  5,  8 }, {  4,  6 },
        { 28,  4 }, { 12,  5 }, { 26,  5 }, { 29,  4 }, { 24,  5 }, { 30,  7 },
        {  7,  7 }, { 27,  6 },
    },
    { /* AC bias group 3, table 8 */
        { 29,  4 }, { 12,  4 }, { 11,  4 }, {  0,  4 }, { 27,  5 }, {  8,  6 },
        {  7,  6 }, { 30,  5 }, {  2,  8 }, { 17,  8 }, { 15,  7 }, { 31,  6 },
        { 23,  3 }, { 28,  4 }, { 26,  5 }, { 13,  5 }, {  9,  3 }, { 10,  3 },
        { 24,  4 }, { 14,  6 }, { 16,  8 }, {  3,  9 }, {  5, 13 }, {  4, 13 },
        { 20, 13 }, {  6, 13 }, { 22, 13 }, { 21, 13 }, { 19, 12 }, { 18, 10 },
        {  1,  7 }, { 25,  5 },
    },
    { /* AC bias group 3, table 9 */
        { 12,  4 }, { 11,  4 }, { 30,  5 }, { 27,  5 }, { 14,  6 }, {  8,  6 },
        { 13,  5 }, { 23,  3 }, { 29,  4 }, {  7,  6 }, { 31,  6 }, { 26,  5 },
        {  0,  4 }, { 24,  4 }, {  9,  3 }, { 10,  3 }, { 28,  4 }, {  1,  6 },
        {  3,  8 }, {  2,  8 }, { 15,  8 }, { 16,  9 }, { 18, 11 }, {  6, 14 },
        {  5, 14 }, { 20, 14 }, { 19, 14 }, { 22, 14 }, { 21, 14 }, {  4, 13 },
        { 17, 10 }, { 25,  5 },
    },
    { /* AC bias group 3, table 10 */
        { 11,  4 }, { 30,  5 }, { 13,  5 }, { 25,  4 }, {  3,  7 }, {  2,  7 },
        {  7,  6 }, { 31,  6 }, {  8,  6 }, { 23,  3 }, { 27,  5 }, { 26,  5 },
        { 29,  4 }, { 24,  4 }, {  0,  4 }, {  9,  3 }, { 10,  3 }, { 16,  9 },
        { 17, 10 }, {  4, 11 }, { 22, 14 }, { 21, 14 }, { 18, 13 }, {  6, 14 },
        {  5, 14 }, { 20, 14 }, { 19, 14 }, { 15,  8 }, { 14,  7 }, {  1,  6 },
        { 12,  5 }, { 28,  4 },
    },
    { /* AC bias group 3, table 11 */
        {  0,  3 }, { 25,  4 }, {  1,  5 }, {  7,  6 }, {  8,  6 }, { 23,  3 },
        { 27,  5 }, {  3,  7 }, {  2,  7 }, { 30,  6 }, { 24,  4 }, { 10,  3 },
        {  9,  3 }, { 26,  5 }, { 12,  5 }, { 29,  4 }, { 11,  5 }, { 13,  6 },
        { 31,  7 }, { 15,  9 }, { 17, 11 }, {  5, 13 }, { 18, 14 }, {  6, 14 },
        { 20, 14 }, { 19, 14 }, { 22, 14 }, { 21, 14 }, { 16, 11 }, {  4, 11 },
        { 14,  8 }, { 28,  4 },
    },
    { /* AC bias group 3, table 12 */
        { 25,  4 }, { 30,  6 }, { 13,  6 }, {  2,  6 }, {  8,  6 }, { 23,  3 },
        {  0,  3 }, { 27,  5 }, { 12,  5 }, { 24,  4 }, { 11,  5 }, {  1,  5 },
        { 26,  5 }, {  4,  9 }, { 15, 10 }, {  6, 13 }, {  5, 13 }, { 18, 13 },
        { 17, 13 }, { 16, 12 }, { 20, 14 }, { 19, 14 }, { 22, 14 }, { 21, 14 },
        { 14,  8 }, { 31,  7 }, {  7,  7 }, {  3,  7 }, { 10,  3 }, {  9,  3 },
        { 28,  4 }, { 29,  4 },
    },
    { /* AC bias group 3, table 13 */
        {  1,  4 }, {  2,  5 }, { 11,  5 }, { 29,  3 }, { 24,  4 }, { 12,  5 },
        {  8,  6 }, { 13,  7 }, { 30,  7 }, {  0,  3 }, { 10,  3 }, {  9,  3 },
        { 27,  5 }, { 26,  5 }, { 25,  5 }, {  7,  7 }, { 31,  8 }, { 14, 10 },
        { 15, 12 }, {  5, 12 }, { 20, 14 }, { 19, 14 }, { 22, 14 }, { 21, 14 },
        { 16, 14 }, {  6, 14 }, { 18, 14 }, { 17, 14 }, {  4,  9 }, {  3,  6 },
        { 23,  4 }, { 28,  4 },
    },
    { /* AC bias group 3, table 14 */
        {  3,  4 }, { 12,  5 }, { 11,  5 }, { 29,  3 }, { 27,  5 }, { 25,  5 },
        { 23,  4 }, {  1,  4 }, { 26,  5 }, {  2,  5 }, {  9,  3 }, { 10,  3 },
        {  0,  3 }, { 24,  5 }, {  7,  7 }, {  4,  7 }, {  8,  7 }, {  5, 10 },
        { 22, 13 }, { 21, 13 }, {  6, 13 }, { 16, 14 }, { 15, 14 }, { 14, 12 },
        { 18, 14 }, { 17, 14 }, { 20, 14 }, { 19, 14 }, { 31,  9 }, { 13,  9 },
        { 30,  9 }, { 28,  4 },
    },
    { /* AC bias group 3, table 15 */
        {  9,  2 }, { 10,  2 }, {  0,  3 }, {  4,  5 }, { 25,  6 }, { 27,  6 },
        {  1,  4 }, {  3,  4 }, { 29,  4 }, {  5,  8 }, {  8,  9 }, {  7, 10 },
        { 18, 14 }, { 17, 14 }, { 20, 14 }, { 19, 14 }, { 14, 14 }, {  6, 14 },
        { 16, 14 }, { 15, 14 }, { 22, 14 }, { 21, 14 }, { 31, 13 }, { 30, 13 },
        { 13, 13 }, { 11,  7 }, { 12,  7 }, { 24,  7 }, {  2,  5 }, { 23,  6 },
        { 26,  6 }, { 28,  5 },
    },
    { /* AC bias group 4, table 0 */
        { 30,  4 }, { 13,  4 }, { 26,  5 }, { 14,  5 }, { 29,  6 }, { 15,  6 },
        { 17,  7 }, { 16,  7 }, {  1,  6 }, { 23,  3 }, { 24,  4 }, {  0,  4 },
        { 12,  4 }, { 11,  4 }, { 27,  6 }, {  7,  6 }, { 28,  5 }, { 25,  5 },
        { 18,  8 }, {  8,  9 }, { 19, 10 }, { 22, 13 }, { 21, 13 }, {  5, 13 },
        { 20, 14 }, {  6, 14 }, {  4, 11 }, {  3,  8 }, {  2,  8 }, { 31,  6 },
        { 10,  3 }, {  9,  3 },
    },
    { /* AC bias group 4, table 1 */
        { 25,  4 }, {  1,  5 }, { 27,  5 }, { 28,  4 }, { 12,  4 }, { 11,  4 },
        { 26,  5 }, {  7,  6 }, { 15,  7 }, {  3,  7 }, { 23,  3 }, { 24,  4 },
        { 30,  5 }, { 13,  5 }, { 14,  6 }, { 31,  6 }, {  2,  7 }, {  8,  9 },
        { 17,  9 }, {  4, 10 }, {  6, 13 }, { 20, 14 }, { 19, 14 }, { 22, 14 },
        { 21, 14 }, {  5, 13 }, { 18, 11 }, { 16,  9 }, { 29,  6 }, {  0,  4 },
        { 10,  3 }, {  9,  3 },
    },
    { /* AC bias group 4, table 2 */
        {  3,  6 }, {  2,  6 }, { 27,  5 }, { 12,  4 }, { 11,  4 }, { 29,  5 },
        {  1,  5 }, { 14,  6 }, {  7,  6 }, { 13,  5 }, { 28,  4 }, { 30,  5 },
        { 26,  5 }, { 24,  4 }, { 23,  3 }, {  9,  3 }, { 10,  3 }, { 22, 12 },
        { 21, 12 }, {  6, 12 }, { 20, 13 }, { 19, 13 }, { 18, 11 }, {  5, 11 },
        { 17,  9 }, {  8,  8 }, {  4,  9 }, { 16,  9 }, { 15,  8 }, { 31,  6 },
        { 25,  5 }, {  0,  4 },
    },
    { /* AC bias group 4, table 3 */
        { 11,  4 }, { 30,  5 }, { 27,  5 }, {  0,  3 }, { 16,  9 }, { 22, 12 },
        { 21, 12 }, { 18, 13 }, {  6, 13 }, { 20, 13 }, { 19, 13 }, { 17, 11 },
        {  5, 11 }, {  8,  8 }, {  4,  8 }, { 15,  8 }, {  2,  6 }, { 31,  6 },
        {  3,  6 }, { 28,  4 }, { 23,  3 }, { 29,  5 }, { 26,  5 }, { 24,  4 },
        {  9,  3 }, { 10,  3 }, {  1,  5 }, { 14,  7 }, {  7,  7 }, { 13,  6 },
        { 25,  5 }, { 12,  5 },
    },
    { /* AC bias group 4, table 4 */
        {  7,  6 }, { 31,  6 }, {  3,  5 }, { 25,  4 }, {  0,  3 }, { 27,  5 },
        { 15,  9 }, { 20, 13 }, { 19, 13 }, { 22, 13 }, { 21, 13 }, { 16, 13 },
        {  6, 13 }, { 18, 13 }, { 17, 13 }, {  5, 10 }, {  8,  8 }, { 14,  8 },
        {  4,  8 }, { 13,  6 }, { 24,  4 }, { 23,  3 }, { 28,  4 }, { 26,  5 },
        { 30,  6 }, {  2,  6 }, {  9,  3 }, { 10,  3 }, { 12,  5 }, { 11,  5 },
        { 29,  5 }, {  1,  5 },
    },
    { /* AC bias group 4, table 5 */
        { 29,  4 }, {  1,  4 }, { 23,  3 }, { 24,  4 }, { 27,  5 }, {  3,  5 },
        {  0,  3 }, { 28,  4 }, { 26,  5 }, {  8,  8 }, {  5,  9 }, { 16, 13 },
        { 15, 13 }, { 18, 13 }, { 17, 13 }, {  6, 12 }, { 20, 14 }, { 19, 14 },
        { 22, 14 }, { 21, 14 }, { 14, 10 }, { 31,  7 }, {  4,  7 }, {  7,  7 },
        { 10,  3 }, {  9,  3 }, { 12,  5 }, { 11,  5 }, { 25,  5 }, { 13,  7 },
        { 30,  7 }, {  2,  6 },
    },
    { /* AC bias group 4, table 6 */
        { 24,  4 }, { 29,  4 }, {  4,  5 }, { 25,  5 }, {  3,  4 }, {  0,  3 },
        { 28,  4 }, {  1,  4 }, { 10,  3 }, {  7,  6 }, { 13,  8 }, { 30,  8 },
        { 31,  9 }, {  8, 10 }, {  6, 11 }, { 18, 14 }, { 17, 14 }, { 20, 14 },
        { 19, 14 }, { 22, 14 }, { 21, 14 }, { 14, 14 }, { 16, 15 }, { 15, 15 },
        {  5,  8 }, {  2,  5 }, { 26,  5 }, { 12,  5 }, {  9,  3 }, { 11,  5 },
        { 27,  5 }, { 23,  4 },
    },
    { /* AC bias group 4, table 7 */
        {  9,  2 }, { 27,  3 }, {  4,  4 }, {  1,  4 }, {  0,  3 }, { 29,  6 },
        {  5,  6 }, { 23,  5 }, { 28,  4 }, {  3,  4 }, {  2,  5 }, { 26,  6 },
        { 11,  7 }, { 12,  9 }, {  6,  9 }, {  8, 13 }, { 14, 14 }, { 13, 14 },
        { 31, 14 }, { 30, 14 }, {  7, 13 }, { 20, 14 }, { 19, 14 }, { 22, 14 },
        { 21, 14 }, { 16, 14 }, { 15, 14 }, { 18, 14 }, { 17, 14 }, { 25, 10 },
        { 24,  9 }, { 10,  3 },
    },
    { /* AC bias group 4, table 8 */
        { 30,  5 }, {  3,  6 }, {  2,  6 }, { 25,  4 }, {  0,  3 }, { 31,  6 },
        {  8,  7 }, { 15,  8 }, { 17, 10 }, {  6, 13 }, {  5, 13 }, { 20, 13 },
        { 19, 13 }, { 22, 13 }, { 21, 13 }, { 18, 12 }, { 16, 10 }, {  4, 10 },
        {  1,  5 }, { 27,  5 }, { 14,  7 }, {  7,  7 }, { 13,  6 }, { 23,  3 },
        { 11,  5 }, { 12,  5 }, { 28,  4 }, { 26,  5 }, { 29,  5 }, { 24,  4 },
        {  9,  3 }, { 10,  3 },
    },
    { /* AC bias group 4, table 9 */
        { 25,  4 }, { 29,  4 }, {  0,  3 }, { 13,  6 }, {  8,  7 }, {  7,  7 },
        { 30,  6 }, {  3,  6 }, { 27,  5 }, { 12,  5 }, { 23,  3 }, { 11,  5 },
        {  2,  6 }, { 31,  7 }, { 14,  8 }, {  4,  9 }, { 15, 10 }, {  5, 12 },
        { 18, 14 }, {  6, 14 }, { 20, 14 }, { 19, 14 }, { 22, 14 }, { 21, 14 },
        { 17, 13 }, { 16, 12 }, { 24,  4 }, {  9,  3 }, { 10,  3 }, { 26,  5 },
        {  1,  5 }, { 28,  4 },
    },
    { /* AC bias group 4, table 10 */
        { 25,  4 }, { 13,  6 }, { 30,  6 }, { 27,  5 }, { 29,  4 }, { 12,  5 },
        { 11,  5 }, { 23,  3 }, {  0,  3 }, {  8,  7 }, {  7,  7 }, { 31,  7 },
        {  4,  8 }, { 15, 10 }, {  6, 13 }, { 18, 14 }, { 17, 14 }, { 16, 12 },
        { 20, 14 }, { 19, 14 }, { 22, 14 }, { 21, 14 }, {  5, 12 }, { 14,  9 },
        { 26,  5 }, { 24,  4 }, {  9,  3 }, { 10,  3 }, { 28,  4 }, {  2,  6 },
        {  3,  6 }, {  1,  5 },
    },
    { /* AC bias group 4, table 11 */
        {  7,  7 }, { 31,  7 }, {  8,  7 }, { 13,  7 }, { 12,  5 }, {  1,  4 },
        { 23,  3 }, { 11,  5 }, {  3,  5 }, { 29,  4 }, { 24,  4 }, { 27,  5 },
        { 26,  5 }, {  0,  3 }, {  9,  3 }, { 10,  3 }, { 28,  4 }, { 30,  7 },
        { 20, 13 }, { 19, 13 }, { 22, 13 }, { 21, 13 }, { 16, 13 }, {  6, 13 },
        { 18, 13 }, { 17, 13 }, { 15, 11 }, {  5, 11 }, { 14,  9 }, {  4,  8 },
        {  2,  6 }, { 25,  5 },
    },
    { /* AC bias group 4, table 12 */
        { 23,  3 }, { 12,  5 }, { 11,  5 }, { 24,  4 }, { 27,  5 }, { 26,  5 },
        {  1,  4 }, { 29,  4 }, {  2,  5 }, {  5,  9 }, { 14, 10 }, { 18, 13 },
        { 17, 13 }, { 20, 13 }, { 19, 13 }, { 22, 13 }, { 21, 13 }, {  6, 13 },
        { 16, 14 }, { 15, 14 }, { 31,  8 }, {  7,  8 }, { 13,  8 }, {  4,  7 },
        { 30,  8 }, {  8,  8 }, {  0,  3 }, { 28,  4 }, { 25,  5 }, {  3,  5 },
        {  9,  3 }, { 10,  3 },
    },
    { /* AC bias group 4, table 13 */
        {  9,  2 }, {  3,  4 }, { 28,  4 }, { 25,  5 }, {  4,  6 }, { 30,  8 },
        {  8,  8 }, { 13,  9 }, {  7,  9 }, {  5,  9 }, { 18, 13 }, { 17, 13 },
        { 20, 13 }, { 19, 13 }, {  6, 13 }, { 16, 14 }, { 15, 14 }, { 22, 14 },
        { 21, 14 }, { 14, 13 }, { 31, 10 }, { 29,  4 }, { 23,  4 }, { 12,  6 },
        { 11,  6 }, { 24,  5 }, {  0,  3 }, {  1,  4 }, {  2,  5 }, { 27,  6 },
        { 26,  6 }, { 10,  3 },
    },
    { /* AC bias group 4, table 14 */
        { 10,  2 }, {  9,  2 }, {  5,  7 }, {  7,  9 }, { 31, 11 }, { 18, 13 },
        { 17, 13 }, { 20, 13 }, { 19, 13 }, { 13, 10 }, { 30, 10 }, {  6, 11 },
        { 22, 13 }, { 21, 13 }, { 14, 13 }, { 16, 14 }, { 15, 14 }, {  8,  9 },
        { 12,  6 }, { 23,  5 }, { 11,  6 }, { 25,  6 }, { 28,  5 }, {  0,  3 },
        {  3,  4 }, {  1,  4 }, { 26,  6 }, { 27,  6 }, { 24,  6 }, {  4,  6 },
        { 29,  5 }, {  2,  5 },
    },
    { /* AC bias group 4, table 15 */
        {  0,  2 }, { 28,  6 }, {  5,  7 }, { 24,  9 }, { 11,  9 }, { 23,  8 },
        {  4,  5 }, {  3,  4 }, { 27,  6 }, { 29,  7 }, { 12,  9 }, {  7, 13 },
        {  6, 13 }, { 14, 13 }, {  8, 13 }, { 31, 13 }, { 30, 13 }, { 13, 12 },
        { 20, 14 }, { 19, 14 }, { 22, 14 }, { 21, 14 }, { 16, 14 }, { 15, 14 },
        { 18, 14 }, { 17, 14 }, { 25, 11 }, { 26,  8 }, {  2,  5 }, {  1,  4 },
        { 10,  2 }, {  9,  2 },
    },
};

#endif /* AVCODEC_VP4DATA_H */
