package bodybuilder.test;

import java.io.File;
import java.io.IOException;

import junit.framework.Test;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import bodybuilder.exception.BodyBuilderException;
import bodybuilder.util.jdom.ExtendedSAXBuilder;

/**
 * XML
 */
public abstract class XML {

    /**
     * SAXビルダー
     */
    private static final ExtendedSAXBuilder builder = new ExtendedSAXBuilder();

    /**
     * ルート要素
     */
    protected Element root = null;

    /**
     * XMLタイプ
     */
    protected String xmlType = null;

    /**
     * ファイル
     */
    protected File file = null;

    /////////////////////////////////////////////////////////////////
    // instance builder

    /**
     * XMLを生成する。
     * 
     * @param filename ファイル名
     * @return XML
     */
    public static XML newXML(String filename) {
        return newXML(new File(filename));
    }

    /**
     * XMLを生成する。
     * 
     * @param file ファイル
     * @return XML
     */
    public static XML newXML(File file) {
        try {
            Document doc = builder.build(file);
            Element root = doc.getRootElement();
            return newXML(file, root);
        } catch (JDOMException e) {
            throw new BodyBuilderException(e);
        } catch (IOException e) {
            throw new BodyBuilderException(e);
        }
    }

    /**
     * XMLを生成する。
     * 
     * @param file ファイル
     * @param root ルート要素
     * @return XML
     */
    public static XML newXML(File file, Element root) {
        String type = root.getName();
        XML xml = TestCaseMapping.getXML(type);
        xml.file = file;
        xml.root = root;
        xml.xmlType = type;
        xml.build();
        return xml;
    }

    /////////////////////////////////////////////////////////////////
    // abstract method

    /**
     * テストを取得する。
     */
    public abstract Test getTest();

    /**
     * XMLを構築する。
     */
    protected abstract void build();

    /////////////////////////////////////////////////////////////////
    // property

    /**
     * ファイルを取得する。
     * 
     * @return ファイル
     */
    public File getFile() {
        return file;
    }

    /**
     * ルート要素を取得する。
     * 
     * @return ルート要素
     */
    public Element getRoot() {
        return root;
    }

    /**
     * XMLタイプを取得する。
     * 
     * @return XMLタイプ
     */
    public String getXmlType() {
        return xmlType;
    }

}