/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.policytool.common.api.impl;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.text.MessageFormat;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.PropertyException;
import javax.xml.bind.Unmarshaller;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;

import org.apache.commons.logging.Log;

import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMPolicies;
import jp.co.ogis_ri.citk.policytool.common.logging.LogWrapperFactory;
import jp.co.ogis_ri.citk.policytool.common.util.ExceptionUtil;

/**
 * 非公開クラス
 */
class OpenAMModelContext {
    /**
     * ロガー.
     */
    private static final Log logger =
            LogWrapperFactory.getLog(OpenAMModelContext.class);

    /**
     * DTD
     */
    private static final String DTD_OPENAM_POLICIES =
            "<!DOCTYPE Policies PUBLIC \"-//OpenSSO Policy Administration DTD//EN\" \"jar://com/sun/identity/policy/policyAdmin.dtd\">";

    /**
     * XML宣言
     */
    private static final String XML_DEFINITION =
            "<?xml version=\"1.0\" encoding=\"{0}\"?>";

    /**
     * OpenAMモデルのコンテキスト
     */
    private static JAXBContext context = null;

    /**
     * OpenAMモデルのマーシャラー
     */
    private static Marshaller marshaller = null;

    /**
     * OpenAMモデルのアンマーシャラー
     */
    private static Unmarshaller unmarshaller = null;

    /**
     * JAXBで利用するエンコード.
     */
    private static String jaxbEncoding = "UTF-8";

    /**
     * JAXBでXML出力時に整形する.
     */
    private static boolean jaxbFormatOutput = false;

    /**
     * JAXBで利用するエンコードを取得する.
     * 
     * @return JAXBで利用するエンコード.
     */
    public static String getJaxbEncoding() {
        return jaxbEncoding;
    }

    /**
     * JAXBで利用するエンコードを設定する.
     * 
     * @param jaxbEncoding JAXBで利用するエンコード.
     */
    public static void setJaxbEncoding(String jaxbEncoding) {
        if (marshaller != null) {
            try {
                marshaller.setProperty(Marshaller.JAXB_ENCODING, jaxbEncoding);
            } catch (PropertyException e) {
                logger.debug(e.getMessage());
                throw ExceptionUtil.convertRuntimeException(e);
            }
        }

        OpenAMModelContext.jaxbEncoding = jaxbEncoding;
    }

    /**
     * JAXBでのXML出力時に整形するかどうかを示す値を取得する.
     * 
     * @return JAXBでのXML出力時に整形するかどうかを示す値.
     */
    public static boolean isJaxbFormatoutput() {
        return jaxbFormatOutput;
    }

    /**
     * JAXBでのXML出力時に整形するかどうかを示す値を設定する.
     * 
     * @param jaxbFormatOutput JAXBでのXML出力時に整形するかどうかを示す値.
     */
    public static void setJaxbFormatOutput(boolean jaxbFormatOutput) {
        if (marshaller != null) {
            try {
                marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT,
                        jaxbFormatOutput);
            } catch (PropertyException e) {
                logger.debug(e.getMessage());
                throw ExceptionUtil.convertRuntimeException(e);
            }
        }

        OpenAMModelContext.jaxbFormatOutput = jaxbFormatOutput;
    }

    /**
     * コンストラクタ.
     */
    private OpenAMModelContext() {
    }

    /**
     * マーシャル.
     * 
     * @param jaxbElement JAXBのXML要素.
     * @param outputStream 出力先のOutputStream.
     */
    public static void marshaller(OpenAMPolicies jaxbElement,
            OutputStream outputStream) {
        String lineSeparator = System.getProperty("line.separator");
        BufferedOutputStream stream = new BufferedOutputStream(outputStream);
        try {
            initContext();
            // エンコード指定はプロパティ出し
            String xmldef = MessageFormat.format(XML_DEFINITION, jaxbEncoding);
            stream.write((xmldef + lineSeparator).getBytes());
            stream.write((DTD_OPENAM_POLICIES + lineSeparator).getBytes());
            marshaller.marshal(jaxbElement, stream);
            stream.close();
        } catch (JAXBException e) {
            logger.debug(e.getMessage());
            throw ExceptionUtil.convertRuntimeException(e);
        } catch (IOException e) {
            logger.debug(e.getMessage());
            throw ExceptionUtil.convertRuntimeException(e);
        } finally {
            if (outputStream != null) {
                try {
                    outputStream.close();
                } catch (IOException e) {
                    // NOP
                }
            }
            if (stream != null) {
                try {
                    stream.close();
                } catch (IOException e) {
                    // NOP
                }
            }
        }
    }

    /**
     * アンマーシャル.
     * 
     * @param clazz アンマーシャル先のクラス.
     * @param reader 入力元のInputStream.
     * @return アンマーシャル先のオブジェクト.
     */
    public static <T> T unmarshaller(Class<T> clazz, InputStream reader) {
        XMLInputFactory factory = XMLInputFactory.newInstance();
        BufferedInputStream stream = new BufferedInputStream(reader);
        XMLStreamReader xmlStreamReader = null;
        try {
            xmlStreamReader = factory.createXMLStreamReader(stream);
        } catch (XMLStreamException e) {
            logger.debug(e.getMessage());
            throw ExceptionUtil.convertRuntimeException(e);
        } finally {
            if (reader != null) {
                try {
                    reader.close();
                } catch (IOException e) {
                    // NOP
                }
            }
        }
        return unmarshaller(clazz, xmlStreamReader);
    }

    /**
     * アンマーシャル.
     * 
     * @param clazz アンマーシャル先のクラス.
     * @param xmlStreamReader 入力元のXMLStreamReader.
     * @return アンマーシャル先のオブジェクト.
     */
    public static <T> T unmarshaller(Class<T> clazz,
            XMLStreamReader xmlStreamReader) {
        T result = null;
        try {
            initContext();
            result = clazz.cast(unmarshaller.unmarshal(xmlStreamReader));
        } catch (JAXBException e) {
            logger.debug(e.getMessage());
            throw ExceptionUtil.convertRuntimeException(e);
        } finally {
            if (xmlStreamReader != null) {
                try {
                    xmlStreamReader.close();
                } catch (XMLStreamException e) {
                    // NOP
                }
            }
        }
        return result;
    }

    /**
     * JAXBContextの初期化.
     * 
     * @throws JAXBException JAXBContextの初期化失敗時
     * 
     */
    private static void initContext() throws JAXBException {
        if (context == null) {
            context =
                    JAXBContext.newInstance(OpenAMPolicies.class.getPackage()
                            .getName());

            marshaller = context.createMarshaller();
            marshaller.setProperty(Marshaller.JAXB_ENCODING, jaxbEncoding);
            marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT,
                    jaxbFormatOutput);
            // XML Headerは独自で出力しているので、このオプションは必須
            marshaller.setProperty(Marshaller.JAXB_FRAGMENT, Boolean.TRUE);
            unmarshaller = context.createUnmarshaller();
        }
    }

}
