/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.policytool.common.api.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang3.builder.ReflectionToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;
import org.apache.commons.logging.Log;

import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMPolicies;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMPolicy;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMReferral;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMResourceName;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMRule;
import jp.co.ogis_ri.citk.policytool.common.logging.LogWrapperFactory;
import jp.co.ogis_ri.citk.policytool.common.util.ApplicationContextUtil;
import jp.co.ogis_ri.citk.policytool.common.util.OpenAMNamingUtil;
import jp.co.ogis_ri.citk.policytool.domain.realm.model.Group;
import jp.co.ogis_ri.citk.policytool.domain.realm.model.Realm;
import jp.co.ogis_ri.citk.policytool.domain.realm.model.Referral;

/**
 * 非公開クラス
 */
class RealmsBuilder {

    /** ﾚﾙﾑを管理する */
    private List<String> realms = new ArrayList<String>();

    /** レルムに属するグループを管理する */
    private Map<String, List<Group>> groups =
            new TreeMap<String, List<Group>>();

    /** レルムに属する参照ポリシーを管理する */
    private Map<String, List<Referral>> refPolicies =
            new TreeMap<String, List<Referral>>();

    /**
     * コンストラクタ.
     */
    public RealmsBuilder() {
    }

    /**
     * ﾚﾙﾑを追加する.
     * 
     * @param realms OpenAMから返却されたﾚﾙﾑのList.
     * 
     */
    public void addRealms(List<String> realms) {
        this.realms.addAll(realms);
    }

    /**
     * ﾚﾙﾑに属するグループを追加する.
     * 
     * @param realmName ﾚﾙﾑ名.
     * @param groupValues OpenAMから返却されたグループ値のList.
     * 
     */
    public void addGroups(String realmName, List<String> groupValues) {
        List<Group> groups = null;
        if (this.groups.containsKey(realmName)) {
            groups = this.groups.get(realmName);
        } else {
            groups = new ArrayList<Group>();
            this.groups.put(realmName, groups);
        }
        for (String groupValue : groupValues) {
            String[] v = groupValue.split("\\(");
            groups.add(new Group(v[0].trim(), getGroupCode(groupValue)));
        }
    }

    /**
     * 参照ポリシーを追加する.
     * 
     * @param policies OpenAMから返却されたポリシー.
     * 
     */
    public void addPolicies(OpenAMPolicies policies) {

        List<Referral> refPolicies = null;

        for (OpenAMPolicy policy : policies.getPolicy()) {
            if (!policy.isReferralPolicy()) {
                continue;
            }

            // ルールがない場合はwarnログを出力して読み飛ばす.
            if (policy.getRule().size() == 0) {
                Log logger = LogWrapperFactory.getLog(RealmsBuilder.class);
                logger.warn(ApplicationContextUtil.getResources().getMessage(
                        "common.realmBuilder.noRuleMsg",
                        new Object[] {ReflectionToStringBuilder.toString(
                                policy, ToStringStyle.DEFAULT_STYLE)}));
                continue;
            }

            // 参照がない場合はwarnログを出力して読み飛ばす.
            if (policy.getReferrals() == null) {
                Log logger = LogWrapperFactory.getLog(RealmsBuilder.class);
                logger.warn(ApplicationContextUtil.getResources().getMessage(
                        "common.realmBuilder.noReferralMsg",
                        new Object[] {ReflectionToStringBuilder.toString(
                                policy, ToStringStyle.DEFAULT_STYLE)}));
                continue;
            }

            for (OpenAMReferral referral : policy.getReferrals().getReferral()) {
                if (!referral.getType().equalsIgnoreCase(
                        OpenAMNamingUtil.SUBJECT_REFERRAL_TYPE)) {
                    continue;
                }
                String value =
                        referral.getAttributeValuePair()
                                .get(0)
                                .getValue()
                                .get(0);
                Map<String, String> v = OpenAMNamingUtil.splitValue(value);
                // レルム名がない場合は読み飛ばす(referralをWarnLogに吐いて、continue)
                // ReflectionToStringBuilder.toString(referral,
                // ToStringStyle.DEFAULT_STYLE);
                // String realmName =
                // v.get(OpenAMNamingUtil.SUBJECT_REFERRAL_OWNER_KEY).trim();
                String realmName =
                        v.get(OpenAMNamingUtil.SUBJECT_REFERRAL_OWNER_KEY);
                if (realmName == null) {
                    Log logger = LogWrapperFactory.getLog(RealmsBuilder.class);
                    logger.warn(ApplicationContextUtil.getResources()
                            .getMessage(
                                    "common.realmBuilder.referralNoRealmnameMsg",
                                    new Object[] {ReflectionToStringBuilder.toString(
                                            referral,
                                            ToStringStyle.DEFAULT_STYLE)}));
                    continue;
                }
                realmName = realmName.trim();

                if (this.refPolicies.containsKey(realmName)) {
                    refPolicies = this.refPolicies.get(realmName);
                } else {
                    refPolicies = new ArrayList<Referral>();
                    this.refPolicies.put(realmName, refPolicies);
                }

                for (OpenAMRule rule : policy.getRule()) {
                    OpenAMResourceName resName = rule.getResourceName();
                    // ルールのリソースネームがない場合は読み飛ばす(ruleをWarnLogに吐いて、continue)
                    // ReflectionToStringBuilder.toString(rule,
                    // ToStringStyle.DEFAULT_STYLE);
                    if (resName == null || resName.getName() == null
                            || resName.getName().length() == 0) {
                        Log logger =
                                LogWrapperFactory.getLog(RealmsBuilder.class);
                        logger.warn(ApplicationContextUtil.getResources()
                                .getMessage(
                                        "common.realmBuilder.ruleNoResourceNameMsg",
                                        new Object[] {ReflectionToStringBuilder.toString(
                                                rule,
                                                ToStringStyle.DEFAULT_STYLE)}));
                        continue;

                    }
                    refPolicies.add(new Referral(rule.getResourceName()
                            .getName()));
                }
            }
        }
    }

    /**
     * ﾚﾙﾑを構築する. ﾚﾙﾑに属するグループ、および参照ポリシーを含んでいる。
     * 
     * @return　構築したﾚﾙﾑのList.
     * 
     */
    public List<Realm> build() {
        List<Realm> realms = new ArrayList<Realm>();
        for (String realmName : this.realms) {
            Realm realm = new Realm(realmName);
            realms.add(realm);
            if (this.groups.containsKey(realmName)) {
                realm.getGroups().addAll(this.groups.get(realmName));
            }
            if (this.refPolicies.containsKey(realmName)) {
                realm.getReferrals().addAll(this.refPolicies.get(realmName));
            }
            // レルムに追加したグループおよび参照ポリシーにレルム自体の参照をセットする.
            for (Group group : realm.getGroups()) {
                group.setRealm(realm);
            }
            for (Referral referral : realm.getReferrals()) {
                referral.setRealm(realm);
            }
        }
        return realms;
    }

    /**
     * OpenAMから返却されたグループ値からグループコードを取得する.
     * 
     * @param groupValue OpenAMから返却されたグループ値.
     * @return グループコード.
     * 
     */
    private String getGroupCode(String groupValue) {
        String text = "";
        Matcher m = Pattern.compile("\\(.*?\\)").matcher(groupValue);
        if (m.find()) {
            // 両端の括弧を取り除いてグループコードのみ取り出す
            text = groupValue.substring(m.start() + 1, m.end() - 1);
        }
        return text;
    }
}
