/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package jp.co.ogis_ri.citk.policytool.domain.policy.model;

import javax.persistence.Entity;
import javax.persistence.EnumType;
import javax.persistence.Enumerated;
import javax.persistence.FetchType;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import javax.persistence.ManyToOne;
import javax.persistence.Table;
import javax.persistence.Transient;
import javax.validation.constraints.NotNull;

import jp.co.ogis_ri.citk.policytool.common.model.AbstractModel;
import jp.co.ogis_ri.citk.policytool.common.validation.annotation.PermitAnnotation;
import jp.co.ogis_ri.citk.policytool.common.validation.annotation.Url;

/**
 * リソースを表すクラス.
 */
@Entity
@Table(name = "t_resource")
public class Resource extends AbstractModel<Long> {

    /**
     * デフォルト・シリアル･バージョンID
     */
    private static final long serialVersionUID = 1L;

    /**
     * JPAで永続化するためのサロゲートキー
     */
    @Id
    @GeneratedValue
    private Long id;

    /**
     * リソースURL
     */
    @Url
    private String resourceUrl = null;

    /**
     * GETの有無
     */
    @Enumerated(EnumType.STRING)
    private Permit getPermit = Permit.INDETERMINATE;

    /**
     * POSTの有無
     */
    @Enumerated(EnumType.STRING)
    private Permit postPermit = Permit.INDETERMINATE;

    /**
     * getPermitに格納する前の値
     */
    @Transient
    @PermitAnnotation
    private String getPermitValue = null;

    /**
     * postPermitに格納する前の値
     */
    @Transient
    @PermitAnnotation
    private String postPermitValue = null;

    /**
     * ポリシー
     */
    @ManyToOne(fetch = FetchType.LAZY)
    @NotNull
    private Policy policy = null;

    @Override
    public Long getId() {
        return this.id;
    }

    /**
     * リソースURLを取得する.
     */
    public String getResourceUrl() {
        return resourceUrl;
    }

    /**
     * リソースURLを設定する.
     * 
     * @param resourceUrl リソースURL.
     */
    public void setResourceURL(String resourceUrl) {
        this.resourceUrl = resourceUrl;
    }

    /**
     * GET権限値を取得する.
     * 
     * @return GET権限値.
     */
    public Permit getGetPermit() {
        return getPermit;
    }

    /**
     * GET権限値を設定する.
     * 
     * @param getPermit GET権限値.
     */
    public void setGetPermit(Permit getPermit) {
        this.getPermit = getPermit;
    }

    /**
     * POST権限値を取得する.
     * 
     * @return POST権限値.
     */
    public Permit getPostPermit() {
        return postPermit;
    }

    /**
     * POST権限値を設定する.
     * 
     * @param postPermit POST権限値.
     */
    public void setPostPermit(Permit postPermit) {
        this.postPermit = postPermit;
    }

    /**
     * GET権限文字列を取得する.
     * 
     * @return GET権限文字列.
     */
    public String getGetPermitValue() {
        return getPermitValue;
    }

    /**
     * GET権限文字列を設定する.
     * 
     * @param getPermitValue GET権限文字列.
     */
    public void setGetPermitValue(String getPermitValue) {
        this.getPermitValue = getPermitValue;
    }

    /**
     * POST権限文字列を取得する.
     * 
     * @return POST権限文字列.
     */
    public String getPostPermitValue() {
        return postPermitValue;
    }

    /**
     * POST権限文字列を設定する.
     * 
     * @param postPermitValue POST権限文字列.
     */
    public void setPostPermitValue(String postPermitValue) {
        this.postPermitValue = postPermitValue;
    }

    /**
     * このリソースを持つポリシーを取得する.
     * 
     * @return policy ポリシー.
     */
    public Policy getPolicy() {
        return policy;
    }

    /**
     * このリソースを持つポリシーを設定する.
     * 
     * @param policy セットする policy
     */
    public void setPolicy(Policy policy) {
        this.policy = policy;
    }

    /**
     * このリソースの権限値に, 権限文字列の情報を反映させる.
     */
    public void resetPermit() {
        this.getPermit = Permit.valueOf(this.getPermitValue);
        this.postPermit = Permit.valueOf(this.postPermitValue);
    }

    /**
     * デフォルト･コンストラクタ
     */
    public Resource() {
    }

    /**
     * コンストラクタ.
     * 
     * @param resourceUrl リソースURL.
     * @param getPermit GET権限値.
     * @param postPermit POST権限値.
     */
    public Resource(String resourceUrl, Permit getPermit, Permit postPermit) {
        this.resourceUrl = resourceUrl;
        this.getPermit = getPermit;
        this.postPermit = postPermit;
    }

    /**
     * コンストラクタ.
     * 
     * @param resourceUrl リソースURL.
     * @param getPermitValue GET権限文字列.
     * @param postPermitValue POST権限文字列.
     */
    public Resource(String resourceUrl, String getPermitValue,
            String postPermitValue) {
        this.resourceUrl = resourceUrl;
        this.getPermitValue = getPermitValue;
        this.postPermitValue = postPermitValue;
    }
}
