/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.policytool.common.api.impl;

import static org.hamcrest.CoreMatchers.is;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.fail;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import jp.co.ogis_ri.citk.policytool.common.CommonConstants;
import jp.co.ogis_ri.citk.policytool.common.api.OpenAMAccess;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMAttribute;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMAttributeValuePair;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMPolicies;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMPolicy;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMReferral;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMReferrals;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMResourceName;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMRule;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMServiceName;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMSubject;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMSubjects;
import jp.co.ogis_ri.citk.policytool.common.api.impl.ssoadm.SsoadmClient;
import jp.co.ogis_ri.citk.policytool.common.api.impl.ssoadm.SsoadmResult;
import jp.co.ogis_ri.citk.policytool.common.exception.OpenAMAccessException;
import jp.co.ogis_ri.citk.policytool.common.http.HttpClientWrapper;
import jp.co.ogis_ri.citk.policytool.common.resource.MessageInfo;
import jp.co.ogis_ri.citk.policytool.common.util.ApplicationContextUtil;
import jp.co.ogis_ri.citk.policytool.common.util.OpenAMNamingUtil;
import jp.co.ogis_ri.citk.policytool.common.util.StreamUtil;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Permit;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Policy;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Resource;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Subject;
import jp.co.ogis_ri.citk.policytool.domain.realm.model.Group;
import jp.co.ogis_ri.citk.policytool.domain.realm.model.Realm;
import jp.co.ogis_ri.citk.policytool.domain.realm.model.Referral;
import mockit.Deencapsulation;
import mockit.Expectations;
import mockit.Mocked;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration({ "OpenAMAccessImplTest-context.xml" })
public class OpenAMAccessImplTest {
    @Autowired
    @Qualifier("policytoolProperty")
    private Properties props;
	
	@Autowired
	@Qualifier("openAMAccessImpl") 
    private OpenAMAccess access;
    
	@Mocked
	private SsoadmClient ssoadmClient;
	
	@Mocked
	private HttpClientWrapper httpClientWrapper;
	
	@SuppressWarnings("unused")
	@Mocked
	private MessageInfo messageInfo;
	
    @Before
    public void setUp() {
        Deencapsulation.setField(ApplicationContextUtil.class, "context", null);
        Deencapsulation.setField(
                ApplicationContextUtil.class,
                "FILE_APPLICATION_CONTEXT",
                "jp/co/ogis_ri/citk/policytool/common/api/impl/OpenAMAccessImplTest-context.xml");
    }
    
    @Test
    public void testGetRealms() {
    	// 取得されるデータ
    	final List<String> realmList = new ArrayList<String>();
    	realmList.add("test4");
    	realmList.add("test7");
    	final SsoadmResult getRealmsResult = new SsoadmResult(realmList, SsoadmResult.EXITCODE_SUCCESS);
        final List<String> groupList1 = new ArrayList<String>();
        groupList1.add("group1 (o=group1,p=123)");
        groupList1.add("group2 (o=group2,p=456)");
    	final SsoadmResult getGroupResult = new SsoadmResult(groupList1, SsoadmResult.EXITCODE_SUCCESS);
        final List<String> groupList2 = new ArrayList<String>();
    	groupList2.add("g11");
    	groupList2.add("g12");
    	final SsoadmResult getGroupResult2 = new SsoadmResult(groupList2, SsoadmResult.EXITCODE_SUCCESS);

        // ポリシーXML 作成
        OpenAMPolicies policies = new OpenAMPolicies();
        OpenAMPolicy policy1 = createPolicyForGetRealms();
        policies.getPolicy().add(policy1);
        String policyXml = toXml(policies);
        
    	final List<String> policyList = new ArrayList<String>();
    	policyList.add(policyXml);
    	
    	final SsoadmResult getPoliciesResult = new SsoadmResult(policyList, SsoadmResult.EXITCODE_SUCCESS);
    	
    	// 期待される処理
    	new Expectations() {
    		{
    			// 初期化
    			new RealmsBuilder();
    			new HttpClientWrapper();
    			httpClientWrapper.setEncoding(props.getProperty(CommonConstants.PROP_KEY_OPENAM_ENCODING));
    			new SsoadmClient(httpClientWrapper,
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_SSOADM_URL),
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_USERID),
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_PASSWORD));
    			
    			// レルム取得→builderに追加
            	ssoadmClient.getRealms("/", "*", false); result = getRealmsResult;
        		
        		// グループ取得→builderに追加
            	ssoadmClient.getGroup("test4", "*"); result = getGroupResult;
            	ssoadmClient.getGroup("test7", "*"); result = getGroupResult2;
        		
        		// ポリシー取得→builderに追加
            	ssoadmClient.getPolicies("/", "*"); result = getPoliciesResult;
        		
        		// シャットダウン
        		httpClientWrapper.shutdown();
        	}
        };
        
    	// テスト実行
        List<Realm> resultRealmList = access.getRealms("/", "*");
        
        // レルム数
        assertThat(resultRealmList.size(), is(2));

        // レルム1
        Realm resultRealm1 = resultRealmList.get(0);
        assertThat(resultRealm1.getRealmName(), is("test4"));
        
        assertThat(resultRealm1.getReferrals().size(), is(1));
        Referral resultReferral11 = resultRealm1.getReferrals().get(0);
        assertThat(resultReferral11.getRefpolicy(), is("nameresource222"));
        assertThat(resultRealm1.getGroups().size(), is(2));
        
        Group resultGroup1 = resultRealm1.getGroups().get(0);
        assertThat(resultGroup1.getGroupName(), is("group1"));
        assertThat(resultGroup1.getGroupCode(), is("o=group1,p=123"));
        Group resultGroup2 = resultRealm1.getGroups().get(1);
        assertThat(resultGroup2.getGroupName(), is("group2"));
        assertThat(resultGroup2.getGroupCode(), is("o=group2,p=456"));
    }
    
    /**
     * 取得レルムが 0 個
     *
     */
    @Test
    public void testGetRealms_Zero() {
    	// 取得されるデータ
    	final List<String> realmList = new ArrayList<String>();
    	final SsoadmResult getRealmsResult = new SsoadmResult(realmList, SsoadmResult.EXITCODE_SUCCESS);

        // ポリシーXML 作成
        OpenAMPolicies policies = new OpenAMPolicies();
        String policyXml = toXml(policies);
        
    	final List<String> policyList = new ArrayList<String>();
    	
    	policyList.add(policyXml);
    	
    	final SsoadmResult getPoliciesResult = new SsoadmResult(policyList, SsoadmResult.EXITCODE_SUCCESS);
    	
    	// 期待される処理
    	new Expectations() {
    		{
    			// 初期化
    			new RealmsBuilder();
    			new HttpClientWrapper();
    			
    			// createSsoadmClient の処理
    			httpClientWrapper.setEncoding(props.getProperty(CommonConstants.PROP_KEY_OPENAM_ENCODING));
    			new SsoadmClient(httpClientWrapper,
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_SSOADM_URL),
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_USERID),
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_PASSWORD));
    			
    			// レルム取得→builderに追加
            	ssoadmClient.getRealms("/", "*", false); result = getRealmsResult;
        		
        		// ポリシー取得→builderに追加
            	ssoadmClient.getPolicies("/", "*"); result = getPoliciesResult;
        		
        		// シャットダウン
        		httpClientWrapper.shutdown();
        	}
        };
        
    	// テスト実行
        List<Realm> resultRealmList = access.getRealms("/", "*");
        
        // レルム数
        assertThat(resultRealmList.size(), is(0));
    }
    
    @Test
    public void testGetRealms_GetRealmsExitCodeNotZero() {
    	// 取得されるデータ
    	final List<String> realmList = new ArrayList<String>();
    	final SsoadmResult getRealmsResult = new SsoadmResult(realmList, SsoadmResult.EXITCODE_SUCCESS + 4);

    	// 期待される処理
    	new Expectations() {
    		{
    			// 初期化
    			new RealmsBuilder();
    			new HttpClientWrapper();
    			
    			// createSsoadmClient の処理
    			httpClientWrapper.setEncoding(props.getProperty(CommonConstants.PROP_KEY_OPENAM_ENCODING));
    			new SsoadmClient(httpClientWrapper,
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_SSOADM_URL),
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_USERID),
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_PASSWORD));
    			
    			// レルム取得→builderに追加
            	ssoadmClient.getRealms("/", "*", false); result = getRealmsResult;
            	
            	// MessageInfo 作成される
            	new MessageInfo("E-0004", SsoadmResult.EXITCODE_SUCCESS + 4);
            	
        		// シャットダウン
        		httpClientWrapper.shutdown();
        	}
        };
        
    	// テスト実行
        try {
        	access.getRealms("/", "*");
        	fail("OpenAMAccessException がスローされませんでした。");
        } catch (OpenAMAccessException e) {
        	assertThat(e.getExitCode(), is(SsoadmResult.EXITCODE_SUCCESS + 4));
        }
    }
    
    @Test
    public void testGetRealms_GetGroupsExitCodeNotZero() {
    	// 取得されるデータ
    	final List<String> realmList = new ArrayList<String>();
    	realmList.add("test4");
    	realmList.add("test7");
    	final SsoadmResult getRealmsResult = new SsoadmResult(realmList, SsoadmResult.EXITCODE_SUCCESS);
        final List<String> groupList1 = new ArrayList<String>();
        groupList1.add("group1 (o=group1,p=123)");
    	final SsoadmResult getGroupResult = new SsoadmResult(groupList1, SsoadmResult.EXITCODE_SUCCESS + 3);

    	// 期待される処理
    	new Expectations() {
    		{
    			// 初期化
    			new RealmsBuilder();
    			new HttpClientWrapper();
    			httpClientWrapper.setEncoding(props.getProperty(CommonConstants.PROP_KEY_OPENAM_ENCODING));
    			new SsoadmClient(httpClientWrapper,
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_SSOADM_URL),
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_USERID),
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_PASSWORD));
    			
    			// レルム取得→builderに追加
            	ssoadmClient.getRealms("/", "*", false); result = getRealmsResult;
        		
        		// グループ取得→builderに追加
            	ssoadmClient.getGroup("test4", "*"); result = getGroupResult;
            	
            	// MessageInfo 作成される
            	new MessageInfo("E-0004", SsoadmResult.EXITCODE_SUCCESS + 3);
        		
        		// シャットダウン
        		httpClientWrapper.shutdown();
        	}
        };
        
    	// テスト実行
        try {
        	access.getRealms("/", "*");
        	fail("OpenAMAccessException がスローされませんでした。");
        } catch (OpenAMAccessException e) {
        	assertThat(e.getExitCode(), is(SsoadmResult.EXITCODE_SUCCESS + 3));
        }
    }
    
    @Test
    public void testGetRealms_GetPoliciesExitCodeNotZero() {
    	// 取得されるデータ
    	final List<String> realmList = new ArrayList<String>();
    	final SsoadmResult getRealmsResult = new SsoadmResult(realmList, SsoadmResult.EXITCODE_SUCCESS);

    	final List<String> policyList = new ArrayList<String>();
    	
    	final SsoadmResult getPoliciesResult = new SsoadmResult(policyList, SsoadmResult.EXITCODE_SUCCESS + 2);
    	
    	// 期待される処理
    	new Expectations() {
    		{
    			// 初期化
    			new RealmsBuilder();
    			new HttpClientWrapper();
    			
    			// createSsoadmClient の処理
    			httpClientWrapper.setEncoding(props.getProperty(CommonConstants.PROP_KEY_OPENAM_ENCODING));
    			new SsoadmClient(httpClientWrapper,
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_SSOADM_URL),
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_USERID),
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_PASSWORD));
    			
    			// レルム取得→builderに追加
            	ssoadmClient.getRealms("/", "*", false); result = getRealmsResult;
        		
        		// ポリシー取得→builderに追加
            	ssoadmClient.getPolicies("/", "*"); result = getPoliciesResult;
        		
            	// MessageInfo 作成される
            	new MessageInfo("E-0004", SsoadmResult.EXITCODE_SUCCESS + 2);
            	
        		// シャットダウン
        		httpClientWrapper.shutdown();
        	}
        };
        
    	// テスト実行
        try {
        	access.getRealms("/", "*");
        	fail("OpenAMAccessException がスローされませんでした。");
        } catch (OpenAMAccessException e) {
        	assertThat(e.getExitCode(), is(SsoadmResult.EXITCODE_SUCCESS + 2));
        }
    }
    
    @Test
    public void testGetPolicies() {
    	// 取得されるデータ
    	
        // ポリシーXML 作成
        OpenAMPolicies policies = new OpenAMPolicies();
        OpenAMPolicy policy1 = createPolicyForGetPolicies();
        policies.getPolicy().add(policy1);
        String policyXml = toXml(policies);
    	
    	final List<String> data = new ArrayList<String>();
    	data.add(policyXml);
    	
    	final SsoadmResult getPoliciesResult = new SsoadmResult(data, SsoadmResult.EXITCODE_SUCCESS);
    	
    	// 期待される処理
    	new Expectations() {
    		{
    			// 初期化
    			new PoliciesBuilder();
    			new HttpClientWrapper();
    			
    			// createSsoadmClient の処理
    			httpClientWrapper.setEncoding(props.getProperty(CommonConstants.PROP_KEY_OPENAM_ENCODING));
    			new SsoadmClient(httpClientWrapper,
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_SSOADM_URL),
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_USERID),
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_PASSWORD));
    			
        		// ポリシー取得→builderに追加
            	ssoadmClient.getPolicies("/", "pol"); result = getPoliciesResult;
        		
        		// シャットダウン
        		httpClientWrapper.shutdown();
        	}
        };
        
    	// テスト実行
        List<Policy> actualPolicyList = access.getPolicies("/", "pol");
        
        assertEquals(1, actualPolicyList.size());
        Policy actualPolicy = actualPolicyList.get(0);
        
        assertEquals("policyname", actualPolicy.getPolicyName());
        
        assertEquals(3, actualPolicy.getSubjects().size());
        assertEquals(1, actualPolicy.getResources().size());
        
        // Subject の確認
        Subject resultSubject1 = actualPolicy.getSubjects().get(0);
        assertEquals("username10", resultSubject1.getSubjectName());
        assertEquals("id=username10,ou=user,o=relmname,ou=services,dc=opensso,dc=java,dc=net", resultSubject1.getSubjectCode());
        
        // Resource の確認
        Resource resultResource1 = actualPolicy.getResources().get(0);
        assertEquals("http://host/resource/*", resultResource1.getResourceUrl());
        assertEquals(Permit.INDETERMINATE, resultResource1.getPostPermit());
        assertEquals(Permit.ALLOW, resultResource1.getGetPermit());
    }
    
    @Test
    public void testGetPolicies_GetPoliciesExitCodeNotZero() {
    	final List<String> data = new ArrayList<String>();
    	final SsoadmResult getPoliciesResult = new SsoadmResult(data, SsoadmResult.EXITCODE_SUCCESS + 9);
    	
    	// 期待される処理
    	new Expectations() {
    		{
    			// 初期化
    			new PoliciesBuilder();
    			new HttpClientWrapper();
    			
    			// createSsoadmClient の処理
    			httpClientWrapper.setEncoding(props.getProperty(CommonConstants.PROP_KEY_OPENAM_ENCODING));
    			new SsoadmClient(httpClientWrapper,
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_SSOADM_URL),
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_USERID),
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_PASSWORD));
    			
        		// ポリシー取得→builderに追加
            	ssoadmClient.getPolicies("/", "pol"); result = getPoliciesResult;
        		
            	// MessageInfo 作成される
            	new MessageInfo("E-0004", SsoadmResult.EXITCODE_SUCCESS + 9);
            	
        		// シャットダウン
        		httpClientWrapper.shutdown();
        	}
        };
        
    	// テスト実行
        try {
        	access.getPolicies("/", "pol");
	    	fail("OpenAMAccessException がスローされませんでした。");
	    } catch (OpenAMAccessException e) {
	    	assertThat(e.getExitCode(), is(SsoadmResult.EXITCODE_SUCCESS + 9));
	    }
    }
    
    @Test
    public void testCreatePolicy() {
    	final Policy policy = new Policy();
    	policy.setRealmName("testName");
    	
    	// 取得されるデータ
    	final SsoadmResult createPoliciesResult = new SsoadmResult(new ArrayList<String>(), SsoadmResult.EXITCODE_SUCCESS);
    	
    	// 期待される処理
    	new Expectations() {
    		{
    			// 初期化
    			new HttpClientWrapper();
    			
    			// createSsoadmClient の処理
    			httpClientWrapper.setEncoding(props.getProperty(CommonConstants.PROP_KEY_OPENAM_ENCODING));
    			new SsoadmClient(httpClientWrapper,
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_SSOADM_URL),
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_USERID),
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_PASSWORD));
    			
        		// ポリシー作成
        		ssoadmClient.createPolicies("testName", anyString); result = createPoliciesResult;
        		
        		// シャットダウン
        		httpClientWrapper.shutdown();
        	}
        };
        
    	// テスト実行
        access.createPolicy(policy);
    }
    
    @Test
    public void testCreatePolicy_CreatePoliciesExitCodeNotZero() {
    	final Policy policy = new Policy();
    	policy.setRealmName("testName");
    	
    	// 取得されるデータ
    	final SsoadmResult createPoliciesResult = new SsoadmResult(new ArrayList<String>(), SsoadmResult.EXITCODE_SUCCESS + 1);
    	
    	// 期待される処理
    	new Expectations() {
    		{
    			// 初期化
    			new HttpClientWrapper();
    			
    			// createSsoadmClient の処理
    			httpClientWrapper.setEncoding(props.getProperty(CommonConstants.PROP_KEY_OPENAM_ENCODING));
    			new SsoadmClient(httpClientWrapper,
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_SSOADM_URL),
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_USERID),
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_PASSWORD));
    			
        		// ポリシー作成
        		ssoadmClient.createPolicies("testName", anyString); result = createPoliciesResult;
        		
            	// MessageInfo 作成される
            	new MessageInfo("E-0004", SsoadmResult.EXITCODE_SUCCESS + 1);
            	
        		// シャットダウン
        		httpClientWrapper.shutdown();
        	}
        };
        
    	// テスト実行
        try {
	        access.createPolicy(policy);
	    	fail("OpenAMAccessException がスローされませんでした。");
	    } catch (OpenAMAccessException e) {
	    	assertThat(e.getExitCode(), is(SsoadmResult.EXITCODE_SUCCESS + 1));
	    }
    }
    
    @Test
    public void testDeletePolicy() {
    	final Policy policy = new Policy();
    	policy.setRealmName("testRealm");
    	policy.setPolicyName("testPolicy");
    	
    	// 取得されるデータ
    	final SsoadmResult deletePolicyResult = new SsoadmResult(new ArrayList<String>(), SsoadmResult.EXITCODE_SUCCESS);
    	
    	// 期待される処理
    	new Expectations() {
    		{
    			// 初期化
    			new HttpClientWrapper();
    			
    			// createSsoadmClient の処理
    			httpClientWrapper.setEncoding(props.getProperty(CommonConstants.PROP_KEY_OPENAM_ENCODING));
    			new SsoadmClient(httpClientWrapper,
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_SSOADM_URL),
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_USERID),
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_PASSWORD));
    			
        		// ポリシー作成
        		ssoadmClient.deletePolicy("testRealm", "testPolicy"); result = deletePolicyResult;
        		
        		// シャットダウン
        		httpClientWrapper.shutdown();
        	}
        };
        
    	// テスト実行
        access.deletePolicy(policy);
    }

    @Test
    public void testDeletePolicy_DeletePolicyExitCodeNotZero() {
    	final Policy policy = new Policy();
    	policy.setRealmName("testRealm");
    	policy.setPolicyName("testPolicy");
    	
    	// 取得されるデータ
    	final SsoadmResult deletePolicyResult = new SsoadmResult(new ArrayList<String>(), SsoadmResult.EXITCODE_SUCCESS + 1);
    	
    	// 期待される処理
    	new Expectations() {
    		{
    			// 初期化
    			new HttpClientWrapper();
    			
    			// createSsoadmClient の処理
    			httpClientWrapper.setEncoding(props.getProperty(CommonConstants.PROP_KEY_OPENAM_ENCODING));
    			new SsoadmClient(httpClientWrapper,
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_SSOADM_URL),
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_USERID),
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_PASSWORD));
    			
        		// ポリシー作成
        		ssoadmClient.deletePolicy("testRealm", "testPolicy"); result = deletePolicyResult;
        		
            	// MessageInfo 作成される
            	new MessageInfo("E-0004", SsoadmResult.EXITCODE_SUCCESS + 1);
            	
        		// シャットダウン
        		httpClientWrapper.shutdown();
        	}
        };
        
    	// テスト実行
        try {
	        access.deletePolicy(policy);
	    	fail("OpenAMAccessException がスローされませんでした。");
	    } catch (OpenAMAccessException e) {
	    	assertThat(e.getExitCode(), is(SsoadmResult.EXITCODE_SUCCESS + 1));
	    }
    }
    
    @Test
    public void testUpdatePolicy() throws IOException {
    	// 入力ポリシー
    	final Policy policy = new Policy();
    	policy.setRealmName("testRealm");
    	policy.setPolicyName("testPolicy");
    	
    	// 取得されるデータ delete
    	final SsoadmResult deletePolicyResult = new SsoadmResult(new ArrayList<String>(), SsoadmResult.EXITCODE_SUCCESS);
    	
    	// 期待される処理 delete
    	new Expectations() {
    		{
    			// 初期化
    			new HttpClientWrapper();
    			
    			// createSsoadmClient の処理
    			httpClientWrapper.setEncoding(props.getProperty(CommonConstants.PROP_KEY_OPENAM_ENCODING));
    			new SsoadmClient(httpClientWrapper,
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_SSOADM_URL),
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_USERID),
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_PASSWORD));
    			
        		// ポリシー作成
        		ssoadmClient.deletePolicy("testRealm", "testPolicy"); result = deletePolicyResult;
        		
        		// シャットダウン
        		httpClientWrapper.shutdown();
        	}
        };
        
    	// 取得されるデータ create
    	final SsoadmResult createPoliciesResult = new SsoadmResult(new ArrayList<String>(), SsoadmResult.EXITCODE_SUCCESS);
    	
    	// 期待される処理 create
    	new Expectations() {
    		{
    			// 初期化
    			new HttpClientWrapper();
    			
    			// createSsoadmClient の処理
    			httpClientWrapper.setEncoding(props.getProperty(CommonConstants.PROP_KEY_OPENAM_ENCODING));
    			new SsoadmClient(httpClientWrapper,
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_SSOADM_URL),
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_USERID),
    					props.getProperty(CommonConstants.PROP_KEY_OPENAM_PASSWORD));
    			
        		// ポリシー作成
        		ssoadmClient.createPolicies("testRealm", anyString); result = createPoliciesResult;
        		
        		// シャットダウン
        		httpClientWrapper.shutdown();
        	}
        };
        
    	// テスト実行
        access.updatePolicy(policy);
    }
    
    private String toXml(OpenAMPolicies policies) {
        String xml = null;
        ByteArrayOutputStream os= null;
        
        try {
            // Open AMのモデルからXMLへマーシャリングする
            os= new ByteArrayOutputStream();
            OpenAMModelContext.setJaxbEncoding(props.getProperty(CommonConstants.PROP_KEY_OPENAM_ENCODING));
            OpenAMModelContext.marshaller(policies, os);
            xml = StreamUtil.toString(new ByteArrayInputStream(os.toByteArray()));
        } finally {
            if(os != null) {
                try {
                    os.close();
                } catch (IOException e) {
                    // NOP
                }
            }
        }
        return xml;
    }    
    
    private OpenAMPolicy createPolicyForGetRealms() {
        OpenAMPolicy policy = new OpenAMPolicy();
        policy.setName("refPolicy");
        policy.setDescription("refpolicy_desc");
        policy.setCreatedby("id=amadmin,ou=user,dc=opensso,dc=java,dc=net");
        policy.setLastmodifiedby("id=amadmin,ou=user,dc=opensso,dc=java,dc=net");
        policy.setCreationdate(1323689719859L);
        policy.setLastmodifieddate(1323689719859L);
        policy.setReferralPolicy(true);
        policy.setActive(true);
        
        OpenAMRule rule = new OpenAMRule();
        rule.setName("http://name");
        policy.getRule().add(rule);
        
        OpenAMServiceName serviceName = new OpenAMServiceName();
        serviceName.setName(OpenAMNamingUtil.SERVICE_NAME);
        rule.setServiceName(serviceName);
        
        OpenAMResourceName resourceName = new OpenAMResourceName();
        resourceName.setName("nameresource222");
        rule.setResourceName(resourceName);
        
        OpenAMReferrals referrals = new OpenAMReferrals();
        referrals.setName("Referrals:1323689679750bRDOs40=");
        referrals.setDescription("");
        policy.setReferrals(referrals);
        
        OpenAMReferral referral = new OpenAMReferral();
        referral.setName("test4ref");
        referral.setType(OpenAMNamingUtil.SUBJECT_REFERRAL_TYPE);
        referrals.getReferral().add(referral);
        
        OpenAMAttributeValuePair attributeValuePair = new OpenAMAttributeValuePair();
        referral.getAttributeValuePair().add(attributeValuePair);
        
        OpenAMAttribute attribute = new OpenAMAttribute();
        attribute.setName(OpenAMNamingUtil.VALUES_ATTR_NAME);
        attributeValuePair.setAttribute(attribute);
        
        attributeValuePair.getValue().add("o=test4,ou=services,dc=opensso,dc=java,dc=net");
        
        return policy;
    }
    
    private OpenAMPolicy createPolicyForGetPolicies() {
        OpenAMPolicies policies = new OpenAMPolicies();

        OpenAMPolicy policy = new OpenAMPolicy();
        policy.setName("policyname");
        policy.setDescription("description");
        policy.setCreatedby("id=amadmin,ou=user,dc=opensso,dc=java,dc=net");
        policy.setLastmodifiedby("id=amadmin123,ou=user,dc=opensso,dc=java,dc=net");
        policy.setCreationdate(1319176315017L);
        policy.setLastmodifieddate(1319176315027L);
        policy.setReferralPolicy(false);
        policy.setActive(true);
        policies.getPolicy().add(policy);
        
        // Rule
        OpenAMRule rule1 = createRuleForGetPolicies();
        policy.getRule().add(rule1);
        
        // Subjects
        OpenAMSubjects subjects = new OpenAMSubjects();
        subjects.setName("subjectsname");
        subjects.setDescription("description");
        policy.setSubjects(subjects);
        
    	// Subject 1 個目
        OpenAMSubject subject1 = createSubjectForGetPolicies();
        subjects.getSubject().add(subject1);

        return policy;
    }
    
    private OpenAMRule createRuleForGetPolicies() {
        OpenAMRule rule1 = new OpenAMRule();
        rule1.setName("rulename1");
        
        OpenAMServiceName serviceName11 = new OpenAMServiceName();
        serviceName11.setName("iPlanetAMWebAgentServiceZZZ");
        rule1.setServiceName(serviceName11);
        
        OpenAMResourceName resourceName11 = new OpenAMResourceName();
        resourceName11.setName("http://host/resource/*");
        rule1.setResourceName(resourceName11);
        
        OpenAMAttributeValuePair attributeValuePair11 = new OpenAMAttributeValuePair();
        rule1.getAttributeValuePair().add(attributeValuePair11);
        
        OpenAMAttribute attribute11 = new OpenAMAttribute();
        attribute11.setName("POST123");
        attributeValuePair11.setAttribute(attribute11);
        
        attributeValuePair11.getValue().add("abc");
        
        OpenAMAttributeValuePair attributeValuePair12 = new OpenAMAttributeValuePair();
        rule1.getAttributeValuePair().add(attributeValuePair12);
        
        OpenAMAttribute attribute12 = new OpenAMAttribute();
        attribute12.setName(OpenAMNamingUtil.RULE_ATTR_GETMETHOD_NAME);
        attributeValuePair12.setAttribute(attribute12);
    
        attributeValuePair12.getValue().add("allow");
        
        return rule1;
    }
    
    private OpenAMSubject createSubjectForGetPolicies() {
        OpenAMSubject subject = new OpenAMSubject();
        subject.setName("subjectname1");
        subject.setType(OpenAMNamingUtil.SUBJECT_TYPE);
        subject.setIncludeType(OpenAMNamingUtil.SUBJECT_INCLUDE_TYPE);
        
        OpenAMAttributeValuePair attributeValuePair = new OpenAMAttributeValuePair();
        subject.getAttributeValuePair().add(attributeValuePair);
        
        OpenAMAttribute attribute = new OpenAMAttribute();
        attribute.setName(OpenAMNamingUtil.VALUES_ATTR_NAME);
        attributeValuePair.setAttribute(attribute);
        
        attributeValuePair.getValue().add("id=username10,ou=user,o=relmname,ou=services,dc=opensso,dc=java,dc=net");
        attributeValuePair.getValue().add("id=username11,ou=user,o=relmname,ou=services,dc=opensso,dc=java,dc=net");
        attributeValuePair.getValue().add("id=username12,ou=user,o=relmname,ou=services,dc=opensso,dc=java,dc=net");
        
        return subject;
    }
    
}
