/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.policytool.common.api.impl;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.util.ArrayList;
import java.util.List;

import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMAttribute;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMAttributeValuePair;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMPolicies;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMPolicy;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMResourceName;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMRule;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMServiceName;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMSubject;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMSubjects;
import jp.co.ogis_ri.citk.policytool.common.util.OpenAMNamingUtil;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Permit;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Policy;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Resource;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Subject;

import org.junit.Test;

public class OpenAMPoliciesBuilderTest {
    /**
     * ポリシーのプロパティ値<br>
     * ルール : POST
     *
     */
    @Test
    public void testBuild_Rule_GET_INDETERMINATE() {
        OpenAMPolicies openAMPolicies = buildOpenAMPolicies();
        
        List<OpenAMPolicy> policyBeanList = openAMPolicies.getPolicy();
        assertEquals(2, policyBeanList.size());
        
        OpenAMPolicy policy = policyBeanList.get(0);
        
        assertEquals(true, policy.isActive());
        assertEquals("", policy.getDescription());
        assertEquals("testPolicy", policy.getName());
        assertEquals(false, policy.isReferralPolicy());
        
        List<OpenAMRule> ruleList = policy.getRule();
        assertEquals(3, ruleList.size());
        
        OpenAMRule rule = ruleList.get(1);
        
        List<OpenAMAttributeValuePair> attributeValuePairList = rule.getAttributeValuePair();
        assertEquals(1, attributeValuePairList.size());
        
        OpenAMAttributeValuePair attributeValuePair = attributeValuePairList.get(0);
        OpenAMAttribute attribute = attributeValuePair.getAttribute();
        assertEquals(OpenAMNamingUtil.RULE_ATTR_POSTMETHOD_NAME, attribute.getName());
        assertEquals(1, attributeValuePair.getValue().size());
        assertEquals("allow", attributeValuePair.getValue().get(0));
    }
    
    /**
     * ルール : GET
     *
     *
     */
    @Test
    public void testBuild_Rule_POST_INDETERMINATE() {
        OpenAMPolicies openAMPolicies = buildOpenAMPolicies();
        
        List<OpenAMPolicy> policyBeanList = openAMPolicies.getPolicy();
        
        OpenAMPolicy policy = policyBeanList.get(0);
        
        List<OpenAMRule> ruleList = policy.getRule();
        
        OpenAMRule rule = ruleList.get(2);
        
        List<OpenAMAttributeValuePair> attributeValuePairList = rule.getAttributeValuePair();
        assertEquals(1, attributeValuePairList.size());
        
        OpenAMAttributeValuePair attributeValuePair = attributeValuePairList.get(0);
        OpenAMAttribute attribute = attributeValuePair.getAttribute();
        assertEquals(OpenAMNamingUtil.RULE_ATTR_GETMETHOD_NAME, attribute.getName());
        assertEquals(1, attributeValuePair.getValue().size());
        assertEquals("deny", attributeValuePair.getValue().get(0));
    }
    
	/**
	 * ルール : GET, POST 両方のルールがある場合, AttributeValuePair が 2 個あることを確認
	 *
	 */
    @Test
    public void testBuild_Rule_2() {
        OpenAMPolicies openAMPolicies = buildOpenAMPolicies();
        
        List<OpenAMPolicy> policyBeanList = openAMPolicies.getPolicy();
        
        // 1番目のポリシー
        OpenAMPolicy policy = policyBeanList.get(0);
        
        List<OpenAMRule> ruleList = policy.getRule();
        
        // 1番目のルール
        OpenAMRule rule = ruleList.get(0);
        assertTrue(rule.getName().startsWith("rule_"));
        
        OpenAMServiceName serviceName = rule.getServiceName();
        assertEquals(OpenAMNamingUtil.SERVICE_NAME, serviceName.getName());
        
        OpenAMResourceName resourceName = rule.getResourceName();
        assertEquals("https://zzzzz", resourceName.getName());
        
        List<OpenAMAttributeValuePair> attributeValuePairList = rule.getAttributeValuePair();
        assertEquals(2, attributeValuePairList.size());
    }
    
    /**
     * Subjects の確認
     *
     */
    @Test
    public void testBuild_Subjects() {
        OpenAMPolicies openAMPolicies = buildOpenAMPolicies();
        
        List<OpenAMPolicy> policyBeanList = openAMPolicies.getPolicy();
        OpenAMPolicy policy = policyBeanList.get(0);
        
        OpenAMSubjects subjectsBean = policy.getSubjects();
        assertTrue(subjectsBean.getName().startsWith("subjects_"));

        List<OpenAMSubject> subjectBeanList = subjectsBean.getSubject();
        assertEquals(1, subjectBeanList.size());
        
        // 1番目のサブジェクト
        OpenAMSubject subjectBean = subjectBeanList.get(0);
        assertEquals("inclusive", subjectBean.getIncludeType());
        assertTrue(subjectBean.getName().startsWith("subject_"));
        
        assertEquals(OpenAMNamingUtil.SUBJECT_TYPE, subjectBean.getType());
        
        List<OpenAMAttributeValuePair> attributeValuePairList = subjectBean.getAttributeValuePair();
        assertEquals(1, attributeValuePairList.size());
        
        OpenAMAttributeValuePair attributeValuePair = attributeValuePairList.get(0);
        
        OpenAMAttribute attribute = attributeValuePair.getAttribute();
        assertEquals(OpenAMNamingUtil.VALUES_ATTR_NAME, attribute.getName());
        
        assertEquals(2, attributeValuePair.getValue().size());
        assertEquals("Scode_1", attributeValuePair.getValue().get(0));
        assertEquals("Scode_2", attributeValuePair.getValue().get(1));
    }
    
    /**
     * ルール, サブジェクトが 0 個のポリシー
     *
     */
    @Test
    public void testBuild_RuleZERO_SubjectZERO() {
        OpenAMPolicies openAMPolicies = buildOpenAMPolicies();
        
        List<OpenAMPolicy> policyBeanList = openAMPolicies.getPolicy();
        
        // ポリシー
        OpenAMPolicy policy = policyBeanList.get(1);
        
        assertEquals(true, policy.isActive());
        assertEquals("", policy.getDescription());
        assertEquals("POLICY_2xxx", policy.getName());
        assertEquals(false, policy.isReferralPolicy());
        
        List<OpenAMRule> ruleList = policy.getRule();
        assertEquals(0, ruleList.size());
        
        OpenAMSubjects subjectsBean = policy.getSubjects();
        assertTrue(subjectsBean.getName().startsWith("subjects_"));
        
        List<OpenAMSubject> subjectBeanList = subjectsBean.getSubject();
        assertEquals(1, subjectBeanList.size());
        
    	// サブジェクト
        OpenAMSubject subjectBean = subjectBeanList.get(0);
        assertNull(subjectBean.getIncludeType());
        assertNull(subjectBean.getName());
        assertNull(subjectBean.getType());
        
        List<OpenAMAttributeValuePair> attributeValuePairList = subjectBean.getAttributeValuePair();
        assertEquals(0, attributeValuePairList.size());
    }
    
    /**
     * ポリシーが 0 個
     *
     *
     */
    @Test
    public void testBuild_PolicyZERO() {
        OpenAMPoliciesBuilder builder = new OpenAMPoliciesBuilder();
        OpenAMPolicies policies = builder.build();
        
        List<OpenAMPolicy> policyBeanList = policies.getPolicy();
        assertEquals(0, policyBeanList.size());
    }

	/**
	 * テスト用 Policy 作成
	 *
	 */
	private List<Policy> createPolicies() {
		// 1番目のポリシー 通常のパターン
        Subject[] subjects = new Subject[] { new Subject("Sname_a", "Scode_1"),
                new Subject("Sname_b", "Scode_2"), };

        Resource[] resources = new Resource[] {
                new Resource("https://zzzzz", Permit.ALLOW, Permit.DENY),
                new Resource("http://aaaaa", Permit.INDETERMINATE, Permit.ALLOW),
                new Resource("http://ttttt", Permit.DENY, Permit.INDETERMINATE),
                };

        Policy policy1 = new Policy("realm1", "testPolicy");
        for (Subject s : subjects) {
            policy1.addSubject(s);
        }
        
        for (Resource r : resources) {
            policy1.addResource(r);
        }
        
        // 2番目のポリシー Subject, Resource が 0 個
        Subject[] subjects2 = new Subject[] { };

        Resource[] resources2 = new Resource[] { };

        Policy policy2 = new Policy("realm2", "POLICY_2xxx");
        for (Subject s: subjects2) {
            policy2.addSubject(s);
        }
        
        for (Resource r: resources2) {
            policy2.addResource(r);
        }
        
        List<Policy> policyList = new ArrayList<Policy>();
        policyList.add(policy1);
        policyList.add(policy2);
        
        return policyList;
	}
	
	private OpenAMPolicies buildOpenAMPolicies() {
        // ポリシー作成
    	List<Policy> policyList = createPolicies();
    	Policy policy1 = policyList.get(0);
    	Policy policy2 = policyList.get(1);
        
        // テスト実行
        OpenAMPoliciesBuilder builder = new OpenAMPoliciesBuilder();
        builder.addPolicy(policy1);
        builder.addPolicy(policy2);
        OpenAMPolicies openAMPolicies = builder.build();
        
        return openAMPolicies;
	}
}
