/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.policytool.common.api.impl;

import static org.hamcrest.CoreMatchers.is;
import static org.junit.Assert.assertThat;

import java.util.ArrayList;
import java.util.List;

import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMAttribute;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMAttributeValuePair;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMPolicies;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMPolicy;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMReferral;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMReferrals;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMResourceName;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMRule;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMServiceName;
import jp.co.ogis_ri.citk.policytool.common.util.OpenAMNamingUtil;
import jp.co.ogis_ri.citk.policytool.domain.realm.model.Group;
import jp.co.ogis_ri.citk.policytool.domain.realm.model.Realm;
import jp.co.ogis_ri.citk.policytool.domain.realm.model.Referral;

import org.junit.Test;

public class RealmsBuilderTest {
	/**
	 * 通常パターン
	 *
	 *
	 */
    @Test
    public void testBuild() {
        // OpenAMPolicies 生成
        OpenAMPolicies policies = new OpenAMPolicies();
        
        // ポリシー作成
        OpenAMPolicy policy1 = createPolicy1();
        policies.getPolicy().add(policy1);
        OpenAMPolicy policy2 = createPolicy2();
        policies.getPolicy().add(policy2);
        OpenAMPolicy policy3 = createPolicy3();
        policies.getPolicy().add(policy3);
        
        // グループリスト作成
        List<String> groupList1 = new ArrayList<String>();
        groupList1.add("group1 (o=group1,p=123)");
        groupList1.add("group2 (o=group2,p=456)");
        List<String> groupList3 = new ArrayList<String>();
        groupList3.add("groupA (o=groupA,p=ABC)");
        
        // レルムリスト作成
        List<String> realmList1 = new ArrayList<String>();
        realmList1.add("test4");
        realmList1.add("test7");
        List<String> realmList2 = new ArrayList<String>();
        realmList2.add("test9");
        
        // Policy が 0 個の Policies
        OpenAMPolicies zeroPolicies = new OpenAMPolicies();
        
        // 作成処理
        RealmsBuilder builder = new RealmsBuilder();
        builder.addRealms(realmList1);
        builder.addRealms(realmList2);
        builder.addPolicies(policies);
        builder.addGroups("test4", groupList1);
        builder.addGroups("test9", groupList3);

        builder.addPolicies(zeroPolicies);
        
        List<Realm> resultRealmList = builder.build();

        // レルム数
        assertThat(resultRealmList.size(), is(3));

        // レルム1
        Realm resultRealm1 = resultRealmList.get(0);
        assertThat(resultRealm1.getRealmName(), is("test4"));
        
        assertThat(resultRealm1.getReferrals().size(), is(2));
        Referral resultReferral11 = resultRealm1.getReferrals().get(0);
        assertThat(resultReferral11.getRefpolicy(), is("nameresource222"));
        Referral resultReferral12 = resultRealm1.getReferrals().get(1);
        assertThat(resultReferral12.getRefpolicy(), is("nameresource333"));
        assertThat(resultRealm1.getGroups().size(), is(2));
        
        Group resultGroup1 = resultRealm1.getGroups().get(0);
        assertThat(resultGroup1.getGroupName(), is("group1"));
        assertThat(resultGroup1.getGroupCode(), is("o=group1,p=123"));
        Group resultGroup2 = resultRealm1.getGroups().get(1);
        assertThat(resultGroup2.getGroupName(), is("group2"));
        assertThat(resultGroup2.getGroupCode(), is("o=group2,p=456"));
        
        // レルム2
        Realm resultRealm2 = resultRealmList.get(1);
        assertThat(resultRealm2.getRealmName(), is("test7"));
        
        assertThat(resultRealm2.getReferrals().size(), is(1));
        Referral resultReferral21 = resultRealm2.getReferrals().get(0);
        assertThat(resultReferral21.getRefpolicy(), is("nameresource000"));
        assertThat(resultRealm2.getGroups().size(), is(0));
        
        // レルム3
        Realm resultRealm3 = resultRealmList.get(2);
        assertThat(resultRealm3.getRealmName(), is("test9"));
        
        assertThat(resultRealm3.getReferrals().size(), is(0));
        
        assertThat(resultRealm3.getGroups().size(), is(1));
        Group resultGroup31 = resultRealm3.getGroups().get(0);
        assertThat(resultGroup31.getGroupName(), is("groupA"));
        assertThat(resultGroup31.getGroupCode(), is("o=groupA,p=ABC"));
    }

	/**
	 * addRealms で加えられたレルム が 0 個
	 *
	 *
	 */
    @Test
    public void testBuild_RealmsZero() {
        // OpenAMPolicies 生成
        OpenAMPolicies policies = new OpenAMPolicies();
        
        OpenAMPolicy policy = createPolicy1();
        
        OpenAMReferrals referrals = new OpenAMReferrals();
        referrals.setName("Referrals:1323689679750bRDOs40=");
        referrals.setDescription("");
        policy.setReferrals(referrals);
        
        // グループ作成
        List<String> groupList1 = new ArrayList<String>();
        groupList1.add("group1 (o=group1,p=123)");
        groupList1.add("group2 (o=group2,p=456)");
        List<String> groupList3 = new ArrayList<String>();
        groupList3.add("groupA (o=groupA,p=ABC)");
        
        RealmsBuilder builder = new RealmsBuilder();
        builder.addPolicies(policies);
        builder.addGroups("test4", groupList1);
        builder.addGroups("test9", groupList3);
        
        List<Realm> resultRealmList = builder.build();
        assertThat(resultRealmList.size(), is(0));
    }
    
    /**
     * 参照ポリシーでない →処理されない
     *
     *
     */
    @Test
    public void testBuild_NotReferralPolicy() {
        // OpenAMPolicies 生成
        OpenAMPolicies policies = new OpenAMPolicies();
        
        // 参照ポリシーでない →処理されない
        OpenAMPolicy policy = new OpenAMPolicy();
        policy.setName("testPolicy1234");
        policy.setReferralPolicy(false);
        policy.setActive(true);
        policies.getPolicy().add(policy);
        
        // レルムリスト作成
        List<String> realmList1 = new ArrayList<String>();
        realmList1.add("test4");
        
        // 作成処理
        RealmsBuilder builder = new RealmsBuilder();
        builder.addRealms(realmList1);
        builder.addPolicies(policies);

        List<Realm> resultRealmList = builder.build();

        // レルム1
        Realm resultRealm1 = resultRealmList.get(0);
        assertThat(resultRealm1.getReferrals().size(), is(0));
    }
    
    /**
     * Referrals 内に Referral がない
     *
     *
     */
    @Test
    public void testPolicy_ReferralNotContainsReferral() {
        // OpenAMPolicies 生成
        OpenAMPolicies policies = new OpenAMPolicies();
        
        // Referrals に Referral がない
        OpenAMPolicy policy = new OpenAMPolicy();
        policy.setName("refPolicy");
        policy.setReferralPolicy(true);
        policy.setActive(true);
        policies.getPolicy().add(policy);
        
        OpenAMRule rule = new OpenAMRule();
        rule.setName("http://name");
        policy.getRule().add(rule);
        
        OpenAMServiceName serviceName = new OpenAMServiceName();
        serviceName.setName(OpenAMNamingUtil.SERVICE_NAME);
        rule.setServiceName(serviceName);
        
        OpenAMResourceName resourceName = new OpenAMResourceName();
        resourceName.setName("nameresource888777");
        rule.setResourceName(resourceName);
        
        OpenAMReferrals referrals = new OpenAMReferrals();
        referrals.setName("Referrals:1323689679750bRDOs40=");
        referrals.setDescription("");
        policy.setReferrals(referrals);
        
        // レルムリスト作成
        List<String> realmList1 = new ArrayList<String>();
        realmList1.add("test4");
        
        // 作成処理
        RealmsBuilder builder = new RealmsBuilder();
        builder.addRealms(realmList1);
        builder.addPolicies(policies);

        List<Realm> resultRealmList = builder.build();

        // レルム数
        assertThat(resultRealmList.size(), is(1));

        // レルム1
        Realm resultRealm1 = resultRealmList.get(0);
        
        assertThat(resultRealm1.getReferrals().size(), is(0));
    }
   
    /**
     * Referral の type が OpenAMNamingUtil.SUBJECT_REFERRAL_TYPE = "SubOrgReferral" でない →処理されない
     *
     *
     */
    public void testBuild_TypeOfReferralInvalid() {
        // OpenAMPolicies 生成
        OpenAMPolicies policies = new OpenAMPolicies();
        
        OpenAMPolicy policy = new OpenAMPolicy();
        policy.setName("refPolicy");
        policy.setReferralPolicy(true);
        policy.setActive(true);
        policies.getPolicy().add(policy);
        
        OpenAMRule rule = new OpenAMRule();
        rule.setName("http://name");
        policy.getRule().add(rule);
        
        OpenAMServiceName serviceName = new OpenAMServiceName();
        serviceName.setName(OpenAMNamingUtil.SERVICE_NAME);
        rule.setServiceName(serviceName);
        
        OpenAMResourceName resourceName = new OpenAMResourceName();
        resourceName.setName("nameresource9909");
        rule.setResourceName(resourceName);
        
        OpenAMReferrals referrals = new OpenAMReferrals();
        referrals.setName("Referrals:1323689679750bRDOs40=");
        referrals.setDescription("");
        policy.setReferrals(referrals);
        
        OpenAMReferral referral = new OpenAMReferral();
        referral.setName("test4ref");
        referral.setType("AAA_SubOrgReferral");
        referrals.getReferral().add(referral);
        
        OpenAMAttributeValuePair attributeValuePair = new OpenAMAttributeValuePair();
        referral.getAttributeValuePair().add(attributeValuePair);
        
        OpenAMAttribute attribute = new OpenAMAttribute();
        attribute.setName(OpenAMNamingUtil.VALUES_ATTR_NAME);
        attributeValuePair.setAttribute(attribute);
        
        attributeValuePair.getValue().add("o=test4,ou=services,dc=opensso,dc=java,dc=net");
        
        // レルムリスト作成
        List<String> realmList1 = new ArrayList<String>();
        realmList1.add("test4");
        
        // 作成処理
        RealmsBuilder builder = new RealmsBuilder();
        builder.addRealms(realmList1);
        builder.addPolicies(policies);

        List<Realm> resultRealmList = builder.build();

        // レルム数
        assertThat(resultRealmList.size(), is(1));

        // レルム1
        Realm resultRealm1 = resultRealmList.get(0);
        
        assertThat(resultRealm1.getReferrals().size(), is(0));
    }
    
    /**
     * ポリシが現在レルムの参照ポリシであるが, <Rule> がない
     *
     *
     */
    public void testBuild_PolicyNotContainsRule() {
        // OpenAMPolicies 生成
        OpenAMPolicies policies = new OpenAMPolicies();
        
        OpenAMPolicy policy = new OpenAMPolicy();
        policy.setName("refPolicy");
        policy.setReferralPolicy(true);
        policy.setActive(true);
        policies.getPolicy().add(policy);
        
        OpenAMReferrals referrals = new OpenAMReferrals();
        referrals.setName("Referrals:1323689679750bRDOs40=");
        referrals.setDescription("");
        policy.setReferrals(referrals);
        
        OpenAMReferral referral = new OpenAMReferral();
        referral.setName("test4ref");
        referral.setType(OpenAMNamingUtil.SUBJECT_REFERRAL_TYPE);
        referrals.getReferral().add(referral);
            
        OpenAMAttributeValuePair attributeValuePair = new OpenAMAttributeValuePair();
        referral.getAttributeValuePair().add(attributeValuePair);
        
        OpenAMAttribute attribute = new OpenAMAttribute();
        attribute.setName(OpenAMNamingUtil.VALUES_ATTR_NAME);
        attributeValuePair.setAttribute(attribute);
        
        attributeValuePair.getValue().add("o=test4,ou=services,dc=opensso,dc=java,dc=net");
        
        // レルムリスト作成
        List<String> realmList1 = new ArrayList<String>();
        realmList1.add("test4");
        
        // 作成処理
        RealmsBuilder builder = new RealmsBuilder();
        builder.addRealms(realmList1);
        builder.addPolicies(policies);

        List<Realm> resultRealmList = builder.build();

        // レルム数
        assertThat(resultRealmList.size(), is(1));

        // レルム1
        Realm resultRealm1 = resultRealmList.get(0);
        
        assertThat(resultRealm1.getReferrals().size(), is(0));
    }
    
    /**
     * Referralsがない
     */
    @Test
    public void testBuild_NoReferrals() {
        // OpenAMPolicies 生成
        OpenAMPolicies policies = new OpenAMPolicies();
        
        // ポリシー作成
        OpenAMPolicy policy = new OpenAMPolicy();
        policy.setName("refPolicy");
        policy.setDescription("参照ポリシ");
        policy.setCreatedby("id=amadmin,ou=user,dc=opensso,dc=java,dc=net");
        policy.setLastmodifiedby("id=amadmin,ou=user,dc=opensso,dc=java,dc=net");
        policy.setCreationdate(1323689719859L);
        policy.setLastmodifieddate(1323689719859L);
        policy.setReferralPolicy(true);
        policy.setActive(true);
        
        // グループリスト作成
        List<String> groupList1 = new ArrayList<String>();
        groupList1.add("group1 (o=group1,p=123)");
        groupList1.add("group2 (o=group2,p=456)");
        List<String> groupList3 = new ArrayList<String>();
        groupList3.add("groupA (o=groupA,p=ABC)");
        
        // レルムリスト作成
        List<String> realmList1 = new ArrayList<String>();
        realmList1.add("test4");
        realmList1.add("test7");
        List<String> realmList2 = new ArrayList<String>();
        realmList2.add("test9");
        
        // 作成処理
        RealmsBuilder builder = new RealmsBuilder();
        builder.addRealms(realmList1);
        builder.addRealms(realmList2);
        builder.addPolicies(policies);
        builder.addGroups("test4", groupList1);
        builder.addGroups("test9", groupList3);

        List<Realm> resultRealmList = builder.build();

        // レルム数
        assertThat(resultRealmList.size(), is(3));
        assertThat(resultRealmList.get(0).getReferrals().size(), is(0));
        assertThat(resultRealmList.get(1).getReferrals().size(), is(0));
        assertThat(resultRealmList.get(2).getReferrals().size(), is(0));
    }

    /**
     * 通常パターンの1番目のポリシー作成
     *
     *
     */
    private OpenAMPolicy createPolicy1() {
        OpenAMPolicy policy = new OpenAMPolicy();
        policy.setName("refPolicy");
        policy.setDescription("参照ポリシ");
        policy.setCreatedby("id=amadmin,ou=user,dc=opensso,dc=java,dc=net");
        policy.setLastmodifiedby("id=amadmin,ou=user,dc=opensso,dc=java,dc=net");
        policy.setCreationdate(1323689719859L);
        policy.setLastmodifieddate(1323689719859L);
        policy.setReferralPolicy(true);
        policy.setActive(true);
        
        OpenAMRule rule = new OpenAMRule();
        rule.setName("http://name");
        policy.getRule().add(rule);
        
        OpenAMServiceName serviceName = new OpenAMServiceName();
        serviceName.setName(OpenAMNamingUtil.SERVICE_NAME);
        rule.setServiceName(serviceName);
        
        OpenAMResourceName resourceName = new OpenAMResourceName();
        resourceName.setName("nameresource222");
        rule.setResourceName(resourceName);
        
        OpenAMReferrals referrals = new OpenAMReferrals();
        referrals.setName("Referrals:1323689679750bRDOs40=");
        referrals.setDescription("");
        policy.setReferrals(referrals);
        
        OpenAMReferral referral = new OpenAMReferral();
        referral.setName("test4ref");
        referral.setType(OpenAMNamingUtil.SUBJECT_REFERRAL_TYPE);
        referrals.getReferral().add(referral);
        
        OpenAMAttributeValuePair attributeValuePair = new OpenAMAttributeValuePair();
        referral.getAttributeValuePair().add(attributeValuePair);
        
        OpenAMAttribute attribute = new OpenAMAttribute();
        attribute.setName(OpenAMNamingUtil.VALUES_ATTR_NAME);
        attributeValuePair.setAttribute(attribute);
        
        attributeValuePair.getValue().add("o=test4,ou=services,dc=opensso,dc=java,dc=net");
        
        return policy;
    }
    
    /**
     * 通常パターンの2番目のポリシー作成
     *
     *
     */
    private OpenAMPolicy createPolicy2() {
        OpenAMPolicy policy = new OpenAMPolicy();
        policy.setName("refPolicy2");
        policy.setDescription("参照ポリシ2");
        policy.setCreatedby("id=amadmin,ou=user,dc=opensso,dc=java,dc=net");
        policy.setLastmodifiedby("id=amadmin,ou=user,dc=opensso,dc=java,dc=net");
        policy.setCreationdate(1323689719859L);
        policy.setLastmodifieddate(1323689719859L);
        policy.setReferralPolicy(true);
        policy.setActive(true);
        
        OpenAMRule rule = new OpenAMRule();
        rule.setName("http://name123");
        policy.getRule().add(rule);
        
        OpenAMServiceName serviceName = new OpenAMServiceName();
        serviceName.setName(OpenAMNamingUtil.SERVICE_NAME);
        rule.setServiceName(serviceName);
        
        OpenAMResourceName resourceName = new OpenAMResourceName();
        resourceName.setName("nameresource333");
        rule.setResourceName(resourceName);
        
        OpenAMReferrals referrals = new OpenAMReferrals();
        referrals.setName("Referrals:1323689679750bRDOs40=");
        referrals.setDescription("");
        policy.setReferrals(referrals);
        
        OpenAMReferral referral = new OpenAMReferral();
        referral.setName("test4ref222");
        referral.setType(OpenAMNamingUtil.SUBJECT_REFERRAL_TYPE);
        referrals.getReferral().add(referral);
        
        OpenAMAttributeValuePair attributeValuePair = new OpenAMAttributeValuePair();
        referral.getAttributeValuePair().add(attributeValuePair);
        
        OpenAMAttribute attribute = new OpenAMAttribute();
        attribute.setName(OpenAMNamingUtil.VALUES_ATTR_NAME);
        attributeValuePair.setAttribute(attribute);
        
        attributeValuePair.getValue().add("o=test4,ou=services,dc=opensso,dc=java,dc=net");
        
        return policy;
    }
    
    /**
     * 通常パターンの3番目のポリシー作成
     *
     *
     */
    private OpenAMPolicy createPolicy3() {
        OpenAMPolicy policy = new OpenAMPolicy();
        policy.setName("refPolicy");
        policy.setDescription("参照ポリシ");
        policy.setCreatedby("id=amadmin,ou=user,dc=opensso,dc=java,dc=net");
        policy.setLastmodifiedby("id=amadmin,ou=user,dc=opensso,dc=java,dc=net");
        policy.setCreationdate(1323689719859L);
        policy.setLastmodifieddate(1323689719859L);
        policy.setReferralPolicy(true);
        policy.setActive(true);
        
        OpenAMRule rule = new OpenAMRule();
        rule.setName("http://name");
        policy.getRule().add(rule);
        
        OpenAMServiceName serviceName = new OpenAMServiceName();
        serviceName.setName(OpenAMNamingUtil.SERVICE_NAME);
        rule.setServiceName(serviceName);
        
        OpenAMResourceName resourceName = new OpenAMResourceName();
        resourceName.setName("nameresource000");
        rule.setResourceName(resourceName);
        
        OpenAMReferrals referrals = new OpenAMReferrals();
        referrals.setName("Referrals:1323689679750bRDOs40=");
        referrals.setDescription("");
        policy.setReferrals(referrals);
        
        OpenAMReferral referral = new OpenAMReferral();
        referral.setName("test7ref");
        referral.setType(OpenAMNamingUtil.SUBJECT_REFERRAL_TYPE);
        referrals.getReferral().add(referral);
        
        OpenAMAttributeValuePair attributeValuePair = new OpenAMAttributeValuePair();
        referral.getAttributeValuePair().add(attributeValuePair);
        
        OpenAMAttribute attribute = new OpenAMAttribute();
        attribute.setName(OpenAMNamingUtil.VALUES_ATTR_NAME);
        attributeValuePair.setAttribute(attribute);
        
        attributeValuePair.getValue().add("o=test7,ou=services,dc=opensso,dc=java,dc=net");
        
        return policy;
    }
}
