/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.policytool.common.util;

import static org.hamcrest.CoreMatchers.is;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.fail;

import java.io.UnsupportedEncodingException;
import java.lang.reflect.InvocationTargetException;
import java.util.HashSet;

import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;

import jp.co.ogis_ri.citk.policytool.common.exception.ApplicationException;
import jp.co.ogis_ri.citk.policytool.common.exception.ApplicationSettingException;
import jp.co.ogis_ri.citk.policytool.common.exception.SystemException;
import jp.co.ogis_ri.citk.policytool.common.resource.MessageInfo;
import mockit.Expectations;
import mockit.Mocked;

import org.apache.http.HttpHost;
import org.apache.http.conn.HttpHostConnectException;
import org.junit.Test;

public class ExceptionUtilTest {
	@Mocked
	private MessageInfo messageInfo;

	@Test
	public void testConvertRuntimeException_HttpHostConnectException() {
		final HttpHost httpHost = new HttpHost("hname");
		final HttpHostConnectException httpHostConnectException = new HttpHostConnectException(
				httpHost, null);

		new Expectations() {
			{
				new MessageInfo("E-0001", httpHost);
			}
		};

		ApplicationSettingException resultException = (ApplicationSettingException) ExceptionUtil
				.convertRuntimeException(httpHostConnectException);
		assertThat((HttpHostConnectException) resultException.getCause(),
				is(httpHostConnectException));
		assertThat(resultException.getMessageInfo().toString(),
				is(messageInfo.toString()));
	}

	@Test
	public void testConvertRuntimeException_UnsupportedEncodingException() {
		final UnsupportedEncodingException unsupportedEncodingException = new UnsupportedEncodingException(
				"msg1");

		new Expectations() {
			{
				new MessageInfo("E-0003", "msg1");
			}
		};

		ApplicationSettingException resultException = (ApplicationSettingException) ExceptionUtil
				.convertRuntimeException(unsupportedEncodingException);
		assertThat((UnsupportedEncodingException) resultException.getCause(),
				is(unsupportedEncodingException));
		assertThat(resultException.getMessageInfo().toString(),
				is(messageInfo.toString()));
	}

	@Test
	public void testConvertRuntimeException_ConstraintViolationException() {
		final ConstraintViolationException constraintViolationException = new ConstraintViolationException(
				"msg1", new HashSet<ConstraintViolation<?>>());

		new Expectations() {
			{
				new MessageInfo("E-0005", "msg1");
			}
		};

		ApplicationException resultException = (ApplicationException) ExceptionUtil
				.convertRuntimeException(constraintViolationException);
		assertThat((ConstraintViolationException) resultException.getCause(),
				is(constraintViolationException));
		assertThat(resultException.getMessageInfo().toString(),
				is(messageInfo.toString()));
	}

	@Test
	public void testConvertRuntimeException_Error() {
		final Error error = new TestError();

		try {
			ExceptionUtil.convertRuntimeException(error);
			fail();
		} catch (Error e) {
			assertThat(e, is(error));
		}
	}

	@Test
	public void testConvertRuntimeException_RuntimeException() {
		final RuntimeException exception = new TestRuntimeException();

		RuntimeException result = ExceptionUtil
				.convertRuntimeException(exception);
		assertThat(result, is(exception));
	}

	@Test
	public void testConvertRuntimeException_InvocationTargetException() {
		final HttpHost httpHost = new HttpHost("hname");
		final HttpHostConnectException httpHostConnectException = new HttpHostConnectException(
				httpHost, null);
		final InvocationTargetException invocationTargetException = new InvocationTargetException(
				httpHostConnectException);

		new Expectations() {
			{
				new MessageInfo("E-0001", httpHost);
			}
		};

		ApplicationSettingException resultException = (ApplicationSettingException) ExceptionUtil
				.convertRuntimeException(invocationTargetException);
		assertThat((HttpHostConnectException) resultException.getCause(),
				is(httpHostConnectException));
		assertThat(resultException.getMessageInfo().toString(),
				is(messageInfo.toString()));
	}

	@Test
	public void testConvertRuntimeException_Others() {
		Exception exception = new Exception();

		SystemException resultException = (SystemException) ExceptionUtil
				.convertRuntimeException(exception);
		assertThat((Exception) resultException.getCause(), is(exception));
	}

	@SuppressWarnings("serial")
	private static class TestError extends Error {
		public TestError() {
			super();
		}
	}

	@SuppressWarnings("serial")
	private static class TestRuntimeException extends RuntimeException {
		public TestRuntimeException() {
			super();
		}
	}

}
