/*
 * brownies and its relative products are published under the terms
 * of the Apache Software License.
 * 
 * Created on 2004/08/10 21:03:17
 */
package test.org.asyrinx.brownie.core.xml.digester;

import java.io.ByteArrayInputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import junit.framework.TestCase;

import org.apache.commons.digester.Rule;
import org.asyrinx.brownie.core.xml.digester.AsyrinxDigester;
import org.asyrinx.brownie.core.xml.digester.DigesterException;
import org.xml.sax.Attributes;

/**
 * @author akima
 */
public class AsyrinxDigesterTest extends TestCase {

    public static void main(String[] args) {
        junit.swingui.TestRunner.run(AsyrinxDigesterTest.class);
    }

    /*
     * Class under test for void addRoot(String, String, Class)
     */
    public void testAddRootStringStringClass() {
        final AsyrinxDigester digester = new AsyrinxDigester();
        digester.addRoot("a", A.class);
        digester.addSetProperties("a");
        //
        final StringBuffer buf = new StringBuffer();
        buf.append("<a value=\"aaa\">");
        buf.append("</a>");
        Object result;
        try {
            result = digester.parse(new ByteArrayInputStream(buf.toString().getBytes()));
        } catch (Exception e) {
            e.printStackTrace();
            fail();
            return;
        }
        assertEquals(A.class, result.getClass());
        final A a = (A) result;
        assertEquals("aaa", a.getValue());
    }

    /*
     * Class under test for void addProp(String, Class, String, boolean)
     */
    public void testAddPropStringClassStringboolean() {
        final AsyrinxDigester digester = new AsyrinxDigester();
        digester.addRoot("b", B.class);
        digester.addSetProperties("b");
        digester.addProp("b/a", A.class, "refA", true);
        //
        final StringBuffer buf = new StringBuffer();
        buf.append("<b value=\"bbb\">");
        buf.append("<a value=\"aaa\"/>");
        buf.append("</b>");
        Object result;
        try {
            result = digester.parse(new ByteArrayInputStream(buf.toString().getBytes()));
        } catch (Exception e) {
            e.printStackTrace();
            fail();
            return;
        }
        assertEquals(B.class, result.getClass());
        final B b = (B) result;
        assertEquals("bbb", b.getValue());
        final A a = b.getRefA();
        assertEquals("aaa", a.getValue());
    }

    /*
     * Class under test for void addProps(String, Class, String)
     */
    public void testAddPropsStringClassString() {
        final AsyrinxDigester digester = new AsyrinxDigester();
        digester.addRoot("c", C.class);
        digester.addSetProperties("c");
        digester.addProps("c/a", C.class, "addValue");
        //
        final StringBuffer buf = new StringBuffer();
        buf.append("<c value=\"ccc\">");
        buf.append("<a value=\"a0\"/>");
        buf.append("<a value=\"a1\"/>");
        buf.append("<a value=\"a2\"/>");
        buf.append("</c>");
        Object result;
        try {
            result = digester.parse(new ByteArrayInputStream(buf.toString().getBytes()));
        } catch (Exception e) {
            e.printStackTrace();
            fail();
            return;
        }
        assertEquals(C.class, result.getClass());
        final C c = (C) result;
        assertEquals("ccc", c.getValue());
        assertEquals(3, c.size());
        final A a0 = c.get(0);
        final A a1 = c.get(1);
        final A a2 = c.get(2);
        assertEquals("a0", a0.getValue());
        assertEquals("a1", a1.getValue());
        assertEquals("a2", a2.getValue());
    }

    /*
     * Class under test for void addProps(String, Class, String)
     */
    public void testAddBodyToProp() {
        final AsyrinxDigester digester = new AsyrinxDigester();
        digester.addRoot("a", A.class);
        digester.addSetProperties("a");
        digester.addBodyToProp("a/comment", "value");
        //
        final StringBuffer buf = new StringBuffer();
        buf.append("<a>");
        buf.append("<comment>");
        buf.append("abc def ghi");
        buf.append("</comment>");
        buf.append("</a>");
        Object result;
        try {
            result = digester.parse(new ByteArrayInputStream(buf.toString().getBytes()));
        } catch (Exception e) {
            e.printStackTrace();
            fail();
            return;
        }
        assertEquals(A.class, result.getClass());
        final A a = (A) result;
        assertEquals("abc def ghi", a.getValue());
    }

    /*
     * Class under test for void addCallMethod(String, String, Class[],
     * String[])
     */
    public void testAddCallMethodStringStringClassArrayStringArray() {
        final AsyrinxDigester digester = new AsyrinxDigester();
        digester.addRoot("c", C.class);
        digester.addSetProperties("c");
        digester.addProps("c/a", C.class, "addValue");
        digester.addCallMethod("c/a", "appendValue", new Class[] { String.class }, new String[] { "suffix" });
        //
        final StringBuffer buf = new StringBuffer();
        buf.append("<c value=\"ccc\">");
        buf.append("<a value=\"a0\" suffix=\"_s0\"/>");
        buf.append("<a value=\"a1\" suffix=\"_s1\"/>");
        buf.append("<a value=\"a2\" suffix=\"_s2\"/>");
        buf.append("</c>");
        Object result;
        try {
            result = digester.parse(new ByteArrayInputStream(buf.toString().getBytes()));
        } catch (Exception e) {
            e.printStackTrace();
            fail();
            return;
        }
        assertEquals(C.class, result.getClass());
        final C c = (C) result;
        assertEquals("ccc", c.getValue());
        assertEquals(3, c.size());
        final A a0 = c.get(0);
        final A a1 = c.get(1);
        final A a2 = c.get(2);
        assertEquals("a0_s0", a0.getValue());
        assertEquals("a1_s1", a1.getValue());
        assertEquals("a2_s2", a2.getValue());
    }

    public void testAddPushProperty() {
        final AsyrinxDigester digester = new AsyrinxDigester();
        digester.addRoot("d", D.class);
        digester.addSetProperties("d");
        //ɂȂpropertyݒ肳Ă܂B
        digester.addProp("d/e", E.class, "object", false);
        //vpeBɂȂĂIuWFNgvbV
        digester.addPushProperty("d/e", "composited");
        digester.addSetProperties("d/e");
        //vbVIuWFNg|bvȂƁÂƂȂ܂B
        digester.addPop("d/e");
        //
        digester.addProp("d/a", A.class, "a", true);
        //
        final StringBuffer buf = new StringBuffer();
        buf.append("<d value=\"ddd\">");
        buf.append("<e value=\"888\"/>");
        buf.append("<a value=\"aaa\"/>");
        buf.append("</d>");
        Object result;
        try {
            result = digester.parse(new ByteArrayInputStream(buf.toString().getBytes()));
        } catch (Exception e) {
            e.printStackTrace();
            fail();
            return;
        }
        assertEquals(D.class, result.getClass());
        final D d = (D) result;
        assertEquals("ddd", d.getValue());
        final E e = d.getObject();
        assertNull(e.getValue());
        assertNotNull(e);
        final A composited = e.getComposited();
        assertNotNull(composited);
        assertEquals("888", composited.getValue());
        //
        assertNotNull(d.getA());
        assertEquals("aaa", d.getA().getValue());
    }

    public void testAddConstructor() {
        final AsyrinxDigester digester = new AsyrinxDigester();
        digester.addRoot("g", G.class);
        digester.addSetProperties("g");
        digester.addProp("g/f", F.class, new Class[] { String.class }, new String[] { "name" }, "f");
        //
        final StringBuffer buf = new StringBuffer();
        buf.append("<g>");
        buf.append("<f name=\"constructor param\" value=\"property\"/>");
        buf.append("</g>");
        Object result;
        try {
            result = digester.parse(new ByteArrayInputStream(buf.toString().getBytes()));
        } catch (Exception e) {
            e.printStackTrace();
            fail();
            return;
        }
        assertEquals(G.class, result.getClass());
        final G g = (G) result;
        final F f = g.getF();
        assertEquals("constructor param", f.getName());
        assertEquals("property", f.getValue());
    }

    public void testAddSetToMap() {
        final AsyrinxDigester digester = new AsyrinxDigester();
        digester.addRoot("h", H.class);
        digester.addSetProperties("h");
        digester.addObjectCreate("h/map", HashMap.class);
        digester.addAsProp("h/map", "options");
        digester.addSetToMap("h/map");
        //
        final StringBuffer buf = new StringBuffer();
        buf.append("<h>");
        buf.append("<map name=\"val1\" value=\"val2\"/>");
        buf.append("</h>");
        Object result;
        try {
            result = digester.parse(new ByteArrayInputStream(buf.toString().getBytes()));
        } catch (Exception e) {
            e.printStackTrace();
            fail();
            return;
        }
        assertEquals(H.class, result.getClass());
        final H h = (H) result;
        assertEquals(2, h.getOptions().keySet().size());
        assertEquals("val1", h.getOptions().get("name"));
        assertEquals("val2", h.getOptions().get("value"));
    }

    public static class A {
        public A() {
            super();
        }

        private String value = null;

        public String getValue() {
            return value;
        }

        public void setValue(String value) {
            this.value = value;
        }

        public void appendValue(String s) {
            value = value + s;
        }
    }

    public static class B {
        private String value = null;

        public String getValue() {
            return value;
        }

        public void setValue(String value) {
            this.value = value;
        }

        private A refA = null;

        public A getRefA() {
            return refA;
        }

        /**
         * @param refA
         *            The refA to set.
         */
        public void setRefA(A refA) {
            this.refA = refA;
        }
    }

    public static class C extends A {
        final List values = new ArrayList();

        public void addValue(A a) {
            values.add(a);
        }

        public A get(int i) {
            return (A) values.get(i);
        }

        public int size() {
            return values.size();
        }
    }

    public static class D {
        private String value = null;

        public String getValue() {
            return value;
        }

        public void setValue(String value) {
            this.value = value;
        }

        private E object = null;

        /**
         * @return Returns the object.
         */
        public E getObject() {
            return object;
        }

        /**
         * @param object
         *            The object to set.
         */
        public void setObject(E object) {
            this.object = object;
        }

        private A a = null;

        /**
         * @return Returns the a.
         */
        public A getA() {
            return a;
        }

        /**
         * @param a
         *            The a to set.
         */
        public void setA(A a) {
            this.a = a;
        }
    }

    public static class E {
        private final A composited = new A();

        /**
         * @return Returns the composited.
         */
        public A getComposited() {
            return composited;
        }

        private String value = null;

        public String getValue() {
            return value;
        }

        public void setValue(String value) {
            this.value = value;
        }

    }

    public static class F {
        public F(String name) {
            this.name = name;
        }

        private final String name;

        private String value = null;

        /**
         * @return Returns the name.
         */
        public String getName() {
            return name;
        }

        /**
         * @return Returns the value.
         */
        public String getValue() {
            return value;
        }

        /**
         * @param value
         *            The value to set.
         */
        public void setValue(String value) {
            this.value = value;
        }
    }

    public static class G {
        private F f = null;

        /**
         * @return Returns the f.
         */
        public F getF() {
            return f;
        }

        /**
         * @param f
         *            The f to set.
         */
        public void setF(F f) {
            this.f = f;
        }
    }

    public static class H {
        private Map options = null;

        /**
         * @return Returns the options.
         */
        public Map getOptions() {
            return options;
        }

        /**
         * @param options
         *            The options to set.
         */
        public void setOptions(Map options) {
            this.options = options;
        }
    }

    public static class ObjectWithOption extends A {
        public ObjectWithOption() {
            super();
        }

        private Map options = new HashMap();

        /**
         * @return Returns the options.
         */
        public Map getOptions() {
            return options;
        }
    }

    public void testWithOption() {
        final AsyrinxDigester digester = new AsyrinxDigester();
        digester.addRoot("obj", ObjectWithOption.class);
        digester.addSetProperties("obj");
        //
        final StringBuffer buf = new StringBuffer();
        buf.append("<obj value=\"value1\" option1=\"opt1\" option2=\"opt2\">");
        buf.append("</obj>");
        Object result;
        try {
            result = digester.parse(new ByteArrayInputStream(buf.toString().getBytes()));
        } catch (Exception e) {
            e.printStackTrace();
            fail();
            return;
        }
        assertEquals(ObjectWithOption.class, result.getClass());
        final ObjectWithOption obj = (ObjectWithOption) result;
        assertEquals("value1", obj.getValue());
        assertEquals("opt1", obj.getOptions().get("option1"));
        assertEquals("opt2", obj.getOptions().get("option2"));
    }

    public void testWithOption2() {
        final AsyrinxDigester digester = new AsyrinxDigester();
        digester.addRoot("c", C.class);
        digester.addSetProperties("c");
        digester.addProps("c/obj", ObjectWithOption.class, "addValue");
        //
        final StringBuffer buf = new StringBuffer();
        buf.append("<c value=\"cccc\">");
        buf.append("<obj value=\"value1\" option1=\"opt1\" option2=\"opt2\" />");
        buf.append("</c>");
        Object result;
        try {
            result = digester.parse(new ByteArrayInputStream(buf.toString().getBytes()));
        } catch (Exception e) {
            e.printStackTrace();
            fail();
            return;
        }
        assertEquals(C.class, result.getClass());
        final C c = (C) result;
        assertEquals("cccc", c.getValue());
        assertEquals(1, c.values.size());
        assertEquals(ObjectWithOption.class, c.values.get(0).getClass());
        final ObjectWithOption obj = (ObjectWithOption) c.values.get(0);
        assertEquals("value1", obj.getValue());
        assertEquals("opt1", obj.getOptions().get("option1"));
        assertEquals("opt2", obj.getOptions().get("option2"));
    }

    public void testDebug() {
        final AsyrinxDigester digester = new AsyrinxDigester();
        digester.addRoot("c", C.class);
        digester.addSetProperties("c");
        digester.addProps("c/obj", ObjectWithOption.class, "addValue");
        //
        final DigesterException testException = new DigesterException("test exception", null);
        digester.addRule("c/obj", new Rule() {
            public void begin(String namespace, String name, Attributes attributes) throws Exception {
                throw testException;
            }
        });
        //
        final StringBuffer buf = new StringBuffer();
        buf.append("<c value=\"cccc\">");
        buf.append("<obj value=\"value1\" option1=\"opt1\" option2=\"opt2\" />");
        buf.append("</c>");
        try {
            digester.parse(new ByteArrayInputStream(buf.toString().getBytes()));
            fail();
        } catch (DigesterException e) {
            //e.printStackTrace();
            assertEquals(testException, e.getCause());
            return;
        } catch (AssertionError e) {
            //thru
        } catch (Exception e) {
            e.printStackTrace();
            fail();
            return;
        }
    }
}