/*
 * brownies and its relative products are published under the terms
 * of the Apache Software License.
 * 
 * Created on 2004/11/25 22:08:15
 */
package test.org.asyrinx.brownie.core.xml.dom;

import java.io.ByteArrayOutputStream;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import junit.framework.TestCase;

import org.asyrinx.brownie.core.lang.StringUtils;
import org.asyrinx.brownie.core.util.jp.JpDateFormat;
import org.asyrinx.brownie.core.xml.dom.DomBuilder;
import org.asyrinx.brownie.core.xml.dom.impl.DomBuilderNode;
import org.asyrinx.brownie.core.xml.dom.impl.XmlBeanWriterImpl;
import org.asyrinx.brownie.core.xml.dom.RenderingAttribute;
import org.asyrinx.brownie.core.xml.dom.RenderingDuplicated;
import org.asyrinx.brownie.core.xml.dom.XmlBeanWriter;

/**
 * @author takeshi
 */
public class XmlBeanWriterTest extends TestCase {

    public static void main(String[] args) {
        junit.swingui.TestRunner.run(XmlBeanWriterTest.class);
    }

    public static class A {
        public A(String valueStr, int valueInt, Date valueDate, A valueA) {
            super();
            this.valueA = valueA;
            this.valueDate = valueDate;
            this.valueInt = valueInt;
            this.valueStr = valueStr;
        }

        private String valueStr = null;

        private int valueInt = 987;

        private Date valueDate = new Date();

        private A valueA = null;

        public A getValueA() {
            return valueA;
        }

        public void setValueA(A valueA) {
            this.valueA = valueA;
        }

        public Date getValueDate() {
            return valueDate;
        }

        public void setValueDate(Date valueDate) {
            this.valueDate = valueDate;
        }

        public int getValueInt() {
            return valueInt;
        }

        public void setValueInt(int valueInt) {
            this.valueInt = valueInt;
        }

        public String getValueStr() {
            return valueStr;
        }

        public void setValueStr(String valueStr) {
            this.valueStr = valueStr;
        }
    }

    final DateFormat df = new SimpleDateFormat("yyyy/MM/dd HH:mm");

    private Date toDate(String value) {
        try {
            return df.parse(value);
        } catch (ParseException e) {
            return null;
        }
    }

    private void checkXml(String expected, Object bean, String root, XmlBeanWriter writer) {
        final ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        try {
            writer.execute(bean, outputStream, root);
        } catch (Exception e) {
            e.printStackTrace();
            fail();
            return;
        }
        final String result = outputStream.toString();
        //System.out.println(StringEscapeUtils.escapeJava(result));
        assertEquals(expected, StringUtils.replace(StringUtils.replace(result, "\n", ""), "\r", ""));
    }

    private static final String BASE_CLASS = "test.org.asyrinx.brownie.core.xml.dom.XmlBeanWriterTest";

    public void testNode1() {
        final A a1 = new A("eXgIuWFNgA1", 123, null, null);
        final A a2 = new A("eXgIuWFNgA2", -77, toDate("2004/11/25 22:19"), a1);
        final A a3 = new A("eXgIuWFNgA3", -32, toDate("2004/11/12 11:19"), a2);
        //
        checkXml("<?xml version=\"1.0\" encoding=\"Shift_JIS\"?>" //
                + "<a>" //
                + "<valueDate>2004/11/12 11:19:00</valueDate>" // 
                + "<valueStr>eXgIuWFNgA3</valueStr>" //
                + "<valueInt>-32</valueInt>" //
                + "<valueA>" //
                + "" + "<valueDate>2004/11/25 22:19:00</valueDate>" // 
                + "" + "<valueStr>eXgIuWFNgA2</valueStr>" //
                + "" + "<valueInt>-77</valueInt>" //
                + "" + "<valueA>" //
                //+ "" + "" + "<valueDate>null</valueDate>" //
                + "" + "" + "<valueStr>eXgIuWFNgA1</valueStr>" //
                + "" + "" + "<valueInt>123</valueInt>" //
                //+ "" + "" + "<valueA>null</valueA>" //
                + "" + "</valueA>" //
                + "</valueA>" //
                + "</a>" //,
        , a3, "a", new XmlBeanWriterImpl(new DomBuilderNode()));
        //
        final DomBuilder domBuilder = new DomBuilderNode();
        domBuilder.setRenderingAttribute(new RenderingAttribute.Composite(new RenderingAttribute[] {
                RenderingAttribute.CLASS, RenderingAttribute.HASH_CODE }));
        checkXml("<?xml version=\"1.0\" encoding=\"Shift_JIS\"?>" //
                + "<a class=\"" + BASE_CLASS + "$A\" hashCode=\"" + a3.hashCode() + "\">" //
                + "<valueDate>2004/11/12 11:19:00</valueDate>" // 
                + "<valueStr>eXgIuWFNgA3</valueStr>" //
                + "<valueInt>-32</valueInt>" //
                + "<valueA class=\"" + BASE_CLASS + "$A\" hashCode=\"" + a2.hashCode() + "\">" //
                + "" + "<valueDate>2004/11/25 22:19:00</valueDate>" // 
                + "" + "<valueStr>eXgIuWFNgA2</valueStr>" //
                + "" + "<valueInt>-77</valueInt>" //
                + "" + "<valueA class=\"" + BASE_CLASS + "$A\" hashCode=\"" + a1.hashCode() + "\">" //
                //+ "" + "" + "<valueDate>null</valueDate>" //
                + "" + "" + "<valueStr>eXgIuWFNgA1</valueStr>" //
                + "" + "" + "<valueInt>123</valueInt>" //
                //+ "" + "" + "<valueA>null</valueA>" //
                + "" + "</valueA>" //
                + "</valueA>" //
                + "</a>" //,
        , a3, "a", new XmlBeanWriterImpl(domBuilder));
    }

    public class B extends A {
        public B(String valueStr, int valueInt, Date valueDate, A valueA) {
            super(valueStr, valueInt, valueDate, valueA);
        }

        private final List beans = new ArrayList();

        public List getBeans() {
            return beans;
        }
    }

    public void testNode2() {
        final A a1 = new A("eXgIuWFNgA1", 123, null, null);
        final A a2 = new A("eXgIuWFNgA2", -77, toDate("2004/11/25 22:19"), a1);
        final A a3 = new A("eXgIuWFNgA3", -32, toDate("2004/11/12 11:19"), a2);
        final B b1 = new B("eXgIuWFNgB1", 999, toDate("2003/10/10 10:10"), null);
        b1.getBeans().add(a1);
        b1.getBeans().add(a3);
        //
        checkXml("<?xml version=\"1.0\" encoding=\"Shift_JIS\"?>" //
                + "<b>" //
                + "<valueDate>2003/10/10 10:10:00</valueDate>" // 
                + "<valueStr>eXgIuWFNgB1</valueStr>" //
                + "<valueInt>999</valueInt>" //
                //+ "<valueA>null</valueA>" //
                + "<beans>" //
                + "" + "<valueStr>eXgIuWFNgA1</valueStr>" //
                + "" + "<valueInt>123</valueInt>" //
                + "</beans>" //
                + "<beans>" //
                + "" + "<valueDate>2004/11/12 11:19:00</valueDate>" // 
                + "" + "<valueStr>eXgIuWFNgA3</valueStr>" //
                + "" + "<valueInt>-32</valueInt>" //
                + "" + "<valueA>" //
                + "" + "" + "<valueDate>2004/11/25 22:19:00</valueDate>" // 
                + "" + "" + "<valueStr>eXgIuWFNgA2</valueStr>" //
                + "" + "" + "<valueInt>-77</valueInt>" //
                + "" + "" + "<valueA>" //
                //+ "" + "" + "" + "<valueDate>null</valueDate>" //
                + "" + "" + "" + "<valueStr>eXgIuWFNgA1</valueStr>" //
                + "" + "" + "" + "<valueInt>123</valueInt>" //
                //+ "" + "" + "" + "<valueA>null</valueA>" //
                + "" + "" + "</valueA>" //
                + "" + "</valueA>" //
                + "</beans>" //
                + "</b>" //
        , b1, "b", new XmlBeanWriterImpl(new DomBuilderNode(RenderingDuplicated.NULL, RenderingAttribute.NULL)));
    }

    public void testCircularReference() {
        final A a1 = new A("eXgIuWFNgA1", 123, null, null);
        final A a2 = new A("eXgIuWFNgA2", -77, toDate("2004/11/25 22:19"), a1);
        final A a3 = new A("eXgIuWFNgA3", -32, toDate("2004/11/12 11:19"), a2);
        a1.setValueA(a3);
        //
        try {
            checkXml("<?xml version=\"1.0\" encoding=\"Shift_JIS\"?>" //
                    + "<a hashCode=\"" + a3.hashCode() + "\">" //
                    + "<valueDate>2004/11/12 11:19:00</valueDate>" // 
                    + "<valueStr>eXgIuWFNgA3</valueStr>" //
                    + "<valueInt>-32</valueInt>" //
                    + "<valueA hashCode=\"" + a2.hashCode() + "\">" //
                    + "" + "<valueDate>2004/11/25 22:19:00</valueDate>" // 
                    + "" + "<valueStr>eXgIuWFNgA2</valueStr>" //
                    + "" + "<valueInt>-77</valueInt>" //
                    + "" + "<valueA hashCode=\"" + a1.hashCode() + "\">" //
                    //+ "" + "" + "<valueDate>null</valueDate>" //
                    + "" + "" + "<valueStr>eXgIuWFNgA1</valueStr>" //
                    + "" + "" + "<valueInt>123</valueInt>" //
                    + "" + "" + "<valueA hashCode=\"" + a3.hashCode() + "\"/>" //
                    + "" + "</valueA>" //
                    + "</valueA>" //
                    + "</a>" //,
            , a3, "a", new XmlBeanWriterImpl(new DomBuilderNode(RenderingDuplicated.TAKE_FIRST,
                    RenderingAttribute.HASH_CODE)));
        } catch (StackOverflowError e) {
            e.printStackTrace();
            fail();
            return;
        }
    }

    public void testDateFormat() {
        final A a1 = new A("eXgIuWFNgA1", 123, toDate("2004/12/01 16:57"), null);
        final DomBuilder domBuilder = new DomBuilderNode(RenderingDuplicated.TAKE_FIRST, null);
        domBuilder.setDateFormat(new JpDateFormat("jEjyyNMMdd HHmm"));
        checkXml("<?xml version=\"1.0\" encoding=\"Shift_JIS\"?>" //
                + "<a>" //,
                + "<valueDate>16N1201 1657</valueDate>" //
                + "<valueStr>eXgIuWFNgA1</valueStr>" //
                + "<valueInt>123</valueInt>" //
                + "</a>" //,
        , a1, "a", new XmlBeanWriterImpl(domBuilder));
        //
        domBuilder.setDateFormat(new JpDateFormat("jEEjyyNMMdd HHmm"));
        checkXml("<?xml version=\"1.0\" encoding=\"Shift_JIS\"?>" //
                + "<a>" //,
                + "<valueDate>16N1201 1657</valueDate>" //
                + "<valueStr>eXgIuWFNgA1</valueStr>" //
                + "<valueInt>123</valueInt>" //
                + "</a>" //,
        , a1, "a", new XmlBeanWriterImpl(domBuilder));
    }
}