package map;
import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Polygon;
import java.awt.RenderingHints;
import java.awt.geom.GeneralPath;
import java.util.Collection;
import java.util.Map;

import javax.swing.JPanel;

import map.data.Curve;
import map.data.MapData;
import map.data.MapMap;
import map.data.Mesh;
import map.data.Node;
import map.data.Road;
import map.labeling.Labeling;
import map.labeling.SimpleLabeling;
import map.route.ShortestPathAlgorithm;
import map.route.DirectDistance;
import map.route.ShortestPathSearch;
import map.route.SearchThread;

/**
 * Graph Class 2005/09/23
 * 
 * @author Masayasu FUJIWARA
 */
public class MapPanel extends JPanel {

	/**
	 * 最短経路探索アルゴリズム
	 */
	private ShortestPathAlgorithm algorithm;

	private int centerX;

	private int centerY;
	
	/**
	 * 市町村まで表示する倍率
	 */
	private final float CITY_SCALE = 0.0002f;

	private double cos;

	/**
	 * 国土数値情報と数値地図25000を切り替える倍率
	 */
	private final float DATA25K_SCALE = 0.0015f;

	private int drawlevel;

	/**
	 * アンチエイリアスのフラグ
	 */
	private boolean isAntialias;

	/**
	 * 水域区間の塗りつぶしフラグ
	 */
	private boolean isFill;

	/**
	 * 操作マニュアルの表示フラグ
	 */
	private boolean isHelp;

	/**
	 * ラベル表示のフラグ
	 */
	private boolean isLabel;

	/**
	 * 頂点表示のフラグ
	 */
	private boolean isNodeView;

	/**
	 * マウス操作のフラグ
	 */
	private boolean isOperation;
	
	/**
	 * 地図情報管理マップ
	 */
	private MapMap maps;

	/**
	 * 頂点のサイズ
	 */
	private int nodeSize = 3;

	/**
	 * オフスクリーンイメージ
	 */
	private Image offs;

	/**
	 * 都道府県ポリゴン
	 */
	private Polygon[] prefecture;

	/**
	 * 回転量
	 */
	private final float ROTATE_SENSE = 0.02f;

	/**
	 * 一般道のルート色
	 */
	private final Color ROUTE_COLOR_GENERAL = Color.YELLOW;

	/**
	 * 高速道路のルート色
	 */
	private final Color ROUTE_COLOR_HIGHWAY = Color.RED;

	/**
	 * 地図の表示倍率
	 */
	private float scale = 0.005f;

	/**
	 * 表示倍率の上限
	 */
	private float SCALE_MAX = 0.1f;
	
	/**
	 * 表示倍率の下限
	 */
	private float SCALE_MIN = 0.000005f;

	/**
	 * 表示倍率の変更精度
	 */
	private float SCALE_SENSE = 0.08f;

	private Polygon screen;
	
	/**
	 * sin の値
	 */
	private double sin;

	/**
	 * 探索の始点
	 */
	private Node start;

	/**
	 * 道路区間，鉄道区間の描画の基本の幅
	 */
	private final int STROKE_WIDTH = 130;
	
	/**
	 * 探索の終点
	 */
	private Node terminal;
	
	/**
	 * XY 方向の回転角度(rad);
	 */
	private double theta;

	/**
	 * 探索を行うスレッド
	 */
	private SearchThread thread;
	private int shiftX;
	private int shiftY;

	/**
	 * 探索端点の変更
	 */
	public synchronized void changeBoundary(int ex, int ey, boolean flag) {

		ex -= this.centerX;
		ey -= this.centerY;
		int x = (int)((this.centerX + ex * this.cos - ey * this.sin) / this.scale + this.shiftX);
		int y = (int)((this.centerY - ex * this.sin - ey * this.cos) / this.scale + this.shiftY);

		Node point = null;
		double nodeDist = Double.POSITIVE_INFINITY;
		for (MapData map : this.maps.values()) {

			if (!map.hasData() || !map.getArea().contains(x, y)) {
				continue;
			}
			for (final Node node : map.getNode()) {
				final double d = node.distance(x, y, this.drawlevel);
				if (nodeDist > d) {
					point = node;
					nodeDist = d;
				}
			}
		}
		if (point == null) {
			return;
		}
		if (flag) {
			this.terminal = point;
			this.start = null;
			if (this.thread != null) {
				this.thread.kill();
			}
		} else {
			this.start = point;
			if (this.terminal != null && this.start != this.terminal) {
				if (this.thread != null) {
					this.thread.kill();
				}
				this.thread = this.algorithm.search(this.start, this.terminal);
			}
		}
	}
	public void clearLocation() {
		this.maps.clearStatus();
	}
	/**
	 * 行政界の描画
	 * 
	 * @param g
	 *            描画するGraphics
	 * @param curves
	 *            描画する行政界
	 * @param color
	 *            描画する色
	 */
	private void drawBorderCurve(Graphics g, Curve[] curves, Color color) {
		if (curves != null) {
			g.setColor(color);
			for (Curve curve : curves) {
				int[] aryX = curve.getArrayX();
				int[] aryY = curve.getArrayY();

				double tx = (aryX[0] - this.shiftX) * this.scale - this.centerX;
				double ty = this.getHeight() - (aryY[0] - this.shiftY) * this.scale - this.centerY;

				int x0 = (int)(  tx * this.cos + ty * this.sin) + this.centerX;
				int y0 = (int)(- tx * this.sin + ty * this.cos) + this.centerY;

				boolean isContains = this.screen.contains(aryX[0], aryY[0]);

				for (int k = 1; k < aryX.length; k++) {
					tx = (aryX[k] - this.shiftX) * this.scale - this.centerX;
					ty = this.getHeight() - (aryY[k] - this.shiftY) * this.scale - this.centerY;

					int x = (int)(  tx * this.cos + ty * this.sin) + this.centerX;
					int y = (int)(- tx * this.sin + ty * this.cos) + this.centerY;

					// 表示領域外であれば次へ
					if (isContains | (isContains = this.screen.contains(aryX[k], aryY[k]))) {
						g.drawLine(x0, y0, x, y);
					}
					x0 = x;
					y0 = y;
				}
			}
		}
	}

	/**
	 * 水域界の描画
	 * @param g 描画するGraphics2D
	 * @param coast 描画する水域界
	 * @param color
	 */
	private void drawCoast(Graphics2D g, Map<Curve, Boolean> coast, Color color1) {
		if (coast != null) {
			g.setColor(color1);
			for (Map.Entry<Curve, Boolean> entry : coast.entrySet()) {
				Curve curve = entry.getKey();
				int[] curveX = curve.getArrayX();
				int[] curveY = curve.getArrayY();


				if (this.isFill && entry.getValue() && curve.getType() == 0) {
					int[] x = new int[curveX.length];
					int[] y = new int[curveY.length];
					for (int i = 0; i < curveX.length; i++) {	
						double tx = (curveX[i] - this.shiftX) * this.scale - this.centerX;
						double ty = this.getHeight() - (curveY[i] - this.shiftY) * this.scale - this.centerY;
						x[i] = (int)(  tx * this.cos + ty * this.sin) + this.centerX;
						y[i] = (int)(- tx * this.sin + ty * this.cos) + this.centerY;
					}
					g.setColor(color1);
					g.fill(new Polygon(x, y, curveX.length));
				} else {
					g.setColor(color1);

					boolean isContains = this.screen.contains(curveX[0], curveY[0]);

					double tx = (curveX[0] - this.shiftX) * this.scale - this.centerX;
					double ty = this.getHeight() - (curveY[0] - this.shiftY) * this.scale - this.centerY;
					int x0 = (int)(  tx * this.cos + ty * this.sin) + this.centerX;
					int y0 = (int)(- tx * this.sin + ty * this.cos) + this.centerY;
					
					
					for (int i = 1; i < curveX.length; i++) {
						tx = (curveX[i] - this.shiftX) * this.scale - this.centerX;
						ty = this.getHeight() - (curveY[i] - this.shiftY) * this.scale - this.centerY;

						int x = (int)(  tx * this.cos + ty * this.sin) + this.centerX;
						int y = (int)(- tx * this.sin + ty * this.cos) + this.centerY;
						// 表示領域外であれば次へ
						if (isContains | (isContains = this.screen.contains(curveX[i], curveY[i]))) {
							g.drawLine(x0, y0, x, y);
						}
						x0 = x;
						y0 = y;
					}
				}
			}
		}
	}

	/**
	 * ヘルプを表示する
	 * @param g 描画するGraphics2D
	 * @param x 左上の座標
	 * @param y 左上の座標
	 */
	public void drawHelp(Graphics2D g, int x, int y) {
		g.setColor(new Color(255, 255, 255, 200));

		int width = 305;
		int height = 270;
		g.fillRect(x, y, width, height);

		g.setColor(Color.BLACK);
		g.setFont(new Font("SansSerif", Font.BOLD, 14));
		FontMetrics metrics = g.getFontMetrics();
		String msg = "操作マニュアル";
		g.drawString(msg,  x + (width - metrics.stringWidth(msg)) / 2, y + 25);
		g.setFont(new Font("SansSerif", Font.BOLD, 12));
		
		g.drawString("↑ / ↓ / ← / →", x + 180, y + 50);
		g.drawString("＋ / ； / PgUp", x + 180, y + 95);
		g.drawString("－ / ＝ / PgDn", x + 180, y + 140);
		g.drawString("？", x + 180, y + 185);

		g.drawString("マウス ドラッグ", x + 20, y + 50);
		g.drawString("マウス ホイール", x + 20, y + 95);
		g.drawString("Shift + マウス ホイール", x + 20, y + 140);
		g.drawString("右クリック", x + 20, y + 185);
		g.drawString("Shift + 右クリック", x + 20, y + 230);

		g.setFont(new Font("SansSerif", Font.PLAIN, 12));
		g.drawString("地図の平行移動",  x + 180, y + 70);
		g.drawString("地図の拡大",     x + 180, y + 115);
 		g.drawString("地図の縮小",     x + 180, y + 160);
		g.drawString("マニュアル表示切替", x + 180, y + 205);

 		g.drawString("地図の平行移動", x + 20, y + 70);
 		g.drawString("地図の拡大縮小", x + 20, y + 115);
 		g.drawString("地図の回転移動", x + 20, y + 160);
 		g.drawString("探索始点の選択 / 探索開始", x + 20, y + 205);
 		g.drawString("探索終点の選択 / 探索中止", x + 20, y + 250);
	}

	/**
	 * 数値地図25000を描画する
	 * 
	 * @param g
	 */
	public synchronized void drawMap(Graphics2D g) {

		int meshSize = (int) (Mesh.SIZE * this.scale) + 1;

		// メッシュの描画
		for (Map.Entry<Integer, MapData> entry : this.maps.entrySet()) {
			MapData map = entry.getValue();
			if (map == null) {
				System.out.println("map null !! code : " + entry.getKey());
				continue;
			}
			if(!map.hasData()) {
				this.drawPolygon(g, map.getPolygon(), true);
			} else {
				int halfMesh = Mesh.SIZE / 2;
				for (Mesh mesh : map.getMesh()) {
					// 表示領域外であれば次へ
					if (!this.screen.intersects(mesh.getX() - halfMesh, mesh.getY() - halfMesh, Mesh.SIZE, Mesh.SIZE)) {
						continue;
					}
					double tx = (mesh.getX() - this.shiftX) * this.scale - this.centerX;
					double ty = this.getHeight() - (mesh.getY() - this.shiftY) * this.scale - this.centerY;

					int x = (int)(  tx * this.cos + ty * this.sin) + this.centerX;
					int y = (int)(- tx * this.sin + ty * this.cos) + this.centerY;

					g.setColor(mesh.getColor());
					g.fillRect(x, y, meshSize, meshSize);
				}
			}
		}
		for (MapData map : this.maps.values()) {

			if (!map.hasData()) {
				continue;
			}

			// 行政界を描画
			this.drawBorderCurve(g, map.getBorder(), Color.GREEN);

			// 沿岸を描画
			this.drawCoast(g, map.getCoast(), Color.BLUE);

			// 河川を描画
			this.drawBorderCurve(g, map.getRiver(), Color.BLUE);
		}
		for (MapData map : this.maps.values()) {

			if (!map.hasData()) {
				continue;
			}
			
			// 道路を描画する
			Road[][] roads = map.getRoad();
			this.drawRoadway(g, roads);

			// 鉄道区間と駅区間を描画する
			this.drawRailway(g, map.getRailway(), map.getStation());

			if (this.isNodeView) {
				g.setColor(Color.GRAY);
				for (Node node : map.getNode()) {
					// 頂点サイズを計算
					int nodeSize = (int) (this.nodeSize / this.scale);
					// 表示領域外であれば次へ
					if (!this.screen.intersects(node.getX() - nodeSize, node.getY() - nodeSize, nodeSize * 2, nodeSize * 2)) {
						continue;
					}

					double tx = (node.getX() - this.shiftX) * this.scale - this.centerX;
					double ty = this.getHeight() - (node.getY() - this.shiftY) * this.scale - this.centerY;

					int x = (int)(  tx * this.cos + ty * this.sin) + this.centerX;
					int y = (int)(- tx * this.sin + ty * this.cos) + this.centerY;

					// draw node
					g.fillRect(x - this.nodeSize, y - this.nodeSize,
							this.nodeSize * 2, this.nodeSize * 2);
				}
			}
		}
		
		this.drawRoute(g);

		g.setStroke(new BasicStroke(1, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
		if (this.isLabel) {

//			Labeling labeling = new DefaultLabeling(g, area, this.shiftX, this.shiftY, this.scale);
			Labeling labeling = new SimpleLabeling(g, this.shiftX, this.shiftY, this.scale, this.screen, this.getWidth(), this.getHeight(), this.theta);
//			Labeling labeling = new Wagner(g, this.screen, this.getWidth(), this.getHeight(), this.scale);
			for (MapData map : this.maps.values()) {

				if (!map.hasData()) {
					continue;
				}

				labeling.add(map.getCity());
				labeling.add(map.getFacility());
			}
			labeling.draw();
		}
	}
	private boolean intersects(Polygon p1, Polygon p2) {
		for (int i = 0; i < p2.npoints; i++) {
			if (p1.contains(p2.xpoints[i], p2.ypoints[i])) {
				return true;
			}
		}
		return false;
	}
	/**
	 * ポリゴンを描画する
	 * @param g
	 * @param polygons
	 * @param fill
	 */
	private void drawPolygon(Graphics g, Polygon[] polygons, boolean fill) {
		for (int i = 0; i < polygons.length; i++) {
			if(this.intersects(this.screen, polygons[i])) {
				int[] curveX = polygons[i].xpoints;
				int[] curveY = polygons[i].ypoints;
				
				int[] polygonX = new int[polygons[i].npoints];
				int[] polygonY = new int[polygons[i].npoints];
		
				for (int j = 0; j < polygons[i].npoints; j++) {
					
					double x = (curveX[j] - this.shiftX) * this.scale - this.centerX;
					double y = this.getHeight() - (curveY[j] - this.shiftY) * this.scale - this.centerY;
					
					polygonX[j] = (int)(  x * this.cos + y * this.sin) + this.centerX;
					polygonY[j] = (int)(- x * this.sin + y * this.cos) + this.centerY;
				}
				if (fill) {
					g.setColor(Color.GREEN.brighter());
					g.fillPolygon(polygonX, polygonY, polygons[i].npoints);
				}
				g.setColor(Color.BLACK);
				g.drawPolygon(polygonX, polygonY, polygons[i].npoints);
			}
		}
	}

	/**
	 * 鉄道区間，駅区間を描画
	 * 
	 * @param g
	 *            描画するGraphics2D
	 * @param rails
	 *            描画する鉄道区間
	 * @param color
	 *            描画する色
	 */
	private void drawRailway(Graphics2D g, Curve[] rails, Curve[] station) {
		int w = (int)(this.STROKE_WIDTH * this.scale * 2f) + 1;
		if (rails != null) {
			GeneralPath jr = new GeneralPath();
			GeneralPath other = new GeneralPath();
			for (Curve curve : rails) {
				GeneralPath rail = (curve.getType() == 0) ? jr : other;
				int[] curveX = curve.getArrayX();
				int[] curveY = curve.getArrayY();

				boolean isContains = this.screen.contains(curveX[0], curveY[0]);
				
				double tx = (curveX[0] - this.shiftX) * this.scale - this.centerX;
				double ty = this.getHeight() - (curveY[0] - this.shiftY) * this.scale - this.centerY;
				
				int x0 = (int)(  tx * this.cos + ty * this.sin) + this.centerX;
				int y0 = (int)(- tx * this.sin + ty * this.cos) + this.centerY;

				rail.moveTo(x0, y0);

				for (int k = 1; k < curveX.length; k++) {

					tx = (curveX[k] - this.shiftX) * this.scale - this.centerX;
					ty = this.getHeight() - (curveY[k] - this.shiftY) * this.scale - this.centerY;
					
					int x = (int)(  tx * this.cos + ty * this.sin) + this.centerX;
					int y = (int)(- tx * this.sin + ty * this.cos) + this.centerY;

					// 表示領域外であれば次へ
					if (isContains | (isContains = this.screen.contains(curveX[k], curveY[k]))) {
						rail.lineTo(x, y);
					} else if (k != 1){
						rail.moveTo(x0, y0);
					}
					x0 = x;
					y0 = y;
				}
			}
			
			g.setStroke(new BasicStroke(w + 2, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
			g.setColor(Color.DARK_GRAY);
			g.draw(other);
			g.setColor(Color.BLACK);
			g.draw(jr);

			g.setColor(Color.BLACK);
			g.setStroke(new BasicStroke(w, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
			g.draw(other);
			g.setColor(Color.WHITE);
			g.setStroke(new BasicStroke(w, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND, 10f, new float[]{w * 3, w * 6}, 0));
			g.draw(jr);
		}
		if (station != null) {
			GeneralPath border = extractRoadway(station);
	
			g.setColor(Color.BLACK);
			g.setStroke(new BasicStroke(w * 3 + 2, BasicStroke.CAP_SQUARE, BasicStroke.JOIN_ROUND));
			g.draw(border);
	
			g.setColor(Color.WHITE);
			g.setStroke(new BasicStroke(w * 3, BasicStroke.CAP_SQUARE, BasicStroke.JOIN_ROUND));
			g.draw(border);
		}
	}

	/**
	 * 道路の描画
	 * 
	 * @param g Graphics2D
	 * @param road 描画する道路(一般道[6], 高速道路[6])
	 */
	private void drawRoadway(Graphics2D g, Road[][] road) {
		GeneralPath[] path = new GeneralPath[road.length];
		int[] width = new int[6];
		for (int i = 5; i >= 0; i--) {
			width[i] = (int)((i + 1) * this.STROKE_WIDTH * this.scale);
			if(width[i] > 0) {
				path[i] = this.extractRoadway(road[i]);
				this.drawlevel = i;
			}
			path[i + 6] = this.extractRoadway(road[i + 6]);
		}
		g.setColor(Color.LIGHT_GRAY);
		for (int i = 0; i < 6; i++) {
			if(path[i] != null) {
				g.setStroke(new BasicStroke(width[i] + 2, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
				g.draw(path[i]);
			}
		}
		g.setColor(Color.WHITE);
		for (int i = 0; i < 6; i++) {
			if(path[i] != null) {
				g.setStroke(new BasicStroke(width[i], BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
				g.draw(path[i]);
			}
		}
		g.setColor(Color.DARK_GRAY);
		for (int i = 0; i < 6; i++) {
			g.setStroke(new BasicStroke(width[i] + 3, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
			g.draw(path[i + 6]);
		}
		g.setColor(Color.GRAY);
		for (int i = 0; i < 6; i++) {
			g.setStroke(new BasicStroke(width[i] + 1, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
			g.draw(path[i + 6]);
		}
	}
	/**
	 * 探索済み頂点を描画する
	 * 
	 * @param g
	 */
	private void drawRoute(Graphics2D g) {
		if (this.thread != null) {
			Collection<Road> route = this.thread.getRoute();
			if (route != null) {
				int[] width = new int[12];
				for (int i = 0; i < 6; i++) {
					width[i] = (int)((i + 1) * this.STROKE_WIDTH * this.scale) + 1;
				}
				for (Road road : route) {
					int[] aryX = road.getArrayX();
					int[] aryY = road.getArrayY();
		
					boolean isContain = this.screen.contains(aryX[0], aryY[0]);
					
					double tx = (aryX[0] - this.shiftX) * this.scale - this.centerX;
					double ty = this.getHeight() - (aryY[0] - this.shiftY) * this.scale - this.centerY;
					
					int x0 = (int)(  tx * this.cos + ty * this.sin) + this.centerX;
					int y0 = (int)(- tx * this.sin + ty * this.cos) + this.centerY;
					
					if (road.getType() == 3) {
						g.setColor(this.ROUTE_COLOR_HIGHWAY);
					} else {
						g.setColor(this.ROUTE_COLOR_GENERAL);
					}
					g.setStroke(new BasicStroke(width[road.getWidth()], BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
		
					for (int k = 1; k < aryX.length; k++) {

						tx = (aryX[k] - this.shiftX) * this.scale - this.centerX;
						ty = this.getHeight() - (aryY[k] - this.shiftY) * this.scale - this.centerY;
						
						int x = (int)(  tx * this.cos + ty * this.sin) + this.centerX;
						int y = (int)(- tx * this.sin + ty * this.cos) + this.centerY;
						
						// 表示領域内であれば描画
						if (isContain | (isContain = this.screen.contains(aryX[k], aryY[k]))) {
							g.drawLine(x0, y0, x, y);
						}
						x0 = x;
						y0 = y;
					}
				}
			}
		}
		int r = (int)(this.scale * 500) + 2;
		g.setStroke(new BasicStroke(r / 2f));

		if (this.start != null) {

			double tx = (this.start.getX() - this.shiftX) * this.scale - this.centerX;
			double ty = this.getHeight() - (this.start.getY() - this.shiftY) * this.scale - this.centerY;
			
			int x = (int)(  tx * this.cos + ty * this.sin) + this.centerX - r;
			int y = (int)(- tx * this.sin + ty * this.cos) + this.centerY - r;

			g.setColor(Color.YELLOW);
			g.fillOval(x, y, r * 2, r * 2);
			g.setColor(Color.BLACK);
			g.drawOval(x, y, r * 2, r * 2);
		}
		if (this.terminal != null) {

			double tx = (this.terminal.getX() - this.shiftX) * this.scale - this.centerX;
			double ty = this.getHeight() - (this.terminal.getY() - this.shiftY) * this.scale - this.centerY;
			
			int x = (int)(  tx * this.cos + ty * this.sin) + this.centerX - r;
			int y = (int)(- tx * this.sin + ty * this.cos) + this.centerY - r;

			g.setColor(Color.RED);
			g.fillOval(x, y, r * 2, r * 2);
			g.setColor(Color.BLACK);
			g.drawOval(x, y, r * 2, r * 2);
		}
	}
	/**
	 * 道路データをGeneralPathに展開する
	 * @param curves 道路データ
	 * @return 展開した道路データ
	 */
	private GeneralPath extractRoadway(Curve[] curves) {
		GeneralPath border = new GeneralPath();
		if (curves != null) {
			for (Curve curve : curves) {
				int[] curveX = curve.getArrayX();
				int[] curveY = curve.getArrayY();
	
				boolean isContains = this.screen.contains(curveX[0], curveY[0]);
				
				double tx = (curveX[0] - this.shiftX) * this.scale - this.centerX;
				double ty = this.getHeight() - (curveY[0] - this.shiftY) * this.scale - this.centerY;
				
				int x0 = (int)(  tx * this.cos + ty * this.sin) + this.centerX;
				int y0 = (int)(- tx * this.sin + ty * this.cos) + this.centerY;
	
				border.moveTo(x0, y0);
				
				for (int k = 1; k < curveX.length; k++) {
	
					tx = (curveX[k] - this.shiftX) * this.scale - this.centerX;
					ty = this.getHeight() - (curveY[k] - this.shiftY) * this.scale - this.centerY;
					
					int x = (int)(  tx * this.cos + ty * this.sin) + this.centerX;
					int y = (int)(- tx * this.sin + ty * this.cos) + this.centerY;

					// 表示領域外であれば次へ
					if (isContains | (isContains = this.screen.contains(curveX[k], curveY[k]))) {
						border.lineTo(x, y);
					} else if (k != 1) {
						border.moveTo(x0, y0);
					}
					x0 = x;
					y0 = y;
				}
			}
		}
		return border;
	}
	public void getLocation(int ex, int ey) {
		int x = (int)(ex / this.scale + this.shiftX);
		int y = (int)((this.getHeight() - ey) / this.scale + this.shiftY);
		this.maps.setStatus(x, y);
	}
	/**
	 * スクリーン情報の取得
	 * 
	 * @return スクリーンのRectangle
	 */
	public Polygon getScreen() {
		return this.screen;
	}
	/**
	 * 初期設定
	 */
	public void init(MapMap map, Polygon[] prefecture) {
		this.algorithm = new ShortestPathSearch(new DirectDistance(), map);
		this.maps = map;
		
		this.prefecture = prefecture;

		this.theta = 0;
		this.sin = 0;
		this.cos = 1;

		this.shiftX = 122 * 3600000;
		this.shiftY = 20 * 3600000;

		this.isLabel = true;
		this.isNodeView = false;
		this.isAntialias = true;

		this.isFill = false;
		
		this.isHelp = true;

		this.isOperation = false;
		
		super.setBackground(Color.CYAN);
		
		double widthScale = (double) this.getWidth() / ((154 - 122) * 3600000);
		double heightScale = (double) this.getHeight() / ((46 - 20) * 3600000);
		this.scale = (float) ((widthScale < heightScale) ? widthScale : heightScale);
		
		this.repaint();
	}
	public boolean isOperation() {
		return this.isOperation;
	}
	/**
	 * どの地図を表示する
	 * @return 
	 */
	public int mode() {
		if(this.scale > this.DATA25K_SCALE) {
			return 2;
		} else if(this.scale > this.CITY_SCALE) {
			return 1;
		} else {
			return 0;
		}
	}
	/**
	 * 平行移動を行う
	 * 
	 * @param dx
	 * @param dy
	 */
	public void moveLocation(int dx, int dy) {
		this.shiftX -= (dx * this.cos - dy * this.sin) / this.scale;
		this.shiftY += (dx * this.sin + dy * this.cos) / this.scale;
		this.repaint();
	}
	@Override
	public void paintComponent(Graphics g) {
		
		// オフスクリーンバッファ
		if(this.maps == null) {
			this.offs = createImage(this.getWidth(), this.getHeight());
			Graphics2D offg = (Graphics2D)this.offs.getGraphics();
			super.paintComponent(offg);
			Font defaultFont = g.getFont();
			offg.setFont(new Font("gothic", Font.PLAIN, 20));
			FontMetrics metrics = offg.getFontMetrics();
			String msg = "Now Loading...";
			offg.drawString(msg, (this.getWidth() - metrics.stringWidth(msg)) / 2, (this.getHeight() - metrics.getHeight()) / 2);
			offg.setFont(defaultFont);
		} else {
			if(this.getWidth() != this.offs.getWidth(null) || this.getHeight() != this.offs.getHeight(null)) {
				this.offs.flush();
				this.offs = createImage(this.getWidth(), this.getHeight());
			}
			Graphics2D offg = (Graphics2D)this.offs.getGraphics();
			super.paintComponent(offg);

			this.centerX = this.getWidth() / 2;
			this.centerY = this.getHeight() / 2;

			int[] x0 = new int[]{0, this.getWidth(), this.getWidth(), 0};
			int[] y0 = new int[]{0, 0, this.getHeight(), this.getHeight()};
			int[] rotateX = new int[4];
			int[] rotateY = new int[4];
			for (int i = 0; i < 4; i++) {
				rotateX[i] = (int)((this.centerX + (x0[i] - this.centerX) * this.cos - (y0[i] - this.centerY) * this.sin) / this.scale + this.shiftX);
				rotateY[i] = (int)((this.centerY - (x0[i] - this.centerX) * this.sin - (y0[i] - this.centerY) * this.cos) / this.scale + this.shiftY);
			}
			this.screen = new Polygon(rotateX, rotateY, 4);
				
			if(this.scale > this.CITY_SCALE) {
				if (!this.isOperation && this.isAntialias) {
					offg.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
					offg.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_OFF);
				}
				if (this.scale >= this.DATA25K_SCALE) {
					this.drawMap(offg);
				} else {
					for (MapData map : this.maps.values()) {
						this.drawPolygon(offg, map.getPolygon(), true);
					}
					offg.setStroke(new BasicStroke(2, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
					this.drawPolygon(offg, this.prefecture, false);
					this.drawRoute(offg);
				}
			} else {
				this.drawPolygon(offg, this.prefecture, true);
			}

			offg.setColor(Color.BLACK);
			offg.drawString("SCALE : " + Float.toString(this.scale * 1000) + "m", 5, 15);
			// アルゴリズム名の表示
			int vp = this.thread != null ? this.thread.getVP() : 0;
			if (vp == 0) {
				offg.drawString(this.algorithm.toString(), 5, 35);
			} else {
				offg.drawString(this.algorithm.toString() + " : " + vp, 5, 35);
			}
			if (this.isHelp) {
				this.drawHelp(offg, this.getWidth() - 325, this.getHeight() - 290);
			}
//			this.drawPolygon(offg, new Polygon[]{this.rotateScreen}, true);
		}
		g.drawImage(this.offs, 0, 0, null);
	}
	/**
	 * リルート計算
	 *
	 */
	public void reroute() {
		if(this.start != null && this.terminal != null && this.start != this.terminal) {
			if (this.thread != null) {
				this.thread.kill();
			}
			this.thread = this.algorithm.search(this.start, this.terminal);
			System.out.printf("s : %d, t : %d\n", this.start.getID(), this.terminal.getID());
		}
	}
	public void rotateLocation(int d) {
		this.theta = (this.theta + d * this.ROTATE_SENSE) % (Math.PI * 2);
		this.sin = Math.sin(this.theta);
		this.cos = Math.cos(this.theta);
		this.repaint();
	}
	/**
	 * マウス操作の状態を設定する
	 * @param flag マウス操作していればtrue
	 */
	public void setOperation(boolean flag) {
		this.isOperation = flag;
		if(!flag) {
			this.repaint();
		}
	}
	/**
	 * 水域界の塗りつぶしを切り替える
	 */
	public void switchFill() {
		this.isFill = !this.isFill;
		this.repaint();
	}
	/**
	 * 操作マニュアルの表示を切り替える
	 */
	public void switchHelp() {
		this.isHelp = !this.isHelp;
		this.repaint();
	}

	/**
	 * ラベル表示を切り替える
	 */
	public void switchLabel() {
		this.isLabel = !this.isLabel;
		this.repaint();
	}
	/**
	 * 頂点表示を切り替える
	 */
	public void switchNodeView() {
		this.isNodeView = !this.isNodeView;
		this.repaint();
	}
	/**
	 * アンチエイリアスを切り替える
	 */
	public void switchRendering() {
		this.isAntialias = !this.isAntialias;
		this.repaint();
	}


	/**
	 * 最短経路を求めるアルゴリズムを切り替える
	 */
	public void switchShortestPathAlgorithm(String algorithm) {
		if(algorithm.endsWith("_a*")) {
			this.algorithm = new ShortestPathSearch(new DirectDistance(), this.maps);
		} else if (algorithm.endsWith("_dijkstra")) {
			this.algorithm = new ShortestPathSearch(null, this.maps);
		}
		this.reroute();
		this.repaint();
	}
	/**
	 * 拡大縮小を行う
	 * 
	 * @param ex
	 * @param ey
	 * @param d
	 */
	public void zoomLocation(int ex, int ey, int d) {
		float newScale = this.scale * (1 + d * this.SCALE_SENSE);
		if (newScale > this.SCALE_MAX) {
			newScale = this.SCALE_MAX;
		} else if (newScale < this.SCALE_MIN) {
			newScale = this.SCALE_MIN;
		}
		
		ex -= this.centerX;
		ey -= this.centerY;

		double x = ex * this.cos - ey * this.sin + this.centerX;
		double y = ex * this.sin + ey * this.cos + this.centerY;
		y = this.getHeight() - y;
		
		this.shiftX = (int) (this.shiftX + x / this.scale - x / newScale + 0.5);
		this.shiftY = (int) (this.shiftY + y / this.scale - y / newScale + 0.5);

		this.scale = newScale;
		repaint();
	}
}