package map.ksj;

import java.awt.Polygon;
import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.PathIterator;
import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Scanner;

import map.data.MapData;
import map.data.MapMap;

public class KsjFactory {
	public static void main(String[] args) {
		try {
			Polygon[] polygons = KsjFactory.readShapes(new BufferedInputStream(new FileInputStream(new File("../ksj/ksj.dat"))));
			KsjFactory.outputPolygon("../ksj/ksj.txt", polygons);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
	/**
	 * Shapeオブジェクトの一覧をストリームに書き込みます。
	 * 文字列中に「,」が含まれていると、正しく読み込めなくなるので注意してください。
	 * @param shapes Shapeオブジェクトと文字列の入ったマップ
	 * @param outputStream 出力ストリーム
	 * @throws UnsupportedEncodingException サポート外エンコーディング例外
	 */
	public static void writeShape(final Map<Shape, String> shapes, final OutputStream outputStream) throws UnsupportedEncodingException {
		PrintWriter out = null;
		try {
			out = new PrintWriter(new OutputStreamWriter(outputStream, "SJIS"));
			out.println("# label, type1, x1, y1, type2, x2, y2, ...");
			for (final Map.Entry<Shape, String> entry : shapes.entrySet()) {
				out.print(entry.getValue());
				final PathIterator iterator = entry.getKey().getPathIterator(new AffineTransform());
				while (!iterator.isDone()) {
					final float[] coords = new float[6];
					final int type = iterator.currentSegment(coords);
					out.print("," + type + "," + coords[0] + "," + coords[1]);
					iterator.next();
				}
				out.println();
			}
		} finally {
			if (out != null) {
				out.close();
			}
		}
	}
	public static Polygon[] readPolygon(final InputStream stream) throws IOException {
		List<Polygon> list = new ArrayList<Polygon>();
		BufferedReader in = null;
		try {
			in = new BufferedReader(new InputStreamReader(stream));
			String data;
			while ((data = in.readLine()) != null) {
				String[] param = data.split(",");
				int length = param.length / 2;
				int[] x = new int[length];
				int[] y = new int[length];	
				for (int i = 0; i < length; i++) {
					int index = i * 2;
					x[i] = Integer.parseInt(param[index]);
					y[i] = Integer.parseInt(param[index + 1]);
				}
				if((param.length % 2) != 0) {
					System.out.println("error KsjFactory : "+ param.length);
				}
				list.add(new Polygon(x, y, length));
			}
		} finally {
			if (in != null) {
				in.close();
			}
		}
		return list.toArray(new Polygon[]{});
	}
	public static Map<String, List<Polygon>> readCityPolygon(final InputStream stream) throws IOException {
		Map<String, List<Polygon>> map = new LinkedHashMap<String, List<Polygon>>();
		BufferedReader in = null;
		try {
			in = new BufferedReader(new InputStreamReader(stream));
			String data;
			while ((data = in.readLine()) != null) {
				String[] param = data.split(",");
				int length = (param.length - 1)/ 2;
				int[] x = new int[length];
				int[] y = new int[length];	
				for (int i = 1; i <= length; i++) {
					int index = i * 2;
					x[i] = Integer.parseInt(param[index]);
					y[i] = Integer.parseInt(param[index + 1]);
				}
				if((param.length % 2) != 0) {
					System.out.println("error!!");
				}
				if(!map.containsKey(param[0])) {
					map.put(param[0], new ArrayList<Polygon>());
				}
				map.get(param[0]).add(new Polygon(x, y, length));
			}
		} finally {
			if (in != null) {
				in.close();
			}
		}
		return map;
	}
	public static void readMapPolygon(final MapMap maps, final InputStream in) throws IOException {
		Map<String, List<Polygon>> map = new LinkedHashMap<String, List<Polygon>>();
		BufferedReader bi = null;
		try {
			bi = new BufferedReader(new InputStreamReader(in));
			String data;
			while ((data = bi.readLine()) != null) {
				String[] param = data.split(",");
				int length = param.length / 2;
				int[] x = new int[length];
				int[] y = new int[length];	
				for (int i = 0; i < length; i++) {
					int index = i * 2 + 1;
					x[i] = Integer.parseInt(param[index]);
					y[i] = Integer.parseInt(param[index + 1]);
				}
				if((param.length % 2) != 1) {
					System.out.println("error!!");
				}
				if(!map.containsKey(param[0])) {
					map.put(param[0], new ArrayList<Polygon>());
				}
				map.get(param[0]).add(new Polygon(x, y, length));
			}
		} finally {
			if(bi != null) {
				bi.close();
			}
		}
		for (Map.Entry<String, List<Polygon>> entry : map.entrySet()) {
			maps.put(Integer.parseInt(entry.getKey()), new MapData(Integer.parseInt(entry.getKey()), entry.getValue().toArray(new Polygon[]{})));
		}
	}
	/**
	 * ストリームからShapeオブジェクトの一覧を読み込みます。
	 * @param in 入力ストリーム
	 * @return Shapeオブジェクトの一覧
	 */
	public static Polygon[] readShapes(final InputStream in) {
		List<Polygon> list = new ArrayList<Polygon>();
		Scanner scanner = null;
		try {
			scanner = new Scanner(in, "SJIS");
			while (scanner.hasNextLine()) {
				final String line = scanner.nextLine();
				if (!line.startsWith("#")) {
					final Scanner scanner2 = new Scanner(line);
					scanner2.useDelimiter(",");
					if (scanner2.hasNext()) {
						scanner2.next();
						List<Float> tmpX = new ArrayList<Float>();
						List<Float> tmpY = new ArrayList<Float>();
						while (scanner2.hasNextInt()) {
							if (scanner2.hasNextInt()) {
								final int type = scanner2.nextInt();
								if (scanner2.hasNextFloat()) {
									final float x = scanner2.nextFloat();
									if (scanner2.hasNextFloat()) {
										final float y = scanner2.nextFloat();
										switch (type) {
										case PathIterator.SEG_MOVETO:
											tmpX = new ArrayList<Float>();
											tmpY = new ArrayList<Float>();
										case PathIterator.SEG_LINETO:
										case PathIterator.SEG_CUBICTO:
										case PathIterator.SEG_QUADTO:
											tmpX.add(x);
											tmpY.add(y);
											break;
										case PathIterator.SEG_CLOSE:
											int[] aryX = new int[tmpX.size()];
											int[] aryY = new int[tmpY.size()];
											for (int i = 0; i < tmpX.size(); i++) {
												aryX[i] = tmpX.get(i).intValue();
												aryY[i] = tmpY.get(i).intValue();
											}
											list.add(new Polygon(aryX, aryY, aryX.length));
											break;
										}
									}
								}
							}
						}
					}
				}
			}
		} finally {
			if (scanner != null) {
				scanner.close();
			}
		}
		return list.toArray(new Polygon[]{});
	}
	/**
	 * 読み込んだポリゴンファイルをx0,y0,x1,y1,x2,y2,...のように
	 * 指定したファイルに出力します。
	 * @param file 出力ファイル
	 * @throws IOException 
	 */
	public static void outputPolygon(String file, Polygon[] polygons) throws IOException {
		BufferedWriter out = null;
		try {
			out = new BufferedWriter(new FileWriter(file));
			for (Polygon polygon : polygons) {
				int[] x = polygon.xpoints;
				int[] y = polygon.ypoints;
				StringBuilder sb = new StringBuilder();
				sb.append(x[0]);
				sb.append(',');
				sb.append(y[0]);
				for (int i = 1; i < polygon.npoints; i++) {
					sb.append(',');
					sb.append(x[i]);
					sb.append(',');
					sb.append(y[i]);
				}
				out.write(sb.toString());
				out.newLine();
			}
			out.flush();
		} finally {
			if(out != null) {
				out.close();
			}
		}
	}
}
