package view;

import java.awt.Graphics;
import java.awt.Image;
import java.awt.Point;
import java.io.IOException;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Random;

import javax.imageio.ImageIO;
import javax.swing.JPanel;

import model.Cast;

/**
 * BattleField
 * 
 * @author Masayasu Fujiwara
 */
public class Field extends JPanel {
	
	/**
	 * はみ出してX座標に描画する量
	 */
	public static final int CLIPPING_X = 10;

	/**
	 * はみ出してY座標に描画する量
	 */
	public static final int CLIPPING_Y = 10;

	/**
	 * フロアのサイズ
	 */
	public static final int FLOOR_SIZE = 20;

	/**
	 * ブロック（迷路作成時の作りかけの壁）
	 */
	private static final int STATE_BLOCK = 2;

	/**
	 * フロア
	 */
	private static final int STATE_FLOOR = 0;

	/**
	 * 壁
	 */
	private static final int STATE_WALL = 1;
	
	/**
	 * 背景画像
	 */
	private Image background;
	
	/**
	 * フロアの状態
	 */
	private int[][] floorStatus;

	/**
	 * 床の画像
	 */
	private final Image IMG_FLOOR;
	
	/**
	 * 壁の画像
	 */
	private final Image[] IMG_WALL;

	/**
	 * オフスクリーンイメージ
	 */
	private Image offs;
	
	/**
	 * 左上のフロアのX座標
	 */
	private int originX;

	/**
	 * 左上のフロアのY座標
	 */
	private int originY;

	/**
	 * 敵
	 */
	private Collection<Cast> predators;

	/**
	 * プレイヤ
	 */
	private Cast prey;

	/**
	 * フィールドのサイズ
	 */
	private final int size;
	
	
	public Field(int size) throws IOException {
		this.size = size - 1;
		this.IMG_FLOOR = ImageIO.read(Field.class.getResource("/img/floor.png"));
		this.IMG_WALL = this.readImages("/img/wall-", 4);
		this.floorStatus = new int[size][size];
	}

	/**
	 * プレイヤーを加えます。
	 * @param predators プレイヤ
	 */
	public void addEnemy(Collection<Cast> predators) {
		this.predators = predators;
	}

	/**
	 * プレイヤーを加えます。
	 * @param player プレイヤ
	 */
	public void addPlayer(Cast player) {
		player.setLocation(this.originX, this.originY);
		this.prey = player;
	}
	
	/**
	 * プレイヤーが下方向へ移動できるか確認します。
	 * @return 移動できればtrue
	 */
	public boolean checkDown(Cast cast) {
		int castY = this.getCastY(cast);
		if (castY + 1 > this.size) {
			return false;
		} else if (this.floorStatus[this.getCastX(cast)][castY + 1] == Field.STATE_WALL) {
			return false;
		}
		return true;
	}

	public boolean checkDown(int x, int startY, int endY) {
		if (startY >= endY) {
			return false;
		}
		for (int y = startY; y > endY; y--) {
			if (this.floorStatus[x][y] == Field.STATE_WALL) {
				return false;
			}
		}
		return true;
	}
	
	/**
	 * 指定した座標へ移動できるか確認します。
	 * @return 移動できればtrue
	 */
	public boolean checkFloor(int x, int y) {
		if (x < 0 || y < 0 || x > this.size || y > this.size || this.floorStatus[x][y] == Field.STATE_WALL) {
			if (x < -1 || y < -1) {
				System.out.printf("check %d, %d\n", x, y);
			}
			return false;
		}
		return true;
	}

	/**
	 * プレイヤーが左方向へ移動できるか確認します。
	 * @return 移動できればtrue
	 */
	public boolean checkLeft(Cast cast) {
		int castX = this.getCastX(cast);
		if (castX - 1 < 0) {
			return false;
		} else if (this.floorStatus[castX - 1][this.getCastY(cast)] == Field.STATE_WALL) {
			return false;
		}
		return true;
	}
	
	public boolean checkLeft(int y, int startX, int endX) {
		if (startX <= endX) {
			return false;
		}
		for (int x = startX; x < endX; x++) {
			if (this.floorStatus[x][y] == Field.STATE_WALL) {
				return false;
			}
		}
		return true;
	}

	/**
	 * プレイヤーが右方向へ移動できるか確認します。
	 * @return 移動できればtrue
	 */
	public boolean checkRight(Cast cast) {
		int castX = this.getCastX(cast);
		if (castX + 1 > this.size) {
			return false;
		} else if (this.floorStatus[castX + 1][this.getCastY(cast)] == Field.STATE_WALL) {
			return false;
		}
		return true;
	}
	
	public boolean checkRight(int y, int startX, int endX) {
		if (startX >= endX) {
			return false;
		}
		for (int x = startX; x > endX; x--) {
			if (this.floorStatus[x][y] == Field.STATE_WALL) {
				return false;
			}
		}
		return true;
	}
	
	/**
	 * プレイヤーが上方向へ移動できるか確認します。
	 * @return 移動できればtrue
	 */
	public boolean checkUp(Cast cast) {
		int castY = this.getCastY(cast);
		if (castY - 1 < 0) {
			return false;
		} else if (this.floorStatus[this.getCastX(cast)][castY - 1] == Field.STATE_WALL) {
			return false;
		}
		return true;
	}

	public boolean checkUp(int x, int startY, int endY) {
		if (startY <= endY) {
			return false;
		}
		for (int y = startY; y < endY; y++) {
			if (this.floorStatus[x][y] == Field.STATE_WALL) {
				return false;
			}
		}
		return true;
	}

	/**
	 * 背景画像を作成します。
	 * @param size サイズ
	 */
	private Image createBackground() {
		int panelWidth = this.getWidth() + Field.CLIPPING_X * 2;
		int panelHeight = this.getHeight() + Field.CLIPPING_Y * 2;
		Image image = this.createImage(panelWidth, panelHeight);
		Graphics offg = image.getGraphics();
		offg.drawLine(0, 0, 100, 100);
		int widthFloor = panelWidth / Field.FLOOR_SIZE;
		int heightFloor = panelHeight / Field.FLOOR_SIZE;
		for (int x = 0; x <= widthFloor; x++) {
			for (int y = 0; y <= heightFloor; y++) {
				offg.drawImage(this.IMG_FLOOR, x * Field.FLOOR_SIZE - Field.CLIPPING_X, y * Field.FLOOR_SIZE - Field.CLIPPING_Y, null);
			}
		}
		int startX = (widthFloor - this.size) / 2;
		int endX = this.size + startX;
		int startY = (heightFloor - this.size) / 2;
		int endY = this.size + startY;
		for (int x = 0; x <= widthFloor; x++) {
			for (int y = 0; y <= heightFloor; y++) {
				if (x < startX || x > endX || y < startY || y > endY) {
					int index = (int)(this.IMG_WALL.length * Math.random());
					offg.drawImage(this.IMG_WALL[index], x * Field.FLOOR_SIZE - Field.CLIPPING_X, y * Field.FLOOR_SIZE - Field.CLIPPING_Y, null);
				}
			}
		}
		this.originX = startX * Field.FLOOR_SIZE - Field.CLIPPING_X;
		this.originY = startY * Field.FLOOR_SIZE - Field.CLIPPING_Y;
		for (int x = startX; x <= endX; x++) {
			for (int y = startY; y <= endY; y++) {
				if (this.floorStatus[x - startX][y - startY] == Field.STATE_WALL) {
					int index = (int)(this.IMG_WALL.length * Math.random());
					offg.drawImage(this.IMG_WALL[index], x * Field.FLOOR_SIZE - Field.CLIPPING_X, y * Field.FLOOR_SIZE - Field.CLIPPING_Y, null);
				}
			}
		}
		return image;
	}

	/**
	 * 迷路の壁を作ります。
	 * @param x
	 * @param y
	 */
	private void createWall(int x, int y, Random generator) {
		if (this.floorStatus[x][y] == Field.STATE_FLOOR) {
			Collection<Point> points = new ArrayList<Point>();
			points.add(new Point(x, y));
			int x1 = x;
			int y1 = y;
			while (true) {
				int direction = generator.nextInt(4);
				int dx;
				int dy;
				if (direction >= 2) {
					dx = 0;
					if (direction == 2) {
						dy = 1;
					} else {
						dy = -1;
					}
				} else {
					dy = 0;
					if (direction == 0) {
						dx = 1;
					} else {
						dx = -1;
					}
				}
				int nextX = x1 + dx * 2;
				int nextY = y1 + dy * 2;
				if (nextX < 0 || nextY < 0 || nextX >= this.size || nextY >= this.size) {
					points.add(new Point(x1 + dx, y1 + dy));
					for (Point point : points) {
						this.floorStatus[point.x][point.y] = Field.STATE_WALL;
					}
					points.clear();
					return;
				} else {
					switch (this.floorStatus[nextX][nextY]) {
						case STATE_FLOOR :
							points.add(new Point(x1 + dx, y1 + dy));
							points.add(new Point(nextX, nextY));
							this.floorStatus[x1 + dx][y1 + dy] = Field.STATE_BLOCK;
							this.floorStatus[nextX][nextY] = Field.STATE_BLOCK;
							x1 = nextX;
							y1 = nextY;
							break;
						case STATE_WALL :
							points.add(new Point(x1 + dx, y1 + dy));
							points.add(new Point(nextX, nextY));
							for (Point point : points) {
								this.floorStatus[point.x][point.y] = Field.STATE_WALL;
							}
							points.clear();
							return;
						case STATE_BLOCK :
							for (Point point : points) {
								this.floorStatus[point.x][point.y] = Field.STATE_FLOOR;
							}
							points.clear();
							points.add(new Point(x, y));
							x1 = x;
							y1 = y;
							break;
					}
				}
			}
		}
	}

	/**
	 * ボンバーマン風のフィールドを作成します。
	 */
	public void fieldBomb() {
		this.fieldClear();
		for (int x = 1; x < this.floorStatus.length; x += 2) {
			for (int y = 1; y < this.floorStatus.length; y += 2) {
				this.floorStatus[x][y] = Field.STATE_WALL;
			}
		}
	}

	/**
	 * すべてフロアのフィールドを作成します。
	 */
	public void fieldClear() {
		for (int x = 0; x < this.floorStatus.length; x++) {
			for (int y = 0; y < this.floorStatus.length; y++) {
				this.floorStatus[x][y] = Field.STATE_FLOOR;
			}
		}
	}

	/**
	 * 迷路のフィールドを作成します。
	 */
	public void fieldMaze() {
		this.fieldClear();
		Random generator = new Random();
		for (int x = 1; x <= this.size; x += 2) {
			for (int y = 1; y <= this.size; y += 2) {
				this.createWall(x, y, generator);
			}
		}
	}

	/**
	 * キャストの位置をフロア単位で返します。
	 * @param cast 位置を返すキャスト
	 * @return キャストのX方向の位置（単位はフロア数）
	 */
	public Point getCastPoint(Cast cast) {
		int x = (cast.getX() - this.originX) / Field.FLOOR_SIZE;
		int y = (cast.getY() - this.originY) / Field.FLOOR_SIZE;
		return new Point(x, y);
	}
	
	/**
	 * キャストのX方向の位置をフロア単位で返します。
	 * @param cast 位置を返すキャスト
	 * @return キャストのX方向の位置（単位はフロア数）
	 */
	public int getCastX(Cast cast) {
		return (cast.getX() - this.originX) / Field.FLOOR_SIZE;
	}

	/**
	 * キャストのY方向の位置をフロア単位で返します。
	 * @param cast 位置を返すキャスト
	 * @return キャストのY方向の位置（単位はフロア数）
	 */
	public int getCastY(Cast cast) {
		return (cast.getY() - this.originY) / Field.FLOOR_SIZE;
	}

	/**
	 * プレイヤの位置をフロア単位で返します。
	 * @param cast 位置を返すキャスト
	 * @return キャストのX方向の位置（単位はフロア数）
	 */
	public Point getPreyPoint() {
		int x = (this.prey.getX() - this.originX) / Field.FLOOR_SIZE;
		int y = (this.prey.getY() - this.originY) / Field.FLOOR_SIZE;
		return new Point(x, y);
	}
	
	/**
	 * プレイヤのX方向の位置をフロア単位で返します。
	 * @return プレイヤのX方向の位置（単位はフロア数）
	 */
	public int getPreyX() {
		return (this.prey.getX() - this.originX) / Field.FLOOR_SIZE;
	}
	
	/**
	 * プレイヤのY方向の位置をフロア単位で返します。
	 * @return プレイヤのY方向の位置（単位はフロア数）
	 */
	public int getPreyY() {
		return (this.prey.getY() - this.originY) / Field.FLOOR_SIZE;
	}
	
	/**
	 * 初期化を行います。
	 * フレーム可視化後に行います。
	 */
	public void init() {
		this.background = this.createBackground();
		this.prey.setLocation(this.originX, this.originY);
		for (Cast cast : this.predators) {
			int enemyX = this.originX + (int)(Math.random() * this.size) * Field.FLOOR_SIZE;
			int enemyY = this.originY + (int)(Math.random() * this.size) * Field.FLOOR_SIZE;
			cast.setLocation(enemyX, enemyY);
		}
		this.repaint();
	}
	@Override
	public void paint(Graphics g) {
		if (this.offs == null) {
			this.offs = this.createImage(this.getWidth(), this.getHeight());
		}
		Graphics offg = this.offs.getGraphics();
		offg.drawImage(this.background, 0, 0, null);
		this.prey.draw(offg);
		for (Cast cast : this.predators) {
			cast.draw(offg);
		}
		g.drawImage(this.offs, 0, 0, null);
	}
	/**
	 * 画像データを読み込みます。
	 * @param label 画像のラベル
	 * @param length 読み込む枚数
	 * @return 読み込んだ画像の配列
	 * @throws IOException 
	 */
	private Image[] readImages(String label, int length) throws IOException {
		NumberFormat format = new DecimalFormat("00");
		List<Image> list = new ArrayList<Image>();
		for (int i = 0; i < length; i++) {
			list.add(ImageIO.read(Field.class.getResource(label + format.format(i)+".png")));
		}
		return list.toArray(new Image[]{});
	}
	/**
	 * 背景を再描画します。
	 */
	public void repaintBackground() {
		this.background = this.createBackground();
		this.repaint();
	}
}
