//******************************************************************************
//
// MIDITrail / MTSettingBGIMGViewCtrl
//
// 背景画像選択ビュークラス
//
// Copyright (C) 2017-2019 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

#import "MTParam.h"
#import "MTSettingBGIMGViewCtrl.h"


//******************************************************************************
// プライベートメソッド定義
//******************************************************************************
@interface MTSettingBGIMGViewCtrl ()

//ファイル一覧生成
- (int)makeFileList;

//テーブルセル作成：ファイル
- (UITableViewCell*)makeFileCellForIndexPath:(NSIndexPath*)indexPath;

//テーブルセル選択イベント：ファイル一覧
- (void)onSelectFileCellForIndexPath:(NSIndexPath*)indexPath tableView:(UITableView*)tableView;

//ファイル削除
- (int)removeFileWithPath:(NSString*)pPath;

//ファイルリネーム
- (int)renameFileWithPath:(NSString*)pSrcPath toPath:(NSString*)pDestPath;

//リフレッシュ
-(void)refreshTable:(id)sender;

@end


@implementation MTSettingBGIMGViewCtrl

//******************************************************************************
// 初期化
//******************************************************************************
- (id)initWithNibName:(NSString *)nibNameOrNil bundle:(NSBundle *)nibBundleOrNil
{
	int result = 0;
	NSString* pFileEditViewNibName = nil;
	
	self = [super initWithNibName:nibNameOrNil bundle:nibBundleOrNil];
	if (self) {
		//ビュー設定
		self.title = @"Background Image";
		
		//ユーザ設定初期化
		m_pUserConf = [[YNUserConf alloc] init];
		if (m_pUserConf == nil) {
			result = YN_SET_ERR(@"Program error.", 0, 0);
			goto EXIT;
		}
		
		//背景画像の設定を取得
		[m_pUserConf setCategory:MT_CONF_CATEGORY_GRAPHIC];
		[m_pUserConf setSection:MT_CONF_SECTION_BGIMG];
		m_pImageFileName = [m_pUserConf strValueForKey:@"ImageFileName" defaultValue:@""];
		
		//ファイル一覧生成
		m_pFileArray = nil;
		[self makeFileList];
		
		//選択行番号
		m_SelectedFileIndex = 0;
		
		//ファイル編集ビュー制御の生成
		if ([[UIDevice currentDevice] userInterfaceIdiom] == UIUserInterfaceIdiomPhone) {
			//iPhone / iPod touch
			pFileEditViewNibName = @"MTFileEditView_iPhone";
		}
		else {
			//iPad
			pFileEditViewNibName = @"MTFileEditView_iPad";
		}
		m_pFileEditViewCtrl = [[MTFileEditViewCtrl alloc] initWithNibName:pFileEditViewNibName bundle:nil];
	}
	
EXIT:;
	if (result != 0) YN_SHOW_ERR();
	return self;
}

//******************************************************************************
// ビュー登録完了
//******************************************************************************
- (void)viewDidLoad
{
	NSNotificationCenter* pCenter = nil;
	
    [super viewDidLoad];
	// Do any additional setup after loading the view.
	
	//ナビゲーションバー右側に編集ボタンを登録
	self.navigationItem.rightBarButtonItem = [self editButtonItem];
	
	//通知先登録：ファイル名更新
	pCenter = [NSNotificationCenter defaultCenter];
	[pCenter addObserver:self
				selector:@selector(onUpdateFileName:)
					name:@"MTSettingBGIMGViewCtrl_onUpdateFileName"
				  object:nil];
	
	[m_pFileEditViewCtrl setNotificationNameForUpdateFileName:@"MTSettingBGIMGViewCtrl_onUpdateFileName"];
	
	//リフレッシュ制御生成生成
	m_pRefreshCtrl = [[UIRefreshControl alloc]init];
	[m_pRefreshCtrl addTarget:self action:@selector(refreshTable:) forControlEvents:UIControlEventValueChanged];
	m_pTableView.alwaysBounceVertical = YES;
	
	//テーブルビューにリフレッシュ制御を登録
    if (@available(iOS 10.0, *)) {
        m_pTableView.refreshControl = m_pRefreshCtrl;
    } else {
        [m_pTableView addSubview:m_pRefreshCtrl];
    }
	
	return;
}

//******************************************************************************
// インターフェース自動回転確認
//******************************************************************************
- (BOOL)shouldAutorotateToInterfaceOrientation:(UIInterfaceOrientation)interfaceOrientation
{
    return (interfaceOrientation == UIInterfaceOrientationPortrait);
}

//******************************************************************************
// インターフェース自動回転確認（iOS6以降）
//******************************************************************************
- (BOOL)shouldAutorotate
{
	//回転を許可する
	return YES;
}

//******************************************************************************
// インターフェース自動回転確認：回転方向（iOS6以降）
//******************************************************************************
- (NSUInteger)supportedInterfaceOrientations
{
	//全方向に対応する
	return UIInterfaceOrientationMaskAll;
}

//******************************************************************************
// ビュー表示
//******************************************************************************
- (void)viewWillAppear:(BOOL)animated
{	
	return;
}

//******************************************************************************
// ビュー非表示
//******************************************************************************
- (void)viewWillDisappear:(BOOL)animated
{
	return;
}

//******************************************************************************
// ファイル一覧生成
//******************************************************************************
- (int)makeFileList
{
	int result = 0;
	int i = 0;
	NSArray* pPathList = nil;
	NSString* pDocDirPath = nil;
	NSString* pPath = nil;
	NSString* pFile = nil;
	NSArray* pFileArrayTmp = nil;
	NSMutableArray* pFileArray = nil;
	BOOL isDir = NO;
	
	pFileArray = [[NSMutableArray alloc] init];
	
	//Documentsディレクトリパスを取得
	pPathList = NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES);
	pDocDirPath = [pPathList objectAtIndex:0];
	
	//ディレクトリ配下のファイル一覧を取得
	pFileArrayTmp = [[NSFileManager defaultManager] contentsOfDirectoryAtPath:pDocDirPath error:nil];
	
	//画像ファイル一覧を作成（ディレクトリは除外する）
	for (i = 0; i < [pFileArrayTmp count]; i++) {
		pFile = [pFileArrayTmp objectAtIndex:i];
		pPath = [pDocDirPath stringByAppendingPathComponent:pFile];
		[[NSFileManager defaultManager] fileExistsAtPath:pPath isDirectory:&isDir];
		if (!isDir) {
			if (([[pPath pathExtension] caseInsensitiveCompare:@"jpg"] == NSOrderedSame) ||
			    ([[pPath pathExtension] caseInsensitiveCompare:@"png"] == NSOrderedSame)) {
				//画像ファイル一覧に追加
				[pFileArray addObject:pFile];
			}
		}
	}
	
	//ファイル名ソート：大文字小文字を区別しない
	[m_pFileArray release];
	m_pFileArray = [pFileArray mutableCopy];
	[m_pFileArray sortUsingComparator:^(id str1, id str2) {
		return [((NSString*)str1) compare:((NSString*)str2) options:NSCaseInsensitiveSearch]; }];
	[m_pFileArray retain];
	
	//テスト：ファイルが存在しない場合
	//m_pFileArray = [[NSArray alloc] init];
	
	//ファイルが存在しない場合は選択不可とする
	if ([m_pFileArray count] == 0) {
		m_pTableView.allowsSelection = NO;
	}
	
	[pFileArray release];
	return result;
}

//******************************************************************************
// セクション数
//******************************************************************************
-(NSInteger)numberOfSectionsInTableView:(UITableView*)tableView
{
    return 1;
} 

//******************************************************************************
// セクションヘッダ
//******************************************************************************
- (NSString*)tableView:(UITableView*)tableView titleForHeaderInSection:(NSInteger)section
{
    return nil;
} 

//******************************************************************************
// セクションごとの項目数
//******************************************************************************
- (NSInteger)tableView:(UITableView *)tableView
 numberOfRowsInSection:(NSInteger)section
{
	NSInteger numOfRows = 0;
	
	switch (section) {
		case 0:
			//ファイル一覧
			numOfRows = [m_pFileArray count];
			
			//ファイルなしの場合はメッセージを表示
			if (numOfRows == 0) {
				numOfRows = 1;
			}
			//ファイルありの場合はnoneを項目に追加
			else {
				numOfRows += 1;
			}
			break;
		default:
			break;
	}
	
	return numOfRows;
}

//******************************************************************************
// 項目表示内容
//******************************************************************************
- (UITableViewCell*)tableView:(UITableView*)tableView
		cellForRowAtIndexPath:(NSIndexPath*)indexPath
{
	UITableViewCell* pCell = nil;
	
	switch (indexPath.section) {
		case 0:
			//ファイル一覧
			pCell = [self makeFileCellForIndexPath:indexPath];
			break;
		default:
			break;
	}
	
	return pCell;
}

//******************************************************************************
// テーブルセル作成：ファイル
//******************************************************************************
- (UITableViewCell*)makeFileCellForIndexPath:(NSIndexPath*)indexPath
{
	static NSString* pCellIdentifier = @"MTSettingBGIMGViewCtrl";
	UITableViewCell* pCell = nil;
    bool isCheckmark = false;
	NSString* pFileName = nil;
	
	//再利用可能セル生成
	pCell = [m_pTableView dequeueReusableCellWithIdentifier:pCellIdentifier];
	if (pCell == nil) {
		pCell = [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleSubtitle
									   reuseIdentifier:pCellIdentifier];
	}
	
	//ラベル設定
	if ([m_pFileArray count] == 0) {
		//ファイルなしの場合はメッセージを表示
		pCell.textLabel.text = @"Image file not found.";
		pCell.detailTextLabel.text = @"You can import your image files(*.jpg *.png) through iTunes File Sharing.";
		pCell.detailTextLabel.numberOfLines = 0;
		pCell.accessoryType = UITableViewCellAccessoryNone;
		pCell.editingAccessoryType = UITableViewCellAccessoryNone;
	}
	else {
		//ファイルありの場合
		if (indexPath.row == 0) {
			//先頭にnoneを表示
			pCell.textLabel.text = @"(none)";
			pCell.detailTextLabel.text = @"";
			pCell.editingAccessoryType = UITableViewCellAccessoryNone;
            if ([m_pImageFileName length] == 0) {
                isCheckmark = true;
            }
		}
		else {
			//ファイル名を表示
			pFileName = [m_pFileArray objectAtIndex:(indexPath.row - 1)];
			pCell.textLabel.text = pFileName;
            if ([pFileName isEqualToString:m_pImageFileName]) {
                isCheckmark = true;
            }
			pCell.editingAccessoryType = UITableViewCellAccessoryDisclosureIndicator;
		}
		
		//チェックマーク設定
        pCell.accessoryType = UITableViewCellAccessoryNone;
		if (isCheckmark) {
			pCell.accessoryType = UITableViewCellAccessoryCheckmark;
		}
	}
	
	return pCell;
}

//******************************************************************************
// テーブルセル選択イベント
//******************************************************************************
- (void)tableView:(UITableView*)tableView didSelectRowAtIndexPath:(NSIndexPath*)indexPath
{
	//選択状態解除
	[tableView deselectRowAtIndexPath:indexPath animated:YES];
	
	switch (indexPath.section) {
		case 0:
			//ファイル一覧
			[self onSelectFileCellForIndexPath:indexPath tableView:tableView];
			break;
		default:
			break;
	}
	
	return;
}

//******************************************************************************
// テーブルセル選択イベント：ファイル一覧
//******************************************************************************
- (void)onSelectFileCellForIndexPath:(NSIndexPath*)indexPath tableView:(UITableView *)tableView
{
	NSString* pFileName = nil;
	
	//NSLog(@"selected %d", indexPath.row);
	
	//ファイルなしの場合は何もしない
	if ([m_pFileArray count] == 0) goto EXIT;
	
	//選択行番号を保持
	m_SelectedFileIndex = indexPath.row - 1; //1行目の(none)を外す
	
	//ファイル選択処理
	if (super.editing) {
		//編集モードであればファイル編集ビューを表示
		//ただし先頭行(none)の場合はなにもしない
		if (indexPath.row != 0) {
			pFileName = [m_pFileArray objectAtIndex:m_SelectedFileIndex];
			[m_pFileEditViewCtrl setFileName:pFileName];
			[self.navigationController pushViewController:m_pFileEditViewCtrl animated:YES];
		}
	}
	else {
		//編集中でなければ選択されたファイル名称を保存
		if (indexPath.row == 0) {
			m_pImageFileName = @"";
		}
		else {
			m_pImageFileName = [m_pFileArray objectAtIndex:m_SelectedFileIndex];
		}
		[m_pUserConf setCategory:MT_CONF_CATEGORY_GRAPHIC];
		[m_pUserConf setSection:MT_CONF_SECTION_BGIMG];
		[m_pUserConf setStr:m_pImageFileName forKey:@"ImageFileName"];
		
		//再表示
		[tableView performSelector:@selector(reloadData) withObject:nil afterDelay:0.1f];
		}
	
	//1階層戻る
	//[self.navigationController popViewControllerAnimated:YES];
	
EXIT:;
	return;
}

//******************************************************************************
// 選択行ファイルパス取得
//******************************************************************************
- (NSString*)selectedFilePath
{
	NSArray* pPathList = nil;
	NSString* pDocDirPath = nil;
	NSString* pFileName = nil;
	NSString* pFilePath = nil;
	
	//Documentsディレクトリパスを取得
	pPathList = NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES);
	pDocDirPath = [pPathList objectAtIndex:0];
	
	//選択行のファイル名
	pFileName = [m_pFileArray objectAtIndex:m_SelectedFileIndex];
	
	//ファイルパス
	pFilePath = [NSString stringWithFormat:@"%@/%@", pDocDirPath, pFileName];
	
	return pFilePath;
}

//******************************************************************************
// 編集開始終了通知
//******************************************************************************
- (void)setEditing:(BOOL)editing animated:(BOOL)animated
{
	//編集開始終了処理
	[super setEditing:editing animated:animated];
	[m_pTableView setEditing:editing animated:animated];
	
	return;
}

//******************************************************************************
// 行単位削除可否判定
//******************************************************************************
- (BOOL)tableView:(UITableView *)tableView canEditRowAtIndexPath:(NSIndexPath *)indexPath
{
	BOOL isDeletable = YES;
	
	//ファイルが存在しない場合は削除不可
	if ([m_pFileArray count] == 0) {
		isDeletable = NO;
	}
	
	//1行目の(none)は削除不可
	if (indexPath.row == 0) {
		isDeletable = NO;
	}
	
	return isDeletable;
}

//******************************************************************************
// 行削除イベント
//******************************************************************************
- (void)tableView:(UITableView *)tableView commitEditingStyle:(UITableViewCellEditingStyle)editingStyle forRowAtIndexPath:(NSIndexPath *)indexPath
{
	int result = 0;
	NSString* pFileName = nil;
	NSString* pFilePath = nil;
	
	//ファイルなしの場合は何もしない
	if ([m_pFileArray count] == 0) goto EXIT;
	
	//1行目の(none)が選択された場合は何もしない
	if (indexPath.row == 0) goto EXIT;
	
	//選択行番号を保持
	m_SelectedFileIndex = indexPath.row - 1; //1行目の(none)を外す
	
	//選択行ファイルパス取得
	pFilePath = [self selectedFilePath];
	
	//ファイル削除
	result = [self removeFileWithPath:pFilePath];
	if (result != 0) goto EXIT;
	
	//選択中のWavetableファイルが削除された場合は(none)を選択状態にする
	pFileName = [m_pFileArray objectAtIndex:m_SelectedFileIndex];
	if ([pFileName isEqualToString:m_pImageFileName]) {
		m_pImageFileName = @"";
		[m_pUserConf setCategory:MT_CONF_CATEGORY_GRAPHIC];
		[m_pUserConf setSection:MT_CONF_SECTION_BGIMG];
		[m_pUserConf setStr:m_pImageFileName forKey:@"ImageFileName"];
	}
	
	//ファイル一覧の該当データを削除
	[m_pFileArray removeObjectAtIndex:m_SelectedFileIndex];
	
	//テーブル再読み込み
	[m_pTableView reloadData];
	
EXIT:;
	//if (result != 0) YN_SHOW_ERR();
	return;
}

//******************************************************************************
// ファイル削除
//******************************************************************************
- (int)removeFileWithPath:(NSString*)pPath
{
	int result = 0;
	NSError* pError = nil;
	NSURL* pPathURL = nil;
	BOOL isDeleted = NO;
	UIAlertController* pAlertController = nil;
	UIAlertAction* pAlertAction = nil;
	
	pPathURL = [NSURL fileURLWithPath:pPath];
	isDeleted = [[NSFileManager defaultManager] removeItemAtURL:pPathURL error:&pError];
	if (!isDeleted) {
		//ファイル削除失敗
		result = YN_SET_ERR(@"File deletion failed.", pError.code, 0);
		
		//アラート表示
		pAlertController = [UIAlertController alertControllerWithTitle:@"File deletion failed."
															   message:[pError localizedDescription]
														preferredStyle:UIAlertControllerStyleAlert];
		pAlertAction = [UIAlertAction actionWithTitle:@"OK"
												style:UIAlertActionStyleDefault
											  handler:nil];
		[pAlertController addAction:pAlertAction];
		[self presentViewController:pAlertController animated:YES completion:nil];
		
		goto EXIT;
	}
	
EXIT:;
	return result;
}

//******************************************************************************
// ファイル名更新イベント
//******************************************************************************
- (void)onUpdateFileName:(NSNotification*)pNotification
{
	int result = 0;
	NSString* pNewFileName = nil;
	NSString* pFileName = nil;
	NSString* pSrcPath = nil;
	NSString* pDestPath = nil;
	
	//変更前ファイルパス
	pSrcPath = [self selectedFilePath];
	
	//変更後ファイルパス
	pNewFileName = [m_pFileEditViewCtrl getNewFileName];
	pDestPath = [[pSrcPath stringByDeletingLastPathComponent] stringByAppendingPathComponent:pNewFileName];
	
	//NSLog(@"SrcFilePath: %@", pSrcPath);
	//NSLog(@"DestFilePath: %@", pDestPath);
	
	//ファイルリネーム
	result = [self renameFileWithPath:pSrcPath toPath:pDestPath];
	if (result != 0) goto EXIT;
	
	//選択中のWavetableファイルが更新された場合は更新後ファイル名を選択状態にする
	pFileName = [m_pFileArray objectAtIndex:m_SelectedFileIndex];
	if ([pFileName isEqualToString:m_pImageFileName]) {
		m_pImageFileName = [pNewFileName retain];
		[m_pUserConf setCategory:MT_CONF_CATEGORY_GRAPHIC];
		[m_pUserConf setSection:MT_CONF_SECTION_BGIMG];
		[m_pUserConf setStr:m_pImageFileName forKey:@"ImageFileName"];
	}
	
	//ファイル一覧の該当データを更新
	[m_pFileArray replaceObjectAtIndex:m_SelectedFileIndex withObject:pNewFileName];
	
	//ファイル一覧ソート：大文字小文字を区別しない
	[m_pFileArray sortUsingComparator:^(id str1, id str2) {
		return [((NSString*)str1) compare:((NSString*)str2) options:NSCaseInsensitiveSearch]; }];
	
	//テーブル再読み込み
	[m_pTableView reloadData];
	
EXIT:;
	//if (result != 0) YN_SHOW_ERR();
	return;
}

//******************************************************************************
// ファイルリネーム
//******************************************************************************
- (int)renameFileWithPath:(NSString*)pSrcPath toPath:(NSString*)pDestPath
{
	int result = 0;
	NSError* pError = nil;
	BOOL isMoved = NO;
	UIAlertController* pAlertController = nil;
	UIAlertAction* pAlertAction = nil;
	
	//リネーム実施
	isMoved = [[NSFileManager defaultManager] moveItemAtPath:pSrcPath toPath:pDestPath error:&pError];
	if (!isMoved) {
		//リネーム失敗
		result = YN_SET_ERR(@"File rename failed.", pError.code, 0);
		
		//アラート表示
		pAlertController = [UIAlertController alertControllerWithTitle:@"File rename failed."
															   message:[pError localizedDescription]
														preferredStyle:UIAlertControllerStyleAlert];
		pAlertAction = [UIAlertAction actionWithTitle:@"OK"
												style:UIAlertActionStyleDefault
											  handler:nil];
		[pAlertController addAction:pAlertAction];
		[self presentViewController:pAlertController animated:YES completion:nil];
		goto EXIT;
	}
	
EXIT:;
	return result;
}

//******************************************************************************
// ファイル一覧更新
//******************************************************************************
- (int)updateFileList
{
	int result = 0;
	
	//ファイル一覧生成
	result = [self makeFileList];
	if (result != 0) goto EXIT;
	
	//テーブル再読み込み
	[m_pTableView reloadData];
	
EXIT:;
	return result;
}

//******************************************************************************
// リフレッシュ
//******************************************************************************
-(void)refreshTable:(id)sender
{
	int result = 0;
	
	//ファイル一覧更新
	result = [self updateFileList];
	if (result != 0) goto EXIT;
	
	//リフレッシュ完了
	[m_pRefreshCtrl endRefreshing];
	
EXIT:;
	if (result != 0) YN_SHOW_ERR();
	return;
}

@end

