/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.db;

import java.sql.DatabaseMetaData;
import java.sql.SQLException;
import java.util.Locale;

/**
 * JAXBで自動生成されたDBIDクラスを拡張したクラスです。
 * （継承しているわけではない）
 * <BR>
 * 以下の属性が追加されています。<BR>
 * dbProductName<BR>
 * dbProductVersion<BR>
 * driverName<BR>
 * driverVersion<BR>
 *
 * @og.rev 4.0.0 (2007/10/25) 新規作成
 * @og.rev 5.1.7.0 (2010/06/01) メソッドの修正、Cloneable の追加(浅いコピー)
 *
 * @version  4.0
 * @author	 高橋正和
 * @since	 JDK5.0,
 */
public class EDbid implements Cloneable {

	private 	String	dbidKey			= "DEFAULT";
	private 	String	url				= null;
	private 	String	user			= null;
	private 	String	password		= null;
	private 	boolean	readonly		= false;
	private 	int		mincount		= 3;
	private 	int		maxcount		= 30;
	private 	int		pooltime		= 3000;
	private 	boolean	isUseAppInfo	= true;

	private 	String	dbProductName	= null;
	private 	String	dbProductVersion= null;
	private 	String	driverName		= null;
	private 	String	driverVersion	= null;

	/**
	 * dbidKeyの取得
	 *
	 * 内部的には、大文字のみで管理します。
	 *
	 * @return String dbidKey
	 */
	public String getDbidKey() {
		return dbidKey;
	}

	/**
	 * dbidKeyの設定
	 *
	 * 内部的には、大文字のみで管理します。
	 *
	 * @og.rev 4.1.0.1 (2008/01/21) 登録時に、大文字に変換する。
	 *
	 * @param value String DBID 
	 */
	protected void setDbidKey( final String value ) {
		if( isNotNull( value ) ) { dbidKey = value.toUpperCase( Locale.JAPAN ); }
	}

	/**
	 * urlの取得
	 *
	 * @return String url
	 */
	public String getUrl() {
		return url;
	}

	/**
	 * urlの設定
	 *
	 * @param value String url文字列
	 */
	protected void setUrl( final String value ) {
		if( isNotNull( value ) ) { url = value; }
	}

	/**
	 * userの取得
	 *
	 * @return String user
	 */
	public String getUser() {
		return user;
	}

	/**
	 * userの設定
	 *
	 * @param value String user文字列
	 */
	protected void setUser( final String value ) {
		if( isNotNull( value ) ) { user = value; }
	}

	/**
	 * passwordの取得
	 *
	 * @return String password
	 */
	public String getPassword() {
		return password;
	}

	/**
	 * passwordの設定
	 *
	 * @param value String password文字列
	 */
	protected void setPassword( final String value ) {
		if( isNotNull( value ) ) { password = value; }
	}

	/**
	 * readonlyの取得
	 *
	 * @return boolean readonly
	 */
	public boolean isReadonly() {
		return readonly ;
	}

	/**
	 * readonlyの設定
	 *
	 * @param value String  readonly文字列(true/false)
	 */
//	protected void setReadonly( final Boolean value ) {
//		if( value != null ) { readonly = value.booleanValue(); }
//	}
	protected void setReadonly( final String value ) {
		if( isNotNull( value ) ) { readonly = Boolean.valueOf( value ).booleanValue(); }
	}

	/**
	 * mincountの取得
	 *
	 * @return int mincount
	 */
	public int getMincount() {
		return mincount;
	}

	/**
	 * mincountの設定
	 *
	 * @param value String 最小数(数字タイプ文字列)
	 */
//	protected void setMincount( final Integer value ) {
//		if( value != null ) { mincount = value.intValue(); }
//	}
	protected void setMincount( final String value ) {
		if( isNotNull( value ) ) { mincount = Integer.valueOf( value ).intValue(); }
	}

	/**
	 * maxcountの取得
	 *
	 * @return int maxcount
	 */
	public int getMaxcount() {
		return maxcount;
	}

	/**
	 * maxcountの設定
	 *
	 * @param value String 最大数(数字タイプ文字列)
	 */
//	protected void setMaxcount( final Integer value ) {
//		if( value != null ) { maxcount = value.intValue(); }
//	}
	protected void setMaxcount( final String value ) {
		if( isNotNull( value ) ) { maxcount = Integer.valueOf( value ).intValue(); }
	}

	/**
	 * pooltimeの取得
	 *
	 * @return int pooltime
	 */
	public int getPooltime() {
		return pooltime;
	}

	/**
	 * pooltimeの設定
	 *
	 * @param value String プールに保管して置く時間(秒数)(数字タイプ文字列)
	 */
//	protected void setPooltime( final Integer value ) {
//		if( value != null ) { pooltime = value.intValue(); }
//	}
	protected void setPooltime( final String value ) {
		if( isNotNull( value ) ) { pooltime = Integer.valueOf( value ).intValue(); }
	}

	/**
	 * isUseAppInfoの取得
	 *
	 * @return boolean isUseAppInfo
	 */
	public boolean isApplicationInfo() {
		return isUseAppInfo ;
	}

	/**
	 * isUseAppInfoの設定
	 *
	 * @param value String ApplicationInfoオブジェクトを使用するかどうか(true/false)
	 */
//	protected void setApplicationInfo( final Boolean value ) {
//		if( value != null ) { isUseAppInfo = value.booleanValue(); }
//	}
	protected void setApplicationInfo( final String value ) {
		if( isNotNull( value ) ) { isUseAppInfo = Boolean.valueOf( value ).booleanValue(); }
	}

	/**
	 * メタデータを受け取って
	 * dbProductName,dbProductVersion,driverName,driverVersion
	 * を一括で設定します
	 *
	 * @og.rev 4.0.0.0 (2007/10/30) 保持情報オブジェクト化に伴う変更
	 *
	 * @param meta DatabaseMetaData
	 * @throws SQLException
	 */
	public void setMetaDataInfo( final DatabaseMetaData meta ) throws SQLException {
		dbProductName	= meta.getDatabaseProductName() ;
		dbProductVersion= meta.getDatabaseProductVersion() ;
		driverName		= meta.getDriverName() ;
		driverVersion	= meta.getDriverVersion() ;

		isUseAppInfo	= isUseAppInfo && "ORACLE".equalsIgnoreCase( dbProductName ) ;
	}

	/**
	 * dbProductNameの取得
	 *
	 * @return String dbProductName
	 */
	public String getDbProductName() {
		return dbProductName;
	}

	/**
	 * dbProductVersionの取得
	 *
	 * @return String dbProductVersion
	 */
	public String getDbProductVersion() {
		return dbProductVersion;
	}

	/**
	 * driverNameの取得
	 *
	 * @return String driverName
	 */
	public String getDriverName() {
		return driverName;
	}

	/**
	 * driverVersionの取得
	 *
	 * @return String driverVersion
	 */
	public String getDriverVersion() {
		return driverVersion;
	}

	/**
	 * 文字列がnullでも空文字列でもないかどうかを評価します。
	 *
	 * スペース文字は、trim() してから、length() しますので、false になります。
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) 新規作成
	 *
	 * @param value String 評価する文字列
	 * @return boolean 結果(true:nullでも空文字列でもない）
	 */
	private boolean isNotNull( final String value ) {
		return ( value != null && value.trim().length() > 0 ) ;
	}

	/**
	 * 自分自身の clone を返します。
	 *
	 * Cloneable の実装で、内部的には、Object クラスの clone メソッドを読んでいます。
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) 新規作成
	 *
	 * @return Object 自分自身の clone を返します。
	 */
	public EDbid clone() {
		try {
			return (EDbid)super.clone() ;
		}
		catch( CloneNotSupportedException ex ) {
			String errMsg = "clone が失敗しました。" ;
			throw new RuntimeException( errMsg,ex );
		}
	}

	/**
	 * 簡易な内部情報の文字列化します。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) toString() の簡易版
	 *
	 * @return String 内部情報の文字列
	 */
	public String info() {
		final String CR = System.getProperty( "line.separator" );
		final StringBuilder buf = new StringBuilder();
		buf.append( "DBID=[" ).append( dbidKey ).append( "]," );
		buf.append( "URL =[" ).append( url     ).append( "]," );
		buf.append( "USER=[" ).append( user    ).append( "]," );
		buf.append( CR );
		buf.append( "DB Product=[" ).append( dbProductName );
		buf.append( "(" ).append( dbProductVersion ).append( ")" ).append( "]" );
		buf.append( CR );
		buf.append( "DB Driver =[" ).append( driverName );
		buf.append( "(" ).append( driverVersion ).append( ")" ).append( "]" );
		buf.append( CR );
		return buf.toString();
	}

	/**
	 * 内部情報を文字列化します。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) info() メソッドを利用するように変更
	 *
	 * @return String 内部情報の文字列
	 */
	public String toString() {
		final String CR = System.getProperty( "line.separator" );
		final StringBuilder buf = new StringBuilder();
		buf.append( info() );
//		buf.append( "DBID=[" ).append( dbidKey ).append( "]," );
//		buf.append( "URL =[" ).append( url     ).append( "]," );
//		buf.append( "USER=[" ).append( user    ).append( "]," );
//		buf.append( CR );
//		buf.append( "DB Product=[" ).append( dbProductName );
//		buf.append( "(" ).append( dbProductVersion ).append( ")" ).append( "]" );
//		buf.append( CR );
//		buf.append( "DB Driver =[" ).append( driverName );
//		buf.append( "(" ).append( driverVersion ).append( ")" ).append( "]" );
//		buf.append( CR );
		buf.append( "MAXCOUNT=[" ).append( maxcount ).append( "]," );
		buf.append( "MINCOUNT=[" ).append( mincount ).append( "]," );
		buf.append( "POOLTIME=[" ).append( pooltime ).append( "]," );
		buf.append( "READONLY=[" ).append( readonly ).append( "]," );
		buf.append( "APPINFO=["  ).append( isUseAppInfo ).append( "]," );
		buf.append( CR );
		return buf.toString();
	}
}
