/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.util;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.InputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.util.Collections;
import java.util.List;

import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;

/**
 * FileUtil.java は、共通的に使用される File関連メソッドを集約した、クラスです。
 *
 * 全変数は、public static final 宣言されており、全メソッドは、public static synchronized 宣言されています。
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class FileUtil {
	private static final NonClosePrintWriter outWriter = new NonClosePrintWriter( System.out );
	private static final NonClosePrintWriter errWriter = new NonClosePrintWriter( System.err );

	/**
	 * すべてが staticメソッドなので、コンストラクタを呼び出さなくしておきます。
	 *
	 */
	private FileUtil() {}

	/** システム依存の改行記号をセットします。	*/
	private static final String CR = System.getProperty("line.separator");

	/**
	 * Fileオブジェクトとエンコードより PrintWriterオブジェクトを作成します。
	 *
	 * @param	file	出力するファイルオブジェクト
	 * @param	encode	ファイルのエンコード
	 *
	 * @return	PrintWriterオブジェクト
	 * @throws RuntimeException 何らかのエラーが発生した場合
	 */
	public static PrintWriter getPrintWriter( final File file,final String encode ) {
		return getPrintWriter( file,encode,false );
	}

	/**
	 * Fileオブジェクトとエンコードより PrintWriterオブジェクトを作成します。
	 *
	 * @param	file	出力するファイルオブジェクト
	 * @param	encode	ファイルのエンコード
	 * @param	append	ファイルを追加モード(true)にするかどうか
	 *
	 * @return	PrintWriterオブジェクト
	 * @throws RuntimeException 何らかのエラーが発生した場合
	 */
	public static PrintWriter getPrintWriter( final File file,final String encode,final boolean append ) {
		final PrintWriter writer ;

		try {
			writer = new PrintWriter(new BufferedWriter(new OutputStreamWriter(
					new FileOutputStream(file,append) ,encode )));
		}
		catch( UnsupportedEncodingException ex ) {
			String errMsg = "指定されたエンコーディングがサポートされていません。" + CR
							+ ex.getMessage() + CR
							+ "File=[" + file + " , encode=[" + encode + "]" ;
			throw new RuntimeException( errMsg,ex );
		}
		catch( FileNotFoundException ex ) {		// 3.6.1.0 (2005/01/05)
			String errMsg = "ファイル名がオープン出来ませんでした。" + CR
							+ ex.getMessage() + CR
							+ "File=[" + file + " , encode=[" + encode + "]" ;
			throw new RuntimeException( errMsg,ex );
		}

		return writer ;
	}

	/**
	 * ファイル名より、PrintWriterオブジェクトを作成する簡易メソッドです。
	 *
	 * これは、ファイル名は、フルパスで、追加モードで、UTF-8 エンコードの
	 * ログファイルを出力する場合に使用します。
	 * また、ファイル名に、"System.out" と、"System.err" を指定できます。
	 * その場合は、標準出力、または、標準エラー出力に出力されます。
	 * "System.out" と、"System.err" を指定した場合は、NonClosePrintWriter
	 * オブジェクトが返されます。これは、close() 処理が呼ばれても、何もしない
	 * クラスです。また、常に内部キャッシュの同じオブジェクトが返されます。
	 *
	 * @param	file	出力するファイル名
	 *
	 * @return	PrintWriterオブジェクト
	 * @throws RuntimeException 何らかのエラーが発生した場合
	 * @throws IllegalArgumentException ファイル名が null の場合
	 */
	public static PrintWriter getLogWriter( final String file ) {
		if( file == null ) {
			String errMsg = "ファイル名に、null は指定できません。";
			throw new IllegalArgumentException( errMsg );
		}

		final PrintWriter writer ;
		if( "System.out".equalsIgnoreCase( file ) ) {
			writer = outWriter ;
		}
		else if( "System.err".equalsIgnoreCase( file ) ) {
			writer = errWriter ;
		}
		else {
			writer = getPrintWriter( new File( file ),"UTF-8",true );
		}

		return writer ;
	}

	/**
	 * OutputStreamとエンコードより PrintWriterオブジェクトを作成します。
	 *
	 * @og.rev 5.5.2.0 (2012/05/01) 新規追加
	 *
	 * @param	os		利用するOutputStream
	 * @param	encode	ファイルのエンコード
	 *
	 * @return	PrintWriterオブジェクト
	 * @throws RuntimeException 何らかのエラーが発生した場合
	 */
	public static PrintWriter getPrintWriter( final OutputStream os,final String encode ) {
		final PrintWriter writer ;

		try {
			writer = new PrintWriter(new BufferedWriter(new OutputStreamWriter(
					os ,encode )));
		}
		catch( UnsupportedEncodingException ex ) {
			String errMsg = "指定されたエンコーディングがサポートされていません。" + CR
							+ ex.getMessage() + CR
							+ "encode=[" + encode + "]" ;
			throw new RuntimeException( errMsg,ex );
		}
		return writer ;
	}

	/**
	 * PrintWriter を継承した、JspWriterなどの Writer 用のクラスを定義します。
	 *
	 * 例えば、JspWriterなどの JSP/Servlet等のフレームワークで使用される
	 * Writer では、flush や close 処理は、フレームワーク内で行われます。
	 * その場合、通常のファイルと同じ用に、flush や close をアプリケーション側で
	 * 行うと、内部処理的に不整合が発生したり、最悪の場合エラーになります。
	 * このクラスは、NonFlushPrintWriter クラスのオブジェクトを返します。
	 * これは、通常の、new PrintWriter( Writer ) で、求めるのと、ほとんど同様の
	 * 処理を行いますが、close() と flush() メソッドが呼ばれても、何もしません。
	 *
	 * @param	writer	出力するWriteオブジェクト(NonFlushPrintWriterクラス)
	 *
	 * @return	PrintWriterオブジェクト
	 */
	public static PrintWriter getNonFlushPrintWriter( final Writer writer ) {
		return new NonFlushPrintWriter( writer );
	}

	/**
	 * Fileオブジェクトとエンコードより BufferedReaderオブジェクトを作成します。
	 *
	 * @param	file	入力するファイルオブジェクト
	 * @param	encode	ファイルのエンコード
	 *
	 * @return	BufferedReaderオブジェクト
	 * @throws RuntimeException 何らかのエラーが発生した場合
	 */
	public static BufferedReader getBufferedReader( final File file,final String encode ) {
		final BufferedReader reader ;

		try {
			reader = new BufferedReader(new InputStreamReader(
							new FileInputStream( file ) ,encode ));
		}
		catch( UnsupportedEncodingException ex ) {
			String errMsg = "指定されたエンコーディングがサポートされていません。" + CR
							+ ex.getMessage() + CR
							+ "FIle=[" + file + " , encode=[" + encode + "]" ;
			throw new RuntimeException( errMsg,ex );
		}
		catch( FileNotFoundException ex ) {
			String errMsg = "ファイル名がオープン出来ませんでした。" + CR
							+ ex.getMessage() + CR
							+ "FIle=[" + file + " , encode=[" + encode + "]" ;
			throw new RuntimeException( errMsg,ex );
		}

		return reader ;
	}

	/**
	 * 指定のファイル名が、実際に存在しているかどうかをチェックします。
	 * 存在しない場合は、２秒毎に、３回確認します。
	 * それでも存在しない場合は、エラーを返します。
	 * return されるFileオブジェクトは、正規の形式(CanonicalFile)です。
	 *
	 * @param	dir			フォルダ名
	 * @param	filename	ファイル名
	 *
	 * @return	存在チェック(なければ null/あれば、CanonicalFile)
	 */
	public static File checkFile( final String dir, final String filename ) {
		return checkFile( dir,filename,3 );
	}

	/**
	 * 指定のファイル名が、実際に存在しているかどうかをチェックします。
	 * 存在しない場合は、２秒毎に、指定の回数分確認します。
	 * それでも存在しない場合は、エラーを返します。
	 * return されるFileオブジェクトは、正規の形式(CanonicalFile)です。
	 *
	 * @param	dir			フォルダ名
	 * @param	filename	ファイル名
	 * @param	count		回数指定
	 *
	 * @return	存在チェック(なければ null/あれば、CanonicalFile)
	 */
	public static File checkFile( final String dir, final String filename,final int count ) {
		File file = null;

		int cnt = count;
		while( cnt > 0 ) {
			file = new File( dir,filename );
			if( file.exists() ) { break; }
			else {
				if( cnt == 1 ) { return null; }		// 残り１回の場合は、２秒待機せずに即抜ける。
				try { Thread.sleep( 2000 );	}	// ２秒待機
				catch ( InterruptedException ex ) {
					System.out.println( "InterruptedException" );
				}
				System.out.println();
				System.out.print( "CHECK File Error! CNT=" + cnt );
				System.out.print( " File=" + file.getAbsolutePath() );
			}
			cnt--;
		}

		// ファイルの正式パス名の取得
		try {
			return file.getCanonicalFile() ;
		}
		catch( IOException ex ) {
			String errMsg = "ファイルの正式パス名が取得できません。[" + file.getAbsolutePath() + "]";
			throw new RuntimeException( errMsg,ex );
		}
	}

	/**
	 * ファイルのバイナリコピーを行います。
	 *
	 * copy( File,File,false ) を呼び出します。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 戻り値に、true/false 指定します。
	 *
	 * @param	fromFile	コピー元ファイル名
	 * @param	toFile		コピー先ファイル名
	 *
	 * @return	バイナリコピーが正常に終了したかどうか[true:成功/false:失敗]
	 * @see		#copy( File,File,boolean )
	 */
	public static boolean copy( final String fromFile,final String toFile ) {
		return copy( new File( fromFile ), new File( toFile ), false );
	}

	/**
	 * ファイルのバイナリコピーを行います。
	 *
	 * copy( File,File,boolean ) を呼び出します。
	 * 第３引数の、keepTimeStamp=true で、コピー元のファイルのタイムスタンプを、
	 * コピー先にもセットします。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 戻り値に、true/false 指定します。
	 *
	 * @param	fromFile	コピー元ファイル名
	 * @param	toFile		コピー先ファイル名
	 * @param	keepTimeStamp	タイムスタンプ維持[true/false]
	 *
	 * @return	バイナリコピーが正常に終了したかどうか[true:成功/false:失敗]
	 * @see		#copy( File,File,boolean )
	 */
//	public static boolean copy( final String fromFile,final String toFile,final boolean changeCrLf ) {
	public static boolean copy( final String fromFile,final String toFile,final boolean keepTimeStamp ) {
		return copy( new File( fromFile ), new File( toFile ), keepTimeStamp );
	}

	/**
	 * ファイルのバイナリコピーを行います。
	 *
	 * copy( File,File,false ) を呼び出します。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 戻り値に、true/false 指定します。
	 *
	 * @param	fromFile	コピー元ファイル
	 * @param	toFile		コピー先ファイル
	 *
	 * @return	バイナリコピーが正常に終了したかどうか[true:成功/false:失敗]
	 * @see		#copy( File,File,boolean )
	 */
	public static boolean copy( final File fromFile,final File toFile ) {
		return copy( fromFile, toFile, false );
	}

	/**
	 * ファイルのバイナリコピーを行います。
	 *
	 * 第３引数の、keepTimeStamp=true で、コピー元のファイルのタイムスタンプを、
	 * コピー先にもセットします。
	 * toFile が、ディレクトリの場合は、fromFile のファイル名をそのままコピーします。
	 * fromFile がディレクトリの場合は、エラーにします。
	 * copyDirectry( File,Fileboolean )を使用してください。(自動処理はしていません)
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規追加
	 *
	 * @param	fromFile	コピー元ファイル
	 * @param	toFile		コピー先ファイル
	 * @param	keepTimeStamp タイムスタンプ維持[true/false]
	 *
	 * @return	バイナリコピーが正常に終了したかどうか[true:成功/false:失敗]
	 * @see		#copyDirectry( File,File,boolean )
	 */
	public static boolean copy( final File fromFile,final File toFile,final boolean keepTimeStamp ) {
		FileInputStream  inFile  = null;
		FileOutputStream outFile = null;
		FileChannel  fin  = null;
		FileChannel  fout = null;

		File tempToFile = toFile ;
		try {
			// fromFileが、ディレクトリの場合は、エラー
			if( fromFile.isDirectory() ) {
				System.err.println( fromFile + " がディレクトリのため、処理できません。" );
				return false;
			}
			// toFileが、ディレクトリの場合は、そのパスでファイル名をfromFileから取り出す。
			if( toFile.isDirectory() ) {
				tempToFile = new File( toFile,fromFile.getName() );
			}
			inFile  = new FileInputStream( fromFile );
			outFile = new FileOutputStream( tempToFile );

			fin  = inFile.getChannel();
			fout = outFile.getChannel();

			ByteBuffer buffer = ByteBuffer.allocateDirect( BUFSIZE );
			while ( (fin.read(buffer) != -1) || buffer.position() > 0) {
				buffer.flip();
				fout.write( buffer );
				buffer.compact();
			}
		}
		catch ( IOException ex ) {
			System.out.println(ex.getMessage());
			return false;
		}
		finally {
			Closer.ioClose( inFile  ) ;
			Closer.ioClose( outFile );
			Closer.ioClose( fin  ) ;
			Closer.ioClose( fout );
		}

		if( keepTimeStamp ) {
			return tempToFile.setLastModified( fromFile.lastModified() );
		}

		return true;
	}
//	public static boolean copy( final File fromFile,final File toFile,final boolean keepTimeStamp ) {
//		BufferedInputStream 	fromStream = null;
//		BufferedOutputStream	toStream   = null;
//		File tempToFile = toFile ;
//		try {
//			// fromFileが、ディレクトリの場合は、エラー
//			if( fromFile.isDirectory() ) {
//				System.err.println( fromFile + " がディレクトリのため、処理できません。" );
//				return false;
//			}
//			// toFileが、ディレクトリの場合は、そのパスでファイル名をfromFileから取り出す。
//			if( toFile.isDirectory() ) {
//				tempToFile = new File( toFile,fromFile.getName() );
//			}
//
//			fromStream = new BufferedInputStream( new FileInputStream( fromFile ) );
//			toStream   = new BufferedOutputStream( new FileOutputStream( tempToFile ) );
//
//			boolean isOK = copy( fromStream,toStream );
//			if( !isOK ) { return false; }
//
//		}
//		catch ( IOException ex ) {
//			System.out.println(ex.getMessage());
//			return false;
//		}
//		finally {
//			Closer.ioClose( fromStream ) ;
//			Closer.ioClose( toStream ) ;
//		}
//
//		if( keepTimeStamp ) {
//			tempToFile.setLastModified( fromFile.lastModified() );
//		}
//
//		return true;
//	}

	private static final byte B_CR = (byte)0x0d ;	// '\r'
	private static final byte B_LF = (byte)0x0a ;	// '\n'
	private static final int  BUFSIZE = 8192 ;		// 5.1.6.0 (2010/05/01)

	/**
	 * ファイルのバイナリコピーを行います。
	 *
	 * このファイルコピーは、バイナリファイルの 改行コードを
	 * CR+LF に統一します。また、UTF-8 の BOM(0xef,0xbb,0xbf) があれば、
	 * 取り除きます。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規追加
	 *
	 * @param	fromFile	コピー元ファイル
	 * @param	toFile		コピー先ファイル
	 *
	 * @return	バイナリコピーが正常に終了したかどうか[true:成功/false:失敗]
	 */
	public static boolean changeCrLfcopy( final File fromFile,final File toFile ) {
		BufferedInputStream 	fromStream = null;
		BufferedOutputStream	toStream   = null;
		File tempToFile = toFile ;
		try {
			// ディレクトリの場合は、そのパスでファイル名をfromFileから取り出す。
			if( toFile.isDirectory() ) {
				tempToFile = new File( toFile,fromFile.getName() );
			}
			fromStream = new BufferedInputStream( new FileInputStream( fromFile ) );
			toStream   = new BufferedOutputStream( new FileOutputStream( tempToFile ) );

//			int BUFSIZE = 8192 ;		// 5.1.6.0 (2010/05/01) static final定義
			byte[] buf = new byte[BUFSIZE];
			int len ;
			// 4.2.3.0 (2008/05/26) changeCrLf 属性対応

			boolean bomCheck = true;	// 最初の一回だけ、ＢＯＭチェックを行う。
			byte    bt = (byte)0x00;	// バッファの最後と最初の比較時に使用
			while( (len = fromStream.read(buf,0,BUFSIZE)) != -1 ) {
				int st = 0;
				if( bomCheck && len >= 3 &&
					buf[0] == (byte)0xef &&
					buf[1] == (byte)0xbb &&
					buf[2] == (byte)0xbf  ) {
						st = 3;
				}
				else {
					// バッファの最後が CR で、先頭が LF の場合、LF をパスします。
					if( bt == B_CR && buf[0] == B_LF ) {
						st = 1 ;
					}
				}
				bomCheck = false;

				for( int i=st;i<len;i++ ) {
					bt = buf[i] ;
					if( bt == B_CR || bt == B_LF ) {
						toStream.write( (int)B_CR );		// CR
						toStream.write( (int)B_LF );		// LF
						// CR+LF の場合
						if( bt == B_CR && i+1 < len && buf[i+1] == B_LF ) {
							i++;
							bt = buf[i] ;
						}
					}
					else {
						toStream.write( (int)bt );
					}
				}
			}
			// 最後が改行コードでなければ、改行コードを追加します。
			// テキストコピーとの互換性のため
			if( bt != B_CR && bt != B_LF ) {
				toStream.write( (int)B_CR );		// CR
				toStream.write( (int)B_LF );		// LF
			}
		}
		catch ( IOException ex ) {
			System.out.println(ex.getMessage());
			return false;
		}
		finally {
			Closer.ioClose( fromStream ) ;
			Closer.ioClose( toStream ) ;
		}

		return true;
	}

	/**
	 * 入出力ストリーム間でデータの転送を行います。
	 *
	 * ここでは、すでに作成されたストリームに基づき、データの入出力を行います。
	 * よって、先にフォルダ作成や、存在チェック、ファイルの削除などの必要な処理は
	 * 済まして置いてください。
	 * また、このメソッド内で、ストリームのクロース処理は行っていません。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規追加
	 *
	 * @param	input	入力ストリーム
	 * @param	output	出力ストリーム
	 *
	 * @return	データ転送が正常に終了したかどうか[true:成功/false:失敗]
	 */
	public static boolean copy( final InputStream input,final OutputStream output ) {
		if( input == null ) {
			System.err.println( "入力ストリームが 作成されていません。" );
			return false;
		}

		if( output == null ) {
			System.err.println( "出力ストリームが 作成されていません。" );
			return false;
		}

		try {
			byte[] buf = new byte[BUFSIZE];
			int len;
			while((len = input.read(buf)) != -1) {
				output.write(buf, 0, len);
			}
		}
		catch ( IOException ex ) {
			System.out.println( ex.getMessage() );
			return false;
		}
	//	finally {
	//		Closer.ioClose( input );
	//		Closer.ioClose( output );
	//	}
		return true ;
	}

	/**
	 * 再帰処理でディレクトリのコピーを行います。
	 *
	 * 指定されたコピー元ディレクトリがディレクトリでなかったり存在しないときは	falseを返します。
	 *
	 * @og.rev 4.3.0.0 (2008/07/24) 追加
	 * @og.rev 5.1.6.0 (2010/05/01) 戻り値に、true/false 指定します。
	 *
	 * @param	fromDir	コピー元ディレクトリ名
	 * @param	toDir	コピー先ディレクトリ名
	 *
	 * @return	ディレクトリのコピーが正常に終了したかどうか[true:成功/false:失敗]
	 */
	public static boolean copyDirectry( final String fromDir, final String toDir ) {
		return copyDirectry( new File( fromDir ), new File( toDir ),false );
	}

	/**
	 * 再帰処理でディレクトリをコピーします。
	 *
	 * 指定されたコピー元ディレクトリがディレクトリでなかったり存在しないときは	falseを返します。
	 *
	 * @og.rev 4.3.0.0 (2008/07/24) 追加
	 * @og.rev 5.1.6.0 (2010/05/01) 内部処理を若干変更します。
	 *
	 * @param	fromDir   コピー元ディレクトリ
	 * @param	toDir     コピー先ディレクトリ
	 *
	 * @return	ディレクトリのコピーが正常に終了したかどうか[true:成功/false:失敗]
	 */
	public static boolean copyDirectry( final File fromDir, final File toDir ) {
		return copyDirectry( fromDir, toDir, false );
	}

	/**
	 * 再帰処理でディレクトリをコピーします。
	 *
	 * 指定されたコピー元ディレクトリがディレクトリでなかったり存在しないときは	falseを返します。
	 *
	 * @og.rev 4.3.0.0 (2008/07/24) 追加
	 * @og.rev 5.1.6.0 (2010/05/01) 内部処理を若干変更します。
	 * @og.rev 5.3.7.0 (2011/07/01) フォルダにアクセスできない場合は、エラーを返します。
	 *
	 * @param	fromDir	コピー元ディレクトリ
	 * @param	toDir	コピー先ディレクトリ
	 * @param	keepTimeStamp タイムスタンプ維持[true/false]
	 *
	 * @return	ディレクトリのコピーが正常に終了したかどうか[true:成功/false:失敗]
	 */
	public static boolean copyDirectry( final File fromDir, final File toDir, final boolean keepTimeStamp ) {
		// コピー元がディレクトリでない場合はfalseを返す
		// 4.3.4.4 (2009/01/01)
		if( !fromDir.exists() || !fromDir.isDirectory() ) {
			System.err.println( fromDir + " が ディレクトリでないか、存在しません。" );
			return false;
		}

		// 4.3.4.4 (2009/01/01)
		if( !toDir.exists() ) {
			// ディレクトリを作成する
			if( !toDir.mkdirs() ) {
				System.err.println( toDir + " の ディレクトリ作成に失敗しました。" );
				return false;
			}
		}

		// ディレクトリ内のファイルをすべて取得する
		File[] files = fromDir.listFiles();

		// 5.3.7.0 (2011/07/01) フォルダにアクセスできない場合は、エラー
		if( files == null ) {
			System.err.println( fromDir + " はアクセスできません。" );
			return false;
		}

		// ディレクトリ内のファイルに対しコピー処理を行う
		boolean flag = true;
		for( int i = 0; files.length>i; i++ ){
			if( files[i].isDirectory() ){ // ディレクトリだった場合は再帰呼び出しを行う
				flag = copyDirectry( files[i], new File( toDir, files[i].getName()),keepTimeStamp );
			}
			else{ // ファイルだった場合はファイルコピー処理を行う
				flag = copy( files[i], new File( toDir, files[i].getName()),keepTimeStamp );
			}
			if( !flag ) { return false; }
		}
		return true;
	}
//	public static boolean copyDirectry( final File fromDirectry, final File toDirectry ) {
//		// コピー元がディレクトリでない場合はfalseを返す
//		// 4.3.4.4 (2009/01/01)
//		if( !fromDirectry.exists() || !fromDirectry.isDirectory() ) { return false; }
//
//		// 4.3.4.4 (2009/01/01)
//		boolean flag = true;
//		if( !toDirectry.exists() ) {
//			// ディレクトリを作成する
//			flag = toDirectry.mkdirs();
//			if( ! flag ) { System.err.println( toDirectry.getName() + " の ディレクトリ作成に失敗しました。" ); }
//		}
//
//		// ディレクトリ内のファイルをすべて取得する
//		File[] files = fromDirectry.listFiles();
//
//		// ディレクトリ内のファイルに対しコピー処理を行う
//		for( int i = 0; files.length>i; i++ ){
//			if( files[i].isDirectory() ){ // ディレクトリだった場合は再帰呼び出しを行う
//				copyDirectry(
//				new File( fromDirectry.toString(), files[i].getName() ),
//				new File( toDirectry.toString(), files[i].getName()));
//			}
//			else{ // ファイルだった場合はファイルコピー処理を行う
//				copy(
//				new File( fromDirectry.toString(), files[i].getName() ),
//				new File( toDirectry.toString(), files[i].getName()) );
//			}
//		}
//		return true;
//	}

	/**
	 * 指定されたファイル及びディレクトを削除します。
	 * ディレクトリの場合はサブフォルダ及びファイルも削除します。
	 * １つでもファイルの削除に失敗した場合、その時点で処理を中断しfalseを返します。
	 *
	 * @og.rev 5.3.7.0 (2011/07/01) フォルダにアクセスできない場合は、エラーを返します。
	 *
	 * @param	file 削除ファイル/ディレクトリ
	 *
	 * @return	ファイル/ディレクトリの削除に終了したかどうか[true:成功/false:失敗]
	 */
	public static boolean deleteFiles( final File file ) {
		if( file.exists() ) {
			if( file.isDirectory() ) {
				File[] list = file.listFiles();

				// 5.3.7.0 (2011/07/01) フォルダにアクセスできない場合は、エラー
				if( list == null ) {
					System.err.println( file + " はアクセスできません。" );
					return false;
				}

				for( int i=0; i<list.length; i++ ) {
					deleteFiles( list[i] );
				}
			}
			if( !file.delete() ) { return false; }
		}
		return true;
	}

	/**
	 * 指定されたディレクトリを基点としたファイル名(パスを含む)の一覧を返します。
	 *
	 * @og.rev 4.3.6.6 (2009/05/15) 新規作成
	 * @og.rev 5.4.3.2 (2012/01/06) 引数isCopy追加
	 *
	 * @param dir 基点となるディレクトリ
	 * @param sort ファイル名でソートするか
	 * @param list ファイル名一覧を格納するList
	 * @param isCopy コピー中ファイルを除外するか [true:含む/false:除外]
	 */
	public static void getFileList( final File dir, final boolean sort, final List<String> list, boolean isCopy ) {
		if( list == null ) { return; }
		if( dir.isFile() ) {
			// コピー中判定はrenameで行う
			if( !isCopy && !dir.renameTo( dir ) ){
				return;
			}
			else{
				list.add( dir.getAbsolutePath() );
			}
		}
		else if( dir.isDirectory() ) {
			File[] files = dir.listFiles();
			for( int i=0; i<files.length; i++ ) {
				getFileList( files[i], sort, list, isCopy );
			}
		}
		if( sort ) {
			Collections.sort( list );
		}
	}

	/**
	 * 指定されたディレクトリを基点としたファイル名(パスを含む)の一覧を返します。
	 * 互換性のため、コピー中ファイルも含みます。
	 *
	 * @og.rev 5.4.3.2 (2012/01/06) コピー中対応のため引数４つを作成する
	 *
	 * @param dir 基点となるディレクトリ
	 * @param sort ファイル名でソートするか
	 * @param list ファイル名一覧を格納するList
	 */
	public static void getFileList( final File dir, final boolean sort, final List<String> list ) {
			getFileList( dir, sort, list, true );
	}

	/**
	 * PrintWriter を継承した、System.out/System.err 用のクラスを定義します。
	 *
	 * 通常の、new PrintWriter( OutputStream ) で、求めるのと、ほとんど同様の
	 * 処理を行います。
	 * ただ、close() メソッドが呼ばれても、何もしません。
	 *
	 */
	private static final class NonClosePrintWriter extends PrintWriter {
		/**
		 * コンストラクター
		 *
		 * new PrintWriter( OutputStream ) を行います。
		 *
		 * @param out OutputStream
		 */
		public NonClosePrintWriter( final OutputStream out ) {
			super( out );
		}

		/**
		 * close() メソッドをオーバーライドします。
		 *
		 * 何もしません。
		 *
		 */
		public void close() {
			// ここでは処理を行いません。
		}
	}

	/**
	 * PrintWriter を継承した、JspWriterなどの Writer 用のクラスを定義します。
	 *
	 * 例えば、JspWriterなどの JSP/Servlet等のフレームワークで使用される
	 * Writer では、flush や close 処理は、フレームワーク内で行われます。
	 * その場合、通常のファイルと同じ用に、flush や close をアプリケーション側で
	 * 行うと、内部処理的に不整合が発生したり、最悪の場合エラーになります。
	 * このクラスは、単に、通常の、new PrintWriter( Writer ) で、求めるのと、
	 * ほとんど同様の処理を行います。
	 * ただ、close() と flush() メソッドが呼ばれても、何もしません。
	 *
	 */
	private static final class NonFlushPrintWriter extends PrintWriter {
		/**
		 * コンストラクター
		 *
		 * new PrintWriter( Writer ) を行います。
		 *
		 * @param writer Writer
		 */
		public NonFlushPrintWriter( final Writer writer ) {
			super( writer );
		}

		/**
		 * close() メソッドをオーバーライドします。
		 *
		 * 何もしません。
		 *
		 */
		public void close() {
			// ここでは処理を行いません。
		}

		/**
		 * flush() メソッドをオーバーライドします。
		 *
		 * 何もしません。
		 *
		 */
		public void flush() {
			// ここでは処理を行いません。
		}
	}

	/**
	 * ファイルをコピーします。
	 *
	 * 引数に <file1> <file2> [<encode1> <encode2>] を指定します。
	 * file1 を読み込み、file2 にコピーします。コピー前に、file2 は、file2_backup にコピーします。
	 * file1 が、ディレクトリの場合は、ディレクトリごとコピーします。
	 * encode1、encode2 を指定すると、エンコード変換しながらコピーになります。
	 * この場合は、ファイル同士のコピーのみになります。
	 *
	 * @og.rev 4.0.0.0 (2007/11/28) メソッドの戻り値をチェックします。
	 * @og.rev 5.1.6.0 (2010/05/01) 引数の並び順、処理を変更します。
	 *
	 * @param	args 引数配列  file1 file2 [encode1 encode2]
	 */
	public static void main( final String[] args ) throws Throwable {
		if( args.length != 2 && args.length != 4 ) {
			LogWriter.log("Usage: java FileUtil <file1> <file2> [<encode1> <encode2>]" );
			return ;
		}

		File file1 = new File( args[0] );
		File file2 = new File( args[1] );

		File tempFile = new File( args[1] + "_backup" );

		if( args.length < 3 ) {
			if( file1.isDirectory() ) {
				FileUtil.copyDirectry( file1, file2, true );
			}
			else {
				FileUtil.copy( file2,tempFile );
				FileUtil.copy( file1,file2, true );
			}
		}
		else {
			String encode1 = args[2];
			String encode2 = args[3];

			FileUtil.copy( file2,tempFile );

			BufferedReader reader = FileUtil.getBufferedReader( file1 ,encode1 );
			PrintWriter    writer = FileUtil.getPrintWriter(    file2 ,encode2 );

			try {
				String line1;
				while((line1 = reader.readLine()) != null) {
					writer.println( line1 );
				}
			}
			finally {
				Closer.ioClose( reader ) ;
				Closer.ioClose( writer ) ;
			}
		}
	}
}
