/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.resource;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.opengion.fukurou.db.DBSimpleTable;
import org.opengion.fukurou.db.DBUtil;
import org.opengion.fukurou.util.ApplicationInfo;
import org.opengion.fukurou.util.Cleanable;
import org.opengion.fukurou.util.HybsEntry;
import org.opengion.fukurou.util.LogWriter;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.common.UserSummary;
import org.opengion.hayabusa.db.DBEditConfig;
import org.opengion.hayabusa.db.DBEditConfigManager;

/**
 * ユーザー情報の取得の為のインターフェースを実装したクラスです。
 *
 * ログイン時のパスワードのチェックや,国名の識別ID，ポータルページのURLなど
 * 個人情報を管理させます。
 * 特に,画面アクセス時の権限やメールの送信，各画面にユーザー情報を表示したり,
 * エラー時のログファイル，テンポラリディレクトリなども管理します。
 *
 * {&#064;USER.XXXX} で、XXXX 部に、UserInfo オブジェクトで定義されている
 * 属性情報を取り出すことが出来ます。
 *
 * 以下の値は UserInfo オブジェクトの項目から取得します。
 * ・JNAME      ユーザー日本語名称
 * ・ID         ユーザーＩＤ
 * ・INFO       ユーザー情報(ユーザーID：日本語名称)
 * ・LANG       言語
 * ・ROLES      ロール
 * ・IPADDRESS  IPアドレス
 * ・LOGINTIME  ログイン時刻
 * ・LASTACCESS 最終アクセス画面ID
 *
 * 以下の値はあらかじめ、動的に作成されます。
 * ・YMD       ８byte の今日のシステム日付
 * ・YMDH    １４byte の今日のシステム日時
 *
 * それ以外は,外部より設定された値です。
 * 従来は、USER.IDNO はUSER.ID が５Byte以上の時のみ先頭１文字を除いたユーザーＩＤとして
 * オブジェクト項目からの取得でしたが、現在は初期値として設定してあります。
 * 外部より再設定可能になっています。
 *
 * @og.group リソース管理
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class UserInfo implements UserSummary , Cleanable {
	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/06/30)

	// ユーザーリソースのキー指定読み込みのクエリー
	private static final String QUERY_PARAM = HybsSystem.sys( "USER_PARAMETER_SQL" );

	// 統計テーブル(GE15)への接続先を、リソースの接続先より取得します。
	private final String DBID = HybsSystem.sys( "RESOURCE_DBID" );

	//private static final String YOYAKU = "|JNAME|ID|INFO|LANG|ROLES|IPADDRESS|LOGINTIME|LASTACCESS|YMD|YMDH|" ;
	private static final String YOYAKU = "|JNAME|ID|INFO|LANG|ROLES|IPADDRESS|LOGINTIME|LASTACCESS|YMD|YMDH|LASTGAMENNM" ; // 4.4.0.1 (2009/08/08)

	private final boolean useAccessTable = HybsSystem.sysBool( "USE_ACCESS_TOKEI_TABLE" );
//	private final String  accessPastDays = HybsSystem.sys( "ACCESS_TOKEI_PAST_DAYS" ); // 4.1.0.1 (2008/01/31)

	private final String	userID	;
//	private final String	lang 	;
	private       String	lang 	;	// 5.1.4.0 (2010/03/01) lang を書き換え可能とする。
	private final String	jname	;
	private final String	roles	;
	private final String	droles	; // 4.4.0.0 (2009/08/02) データロール対応
//	private final String[]	userRoles	;
	private final String	ipAddress	;
	private final long		loginTime	;
	private final Map<String,String>	attribute  ;
	private final RoleMode	roleMode ;		// ロールズとモードを管理するオブジェクト
	private final DataRole	dataRole ;		// データロールを管理するオブジェクト

//	private final boolean	isRootRole	;				// 4.3.0.0 (2008/07/04) 廃止
	private final int		hashcode	;				// 3.5.6.0 (2004/06/18)
	private Map<String,GUIInfo>	guiMap	;				// 4.0.0 (2005/01/31)
	private 	  boolean	isInfoSet	= false;			// 4.0.0 (2005/01/31)
	private 	  long		usedTime 	= 0L;				// 4.0.0 (2005/01/31)
	private final Map<String,String>	paramMap = new HashMap<String,String>();
	private final Object	guiLock		= new Object();
	private final String	systemId	;
	private Map<String,FavoriteGUIData> favoriteGuiMap = null;	// 4.1.1.0 (2008/01/22)
	private Set<String> forbidAddrSet 	= null;			// 5.2.0.0 (2010/09/01)
	private final DBEditConfigManager editMgr = new DBEditConfigManager(); // 5.3.6.0 (2011/06/01)

	/** コネクションにアプリケーション情報を追記するかどうか指定 */
	public static final boolean USE_DB_APPLICATION_INFO  = HybsSystem.sysBool( "USE_DB_APPLICATION_INFO" ) ;

	// 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	private final ApplicationInfo appInfo;

	// GE20 ユーザーロール情報永続化テーブル 読み込み用SQL
	// 4.3.4.0 (2008/12/01) ROLE='*'も検索できるようにする
	// 5.3.6.0 (2011/06/01) USERID='*'も検索できるようにする
	private static final String QUERY_GE20		= "select PARAM_ID,PARAM from GE20 where SYSTEM_ID = ? and USERID in ( ?, '*' ) and ROLES in ( ?, '*' ) and FGJ = '1' order by USERID,ROLES";

	// 4.3.4.0 (2008/12/01) GE20にセーブ時に存在チェックを行うためのSQL
	// 4.3.4.4 (2009/01/01) private static を付加
	private static final String QUERY_GE20_KEY	= "select KBSET from GE20 where SYSTEM_ID = ? and USERID = ? and ROLES = ? and PARAM_ID = ? and FGJ = '1'";

	// 5.2.3.0 (2010/12/01) アクセス履歴管理
	private GUIInfo lastGuiInfo = null;

	/**
	 * コンストラクター
	 *
	 * @og.rev 3.0.0.1 (2003/02/14) ユーザー毎のエンコード指定方法を廃止します。
	 * @og.rev 3.1.3.0 (2003/04/10) ユーザー情報から、エンコード情報を削除する。
	 * @og.rev 3.4.0.3 (2003/09/10) "root","manager","admin" のロールを、すべて root 権限を与える。
	 * @og.rev 3.8.5.3 (2006/06/30) USE_USER_IDNO_C_SAPLESS を判定条件に加える。
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.3.0.0 (2008/07/04) ロールモードマルチ対応
	 * @og.rev 4.4.0.0 (2009/08/02) データロール対応
	 * @og.rev 5.3.6.0 (2011/06/01) GE20の読み込みをUserInfoFactoryから移動
	 *
	 * @param	userID		ユーザー
	 * @param	lang		言語
	 * @param	jname		日本語名称
	 * @param	roles		ロール
	 * @param	droles		データロール
	 * @param	systemId	システムID
	 * @param	ipAddress	IPアドレス
	 * @param   appInfo		アプリ情報オブジェクト
	 */
	public UserInfo( final String userID		,
					 final String lang			,
					 final String jname			,
					 final String roles			,
					 final String droles		, // 4.4.0.0 (2009/08/02)
					 final String systemId		,
					 final String ipAddress		,
					 final ApplicationInfo appInfo  ) {
		this.userID	 	= userID	;
		this.lang	 	= lang		;
		this.jname	 	= jname		;
		this.roles		= roles		;
		this.droles		= droles	; // 4.4.0.0 (2009/08/02)
		this.systemId	= systemId	;
//		userRoles		= StringUtil.csv2Array( roles,HybsSystem.GUI_DELIMITER );
		this.roleMode	= RoleMode.newInstance( roles );	// 4.3.0.0 (2008/07/04) ロールモード
		this.dataRole	= DataRole.newInstance( droles, systemId, appInfo ); // 4.4.0.0 (2009/08/02)
		this.ipAddress	= ipAddress	;
		this.appInfo	= appInfo	;
		loginTime		= System.currentTimeMillis();
		usedTime		= loginTime;
		attribute		= new HashMap<String,String>();

		// 3.4.0.3 (2003/09/10) "root","manager","admin" のロールを、すべて root 権限を与える。
		// 4.3.0.0 (2008/07/04) 廃止
//		isRootRole = "root".equals( roles ) || "manager".equals( roles ) || "admin".equals( roles ) ;

		// 3.5.6.0 (2004/06/18) hashCode を計算しておきます。
		hashcode = (int)(loginTime^(loginTime>>>32)) ;

	 	// 3.8.1.2 (2005/12/19) USER.IDNO をAttributeにセットする。
	 	// 3.8.5.3 (2006/06/30) USE_USER_IDNO_C_SAPLESS を判定条件に加える。
		boolean IDNO_C_SAPLESS = HybsSystem.sysBool( "USE_USER_IDNO_C_SAPLESS" );
		String idno = ( userID.length() > 5 && IDNO_C_SAPLESS ) ? userID.substring(1) : userID ;
		attribute.put( "IDNO",idno ) ;

		// ユーザーパラメータなどの初期設定を行います。
		initLoad() ;

		// 5.3.6.0 (2011/06/01) GE20からDBに保存されたUserInfo情報を読み出します。
		dbLoad();

		// 5.3.6.0 (2011/06/01) Edit情報の一覧を作成します。
		makeEditConfigMap();
	}

	/**
	 * ユーザーパラメータを取得します。
	 * ユーザーパラメーターは、通常、GE16 テーブルより取得します。
	 * 取得するSQL文は、SystemData の USER_PARAMETER_SQL に記述しておきます。
	 * ユーザーパラメータに、値が存在しない場合は、システムリソースより
	 * 取得します。
	 *
	 * @param	key	パラメータキー
	 *
	 * @return	パラメータ値(ユーザーパラメータになければ、システムリソースより取得
	 */
	public String getParameter( final String key ) {
		String val = null;
		if( key != null ) {
			synchronized( paramMap ) {
				val = paramMap.get( key );
			}
			if( val == null ) { val = HybsSystem.sys( key ); }
		}
		return val;
	}

	/**
	 * ユーザーログイン時刻を取得します。
	 *
	 * @return	ログイン時刻
	 */
	public long getLoginTime() {
		return loginTime;
	}

	/**
	 * ユーザーのログインIPアドレスを取得します。
	 *
	 * @return	IPアドレス
	 *
	 */
	public String getIPAddress() {
		return ipAddress;
	}

	/**
	 * ユーザーを取得します。
	 *
	 * @return	ユーザー
	 *
	 */
	public String getUserID() {
		return userID;
	}

	/**
	 * ユーザー情報ロケール(言語)を取得します。
	 *
	 * @return	ロケール(言語)
	 */
	public String getLang() {
		return lang ;
	}

	/**
	 * ユーザー情報ロケール(言語)をセットします。
	 *
	 * @og.rev 5.1.4.0 (2010/03/01) lang を書き換え可能とする。
	 *
	 * @param newLang	ロケール(言語)
	 */
	public void setLang( final String newLang ) {
		lang = newLang ;
	}

	/**
	 * ユーザー情報 名称(日本語)を取得します。
	 *
	 * @return	名称(日本語)
	 */
	public String getJname() {
		return jname ;
	}

	/**
	 * ユーザー情報 ロール(役割)を取得します。
	 *
	 * @return	ロール(役割)
	 */
	public String getRoles() {
		return roles ;
	}

	/**
	 * ロールモード情報を取得します。
	 *
	 * @og.rev 4.3.0.0 (2008/07/04) 新規追加
	 *
	 * @return	ロールモード情報
	 */
	public RoleMode getRoleMode() {
		return roleMode ;
	}

	/**
	 * オブジェクトの識別子として，ユーザー情報を返します。
	 *
	 * @return  ユーザー情報
	 */
	public String getInfo() {
		StringBuilder rtn = new StringBuilder( HybsSystem.BUFFER_MIDDLE );
		rtn.append( userID ).append( " ： " );
		rtn.append( jname  ).append( HybsSystem.CR );
		return rtn.toString();
	}

	/**
	 * UserInfoの属性文字列を登録します。
	 *
	 * 予約文字(JNAME,ID,INFO,LANG,ROLES,IPADDRESS,LOGINTIME,LASTACCESS,YMD,YMDH,LASTGAMENNM)を
	 * 登録しようとした場合は、エラーにします。
	 *
	 * @param	key  	キー
	 * @param	value	値
	 * @param	save 	GE20(ユーザー定数)に情報を保存するか
	 * @throws HybsSystemException
	 */
	public void setAttribute( final String key,final String value, final boolean save ) {
		setAttribute( key, value, save, false );
	}

	/**
	 * UserInfoの属性文字列を登録します。
	 *
	 * 予約文字(JNAME,ID,INFO,LANG,ROLES,IPADDRESS,LOGINTIME,LASTACCESS,YMD,YMDH,LASTGAMENNM)を
	 * 登録しようとした場合は、エラーにします。
	 *
	 * @og.rev 3.5.6.0 (2004/06/18) synchronized をattribute に行います。
	 * @og.rev 4.3.4.0 (2008/12/01) GE20(ユーザー定数)へ登録するかのフラグを追加
	 * @og.rev 5.3.6.0 (2011/06/01) GE20(ユーザー定数)へ登録時に全ユーザー公開するかのフラグを追加
	 *
	 * @param	key  	キー
	 * @param	value	値
	 * @param	save 	GE20(ユーザー定数)に情報を保存するか
	 * @param	common 	GE20(ユーザー定数)に保存した情報を全ユーザー公開するか
	 * @throws HybsSystemException
	 */
//	public void setAttribute( final String key,final String value ) {
//	public void setAttribute( final String key,final String value, final boolean save ) {
	private void setAttribute( final String key,final String value, final boolean save, final boolean common ) {
		if( key != null && YOYAKU.indexOf( "|" + key + "|" ) < 0 ) {
			synchronized( attribute ) {
				attribute.put( key,value ) ;
			}

			// 4.3.4.0 (2008/12/01) 追加
			if( save ) {
				savePermanently( key ,value, common );
			}
		}
		else {
			String errMsg = "属性文字列キーが不正です。 key=[" + key + "]"
						+ HybsSystem.CR
						+ "null または予約語(" + YOYAKU + ") は指定できません。";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * ユーザーのロールが、"root" 権限があるかどうか、返します。
	 *
	 * "root" 権限とは、ロールが、"root","manager","admin" のいずれかの場合。
	 *
	 * @og.rev 3.4.0.3 (2003/09/10) 新規追加
	 * @og.rev 4.3.0.0 (2008/07/04) 廃止
	 *
	 * @return  "root" 権限かどうか
	 */
//	public boolean isRoot() {
//		return isRootRole ;
//	}

	/**
	 * UserInfoの属性文字列を取得します。
	 *
	 * 以下の値は UserInfo オブジェクトの項目から取得します。
	 * <pre>
	 * ・JNAME      ユーザー日本語名称
	 * ・ID         ユーザーＩＤ
	 * ・IDNO       (初期値)USER.ID が５Byte以上の時のみ先頭１文字を除いたユーザーＩＤ
	 * ・INFO       ユーザー情報(ユーザーID：日本語名称)
	 * ・LANG       言語
	 * ・ROLES      ロール
	 * ・IPADDRESS  IPアドレス
	 * ・LOGINTIME  ログイン時刻
	 * ・LASTACCESS 最終アクセス画面ID
	 * ・LASTGAMENNM 最終アクセス画面名
	 *
	 * 以下の値はあらかじめ、動的に作成されます。
	 * ・YMD       ８byte の今日のシステム日付
	 * ・YMDH    １４byte の今日のシステム日時
	 * </pre>
	 *
	 * それ以外は,外部より設定された値です。
	 *
	 * @og.rev 2.1.0.2 (2002/11/07) USER.IDNO の返す値をUSER.ID が５Byte以上の時のみ、
	 * 先頭1文字を除いた値を返す様に変更。それ以外は、USER.IDを返す。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語(国際化)対応  ENCODE 追加
	 * @og.rev 3.1.3.0 (2003/04/10) ユーザー情報から、エンコード情報を削除する。
	 * @og.rev 3.5.4.2 (2003/12/15) ENAME,MAILTO、MAILUSERID、MAILPASSWD、GROUP、PROJECTを削除する。
	 * @og.rev 3.5.6.0 (2004/06/18) synchronized をattribute に行います。
	 * @og.rev 3.6.0.0 (2004/09/17) PASSWD を削除する。
	 * @og.rev 3.8.1.2 (2005/12/19) USER.IDNO を削除する。(外部設定可能にするため)
	 * @og.rev 3.8.7.0 (2006/12/15) ApplicationInfoオブジェクトから最終アクセス画面を取得
	 * @og.rev 4.4.0.0 (2009/08/02) データロール属性対応
	 * @og.rev 4.4.0.1 (2009/08/08) LASTGAMENNM追加
	 *
	 * @param	key  	キー
	 *
	 * @return	UserInfoの属性文字列
	 */
	public String getAttribute( final String key ) {
		if( key == null ) { return null; }
		String rtn = null;

		if(      key.equalsIgnoreCase( "JNAME"	) ) { rtn = jname; }
		else if( key.equalsIgnoreCase( "ID"		) ) { rtn = userID; }
		else if( key.equalsIgnoreCase( "INFO"		) ) { rtn = getInfo(); }
		else if( key.equalsIgnoreCase( "LANG"		) ) { rtn = lang; }
		else if( key.equalsIgnoreCase( "ROLE"		) ) { rtn = roles; }
		else if( key.equalsIgnoreCase( "ROLES"		) ) { rtn = roles; }
		else if( key.equalsIgnoreCase( "DROLES"		) ) { rtn = droles; } // 4.4.0.0 (2009/08/02)
		else if( key.equalsIgnoreCase( "IPADDRESS"	) ) { rtn = ipAddress; }
		else if( key.equalsIgnoreCase( "LOGINTIME"	) ) {
			rtn = HybsSystem.getDate( loginTime );
		}
		else if( key.equalsIgnoreCase( "LASTACCESS"  ) ) {		// 3.8.7.0 (2006/12/15)
			if( appInfo != null ) { rtn = appInfo.getGamenId(); }
		}
		else if( key.equalsIgnoreCase( "YMD" ) ) {
			rtn = HybsSystem.getDate( "yyyyMMdd" );
		}
		else if ( key.equalsIgnoreCase( "LASTGAMENNM" ) ){		// 4.4.0.1 (2009/08/08)
			if( appInfo != null && appInfo.getGamenId() != null){
				if( getGUIInfo( appInfo.getGamenId() ) != null){
					rtn = getGUIInfo( appInfo.getGamenId() ).getName();
				}
			}
		}
		else if( key.equalsIgnoreCase( "YMDH" ) ) {
			rtn = HybsSystem.getDate( "yyyyMMddHHmmss" );
		}
		else {
			synchronized( attribute ) {
				rtn = attribute.get( key ) ;
			}
		}
		return rtn ;
	}

	/**
	 * UserInfoの属性文字列の内部情報を返します。
	 * この内部情報の中には、UserInfo 自身の管理情報も含めます。
	 * 独自管理キーは、JNAME,ID,IDNO,INFO,LANG,ROLES,IPADDRESS,LOGINTIME,LASTACCESS,LASTGAMENNM です。
	 *
	 * それ以外は,外部より設定された値です。
	 *
	 * @og.rev 4.0.0 (2004/12/31) 新規作成
	 * @og.rev 4.4.0.1 (2009/08/08) LASTGAMENNM追加
 	 *
	 * @return 属性文字列のHybsEntryオブジェクト配列
	 */
	public HybsEntry[] getEntrys() {
		List<HybsEntry> list = new ArrayList<HybsEntry>();

		list.add( new HybsEntry( "JNAME"		, getAttribute( "JNAME"		) ,"ユーザー日本語名称" ) );
		list.add( new HybsEntry( "ID"			, getAttribute( "ID"		) ,"ユーザーＩＤ" ) );
		list.add( new HybsEntry( "IDNO" 		, getAttribute( "IDNO"		) ,"USER.ID が５Byte以上の時のみ先頭１文字を除いたユーザーＩＤ" ) );
		list.add( new HybsEntry( "INFO" 		, getAttribute( "INFO"		) ,"ユーザー情報(ユーザーID：日本語名称)" ) );
		list.add( new HybsEntry( "LANG" 		, getAttribute( "LANG"		) ,"言語" ) );
		list.add( new HybsEntry( "ROLES"		, getAttribute( "ROLES"		) ,"ロールズ" ) );
		list.add( new HybsEntry( "IPADDRESS"	, getAttribute( "IPADDRESS" ) ,"IPアドレス" ) );
		list.add( new HybsEntry( "LOGINTIME"	, getAttribute( "LOGINTIME" ) ,"ログイン時刻" ) );
		list.add( new HybsEntry( "LASTACCESS"	, getAttribute( "LASTACCESS" ) ,"最終アクセス画面ID" ) );
		list.add( new HybsEntry( "LASTGAMENNM" 	, getAttribute( "LASTGAMENNM") ,"最終アクセス画面名" ) ); // 4.4.0.1 (2009/08/08)
		list.add( new HybsEntry( "YMD"			, getAttribute( "YMD"		) ,"　８byte の今日のシステム日付" ) );
		list.add( new HybsEntry( "YMDH" 		, getAttribute( "YMDH"		) ,"１４byte の今日のシステム日時" ) );

		synchronized( attribute ) {
			String[] keys = attribute.keySet().toArray( new String[attribute.size()] );
			Arrays.sort( keys );
			for( int i=0; i<keys.length; i++ ) {
				list.add( new HybsEntry( keys[i],getAttribute( keys[i] ) ) );
			}
		}
		return list.toArray( new HybsEntry[list.size()] );
	}

	/**
	 * UserInfoの属性文字列を削除します。
	 *
	 * @param	key  	キー
	 * @param	save 	GE20(ユーザー定数)から情報を削除するか
	 */
	public void removeAttribute( final String key, final boolean save ) {
		removeAttribute( key, save, false );
	}

	/**
	 * UserInfoの属性文字列を削除します。
	 *
	 * @og.rev 3.5.6.0 (2004/06/18) synchronized をattribute に行います。
	 * @og.rev 5.3.6.0 (2011/06/01) GE20(ユーザー定数)から削除するかのフラグを追加
	 *
	 * @param	key  	キー
	 * @param	save 	GE20(ユーザー定数)から情報を削除するか
	 * @param	common 	GE20(ユーザー定数)から情報削除時、全ユーザー公開情報を削除するか
	 */
	private void removeAttribute( final String key, final boolean save, final boolean common ) {
		synchronized( attribute ) {
			attribute.remove( key ) ;
		}

		if( save ) {
			deletePermanently( key, common );
		}
	}

	/**
	 * ユーザー個別の画面オブジェクトのマップをセットします。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.0.0 (2005/01/31) 新規追加
	 * @og.rev 4.1.1.0 (2008/01/29) 画面の格上げとお気に入りマップ作成はクラスUserAccessTableに依頼
	 * @og.rev 5.2.0.0 (2010/09/01) アクセス禁止アドレスによる不正アクセス防止機能追加
	 *
	 * @param	newGuiMap			画面オブジェクトのマップ
	 * @param	newForbidAddrSet	アクセス禁止アドレスセット
	 */
//	public void setGUIMap( final Map<String,GUIInfo> newGuiMap ) {
	public void setGUIMap( final Map<String,GUIInfo> newGuiMap, final Set<String> newForbidAddrSet ) {
		if( newGuiMap != null ) {
			synchronized( guiLock ) {
				guiMap = newGuiMap ;
				forbidAddrSet = newForbidAddrSet;
				favoriteGuiMap = UserAccessTable.makeAccessDB( guiMap,systemId,userID,lang );
				isInfoSet = true;
			}
		}
	}

	/**
	 * ユーザー個別の画面オブジェクトを取得します。
	 * アクセスできない画面IDが指定されたときは、null が返ります。
	 *
	 * @og.rev 4.0.0 (2005/01/31) 新規追加
	 *
	 * @param	gamenId	画面ID
	 *
	 * @return	画面オブジェクト
	 */
	public GUIInfo getGUIInfo( final String gamenId ) {
		synchronized( guiLock ) {
			return guiMap.get( gamenId );
		}
	}

	/**
	 * ユーザー個別の画面オブジェクトのマップを取得します。
	 *
	 * @og.rev 4.0.0 (2005/01/31) 新規追加
	 *
	 * @return	画面オブジェクトの配列
	 */
	public GUIInfo[] getGUIInfos() {
		synchronized( guiLock ) {
			return guiMap.values().toArray( new GUIInfo[ guiMap.size() ] ) ;
		}
	}

	/**
	 * アクセスが許可されているアドレスかどうかをチェックします。
	 *
	 * チェックの方法は、ブラックリスト方式です。
	 * 画面リソースに登録された一覧の内、そのユーザーが許可されていないアドレスに
	 * 対してのみfalseが返ります。
	 * 画面リソースに登録されていないアドレスや、アドレスにURLの区切り文字(/)が
	 * 含まれている場合はチェックされません。(trueが返ります)
	 *
	 * @og.rev 5.2.0.0 (2010/09/01) 新規追加
	 *
	 * @param addr チェック対象のアドレス
	 *
	 * @return アクセスが許可されているアドレスかどうか
	 */
	public boolean isValidAddr( final String addr ) {
		synchronized( guiLock ) {
			return ( forbidAddrSet == null || !forbidAddrSet.contains( addr ) );
		}
	}

	/**
	 * ユーザー個別のお気に入り画面オブジェクトのマップを取得します。
	 *
	 * @og.rev 4.1.1.0 (2008/01/31) 新規追加
	 *
	 * @return	お気に入り画面オブジェクトのマップ
	 */
	public Map<String,FavoriteGUIData> getFavoriteMap() {
		return favoriteGuiMap;
	}

	/**
	 * 画面オブジェクトのマップがセット済みかどうかを取得します。
	 *
	 * @og.rev 4.0.0 (2005/01/31) 新規追加
	 *
	 * @return 画面オブジェクトのマップがセット済みかどうか(true:セット済み / false:未セット)
	 */
	public boolean isGUIInfoSet() {
		return isInfoSet ;
	}

	/**
	 * 指定のユーザーロールに対する最終的なアクセス条件を取得します。
	 * アクセス条件は、複数あるユーザーロールの中で、最大のアクセス条件を算出します。
	 * 例えば、AAA(-r)|BBB(-w)|CCC(mr) の３つのロール/モードが設定されている場合、
	 * ユーザーが、AAA だけの場合は、-r ですが、AAA|BBB を持っている場合は、-w になります。
	 * さらに、BBB|CCC と持っている場合は、(-w:書き込み許可)と(mr:メニューから読取許可)の
	 * 権限により、mw:メニューからの書き込み許可が与えられます。
	 * モード指定がある場合は、AND演算になります。
	 * 例えば、AAA(-r)|BBB(-w)|CCC(mr) と BBB|CCC(-r) の場合、(-r)+(-w)+(mr)*(-r)=-w に
	 * なります。ロールは、OR ですが、モードは、同一ロールでのAND になります。
	 * 実際には、メニュー表示の可否は、ポップアップ系によく用いられますので、上記のような
	 * 許可が実際にあるかどうかは不明ですが、すべてのモードのOR条件での結合になります。
	 *
	 * @og.rev 4.3.0.0 (2008/07/04) ロールモードマルチ対応
	 *
	 * @param	other	ロールモード
	 *
	 * @return アクセスビット
	 */
	public byte getAccessBitMode( final RoleMode other ) {
		return roleMode.getAccessBitMode( other );
	}

	/**
	 * 指定のユーザーロールに対する最終的なアクセス条件を取得します。
	 *
	 * @og.rev 4.3.0.0 (2008/07/04) ロールモードマルチ対応
	 *
	 * @param	roles	ロール文字列
	 *
	 * @return アクセスビット
	 */
//	public byte getAccessBitMode( final String roles ) {
//		return roleMode.getAccessBitMode( RoleMode.newInstance( roles ) );
//	}

	/**
	 * このユーザーの権限で、指定のロールが許可されているかどうかを判定します。
	 *
	 * @og.rev 4.3.0.0 (2008/07/04) ロールモードマルチ対応
	 * @og.rev 4.3.0.1 (2008/08/11) ロールチェック時の引数間違い、是正
	 *
	 * @param	role	チェックを行うロール
	 *
	 * @return	アクセスできる(true)/出来ない(false)
	 */
	public boolean isAccess( final String role ) {
		if( role == null || role.length() == 0 ) {
			return true;
		}

//		return ( roleMode.getAccessBitMode( RoleMode.newInstance( roles ) ) > 0 );
		return ( roleMode.getAccessBitMode( RoleMode.newInstance( role ) ) > 0 );		// 4.3.0.1 (2008/08/11)
	}

	/**
	 * このユーザーの権限で、指定のロールが許可されているかどうかを判定します。
	 *
	 * @og.rev 4.3.0.0 (2008/07/04) ロールモードマルチ対応
	 *
	 * @param otherRoles チェックを行うロール
	 *
	 * @return アクセスできる(true)/出来ない(false)
	 */
////	public boolean isAccess( final String role ) {
//	public boolean isAccess( final String otherRoles ) {
//		// ユーザーがルートか、指定のロールが無い場合は、無制限アクセス可能
////		if( isRootRole || role == null || role.length() == 0 ) {
//		if( otherRoles == null || otherRoles.length() == 0 ) {
//			return true;
//		}
//
//		// ユーザーロールが無い場合は、アクセス不許可
//		if( roles == null || roles.length() == 0 ) {
//			return false;
//		}
//
//	 	// 4.3.0.0 (2008/07/04) ロールモードマルチ対応
//		RoleMode otherRoleMode = RoleMode.newInstance( otherRoles );
//		byte bit = roleMode.getAccessBitMode( otherRoleMode );
//
//		return bit > 0 ;
//
////		// ユーザーロールが無い場合は、アクセス不許可
////		if( userRoles == null || userRoles.length == 0 ) {
////			return false;
////		}
////
////		String[] otherRoles = StringUtil.csv2Array( role,HybsSystem.GUI_DELIMITER );
////		for( int g=0; g<otherRoles.length; g++ ) {
////			if( otherRoles[g] != null ) {
////				for( int u=0; u<userRoles.length; u++ ) {
////					if( otherRoles[g].equalsIgnoreCase( userRoles[u] ) ) {
////						return true;
////					}
////				}
////			}
////		}
////		return false;
//	}

	/**
	 * 初期化(クリア)します(org.opengion.fukurou.util.Cleanable の実装)。
	 * 画面オブジェクトのマップをクリアし、セット済みフラグを未セットに設定します。
	 * システムリソースの USE_ACCESS_TOKEI_TABLE が true の場合は、
	 * 画面アクセス状況を、アクセス統計テーブル(GE15)に書き込みます。
	 * ユーザー単位のパラメータは、システムリソースの USER_PARAMETER_SQL で
	 * 定義された値を検索して、取り込みます。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.0.0 (2005/01/31) 新規追加
	 */
	public void clear() {
		if( useAccessTable && isInfoSet ) { saveGUIAccessInfo(); }
		initLoad() ;
	}

	/**
	 * ユーザーパラメータを取得します。
	 * 画面オブジェクトのマップをクリアし、セット済みフラグを未セットに設定します。
	 * ユーザー単位のパラメータは、システムリソースの USER_PARAMETER_SQL で
	 * 定義された値を検索して、取り込みます。
	 *
	 * @og.rev 4.0.0 (2005/01/31) 新規追加
	 */
	private void initLoad() {

		// ユーザーパラメータの取得
		if( QUERY_PARAM != null && QUERY_PARAM.length() > 0 ) {
			String[] args = new String[] { systemId,userID };
			// 画面ID,操作,プログラムID
			if( appInfo != null ) {
				// 画面ID,操作,プログラムID
				appInfo.setModuleInfo( "UserInfo",null,"initLoad" );
			}
			String[][] vals = DBUtil.dbExecute( QUERY_PARAM,args,appInfo,DBID );

			synchronized( paramMap ) {
				paramMap.clear();
				for( int i=0; i<vals.length; i++ ) {
					String key   = vals[i][0];
					String val   = vals[i][1];
					if( val != null && val.length() == 0 ) { continue; }
					paramMap.put( key,val );
				}
			}
		}
	}

	/**
	 * GE20からDBに保存されたUserInfoのパラメーターを取得します。
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) 新規追加
	 */
	private void dbLoad() {
		// GE20 ユーザーロール情報永続化テーブル読み込み
		String[] argsGe20 = new String[] { systemId,userID,roles };
		String[][] valsGe20 = DBUtil.dbExecute( QUERY_GE20,argsGe20,appInfo,DBID );

		for( int i=0; i<valsGe20.length; i++ ) {
			// 4.3.4.0 (2008/12/01) GE20から読み込んでいるので、当然保存しない
			setAttribute( valsGe20[i][0], valsGe20[i][1], false );
		}
	}

	/**
	 * 属性一覧からEDIT設定情報をオブジェクト化し、画面ID毎のマップに登録します。
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) 新規追加
	 */
	private void makeEditConfigMap() {
		String[] keys = attribute.keySet().toArray( new String[0] );
		String[][] keySet = DBEditConfig.getKeySet( keys );
		if( keySet != null ) {
			for( String[] set : keySet ) {
				String guikey = set[0];
				String editName = set[1];
				String[] editKeys = DBEditConfig.getEditKeys( guikey, editName );
				String[] editVals = new String[editKeys.length];
				for( int i=0; i<editKeys.length; i++ ) {
					editVals[i] = attribute.get( editKeys[i] );
				}
				editMgr.addEditConfig( guikey, editName, new DBEditConfig( editVals ) );
			}
		}
	}

	/**
	 * 引数の画面で登録されているエディット設定を配列で返します。
	 * 返される配列は、エディット名順にソートされた状態で返されます。
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) 新規追加
	 *
	 * @param guikey 画面ID
	 *
	 * @return エディット設定(配列)
	 */
	public DBEditConfig[] getEditConfigs( final String guikey ) {
		return editMgr.getEditConfigs( guikey );
	}

	/**
	 * 画面ID、エディット名よりエディット設定オブジェクトを返します。
	 * また、ここで指定されたエディット名がこの画面での選択済みエディットとして登録されます。
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) 新規追加
	 *
	 * @param guikey 画面ID
	 * @param editName エディット名
	 *
	 * @return エディット配列
	 */
	public DBEditConfig getEditConfig( final String guikey, final String editName ) {
		if( editName != null ) {
			String selEditName = getSelectedEdit( guikey );
			if( !editName.equals( selEditName ) ) {
				setSelectedEdit( guikey, editName );
			}
		}
		else {
			setSelectedEdit( guikey, null );
		}
		return editMgr.getEditConfig( guikey, editName );
	}

	/**
	 * 指定の画面ID、エディット名でエディット設定オブジェクトを追加します。
	 * 既に登録されている場合は、既存のエディット情報を更新します。
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) 新規追加
	 *
	 * @param guikey 画面ID
	 * @param editName エディット名
	 * @param config エディット設定オブジェクト
	 */
	public void addEditConfig( final String guikey, final String editName, final DBEditConfig config ) {
		if( config == null ) { return; }

		boolean isCommon = config.isCommon();
		String[] editKeys = DBEditConfig.getEditKeys( guikey, editName );
		String[] editVals = config.getEditVals();

		// 個別設定の場合、同じキーで共通情報が存在していた場合はエラーとする。
		if( !isCommon && isExistValue( editKeys[0], "*", "*" ) ) {
			String msg = "同じ編集名で共通設定されているため個別編集を保存できません。";
			throw new HybsSystemException( msg );
		}
		// 共通設定の場合、同じキーで個別情報が存在していた場合はエラーとする。
		if( isCommon && isExistValue( editKeys[0], userID, "*" ) ) {
			String msg = "同じ編集名で個別設定されているため共通編集を保存できません。";
			throw new HybsSystemException( msg );
		}

		editMgr.addEditConfig( guikey, editName, config );
		for( int i=0; i<editKeys.length; i++ ) {
			if( editVals[i] != null && editVals[i].length() > 0 ) {
				setAttribute( editKeys[i], editVals[i], true, isCommon );
			}
			else {
				removeAttribute( editKeys[i], true, isCommon );
			}
		}
	}

	/**
	 * 指定の画面ID、エディット名のエディット設定を削除します。
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) 新規追加
	 *
	 * @param guikey 画面ID
	 * @param editName エディット名
	 */
	public void deleteEditConfig( final String guikey, final String editName ) {
		DBEditConfig config = editMgr.deleteEditConfig( guikey, editName );
		if( config != null ) {
			boolean isCommon = config.isCommon();
			String[] editKeys = DBEditConfig.getEditKeys( guikey, editName );
			// エディット設定が存在しない場合エラー。
			if( !isExistValue( editKeys[0], ( isCommon ? "*" : userID ), "*" ) ) {
				String msg = "エディット設定が存在しません。";
				throw new HybsSystemException( msg );
			}
			for( int i=0; i<editKeys.length; i++ ) {
				removeAttribute( editKeys[i], true, isCommon );
			}
		}

		if( editName != null ) {
			String selEditName = getSelectedEdit( guikey );
			if( !editName.equals( selEditName ) ) {
				setSelectedEdit( guikey, null );
			}
		}
	}

	/**
	 * 指定の画面IDに対して選択済みのエディット名を登録します。
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) 新規追加
	 *
	 * @param guikey 画面ID
	 * @param editName エディット名
	 */
	public void setSelectedEdit( final String guikey, final String editName ) {
		setAttribute( "EDIT_NAME_SELECTED_" + guikey, editName, true );
	}

	/**
	 * 指定の画面IDに対して選択済みのエディット名を返します。
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) 新規追加
	 *
	 * @param guikey 画面ID
	 *
	 * @return 選択済みエディット名
	 */
	public String getSelectedEdit( final String guikey ) {
		return getAttribute( "EDIT_NAME_SELECTED_" + guikey );
	}

	/**
	 * アクセスログ取得の為,ApplicationInfoオブジェクトを返します。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) 新規追加
	 *
	 * @param	gamenId	実行中の画面ID
	 * @param	prgId	実行中のプログラムID
	 *
	 * @return	ApplicationInfoオブジェクト
	 */
	public ApplicationInfo getApplicationInfo( final String gamenId,final String prgId ) {
		if( appInfo != null ) {
			// 画面ID,操作,プログラムID
			appInfo.setModuleInfo( gamenId,null,prgId );
		}
		return appInfo;
	}

	/**
	 * 自然比較メソッド
	 * インタフェース Comparable の 実装です。
	 * ユーザーの順序は、ユーザーID そのものの順序であらわされます。
	 * 同一ユーザーの場合は，ログインタイムの順番になります。
	 *
	 * @og.rev 5.1.8.0 (2010/07/01) UserSummary の Comparable を型設定
	 *
	 * @param	object	比較対象のObject
	 *
	 * @return	このオブジェクトが指定されたオブジェクトより小さい場合は負の整数、等しい場合はゼロ、大きい場合は正の整数
	 */
	@Override
	public int compareTo( final UserSummary object ) {
//		if( object instanceof UserInfo ) {
//			int test1 = userID.compareTo( ((UserInfo)object).getUserID() );
			int test1 = userID.compareTo( object.getUserID() );
			if( test1 == 0 ) {
//				test1 = (int)( loginTime - ((UserInfo)object).getLoginTime() ) ;
				test1 = (int)( loginTime - object.getLoginTime() ) ;
			}
			return test1;
//		}
//		throw new ClassCastException();
	}

	/**
	 * このオブジェクトと他のオブジェクトが等しいかどうかを示します。
	 * インタフェース Comparable の 実装に関連して、再定義しています。
	 * ユーザーは、ユーザーIDが等しく、かつ ログイン時刻が同一の場合に、
	 * 等しいと判断されます。
	 *
	 * @param   object 比較対象の参照オブジェクト
	 *
	 * @return	引数に指定されたオブジェクトとこのオブジェクトが等しい場合は true、そうでない場合は false
	 */
	@Override
	public boolean equals( final Object object ) {
		if( object instanceof UserInfo ) {
			return ( userID.equals( ((UserInfo)object).getUserID()  )  &&
					 loginTime == ( ((UserInfo)object).getLoginTime() )   );
		}
		return false ;
	}

	/**
	 * オブジェクトのハッシュコード値を返します。
	 * このメソッドは、java.util.Hashtable によって提供されるような
	 * ハッシュテーブルで使用するために用意されています。
	 * equals( Object ) メソッドをオーバーライトした場合は、hashCode() メソッドも
	 * 必ず 記述する必要があります。
	 * ここでは、ログイン時刻(long 値)の上位 32 ビットと下位 32 ビットの排他的論理和
	 * を求めています。
	 * (int)(this.longValue()^(this.longValue()&gt;&gt;&gt;32))
	 *
	 * ※ hashCode の 同一オブジェクトには同一ハッシュコードという規則と
	 *    発生頻度,ランダム性を考慮すれば、ログイン時刻そのもの(long)の
	 *    ハッシュコードでも運用上は全く問題ないと考えられます。
	 *
	 * @og.rev 3.5.6.0 (2004/06/18) 新規追加
	 *
	 * @return  このオブジェクトのハッシュコード値
	 *
	 */
	@Override
	public int hashCode() {
		return hashcode ;
	}

	/**
	 * オブジェクトの識別子として，詳細なユーザー情報を返します。
	 *
	 * @return  詳細なユーザー情報
	 */
	@Override
	public String toString() {
		StringBuilder rtn = new StringBuilder( HybsSystem.BUFFER_MIDDLE );
		rtn.append( "userID   :" ).append( userID	 ).append( HybsSystem.CR );
		rtn.append( "lang     :" ).append( lang		 ).append( HybsSystem.CR );
		rtn.append( "jname    :" ).append( jname	 ).append( HybsSystem.CR );
		rtn.append( "roles    :" ).append( roles	 ).append( HybsSystem.CR );
		rtn.append( "IPAddress:" ).append( ipAddress ).append( HybsSystem.CR );
		rtn.append( "loginTime:" ).append( loginTime ).append( HybsSystem.CR );
		return rtn.toString();
	}

	// saveGUIAccessInfo() メソッドでしか使用しない、定数宣言
	private static final int C_SYSTEM_ID		= 0 ;
	private static final int C_USERID			= 1 ;
	private static final int C_USERADRS			= 2 ;
	private static final int C_HOSTADRS			= 3 ;
	private static final int C_GUIKEY			= 4 ;
	private static final int C_DYLOGIN			= 5 ;
	private static final int C_DYLOGOUT			= 6 ;
	private static final int C_USED_TIME		= 7 ;
	private static final int C_CNT_ACCESS		= 8 ;
	private static final int C_CNT_ERROR		= 9 ;
	private static final int C_CNT_READ			= 10 ;
	private static final int C_CNT_WRITE		= 11 ;
	private static final int C_TM_TOTAL_QUERY	= 12 ;
	private static final int C_TM_MAX_QUERY		= 13 ;
	private static final int C_MAX_QUERY		= 14 ;
	private static final int C_FGJ				= 15 ;
	private static final int C_DYSET			= 16;
	private static final int C_DYUPD			= 17;
	private static final int C_USRSET			= 18;
	private static final int C_USRUPD			= 19;
	private static final int C_PGUPD			= 20;

	/**
	 * ユーザー個別の画面オブジェクトの明細情報をアクセス統計テーブル(GE15)に登録します。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.0.0 (2005/01/31) 新規追加
	 * @og.rev 4.0.0.0 (2007/10/05) SQLServer 互換性の為、SUBSTRB を廃止します。
	 * @og.rev 4.1.1.0 (2008/01/30) ユーザーアクセス画面管理テーブルに画面の最終アクセス時間を更新
	 * @og.rev 5.0.2.0 (2009/11/01) 作成・更新日付がセットされていないバグを修正
	 * @og.rev 5.2.3.0 (2010/12/01) 画面アクセスの履歴(順番)を管理する機能を追加
	 */
	private void saveGUIAccessInfo() {
	//	if( !useAccessTable || !isInfoSet ) { return ; }

		final GUIInfo[] infos ;
		synchronized( guiLock ) {
			infos = getGUIInfos() ;
			guiMap = null;
			isInfoSet = false;
		}

		long crntTime = System.currentTimeMillis();

		String[] names = new String[] { "SYSTEM_ID","USERID","USERADRS","HOSTADRS","GUIKEY","DYLOGIN","DYLOGOUT",
										"USED_TIME","CNT_ACCESS","CNT_ERROR","CNT_READ","CNT_WRITE",
//										"TM_TOTAL_QUERY","TM_MAX_QUERY","MAX_QUERY","FGJ" };
										"TM_TOTAL_QUERY","TM_MAX_QUERY","MAX_QUERY","FGJ","DYSET","DYUPD","USRSET","USRUPD","PGUPD" };
		String[] values = new String[names.length];

		values[C_SYSTEM_ID		] = HybsSystem.sys( "SYSTEM_ID" );
		values[C_USERID			] = userID;
		values[C_USERADRS		] = ipAddress;
		values[C_HOSTADRS		] = HybsSystem.sys( "HOST_ADRS" );
		values[C_GUIKEY			] = "";
		values[C_DYLOGIN		] = HybsSystem.getDate( loginTime,"yyyyMMddHHmmss" );
		values[C_DYLOGOUT 		] = HybsSystem.getDate( "yyyyMMddHHmmss" );
		values[C_USED_TIME		] = String.valueOf( Math.round( (crntTime-usedTime) / 1000.0d ) );	// 秒に変換
		values[C_CNT_ACCESS		] = "0";
		values[C_CNT_ERROR		] = "0";
		values[C_CNT_READ 		] = "0";
		values[C_CNT_WRITE		] = "0";
		values[C_TM_TOTAL_QUERY	] = "0";
		values[C_TM_MAX_QUERY	] = "0";
		values[C_MAX_QUERY		] = "";
		values[C_FGJ			] = "1";
		values[C_DYSET			] = HybsSystem.getDate( "yyyyMMddHHmmss" );
		values[C_DYUPD			] = HybsSystem.getDate( "yyyyMMddHHmmss" );
		values[C_USRSET			] = "userInfo";
		values[C_USRUPD			] = "userInfo";
		values[C_PGUPD			] = "userInfo";

		usedTime = crntTime ;

		DBSimpleTable dbTable = new DBSimpleTable( names );
		// 画面ID,操作,プログラムID
		getApplicationInfo( "UserInfo","saveGUI" );
		dbTable.setApplicationInfo( appInfo );	// 3.8.7.0 (2006/12/15)
		dbTable.setTable( "GE15" );
		// 4.0.0.0 (2007/10/05) SQLServer 互換性の為、CLOB化します。
	//	dbTable.addConstrain( names[C_MAX_QUERY],"SUBSTRB(?,1,4000)" );

		boolean okFlag = false;
		try {
			dbTable.startInsert();

			// UserInfo に関する情報の登録
			dbTable.execute( values );

			// GUIInfo に関する情報の登録
			if( infos != null ) {
				values[C_USED_TIME] = "0";	// USED_TIME をクリアしておきます。
				String logoutTime = HybsSystem.getDate( "yyyyMMddHHmmss" );
				for( int i=0; i<infos.length; i++ ) {
					GUIAccessCount access = infos[i].getGUIAccessCount();
					int cnt = access.getAccessCount();
					if( cnt > 0 ) {
						values[C_GUIKEY			] = access.getKey();
						values[C_CNT_ACCESS		] = String.valueOf( cnt );
						values[C_CNT_ERROR		] = String.valueOf( access.getErrorCount() );
						values[C_CNT_READ 		] = String.valueOf( access.getReadCount() );
						values[C_CNT_WRITE		] = String.valueOf( access.getWriteCount() );
						values[C_TM_TOTAL_QUERY	] = String.valueOf( access.getQueryTime() );
						values[C_TM_MAX_QUERY	] = String.valueOf( access.getMaxQueryTime() );
						values[C_MAX_QUERY		] = access.getMaxQuery();
	//					dbTable.addValues( values );
						dbTable.execute( values );
						// 4.1.1.0(2008/01/28)画面アクセス時間の更新
						// 5.2.3.0 (2010/12/01) 画面アクセスの履歴(順番)を管理する機能を追加
						String keys = infos[i].getNextGuiKeys();
//						UserAccessTable.updateLastAccessTime( systemId,userID,access.getKey(),logoutTime );
						UserAccessTable.updateLastAccessTime( systemId,userID,access.getKey(),logoutTime,keys );
					}
				}
			}
			okFlag = true;
		}
		catch (SQLException ex) {
			LogWriter.log( "  userID=[" + userID + "] アクセス統計テーブル登録時にエラーが発生しました" );
			LogWriter.log( ex.getMessage() );
		}
		finally {
			int cnt = dbTable.close( okFlag );
			System.out.println();
			System.out.println( "  userID=[" + userID + "] アクセス統計テーブルに、[" + cnt + "]件、追加しました。" );
		}
	}

	// 永続化情報設定でしか使用しない変数の宣言
	private static final int C_GE20_SYSTEM_ID	= 0;
	private static final int C_GE20_USERID		= 1;
	private static final int C_GE20_ROLES		= 2;
	private static final int C_GE20_PARAM_ID	= 3;
	private static final int C_GE20_PARAM		= 4;
	private static final int C_GE20_KBSET		= 5;
	private static final int C_GE20_FGJ			= 6;
	private static final int C_GE20_DYSET		= 7;
	private static final int C_GE20_DYUPD		= 8;
	private static final int C_GE20_USRSET		= 9;
	private static final int C_GE20_USRUPD		= 10;
	private static final int C_GE20_PGUPD		= 11;

	private static final int GE20_KBSET_READONLY	= 1;
	private static final int GE20_KBSET_WRITABLE	= 2;

	// ロールは全て*で登録する。アプリケーションから動的に登録される値を、
	// ロール単位設定しても、ロール変更時に整合性が合わない可能性大なので、
	// UserInfoで設定する場合は、全てのロールで有効とする。
	private static final String GE20_ROLES = "*";

	/**
	 * userInfoにセットされた値/キーをDBに登録します。
	 * 既にキーが存在している場合は、既存データを更新し、なければ追加します。
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) 全ユーザー情報として保存できるように対応
	 *
	 * @param key キー
	 * @param value 値
	 * @param isCommon ユーザーID='*'(全ユーザー公開)として登録するかどうか
	 */
//	private void savePermanently( final String key, final String value ) {
	private void savePermanently( final String key, final String value, final boolean isCommon ) {

		// 追加変更時に共通でセットされる値を設定
		String[] names = new String[] { "SYSTEM_ID","USERID","ROLES","PARAM_ID","PARAM","KBSET"
										,"FGJ","DYSET","DYUPD","USRSET","USRUPD","PGUPD" };
		String[] values = new String[names.length];
		values[C_GE20_SYSTEM_ID	] = HybsSystem.sys( "SYSTEM_ID" );
//		values[C_GE20_USERID	] = userID;
		values[C_GE20_USERID	] = ( isCommon ? "*" : userID );
		values[C_GE20_ROLES		] = GE20_ROLES;
		values[C_GE20_PARAM_ID	] = key;
		values[C_GE20_PARAM		] = value;
		values[C_GE20_KBSET		] = String.valueOf( GE20_KBSET_WRITABLE );
		values[C_GE20_FGJ		] = "1";
		values[C_GE20_DYSET		] = HybsSystem.getDate( "yyyyMMddHHmmss" );
		values[C_GE20_DYUPD		] = HybsSystem.getDate( "yyyyMMddHHmmss" );
		values[C_GE20_USRSET 	] = userID;
		values[C_GE20_USRUPD	] = userID;
		values[C_GE20_PGUPD	] = "UserInfo";

		// 画面ID,操作,プログラムID
		getApplicationInfo( "UserInfo","registValueToDB" );

		DBSimpleTable dbTable = new DBSimpleTable( names );
		dbTable.setApplicationInfo( appInfo );	// 3.8.7.0 (2006/12/15)
		dbTable.setTable( "GE20" );

		boolean okFlag = false;
		try {
//			if( isExistValue( key, GE20_ROLES ) ) {
			if( isExistValue( key, ( isCommon ? "*" : userID ), GE20_ROLES ) ) {
				String where = "SYSTEM_ID = [SYSTEM_ID] and USERID = [USERID] and ROLES = [ROLES] and PARAM_ID = [PARAM_ID] and FGJ='1'";
				dbTable.setWhere( where );
				dbTable.startUpdate();
			}
			else {
				dbTable.startInsert();
			}
			dbTable.execute( values );
			okFlag = true;
		}
		catch ( SQLException ex ) {
			throw new HybsSystemException( "ユーザー永続化情報設定時にエラーが発生しました", ex );
		}
		finally {
			dbTable.close( okFlag );
		}
	}

	/**
	 * userInfoから削除された値/キーをDBからも削除します。
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) 新規追加
	 *
	 * @param key キー
	 * @param isCommon ユーザーID='*'(全ユーザー公開)から削除するかどうか
	 */
	private void deletePermanently( final String key, final boolean isCommon ) {

		// 追加変更時に共通でセットされる値を設定
		String[] names = new String[] { "SYSTEM_ID","USERID","ROLES","PARAM_ID" };
		String[] values = new String[names.length];
		values[C_GE20_SYSTEM_ID	] = HybsSystem.sys( "SYSTEM_ID" );
		values[C_GE20_USERID	] = ( isCommon ? "*" : userID );
		values[C_GE20_ROLES		] = GE20_ROLES;
		values[C_GE20_PARAM_ID	] = key;

		// 画面ID,操作,プログラムID
		getApplicationInfo( "UserInfo","deleteValueFromDB" );

		DBSimpleTable dbTable = new DBSimpleTable( names );
		dbTable.setApplicationInfo( appInfo );
		dbTable.setTable( "GE20" );

		boolean okFlag = false;
		try {
			String where = "SYSTEM_ID = [SYSTEM_ID] and USERID = [USERID] and ROLES = [ROLES] and PARAM_ID = [PARAM_ID] and FGJ='1'";
			dbTable.setWhere( where );
			dbTable.startDelete();
			dbTable.execute( values );
			okFlag = true;
		}
		catch ( SQLException ex ) {
			throw new HybsSystemException( "ユーザー永続化情報削除時にエラーが発生しました", ex );
		}
		finally {
			dbTable.close( okFlag );
		}
	}

	/**
	 * ユーザー定数(GE20)に該当のキーが存在するかをチェックします。
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) 全ユーザー情報として保存できるように対応
	 *
	 * @param key キー
	 * @param userid ユーザーID
	 * @param roles ロール
	 *
	 * @return true:存在している/false:存在していない
	 */
//	private boolean isExistValue( final String key, final String roles ) {
	private boolean isExistValue( final String key, final String userid, final String roles ) {
//		String[] args = { HybsSystem.sys( "SYSTEM_ID" ), userID, roles, key };
		String[] args = { HybsSystem.sys( "SYSTEM_ID" ), userid, roles, key };

		// 画面ID,操作,プログラムID
		getApplicationInfo( "UserInfo","isExistValue" );

		String[][] rtn = DBUtil.dbExecute( QUERY_GE20_KEY, args, appInfo );
		if( rtn == null || rtn.length == 0 ) {
			return false;
		}
		else if( rtn[0].length > 0 ) {
			if( String.valueOf( GE20_KBSET_READONLY ).equals( rtn[0][0] ) ) {
				throw new HybsSystemException( "読み取り専用情報のため、書き込みできません" );
			}
			else {
				return true;
			}
		}
		else {
			throw new HybsSystemException( "永続化情報検索時にエラーが発生しました。" );
		}
	}

	/**
	 * 指定されたカラムキーに対応するデータの条件式を返します。
	 *
	 * @og.rev 4.4.0.0 (2009/08/02) 新規追加
	 *
	 * @param clm カラム名
	 *
	 * @return データの条件式
	 */
	public String getDataCondition ( final String clm ) {
		return dataRole.getCondition( clm );
	}

	/**
	 * このユーザーでアクセスされた画面オブジェクトを設定します。
	 *
	 * これは、画面アクセスの履歴(順番)を管理する機能に使います。
	 *
	 * @og.rev 5.2.3.0 (2010/12/01) 新規追加
	 *
	 * @param guiInfo 画面オブジェクト
	 */
	public void setAccessGui( final GUIInfo guiInfo ) {
		if( lastGuiInfo != null && guiInfo != null ) {
			lastGuiInfo.setNextGuiKey( guiInfo.getKey() );
		}
		lastGuiInfo = guiInfo ;		// 最後にアクセスした GUIInfo を設定
	}
}
