/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.query;

import org.opengion.hayabusa.db.AbstractQuery;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.Closer;
import org.opengion.fukurou.model.Formatter;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ParameterMetaData;
import java.sql.SQLException;

/**
 * 引数引き当て(PreparedStatement) を利用した登録系Queryです。
 *
 * java.sql.PreparedStatement を用いて、データベース検索処理を行います。
 * 引数の指定方法は、DBTableModele のカラム名に対応する名称を、SQL文の[カラム名]形式で
 * 記述します。これを解析して、実際に実行する PreparedStatement に対応する文字列を
 * 作成します。
 * たとえば、INSERT INTO GEXX (CLM,NAME_JA,LABEL_NAME) VALUES ([CLM],[NAME_JA],[LABEL_NAME] )
 * と記述すれば、内部で、DBTableModele のカラム名に対応する値を取り出し、SQL文として、
 * INSERT INTO GEXX (CLM,NAME_JA,LABEL_NAME) VALUES (?,?,? ) を実行します。
 *
 * @og.formSample
 * ●使用例
 *
 *    ・QUERYを直接書く場合
 *    【entry.jsp】
 *        &lt;og:tableUpdate
 *            command   = &quot;{&#064;command}&quot;
 *            queryType = &quot;JDBCTableUpdate&quot;
 *        &gt;
 *            INSERT INTO GE41
 *                (CLM,NAME_JA,LABEL_NAME,KBSAKU,SYSTEM_ID,LANG,
 *                 FGJ,DYSET,DYUPD,USRSET,USRUPD,PGUPD)
 *            VALUES
 *                ([CLM],[NAME_JA],[LABEL_NAME],[KBSAKU],[SYSTEM_ID],[LANG],
 *                 &#39;1&#39;,&#39;{&#064;USER.YMDH}&#39;,&#39;{&#064;USER.YMDH}&#39;,&#39;{&#064;USER.ID}&#39;,&#39;{&#064;USER.ID}&#39;,&#39;{&#064;GUI.KEY}&#39;)
 *        &lt;/og:tableUpdate&gt;
 *
 * @og.rev 4.0.0 (2005/01/31) 新規作成
 * @og.group データ編集
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Query_JDBCTableUpdate extends AbstractQuery {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	/**
	 * 引数配列付のクエリーを実行します。
	 * 処理自体は, #execute() と同様に、各サブクラスの実装に依存します。
	 * これは、PreparedQuery で使用する引数を配列でセットするものです。
	 * select * from emp where deptno = ? and job = ? などの PreparedQuery の
	 * [カラム名] 部分の引数を、DBTableModelから順番にセットしていきます。
	 *
	 * @og.rev 3.8.0.8 (2005/10/03) エラーメッセージの出力順をメッセージ＋Queryに変更します。
	 * @og.rev 4.0.0 (2007/05/09) ParameterMetaData を使用したパラメータ設定追加。
	 * @og.rev 4.0.0.0 (2007/09/25) isOracle から useParamMetaData に変更
	 * @og.rev 5.3.8.0 (2011/08/01) useParamMetaData を ConnectionFactory経由で取得。(PostgreSQL対応)、setNull 対応
	 *
	 * @param   rowNo 選択された行番号配列(登録する対象行)
	 * @param   table DBTableModelオブジェクト(登録する元データ)
	 */
	@Override
	public void execute( final int[] rowNo, final DBTableModel table ) {
		PreparedStatement pstmt = null ;
		ParameterMetaData pMeta = null ;

		int row = 0;			// エラー時に表示するエラー行番号
		try {
			int executeCount = 0;	// 処理件数
			Formatter form = new Formatter( table );
			form.setFormat( getStatement() );
			int[] clmNos = form.getClmNos();		// 引数の個数分の配列。カラム番号を保存
			String query = form.getQueryFormatString();
			int   cnt    = clmNos.length;			// 引数の個数(カラムの個数ではありません。)

			Connection conn = getConnection();
			pstmt = conn.prepareStatement( query );
			// 4.0.0.0 (2007/09/25) isOracle から useParamMetaData に変更
//			boolean useParamMetaData = ApplicationInfo.useParameterMetaData( conn );
			boolean useParamMetaData = useParameterMetaData();	// 5.3.8.0 (2011/08/01)
			pstmt.setQueryTimeout( DB_MAX_QUERY_TIMEOUT );
		//	((oracle.jdbc.OraclePreparedStatement)pstmt).setExecuteBatch(50);

			if( useParamMetaData ) { pMeta = pstmt.getParameterMetaData(); }
			for( int i=0; i<rowNo.length; i++ ) {
				row = rowNo[i];
				for( int j=0; j<cnt; j++ ) {
//					String val = table.getValue( row,clmNos[j] ) ;	// 5.3.8.0 (2011/08/01) 簡素化
					String val = StringUtil.rTrim( table.getValue( row,clmNos[j] ) );
					// 4.0.0.0 (2007/09/25) ParameterMetaData を使用したパラメータ設定追加
					if( useParamMetaData ) {
						int type = pMeta.getParameterType( j+1 );
						// 5.3.8.0 (2011/08/01) setNull 対応
//						pstmt.setObject( j+1,StringUtil.rTrim( val ),type );
						if( val == null || val.isEmpty() ) {
							pstmt.setNull( j+1, type );
						}
						else {
							pstmt.setObject( j+1, val, type );
						}
					}
					else {
//						pstmt.setObject( j+1,StringUtil.rTrim( val ) );
						pstmt.setObject( j+1,val );
					}
				}
				executeCount += pstmt.executeUpdate();
			}
			setExecuteCount( executeCount );
			setErrorCode( ErrorMessage.OK );
		}
		catch (SQLException ex) {
			setErrorCode( ErrorMessage.EXCEPTION );
			String errMsg = ex.getMessage() + ":" + ex.getSQLState() + HybsSystem.CR
						+ "QUERY=" + getStatement() + HybsSystem.CR
						+ "ROW=[" + (row+1) + "]"
						+ " VALS=[" + StringUtil.array2csv( table.getValues(row) )
						+ HybsSystem.CR ;
			rollback();
			realClose();
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}
		finally {
			Closer.stmtClose( pstmt );
		}
	}
}
