/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.io;

import org.opengion.hayabusa.db.DBTableModel;

import java.awt.Color;							// 6.0.2.2 (2014/10/03)
import java.sql.Connection;
import java.sql.SQLException;

import org.jfree.data.category.CategoryDataset;
import org.jfree.data.Range;

/**
 * HybsJDBCCategoryDataset は、org.jfree.data.jdbc.JDBCCategoryDataset を継承したサブクラスで、
 * executeQuery(Connection , String )  をオーバーライドしています。
 * これは、元のソースのデータベース検索結果を内部で持っておき、getValue(int row, int column)
 * メソッドで直接値を返します。
 * select category,series1,series2,series3,･･･ from ･･･
 * series の横持ち(標準と同じ) 対応です。
 * category カラムの値は、カテゴリのラベルになり、series1,2,3 のラベルがシリーズラベル、値が
 * seriesの値になります。
 *
 * カテゴリのカラー名の指定を行う場合、最後のカラムが、カラー名の文字列になります。
 * select category,series1,series2,series3,･･･,color from ･･･
 * color文字列の検索結果は、Dataset には含まれません。
 *
 * なお、Colorコードは、このクラスで作成しますが、Renderer に与える必要があります。
 * 通常のRenderer には、categoryにカラーを指定する機能がありませんので、HybsBarRenderer に
 * setCategoryColor( Color[] ) メソッドを用意します。(正確には、HybsDrawItem インターフェース)
 * このRenderer で、getItemPaint( int  , int )メソッドをオーバーライドすることで、カテゴリごとの
 * 色を返します。
 *
 * @og.rev 3.8.9.2 (2007/07/28) 新規作成
 *
 * @version  0.9.0  2001/05/05
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.1,
 */
public interface HybsDataset extends CategoryDataset {

	/**
	 * レンジオブジェクトを取得します。(独自メソッド)
	 *
	 * @return	レンジオブジェクト
	 */
	Range getRange() ;

	/**
	 * CategoryDataset を構築するに当たり、初期パラメータを設定します。
	 *
	 * @og.rev 6.0.2.2 (2014/10/03) 新規追加
	 *
	 * @param lbls  シリーズのラベル名配列
	 * @param isColCate  カテゴリのカラー名の指定有無(true:使用する)
	 * @param isPareto   パレート図用のDatasetとして処理するかどうか(true:処理する)
	 */
	void initParam( final String[] lbls , final boolean isColCate , final boolean isPareto );

	/**
	 * コネクションと、SQL文字列から、CategoryDataset のデータを作成します。
	 * 元となる処理は、org.jfree.data.jdbc.JDBCCategoryDataset#executeQuery( Connection,String ) です。
	 *
	 * このメソッドでは、先に #initParam(String[],boolean,isPareto) のパラメータを使用して
	 * 検索した結果のデータを加工、処理します。
	 * また、内部的に、データをキャッシュする事と、データ範囲を示す レンジオブジェクト を作成します。
	 *
	 * @og.rev 6.0.2.2 (2014/10/03) 新規追加
	 *
	 * @param con  コネクション
	 * @param query  SQL文字列
	 *
	 * @throws SQLException データベースアクセス時のエラー
	 * @see		org.jfree.data.jdbc.JDBCCategoryDataset#executeQuery( Connection,String )
	 */
	void execute( final Connection con, final String query ) throws SQLException ;

	/**
	 * DBTableModelオブジェクトから、CategoryDataset のデータを作成します。
	 * openGionの独自処理メソッドです。
	 *
	 * このメソッドでは、先に #initParam(String[],boolean,isPareto) のパラメータを使用して
	 * 検索した結果のデータを加工、処理します。
	 * また、内部的に、データをキャッシュする事と、データ範囲を示す レンジオブジェクト を作成します。
	 *
	 * @og.rev 6.0.2.2 (2014/10/03) 新規追加
	 *
	 * @param table DBTableModelオブジェクト
	 * @see		#execute( Connection,String )
	 */
	void execute( final DBTableModel table ) ;

	/**
	 * categoryカラー配列を取得します。(独自メソッド)
	 *
	 * このクラスは、一番最後のカラムを、色文字列として処理し、categoryにColorを指定できます。
	 * select文で指定されていなかった場合は、null を返します。
	 *
	 * select category,series1,series2,series3,･･･,color from ･･･
	 *
	 * なお、Colorコードは、このクラスで作成しますが、Renderer に与える必要があります。
	 * 通常のRenderer には、categoryにカラーを指定する機能がありませんので、HybsBarRenderer に
	 * setCategoryColor( Color[] ) メソッドを用意します。(正確には、HybsDrawItem インターフェース)
	 * このRenderer で、getItemPaint( int  , int )メソッドをオーバーライドすることで、カテゴリごとの
	 * 色を返します。
	 * この設定を行うと、シリーズは、カテゴリと同一色になります。
	 *
	 * @og.rev 6.0.2.2 (2014/10/03) 新規追加
	 *
	 * @return	categoryカラー配列(なければ null)
	 */
	Color[] getCategoryColor() ;
}
