package jp.sf.amateras.functions.utils;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import jp.sf.amateras.functions.filter.FunctionsFilter;

/**
 * Java Standard EL Functionsの各クラスが<code>HttpServletRequest</code>にアクセスするためのユーティリティです。
 * <p>
 * このユーティリティを使用するには{@link FunctionsFilter}を<tt>web.xml</tt>に適切に設定する必要があります。
 * 
 * @author Naoki Takezoe
 */
public class RequestUtils {
	
	private static ThreadLocal<HttpServletRequest> requests = new ThreadLocal<HttpServletRequest>();
	
	/**
	 * 現在のスレッドで使用する<code>HttpServletRequest</code>オブジェクトをセットします。
	 * 
	 * @param request 現在のスレッドで使用する<code>HttpServletRequest</code>オブジェクト
	 */
	public static void setRequest(HttpServletRequest request){
		requests.set(request);
	}
	
	/**
	 * 現在のスレッドで使用する<code>HttpServletRequest</code>オブジェクトを取得します。
	 * 
	 * @return 現在のスレッドで使用する<code>HttpServletRequest</code>オブジェクト
	 */
	public static HttpServletRequest getRequest(){
		return requests.get();
	}
	
	public static String getEncoding(){
		HttpServletRequest request = getRequest();
		if(request != null){
			return request.getCharacterEncoding();
		}
		return null;
	}
	
	/**
	 * 現在のスレッドで使用する<code>HttpServletRequest</code>オブジェクトを削除します。
	 * スレッドでの処理が終了する前に必ずこのメソッドを呼び出してください。
	 */
	public static void removeRequest(){
		requests.remove();
	}
	
	/**
	 * リクエスト、セッション、サーブレットコンテキストの順で属性を検索し、
	 * 最初に見つかったオブジェクトを返却します。
	 * 
	 * @param name リクエスト、セッション、サーブレットコンテキストの属性名
	 * @return リクエスト、セッション、サーブレットコンテキストのいずれかの属性にセットされたオブジェクト
	 */
	public static Object getAttribute(String name){
		HttpServletRequest request = RequestUtils.getRequest();
		
		// まずはリクエストから検索
		Object obj = request.getAttribute(name);
		
		if(obj == null){
			// リクエストに見つからなければセッションを検索
			HttpSession session = request.getSession();
			obj = session.getAttribute(name);
			
			if(obj == null){
				// セッションにも見つからなかったらServletContextを検索
				obj = session.getServletContext().getAttribute(name);
			}
		}
		
		return obj;
	}
	
}
