/**
 * @file doxygen_firmware.c
 * @author Shinichiro Nakamura
 * @brief Doxygen用のファイル。ファームウェアに関するトピックを記述してある。
 * @details
 *
 * @page firmware ファームウェアに関するトピック
 *
 * ファームウェアに関する既知の問題点については @ref problems_firmware をご覧下さい。
 *
 * @section task_structure_design タスク設計
 *
 * @subsection task_userinput ユーザインプットタスク(task_userinput)
 *
 *  ユーザインプットタスクは４系統のユーザ入力の変化を観察する
 *  タスクです。
 *  変化があればデータキューを介してシステムコントロールタスクに
 *  通知されます。
 *
 * @subsection task_menu メニュータスク(task_menu)
 *
 *  メニュータスクはユーザにシステムのサービスを提示し、
 *  ユーザからの要求をシステムに伝達する役目を果たします。
 *
 *  ユーザは各タスクから次のマクロのみで指示することができます。
 *
 *  @code
 *  USERMSG(device, value);
 *  @endcode
 *
 *  - deviceは以下から選択できます。
 *    - スイッチ: SW0, SW1, SW2, SW3
 *    - ボリューム: VOL0, VOL1, VOL2, VOL3
 *    .
 *  .
 *
 *  - Typeはビットマッピングです。
 *    - 長押し状態フラグ：SW_LONG_PUSH
 *    .
 *  .
 *
 *  このタスクへの指示はデータキューを介して行われます。
 *  データキュー内部データ構造は以下のようになっています。
 *
 *  <pre>
 *  +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+------------------+
 *  |15|14|13|12|11|10| 9| 8| 7| 6| 5| 4| 3| 2| 1| 0| Description      |
 *  +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+------------------+
 *  |                 |<-          [9:0]          ->| Value            |
 *  +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+------------------+
 *  |           |11:10|                             | Type             |
 *  +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+------------------+
 *  |<-[15:12]->|                                   | Device           |
 *  +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+------------------+
 *  </pre>
 *
 * @subsection task_led LEDタスク(task_led)
 *
 *  LEDタスクはLEDの点灯を制御するタスクです。
 *
 *  点灯は各タスクから次のマクロのみで指示することができます。
 *
 *  @code
 *  LEDMSG(target, control);
 *  @endcode
 *
 *  - targetは以下から選択できます。
 *    - デバッグ用LED: DBLED0, DBLED1, DBLED2, DBLED3
 *    - スイッチ用LED: SWLED0, SWLED1, SWLED2, SWLED3
 *    .
 *  .
 *
 *  - controlは以下から選択できます。
 *    - LED点灯：LEDON
 *    - LED消灯：LEDOFF
 *    .
 *  .
 *
 *  このタスクへの指示はデータキューを介して行われます。
 *  データキュー内部データ構造は以下のようになっています。
 *
 *  <pre>
 *  +---+---+---+---+---+---+---+---+-------------+
 *  | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 | Description |
 *  +---+---+---+---+---+---+---+---+-------------+
 *  |               |<-   [3:0]   ->| Target      |
 *  +---+---+---+---+---+---+---+---+-------------+
 *  |   | x   x   x |               | Reserved    |
 *  +---+---+---+---+---+---+---+---+-------------+
 *  |[7]|                           | Control     |
 *  +---+---+---+---+---+---+---+---+-------------+
 *  </pre>
 *
 * @subsection task_display ディスプレイタスク(task_display)
 *  ディスプレイタスクは有機ELディスプレイを制御するタスクです。
 *
 *  ディスプレイの制御は以下のマクロを使って行うことができます。
 *
 *  @code
 *  DISP_CLEAR(R, G, B);
 *  DISP_LINE(X1, Y1, X2, Y2, R, G, B);
 *  DISP_BOX(X1, Y1, X2, Y2, R, G, B);
 *  DISP_FILLBOX(X1, Y1, X2, Y2, R1, G1, B1, R2, G2, B2);
 *  DISP_TEXT(X, Y, R, G, B, "TEXT");
 *  DISP_BMPFILE("0:FILENAME");
 *  DISP_AUDIO_LEVELMETER(L, R);
 *  @endcode
 *
 *  タスク間はメールボックスでやりとりされます。
 *  マクロはこれを隠蔽した実装としました。
 *  メールボックスで陥りがちな同期問題を未然に防ぐ対策として
 *  メモリプール管理機能を使ってデータ領域に対するアクセスを
 *  管理しています。
 *
 *  今回の実装では受信側が処理を完了するまで送信側が次の処理に
 *  遷移しない実装にしてあります。
 *  今回のタスク設計ではメニュータスクがブロックの対象となります。
 *
 * @subsection task_init 初期化タスク(task_init)
 * 初期化するタスクです。
 *
 * @subsection task_ntshell ナチュラルタイニーシェルタスク(task_ntshell)
 *  システムをコンソールから制御することのできるインターフェース
 *  タスクです。
 *
 * @subsection task_audio オーディオタスク(task_audio)
 *  オーディオを処理するためのタスクです。
 *
 *  このタスクにはパラメータを指定するためのマクロがあります。
 *
 *  @code
 *  AUDIO_PARAM(TARGET,VALUE);
 *  @endcode
 *
 *  - TARGETとVALUEは以下から選択します。
 *    - TARGET: VAR0, VAR1, VAR2, VAR3
 *    - VALUE : The value of the target.
 *    .
 *  .
 *
 *  <pre>
 *  +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+------------------+
 *  |15|14|13|12|11|10| 9| 8| 7| 6| 5| 4| 3| 2| 1| 0| Description      |
 *  +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+------------------+
 *  |           |<-            [11:0]             ->| Value            |
 *  +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+------------------+
 *  |<-[15:12]->|                                   | Target           |
 *  +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+------------------+
 *  </pre>
 *
 *  VAR0, VAR1, VAR2, VAR3の各値をどのように使うのかは実装される
 *  オーディオエフェクトに依存します。
 *  詳しくはaudio_effect.cを御参照下さい。
 *
 * @section application_examples アプリケーション例
 *
 * - オーディオエフェクタ
 * - メトロノーム
 * - 多機能時計
 * - ピンボールゲーム
 *
 * @section task_priority プライオリティ
 *
 * タスクプライオリティに関しては以下のように定めた。
 * 一般にRTOSのプライオリティはサーバ＞クライアントとなる。
 * これに層別の視点を加えてプライオリティを決定した。
 *
 * <table>
 *   <tr>
 *     <th>カテゴリ</th>
 *     <th>タスク</th>
 *     <th>プライオリティ</th>
 *   </tr>
 *   <tr>
 *     <td>リアルタイム・タスク</td>
 *     <td>@ref task_audio</td>
 *     <td>10</td>
 *   </tr>
 *   <tr>
 *     <td rowspan="2">サーバ・タスク</td>
 *     <td>@ref task_display</td>
 *     <td>11</td>
 *   </tr>
 *   <tr>
 *     <td>@ref task_led</td>
 *     <td>11</td>
 *   </tr>
 *   <tr>
 *     <td rowspan="2">クライアント・タスク</td>
 *     <td>@ref task_menu</td>
 *     <td>12</td>
 *   </tr>
 *   <tr>
 *     <td>@ref task_userinput</td>
 *     <td>12</td>
 *   </tr>
 *   <tr>
 *     <td>イニシャライザ</td>
 *     <td>@ref task_init</td>
 *     <td>13</td>
 *   </tr>
 *   <tr>
 *     <td>デバッガ</td>
 *     <td>@ref task_ntshell</td>
 *     <td>14</td>
 *   </tr>
 * </table>
 *
 * @section sec_audio_effect_refactoring オーディオ処理の効率改善。
 *
 * ここではオーディオ処理の効率改善について記す。
 *
 * @subsection sec_audio_effect_refactoring1 オリジナルの設計
 *
 * オリジナルの実装では以下のような処理となっていた。
 *
 * - DMA転送されたオーディオバッファサイズ分のデータがrxbufに入っている。
 * - 並び替えながらrxbufからaudio_data.inputBufferにコピーする。
 * - オーディオ処理を実行する。
 *   - ここでは処理に応じてaudio_data.inputBufferからaudio_data.outputBufferへのコピーが発生する。
 *   .
 * - 並び替えながらaudio_data.outputBufferからtxbufにコピーする。
 * .
 *
 * 図を用いて整理すると以下のようになる。
 *
 * <pre>
 * +-----+      +-----------+      +------------+      +-----+
 * |rxbuf|  ->  |inputBuffer|  ->  |outputBuffer|  ->  |txbuf|
 * +-----+      +-----------+      +------------+      +-----+
 * </pre>
 *
 * 要するに主記憶上におけるメモリコピーが少なくとも３回発生している。
 *
 * また、このメモリコピーはforループで実装されており、３回のforループによる性能への影響も気になる。
 *
 * 通常、性能という観点で見た場合、メモリコピーや重複したforループは性能劣化の主要な要因のひとつとなる。
 *
 * そこで、今回は上記処理の効率改善を行う。
 *
 * まず初めに、最小限の処理について考えてみる。
 *
 * <pre>
 * +-----+      +-----+
 * |rxbuf|  ->  |txbuf|
 * +-----+      +-----+
 * </pre>
 *
 * 入力をそのまま出力に伝達する場合、単なるメモリコピーで良い。
 *
 * 入力に何らかの処理を加え、出力に伝える場合でもこの入出力間のメモリコピーの間に何らかの処理を追加するだけで済むので、本質的に上記と変わらない。
 *
 * オリジナルの実装ではコーデックのデータ形式を鑑みて工夫がしてある。
 *
 * この処理を順次見ていくことにする。
 *
 * まず初めに入力されたデータを内部で扱いやすい形式にメモリコピーする。
 *
 * これはrxbufからinputBufferへのコピーである。
 *
 * <pre>
 * +--------------------------+
 * |+-----+      +-----------+|      +------------+      +-----+
 * ||rxbuf|  ->  |inputBuffer||  ->  |outputBuffer|  ->  |txbuf|
 * |+-----+      +-----------+|      +------------+      +-----+
 * +--------------------------+
 * </pre>
 *
 * 次にオーディオの処理を実行する。
 *
 * これはinputBufferからoutputBufferへのコピーである。
 *
 * オーディオの処理を実装する過程で、ここに様々な演算が入ることになる。
 *
 * <pre>
 *              +---------------------------------+
 * +-----+      |+-----------+      +------------+|      +-----+
 * |rxbuf|  ->  ||inputBuffer|  ->  |outputBuffer||  ->  |txbuf|
 * +-----+      |+-----------+      +------------+|      +-----+
 *              +---------------------------------+
 * </pre>
 *
 * 最後に結果を出力バッファに書き込む。
 *
 * outputBufferからtxbufへのコピーである。
 *
 * これはオーディオバッファの内容を、都合の良い形式に前段で並び替えた結果発生する作業である。
 *
 * <pre>
 *                                 +---------------------------+
 * +-----+      +-----------+      |+------------+      +-----+|
 * |rxbuf|  ->  |inputBuffer|  ->  ||outputBuffer|  ->  |txbuf||
 * +-----+      +-----------+      |+------------+      +-----+|
 *                                 +---------------------------+
 * </pre>
 *
 * まとめると以下のようになる。
 * - コーデックから得られたデータ形式は扱いにくいので並び替える。
 * - 並び替えは主記憶上でMCUが実行する。
 * - 並び替えたデータは、コーデックがそのまま扱えないので再変換する。
 * .
 *
 * コードブロックは以下のようになっていた。(一部はオリジナルと少し異なる。)
 *
 * @code
 *  index = 0;
 *  for (sample = 0; sample < AUDIOBUFSIZE / 2; sample++) {
 *      for (ch = 0; ch < 2; ch++) {
 *          audio_data.inputBuffer[ch][sample] = rxbuf[index++];
 *      }
 *  }
 *  audio_effect_through(
 *          &effect_param,
 *          audio_data.inputBuffer,
 *          audio_data.outputBuffer,
 *          AUDIOBUFSIZE / 2);
 *  index = 0;
 *  for (sample = 0; sample < AUDIOBUFSIZE / 2; sample++) {
 *      for (ch = 0; ch < 2; ch++) {
 *          txbuf[index++] = audio_data.outputBuffer[ch][sample];
 *      }
 *  }
 * @endcode
 *
 * オーディオエフェクト処理の前後でデータ形式変換を行なっていることがわかる。
 *
 * 前段と後段で各((AUDIOBUFSIZE / 2) x 2)回分のメモリコピーを行なっている。
 *
 * 実際にオーディオ処理関数内部の実装も見る。(一部はオリジナルと少し異なる。)
 *
 * @code
 *  void audio_effect_through(
 *          effect_param_t *param,
 *          AUDIOSAMPLE input[2][AUDIOBUFSIZE / 2],
 *          AUDIOSAMPLE output[2][AUDIOBUFSIZE / 2],
 *          int count)
 *  {
 *      int i;
 *
 *      const int var0 = param->var0;
 *      const int var1 = param->var1;
 *      for (i = 0; i < count; i++)
 *      {
 *          output[LCH][i] = (input[LCH][i] >> 10) * var0;
 *          output[RCH][i] = (input[RCH][i] >> 10) * var1;
 *      }
 *  }
 * @endcode
 *
 * ここで上位から渡されるcountは(AUDIOBUFSIZE / 2)である。
 *
 * よって、ここでも((AUDIOBUFSIZE / 2) x 2)回分のメモリコピーを行なっていることになる。
 *
 * @subsection sec_audio_effect_refactoring2 改善の提案
 *
 * ここまではオリジナルの設計について述べた。
 *
 * それでは実際にオーディオ処理の効率改善について述べる。
 *
 * 基本的な思想は以下の通りである。
 * - 主記憶上におけるメモリコピーは性能に対して著しい劣化を伴う。
 * - より多くの処理を実現するためにはメモリコピーを排除すれば良い。
 * - メモリコピーを行なっている主な理由はデータ形式変換である。
 * - データ形式変換が不要となるような枠組みを用意すれば、データ形式変換が不要となるはずである。
 * - データ形式変換が不要となれば、必要となるバッファも削減することができ、RAM容量という観点から見ても有利である。
 * .
 *
 * 要するに「データ形式変換」を実現しながらも、「メモリコピー」を発生させないという矛盾を解決すれば良い事になる。
 * この中でforループについても削減可能と判断した。
 *
 * オリジナルの実装ではオーディオ処理関数に渡るデータ形式が重要であった。
 *
 * この点は改善案でも特に変わるものではない。
 *
 * オリジナルと異なるのはその実現手法である。
 *
 * ここで実際に改善したコードを示す。
 *
 * @code
 *  for (index = 0; index < AUDIOBUFSIZE; index+=2) {
 *      audio_effect_through(
 *              &effect_param,
 *              rxbuf + (index + 0), rxbuf + (index + 1),
 *              txbuf + (index + 0), txbuf + (index + 1));
 *  }
 * @endcode
 *
 * オーディオ処理関数へL-Rのステレオデータを揃えて渡す部分はコールバック関数とし、オーディオ処理関数内部で直接出力データを格納させる形式とした。
 *
 * これによりオリジナルに存在した前段と後段でのメモリコピーを排除できる。
 *
 * オリジナルの実装では、前段と後段で(AUDIOBUFSIZE / 2) x 2回分のメモリコピーを必要としていたが、これを置きかえる形となる。
 *
 * これによりinputBufferとoutputBufferも不要となった。
 *
 * audio_effect_throught関数はオーディオデータをスルーコピーする関数である。
 *
 * @code
 *  void audio_effect_through(
 *          const effect_param_t *param,
 *          const AUDIOSAMPLE *in_left,
 *          const AUDIOSAMPLE *in_right,
 *          AUDIOSAMPLE *out_left,
 *          AUDIOSAMPLE *out_right)
 *  {
 *      const int var0 = param->var0;
 *      const int var1 = param->var1;
 *      *out_left = ((*in_left) >> 10) * var0;
 *      *out_right = ((*in_right) >> 10) * var1;
 *  }
 * @endcode
 *
 * オーディオエフェクト関数には１サンプル毎に処理を依頼する。
 *
 * 処理結果は関数に渡されたバッファへのポインタを用いて直接格納する。
 *
 * 上記により中間バッファを排除しながらも、渡されるデータ形式はL-Rのステレオで揃っているという状況を作ることができる。
 *
 * また、バッファサイズ分のforループも１回で済むようになり、効率改善が期待できる。
 *
 * @subsection sec_audio_effect_refactoring3 改善の効果
 *
 * ここで実際の処理時間に与える影響を調査した。
 *
 * 処理時間はオーディオ処理ブロックに差し掛かったところでGPIOをハイレベルにし、オシロスコープにより観測した。
 *
 * ここでの処理対象は１オーディオサンプルブロックでAUDIOBUFSIZE分のデータである。
 *
 * 初めにオリジナル実装でオーディオスルーにかかっている処理時間を示す。
 *
 * 約70[us]の時間を要している。
 *
 * @image html task_audio_before.bmp
 *
 * 次に示すのは改良した実装でオーディオスルーを行なった場合の処理時間である。
 *
 * 約25[us]の時間で処理できている。
 *
 * @image html task_audio_after.bmp
 *
 * 上記のようにオリジナルの実装に対して35.7%の時間で同等の処理が実現できる事が確認できた。
 *
 * 削減できた約45[us]は別の演算に割り当てることができる。
 *
 * 従来より高度な演算も可能となる。
 *
 * @subsection sec_audio_effect_refactoring4 改善のまとめ
 *
 * 本改善提案によれば、処理を非常に簡単に記述することが出来る上、主記憶上のメモリコピーが必要ない。
 *
 * このため大幅な処理時間の短縮が可能となった。
 *
 * これによりMCUの演算能力を効率的にオーディオ処理に割り当てる事が可能となる。
 *
 * オーディオエフェクト関数を同じパラメータで作成すれば、関数ポインタの切り替えのみでオーディオエフェクト処理を切り替えることが可能となる。
 *
 * 通常、オーディオ処理では特定サンプルに対して、時間軸方向前後のデータも用いてフィルタリングを行なう。
 *
 * 提案手法ではオーディオ処理関数に渡ってくるデータは１サンプル分のみであるが、オーディオエフェクト関数内部で静的メモリを保持し、バッファリングしながら処理をすれば、時間軸の前後方向のデータも用いて処理することが可能である。
 *
 * なお、実験結果はコンパイラの最適化オプションを外した状態で行なった。最適化を施すことで、場合によってはより多くのパフォーマンス改善が得られる可能性もある。
 */

