﻿/*
 * FormUtauVoiceConfig.cs
 * Copyright (c) 2009 kbinani
 *
 * This file is part of Boare.Cadencii.
 *
 * Boare.Cadencii is free software; you can redistribute it and/or
 * modify it under the terms of the GPLv3 License.
 *
 * Boare.Cadencii is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
#if VER22
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.IO;
using System.Threading;
using System.Windows.Forms;

using Boare.Lib.AppUtil;
using Boare.Lib.Media;
using bocoree;

namespace Boare.Cadencii {

    public partial class FormUtauVoiceConfig : Form {
        enum MouseMode {
            None,
            MiddleDrag,
            MoveOffset,
            MoveConsonant,
            MoveBlank,
            MovePreUtterance,
            MoveOverlap,
        }

        enum FlagType : int {
            Offset = 0,
            Consonant = 1,
            Blank = 2,
            PreUtterance = 3,
            Overlap = 4,
        }

        const int LINE_HEIGHT = 20;
        const int TRACKBAR_MAX = 35;
        const int TRACKBAR_MIN = 15;
        const int ORDER = 10;

        private List<WaveDrawContext> m_drawer = new List<WaveDrawContext>();
        private int m_index = -1;
        private double m_px_per_sec = 1000;
        private float m_start_to_draw = 0.0f;
        private SolidBrush m_brs_offset = new SolidBrush( Color.FromArgb( 192, 192, 255 ) );
        private SolidBrush m_brs_consonant = new SolidBrush( Color.FromArgb( 255, 192, 255 ) );
        private SolidBrush m_brs_blank = new SolidBrush( Color.FromArgb( 192, 192, 255 ) );
        private Pen m_pen_overlap = new Pen( Color.FromArgb( 0, 255, 0 ) );
        private Pen m_pen_preutterance = new Pen( Color.FromArgb( 255, 0, 0 ) );
        private int m_offset;
        private int m_consonant;
        private int m_blank;
        private int m_pre_utterance;
        private int m_overlap;
        private string m_file;
        private Rectangle[] m_flag_box = new Rectangle[5];
        private float m_length;
        private int m_font_draw_offset = 0;
        private string m_font_name = "";
        private float m_font_size = -1;
        private MouseMode m_mode = MouseMode.None;
        private int m_move_init;
        /// <summary>
        /// MouseDownイベントが起きた座標
        /// </summary>
        private Point m_mouse_downed;
        /// <summary>
        /// mouseDownイベントが起きた時点での、m_start_to_drawの値
        /// </summary>
        private float m_mouse_downed_start_to_draw;
        private MediaPlayer m_player;
        private string m_oto_ini = "";
        private int m_last_preview = 0;
        private bool m_edited = false;
        private int m_trackbar_value = 20;
        private Thread m_mouse_hover_generator = null;
        private bool m_cancel_required = false;

        public FormUtauVoiceConfig() {
            InitializeComponent();
            pictWave.MouseWheel += new MouseEventHandler( pictWave_MouseWheel );
            splitContainerIn.Panel1.BorderStyle = BorderStyle.None;
            splitContainerIn.Panel2.BorderStyle = BorderStyle.FixedSingle;
            splitContainerIn.Panel2.BorderColor = SystemColors.ControlDark;

            splitContainerOut.Panel1.BorderStyle = BorderStyle.None;
            splitContainerOut.Panel2.BorderStyle = BorderStyle.FixedSingle;
            splitContainerOut.Panel2.BorderColor = SystemColors.ControlDark;

            splitContainerIn.Panel1.Controls.Add( listFiles );
            listFiles.Dock = DockStyle.Fill;
            splitContainerIn.Panel2.Controls.Add( panel1 );
            panel1.Dock = DockStyle.Fill;

            splitContainerOut.Panel1.Controls.Add( splitContainerIn );
            splitContainerIn.Dock = DockStyle.Fill;
            splitContainerOut.Panel2.Controls.Add( panel2 );
            panel2.Dock = DockStyle.Fill;

            splitContainerOut.Dock = DockStyle.Fill;

            UpdateScale();
            m_player = new MediaPlayer();
            ApplyLanguage();
        }

        public FormConfigUtauVoiceConfig CurrentConfig {
            get {
                FormConfigUtauVoiceConfig ret = new FormConfigUtauVoiceConfig();
                ret.Size = this.Size;
                ret.InnerSplitterDistancePercentage = splitContainerIn.SplitterDistance / (float)splitContainerIn.Width * 100.0f;
                ret.OuterSplitterDistancePercentage = splitContainerOut.SplitterDistance / (float)splitContainerOut.Height * 100.0f;
                ret.WaveViewScale = m_trackbar_value;
                return ret;
            }
            set {
                this.Size = value.Size;
                splitContainerIn.SplitterDistance = (int)(splitContainerIn.Width * value.InnerSplitterDistancePercentage / 100.0f);
                splitContainerOut.SplitterDistance = (int)(splitContainerOut.Height * value.OuterSplitterDistancePercentage / 100.0f);
                if ( value.WaveViewScale < TRACKBAR_MIN ) {
                    m_trackbar_value = TRACKBAR_MIN;
                } else if ( TRACKBAR_MAX < value.WaveViewScale ) {
                    m_trackbar_value = TRACKBAR_MAX;
                } else {
                    m_trackbar_value = value.WaveViewScale;
                }
                UpdateScale();
            }
        }

        public void ApplyFont( Font font ) {
            this.Font = font;
            foreach ( Control c in this.Controls ) {
                Misc.ApplyFontRecurse( c, font );
            }
            Misc.ApplyToolStripFontRecurse( menuFile, font );
        }

        private void HoverWaitThread(){
            Thread.Sleep( SystemInformation.MouseHoverTime );
            EventHandler eh = new EventHandler( pictWave_MouseHover );
            this.Invoke( eh, pictWave, new EventArgs() );
        }

        public void ApplyLanguage() {
            UpdateFormTitle();
            
            menuFile.Text = _( "File" ) + "(&F)";
            menuFileOpen.Text = _( "Open" ) + "(&O)";
            menuFileSave.Text = _( "Save" ) + "(&S)";
            menuFileSaveAs.Text = _( "Save As" ) + "(&A)";
            menuFileQuit.Text = _( "Quit" ) + "(&Q)";

            lblFileName.Text = _( "File Name" );
            lblAlias.Text = _( "Alias" );
            lblOffset.Text = _( "Offset" );
            lblConsonant.Text = _( "Consonant" );
            lblBlank.Text = _( "Blank" );
            lblPreUtterance.Text = _( "Pre Utterance" );
            lblOverlap.Text = _( "Overlap" );

            columnHeaderFilename.Text = _( "File Name" );
            columnHeaderAlias.Text = _( "Alias" );
            columnHeaderOffset.Text = _( "Offset" );
            columnHeaderConsonant.Text = _( "Consonant" );
            columnHeaderBlank.Text = _( "Blank" );
            columnHeaderPreUtterance.Text = _( "Pre Utterance" );
            columnHeaderOverlap.Text = _( "Overlap" );

            try {
                openFileDialog.Filter = _( "Voice DB Config(*.ini)|*.ini" ) + "|" + _( "All Files(*.*)|*.*" );
                saveFileDialog.Filter = _( "Voice DB Config(*.ini)|*.ini" ) + "|" + _( "All Files(*.*)|*.*" );
            } catch {
                openFileDialog.Filter = "Voice DB Config(*.ini)|*.ini|All Files(*.*)|*.*";
                saveFileDialog.Filter = "Voice DB Config(*.ini)|*.ini|All Files(*.*)|*.*";
            }
        }

        private void UpdateFormTitle() {
            string f = m_oto_ini;
            if ( f == "" ){
                f = "Untitled";
            }
            string title = _( "Voice DB Config" ) + " - " + f + (m_edited ? " *" : "");
            if ( title != this.Text ) {
                this.Text = title;
            }
        }

        private static string _( string id ) {
            return Messaging.GetMessage( id );
        }

        private void pictWave_MouseWheel( object sender, MouseEventArgs e ) {
            int draft = hScroll.Value - e.Delta / 120 * hScroll.LargeChange / 2;
            if ( draft > hScroll.Maximum - hScroll.LargeChange + 1 ) {
                draft = hScroll.Maximum - hScroll.LargeChange + 1;
            }
            if ( draft < 0 ) {
                draft = 0;
            }
            if ( draft != hScroll.Value ) {
                hScroll.Value = draft;
            }
        }

        public void Open( string oto_ini_path ) {
            m_oto_ini = oto_ini_path;
            UpdateFormTitle();
            m_cancel_required = false;
            listFiles.Items.Clear();
            bgWork.RunWorkerAsync( oto_ini_path );
        }

        private void AddItem( string[] columns ) {
            ListViewItem item = new ListViewItem( columns );
            listFiles.Items.Add( item );
        }

        private void bgWork_DoWork( object sender, DoWorkEventArgs e ) {
            string oto_ini_path = (string)e.Argument;
            int c = m_drawer.Count;
            for ( int i = 0; i < c; i++ ) {
                m_drawer[i].Dispose();
            }
            m_drawer.Clear();

            if ( !File.Exists( oto_ini_path ) ) {
                return;
            }

            string dir = Path.GetDirectoryName( oto_ini_path );
#if DEBUG
            Console.WriteLine( "FormUtauVoiceConfig.Open" );
#endif
            using ( cp932reader sr = new cp932reader( oto_ini_path ) ) {
                while ( sr.Peek() >= 0 ) {
                    if ( m_cancel_required ) {
                        break;
                    }
                    string line = sr.ReadLine();
                    int eq = line.IndexOf( '=' );
                    if ( eq <= 0 ) {
                        continue;
                    }
                    string f = line.Substring( 0, eq );
#if DEBUG
                    Console.WriteLine( "f=" + f );
#endif
                    line = line.Substring( eq + 1 );
#if DEBUG
                    Console.WriteLine( "line=" + line );
#endif
                    string[] spl = line.Split( ',' );
                    string file = Path.Combine( dir, f );
                    if ( File.Exists( file ) ) {
                        WaveDrawContext wdc = new WaveDrawContext( file );
                        wdc.Name = f;
                        string f2 = f.Replace( ".wav", "_wav.frq" );
                        string freq = Path.Combine( dir, f2 );
                        bool freq_exists = File.Exists( freq );
                        if ( freq_exists ){
                            wdc.Freq = UtauFreq.FromFrq( freq );
                        }
                        m_drawer.Add( wdc );
                        List<string> columns = new List<string>( spl );
                        columns.Insert( 0, f );
                        columns.Add( freq_exists ? "○" : "" );
                        BSimpleDelegate<string[]> deleg = new BSimpleDelegate<string[]>( AddItem );
                        this.Invoke( deleg, new object[] { columns.ToArray() } );
                    }
                }
            }
            m_edited = false;
        }

        private void listFiles_SelectedIndexChanged( object sender, EventArgs e ) {
            if ( listFiles.SelectedIndices.Count <= 0 ){
                return;
            }
            int index = listFiles.SelectedIndices[0];
            string name = listFiles.Items[index].Text;
            int c = m_drawer.Count;
            m_index = -1;
            for ( int i = 0; i < c; i++ ) {
                if ( name == m_drawer[i].Name ) {
                    m_index = i;
                    m_length = m_drawer[i].Length;
                    int c2 = listFiles.Items[m_index].SubItems.Count;
                    string[] spl = new string[c];
                    for ( int i2 = 0; i2 < c2; i2++ ) {
                        spl[i2] = listFiles.Items[m_index].SubItems[i2].Text;
                    }
                    bool old = Edited;
                    txtFileName.Text = spl[0];
                    m_file = spl[0];
                    txtAlias.Text = spl[1];
                    int o;
                    txtOffset.Text = spl[2];
                    if ( int.TryParse( txtOffset.Text, out o ) ) {
                        m_offset = o;
                    }
                    txtConsonant.Text = spl[3];
                    if ( int.TryParse( txtConsonant.Text, out o ) ) {
                        m_consonant = o;
                    }
                    txtBlank.Text = spl[4];
                    if ( int.TryParse( txtBlank.Text, out o ) ){
                        m_blank = o;
                    }
                    txtPreUtterance.Text = spl[5];
                    if ( int.TryParse( txtPreUtterance.Text, out o ) ) {
                        m_pre_utterance = o;
                    }
                    txtOverlap.Text = spl[6];
                    if ( int.TryParse( txtOverlap.Text, out o ) ) {
                        m_overlap = o;
                    }
                    Edited = old;
                    hScroll.Maximum = (int)(m_drawer[i].Length * 1000 * ORDER);
                    break;
                }
            }
            pictWave.Invalidate();
        }

        private void pictWave_Paint( object sender, PaintEventArgs e ) {
            if ( 0 <= m_index && m_index < m_drawer.Count ) {
                int c = listFiles.Items[m_index].SubItems.Count;
                string[] spl = new string[c];
                for ( int i = 0; i < c; i++ ) {
                    spl[i] = listFiles.Items[m_index].SubItems[i].Text;
                }
                e.Graphics.Clear( Color.White );
                int h = pictWave.Height;

                int x_offset = XFromSec( 0.0f );
                Rectangle rc = new Rectangle( x_offset, 0, (int)(m_offset / 1000.0f * m_px_per_sec), h );
                e.Graphics.FillRectangle( m_brs_offset, rc );

                int x_consonant = XFromSec( m_offset / 1000.0f );
                rc = new Rectangle( x_consonant, 0, (int)(m_consonant / 1000.0f * m_px_per_sec), h );
                e.Graphics.FillRectangle( m_brs_consonant, rc );

                int x_blank = XFromSec( m_drawer[m_index].Length - m_blank / 1000.0f );
                rc = new Rectangle( x_blank, 0, (int)(m_blank / 1000.0f * m_px_per_sec), h );
                e.Graphics.FillRectangle( m_brs_blank, rc );

                m_drawer[m_index].Draw( e.Graphics,
                                        Pens.Black,
                                        new Rectangle( 0, 0, pictWave.Width, h ),
                                        m_start_to_draw,
                                        (float)(m_start_to_draw + pictWave.Width / m_px_per_sec) );

                int x_overlap = XFromSec( m_overlap / 1000.0f );
                e.Graphics.DrawLine( m_pen_overlap, new Point( x_overlap, 0 ), new Point( x_overlap, h ) );

                int x_pre_utterance = XFromSec( m_pre_utterance / 1000.0f );
                e.Graphics.DrawLine( m_pen_preutterance, new Point( x_pre_utterance, 0 ), new Point( x_pre_utterance, h ) );

                Font font = AppManager.EditorConfig.BaseFont;

                int x_lastpreview = XFromSec( m_last_preview / 1000.0f );
                e.Graphics.DrawLine( Pens.Blue, new Point( x_lastpreview, 0 ), new Point( x_lastpreview, h ) );
                e.Graphics.DrawString( m_last_preview + " ms", font, Brushes.Black, new PointF( x_lastpreview + 1, 1 ) );

                e.Graphics.DrawString( m_file, font, Brushes.Black, new PointF( 1, LINE_HEIGHT ) );

                if ( font.Name != m_font_name || font.SizeInPoints != m_font_size ) {
                    m_font_draw_offset = Misc.GetStringDrawOffset( font ) - 1;
                    m_font_name = font.Name;
                    m_font_size = font.SizeInPoints;
                }

                using ( SolidBrush brs = new SolidBrush( Color.FromArgb( 180, Color.White ) ) ) {
                    rc = GetFlagRect( FlagType.Offset, font );
                    m_flag_box[0] = rc;
                    e.Graphics.FillRectangle( brs, rc );
                    e.Graphics.DrawRectangle( Pens.Black, rc );
                    e.Graphics.DrawString( _( "Offset" ) + ": " + m_offset + " ms",
                                           font, 
                                           Brushes.Black, 
                                           rc.X + 1,
                                           rc.Y + rc.Height * 0.5f - m_font_draw_offset );

                    rc = GetFlagRect( FlagType.Consonant, font );
                    m_flag_box[1] = rc;
                    e.Graphics.FillRectangle( brs, rc );
                    e.Graphics.DrawRectangle( Pens.Black, rc );
                    e.Graphics.DrawString( _( "Consonant" ) + ": " + m_consonant + " ms", 
                                           font,
                                           Brushes.Black,
                                           rc.X + 1,
                                           rc.Y + rc.Height * 0.5f - m_font_draw_offset );

                    rc = GetFlagRect( FlagType.Blank, font );
                    m_flag_box[2] = rc;
                    e.Graphics.FillRectangle( brs, rc );
                    e.Graphics.DrawRectangle( Pens.Black, rc );
                    e.Graphics.DrawString( _( "Blank" ) + ": " + m_blank + " ms",
                                           font,
                                           Brushes.Black,
                                           rc.X + 1,
                                           rc.Y + rc.Height * 0.5f - m_font_draw_offset );

                    rc = GetFlagRect( FlagType.PreUtterance, font );
                    m_flag_box[3] = rc;
                    e.Graphics.FillRectangle( brs, rc );
                    e.Graphics.DrawRectangle( Pens.Black, rc );
                    e.Graphics.DrawString( _( "Pre Utterance" ) + ": " + m_pre_utterance + " ms",
                                           font, 
                                           Brushes.Black, 
                                           rc.X + 1,
                                           rc.Y + rc.Height * 0.5f - m_font_draw_offset );

                    rc = GetFlagRect( FlagType.Overlap, font );
                    m_flag_box[4] = rc;
                    e.Graphics.FillRectangle( brs, rc );
                    e.Graphics.DrawRectangle( Pens.Black, rc );
                    e.Graphics.DrawString( _( "Overlap" ) + ": " + m_overlap + " ms",
                                           font, 
                                           Brushes.Black, 
                                           rc.X + 1,
                                           rc.Y + rc.Height * 0.5f - m_font_draw_offset );
                }
            }
        }

        private Rectangle GetFlagRect( FlagType type, Font font ) {
            int i = (int)type;
            int x = 0;
            string s = "";
            switch ( type ) {
                case FlagType.Offset:
                    x = XFromSec( m_offset / 1000.0f );
                    s = _( "Offset" ) + ": " + m_offset + " ms";
                    break;
                case FlagType.Consonant:
                    x = XFromSec( (m_offset + m_consonant) / 1000.0f );
                    s = _( "Consonant" ) + ": " + m_consonant + " ms";
                    break;
                case FlagType.Blank:
                    x = XFromSec( m_length - m_blank / 1000.0f );
                    s = _( "Blank" ) + ": " + m_blank + " ms";
                    break;
                case FlagType.PreUtterance:
                    x = XFromSec( m_pre_utterance / 1000.0f );
                    s = _( "Pre Utterance" ) + ": " + m_pre_utterance + " ms";
                    break;
                case FlagType.Overlap:
                    x = XFromSec( m_overlap / 1000.0f );
                    s = _( "Overlap" ) + ": " + m_overlap + " ms";
                    break;
            }
            SizeF size = Misc.MeasureString( s, font );
            return new Rectangle( x, LINE_HEIGHT * (i + 2), (int)(size.Width * 1.1f), (int)(size.Height * 1.1f) );
        }

        private int XFromSec( float sec ) {
            return (int)((sec - m_start_to_draw) * m_px_per_sec);
        }

        private float SecFromX( int x ) {
            return x / (float)m_px_per_sec + m_start_to_draw;
        }

        private void UpdateScale() {
            m_px_per_sec = 10.0 * Math.Pow( 10.0, m_trackbar_value / 10.0 );
            hScroll.LargeChange = (int)(pictWave.Width / m_px_per_sec * 1000 * ORDER);
            pictWave.Invalidate();
        }

        private void hScroll_ValueChanged( object sender, EventArgs e ) {
            m_start_to_draw = hScroll.Value / 1000.0f / ORDER;
            pictWave.Invalidate();
        }

        private void pictWave_MouseDown( object sender, MouseEventArgs e ) {
            m_mouse_downed = e.Location;
            m_mouse_downed_start_to_draw = m_start_to_draw;
            if ( m_mouse_hover_generator != null && m_mouse_hover_generator.IsAlive ) {
                m_mouse_hover_generator.Abort();
                while ( m_mouse_hover_generator.IsAlive ) {
                    Application.DoEvents();
                }
            }
            m_mouse_hover_generator = new Thread( new ThreadStart( this.HoverWaitThread ) );
            m_mouse_hover_generator.Start();
            if ( e.Button == MouseButtons.Middle ) {
                m_mode = MouseMode.MiddleDrag;
            } else if ( e.Button == MouseButtons.Left ) {
                m_mode = MouseMode.None;
                for ( int i = 0; i < m_flag_box.Length; i++ ) {
                    Rectangle rc = m_flag_box[i];
                    if ( AppManager.IsInRect( e.Location, rc ) ) {
                        switch ( i ) {
                            case 0:
                                m_mode = MouseMode.MoveOffset;
                                m_move_init = m_offset;
                                return;
                            case 1:
                                m_mode = MouseMode.MoveConsonant;
                                m_move_init = m_consonant;
                                return;
                            case 2:
                                m_mode = MouseMode.MoveBlank;
                                m_move_init = m_blank;
                                return;
                            case 3:
                                m_mode = MouseMode.MovePreUtterance;
                                m_move_init = m_pre_utterance;
                                return;
                            case 4:
                                m_mode = MouseMode.MoveOverlap;
                                m_move_init = m_overlap;
                                return;
                        }
                        break;
                    }
                }

                if ( 0 <= m_index && m_index < listFiles.Items.Count ){
                    string file = Path.Combine( Path.GetDirectoryName( m_oto_ini ), listFiles.Items[m_index].SubItems[0].Text );
                    if ( File.Exists( file ) && m_player.SoundLocation != file ) {
                        m_player.Close();
                        m_player.Load( file );
                    }
                    double sec = SecFromX( e.X );
                    m_last_preview = (int)(sec * 1000);
                    m_player.PlayFrom( m_last_preview / 1000.0 );
                    pictWave.Invalidate();
                }
            }
        }

        private void pictWave_MouseMove( object sender, MouseEventArgs e ) {
            if ( m_mouse_hover_generator != null && m_mouse_hover_generator.IsAlive ) {
                m_mouse_hover_generator.Abort();
            }
            if ( m_mode == MouseMode.MiddleDrag ) {
                int dx = e.X - m_mouse_downed.X;
                int draft = (int)((m_mouse_downed_start_to_draw - dx / (float)m_px_per_sec) * 1000 * ORDER);
                if ( draft > hScroll.Maximum - hScroll.LargeChange + 1 ) {
                    draft = hScroll.Maximum - hScroll.LargeChange + 1;
                }
                if ( draft < 0 ) {
                    draft = 0;
                }
                if ( hScroll.Value != draft ) {
                    hScroll.Value = draft;
                }
            } else if ( m_mode == MouseMode.MoveOffset ) {
                int dx = e.X - m_mouse_downed.X;
                int draft = m_move_init + (int)(dx / m_px_per_sec * 1000);
                txtOffset.Text = draft.ToString();
            } else if ( m_mode == MouseMode.MoveConsonant ) {
                int dx = e.X - m_mouse_downed.X;
                int draft = m_move_init + (int)(dx / m_px_per_sec * 1000);
                txtConsonant.Text = draft.ToString();
            } else if ( m_mode == MouseMode.MoveBlank ) {
                int dx = e.X - m_mouse_downed.X;
                int draft = m_move_init - (int)(dx / m_px_per_sec * 1000);
                txtBlank.Text = draft.ToString();
            } else if ( m_mode == MouseMode.MovePreUtterance ) {
                int dx = e.X - m_mouse_downed.X;
                int draft = m_move_init + (int)(dx / m_px_per_sec * 1000);
                txtPreUtterance.Text = draft.ToString();
            } else if ( m_mode == MouseMode.MoveOverlap ) {
                int dx = e.X - m_mouse_downed.X;
                int draft = m_move_init + (int)(dx / m_px_per_sec * 1000);
                txtOverlap.Text = draft.ToString();
            }
        }

        private bool Edited {
            get {
                return m_edited;
            }
            set {
                bool old = m_edited;
                m_edited = value;
                if ( m_edited != old ) {
                    UpdateFormTitle();
                }
            }
        }

        private void pictWave_MouseUp( object sender, MouseEventArgs e ) {
            m_mode = MouseMode.None;
        }

        private void txtAlias_TextChanged( object sender, EventArgs e ) {
            if ( 0 <= m_index && m_index < listFiles.Items.Count ) {
                listFiles.Items[m_index].SubItems[1].Text = txtAlias.Text;
                Edited = true;
                pictWave.Invalidate();
            }
        }

        private void txtOffset_TextChanged( object sender, EventArgs e ) {
            int i;
            if ( !int.TryParse( txtOffset.Text, out i ) ) {
                return;
            }
            m_offset = i;
            if ( 0 <= m_index && m_index < listFiles.Items.Count ) {
                listFiles.Items[m_index].SubItems[2].Text = txtOffset.Text;
                Edited = true;
                pictWave.Invalidate();
            }
        }

        private void txtConsonant_TextChanged( object sender, EventArgs e ) {
            int i;
            if ( !int.TryParse( txtConsonant.Text, out i ) ) {
                return;
            }
            m_consonant = i;
            if ( 0 <= m_index && m_index < listFiles.Items.Count ) {
                listFiles.Items[m_index].SubItems[3].Text = txtConsonant.Text;
                Edited = true;
                pictWave.Invalidate();
            }
        }

        private void txtBlank_TextChanged( object sender, EventArgs e ) {
            int i;
            if ( !int.TryParse( txtBlank.Text, out i ) ) {
                return;
            }
            m_blank = i;
            if ( 0 <= m_index && m_index < listFiles.Items.Count ) {
                listFiles.Items[m_index].SubItems[4].Text = txtBlank.Text;
                Edited = true;
                pictWave.Invalidate();
            }
        }

        private void txtPreUtterance_TextChanged( object sender, EventArgs e ) {
            int i;
            if ( !int.TryParse( txtPreUtterance.Text, out i ) ) {
                return;
            }
            m_pre_utterance = i;
            if ( 0 <= m_index && m_index < listFiles.Items.Count ) {
                listFiles.Items[m_index].SubItems[5].Text = txtPreUtterance.Text;
                Edited = true;
                pictWave.Invalidate();
            }
        }

        private void txtOverlap_TextChanged( object sender, EventArgs e ) {
            int i;
            if ( !int.TryParse( txtOverlap.Text, out i ) ) {
                return;
            }
            m_overlap = i;
            if ( 0 <= m_index && m_index < listFiles.Items.Count ) {
                listFiles.Items[m_index].SubItems[6].Text = txtOverlap.Text;
                Edited = true;
                pictWave.Invalidate();
            }
        }

        private void FormUtauVoiceConfig_FormClosed( object sender, FormClosedEventArgs e ) {
            m_cancel_required = true;
            while ( bgWork.IsBusy ) {
                Application.DoEvents();
            }
            if ( m_player != null ) {
                m_player.Close();
            }
        }

        private void btnMinus_Click( object sender, EventArgs e ) {
            if ( TRACKBAR_MIN < m_trackbar_value ) {
                m_trackbar_value--;
                UpdateScale();
            }
        }

        private void btnPlus_Click( object sender, EventArgs e ) {
            if ( m_trackbar_value < TRACKBAR_MAX ) {
                m_trackbar_value++;
                UpdateScale();
            }
        }

        private void bgWork_RunWorkerCompleted( object sender, RunWorkerCompletedEventArgs e ) {
            Edited = false;
        }

        private void pictWave_MouseHover( object sender, EventArgs e ) {

        }

        private void menuFileOpen_Click( object sender, EventArgs e ) {
            if ( openFileDialog.ShowDialog() == DialogResult.OK ) {
                Open( openFileDialog.FileName );
            }
        }

        private void menuFileQuit_Click( object sender, EventArgs e ) {
            this.Close();
        }

        private void menuFileSave_Click( object sender, EventArgs e ) {
            if ( m_oto_ini == "" ) {
                if ( saveFileDialog.ShowDialog() != DialogResult.OK ) {
                    return;
                }
                m_oto_ini = saveFileDialog.FileName;
            }
            SaveCor( m_oto_ini );
            Edited = false;
        }

        private void menuFileSaveAs_Click( object sender, EventArgs e ) {
            if ( saveFileDialog.ShowDialog() != DialogResult.OK ) {
                return;
            }
            m_oto_ini = saveFileDialog.FileName;
            SaveCor( m_oto_ini );
            Edited = false;
        }

        private void SaveCor( string file ) {
            using ( cp932writer sw = new cp932writer( file ) ) {
                int i1 = listFiles.Items.Count;
                for ( int i = 0; i < i1; i++ ) {
                    int i2 = listFiles.Items[i].SubItems.Count;
                    sw.Write( listFiles.Items[i].SubItems[0].Text + "=" );
                    for ( int j = 1; j <= 6; j++ ) {
                        sw.Write( (j > 1 ? "," : "") + listFiles.Items[i].SubItems[j].Text );
                    }
                    sw.WriteLine( "" );
                }
            }
        }

        private void menuFileOpen_MouseEnter( object sender, EventArgs e ) {
            statusLblTootip.Text = _( "Open Voice DB configuration file" );
        }

        private void menuFileSave_MouseEnter( object sender, EventArgs e ) {
            statusLblTootip.Text = _( "Save Voice DB configuration file." );
        }

        private void menuFileSaveAs_MouseEnter( object sender, EventArgs e ) {
            statusLblTootip.Text = _( "Save Voice DB configuration file with new name." );
        }

        private void menuFileQuit_MouseEnter( object sender, EventArgs e ) {
            statusLblTootip.Text = _( "Close this window." );
        }
    }

    public struct FormConfigUtauVoiceConfig {
        public Size Size;// = new Size( 714, 533 );
        public float InnerSplitterDistancePercentage;// = 70.0f;
        public float OuterSplitterDistancePercentage;// = 60.0f;
        public int WaveViewScale;

        public FormConfigUtauVoiceConfig( Size size, float inner_splitter_distance_percentage, float outer_splitter_distance_percentage, int wave_view_scale ) {
            Size = size;
            InnerSplitterDistancePercentage = inner_splitter_distance_percentage;
            OuterSplitterDistancePercentage = outer_splitter_distance_percentage;
            WaveViewScale = wave_view_scale;
        }
    }

}
#endif
