﻿/*
 * VocaloSysUtil.s
 * Copyright (c) 2009 kbinani
 *
 * This file is part of Boare.Lib.Vsq.
 *
 * Boare.Lib.Vsq is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Lib.Vsq is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.IO;
using System.Collections.Generic;
using Microsoft.Win32;

namespace Boare.Lib.Vsq {

    public static class VocaloSysUtil {
        private static bool s_initialized = false;

        private static string s_dll_path2 = "";
        private static string s_editor_path2 = "";
        private static string s_voicedbdir2 = "";
        private static List<SingerConfig> s_installed_singers2 = new List<SingerConfig>();
        private static List<SingerConfig> s_singer_configs2 = new List<SingerConfig>();

        private static string s_dll_path1 = "";
        private static string s_editor_path1 = "";
        private static string s_voicedbdir1 = "";
        private static List<SingerConfig> s_installed_singers1 = new List<SingerConfig>();
        private static List<SingerConfig> s_singer_configs1 = new List<SingerConfig>();

        private const int MAX_SINGERS = 0x4000;

        static VocaloSysUtil() {
            init_vocalo2();
            init_vocalo1();
        }

        /// <summary>
        /// Gets the name of original singer of specified program change.
        /// </summary>
        /// <param name="singer"></param>
        /// <returns></returns>
        public static string getOriginalSinger1( string singer ) {
            string voiceidstr = "";
            for ( int i = 0; i < s_singer_configs1.Count; i++ ) {
                if ( singer == s_singer_configs1[i].VOICENAME ) {
                    voiceidstr = s_singer_configs1[i].VOICEIDSTR;
                }
            }
            if ( voiceidstr == "" ) {
                return "";
            }
            for ( int i = 0; i < s_installed_singers1.Count; i++ ) {
                if ( voiceidstr == s_installed_singers1[i].VOICEIDSTR ) {
                    return s_installed_singers1[i].VOICENAME;
                }
            }
            return "";
        }

        /// <summary>
        /// Gets the name of original singer of specified program change.
        /// </summary>
        /// <param name="singer"></param>
        /// <returns></returns>
        public static string getOriginalSinger2( string singer ) {
            string voiceidstr = "";
            for ( int i = 0; i < s_singer_configs2.Count; i++ ) {
                if ( singer == s_singer_configs2[i].VOICENAME ) {
                    voiceidstr = s_singer_configs2[i].VOICEIDSTR;
                }
            }
            if ( voiceidstr == "" ) {
                return "";
            }
            for ( int i = 0; i < s_installed_singers2.Count; i++ ) {
                if ( voiceidstr == s_installed_singers2[i].VOICEIDSTR ) {
                    return s_installed_singers2[i].VOICENAME;
                }
            }
            return "";
        }

        /// <summary>
        /// Gets the voice language of specified program change
        /// </summary>
        /// <param name="name">name of singer</param>
        /// <returns></returns>
        private static VsqVoiceLanguage getLanguageFromName( string name ) {
            switch ( name.ToLower() ) {
                case "meiko":
                case "kaito":
                case "miku":
                case "rin":
                case "len":
                case "rin_act2":
                case "len_act2":
                case "gackpoid":
                case "luka_jpn":
                case "megpoid":
                    return VsqVoiceLanguage.Japanese;
                case "sweet_ann":
                case "prima":
                case "luka_eng":
                case "sonika":
                    return VsqVoiceLanguage.English;
            }
            return VsqVoiceLanguage.Default;
        }

        public static VsqID getSingerID1( string singer_name ) {
            VsqID ret = new VsqID( 0 );
            ret.type = VsqIDType.Singer;
            SingerConfig sc = null;
            for ( int i = 0; i < s_singer_configs1.Count; i++ ) {
                if ( s_singer_configs1[i].VOICENAME == singer_name ) {
                    sc = s_singer_configs1[i];
                    break;
                }
            }
            if ( sc == null ) {
                sc = new SingerConfig();
            }
            int lang = 0;
            foreach ( SingerConfig sc2 in s_installed_singers1 ) {
                if ( sc.VOICEIDSTR == sc2.VOICEIDSTR ) {
                    lang = (int)getLanguageFromName( sc.VOICENAME );
                    break;
                }
            }
            ret.IconHandle = new IconHandle();
            ret.IconHandle.IconID = "$0701" + sc.Program.ToString( "0000" );
            ret.IconHandle.IDS = sc.VOICENAME;
            ret.IconHandle.Index = 0;
            ret.IconHandle.Language = lang;
            ret.IconHandle.Length = 1;
            ret.IconHandle.Original = sc.Original;
            ret.IconHandle.Program = sc.Program;
            ret.IconHandle.Caption = "";
            return ret;
        }

        public static VsqID getSingerID2( string singer_name ) {
            VsqID ret = new VsqID( 0 );
            ret.type = VsqIDType.Singer;
            SingerConfig sc = null;
            for ( int i = 0; i < s_singer_configs2.Count; i++ ) {
                if ( s_singer_configs2[i].VOICENAME == singer_name ) {
                    sc = s_singer_configs2[i];
                    break;
                }
            }
            if ( sc == null ) {
                sc = new SingerConfig();
            }
            int lang = 0;
            foreach ( SingerConfig sc2 in s_installed_singers2 ) {
                if ( sc.VOICEIDSTR == sc2.VOICEIDSTR ) {
                    lang = (int)getLanguageFromName( sc.VOICENAME );
                    break;
                }
            }
            ret.IconHandle = new IconHandle();
            ret.IconHandle.IconID = "$0701" + sc.Program.ToString( "0000" );
            ret.IconHandle.IDS = sc.VOICENAME;
            ret.IconHandle.Index = 0;
            ret.IconHandle.Language = lang;
            ret.IconHandle.Length = 1;
            ret.IconHandle.Original = sc.Original;
            ret.IconHandle.Program = sc.Program;
            ret.IconHandle.Caption = "";
            return ret;
        }

        public static SingerConfig[] getSingerConfigs1() {
            return s_singer_configs1.ToArray();
        }

        public static SingerConfig[] getSingerConfigs2() {
            return s_singer_configs2.ToArray();
        }

        public static double getAmplifyCoeffFromPanLeft( int pan ) {
            return pan / -64.0 + 1.0;
        }

        public static double getAmplifyCoeffFromPanRight( int pan ) {
            return pan / 64.0 + 1.0;
        }

        public static double getAmplifyCoeffFromFeder( int feder ) {
            return Math.Exp( -1.26697245e-02 + 1.18448420e-01 * feder / 10.0 );
        }

        public static string getEditorPath2() {
            return s_editor_path2;
        }

        public static string getEditorPath1() {
            return s_editor_path1;
        }

        public static string getDllPathVsti2() {
            return s_dll_path2;
        }

        public static string getDllPathVsti1() {
            return s_dll_path1;
        }

        /// <summary>
        /// VOCALOID1システムのプロパティを取得
        /// </summary>
        private static void init_vocalo1() {
            // vocaloid1 dll path
            RegistryKey v1application = null;
            v1application = Registry.LocalMachine.OpenSubKey( "SOFTWARE\\VOCALOID\\APPLICATION", false );
            if ( v1application != null ) {
                string[] keys = v1application.GetSubKeyNames();
                for ( int i = 0; i < keys.Length; i++ ) {
                    RegistryKey key = v1application.OpenSubKey( keys[i], false );
                    if ( key != null ) {
                        string name = (string)key.GetValue( "PATH" );
                        if ( name.ToLower().EndsWith( "\\vocaloid.dll" ) ) {
                            s_dll_path1 = name;
                        } else if ( name.ToLower().EndsWith( "\\vocaloid.exe" ) ) {
                            s_editor_path1 = name;
                        }
                        key.Close();
                    }
                }
                v1application.Close();
            }

            // voicedbdir for vocaloid1
            RegistryKey v1database = Registry.LocalMachine.OpenSubKey( "SOFTWARE\\VOCALOID\\DATABASE\\VOICE", false );
            if ( v1database != null ) {
                s_voicedbdir1 = (string)v1database.GetValue( "VOICEDIR", "" );
#if DEBUG
                Console.WriteLine( "s_voicedbdir1=" + s_voicedbdir1 );
#endif
                // インストールされている歌手のVOICEIDSTRを列挙
                string[] singer_voiceidstrs = v1database.GetSubKeyNames();
                List<string> vvoice_keys = new List<string>();
                List<SingerConfig> vvoice_values = new List<SingerConfig>();
                foreach ( string voiceidstr in singer_voiceidstrs ) {
                    RegistryKey singer = v1database.OpenSubKey( voiceidstr );
                    if ( singer == null ) {
                        continue;
                    }
                    RegistryKey vvoice = singer.OpenSubKey( "vvoice" );
                    if ( vvoice != null ) {
                        string[] vvoices = vvoice.GetValueNames();

                        // インストールされた歌手の.vvdを読みにいく
                        // installdir以下の、拡張子.vvdのファイルを探す
                        foreach ( string file in Directory.GetFiles( Path.Combine( s_voicedbdir1, voiceidstr ), "*.vvd" ) ) {
                            SingerConfig config = SingerConfig.readSingerConfig( file, 0 ); //とりあえずプログラムチェンジは0
                            s_installed_singers1.Add( config );
                        }

                        // vvoice*.vvdを読みにいく。
                        foreach ( string s in vvoices ) {
#if DEBUG
                            Console.WriteLine( "s=" + s );
#endif
                            string file = Path.Combine( s_voicedbdir1, s + ".vvd" );
                            if ( File.Exists( file ) ) {
                                SingerConfig config = SingerConfig.readSingerConfig( file, 0 );
                                vvoice_keys.Add( s );
                                vvoice_values.Add( config );
                            }
                        }
                    }
                    singer.Close();
                }

                // voice.mapを読み込んで、s_singer_configs1のプログラムチェンジを更新する
                string map = Path.Combine( s_voicedbdir1, "voice.map" );
                if ( File.Exists( map ) ) {
                    using ( FileStream fs = new FileStream( map, FileMode.Open, FileAccess.Read ) ) {
                        byte[] dat = new byte[8];
                        fs.Seek( 0x20, SeekOrigin.Begin );
                        for ( int i = 0; i < MAX_SINGERS; i++ ) {
                            fs.Read( dat, 0, 8 );
                            ulong value = makelong_le( dat );
                            if ( value >= 1 ) {
#if DEBUG
                                Console.WriteLine( "value=" + value );
#endif
                                for ( int j = 0; j < vvoice_keys.Count; j++ ) {
                                    if ( vvoice_keys[j] == "vvoice" + value ) {
                                        vvoice_values[j].Program = i;
                                    }
                                }
                            }
                        }
                    }
                }

                // s_installed_singers1のSingerConfigのProgramとOriginalを適当に頒番する
                for ( int i = 0; i < s_installed_singers1.Count; i++ ) {
                    s_installed_singers1[i].Program = i;
                    s_installed_singers1[i].Original = i;
                }

                // s_singer_configs1を更新
                for ( int i = 0; i < vvoice_values.Count; i++ ) {
                    for ( int j = 0; j < s_installed_singers1.Count; j++ ) {
                        if ( vvoice_values[i].VOICEIDSTR == s_installed_singers1[j].VOICEIDSTR ) {
                            vvoice_values[i].Original = s_installed_singers1[j].Program;
                            break;
                        }
                    }
                    s_singer_configs1.Add( vvoice_values[i] );
                }
                v1database.Close();
            }
#if DEBUG
            Console.WriteLine( "installed" );
            foreach ( SingerConfig sc in s_installed_singers1 ) {
                Console.WriteLine( "VOICENAME=" + sc.VOICENAME + "; VOICEIDSTR=" + sc.VOICEIDSTR + "; Program=" + sc.Program + "; Original=" + sc.Original );
            }
            Console.WriteLine( "singer configs" );
            foreach ( SingerConfig sc in s_singer_configs1 ) {
                Console.WriteLine( "VOICENAME=" + sc.VOICENAME + "; VOICEIDSTR=" + sc.VOICEIDSTR + "; Program=" + sc.Program + "; Original=" + sc.Original );
            }
#endif
        }

        /// <summary>
        /// VOCALOID2システムのプロパティを取得
        /// </summary>
        private static void init_vocalo2() {
            // 最初はvstiとeditorのパスを取得
            RegistryKey v2application = Registry.LocalMachine.OpenSubKey( "SOFTWARE\\VOCALOID2\\APPLICATION", false );
            if ( v2application == null ) {
                v2application = Registry.LocalMachine.OpenSubKey( "SOFTWARE\\VOCALOID2_DEMO\\APPLICATION", false );
            }
            if ( v2application != null ) {
                string[] keys = v2application.GetSubKeyNames();
                for ( int i = 0; i < keys.Length; i++ ) {
                    RegistryKey key = v2application.OpenSubKey( keys[i], false );
                    if ( key != null ) {
                        string name = (string)key.GetValue( "PATH" );
                        if ( name.ToLower().EndsWith( "\\vocaloid2.dll" ) ) {
                            s_dll_path2 = name;
                        } else if ( name.ToLower().EndsWith( "\\vocaloid2_demo.dll" ) ) {
                            s_dll_path2 = name;
                        } else if ( name.ToLower().EndsWith( "\\vocaloid2.exe" ) ) {
                            s_editor_path2 = name;
                        }
                        key.Close();
                    }
                }
                v2application.Close();
            }

            // 歌声データベースを取得
            RegistryKey v2database = Registry.LocalMachine.OpenSubKey( "SOFTWARE\\VOCALOID2\\DATABASE\\VOICE", false );
            if ( v2database != null ) {
                // データベース（というよりもvoice.map）が保存されているパスを取得
                s_voicedbdir2 = (string)v2database.GetValue( "VOICEDIR", "" );
                // インストールされている歌手のVOICEIDSTRを列挙
                string[] singer_voiceidstrs = v2database.GetSubKeyNames();
                List<string> vvoice_keys = new List<string>();
                List<SingerConfig> vvoice_values = new List<SingerConfig>();
                foreach ( string voiceidstr in singer_voiceidstrs ) {
                    RegistryKey singer = v2database.OpenSubKey( voiceidstr );
                    if ( singer == null ) {
                        continue;
                    }
                    string installdir = (string)singer.GetValue( "INSTALLDIR", "" );
#if DEBUG
                    Console.WriteLine( "installdir=" + installdir );
#endif
                    RegistryKey vvoice = singer.OpenSubKey( "vvoice" );
                    if ( vvoice != null ) {
                        string[] vvoices = vvoice.GetValueNames();

                        // インストールされた歌手の.vvdを読みにいく
                        // installdir以下の、拡張子.vvdのファイルを探す
                        foreach ( string file in Directory.GetFiles( Path.Combine( installdir, voiceidstr ), "*.vvd" ) ) {
                            SingerConfig config = SingerConfig.readSingerConfig( file, 0 ); //とりあえずプログラムチェンジは0
                            s_installed_singers2.Add( config );
                        }

                        // vvoice*.vvdを読みにいく。場所は、installdirではなく、s_voicedbdir2
                        foreach ( string s in vvoices ) {
                            string file = Path.Combine( s_voicedbdir2, s + ".vvd" );
                            if ( File.Exists( file ) ) {
                                SingerConfig config = SingerConfig.readSingerConfig( file, 0 );
                                vvoice_keys.Add( s );
                                vvoice_values.Add( config );
                            }
                        }
                    }
                    singer.Close();
                }

                // voice.mapを読み込んで、s_singer_configs2のプログラムチェンジを更新する
                string map = Path.Combine( s_voicedbdir2, "voice.map" );
                if ( File.Exists( map ) ) {
                    using ( FileStream fs = new FileStream( map, FileMode.Open, FileAccess.Read ) ) {
                        byte[] dat = new byte[8];
                        fs.Seek( 0x20, SeekOrigin.Begin );
                        for ( int i = 0; i < MAX_SINGERS; i++ ) {
                            fs.Read( dat, 0, 8 );
                            ulong value = makelong_le( dat );
                            if ( value >= 1 ) {
#if DEBUG
                                Console.WriteLine( "value=" + value );
#endif
                                for ( int j = 0; j < vvoice_keys.Count; j++ ) {
                                    if ( vvoice_keys[j] == "vvoice" + value ) {
                                        vvoice_values[j].Program = i;
                                    }
                                }
                            }
                        }
                    }
                }

                // s_installed_singers2のSingerConfigのProgramとOriginalを適当に頒番する
                for ( int i = 0; i < s_installed_singers2.Count; i++ ) {
                    s_installed_singers2[i].Program = i;
                    s_installed_singers2[i].Original = i;
                }

                // s_singer_configs2を更新
                for ( int i = 0; i < vvoice_values.Count; i++ ) {
                    for ( int j = 0; j < s_installed_singers2.Count; j++ ) {
                        if ( vvoice_values[i].VOICEIDSTR == s_installed_singers2[j].VOICEIDSTR ) {
                            vvoice_values[i].Original = s_installed_singers2[j].Program;
                            break;
                        }
                    }
                    s_singer_configs2.Add( vvoice_values[i] );
                }
                v2database.Close();
            }
#if DEBUG
            Console.WriteLine( "installed" );
            foreach ( SingerConfig sc in s_installed_singers2 ) {
                Console.WriteLine( "VOICENAME=" + sc.VOICENAME + "; VOICEIDSTR=" + sc.VOICEIDSTR + "; Program=" + sc.Program + "; Original=" + sc.Original );
            }
            Console.WriteLine( "singer configs" );
            foreach ( SingerConfig sc in s_singer_configs2 ) {
                Console.WriteLine( "VOICENAME=" + sc.VOICENAME + "; VOICEIDSTR=" + sc.VOICEIDSTR + "; Program=" + sc.Program + "; Original=" + sc.Original );
            }
#endif
        }

        /// <summary>
        /// Transform the byte array(length=8) to unsigned long, assuming that the byte array is little endian.
        /// </summary>
        /// <param name="oct"></param>
        /// <returns></returns>
        private static ulong makelong_le( byte[] oct ) {
            return (ulong)oct[7] << 56 | (ulong)oct[6] << 48 | (ulong)oct[5] << 40 | (ulong)oct[4] << 32 | (ulong)oct[3] << 24 | (ulong)oct[2] << 16 | (ulong)oct[1] << 8 | (ulong)oct[0];
        }
    }

}
