#ifndef INCLUDED_HTASK
#define INCLUDED_HTASK

#include "base.h"
#include "types.h"
#include "Task.h"
#include "TaskQueue.h"
#include "QueueInfo.h"
//#include <stdio.h>

class TaskManagerImpl;
class SchedTask;

typedef void (*PostFunction)(SchedTask *s, void *read, void *write);

extern QueueInfo<TaskQueue>* taskQueuePool;

/*!
  @class
  
  @brief
  
  Cerium の Task で、spawn() でキューに格納されて順次実行される。
  cpu の指定があれば並列に実行される。
  特定の Task を待ち合わせる事が可能。
  Task の入出力は dma などで copy される。
 */

#include "SimpleTask.h"

class HTask : public SimpleTask {
public:
    BASE_NEW_DELETE(HTask);

    QueueInfo<TaskQueue> *wait_me;  // List of task waiting for me
    QueueInfo<TaskQueue> *wait_i;   // List of task for which I am waiting
    PostFunction post_func;
    void *post_arg1;
    void *post_arg2;
    CPU_TYPE cpu_type;
    TaskManagerImpl *mimpl;

    HTask *waiter;
    HTask *next;
    HTask *prev;

    struct {
	unsigned no_auto_free:1;        //      bit 0    auto free flag (0 .. auto, 1 manual)
    } flag;

    void spawn();
    void wait_for(HTask *);
    void set_cpu(CPU_TYPE type);    
    void set_post(PostFunction func, void *read, void *write);
    Task *create_task_array(int task_id, int num_task, int num_param, int num_inData, int num_outData);
    Task *next_task_array(int task_id, Task *t);
    void spawn_task_array(Task *t);

    HTask *init(int cmd, memaddr rbuf, int rs, memaddr wbuf, int ws) {
	init(cmd);
	set_input(rbuf, rs);
	set_output(wbuf, ws);
	return this;
    }

    void initOnce() {
       wait_me = new QueueInfo<TaskQueue>(taskQueuePool);
       wait_i = new QueueInfo<TaskQueue>(taskQueuePool);
    }

    void freeOnce() {
       delete wait_me;
       delete wait_i;
    }

    private:

    int param_index;
    int in_index;
    int out_index;

// compatibility
    public: // functions
    void add_inData_t(memaddr addr, int size) {
	Task *t = (Task*)rbuf;
	t->set_inData_t(in_index++, addr,size);
    }
    void add_outData_t(memaddr addr, int size) {
	Task *t = (Task*)rbuf;
	t->set_outData_t(out_index++, addr,size);
    }
    void set_inData_t(int index, memaddr addr, int size) {
#ifdef EARLY_TOUCH
        if ((unsigned long)addr&0xf) {
	  printf("inData is not aligned. command = %d, index = %d, addr = 0x%lx, size = %d\n", 
		 command, index, (unsigned long)addr, size);
	}
        char *p = (char *)addr; char b = *p;
        p = (char *)(addr+size-1); b += *p;
#endif
	Task *t = (Task*)rbuf;
	t->set_inData_t(index, addr,size);
    }
    void set_outData_t(int index, memaddr addr, int size) { 
#ifdef EARLY_TOUCH
	if ((unsigned long)addr&0xf) {
	  printf("inData is not aligned. command = %d, index = %d, addr = 0x%lx, size = %d\n", 
		 command, index, (unsigned long)addr, size);
	}
        char *p = (char *)addr; char b = *p;
        p = (char *)(addr+size-1); b += *p;
#endif
	Task *t = (Task*)rbuf;
	t->set_outData_t(index, addr,size);
    }
    void add_param_t(memaddr param) {
	Task *t = (Task*)rbuf;
	t->set_param_t(param_index++, param);
    }
    void set_param_t(int index, memaddr param) {
	if (command==TaskArray1) {
	    Task *t = (Task*)rbuf;
	    t->set_param_t(index, param);
	} else {
	    this->param = param;
	}
    }

    void no_auto_free() {
	flag.no_auto_free = 1;
    }
    void auto_free() {
	flag.no_auto_free = 0;
    }

    void init() {
	next = prev = NULL;
	waiter = NULL;
    }

    void init(int cmd) {
	command  = cmd;
	param_index = 0;
	in_index = 0;
	out_index = 0;
	flag.no_auto_free = 0;
	self = (memaddr) this;

	post_func = NULL;
	mimpl     = NULL;
	cpu_type  = CPU_PPE;

	post_arg1 = NULL;
	post_arg2 = NULL;
    }
#define add_param(param) add_param_t((memaddr)(param))
#define set_param(index,param) set_param_t(index, (memaddr) (param))

#define add_inData(addr, size)                  \
    add_inData_t((memaddr)(addr), (size));
#define add_outData(addr, size)                 \
    add_outData_t((memaddr)(addr), (size));


}  __attribute__ ((aligned (DEFAULT_ALIGNMENT)));

typedef HTask* HTaskPtr;

#endif
