﻿/* ------------------------------------------------------------------------- */
//
// Copyright (c) 2010 CubeSoft, Inc.
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Affero General Public License as published
// by the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Affero General Public License for more details.
//
// You should have received a copy of the GNU Affero General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/* ------------------------------------------------------------------------- */
using Cube.FileSystem;
using Cube.Tests;
using NUnit.Framework;
using System.Linq;
using System.Windows.Media;

namespace Cube.Pdf.Editor.Tests.Presenters
{
    /* --------------------------------------------------------------------- */
    ///
    /// OpenTest
    ///
    /// <summary>
    /// Tests Open commands of the MainViewModel class.
    /// </summary>
    ///
    /* --------------------------------------------------------------------- */
    [TestFixture]
    class OpenTest : ViewModelFixture
    {
        #region Tests

        /* ----------------------------------------------------------------- */
        ///
        /// OpenFile
        ///
        /// <summary>
        /// Executes the test for opening a PDF document and creating
        /// images.
        /// </summary>
        ///
        /* ----------------------------------------------------------------- */
        [TestCase("Sample.pdf",       ""        )]
        [TestCase("SampleAes128.pdf", "password")]
        public void OpenFile(string filename, string password) => Open(filename, password, vm =>
        {
            var dummy = vm.Data.Images.Preferences.Dummy;
            var dest  = vm.Data.Images.First();
            vm.Test(vm.Ribbon.Refresh);
            Assert.That(dest.Stretch, Is.EqualTo(Stretch.None));
            Assert.That(Wait.For(() => dest.Image != dummy), "Timeout");
            Assert.That(dest.Stretch, Is.EqualTo(Stretch.UniformToFill));
        });

        /* ----------------------------------------------------------------- */
        ///
        /// OpenLink
        ///
        /// <summary>
        /// Executes the test of the OpenLink command.
        /// </summary>
        ///
        /* ----------------------------------------------------------------- */
        [TestCase("Sample.pdf")]
        public void OpenLink(string filename)
        {
            Link(Get(filename), GetSource(filename));
            Create(vm =>
            {
                var src = IO.Get(Get($"{filename}.lnk"));
                Assert.That(src.Exists, Is.True);
                vm.Recent.Open.Execute(src);
                Assert.That(Wait.For(() => vm.Data.Count.Value == 2), "Timeout");
            });
        }

        /* ----------------------------------------------------------------- */
        ///
        /// OpenLink_NotFound
        ///
        /// <summary>
        /// Confirms the behavior when the specified link is invalid.
        /// </summary>
        ///
        /* ----------------------------------------------------------------- */
        [TestCase("Sample.pdf")]
        public void OpenLink_NotFound(string filename)
        {
            var tmp = Get(filename);
            var src = $"{tmp}.lnk";

            IO.Copy(GetSource(filename), tmp);
            Link(src, tmp);
            IO.TryDelete(tmp);
            Assert.That(IO.Exists(src), Is.True,  src);
            Assert.That(IO.Exists(tmp), Is.False, tmp);

            Create(vm =>
            {
                vm.Recent.Open.Execute(IO.Get(src));
                Assert.That(Wait.For(() => !IO.Exists(src)), src);
                Assert.That(vm.Data.IsOpen(), Is.False);
            });
        }

        /* ----------------------------------------------------------------- */
        ///
        /// OpenLink_Null
        ///
        /// <summary>
        /// Confirms the behavior when the specified link is null.
        /// </summary>
        ///
        /* ----------------------------------------------------------------- */
        [Test]
        public void OpenLink_Null() => Create(vm =>
        {
            vm.Recent.Open.Execute(null);
            Assert.That(Wait.For(() => !vm.Data.Busy.Value), "Timeout");
            Assert.That(vm.Data.IsOpen(), Is.False);
        });

        #endregion

        #region Others

        /* ----------------------------------------------------------------- */
        ///
        /// Link
        ///
        /// <summary>
        /// Creates a link with the specified path.
        /// </summary>
        ///
        /* ----------------------------------------------------------------- */
        private void Link(string link, string src) => new Shortcut(IO)
        {
            FullName     = link,
            Target       = src,
            IconLocation = src,
        }.Create();

        #endregion
    }
}
