/*
 * Copyright 2009-2010 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.fess.solr;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;

import jp.sf.fess.FessSystemException;
import jp.sf.fess.helper.SystemHelper;
import jp.sf.fess.solr.response.ReplicationResponse;

import org.apache.commons.httpclient.Credentials;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.auth.AuthScope;
import org.apache.commons.io.FileUtils;
import org.apache.solr.client.solrj.ResponseParser;
import org.apache.solr.client.solrj.impl.BinaryResponseParser;
import org.apache.solr.client.solrj.impl.CommonsHttpSolrServer;
import org.seasar.framework.container.SingletonS2Container;
import org.seasar.framework.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FessSolrServer extends CommonsHttpSolrServer {

    private static final long serialVersionUID = 1L;

    private static final Logger logger = LoggerFactory
            .getLogger(FessSolrServer.class);

    protected String dataDir;

    public FessSolrServer(String solrServerUrl) throws MalformedURLException {
        super(new URL(solrServerUrl));
    }

    public FessSolrServer(String solrServerUrl, HttpClient httpClient)
            throws MalformedURLException {
        super(new URL(solrServerUrl), httpClient,
                ((ResponseParser) (new BinaryResponseParser())), false);
    }

    public FessSolrServer(String solrServerUrl, HttpClient httpClient,
            boolean useMultiPartPost) throws MalformedURLException {
        super(new URL(solrServerUrl), httpClient,
                ((ResponseParser) (new BinaryResponseParser())),
                useMultiPartPost);
    }

    public FessSolrServer(String solrServerUrl, HttpClient httpClient,
            ResponseParser parser) throws MalformedURLException {
        super(new URL(solrServerUrl), httpClient, parser, false);
    }

    public FessSolrServer(URL baseURL) {
        super(baseURL, null, ((ResponseParser) (new BinaryResponseParser())),
                false);
    }

    public FessSolrServer(URL baseURL, HttpClient client) {
        super(baseURL, client, ((ResponseParser) (new BinaryResponseParser())),
                false);
    }

    public FessSolrServer(URL baseURL, HttpClient client,
            boolean useMultiPartPost) {
        super(baseURL, client, ((ResponseParser) (new BinaryResponseParser())),
                useMultiPartPost);
    }

    public FessSolrServer(URL baseURL, HttpClient client,
            ResponseParser parser, boolean useMultiPartPost) {
        super(baseURL, client, parser, useMultiPartPost);
    }

    public void setCredentials(AuthScope authscope, Credentials credentials) {
        getHttpClient().getState().setCredentials(authscope, credentials);
    }

    public String getDataDir() {
        return dataDir;
    }

    public void setDataDir(String dataDir) {
        if (StringUtil.isNotBlank(dataDir)) {
            String solrDataDir = System.getProperty("solr.data.dir");
            if (solrDataDir != null) {
                dataDir = dataDir.replaceAll("\\$\\{solr.data.dir\\}",
                        solrDataDir);
            }
            if (dataDir.endsWith("/")) {
                this.dataDir = dataDir.replaceAll("/+$", "");
            } else {
                this.dataDir = dataDir;
            }
        } else {
            this.dataDir = dataDir;
        }
    }

    public ReplicationResponse replicate(File snapshotDir) {
        if (snapshotDir == null) {
            throw new FessSystemException("snapshotDir is null.");
        }

        ReplicationResponse response = new ReplicationResponse();
        response.setRequestUrl(snapshotDir.getAbsolutePath());
        if (StringUtil.isBlank(getDataDir())) {
            throw new FessSolrReplicationException("Skipped a replication. "
                    + "DataDir is a blank. BaseURL is " + getBaseURL());
        }

        if (!snapshotDir.exists()) {
            throw new FessSolrReplicationException(
                    "Invalid snapshot directory: "
                            + snapshotDir.getAbsolutePath());
        }

        long startTime = System.currentTimeMillis();

        SystemHelper systemHelper = SingletonS2Container
                .getComponent("systemHelper");
        String sessionId = systemHelper.getSessionId();

        File indexDir = new File(getDataDir(), "index");
        File newIndexDir = new File(getDataDir(), "index");
        File newSnapshotDir = new File(getDataDir(), "index.new." + sessionId);
        File oldSnapshotDir = new File(getDataDir(), "index.old." + sessionId);

        try {
            try {
                FileUtils.copyDirectory(snapshotDir, newSnapshotDir);
            } catch (IOException e) {
                throw new FessSolrReplicationException("Could not copy "
                        + snapshotDir.getAbsolutePath() + " to"
                        + newSnapshotDir.getAbsolutePath());
            }
            if (!indexDir.renameTo(oldSnapshotDir)) {
                throw new FessSolrException(
                        "Could not rename an old dataDir to "
                                + oldSnapshotDir.getAbsolutePath());
            }
            if (!newSnapshotDir.renameTo(newIndexDir)) {
                // TODO should rollback above?
                throw new FessSolrException(
                        "Could not rename a new dataDir to "
                                + newIndexDir.getAbsolutePath());
            }
        } catch (FessSolrException e) {
            throw e;
        } catch (Exception e) {
            throw new FessSolrException("Failed to replicate index data.", e);
        }
        try {
            FileUtils.deleteDirectory(oldSnapshotDir);
        } catch (Exception e) {
            oldSnapshotDir.deleteOnExit();
            logger.warn("Could not delete old index directory: "
                    + oldSnapshotDir.getAbsolutePath(), e);
        }
        response.setElapsedTime(System.currentTimeMillis() - startTime);
        return response;
    }

}
