/*
 * Copyright 2009-2009 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.fess.action.admin;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import jp.sf.fess.Constants;
import jp.sf.fess.crud.util.SAStrutsUtil;
import jp.sf.fess.form.admin.SolrForm;
import jp.sf.fess.helper.SystemHelper;
import jp.sf.fess.solr.SolrServerGroup;
import jp.sf.fess.solr.SolrServerManager;
import jp.sf.fess.util.FessProperties;

import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.SolrQuery.ORDER;
import org.apache.solr.client.solrj.response.FacetField;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.seasar.struts.annotation.ActionForm;
import org.seasar.struts.annotation.Execute;
import org.seasar.struts.exception.ActionMessagesException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SolrAction implements Serializable {
    private static final Logger logger = LoggerFactory
            .getLogger(SolrAction.class);

    private static final long serialVersionUID = 1L;

    @ActionForm
    @Resource
    protected SolrForm solrForm;

    @Resource
    protected FessProperties solrServerProperties;

    @Resource
    protected SolrServerManager solrServerManager;

    @Resource
    protected SystemHelper systemHelper;

    protected String showIndex(boolean redirect) {
        String[] serverNames = solrServerManager.getSolrServerNames();
        for (String serverName : serverNames) {
            HashMap<String, String> map = new HashMap<String, String>(2);
            map.put("serverName", serverName.replace('.', '/'));
            String status = solrServerProperties.getProperty(serverName
                    + ".status");
            map.put("status", status != null ? status : Constants.ACTIVE);
            solrForm.serverStatusList.add(map);
        }

        solrForm.currentServerForSelect = solrServerProperties
                .getProperty(Constants.CURRENT_SERVER_FOR_SELECT);
        solrForm.currentServerForUpdate = solrServerProperties
                .getProperty(Constants.CURRENT_SERVER_FOR_UPDATE);
        if (redirect) {
            return "index?redirect=true";
        } else {
            return "index.jsp";
        }
    }

    @Execute(validator = false)
    public String index() {
        return showIndex(false);
    }

    @Execute(validator = true, input = "index.jsp")
    public String update() {
        // server status
        for (Map<String, String> statusMap : solrForm.serverStatusList) {
            for (Map.Entry<String, String> entry : statusMap.entrySet()) {
                String value = entry.getValue();
                solrServerProperties.setProperty(entry.getKey().replace('/',
                        '.')
                        + ".status", value);
                if (Constants.ACTIVE.equals(value)) {
                    String groupKey = entry.getKey().split("/")[0] + ".status";
                    if (!Constants.ACTIVE.equals(solrServerProperties
                            .getProperty(groupKey))) {
                        solrServerProperties.remove(groupKey);
                    }
                }
            }
        }
        try {
            solrServerProperties.store();
            SAStrutsUtil.addSessionMessage("success.update_solr_params");
        } catch (Exception e) {
            logger.error("Failed to update solr parameters.", e);
            throw new ActionMessagesException(
                    "errors.failed_to_update_solr_params", e);
        }

        return showIndex(true);

    }

    @Execute(validator = true, input = "index")
    public String commit() {
        if (systemHelper.isCrawlProcessRunning()) {
            throw new ActionMessagesException(
                    "errors.failed_to_start_solr_process_because_of_running");
        }
        final SolrServerGroup serverGroup = solrServerManager
                .getSolrServerGroup(solrForm.groupName);
        if (serverGroup == null) {
            throw new ActionMessagesException(
                    "errors.failed_to_commit_solr_index");
        } else {
            Thread thread = new Thread(new Runnable() {
                @Override
                public void run() {
                    if (systemHelper.readyCrawlProcess()) {
                        long execTime = System.currentTimeMillis();
                        try {
                            serverGroup.commit();
                            if (logger.isInfoEnabled()) {
                                logger
                                        .info("[EXEC TIME] index commit time: "
                                                + (System.currentTimeMillis() - execTime)
                                                + "ms");
                            }
                        } catch (Exception e) {
                            logger.error("Failed to commit index.", e);
                        } finally {
                            systemHelper.finishCrawlProcess();
                        }
                    } else {
                        if (logger.isInfoEnabled()) {
                            logger.info("could not start index cleanup process"
                                    + " because of running solr process.");
                        }
                    }
                }
            });
            thread.start();
            SAStrutsUtil.addSessionMessage("success.commit_solr_index");
        }
        return showIndex(true);
    }

    @Execute(validator = true, input = "index")
    public String optimize() {
        if (systemHelper.isCrawlProcessRunning()) {
            throw new ActionMessagesException(
                    "errors.failed_to_start_solr_process_because_of_running");
        }
        final SolrServerGroup serverGroup = solrServerManager
                .getSolrServerGroup(solrForm.groupName);
        if (serverGroup == null) {
            throw new ActionMessagesException(
                    "errors.failed_to_optimize_solr_index");
        } else {
            Thread thread = new Thread(new Runnable() {
                @Override
                public void run() {
                    if (systemHelper.readyCrawlProcess()) {
                        long execTime = System.currentTimeMillis();
                        try {
                            serverGroup.optimize();
                            if (logger.isInfoEnabled()) {
                                logger
                                        .info("[EXEC TIME] index optimize time: "
                                                + (System.currentTimeMillis() - execTime)
                                                + "ms");
                            }
                        } catch (Exception e) {
                            logger.error("Failed to optimize index.", e);
                        } finally {
                            systemHelper.finishCrawlProcess();
                        }
                    } else {
                        if (logger.isInfoEnabled()) {
                            logger.info("could not start index cleanup process"
                                    + " because of running solr process.");
                        }
                    }
                }
            });
            thread.start();
            SAStrutsUtil.addSessionMessage("success.optimize_solr_index");
        }
        return showIndex(true);
    }

    @Execute(validator = true, input = "index")
    public String delete() {
        if (systemHelper.isCrawlProcessRunning()) {
            throw new ActionMessagesException(
                    "errors.failed_to_start_solr_process_because_of_running");
        }
        final SolrServerGroup serverGroup = solrServerManager
                .getSolrServerGroup(solrForm.groupName);
        final String sessionId = solrForm.sessionId;
        if (serverGroup == null) {
            throw new ActionMessagesException(
                    "errors.failed_to_delete_solr_index");
        } else {
            Thread thread = new Thread(new Runnable() {
                @Override
                public void run() {
                    if (systemHelper.readyCrawlProcess()) {
                        long execTime = System.currentTimeMillis();
                        try {
                            if ("*".equals(sessionId)) {
                                serverGroup.deleteByQuery("*:*");
                            } else {
                                serverGroup.deleteByQuery("segment:"
                                        + sessionId);
                            }
                            serverGroup.optimize();
                            if (logger.isInfoEnabled()) {
                                logger
                                        .info("[EXEC TIME] index cleanup time: "
                                                + (System.currentTimeMillis() - execTime)
                                                + "ms");
                            }
                        } catch (Exception e) {
                            logger.error("Failed to delete index (segment:"
                                    + sessionId + ").", e);
                        } finally {
                            systemHelper.finishCrawlProcess();
                        }
                    } else {
                        if (logger.isInfoEnabled()) {
                            logger.info("could not start index cleanup process"
                                    + " because of running solr process.");
                        }
                    }
                }
            });
            thread.start();
            SAStrutsUtil.addSessionMessage("success.delete_solr_index");
        }
        return showIndex(true);
    }

    public List<Map<String, Object>> getGroupActionItems() {
        List<Map<String, Object>> groupActionItems = new ArrayList<Map<String, Object>>();
        try {
            for (String groupName : solrServerManager.getSolrServerGroupNames()) {
                Map<String, Object> map = new HashMap<String, Object>();
                map.put("groupName", groupName);
                map.put("sessionIdItems", getSessionIdList(groupName));
                groupActionItems.add(map);
            }
        } catch (Exception e) {
            logger.info("could not get server groups.", e);
        }
        return groupActionItems;
    }

    protected List<Map<String, String>> getSessionIdList(String groupName) {
        List<Map<String, String>> sessionIdList = new ArrayList<Map<String, String>>();

        SolrServerGroup serverGroup = solrServerManager
                .getSolrServerGroup(groupName);

        SolrQuery query = new SolrQuery();
        query.setQuery("*:*");
        query.setFacet(true);
        query.addFacetField("segment");
        query.addSortField("segment", ORDER.desc);

        QueryResponse queryResponse = serverGroup.query(query);
        List<FacetField> facets = queryResponse.getFacetFields();
        for (FacetField facet : facets) {
            List<FacetField.Count> facetEntries = facet.getValues();
            if (facetEntries != null) {
                for (FacetField.Count fcount : facetEntries) {
                    Map<String, String> map = new HashMap<String, String>();
                    map.put("label", fcount.getName() + " ("
                            + fcount.getCount() + ")");
                    map.put("value", fcount.getName());
                    sessionIdList.add(map);
                }
            }
        }
        return sessionIdList;
    }

    public boolean isSolrProcessRunning() {
        return systemHelper.isCrawlProcessRunning();
    }
}