/*
 * Copyright 2009-2011 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package jp.sf.fess.action;

import java.awt.Desktop;
import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.io.StringWriter;
import java.sql.Timestamp;
import java.text.NumberFormat;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import jp.sf.fess.Constants;
import jp.sf.fess.FessSystemException;
import jp.sf.fess.crypto.FessCipher;
import jp.sf.fess.db.allcommon.CDef;
import jp.sf.fess.db.exentity.ClickLog;
import jp.sf.fess.db.exentity.SearchLog;
import jp.sf.fess.entity.GeoInfo;
import jp.sf.fess.form.IndexForm;
import jp.sf.fess.helper.BrowserTypeHelper;
import jp.sf.fess.helper.HotSearchWordHelper;
import jp.sf.fess.helper.HotSearchWordHelper.Range;
import jp.sf.fess.helper.LabelTypeHelper;
import jp.sf.fess.helper.SearchLogHelper;
import jp.sf.fess.service.SearchService;
import jp.sf.fess.util.FacetResponse;
import jp.sf.fess.util.FacetResponse.Field;
import jp.sf.fess.util.FessProperties;
import jp.sf.fess.util.QueryResponseList;

import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.lang.StringUtils;
import org.seasar.framework.beans.util.Beans;
import org.seasar.framework.container.SingletonS2Container;
import org.seasar.framework.util.ArrayUtil;
import org.seasar.framework.util.StringUtil;
import org.seasar.struts.annotation.ActionForm;
import org.seasar.struts.annotation.Execute;
import org.seasar.struts.taglib.S2Functions;
import org.seasar.struts.util.MessageResourcesUtil;
import org.seasar.struts.util.RequestUtil;
import org.seasar.struts.util.ResponseUtil;

public class IndexAction implements Serializable {

    private static final long serialVersionUID = 1L;

    protected static final int DEFAULT_PAGE_SIZE = 20;

    protected static final long DEFAULT_START_COUNT = 0;

    @ActionForm
    @Resource
    protected IndexForm indexForm;

    @Resource
    protected SearchService searchService;

    @Resource
    protected BrowserTypeHelper browserTypeHelper;

    @Resource
    protected LabelTypeHelper labelTypeHelper;

    @Resource
    protected FessProperties crawlerProperties;

    @Resource
    protected HttpServletRequest request;

    public List<Map<String, Object>> documentItems;

    public FacetResponse facetResponse;

    public String pageSize;

    public String currentPageNumber;

    public String allRecordCount;

    public String allPageCount;

    public boolean existNextPage;

    public boolean existPrevPage;

    public String currentStartRecordNumber;

    public String currentEndRecordNumber;

    public List<String> pageNumberList;

    public String execTime;

    public List<Map<String, String>> labelTypeItems;

    public String errorMessage;

    private String pagingQuery = null;

    public String getPagingQuery() {
        if (pagingQuery == null) {
            final StringBuilder buf = new StringBuilder();
            if (StringUtil.isNotBlank(indexForm.additional)) {
                buf.append("&additional=").append(
                        S2Functions.u(indexForm.additional));
            }
            if (StringUtil.isNotBlank(indexForm.distance)) {
                buf.append("&distance=").append(
                        S2Functions.u(indexForm.distance));
            }
            if (StringUtil.isNotBlank(indexForm.latitude)) {
                buf.append("&latitude=").append(
                        S2Functions.u(indexForm.latitude));
            }
            if (StringUtil.isNotBlank(indexForm.longitude)) {
                buf.append("&longitude=").append(
                        S2Functions.u(indexForm.longitude));
            }
            if (StringUtil.isNotBlank(indexForm.labelTypeValue)) {
                buf.append("&labelTypeValue=").append(
                        S2Functions.u(indexForm.labelTypeValue));
            }

            pagingQuery = buf.toString();
        }
        return pagingQuery;
    }

    @Execute(validator = false)
    public String index() {
        if (isMobile()) {
            return "/mobile/?redirect=true";
        }

        buildLabelTypeItems();

        return "index.jsp";
    }

    protected String doSearch() {
        if (isMobile()) {
            return "/mobile/?redirect=true";
        }

        if (StringUtil.isBlank(indexForm.query)
                && StringUtil.isBlank(indexForm.labelTypeValue)) {
            // redirect to index page
            indexForm.query = null;
            return "index?redirect=true";
        }

        doSearchInternal();

        return "search.jsp";
    }

    @Execute(validator = true, input = "index")
    public String go() throws IOException {
        if (Constants.TRUE.equals(crawlerProperties.getProperty(
                Constants.SEARCH_LOG_PROPERTY, Constants.TRUE))) {
            final String userSessionId = getUserSessionId();
            if (userSessionId != null) {
                final SearchLogHelper searchLogHelper = SingletonS2Container
                        .getComponent(SearchLogHelper.class);
                final ClickLog clickLog = new ClickLog();
                clickLog.setUrl(indexForm.u);
                clickLog.setRequestedTime(new Timestamp(System
                        .currentTimeMillis()));
                clickLog.setQueryRequestedTime(new Timestamp(Long
                        .parseLong(indexForm.rt)));
                clickLog.setUserSessionId(userSessionId);
                searchLogHelper.addClickLog(clickLog);
            }
        }
        if (indexForm.u.startsWith("file:")) {
            if (Constants.TRUE.equals(crawlerProperties.getProperty(
                    Constants.SEARCH_DESKTOP_PROPERTY, Constants.FALSE))) {
                final Pattern pattern = Pattern.compile("file:/+([a-zA-Z]:.*)");
                final Matcher matcher = pattern.matcher(indexForm.u);
                String path;
                if (matcher.matches()) {
                    path = matcher.group(1);
                } else {
                    path = indexForm.u.replaceFirst("file:/+", "//");
                }
                final File file = new File(path);
                if (!file.exists()) {
                    errorMessage = MessageResourcesUtil.getMessage(RequestUtil
                            .getRequest().getLocale(),
                            "errors.not_found_on_file_system", indexForm.u);
                    return "error.jsp";
                }
                final Desktop desktop = Desktop.getDesktop();
                try {
                    desktop.open(file);
                } catch (final Exception e) {
                    errorMessage = MessageResourcesUtil.getMessage(RequestUtil
                            .getRequest().getLocale(),
                            "errors.could_not_open_on_system", indexForm.u);
                    return "error.jsp";
                }

                ResponseUtil.getResponse().setStatus(
                        HttpServletResponse.SC_NO_CONTENT);
                return null;
            } else {
                final String text = "<html><head><meta http-equiv=\"refresh\" content=\"0;URL="
                        + indexForm.u + "\"/></head></html>";
                ResponseUtil.write(text, "text/html");
            }
        } else {
            ResponseUtil.getResponse().sendRedirect(indexForm.u);
        }
        return null;
    }

    private String doSearchInternal() {
        final StringBuilder queryBuf = new StringBuilder(255);
        if (StringUtil.isNotBlank(indexForm.query)) {
            queryBuf.append(indexForm.query);
        }
        if (StringUtil.isNotBlank(indexForm.labelTypeValue)) {
            queryBuf.append(" label:\"").append(indexForm.labelTypeValue)
                    .append('\"');
        }
        if (StringUtil.isNotBlank(indexForm.additional)) {
            queryBuf.append(indexForm.additional);
        }

        final String query = queryBuf.toString().trim();

        // init pager
        if (StringUtil.isBlank(indexForm.start)) {
            indexForm.start = String.valueOf(DEFAULT_START_COUNT);
        } else {
            try {
                Long.parseLong(indexForm.start);
            } catch (final NumberFormatException e) {
                indexForm.start = String.valueOf(DEFAULT_START_COUNT);
            }
        }
        if (StringUtil.isBlank(indexForm.num)) {
            indexForm.num = String.valueOf(DEFAULT_PAGE_SIZE);
        } else {
            try {
                final int num = Integer.parseInt(indexForm.num);
                if (num > 100) {
                    // max page size
                    indexForm.num = "100";
                }
            } catch (final NumberFormatException e) {
                indexForm.num = String.valueOf(DEFAULT_PAGE_SIZE);
            }
        }

        if (indexForm.facet != null && ArrayUtil.isEmpty(indexForm.facet.field)
                && ArrayUtil.isEmpty(indexForm.facet.query)) {
            indexForm.facet = null;
        }

        final GeoInfo geoInfo = Beans.createAndCopy(GeoInfo.class, indexForm)
                .includes("latitude", "longitude", "distance").execute();

        final int pageOffset = Integer.parseInt(indexForm.start);
        final int pageSize = Integer.parseInt(indexForm.num);
        documentItems = searchService.selectList(query, indexForm.facet,
                pageOffset, pageSize, geoInfo.isAvailable() ? geoInfo : null);

        final QueryResponseList queryResponseList = (QueryResponseList) documentItems;
        facetResponse = queryResponseList.getFacetResponse();
        final NumberFormat nf = NumberFormat.getInstance(RequestUtil
                .getRequest().getLocale());
        nf.setMaximumIntegerDigits(2);
        nf.setMaximumFractionDigits(2);
        try {
            execTime = nf
                    .format((double) queryResponseList.getExecTime() / 1000);
        } catch (final Exception e) {
        }

        final long rt = System.currentTimeMillis();
        indexForm.rt = Long.toString(rt);

        // search log
        if (Constants.TRUE.equals(crawlerProperties.getProperty(
                Constants.SEARCH_LOG_PROPERTY, Constants.TRUE))) {
            final SearchLogHelper searchLogHelper = SingletonS2Container
                    .getComponent(SearchLogHelper.class);
            final SearchLog searchLog = new SearchLog();
            searchLog.setHitCount(queryResponseList.getAllRecordCount());
            searchLog.setResponseTime(Integer.valueOf((int) queryResponseList
                    .getExecTime()));
            searchLog.setSearchWord(query);
            searchLog.setSearchQuery(StringUtils.abbreviate(
                    queryResponseList.getSearchQuery(), 1000));
            searchLog.setSolrQuery(StringUtils.abbreviate(
                    queryResponseList.getSolrQuery(), 1000));
            searchLog.setRequestedTime(new Timestamp(rt));
            searchLog.setQueryOffset(pageOffset);
            searchLog.setQueryPageSize(pageSize);

            searchLog.setClientIp(StringUtils.abbreviate(
                    request.getRemoteAddr(), 50));
            searchLog.setReferer(StringUtils.abbreviate(
                    request.getHeader("referer"), 1000));
            searchLog.setUserAgent(StringUtils.abbreviate(
                    request.getHeader("user-agent"), 255));
            final String userSessionId = getUserSessionId();
            if (userSessionId != null) {
                searchLog.setUserSessionId(userSessionId);
            }
            final Object accessType = request
                    .getAttribute(Constants.SEARCH_LOG_ACCESS_TYPE);
            if (accessType != null && accessType instanceof CDef.AccessType) {
                searchLog.setAccessType(((CDef.AccessType) accessType).code());
            } else {
                searchLog.setAccessType(CDef.AccessType.Web.code());
            }
            searchLogHelper.addSearchLog(searchLog);
        }

        Beans.copy(documentItems, this)
                .includes("pageSize", "currentPageNumber", "allRecordCount",
                        "allPageCount", "existNextPage", "existPrevPage",
                        "currentStartRecordNumber", "currentEndRecordNumber",
                        "pageNumberList").execute();

        buildLabelTypeItems();

        return query;
    }

    @Execute(validator = false)
    public String search() {
        return doSearch();
    }

    @Execute(validator = false)
    public String prev() {
        return doMove(-1);
    }

    @Execute(validator = false)
    public String next() {
        return doMove(1);
    }

    @Execute(validator = false)
    public String move() {
        return doMove(0);
    }

    protected String doMove(final int move) {
        int pageSize = DEFAULT_PAGE_SIZE;
        if (StringUtil.isBlank(indexForm.num)) {
            indexForm.num = String.valueOf(DEFAULT_PAGE_SIZE);
        } else {
            try {
                pageSize = Integer.parseInt(indexForm.num);
            } catch (final NumberFormatException e) {
                indexForm.num = String.valueOf(DEFAULT_PAGE_SIZE);
            }
        }

        if (StringUtil.isBlank(indexForm.pn)) {
            indexForm.start = String.valueOf(DEFAULT_START_COUNT);
        } else {
            Integer pageNumber = Integer.parseInt(indexForm.pn);
            if (pageNumber != null && pageNumber > 0) {
                pageNumber = pageNumber + move;
                if (pageNumber < 1) {
                    pageNumber = 1;
                }
                indexForm.start = String.valueOf((pageNumber - 1) * pageSize);
            } else {
                indexForm.start = String.valueOf(DEFAULT_START_COUNT);
            }
        }

        return doSearch();
    }

    @Execute(validator = false)
    public String xml() throws IOException {
        if (Constants.FALSE.equals(crawlerProperties.getProperty(
                Constants.WEB_API_XML_PROPERTY, Constants.TRUE))) {
            return "index?redirect=true";
        }

        switch (getFormatType()) {
        case SEARCH:
            return searchByXml();
        case LABEL:
            return labelByXml();
        default:
            ResponseUtil.getResponse().sendError(
                    HttpServletResponse.SC_NOT_FOUND);
            return null;
        }
    }

    private String searchByXml() {
        int status = 0;
        String errMsg = Constants.EMPTY_STRING;
        final StringBuilder buf = new StringBuilder(1000);
        String query = null;
        request.setAttribute(Constants.SEARCH_LOG_ACCESS_TYPE,
                CDef.AccessType.Xml);
        try {
            query = doSearchInternal();
            buf.append("<query>");
            buf.append(StringEscapeUtils.escapeXml(query));
            buf.append("</query>");
            buf.append("<exec-time>");
            buf.append(execTime);
            buf.append("</exec-time>");
            buf.append("<page-size>");
            buf.append(pageSize);
            buf.append("</page-size>");
            buf.append("<page-number>");
            buf.append(currentPageNumber);
            buf.append("</page-number>");
            buf.append("<record-count>");
            buf.append(allRecordCount);
            buf.append("</record-count>");
            buf.append("<page-count>");
            buf.append(allPageCount);
            buf.append("</page-count>");
            buf.append("<result>");
            for (final Map<String, Object> document : documentItems) {
                buf.append("<doc>");
                for (final Map.Entry<String, Object> entry : document
                        .entrySet()) {
                    if (StringUtil.isNotBlank(entry.getKey())
                            && entry.getValue() != null) {
                        final String tagName = StringUtil
                                .decamelize(entry.getKey())
                                .replaceAll("_", "-").toLowerCase();
                        buf.append('<');
                        buf.append(tagName);
                        buf.append('>');
                        buf.append(StringEscapeUtils.escapeXml(entry.getValue()
                                .toString()));
                        buf.append("</");
                        buf.append(tagName);
                        buf.append('>');
                    }
                }
                buf.append("</doc>");
            }
            buf.append("</result>");
            if (facetResponse != null && facetResponse.hasFacetResponse()) {
                buf.append("<facet>");
                // facet field
                if (facetResponse.getFieldList() != null) {
                    for (final Field field : facetResponse.getFieldList()) {
                        buf.append("<field name=\"");
                        buf.append(StringEscapeUtils.escapeXml(field.getName()));
                        buf.append("\">");
                        for (final Map.Entry<String, Long> entry : field
                                .getValueCountMap().entrySet()) {
                            buf.append("<value count=\"");
                            buf.append(entry.getValue());
                            buf.append("\">");
                            buf.append(StringEscapeUtils.escapeXml(entry
                                    .getKey()));
                            buf.append("</value>");
                        }
                        buf.append("</field>");
                    }
                }
                // facet query
                if (facetResponse.getQueryCountMap() != null) {
                    buf.append("<query>");
                    for (final Map.Entry<String, Long> entry : facetResponse
                            .getQueryCountMap().entrySet()) {
                        buf.append("<value count=\"");
                        buf.append(entry.getValue());
                        buf.append("\">");
                        buf.append(StringEscapeUtils.escapeXml(entry.getKey()));
                        buf.append("</value>");
                    }
                    buf.append("</query>");
                }
                buf.append("</facet>");
            }
        } catch (final Exception e) {
            status = 1;
            errMsg = e.getMessage();
            if (errMsg == null) {
                errMsg = e.getClass().getName();
            }
        }

        writeXmlResponse(status, buf.toString(), errMsg);
        return null;
    }

    private String labelByXml() {
        int status = 0;
        String errMsg = Constants.EMPTY_STRING;
        final StringBuilder buf = new StringBuilder(255);
        try {
            labelTypeItems = labelTypeHelper.getLabelTypeItems();
            buf.append("<record-count>");
            buf.append(labelTypeItems.size());
            buf.append("</record-count>");
            buf.append("<result>");
            for (final Map<String, String> labelMap : labelTypeItems) {
                buf.append("<label>");
                buf.append("<name>");
                buf.append(StringEscapeUtils.escapeXml(labelMap
                        .get(Constants.ITEM_LABEL)));
                buf.append("</name>");
                buf.append("<value>");
                buf.append(StringEscapeUtils.escapeXml(labelMap
                        .get(Constants.ITEM_VALUE)));
                buf.append("</value>");
                buf.append("</label>");
            }
            buf.append("</result>");
        } catch (final Exception e) {
            status = 1;
            errMsg = e.getMessage();
        }

        writeXmlResponse(status, buf.toString(), errMsg);
        return null;
    }

    private void writeXmlResponse(final int status, final String body,
            final String errMsg) {
        final StringBuilder buf = new StringBuilder(1000);
        buf.append("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
        buf.append("<response>");
        buf.append("<version>");
        buf.append(Constants.WEB_API_VERSION);
        buf.append("</version>");
        buf.append("<status>");
        buf.append(status);
        buf.append("</status>");
        if (status == 0) {
            buf.append(body);
        } else {
            buf.append("<message>");
            buf.append(StringEscapeUtils.escapeXml(errMsg));
            buf.append("</message>");
        }
        buf.append("</response>");
        ResponseUtil.write(buf.toString(), "text/xml", Constants.UTF_8);

    }

    @Execute(validator = false)
    public String json() throws IOException {
        if (Constants.FALSE.equals(crawlerProperties.getProperty(
                Constants.WEB_API_JSON_PROPERTY, Constants.TRUE))) {
            return "index?redirect=true";
        }

        switch (getFormatType()) {
        case SEARCH:
            return searchByJson();
        case LABEL:
            return labelByJson();
        default:
            ResponseUtil.getResponse().sendError(
                    HttpServletResponse.SC_NOT_FOUND);
            return null;
        }
    }

    private String searchByJson() {
        int status = 0;
        String errMsg = Constants.EMPTY_STRING;
        String query = null;
        final StringBuilder buf = new StringBuilder(1000);
        request.setAttribute(Constants.SEARCH_LOG_ACCESS_TYPE,
                CDef.AccessType.Json);
        try {
            query = doSearchInternal();
            buf.append("\"query\":\"");
            buf.append(StringEscapeUtils.escapeJavaScript(query));
            buf.append("\",");
            buf.append("\"execTime\":");
            buf.append(execTime);
            buf.append(',');
            buf.append("\"pageSize\":");
            buf.append(pageSize);
            buf.append(',');
            buf.append("\"pageNumber\":");
            buf.append(currentPageNumber);
            buf.append(',');
            buf.append("\"recordCount\":");
            buf.append(allRecordCount);
            buf.append(',');
            buf.append("\"pageCount\":");
            buf.append(allPageCount);
            if (!documentItems.isEmpty()) {
                buf.append(',');
                buf.append("\"result\":[");
                boolean first1 = true;
                for (final Map<String, Object> document : documentItems) {
                    if (!first1) {
                        buf.append(',');
                    } else {
                        first1 = false;
                    }
                    buf.append('{');
                    boolean first2 = true;
                    for (final Map.Entry<String, Object> entry : document
                            .entrySet()) {
                        if (StringUtil.isNotBlank(entry.getKey())
                                && entry.getValue() != null) {
                            if (!first2) {
                                buf.append(',');
                            } else {
                                first2 = false;
                            }
                            buf.append('\"');
                            buf.append(escapeJsonString(entry.getKey()));
                            buf.append("\":\"");
                            buf.append(escapeJsonString(entry.getValue()
                                    .toString()));
                            buf.append('\"');
                        }
                    }
                    buf.append('}');
                }
                buf.append(']');
            }
            if (facetResponse != null && facetResponse.hasFacetResponse()) {
                // facet field
                if (facetResponse.getFieldList() != null) {
                    buf.append(',');
                    buf.append("\"facetField\":[");
                    boolean first1 = true;
                    for (final Field field : facetResponse.getFieldList()) {
                        if (!first1) {
                            buf.append(',');
                        } else {
                            first1 = false;
                        }
                        buf.append("{\"name\":\"");
                        buf.append(escapeJsonString(field.getName()));
                        buf.append("\",\"result\":[");
                        boolean first2 = true;
                        for (final Map.Entry<String, Long> entry : field
                                .getValueCountMap().entrySet()) {
                            if (!first2) {
                                buf.append(',');
                            } else {
                                first2 = false;
                            }
                            buf.append("{\"value\":\"");
                            buf.append(escapeJsonString(entry.getKey()));
                            buf.append("\",\"count\":");
                            buf.append(entry.getValue());
                            buf.append('}');
                        }
                        buf.append(']');
                        buf.append('}');
                    }
                    buf.append(']');
                }
                // facet query
                if (facetResponse.getQueryCountMap() != null) {
                    buf.append(',');
                    buf.append("\"facetQuery\":[");
                    boolean first1 = true;
                    for (final Map.Entry<String, Long> entry : facetResponse
                            .getQueryCountMap().entrySet()) {
                        if (!first1) {
                            buf.append(',');
                        } else {
                            first1 = false;
                        }
                        buf.append("{\"value\":\"");
                        buf.append(escapeJsonString(entry.getKey()));
                        buf.append("\",\"count\":");
                        buf.append(entry.getValue());
                        buf.append('}');
                    }
                    buf.append(']');
                }
            }
        } catch (final Exception e) {
            status = 1;
            errMsg = e.getMessage();
            if (errMsg == null) {
                errMsg = e.getClass().getName();
            }
        }

        writeJsonResponse(status, buf.toString(), errMsg);

        return null;
    }

    private String labelByJson() {
        int status = 0;
        String errMsg = Constants.EMPTY_STRING;
        final StringBuilder buf = new StringBuilder(255);
        try {
            labelTypeItems = labelTypeHelper.getLabelTypeItems();
            buf.append("\"recordCount\":");
            buf.append(labelTypeItems.size());
            if (!labelTypeItems.isEmpty()) {
                buf.append(',');
                buf.append("\"result\":[");
                boolean first1 = true;
                for (final Map<String, String> labelMap : labelTypeItems) {
                    if (!first1) {
                        buf.append(',');
                    } else {
                        first1 = false;
                    }
                    buf.append("{\"label\":\"");
                    buf.append(escapeJsonString(labelMap
                            .get(Constants.ITEM_LABEL)));
                    buf.append("\", \"value\":\"");
                    buf.append(escapeJsonString(labelMap
                            .get(Constants.ITEM_VALUE)));
                    buf.append("\"}");
                }
                buf.append(']');
            }
        } catch (final Exception e) {
            status = 1;
            errMsg = e.getMessage();
        }

        writeJsonResponse(status, buf.toString(), errMsg);

        return null;
    }

    private void writeJsonResponse(final int status, final String body,
            final String errMsg) {
        final boolean isJsonp = StringUtil.isNotBlank(indexForm.callback);

        final StringBuilder buf = new StringBuilder(1000);
        if (isJsonp) {
            buf.append(escapeCallbackName(indexForm.callback));
            buf.append('(');
        }
        buf.append("{\"response\":");
        buf.append("{\"version\":");
        buf.append(Constants.WEB_API_VERSION);
        buf.append(',');
        buf.append("\"status\":");
        buf.append(status);
        buf.append(',');
        if (status == 0) {
            buf.append(body);
        } else {
            buf.append("\"message\":\"");
            buf.append(StringEscapeUtils.escapeXml(errMsg));
            buf.append('\"');
        }
        buf.append('}');
        buf.append('}');
        if (isJsonp) {
            buf.append(')');
        }
        ResponseUtil.write(buf.toString(), "text/javascript+json",
                Constants.UTF_8);

    }

    private String escapeCallbackName(final String callbackName) {
        return callbackName.replaceAll("[^0-9a-zA-Z_\\$\\.]", "");
    }

    private String escapeJsonString(final String str) {
        if (str == null) {
            return "";
        }

        final StringWriter out = new StringWriter(str.length() * 2);
        int sz;
        sz = str.length();
        for (int i = 0; i < sz; i++) {
            final char ch = str.charAt(i);

            // handle unicode
            if (ch > 0xfff) {
                out.write("\\u" + hex(ch));
            } else if (ch > 0xff) {
                out.write("\\u0" + hex(ch));
            } else if (ch > 0x7f) {
                out.write("\\u00" + hex(ch));
            } else if (ch < 32) {
                switch (ch) {
                case '\b':
                    out.write('\\');
                    out.write('b');
                    break;
                case '\n':
                    out.write('\\');
                    out.write('n');
                    break;
                case '\t':
                    out.write('\\');
                    out.write('t');
                    break;
                case '\f':
                    out.write('\\');
                    out.write('f');
                    break;
                case '\r':
                    out.write('\\');
                    out.write('r');
                    break;
                default:
                    if (ch > 0xf) {
                        out.write("\\u00" + hex(ch));
                    } else {
                        out.write("\\u000" + hex(ch));
                    }
                    break;
                }
            } else {
                switch (ch) {
                case '"':
                    out.write("\\u0022");
                    break;
                case '\\':
                    out.write("\\u005C");
                    break;
                case '/':
                    out.write("\\u002F");
                    break;
                default:
                    out.write(ch);
                    break;
                }
            }
        }
        return out.toString();
    }

    private String hex(final char ch) {
        return Integer.toHexString(ch).toUpperCase();
    }

    protected boolean isMobile() {
        final String supportedSearch = crawlerProperties.getProperty(
                Constants.SUPPORTED_SEARCH_FEATURE_PROPERTY,
                Constants.SUPPORTED_SEARCH_WEB_MOBILE);
        if (Constants.SUPPORTED_SEARCH_MOBILE.equals(supportedSearch)) {
            return true;
        } else if (Constants.SUPPORTED_SEARCH_NONE.equals(supportedSearch)) {
            throw new FessSystemException("A search is not supported.");
        }

        if (browserTypeHelper == null) {
            return false;
        }

        return browserTypeHelper.isMobile();
    }

    public List<Map<String, String>> getLabelTypeItems() {
        return labelTypeItems;
    }

    public boolean isDisplayLabelTypeItems() {
        if (labelTypeItems != null) {
            return !labelTypeItems.isEmpty();
        } else {
            return false;
        }
    }

    public String getDisplayQuery() {
        final StringBuilder buf = new StringBuilder();
        buf.append(indexForm.query);
        if (StringUtil.isNotBlank(indexForm.labelTypeValue)) {
            for (final Map<String, String> map : labelTypeItems) {
                if (map.get(Constants.ITEM_VALUE).equals(
                        indexForm.labelTypeValue)) {
                    buf.append(' ');
                    buf.append(map.get(Constants.ITEM_LABEL));
                    break;
                }
            }
        }
        return buf.toString();
    }

    private void buildLabelTypeItems() {
        // label
        labelTypeItems = labelTypeHelper.getLabelTypeItems();

        if (!labelTypeItems.isEmpty() && indexForm.labelTypeValue == null) {
            indexForm.labelTypeValue = crawlerProperties
                    .getProperty(Constants.DEFAULT_LABEL_VALUE_PROPERTY);
        }

    }

    private FormatType getFormatType() {
        if (indexForm.type == null) {
            return FormatType.SEARCH;
        }
        final String type = indexForm.type.toUpperCase();
        if (FormatType.SEARCH.name().equals(type)) {
            return FormatType.SEARCH;
        } else if (FormatType.LABEL.name().equals(type)) {
            return FormatType.LABEL;
        } else {
            // default
            return FormatType.SEARCH;
        }
    }

    private String getUserSessionId() {
        final HttpSession session = request.getSession(false);
        if (session != null) {
            final FessCipher cipher = SingletonS2Container
                    .getComponent(FessCipher.class);
            return cipher.encryptoText(session.getId());
        }
        return null;
    }

    private static enum FormatType {
        SEARCH, LABEL;
    }

    @Execute(validator = false)
    public String hotsearchword() throws IOException {
        if (Constants.FALSE.equals(crawlerProperties.getProperty(
                Constants.WEB_API_HOT_SEARCH_WORD_PROPERTY, Constants.TRUE))) {
            return "index?redirect=true";
        }

        Range range;
        if (indexForm.range == null) {
            range = Range.ENTIRE;
        } else if (indexForm.range.equals("day") || indexForm.range.equals("1")) {
            range = Range.ONE_DAY;
        } else if (indexForm.range.equals("week")
                || indexForm.range.equals("7")) {
            range = Range.ONE_DAY;
        } else if (indexForm.range.equals("month")
                || indexForm.range.equals("30")) {
            range = Range.ONE_DAY;
        } else if (indexForm.range.equals("year")
                || indexForm.range.equals("365")) {
            range = Range.ONE_DAY;
        } else {
            range = Range.ENTIRE;
        }

        int status = 0;
        String errMsg = Constants.EMPTY_STRING;
        final StringBuilder buf = new StringBuilder(255);
        try {
            final HotSearchWordHelper hotSearchWordHelper = SingletonS2Container
                    .getComponent(HotSearchWordHelper.class);
            final List<String> wordList = hotSearchWordHelper
                    .getHotSearchWordList(range);
            buf.append("\"result\":[");
            boolean first1 = true;
            for (final String word : wordList) {
                if (!first1) {
                    buf.append(',');
                } else {
                    first1 = false;
                }
                buf.append("\"");
                buf.append(escapeJsonString(word));
                buf.append("\"");
            }
            buf.append(']');
        } catch (final Exception e) {
            status = 1;
            errMsg = e.getMessage();
        }

        writeJsonResponse(status, buf.toString(), errMsg);

        return null;
    }
}