﻿// Copyright (C) 2003 Daisuke Arai <darai@users.sourceforge.jp>
// Copyright (C) 2010 panacoran <panacoran@users.sourceforge.jp>
// Copyright (C) 2012 Daisuke Arai <darai@users.sourceforge.jp>
// 
// This program is part of Protra.
//
// Protra is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
// 
// $Id: PtFileTreeView.cs 438 2012-06-22 16:54:17Z darai $

using System;
using System.Collections.Generic;
using System.Windows.Forms;
using System.IO;

namespace PtSim.Controls
{
    /// <summary>
    /// 拡張子がptのファイルのTreeViewを実装するクラス
    /// </summary>
    public partial class PtFileTreeView : TreeView
    {
        private TreeNode previousSelectedNode;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PtFileTreeView()
        {
            InitializeComponent();
        }

        private string root;
        
        /// <summary>
        /// ルートディレクトリを取得または設定する。
        /// </summary>
        public string Root
        {
            get { return root; }
            set
            {
                root = value;
                SetNodes(Nodes, root);
            }
        }

        /// <summary>
        /// ディレクトリとファイルを表すノードを設定する。
        /// </summary>
        /// <param name="nodes">追加するTreeNodeCollection</param>
        /// <param name="path">パス</param>
        private void SetNodes(TreeNodeCollection nodes, string path)
        {
            nodes.Clear();
            if (!Directory.Exists(path))
                return;
            foreach (string d in Directory.GetDirectories(path))
            {
                var key = Path.GetFileName(d);
                var node = nodes.Add(key, key, 0, 0);
                SetNodes(node.Nodes, d);
            }
            foreach (string f in Directory.GetFiles(path, "*.pt"))
            {
                var key = Path.GetFileNameWithoutExtension(f);
                nodes.Add(key, key, 2, 2);
            }
        }

        /// <summary>
        /// 選択されているファイルの相対パスを取得する。
        /// </summary>
        public string SelectedRelativeFile
        {
            get
            {
                if (SelectedNode == null)
                    return null;
                string result = SelectedNode.FullPath + ".pt";
                if (!File.Exists(Path.Combine(root, result)))
                    return null;
                return result;
            }
        }

        /// <summary>
        /// BeforeExpandイベントを処理する。
        /// </summary>
        /// <param name="e">BeforeExpandイベントの引数</param>
        protected override void OnBeforeExpand(TreeViewCancelEventArgs e)
        {
            base.OnBeforeExpand(e);
            e.Node.ImageIndex = 1;
            e.Node.SelectedImageIndex = 1;
        }

        /// <summary>
        /// BeforeCollapseイベントを処理する。
        /// </summary>
        /// <param name="e">BeforeCollapseイベントの引数</param>
        protected override void OnBeforeCollapse(TreeViewCancelEventArgs e)
        {
            base.OnBeforeCollapse(e);
            e.Node.ImageIndex = 0;
            e.Node.SelectedImageIndex = 0;
        }


        private void PtFileTreeView_Validating(object sender, System.ComponentModel.CancelEventArgs e)
        {
            // フォーカスを失ったあとも、選択されたノードに色を付けておくための対処
            this.SelectedNode.BackColor = System.Drawing.SystemColors.Highlight;
            this.SelectedNode.ForeColor = System.Drawing.SystemColors.HighlightText;
            previousSelectedNode = this.SelectedNode;
        }

        private void PtFileTreeView_AfterSelect(object sender, TreeViewEventArgs e)
        {
            // フォーカスを失ったあとも、選択されたノードに色を付けておくための対処
            if (previousSelectedNode != null)
            {
                previousSelectedNode.BackColor = this.BackColor;
                previousSelectedNode.ForeColor = this.ForeColor;
            }
        }
    }
}
