﻿// Copyright (C) 2010 panacoran <panacoran@users.sourceforge.jp>
// 
// This program is part of Protra.
//
// Protra is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
// 
// $Id: ProtraConfig.cs 308 2010-03-19 14:51:46Z panacoran $

using System.Collections.Generic;
using System.Drawing;


namespace Protra.Lib.Config
{
    /// <summary>
    /// チャートの設定を格納するクラス。
    /// </summary>
    public class ChartConfig
    {
        /// <summary>
        /// 高さを取得または設定する。
        /// </summary>
        public int Height { get; set; }

        /// <summary>
        /// 日足のチャートを取得または設定する。
        /// </summary>
        public string DailyFile { get; set; }

        /// <summary>
        /// 週足のチャートを取得または設定する。
        /// </summary>
        public string WeeklyFile { get; set; }
    }

    /// <summary>
    /// Protraの設定を読み書きするクラス。
    /// </summary>
    public class ProtraConfig : ConfigBase
    {
        // フォーム

        /// <summary>
        /// 幅の設定を取得または設定する。
        /// </summary>
        public int Width { get; set; }

        /// <summary>
        /// 高さの設定を取得または設定する。
        /// </summary>
        public int Height { get; set; }

        /// <summary>
        /// パネルの左側の幅の設定を取得または設定する。
        /// </summary>
        public int PanelLeftWidth { get; set; }
        
        /// <summary>
        /// 価格情報の高さの設定を取得または設定する。
        /// </summary>
        public int PriceInfoHeight { get; set; }

        /// <summary>
        /// 最大化しているかの設定を取得または設定する。
        /// </summary>
        public bool Maximized { get; set; }

        // チャート

        /// <summary>
        /// チャートの表示モードの設定を取得または設定する。
        /// </summary>
        public int ChartMode { get; set; }

        /// <summary>
        /// チャートの前面色の設定を取得または設定する。
        /// </summary>
        public int ChartForeColor { get; set; }

        /// <summary>
        /// チャートの背面色の設定を取得または設定する。
        /// </summary>
        public int ChartBackColor { get; set; }

        /// <summary>
        /// 不完全な週足を描画するかの設定を取得または設定する。
        /// </summary>
        public bool DrawLastWeek { get; set; }

        /// <summary>
        /// 日足と週足で異なるチャートを使用するかの設定を取得または設定する。
        /// </summary>
        public bool UseDifferentChart { get; set; }

        /// <summary>
        /// チャートの個数の設定を取得または設定する。
        /// </summary>
        public int ChartNum { get; set; }

        /// <summary>
        /// チャートの設定のリストを取得または設定する。
        /// </summary>
        public List<ChartConfig> ChartList { get; set; }

        /// <summary>
        /// 設定ファイルの名前を取得する。
        /// </summary>
        protected override string ConfigName
        {
            get { return "protra"; }
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ProtraConfig() : base(false)
        {
            ChartNum = 3;
            ChartForeColor = Color.FromKnownColor(KnownColor.ControlText).ToArgb();
            ChartBackColor = Color.FromKnownColor(KnownColor.Control).ToArgb();
            ChartList = new List<ChartConfig>();
        }

        /// <summary>
        /// 古い設定ファイルを読み込む。
        /// </summary>
        protected override void ReadOldConfig()
        {
            ConfigFile ptopt = new ConfigFile("ptopt");
            try
            {
                ptopt.Read();
                ChartBackColor = int.Parse(ptopt["ChartBackColor"]);
                ChartForeColor = int.Parse(ptopt["ChartForeColor"]);
                DrawLastWeek = bool.Parse(ptopt["DrawLastWeek"]);
                UseDifferentChart = bool.Parse(ptopt["UseDifferentChart"]);
                ChartNum = int.Parse(ptopt["ChartMode"]);
            }
            catch
            { }

            ConfigFile ptenv = new ConfigFile("ptenv");
            try
            {
                ptenv.Read();
                // フォーム
                Width = int.Parse(ptenv["Width"]);
                Height = int.Parse(ptenv["Height"]);
                Maximized = bool.Parse(ptenv["Maximized"]);
                PanelLeftWidth = int.Parse(ptenv["PanelLeftWidth"]);
                PriceInfoHeight = int.Parse(ptenv["TextBoxPriceInfoHeight"]);
                ChartMode = int.Parse(ptenv["ChartMode"]);
                // チャート
                for (int i = 0; i < ChartNum; i++)
                {
                    ChartConfig config = new ChartConfig();
                    config.Height = int.Parse(ptenv["ChartHeight" + i.ToString()]);
                    config.DailyFile = ptenv["ChartFile" + i.ToString()];
                    config.WeeklyFile = ptenv["WeeklyChartFile" + i.ToString()];
                    ChartList.Add(config);
                }
            }
            catch
            { }
        }
    }
}
